indexing

	description:

		"Interface for input streams"

	library: "Gobo Eiffel Kernel Library"
	copyright: "Copyright (c) 2001, Eric Bezault and others"
	license: "Eiffel Forum License v2 (see forum.txt)"
	date: "$Date: 2003/10/24 20:52:59 $"
	revision: "$Revision: 1.16 $"

deferred class KI_INPUT_STREAM [G]

feature -- Input

	read is
			-- Read the next item in input stream.
			-- Make the result available in `last_item'.
		require
			is_open_read: is_open_read
			not_end_of_input: not end_of_input
		deferred
		end

	unread (an_item: G) is
			-- Put `an_item' back in input stream.
			-- This item will be read first by the next
			-- call to a read routine.
		require
			is_open_read: is_open_read
			an_item_valid: valid_unread_item (an_item)
		deferred
		ensure
			not_end_of_input: not end_of_input
			last_item_set: last_item = an_item
		end

	read_to_buffer (a_buffer: KI_BUFFER [G]; pos, nb: INTEGER): INTEGER is
			-- Fill `a_buffer', starting at position `pos', with
			-- at most `nb' items read from input stream.
			-- Return the number of items actually read.
		require
			is_open_read: is_open_read
			not_end_of_input: not end_of_input
			a_buffer_not_void: a_buffer /= Void
			pos_large_enough: pos >= 1
			nb_large_enough: nb > 0
			enough_space: (pos + nb - 1) <= a_buffer.count
		local
			i, end_pos: INTEGER
		do
			end_pos := pos + nb - 1
			from i := pos until i > end_pos loop
				read
				if not end_of_input then
					a_buffer.put (last_item, i)
					i := i + 1
				else
					Result := i - pos - nb
					i := end_pos + 1 -- Jump out of the loop.
				end
			end
			Result := Result + i - pos
		ensure
			nb_item_read_large_enough: Result >= 0
			nb_item_read_small_enough: Result <= nb
			not_end_of_input: not end_of_input implies Result > 0
		end

feature -- Status report

	is_closable: BOOLEAN is
			-- Can current input stream be closed?
		do
			Result := False
		ensure
			is_open: Result implies is_open_read
		end

	is_open_read: BOOLEAN is
			-- Can items be read from input stream?
		deferred
		end

	is_rewindable: BOOLEAN is
			-- Can current input stream be rewound to return input from
			-- the beginning of the stream?
		do
			Result := False
		ensure
			rewind_implies_open: Result implies is_open_read
		end

	end_of_input: BOOLEAN is
			-- Has the end of input stream been reached?
		require
			is_open_read: is_open_read
		deferred
		end

	valid_unread_item (an_item: G): BOOLEAN is
			-- Can `an_item' be put back in input stream?
		deferred
		end

feature -- Access

	name: STRING is
			-- Name of input stream
		deferred
		ensure
			name_not_void: Result /= Void
		end

	last_item: G is
			-- Last item read
		require
			is_open_read: is_open_read
			not_end_of_input: not end_of_input
		deferred
		end

feature -- Basic operations

	close is
			-- Try to close input stream if it is closable. Set
			-- `is_open_read' to false if operation was successful.
		require
			is_closable: is_closable
		do
			-- nothing
		end

	rewind is
			-- Move input position to the beginning of stream.
		require
			can_rewind: is_rewindable
		do
			-- nothing
		end

end
