/*
 * GPG Keys 
 * ckeyserver.cpp
 * (c) 2001 Peter Mathiasson <peter@mathiasson.nu>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation
 */

#include <qstring.h>
#include <qregexp.h>
#include <qsocket.h>
#include "ckeyserver.h"
#include "functions.h"
#include "config.h"

/* --------------------------------------------------------------------------------- */

enum {
    OP_SEARCH,
    OP_GET,
    OP_SEND
};

/* --------------------------------------------------------------------------------- */

CKeyServer::CKeyServer( QObject *parent ) : QObject( parent, "CKeyServer" )
{
    /* Create Socket Object */
    socket = new QSocket( this );
    connect( socket, SIGNAL(connectionClosed()), this, SLOT(socketDone()) );
    connect( socket, SIGNAL(error(int)), this, SLOT(socketError()) );
    connect( socket, SIGNAL(connected()), this, SLOT(socketConnected()) );
    connect( socket, SIGNAL(readyRead()), this, SLOT(socketRead()) );
}

/* --------------------------------------------------------------------------------- */

void CKeyServer::checkConfig()
{
    /* Keyserver */
    server = readConfigString( "keyserver", KEYSERVER );
    port = 11371;

    /* Search Command */
    if ( readConfigInt( "serversigs", SERVERSIGS ) ) {
        searchCommand = "vindex";
    } else {
        searchCommand = "index";
    }

    /* Request Timeout */
    requestTimeout = 90;
}

/* --------------------------------------------------------------------------------- */

void CKeyServer::search( QString searchString )
{
    /* Check Configuration */
    checkConfig();

    /* Set Operation */
    operation = OP_SEARCH;

    /* Build Request */
    searchString.replace( QRegExp(" "), "+" );
    request = "GET /pks/lookup?op=" + searchCommand + "&search=" + searchString + " HTTP/1.0\n\n";
    
    /* Reset Entry Counter */
    entriesFound = 0;

    /* Connect to Server */
    emit statusChange( Connecting );
    socket->connectToHost( server, port );
}

/* --------------------------------------------------------------------------------- */

void CKeyServer::getKey( QString keyID )
{
    /* Check Configuration */
    checkConfig();

    /* Set Operation */
    operation = OP_GET;

    /* Build Request */
    request = "GET /pks/lookup?op=get&search=0x" + keyID + " HTTP/1.0\n\n";
    
    /* Clear Get Data */
    data = QString::null;

    /* Connect to Server */
    emit statusChange( Connecting );
    socket->connectToHost( server, port );
}

/* --------------------------------------------------------------------------------- */

void CKeyServer::socketDone()
{
    emit statusChange( Idle );
    switch ( operation ) {
    
    case OP_SEARCH:
        emit searchDone();
        break;
    case OP_GET:
        emit getDone( data );
        break;
    case OP_SEND:
        emit sendDone();
    default:
        break;
    }
}

/* --------------------------------------------------------------------------------- */

void CKeyServer::socketError()
{
    emit connectionFailed();
    emit statusChange( Idle );
}

/* --------------------------------------------------------------------------------- */

void CKeyServer::socketConnected()
{
    emit statusChange( Sending );
    socket->writeBlock( request, request.length() );
    emit statusChange( Waiting );
}

/* --------------------------------------------------------------------------------- */

void CKeyServer::socketRead()
{
    emit statusChange( Recieving );
    switch ( operation ) {
    case OP_SEARCH:
        socketReadSearch();
        break;
    case OP_GET:
        socketReadGet();
        break;
    case OP_SEND:
        socketReadSend();
    default:
        break;
    }
}

/* --------------------------------------------------------------------------------- */

void CKeyServer::socketReadSearch()
{
    QString line, type, id, date, name;
    unsigned int i;
    
    while ( socket->canReadLine() ) {

        /* Read and Clean-up Line */
        line = removeHTMLTags( socket->readLine().stripWhiteSpace() );
        line.replace( QRegExp( "&gt;" ), ">" );
        line.replace( QRegExp( "&lt;" ), "<" );

        /* Check for valid entry */
        type = line.left(3);
        if ( type == "pub" || type == "sig" ) {
            
            /* Parse String */
            if ( type == "pub" ) {
                for ( i = 0; i < line.length() && line[i] != '/'; i++ );
                i++;
            } else {
                for ( i = 6; i < line.length() && line[i] == ' '; i++ );
            }            
            id = line.mid( i, 8 );
            if ( type == "pub" ) {
                for ( i += 8; i < line.length() && line[i] == ' '; i++ );
                date = line.mid( i, 10 );
            } else {
                date = QString::null;
                i += 8;
            }
            for ( i += 11; i < line.length() && line[i] == ' '; i++ );
            name = line.mid( i );

            /* Increase Entry Counter */
            entriesFound += ( type == "pub" );

            /* Emit Signal */
            emit itemFound( type, name, date, id );
            
        }

    }
}

/* --------------------------------------------------------------------------------- */

QString CKeyServer::removeHTMLTags( const QString string )
{
    QString newString = QString::null;
    bool skip = FALSE;

    for ( unsigned int i = 0; i < string.length(); i++ ) {
        
        if ( string[i] == '<' ) {
            skip = TRUE;
        }         
        if ( !skip ) {
            newString += string[i];
        }        
        if ( string[i] == '>' ) {
            skip = FALSE;
        }

    }

    return newString;
}

/* --------------------------------------------------------------------------------- */

void CKeyServer::socketReadGet()
{
    QString line;
    static bool skip = TRUE;

    while ( socket->canReadLine() ) {
        line = socket->readLine().stripWhiteSpace();

        if ( line == "-----BEGIN PGP PUBLIC KEY BLOCK-----" )
            skip = FALSE;
        if ( !skip )
            data += line + "\n";
        if ( line == "-----END PGP PUBLIC KEY BLOCK-----")
            skip = TRUE;
    }
}

/* --------------------------------------------------------------------------------- */

const QString CKeyServer::urlEncode( const QString data )
{
    QString returnString = QString::null;
    QString tempString;

    for ( unsigned int i = 0; i < data.length(); i++ ) {
        if ( data[i].isLetterOrNumber() || data[i] == '-' || data[i] == '_' || data[i] == '.' ||
             data[i] == '\n' ) {
            returnString += data[i];
        } else {
            tempString.sprintf( "%%%.2X", ((unsigned int)data[i]&0xFF) );
            returnString += tempString;
        }
    }

    return returnString;
}

/* --------------------------------------------------------------------------------- */

void CKeyServer::sendKey( QString keyData )
{
    /* Check Configuration */
    checkConfig();

    /* Set Operation */
    operation = OP_SEND;

    /* Clear Data */
    data = QString::null;

    /* Build Request */
    QString tmpString;
    QString encodedData = "keytext=" + urlEncode( keyData );
    tmpString.sprintf( "%d", encodedData.length()  );
    request = "POST /pks/add HTTP/1.0\n"
              "Content-type: application/x-www-form-urlencoded\n"
              "Content-length: " + tmpString + "\n\n" + encodedData;

    /* Connect to Server */
    emit statusChange( Connecting );
    socket->connectToHost( server, port );

}

/* --------------------------------------------------------------------------------- */

void CKeyServer::socketReadSend()
{
    while ( socket->canReadLine() ) {
        data += socket->readLine().stripWhiteSpace() + '\n';
    }
}

/* --------------------------------------------------------------------------------- */
