/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*---------------------------------------------------------------------------

   piv2h5 - converts PIV data to hdf5 format data

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

------------------------------------------------------------------------*/

#include <stdio.h> 
#include <stdlib.h> 
#include <time.h>
#include <gpiv.h>

/* #define PARFILE "scale.par"    */  /* Parameter file name */
#define PARFILE "gpivrc"     /* Parameter file name */
#define USAGE "\
Usage: piv2hdf [-e] [-h | --help] [-i] [-p | --print] [-v | --version] \n\
               filename \n\
\n\
keys: \n\
-e:                    exclude data \n\
-h | --help:           this on-line help \n\
-i:                    includes image data \n\
-p | --print:          print parameters to stdout \n\
-v | --version:        version number \n\
"

#define HELP  "\
piv2hdf - converts ASCII PIV-data to hdf5 formatted data"

#define RCSID "$Id: piv2hdf.c,v 1.9 2006/01/31 14:18:04 gerber Exp $"

gboolean print_par = FALSE,  exclude_data = FALSE, include_image = FALSE;

void 
command_args(int argc, 
             char *argv[], 
             char fname[GPIV_MAX_CHARS]
             )
/*-----------------------------------------------------------------------------
 * Command line argument handling
 */
{
    char c = '\0';
    int argc_next;


    while (--argc > 0 && (*++argv)[0] == '-') {

/*
 * argc_next is set to 1 if the next cmd line argument has to be searched for; 
 * in case that the command line argument concerns more than one char or cmd 
 * line argument needs a parameter 
 */
        argc_next = 0;
	while (argc_next == 0 && (c = *++argv[0]))

            switch (c) {
            case 'v':
                printf("%s\n", RCSID); 
                exit(0);
                break;
            case 'e':
                exclude_data = TRUE;
                break;
            case 'h':
                printf("%s\n", RCSID); 
                printf("%s\n",HELP);
                printf("%s\n",USAGE);
                exit(0);
                break;
            case 'i':
                include_image = TRUE;
                break;
            case 'p':
                print_par = TRUE;
                break;

/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                } else if (strcmp("-print", *argv) == 0) {
		    print_par = TRUE;
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);
                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

            default:
                gpiv_error(USAGE);
                break;
            }
     }

     if(argc != 1) { 
	  gpiv_error("%s: %s", RCSID, USAGE);
     }
     strcpy(fname, argv[0]); 


}


void 
make_fname(char *fname, 
           char *fname_img, 
           char *fname_piv, 
           char *fname_vor, 
           char *fname_nstrain, 
           char *fname_sstrain, 
           char *fname_out
           )
/*-----------------------------------------------------------------------------
 * generates filenames
 */
{
     if (include_image) {
         gpiv_io_make_fname(fname, GPIV_EXT_RAW_IMAGE, fname_img);
         if (print_par) printf("# Image data file: %s\n",fname_img);
     }

     if (exclude_data == FALSE) {
         gpiv_io_make_fname(fname, GPIV_EXT_PIV, fname_piv);
         if (print_par) printf("# Input data file: %s\n",fname_piv);
         
         gpiv_io_make_fname(fname, GPIV_EXT_VOR, fname_vor);
         if (print_par) printf("# Input data file: %s\n",fname_vor);
         
         gpiv_io_make_fname(fname, GPIV_EXT_NSTR, fname_nstrain);
         if (print_par) printf("# Input data file: %s\n",fname_nstrain);
         
         gpiv_io_make_fname(fname, GPIV_EXT_SSTR, fname_sstrain);
         if (print_par) printf("# Input data file: %s\n",fname_sstrain);
         
     }

         gpiv_io_make_fname(fname, GPIV_EXT_GPIV, fname_out);
         if (print_par) printf("# Output file: %s\n",fname_out);
}



   



int 
main(int argc, 
     char *argv[]
     )
/*-----------------------------------------------------------------------------
 * main routine to converts ASCII PIV-data to hdf5 formatted data
 */
{
    char *err_msg = NULL;
    FILE *fp;
    char fname[GPIV_MAX_CHARS], 
        fname_out[GPIV_MAX_CHARS], 
        fname_vor[GPIV_MAX_CHARS],
        fname_nstrain[GPIV_MAX_CHARS],
        fname_sstrain[GPIV_MAX_CHARS],
        fname_piv[GPIV_MAX_CHARS],
        fname_header[GPIV_MAX_CHARS],
        fname_img[GPIV_MAX_CHARS],
        fname_parameter[GPIV_MAX_CHARS];
    time_t itime;
    char  c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    int nc_lines=0, var_scale=0;
    int read_img = 0, read_piv = 0, read_vor = 0, read_nstrain = 0, 
        read_sstrain = 0;

    GpivImagePar image_par;
    GpivEvalPar piv_eval_par;
    GpivValidPar piv_valid_par;
    GpivPostPar piv_post_par;

    GpivPivData piv_data;
    GpivScalarData scal_data;

    piv_data.nx = 0;
    piv_data.ny = 0;
    piv_data.point_x = NULL;
    piv_data.point_y = NULL;
    piv_data.dx = NULL;
    piv_data.dy = NULL;
    piv_data.snr = NULL;
    piv_data.peak_no = NULL;

    scal_data.nx = 0;
    scal_data.ny = 0;
    scal_data.point_x = NULL;
    scal_data.point_y = NULL;
    scal_data.scalar = NULL;
    scal_data.flag = NULL;


/*
 * Initializing parameters
 */
    gpiv_img_parameters_logic(&image_par, FALSE);
    gpiv_piv_parameters_logic(&piv_eval_par, FALSE);
    gpiv_valid_parameters_logic(&piv_valid_par, FALSE);
    gpiv_post_parameters_logic(&piv_post_par, FALSE);


    command_args(argc, argv, fname);
    make_fname(fname,  fname_img, fname_piv, fname_vor, fname_nstrain, 
               fname_sstrain, fname_out);
    gpiv_io_make_fname(fname, GPIV_EXT_HEADER, fname_header);
    gpiv_io_make_fname(fname, GPIV_EXT_PAR, fname_parameter);
	  
/*
 * Reads piv program parameters from fname_parameter
 */
    gpiv_scan_parameter("", fname_header, &image_par, print_par);
    gpiv_scan_parameter(GPIV_IMAGE_PAR_KEY, fname_parameter, &image_par, 
                        print_par);
    if ((err_msg =
         gpiv_scan_resourcefiles(GPIV_IMAGE_PAR_KEY, &image_par, print_par))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

    if (image_par.creation_date_logic == FALSE) {
        char time_string[GPIV_MAX_CHARS];
        struct tm time_struct;

        time(&itime);
        time_struct = *gmtime(&itime);
        strftime(time_string, GPIV_MAX_CHARS, "%a %b %s %Y %T", &time_struct);
        if (print_par)
            gpiv_warning("Setting new time = %s", time_string);

        snprintf(image_par.creation_date, GPIV_MAX_CHARS, "%s", time_string);
        image_par.creation_date_logic = TRUE;
    }
    gpiv_scan_parameter(GPIV_EVAL_PAR_KEY, fname_parameter, &piv_eval_par, 
                        print_par);
    gpiv_scan_parameter(GPIV_VALID_PAR_KEY, fname_parameter, &piv_valid_par, 
                        print_par);
    gpiv_scan_parameter(GPIV_POST_PAR_KEY, fname_parameter, &piv_post_par, 
                        print_par);

    if ((err_msg =
         gpiv_scan_resourcefiles(GPIV_EVAL_PAR_KEY, &piv_eval_par, print_par))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
    if ((err_msg =
         gpiv_scan_resourcefiles(GPIV_VALID_PAR_KEY, &piv_eval_par, print_par))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
    if ((err_msg =
         gpiv_scan_resourcefiles(GPIV_POST_PAR_KEY, &piv_eval_par, print_par))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
/* 
 * writing to output int hdf5
 */
    if ((err_msg =
         gpiv_fcreate_hdf5 (fname_out))
        != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);


/*
 * Image data
 */
     if ((err_msg =
          gpiv_img_fwrite_hdf5_parameters (fname_out, &image_par))
         != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

     if (include_image) {
         if((fp=fopen(fname_img, "r")) == NULL) {
             read_img = 0;
             if (print_par) gpiv_warning("no image data to be included");
         } else {
             read_img = 1;
             if (print_par) gpiv_warning("including image data");
         }

         if (read_img) {
             guint16 **img1 = NULL, **img2 = NULL;
#ifdef HD5_IMAGE_INT
             int **i_img1 = NULL, **i_img2 = NULL;
             int i, j;
#endif
             fclose(fp);

             img1 = gpiv_alloc_img(image_par);
             if (image_par.x_corr) 
                 img2 = gpiv_alloc_img(image_par);
             gpiv_fread_image(fname_img, img1, img2, image_par);
             
#ifdef HD5_IMAGE_INT
             i_img1 = gpiv_imatrix(0, image_par.nrows - 1, 0, 
                                  image_par.ncolumns - 1);
             if (image_par.x_corr) 
                 i_img2 = gpiv_imatrix(0, image_par.nrows - 1, 0, 
                                      image_par.ncolumns - 1);
             for (i = 0; i < image_par.nrows; i++) {
                 for (j = 0; j < image_par.ncolumns; j++) {
                     i_img1[i][j] =  img1[i][j];
                     if (image_par.x_corr) i_img2[i][j] = img2[i][j];
                 }
             }

             if ((err_msg =
                  gpiv_fwrite_hdf5_image (fname_out, i_img1, i_img2, 
                                          &image_par))
                 != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

#else
             if ((err_msg =
                  gpiv_fwrite_hdf5_image (fname_out, img1, img2, &image_par))
                 != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
#endif


             gpiv_free_img(img1, image_par);
             if (image_par.x_corr) 
                 gpiv_free_img(img2, image_par);
#ifdef HD5_IMAGE_INT
             gpiv_free_imatrix(i_img1, 0, image_par.nrows - 1, 0, 
                                image_par.ncolumns - 1);
             if (image_par.x_corr) 
                 gpiv_free_imatrix(i_img2, 0, image_par.nrows - 1, 0, 
                                    image_par.ncolumns - 1);
#endif
         }
     }


/*
 * Piv data
 */
     if (exclude_data == FALSE) {
         if((fp=fopen(fname_piv, "r")) == NULL) {
             read_piv = 0;
             if (print_par) gpiv_warning("no piv data to be included");
         } else {
             read_piv = 1;
             if (print_par) gpiv_warning("including piv data");
         }
         
         if (read_piv) {
             fclose(fp);
             
             if ((var_scale = gpiv_fcount_pivdata(fname_piv, &piv_data)) == -1
                 ) {
                 gpiv_error("%s: Failure calling gpiv_count_pivdata", RCSID);
             }
             if (print_par) gpiv_warning("nx=%d ny=%d", piv_data.nx, 
                                         piv_data.ny);
             gpiv_alloc_pivdata(&piv_data);
             gpiv_fread_pivdata(fname_piv, &piv_data, c_line, &nc_lines);
             if ((err_msg =
                  gpiv_fwrite_hdf5_piv_position(fname_out, &piv_data, 
                                                RCSID, &var_scale))
                 != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

             if ((err_msg =
                  gpiv_fwrite_hdf5_pivdata (fname_out, &piv_data,  "PIV", 
                                            RCSID, &var_scale))
                 != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

/* Other option: */
/*      gpiv_fwrite_hdf5_parameters (fname_out, GPIV_EVAL_PAR_KEY, &piv_eval_par); */
             if ((err_msg =
                  gpiv_piv_fwrite_hdf5_parameters (fname_out, &piv_eval_par))
                 != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

              if ((err_msg =
                   gpiv_valid_fwrite_hdf5_parameters (fname_out, 
                                                      &piv_valid_par))
                 != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

             gpiv_free_pivdata(&piv_data);
         }


/*
 * Scalar data: vorticity
 */
         if((fp=fopen(fname_vor, "r")) == NULL) {
             read_vor = 0;
             if (print_par) gpiv_warning("no vorticity data to be included");
         } else {
             read_vor = 1;
             if (print_par) gpiv_warning("including vorticity data");
         }
         
         if (read_vor) {
             fclose(fp);
             scal_data.nx = piv_data.nx;
             scal_data.ny = piv_data.ny;
             gpiv_alloc_scdata(&scal_data);
             gpiv_fread_scdata(fname_vor, &scal_data, c_line, &nc_lines);
/*     gpiv_fwrite_hdf5_scdata(fname_out, scal_data, "SHEAR_STRESS"); */
/*     gpiv_fwrite_hdf5_scdata(fname_out, scal_data, "NORMAL_STRESS"); */
             if ((err_msg =
                  gpiv_fwrite_hdf5_scdata(fname_out, &scal_data, "GPIV_VORTICITY",
                                          RCSID))
                 != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
             
             if ((err_msg =
                  gpiv_post_fwrite_hdf5_parameters (fname_out, &piv_post_par))
                 != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

             gpiv_free_scdata(&scal_data);
         }

/*
 * Scalar data: sstrain
 */
         if((fp=fopen(fname_sstrain, "r")) == NULL) {
             read_sstrain = 0;
             if (print_par) gpiv_warning("no shear strain data to be included");
         } else {
             read_sstrain = 1;
             if (print_par) gpiv_warning("including shear strain data");
         }

         if (read_sstrain) {
             fclose(fp);
             scal_data.nx = piv_data.nx;
             scal_data.ny = piv_data.ny;
             gpiv_alloc_scdata(&scal_data);
             gpiv_fread_scdata(fname_sstrain, &scal_data, c_line, &nc_lines);
             if ((err_msg =
                  gpiv_fwrite_hdf5_scdata(fname_out, &scal_data, 
                                          "SHEAR_STRESS", RCSID))
                != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

             if ((err_msg =
                  gpiv_post_fwrite_hdf5_parameters (fname_out, &piv_post_par))
                != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

             gpiv_free_scdata(&scal_data);
         }
         
/*
 * Scalar data: normal strain
 */
         if((fp=fopen(fname_nstrain, "r")) == NULL) {
             read_nstrain = 0;
             if (print_par) gpiv_warning("no normal strain data to be included");
         } else {
             read_nstrain = 1;
             if (print_par) gpiv_warning("including normal strain data");
         }
         
         if (read_nstrain) {
             fclose(fp);
             scal_data.nx = piv_data.nx;
             scal_data.ny = piv_data.ny;
             gpiv_alloc_scdata(&scal_data);
             gpiv_fread_scdata(fname_nstrain, &scal_data, c_line, &nc_lines);
             if ((err_msg =
             gpiv_fwrite_hdf5_scdata(fname_out, &scal_data, 
                                     "NORMAL_STRESS", RCSID))
                != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

             if ((err_msg =
                  gpiv_post_fwrite_hdf5_parameters (fname_out, &piv_post_par))
                != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);

             gpiv_free_scdata(&scal_data);
         }
     }


    exit(0);
}
