#include <stdio.h>
#include <ctype.h>

#include <glib.h>

#include "gql++/object.h"

namespace GQL
{

SQLType::SQLType()
{
  type_ = VOID;
  length_ = 0;
  decimals_ = 0;
}

SQLType::SQLType(TypeCode type, ...)
{
  va_list args;
  
  type_ = type;
  length_ = 0;
  decimals_ = 0;

  va_start(args, type);
  
  switch (type)
  {
    case SQLType::VOID:
    case SQLType::BOOLEAN:
    case SQLType::SMALLINT:
    case SQLType::INTEGER:
    case SQLType::INTERVAL:
    case SQLType::DATE:
    case SQLType::TIME:
    case SQLType::TIME_WITH_TZ:
      break;
    case SQLType::NUMERIC:
    case SQLType::DECIMAL:
      length_ = va_arg(args, int);
      decimals_ = va_arg(args, int);
      break;
    case SQLType::FLOAT:
      decimals_ = va_arg(args, int);
      break;
    case SQLType::CHARACTER_VARYING:
    case SQLType::CHARACTER:
    case SQLType::BLOB:
      length_ = va_arg(args, int);
      break;
    default:
      g_assert_not_reached();
  }

  va_end(args);
}

Blob::~Blob()
{
}

string SQLObject::output() const
{
  if (is_null())
    return("NULL");
  
  switch (type_)
  {
    case VOID:
      return("VOID");
      
    case STRING:
      return("'" + value_ + "'");

    case TYPE:
    case INT:
      return(value_);

    case BLOB:
      throw SQLException("can't output BLOB objects");
      
    default:
      assert(0);
  }
}

bool SQLObject::input(const string& str)
{
  int c = str[0];
  
  set_null(false);
  
  if (c == '\'')
  {
    type_ = STRING;
    value_ = str.substr(0, str.length() - 1);
  }
  else if (isdigit(c) || c == '-' || c == '+')
  {
    type_ = INT;
    value_ = str;
  }
  else if (str == "NULL")
    set_null(true);
  else
    throw SQLException("invalid input for SQLObject");
  
  return(true);
}

string SQLObject::to_string() const
{
  return(value_);
}

long SQLObject::to_int() const
{
  switch (type_)
  {
    case STRING:
    case INT:
    {
      char *endptr;
      long l = strtol(value_.c_str(), &endptr, 10);
      if (*endptr == '\0')
        return l;
      else
	break;
    }
    case FLOAT:
    {
      char *endptr;
      double d = strtod(value_.c_str(), &endptr);
      if (*endptr == '\0')
        return (long)d;
      else
	break;
    }
    case TYPE:
    case BLOB:
    case VOID:
      break;
    default:
      g_assert_not_reached();
  }
  throw SQLException("object type mismatch");
}

double SQLObject::to_real() const
{
  switch (type_)
  {
    case STRING:
    case INT:
    {
      char *endptr;
      long l = strtol(value_.c_str(), &endptr, 10);
      if (*endptr == '\0')
        return l;
      else
	break;
    }
    case FLOAT:
    {
      char *endptr;
      double d = strtod(value_.c_str(), &endptr);
      if (*endptr == '\0')
        return d;
      else
	break;
    }
    case TYPE:
    case BLOB:
    case VOID:
      break;
    default:
      g_assert_not_reached();
  }
  throw SQLException("object type mismatch");
}

bool SQLObject::to_boolean() const
{
  switch (type_)
  {
    case STRING:
    {
      if (value_ == "1" || value_ == "t")
        return true;
      else
        return false;
    }
    case INT:
    {
      char *endptr;
      long l = strtol(value_.c_str(), &endptr, 10);
      if (*endptr == '\0')
        return l;
      else
	break;
    }
    case FLOAT:
    {
      char *endptr;
      double d = strtod(value_.c_str(), &endptr);
      if (*endptr == '\0')
        return (bool)d;
      else
	break;
    }
    case TYPE:
    case BLOB:
    case VOID:
      break;
    default:
      g_assert_not_reached();
  }
  throw SQLException("object type mismatch");
}


#if 0
SQLType basicSQLObject::to_type() const
{
  SQLType type;
  return(type);
}
#endif

Blob *SQLObject::to_blob() const
{
  throw SQLException("BLOBs not supported by driver");
}

bool SQLObject::from_string(const string& s)
{
  type_ = STRING;
  value_ = s;
  is_null_ = false;

  return(true);
}

bool SQLObject::from_int(long l)
{
  char buffer[64];

  sprintf(buffer, "%li", l);
  value_ = buffer;
  type_ = INT;
  is_null_ = false;

  return(true);
}

bool SQLObject::from_real(double d)
{
  char buffer[64];

  sprintf(buffer, "%f", d);
  value_ = buffer;
  type_ = FLOAT;
  is_null_ = false;

  return(true);
}

bool SQLObject::from_boolean(bool b)
{
  value_ = b ? "t" : "f";
  type_ = STRING;
  is_null_ = false;
  
  return true;
}

bool SQLObject::from_blob(const Blob *blob)
{
  throw SQLException("BLOBs not supported by driver");
}
  
inline void add_length_decimals(string& value, const GQL::SQLType& type)
{
  char buffer[64];
  sprintf(buffer, "(%i, ", type.length());
  value += buffer;
  sprintf(buffer, "%i)", type.decimals());
  value += buffer;
}

inline void add_length(string& value, const GQL::SQLType& type)
{
  char buffer[64];
  sprintf(buffer, "(%i)", type.length());
  value += buffer;
}

inline void add_decimals(string& value, const GQL::SQLType& type)
{
  char buffer[64];
  sprintf(buffer, "(%i)", type.decimals());
  value += buffer;
}

bool SQLObject::from_type(const SQLType& type)
{
  static const char *types[SQLType::MAX_TYPE] = { 0 };
  static bool initialized = false;
  
  if (!initialized)
  {
    types[SQLType::VOID] = "VOID";
    types[SQLType::BOOLEAN] = "BOOLEAN";
    types[SQLType::DATE] = "DATE";
    types[SQLType::DECIMAL] = "DECIMAL";
    types[SQLType::FLOAT] = "FLOAT";
    types[SQLType::SMALLINT] = "SMALLINT";
    types[SQLType::INTEGER] = "INTEGER";
    types[SQLType::INTERVAL] = "INTERVAL";
    types[SQLType::NUMERIC] = "NUMERIC";
    types[SQLType::TIME] = "TIME";
    types[SQLType::TIME_WITH_TZ] = "TIME WITH TIMEZONE";
    types[SQLType::TIMESTAMP_WITH_TZ] = "TIMESTAMP WITH TIMEZONE";
    types[SQLType::CHARACTER] = "CHARACTER";
    types[SQLType::CHARACTER_VARYING] = "CHARACTER VARYING";
    types[SQLType::BLOB] = "BLOB";
    
    initialized = true;
  }
 
  type_ = TYPE;
  value_ = types[type.typecode()];
  is_null_ = false;
  
  return true;
}

}
