/* Copyright (C) 1999-2000 Aaron Lehmann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */



#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include <stdio.h>
#include <stdlib.h>

#include "gtkplot.h"
#include "gtkplotlayout.h"
#include "gtksheet.h"
#include "gtkfontcombo.h"
#include "gtkpsfont.h"

#include "app.h"
#include "version.h"
#include "mathparsesym.h"
#include "mathparse.h"
#include "prefs.h"
#include "dialogs.h"
#include "about.h"
#include "data.h"


#include "xpm/data-diamond.xpm"
#include "xpm/data-square.xpm"
#include "xpm/data-plus.xpm"
#include "xpm/data-cross.xpm"
#include "xpm/data-star.xpm"
#include "xpm/data-triup.xpm"
#include "xpm/data-tridown.xpm"


/* Size of color preview rectangles in the prefs editor */
#define PREVIEW_SIZE_X 16
#define PREVIEW_SIZE_Y 16


struct cbk_record
{
    int type;
    GtkWidget *dlg, *text;
    GtkFontCombo *font_combo;
};

struct cbk_color
{
    GtkWidget *colorsel_toplevel;	    /* Color selection window */
    GdkColor color;			    /* The selected color */
    GtkWidget *colorbox;		    /* Widget receiving the selected color */
};

struct cbk_prf
{
    GtkWidget *legend, *showbottom, *showleft, *showtop, *showvgrid, *showhgrid;
    GtkWidget *trigunit;
    GtkWidget *calcmode_select, *calcmode_append, *errappend;

    struct cbk_color bg_color, grid_color, axes_color;
};

static struct cbk_prf *cbkprf = 0;

struct cbk_data
{
    GtkWidget *regrnone, *regrlin, *regrlog, *regrexp, *regrpow;
    GtkWidget *autoresize, *icondia, *iconsq, *iconcr, *iconpl, *iconst, *iconup, *icondown, *connectnone, *connectstraight, *connectspline;
    thing *its_thing;
    struct cbk_color symbol_color, line_color;
};

struct cbk_rename
{
    GtkWidget *dlg, *text;
    union
    {
	func_thing *function;
	thing *dataset;
    } things;
};




extern GtkWidget *active_plot;
extern GtkWidget *layout;
extern GtkWidget *opmenu;
extern gint nbrrealfuncs;


static void dialog_make_color_edit_line (GtkWidget *vbox, const char *text, struct cbk_color *cbkcol,
					 GdkColor *gdkcol);
static void dialog_set_preview_color (GtkWidget *w, GdkColor* col);
static void dialog_label_new_font(GtkFontCombo *font_combo, gpointer data);




void pref_destroy (GtkWidget *w, gpointer data)
{
    struct cbk_prf **cbkstruct_p = data;
    struct cbk_prf *cbkstruct = *cbkstruct_p;

    if (cbkstruct->bg_color.colorsel_toplevel)
	gtk_widget_destroy (GTK_WIDGET(cbkstruct->bg_color.colorsel_toplevel));
    if (cbkstruct->grid_color.colorsel_toplevel)
	gtk_widget_destroy (GTK_WIDGET(cbkstruct->grid_color.colorsel_toplevel));
    if (cbkstruct->axes_color.colorsel_toplevel)
	gtk_widget_destroy (GTK_WIDGET(cbkstruct->axes_color.colorsel_toplevel));

    free (cbkstruct);
    *cbkstruct_p = 0;
}

void dialog_cancel (GtkWidget *w, gpointer data)
{
    /* Destroy a widget, make the global that points to it = 0 */
    /* You must pass a pointer to a pointer to this function */
    GtkWidget **wid = data;
    GtkWidget *widg = *wid;
    gtk_widget_destroy (GTK_WIDGET(widg));
    *wid = 0;
}


void dialog_cancel_simple (GtkWidget *w, gpointer data)
{
    gtk_widget_destroy (GTK_WIDGET(data));
}


void dialog_prefs (void)
{
    GtkWidget *prefdlg, *button, *chkbutton, *frame, *vbox1, *label, *notebook;
    GSList *group;
    /*int i;*/
    struct cbk_prf *cbkstruct;

    if (cbkprf) /* Return if prefs dialog already is open */
    {
	return;
    }
    
    cbkstruct = (struct cbk_prf *) malloc (sizeof(struct cbk_prf));
    cbkprf = cbkstruct;
    
    prefdlg = gtk_dialog_new ();
    gtk_window_set_title(GTK_WINDOW(prefdlg), _("GtkGraph Preferences"));
    gtk_container_set_border_width (GTK_CONTAINER(GTK_DIALOG(prefdlg)->vbox), 5);
    
    notebook = gtk_notebook_new ();
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(prefdlg)->vbox), notebook, FALSE, FALSE, 0);
    gtk_widget_show (notebook);
    
    /* Create a Frame */
    frame = gtk_frame_new(NULL);
    /* Set the frames label */
    gtk_frame_set_label( GTK_FRAME(frame), _("Toggles") );
    
    /* Align the label at the right of the frame */
    gtk_frame_set_label_align( GTK_FRAME(frame), 1.0, 0.0);
    
    /* Set the style of the frame */
    gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
    gtk_widget_show(frame);
    label = gtk_label_new (_("Toggles"));

    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
    gtk_widget_show (vbox1);

    chkbutton = gtk_check_button_new_with_label (_("Show legend"));
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), prf.showlegend);
    cbkstruct->legend = chkbutton;
    gtk_widget_show (chkbutton);
    
    chkbutton = gtk_check_button_new_with_label (_("Show X axis title"));
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), prf.bottomtitle);
    cbkstruct->showbottom = chkbutton;
    gtk_widget_show (chkbutton);
    
    chkbutton = gtk_check_button_new_with_label (_("Show Y axis title"));
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), prf.lefttitle);
    cbkstruct->showleft = chkbutton;
    gtk_widget_show (chkbutton);
    
    chkbutton = gtk_check_button_new_with_label (_("Show plot title"));
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), prf.toptitle);
    cbkstruct->showtop = chkbutton;
    gtk_widget_show (chkbutton);
    
    chkbutton = gtk_check_button_new_with_label (_("Show vertical grid"));
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), prf.showvgrid);
    cbkstruct->showvgrid = chkbutton;
    gtk_widget_show (chkbutton);
    
    chkbutton = gtk_check_button_new_with_label (_("Show horizontal grid"));
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), prf.showhgrid);
    cbkstruct->showhgrid = chkbutton;
    gtk_widget_show (chkbutton);
    
    
    /* Create a Frame */
    frame = gtk_frame_new(NULL);
    /* Set the frames label */
    gtk_frame_set_label( GTK_FRAME(frame), _("Trig. unit") );
    
    /* Align the label at the right of the frame */
    gtk_frame_set_label_align( GTK_FRAME(frame), 1.0, 0.0);
    
    /* Set the style of the frame */
    gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
    gtk_widget_show(frame);
    
    label = gtk_label_new (_("Units"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    

    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
    gtk_widget_show (vbox1);
    
    button = gtk_radio_button_new_with_label (NULL, _("Radians"));
    cbkstruct->trigunit = button;
    gtk_box_pack_start (GTK_BOX(vbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    
    group = gtk_radio_button_group (GTK_RADIO_BUTTON(button));
    if (prf.trigunit == MP_UNIT_RADIAN) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    button = gtk_radio_button_new_with_label(group, _("Degrees"));
    if (prf.trigunit == MP_UNIT_DEGREE) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_box_pack_start (GTK_BOX(vbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);

    /* Create a Frame */
    frame = gtk_frame_new(NULL);
    /* Set the frames label */
    gtk_frame_set_label( GTK_FRAME(frame), _("Calculator Edit Mode") );
    
    /* Align the label at the right of the frame */
    gtk_frame_set_label_align( GTK_FRAME(frame), 1.0, 0.0);
    
    /* Set the style of the frame */
    gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
    gtk_widget_show(frame);
    
    label = gtk_label_new (_("Calculator"));
    
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    
    
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
    gtk_widget_show (vbox1);
  

    button = gtk_radio_button_new_with_label (NULL, _("Select"));
    cbkstruct->calcmode_select = button;
    gtk_box_pack_start (GTK_BOX(vbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);


    if (prf.calcmode == CALCMODE_SELECT) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    
    button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(button), _("Append"));
    if (prf.calcmode == CALCMODE_APPEND) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    cbkstruct->calcmode_append = button;
    gtk_box_pack_start (GTK_BOX(vbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(button), _("Clear"));
    if (prf.calcmode == CALCMODE_CLEAR) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_box_pack_start (GTK_BOX(vbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    chkbutton = gtk_check_button_new_with_label (_("Always append on error"));
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), prf.errappend);
    cbkstruct->errappend = chkbutton;
    gtk_widget_show (chkbutton);
    
    /* Create a Frame */
    frame = gtk_frame_new(NULL);
    /* Set the frames label */
    gtk_frame_set_label( GTK_FRAME(frame), _("Colors") );
    
    /* Align the label at the right of the frame */
    gtk_frame_set_label_align( GTK_FRAME(frame), 1.0, 0.0);
    
    /* Set the style of the frame */
    gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
    gtk_widget_show(frame);
    label = gtk_label_new (_("Colors"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    
    
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
    gtk_widget_show (vbox1);
    

    dialog_make_color_edit_line (vbox1, _("Axes"), &cbkstruct->axes_color, &prf.axiscolor);
    dialog_make_color_edit_line (vbox1, _("Background"), &cbkstruct->bg_color, &prf.bgcolor);
    dialog_make_color_edit_line (vbox1, _("Grid"), &cbkstruct->grid_color, &prf.gridcolor);

#if 0
    for (i=0; i != MAX_FUNC; ++i)
    {
	char labname[32];
	sprintf (labname, "%s %i:", _("Function"), i+1);
	dialog_make_color_edit_line (vbox1, labname, ???, ???);
    }
#endif
    
    button = gtk_button_new_with_label (_("Save"));
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(prefdlg)->action_area), button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_grab_focus (button);
    gtk_widget_grab_default (button);
    gtk_widget_show (button);
    gtk_signal_connect_after (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_cancel_simple), (gpointer) prefdlg);
    gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_prefs_ok), cbkstruct);

    button = gtk_button_new_with_label (_("Cancel"));
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(prefdlg)->action_area), button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_show (button);
    gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_cancel_simple), (gpointer) prefdlg);

    gtk_signal_connect (GTK_OBJECT (prefdlg), "destroy", GTK_SIGNAL_FUNC (pref_destroy), &cbkprf);
    
    gtk_widget_show (prefdlg);
}

static void
dialog_make_color_edit_line (GtkWidget *vbox, const char *text, struct cbk_color *cbkcol, GdkColor *gdkcol)
{
    GtkWidget *hbox1, *label, *button, *preview;
    char labname[20];

    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);

    sprintf (labname, "%s:", text);
    label = gtk_label_new (labname);
    gtk_box_pack_start (GTK_BOX(hbox1), label, FALSE, FALSE, 0);
    gtk_widget_show (label);

    button = gtk_button_new ();
    gtk_box_pack_end (GTK_BOX(hbox1), button, FALSE, FALSE, 0);
    gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_prefs_color), cbkcol);
    gtk_widget_show (button);

    preview = gtk_preview_new (GTK_PREVIEW_COLOR);
    gtk_preview_size (GTK_PREVIEW (preview), PREVIEW_SIZE_X, PREVIEW_SIZE_Y);
    gtk_preview_set_expand (GTK_PREVIEW (preview), FALSE);
    gtk_container_add (GTK_CONTAINER (button), preview);
    gtk_widget_show (preview);

    cbkcol->colorsel_toplevel = 0;
    cbkcol->color = *gdkcol;
    cbkcol->colorbox = preview;

    dialog_set_preview_color(cbkcol->colorbox, &cbkcol->color);
}

void dialog_prefs_ok (GtkWidget *w, gpointer data)
{
    struct cbk_prf *cbkstruct = data;

    prf.showlegend = GTK_TOGGLE_BUTTON(cbkstruct->legend)->active;
    prf.lefttitle = GTK_TOGGLE_BUTTON(cbkstruct->showleft)->active;
    prf.bottomtitle = GTK_TOGGLE_BUTTON(cbkstruct->showbottom)->active;
    prf.toptitle = GTK_TOGGLE_BUTTON(cbkstruct->showtop)->active;
    prf.showvgrid = GTK_TOGGLE_BUTTON(cbkstruct->showvgrid)->active;
    prf.showhgrid = GTK_TOGGLE_BUTTON(cbkstruct->showhgrid)->active;
    
    if (GTK_TOGGLE_BUTTON(cbkstruct->trigunit)->active)
    {
	prf.trigunit = MP_UNIT_RADIAN;
    }
    else
    {
	prf.trigunit = MP_UNIT_DEGREE;
    }
    
    prf.errappend = ( GTK_TOGGLE_BUTTON(cbkstruct->errappend)->active);
    
    if (GTK_TOGGLE_BUTTON(cbkstruct->calcmode_append)->active)
    {
	prf.calcmode = CALCMODE_APPEND;
    }
    
    else if (GTK_TOGGLE_BUTTON(cbkstruct->calcmode_select)->active)
    {
	prf.calcmode = CALCMODE_SELECT;
    }
    
    else
    {
	prf.calcmode = CALCMODE_CLEAR;
    }

    prf.bgcolor = cbkstruct->bg_color.color;
    prf.gridcolor = cbkstruct->grid_color.color;
    prf.axiscolor = cbkstruct->axes_color.color;

    gdk_colormap_free_colors (gdk_colormap_get_system(), &prf.bgcolor, 1);
    gdk_color_alloc (gdk_colormap_get_system(), &prf.bgcolor);
    
    gdk_colormap_free_colors (gdk_colormap_get_system(), &prf.gridcolor, 1);
    gdk_color_alloc (gdk_colormap_get_system(), &prf.gridcolor);
    
    gdk_colormap_free_colors (gdk_colormap_get_system(), &prf.axiscolor, 1);
    gdk_color_alloc (gdk_colormap_get_system(), &prf.axiscolor);

    reset_flags ();
    prefs_save ();
}


void dialog_prefs_color (GtkWidget *w, gpointer data)
{
    GtkWidget *colorsel;
    struct cbk_color *p = data;
    gdouble color[3];

    if (p->colorsel_toplevel) /* Return if this dialog is already open */
    {
	return;
    }

    colorsel = gtk_color_selection_dialog_new (_("Color Selection"));
    p->colorsel_toplevel = colorsel;

    color[0] = p->color.red / 65535.0;
    color[1] = p->color.green / 65535.0;
    color[2] = p->color.blue / 65535.0;

    gtk_color_selection_set_color (GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(colorsel)->colorsel),
				   &color[0]);
    gtk_widget_destroy (GTK_COLOR_SELECTION_DIALOG(colorsel)->help_button);
    gtk_signal_connect (GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG(colorsel)->ok_button), "clicked", GTK_SIGNAL_FUNC (dialog_prefs_color_ok), p);
    gtk_signal_connect_after (GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG(colorsel)->ok_button), "clicked", GTK_SIGNAL_FUNC (dialog_cancel), (gpointer) &p->colorsel_toplevel);
    gtk_signal_connect (GTK_OBJECT (GTK_COLOR_SELECTION_DIALOG(colorsel)->cancel_button), "clicked", GTK_SIGNAL_FUNC (dialog_cancel), (gpointer) &p->colorsel_toplevel);
    gtk_signal_connect (GTK_OBJECT (colorsel), "destroy", GTK_SIGNAL_FUNC (dialog_cancel), (gpointer) &p->colorsel_toplevel);

    gtk_widget_show (colorsel);
}


void dialog_prefs_color_ok (GtkWidget *w, gpointer data)
{
    gdouble color[3];
    struct cbk_color *p = data;

    GtkColorSelection *colsel = GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(p->colorsel_toplevel)->colorsel);
    gtk_color_selection_get_color (colsel, &color[0]);
    p->color.red = (guint16)(color[0]*65535.0);
    p->color.green = (guint16)(color[1]*65535.0);
    p->color.blue = (guint16)(color[2]*65535.0);
    dialog_set_preview_color (p->colorbox, &p->color);

#if 0
/* Will need to be fixed in the future */
    if (p->num < 10)
    {
	gtk_plot_dataset_set_line_attributes(dataset[p->num], GTK_PLOT_LINE_SOLID, 0, prf.color[p->num]);
    }
#endif
}

static void dialog_set_preview_color (GtkWidget *w, GdkColor* col)
{
    int xsize = PREVIEW_SIZE_X;
    int ysize = PREVIEW_SIZE_Y;
    guchar *sample_buf = g_malloc (3 * xsize * sizeof(guchar));
    int x, y;

    for (y = 0; y != ysize; ++y)
    {
	int i = 0;
	for (x = 0; x != xsize; ++x)
	{
	    sample_buf[i++] = col->red / 65535.0 * 255.0;
	    sample_buf[i++] = col->green / 65535.0 * 255.0;
	    sample_buf[i++] = col->blue / 65535.0 * 255.0;
	}
	gtk_preview_draw_row (GTK_PREVIEW (w), sample_buf, 0, y, xsize);
    }
    gtk_widget_queue_draw (w);
    g_free (sample_buf);
}

void dialog_about (void)
{
    gtk_widget_show (create_frmAbout());
}


void dialog_label (int type, char *name, char *curr)
{
    GtkWidget *dialog, *button, *hbox1, *new, *label, *font_combo;
    struct cbk_record *rcd;
    rcd = (struct cbk_record *) malloc (sizeof(struct cbk_record));
    
    rcd->type = type;
    
    
    dialog = gtk_dialog_new ();
    gtk_window_set_title(GTK_WINDOW(dialog), _("Set Title"));
    
    rcd->dlg = dialog;
    
    font_combo = gtk_font_combo_new();
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),font_combo, FALSE, FALSE, 0);
    rcd->font_combo = GTK_FONT_COMBO(font_combo);
    gtk_widget_show(font_combo);

    hbox1=gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);

    label = gtk_label_new (name);
    gtk_box_pack_start (GTK_BOX(hbox1), label, FALSE, FALSE, 0);
    gtk_widget_show (label);

    new = gtk_entry_new_with_max_length(50);
    gtk_entry_set_text (GTK_ENTRY(new), curr);
    gtk_editable_select_region (GTK_EDITABLE(new), 0, -1); /* Select field */
    gtk_box_pack_start (GTK_BOX(hbox1), new, FALSE, FALSE, 0);
    gtk_widget_grab_focus (new);
    gtk_signal_connect (GTK_OBJECT(new), "activate", dialog_label_ok, rcd);
    gtk_widget_show (new);
    rcd->text = new;
    
    dialog_label_new_font (GTK_FONT_COMBO(font_combo), new);
    gtk_signal_connect (GTK_OBJECT(font_combo), "changed", GTK_SIGNAL_FUNC(dialog_label_new_font), new);
    
    button = gtk_button_new_with_label (_("OK"));
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_label_ok), rcd);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (button);
    gtk_widget_show (button);
    
    button = gtk_button_new_with_label (_("Cancel"));
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_show (button);
    gtk_signal_connect_after (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC(dialog_cancel_simple), (gpointer)dialog);
    gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC(dialog_label_cleanup), (gpointer)rcd);
    
    gtk_widget_show (dialog);
}


static void dialog_label_new_font (GtkFontCombo *font_combo, gpointer data)
{
    GtkWidget *entry;
    GtkStyle *style;
    
    entry = GTK_WIDGET(data);
    

    style = gtk_style_new ();
    gdk_font_unref (style->font);
    style->font = font_combo->font;
    gdk_font_ref (style->font);
    
    gtk_widget_set_style (entry, style);
    gtk_style_unref(style);
}


void dialog_label_ok (GtkWidget *w, gpointer data)
{
    struct cbk_record *datas = (struct cbk_record *) data;
    
    GtkPSFont *psfont= datas->font_combo->psfont;
    char *name = psfont->fontname;
    gint height = datas->font_combo->height;
    GdkColor white;
    GdkColor black;
    
    gdk_color_parse ("black", &black);
    gdk_color_alloc (gdk_colormap_get_system(), &black);

    gdk_color_parse ("white", &white);
    gdk_color_alloc (gdk_colormap_get_system(), &white);

    switch (datas->type)
    {
    case DIALOG_LABEL_X:
    {
	gtk_plot_axis_set_title (GTK_PLOT(active_plot), GTK_PLOT_AXIS_BOTTOM, gtk_entry_get_text (GTK_ENTRY(datas->text)));
	gtk_plot_axis_move_title (GTK_PLOT(active_plot), GTK_PLOT_AXIS_BOTTOM, 0, 0.5, GTK_PLOT(active_plot)->bottom.title.y);
	gtk_plot_axis_labels_set_attributes (GTK_PLOT(active_plot), GTK_PLOT_AXIS_BOTTOM, name, height, black, white);
	break;
    }
    case DIALOG_LABEL_Y:
    {
	gtk_plot_axis_set_title (GTK_PLOT(active_plot), GTK_PLOT_AXIS_LEFT, gtk_entry_get_text (GTK_ENTRY(datas->text)));
	gtk_plot_axis_move_title (GTK_PLOT(active_plot), GTK_PLOT_AXIS_LEFT, 0, GTK_PLOT(active_plot)->left.title.x, 0.5);
	gtk_plot_axis_labels_set_attributes (GTK_PLOT(active_plot), GTK_PLOT_AXIS_LEFT, name, height, black, white);
	break;
    }
    case DIALOG_LABEL_TITLE:
    {
	gtk_plot_axis_set_title (GTK_PLOT(active_plot), GTK_PLOT_AXIS_TOP, gtk_entry_get_text (GTK_ENTRY(datas->text)));
	gtk_plot_axis_move_title (GTK_PLOT(active_plot), GTK_PLOT_AXIS_TOP, 0, 0.5, GTK_PLOT(active_plot)->top.title.y);
	gtk_plot_axis_labels_set_attributes (GTK_PLOT(active_plot), GTK_PLOT_AXIS_TOP, name, height, black, white);
	break;
    }
    }
    

    gtk_widget_queue_draw (layout);
    
    gtk_widget_destroy (datas->dlg);
    free (datas);
}


void dialog_label_cleanup (GtkWidget *w, gpointer data)
{
    struct cbk_record *in = (struct cbk_record *) data;
    free (in);
}


/* Rename a dataset or function */
void dialog_rename (gpointer thingie, int type)
{
    GtkWidget *dialog, *button, *hbox1, *new, *label;
    struct cbk_rename *rcd;
    rcd = (struct cbk_rename *) malloc (sizeof(struct cbk_rename));
    
    switch (type)
    {
    case 0: /* Function */
	rcd->things.function = thingie;
	break;
    case 1:
	rcd->things.dataset = thingie;
	break;
    }
    
    dialog = gtk_dialog_new ();
    gtk_window_set_title(GTK_WINDOW(dialog), _("Rename"));
    
    rcd->dlg = dialog;
    
    hbox1=gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);

    label = gtk_label_new ("Rename to: ");
    gtk_box_pack_start (GTK_BOX(hbox1), label, FALSE, FALSE, 0);
    gtk_widget_show (label);

    new = gtk_entry_new_with_max_length(50);
    if (type == 0 && rcd->things.function->name)
	gtk_entry_set_text (GTK_ENTRY(new), rcd->things.function->name);
    else if (type == 1 && rcd->things.dataset->name)
	gtk_entry_set_text (GTK_ENTRY(new), rcd->things.dataset->name);
    gtk_editable_select_region (GTK_EDITABLE(new), 0, -1); /* Select field */
    gtk_box_pack_start (GTK_BOX(hbox1), new, FALSE, FALSE, 0);
    gtk_widget_grab_focus (new);
    gtk_widget_show (new);
    rcd->text = new;
    
    button = gtk_button_new_with_label (_("OK"));
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
    if (type == 0)
    {
	gtk_signal_connect (GTK_OBJECT(button), "clicked", dialog_rename_function_ok, rcd);
	gtk_signal_connect (GTK_OBJECT(new), "activate", dialog_rename_function_ok, rcd);
    }
    else if (type == 1)
    {
	gtk_signal_connect (GTK_OBJECT(button), "clicked", dialog_rename_dataset_ok, rcd);
	gtk_signal_connect (GTK_OBJECT(new), "activate", dialog_rename_dataset_ok, rcd);
    }
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (button);
    gtk_widget_show (button);
    
    button = gtk_button_new_with_label (_("Cancel"));
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_show (button);
    gtk_signal_connect_after (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC(dialog_cancel_simple), (gpointer)dialog);
    gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC(dialog_rename_cleanup), (gpointer)rcd);
    gtk_widget_show (dialog);
}


void dialog_rename_function_ok (GtkWidget *w, gpointer data)
{
    struct cbk_rename *in = (struct cbk_rename *) data;
    free (in->things.function->name);
    in->things.function->name = gtk_editable_get_chars (GTK_EDITABLE(in->text), 0, -1);
    gtk_label_set_text (GTK_LABEL(in->things.function->label), in->things.function->name);
    /* Change the name twice 'cause gtk doesn't want to refresh it if it's the current selection */
    gtk_option_menu_set_history (GTK_OPTION_MENU(opmenu), nbrrealfuncs+3);
    gtk_option_menu_set_history (GTK_OPTION_MENU(opmenu), in->things.function->index);
    gtk_plot_dataset_set_legend (in->things.function->real_dataset, in->things.function->name);
    gtk_widget_queue_draw (active_plot);
    gtk_widget_destroy (in->dlg);
    free (in);
}


void dialog_rename_dataset_ok (GtkWidget *w, gpointer data)
{
    struct cbk_rename *in = (struct cbk_rename *) data;
    char *regname;
    free (in->things.dataset->name);
    in->things.dataset->name = gtk_editable_get_chars (GTK_EDITABLE(in->text), 0, -1);
    gtk_label_set_text (GTK_LABEL(in->things.dataset->label), in->things.dataset->name);
    gtk_plot_dataset_set_legend(in->things.dataset->real_dataset, in->things.dataset->name);
    regname = (char *) malloc (strlen(in->things.dataset->name)+strlen(_("curve fit"))+4);
    sprintf (regname, "%s (%s)", in->things.dataset->name, _("curve fit"));
    gtk_plot_dataset_set_legend (in->things.dataset->reg_dataset, regname);
    free (regname);
    gtk_widget_queue_draw (active_plot);
    gtk_widget_destroy (in->dlg);
    free (in);
}


void dialog_rename_cleanup (GtkWidget *w, gpointer data)
{
    struct cbk_rename *in = (struct cbk_rename *) data;
    free (in);
}


void dialog_solve (void)
{
    /* nothing here yet */
}


void dialog_data (GtkWidget *w, gpointer data)
{
    GtkWidget *datadlg, *button, *chkbutton, *frame, *vbox1, *hbox1, *label, *notebook, *pixmapwid, *entry;
    GdkPixmap *pixmap;
    GdkBitmap *mask;
    GtkStyle *style;
    gint symbol;
    
    struct cbk_data *cbkstruct;
    
    
    cbkstruct = (struct cbk_data *) malloc (sizeof(struct cbk_data));
    
    cbkstruct->its_thing = (thing *) data;
    symbol = cbkstruct->its_thing->real_dataset->symbol.symbol_type;

    if (cbkstruct->its_thing->dlg_data) /* Return if dialog is already open */
    {
	return;
    }

    datadlg = gtk_dialog_new ();
    cbkstruct->its_thing->dlg_data = datadlg;
    gtk_window_set_title(GTK_WINDOW(datadlg), _("Dataset settings"));
    gtk_container_set_border_width (GTK_CONTAINER(GTK_DIALOG(datadlg)->vbox), 5);
    gtk_widget_realize (datadlg);
    
    notebook = gtk_notebook_new ();
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(datadlg)->vbox), notebook, FALSE, FALSE, 0);
    gtk_widget_show (notebook);
    
    /* Create a Frame */
    frame = gtk_frame_new(NULL);
    /* Set the frames label */
    gtk_frame_set_label( GTK_FRAME(frame), _("Regression") );
    
    /* Align the label at the right of the frame */
    gtk_frame_set_label_align( GTK_FRAME(frame), 1.0, 0.0);
    
    /* Set the style of the frame */
    gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
    gtk_widget_show(frame);
    label = gtk_label_new (_("Regression"));

    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
    gtk_widget_show (vbox1);

    chkbutton = gtk_radio_button_new_with_label (NULL, _("None"));
    cbkstruct->regrnone = chkbutton;
    if (cbkstruct->its_thing->regrtype == REGRESSION_NONE)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), TRUE);
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_widget_show (chkbutton);

    chkbutton = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(chkbutton), _("Linear, y=M*x+B"));
    cbkstruct->regrlin = chkbutton;
    if (cbkstruct->its_thing->regrtype == REGRESSION_LINEAR)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), TRUE);
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_widget_show (chkbutton);

    chkbutton = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(chkbutton), _("Logarithmic, y=M*log(x)+B"));
    cbkstruct->regrlog = chkbutton;
    if (cbkstruct->its_thing->regrtype == REGRESSION_LOGARITHMIC)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), TRUE);
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_widget_show (chkbutton);

    chkbutton = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(chkbutton), _("Exponential, y=B*exp(M*x)"));
    cbkstruct->regrexp = chkbutton;
    if (cbkstruct->its_thing->regrtype == REGRESSION_EXPONENTIAL)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), TRUE);
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_widget_show (chkbutton);

    chkbutton = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(chkbutton), _("Power, y=B*x^M"));
    cbkstruct->regrpow = chkbutton;
    if (cbkstruct->its_thing->regrtype == REGRESSION_POWER)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), TRUE);
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_widget_show (chkbutton);

    
    /* Create a Frame */
    frame = gtk_frame_new(NULL);
    /* Set the frames label */
    gtk_frame_set_label( GTK_FRAME(frame), _("Icon") );
    
    /* Align the label at the right of the frame */
    gtk_frame_set_label_align( GTK_FRAME(frame), 1.0, 0.0);
    
    /* Set the style of the frame */
    gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
    gtk_widget_show(frame);
    
    label = gtk_label_new (_("Icon"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    
    
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
    gtk_widget_show (vbox1);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);

    style = gtk_widget_get_style (datadlg);
    
    pixmap = gdk_pixmap_create_from_xpm_d (datadlg->window, &mask, &style->bg[GTK_STATE_NORMAL], (gchar **)data_diamond_xpm);
    pixmapwid = gtk_pixmap_new (pixmap, mask);
    gtk_box_pack_end(GTK_BOX(hbox1), pixmapwid, FALSE, FALSE, 0);
    gtk_widget_show (pixmapwid);
    
    button = gtk_radio_button_new_with_label (NULL, _("Diamond"));
    cbkstruct->icondia = button;
    if (symbol == GTK_PLOT_SYMBOL_DIAMOND) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_box_pack_start (GTK_BOX(hbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);
    
    pixmap = gdk_pixmap_create_from_xpm_d (datadlg->window, &mask, &style->bg[GTK_STATE_NORMAL], (gchar **)data_square_xpm);
    pixmapwid = gtk_pixmap_new (pixmap, mask);
    gtk_box_pack_end(GTK_BOX(hbox1), pixmapwid, FALSE, FALSE, 0);
    gtk_widget_show (pixmapwid);
    
    button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(button), _("Square"));
    cbkstruct->iconsq = button;
    if (symbol == GTK_PLOT_SYMBOL_SQUARE) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_box_pack_start (GTK_BOX(hbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);
    
    pixmap = gdk_pixmap_create_from_xpm_d (datadlg->window, &mask, &style->bg[GTK_STATE_NORMAL], (gchar **)data_plus_xpm);
    pixmapwid = gtk_pixmap_new (pixmap, mask);
    gtk_box_pack_end(GTK_BOX(hbox1), pixmapwid, FALSE, FALSE, 0);
    gtk_widget_show (pixmapwid);
    
    button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(button), _("Plus"));
    cbkstruct->iconpl = button;
    if (symbol == GTK_PLOT_SYMBOL_PLUS) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_box_pack_start (GTK_BOX(hbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);
    
    pixmap = gdk_pixmap_create_from_xpm_d (datadlg->window, &mask, &style->bg[GTK_STATE_NORMAL], (gchar **)data_cross_xpm);
    pixmapwid = gtk_pixmap_new (pixmap, mask);
    gtk_box_pack_end(GTK_BOX(hbox1), pixmapwid, FALSE, FALSE, 0);
    gtk_widget_show (pixmapwid);
    
    button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(button), _("Cross"));
    cbkstruct->iconcr = button;
    if (symbol == GTK_PLOT_SYMBOL_CROSS) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_box_pack_start (GTK_BOX(hbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);
    
    pixmap = gdk_pixmap_create_from_xpm_d (datadlg->window, &mask, &style->bg[GTK_STATE_NORMAL], (gchar **)data_star_xpm);
    pixmapwid = gtk_pixmap_new (pixmap, mask);
    gtk_box_pack_end(GTK_BOX(hbox1), pixmapwid, FALSE, FALSE, 0);
    gtk_widget_show (pixmapwid);
    
    button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(button), _("Star"));
    cbkstruct->iconst = button;
    if (symbol == GTK_PLOT_SYMBOL_STAR) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_box_pack_start (GTK_BOX(hbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);
    
    pixmap = gdk_pixmap_create_from_xpm_d (datadlg->window, &mask, &style->bg[GTK_STATE_NORMAL], (gchar **)data_triup_xpm);
    pixmapwid = gtk_pixmap_new (pixmap, mask);
    gtk_box_pack_end(GTK_BOX(hbox1), pixmapwid, FALSE, FALSE, 0);
    gtk_widget_show (pixmapwid);
    
    button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(button), _("Upward triangle"));
    cbkstruct->iconup = button;
    if (symbol == GTK_PLOT_SYMBOL_UP_TRIANGLE) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_box_pack_start (GTK_BOX(hbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);
    
    pixmap = gdk_pixmap_create_from_xpm_d (datadlg->window, &mask, &style->bg[GTK_STATE_NORMAL], (gchar **)data_tridown_xpm);
    pixmapwid = gtk_pixmap_new (pixmap, mask);
    gtk_box_pack_end(GTK_BOX(hbox1), pixmapwid, FALSE, FALSE, 0);
    gtk_widget_show (pixmapwid);
    
    button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(button), _("Downward triangle"));
    cbkstruct->icondown = button;
    if (symbol == GTK_PLOT_SYMBOL_DOWN_TRIANGLE) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_box_pack_start (GTK_BOX(hbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    
    
    /* Create a Frame */
    frame = gtk_frame_new(NULL);
    /* Set the frames label */
    gtk_frame_set_label( GTK_FRAME(frame), _("X Coordinate") );
    
    /* Align the label at the right of the frame */
    gtk_frame_set_label_align( GTK_FRAME(frame), 1.0, 0.0);
    
    /* Set the style of the frame */
    gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
    gtk_widget_show(frame);
    
    label = gtk_label_new (_("X"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
    gtk_widget_show (vbox1);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);
    
    chkbutton = gtk_check_button_new_with_label (_("Significant figures"));
    gtk_box_pack_start (GTK_BOX(hbox1), chkbutton, FALSE, FALSE, 0);
    gtk_widget_show (chkbutton);
    
    entry = gtk_entry_new ();
    gtk_box_pack_end (GTK_BOX(hbox1), entry, FALSE, FALSE, 0);
    gtk_widget_set_usize (entry, 30, 20);
    gtk_widget_show (entry);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);
    
    label = gtk_label_new (_("Unit:"));
    gtk_box_pack_start (GTK_BOX(hbox1), label, FALSE, FALSE, 0);
    gtk_widget_show (label);
    
    entry = gtk_entry_new ();
    gtk_box_pack_end (GTK_BOX(hbox1), entry, FALSE, FALSE, 0);
    gtk_widget_set_usize (entry, 90, 20);
    gtk_widget_show (entry);
    
    
    
    /* Create a Frame */
    frame = gtk_frame_new(NULL);
    /* Set the frames label */
    gtk_frame_set_label( GTK_FRAME(frame), _("Y Coordinate") );
    
    /* Align the label at the right of the frame */
    gtk_frame_set_label_align( GTK_FRAME(frame), 1.0, 0.0);
    
    /* Set the style of the frame */
    gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
    gtk_widget_show(frame);
    
    label = gtk_label_new (_("Y"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
    gtk_widget_show (vbox1);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);
    
    chkbutton = gtk_check_button_new_with_label (_("Significant figures"));
    gtk_box_pack_start (GTK_BOX(hbox1), chkbutton, FALSE, FALSE, 0);
    gtk_widget_show (chkbutton);
    
    entry = gtk_entry_new ();
    gtk_box_pack_end (GTK_BOX(hbox1), entry, FALSE, FALSE, 0);
    gtk_widget_set_usize (entry, 30, 20);
    gtk_widget_show (entry);
    
    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX(vbox1), hbox1, FALSE, FALSE, 0);
    gtk_widget_show (hbox1);
    
    label = gtk_label_new (_("Unit:"));
    gtk_box_pack_start (GTK_BOX(hbox1), label, FALSE, FALSE, 0);
    gtk_widget_show (label);
    
    entry = gtk_entry_new ();
    gtk_box_pack_end (GTK_BOX(hbox1), entry, FALSE, FALSE, 0);
    gtk_widget_set_usize (entry, 90, 20);
    gtk_widget_show (entry);
    
    
    /* Create a Frame */
    frame = gtk_frame_new(NULL);
    /* Set the frames label */
    gtk_frame_set_label( GTK_FRAME(frame), _("Miscellaneous") );
    
    /* Align the label at the right of the frame */
    gtk_frame_set_label_align( GTK_FRAME(frame), 1.0, 0.0);
    
    /* Set the style of the frame */
    gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
    gtk_widget_show(frame);
    
    label = gtk_label_new (_("Miscellaneous"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
    gtk_widget_show (vbox1);
    
    chkbutton = gtk_check_button_new_with_label (_("Resize plot automatically to fit points"));
    cbkstruct->autoresize = chkbutton;
    if (cbkstruct->its_thing->autoresize)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(chkbutton), TRUE);
    gtk_box_pack_start (GTK_BOX(vbox1), chkbutton, FALSE, FALSE, 0);
    gtk_widget_show (chkbutton);
    
    button = gtk_radio_button_new_with_label (NULL, _("Don't connect points"));
    cbkstruct->connectnone = button;
    gtk_box_pack_start (GTK_BOX(vbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);


    if (cbkstruct->its_thing->connect == GTK_PLOT_CONNECT_NONE) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    
    button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(button), _("Connect points with lines"));
    if (cbkstruct->its_thing->connect == GTK_PLOT_CONNECT_STRAIGHT)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    cbkstruct->connectstraight = button;
    gtk_box_pack_start (GTK_BOX(vbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    
    button = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON(button), _("Connect points with splines"));
    if (cbkstruct->its_thing->connect == GTK_PLOT_CONNECT_SPLINE)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);
    cbkstruct->connectspline = button;
    gtk_box_pack_start (GTK_BOX(vbox1), button, FALSE, FALSE, 0);
    gtk_widget_show (button);

    
    
    /* Create a Frame */
    frame = gtk_frame_new(NULL);
    /* Set the frames label */
    gtk_frame_set_label( GTK_FRAME(frame), _("Colors") );
    
    /* Align the label at the right of the frame */
    gtk_frame_set_label_align( GTK_FRAME(frame), 1.0, 0.0);
    
    /* Set the style of the frame */
    gtk_frame_set_shadow_type( GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
    gtk_widget_show(frame);
    
    label = gtk_label_new (_("Colors"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    
    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), vbox1);
    gtk_container_set_border_width (GTK_CONTAINER(vbox1), 5);
    gtk_widget_show (vbox1);


    dialog_make_color_edit_line (vbox1, _("Points"), &cbkstruct->symbol_color, &cbkstruct->its_thing->symbol_color);
    dialog_make_color_edit_line (vbox1, _("Curve fit"), &cbkstruct->line_color, &cbkstruct->its_thing->line_color);

    button = gtk_button_new_with_label (_("OK"));
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(datadlg)->action_area), button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_grab_focus (button);
    gtk_widget_grab_default (button);
    gtk_widget_show (button);
    gtk_signal_connect_after (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_cancel), (gpointer) &cbkstruct->its_thing->dlg_data);
    gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_data_ok), cbkstruct);

    button = gtk_button_new_with_label (_("Cancel"));
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(datadlg)->action_area), button, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_show (button);
    gtk_signal_connect_after (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_cancel), (gpointer) &cbkstruct->its_thing->dlg_data);
    gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (dialog_data_cleanup), cbkstruct);
    
    gtk_widget_show (datadlg);
};


void dialog_data_ok (GtkWidget *w, gpointer data)
{
    GdkColor color;
    
    struct cbk_data *in = (struct cbk_data *) data;

    gdk_color_parse("red", &color);
    gdk_color_alloc(gdk_colormap_get_system(), &color);
    
    if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->icondia)->active)
    {
	gtk_plot_dataset_set_symbol(in->its_thing->real_dataset,
				    GTK_PLOT_SYMBOL_DIAMOND,
				    GTK_PLOT_SYMBOL_OPAQUE,
				    10, 2, color);
    }
    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->iconsq)->active)
    {
	gtk_plot_dataset_set_symbol(in->its_thing->real_dataset,
				    GTK_PLOT_SYMBOL_SQUARE,
				    GTK_PLOT_SYMBOL_OPAQUE,
				    10, 2, color);
    }
    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->iconpl)->active)
    {
	gtk_plot_dataset_set_symbol(in->its_thing->real_dataset,
				    GTK_PLOT_SYMBOL_PLUS,
				    GTK_PLOT_SYMBOL_OPAQUE,
				    10, 2, color);
    }
    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->iconcr)->active)
    {
	gtk_plot_dataset_set_symbol(in->its_thing->real_dataset,
				    GTK_PLOT_SYMBOL_CROSS,
				    GTK_PLOT_SYMBOL_OPAQUE,
				    10, 2, color);
    }
    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->iconst)->active)
    {
	gtk_plot_dataset_set_symbol(in->its_thing->real_dataset,
				    GTK_PLOT_SYMBOL_STAR,
				    GTK_PLOT_SYMBOL_OPAQUE,
				    10, 2, color);
    }
    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->iconup)->active)
    {
	gtk_plot_dataset_set_symbol(in->its_thing->real_dataset,
				    GTK_PLOT_SYMBOL_UP_TRIANGLE,
				    GTK_PLOT_SYMBOL_OPAQUE,
				    10, 2, color);
    }
    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->icondown)->active)
    {
	gtk_plot_dataset_set_symbol(in->its_thing->real_dataset,
				    GTK_PLOT_SYMBOL_DOWN_TRIANGLE,
				    GTK_PLOT_SYMBOL_OPAQUE,
				    10, 2, color);
    }
    
    if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->connectnone)->active)
    {
	in->its_thing->connect = GTK_PLOT_CONNECT_NONE;
	gtk_plot_dataset_set_connector (in->its_thing->real_dataset,
					GTK_PLOT_CONNECT_NONE);
    }

    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->connectstraight)->active)
    {
	in->its_thing->connect = GTK_PLOT_CONNECT_STRAIGHT;
	gtk_plot_dataset_set_connector (in->its_thing->real_dataset,
					GTK_PLOT_CONNECT_STRAIGHT);
    }

    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->connectspline)->active)
    {
	in->its_thing->connect = GTK_PLOT_CONNECT_SPLINE;
	gtk_plot_dataset_set_connector (in->its_thing->real_dataset,
					GTK_PLOT_CONNECT_SPLINE);
    }

    if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->regrlin)->active)
	in->its_thing->regrtype = REGRESSION_LINEAR;
    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->regrlog)->active)
	in->its_thing->regrtype = REGRESSION_LOGARITHMIC;
    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->regrexp)->active)
	in->its_thing->regrtype = REGRESSION_EXPONENTIAL;
    else if (GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->regrpow)->active)
	in->its_thing->regrtype = REGRESSION_POWER;
    else
	in->its_thing->regrtype = REGRESSION_NONE;

    in->its_thing->autoresize = GTK_TOGGLE_BUTTON(((struct cbk_data *) data)->autoresize)->active;
    
    
    in->its_thing->symbol_color = in->symbol_color.color;
    in->its_thing->line_color = in->line_color.color;
    
    
    gdk_colormap_free_colors (gdk_colormap_get_system(), &in->its_thing->symbol_color, 1);
    gdk_color_alloc (gdk_colormap_get_system(), &in->its_thing->symbol_color);
    
    gdk_colormap_free_colors (gdk_colormap_get_system(), &in->its_thing->line_color, 1);
    gdk_color_alloc (gdk_colormap_get_system(), &in->its_thing->line_color);
    

    gtk_plot_dataset_set_symbol (in->its_thing->real_dataset,
				 in->its_thing->real_dataset->symbol.symbol_type,
				 GTK_PLOT_SYMBOL_OPAQUE,
				 10, 2, in->its_thing->symbol_color);
    gtk_plot_dataset_set_line_attributes (in->its_thing->real_dataset,
					  GTK_PLOT_LINE_SOLID,
					  1,
					  in->its_thing->line_color);
    
    gtk_plot_dataset_set_line_attributes (in->its_thing->reg_dataset,
					  GTK_PLOT_LINE_SOLID,
					  0, in->its_thing->line_color);


    
    set_cell (in->its_thing->sheet, 0, 0);
    
    /* No need to redraw the widget, set_cell does it */

    free (in);
}


void dialog_data_cleanup (GtkWidget *w, gpointer data)
{
    struct cbk_data *in = (struct cbk_data *) data;
    free (in);
}
