/* 
 * Copyright (c) Tony Bybell 2001.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 */

#include <stdio.h>

#ifndef _MSC_VER
#include <unistd.h>
#include <sys/mman.h>
#else
#include <windows.h>
#endif


#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "aet.h"
#include "vcd.h"
#include "lxt.h"
#include "bsearch.h"

/****************************************************************************/

#define LXTHDR "LXTLOAD | "


/*
 * globals
 */
char is_lxt = 0;
char lxt_clock_compress_to_z = 0;

static void *mm;
static int version;

static struct fac *mvlfacs=NULL;
static int first_cycle, last_cycle, total_cycles;
static int maxchange=0, maxindex=0;
static int trace_start=0;
static off_t f_len;

static int *positional_information=NULL, *time_information=NULL;

static unsigned int change_field_offset=0;
static unsigned int facname_offset=0;
static unsigned int facgeometry_offset=0;
static unsigned int time_table_offset=0;
static unsigned int sync_table_offset=0;
static unsigned int initial_value_offset=0;
static unsigned int timescale_offset=0;
static unsigned int double_test_offset=0;


/****************************************************************************/

/*
 * reconstruct 8/16/24/32 bits out of the aet's representation
 * of a big-endian integer.  this can be optimized for 
 * big-endian platforms such as PPC but hasn't been.
 */
static unsigned int get_byte(int offset)
{
return(*((unsigned char *)mm+offset));
}

static unsigned int get_16(int offset)
{
unsigned char *nn=(unsigned char *)mm+offset;
unsigned int m1=*((unsigned char *)(nn++));
unsigned int m2=*((unsigned char *)nn);
return((m1<<8)|m2);
}

static unsigned int get_24(int offset)
{
unsigned char *nn=(unsigned char *)mm+offset;
unsigned long m1=*((unsigned char *)(nn++));
unsigned long m2=*((unsigned char *)(nn++));
unsigned long m3=*((unsigned char *)nn);
return((m1<<16)|(m2<<8)|m3);
}

static unsigned int get_32(int offset)
{
unsigned char *nn=(unsigned char *)mm+offset;
unsigned long m1=*((unsigned char *)(nn++));
unsigned long m2=*((unsigned char *)(nn++));
unsigned long m3=*((unsigned char *)(nn++));
unsigned long m4=*((unsigned char *)nn);
return((m1<<24)|(m2<<16)|(m3<<8)|m4);
}

/****************************************************************************/

static char double_mask[8]={0,0,0,0,0,0,0,0};
static char double_is_native=0;

static void create_double_endian_mask(int offset)
{
static double p = 3.14159;
double d;
int i, j;

d= *((double *)((unsigned char *)mm+offset));
if(p==d) 
	{
	double_is_native=1;
	}
	else
	{
	char *remote, *here;

	remote = (char *)&d;
	here = (char *)&p;
	for(i=0;i<8;i++)
		{
		for(j=0;j<8;j++)
			{
			if(here[i]==remote[j])
				{
				double_mask[i]=j;
				break;
				}
			}
		}	
	}
}

static char *swab_double_via_mask(int offset)
{
char swapbuf[8];
char *pnt = malloc_2(8*sizeof(char));
int i;

memcpy(swapbuf, ((unsigned char *)mm+offset), 8);
for(i=0;i<8;i++)
	{
	pnt[i]=swapbuf[double_mask[i]];
	}

return(pnt);
}

/****************************************************************************/

/*
 * convert 0123 to an mvl character representation
 */
static unsigned char convert_mvl(int value)
{
return("01zx1xx0xxxxxxxx"[value&15]);
}


/*
 * given an offset into the aet, calculate the "time" of that
 * offset in simulation.  this func similar to how gtkwave can
 * find the most recent valuechange that corresponds with
 * an arbitrary timevalue.
 */
static int max_compare_time_tc, max_compare_pos_tc;
static int compar_mvl_timechain(const void *s1, const void *s2)
{
int key, obj, delta;
int rv;

key=*((int *)s1);
obj=*((int *)s2);

if((obj<=key)&&(obj>max_compare_time_tc))
        {
        max_compare_time_tc=obj;
        max_compare_pos_tc=(int *)s2 - positional_information;
        }

delta=key-obj;
if(delta<0) rv=-1;
else if(delta>0) rv=1;
else rv=0;

return(rv);
}

static int bsearch_mvl_timechain(int key)
{
max_compare_time_tc=-1; max_compare_pos_tc=-1; 

bsearch((void *)&key, (void *)positional_information, total_cycles, sizeof(int), compar_mvl_timechain);
if((max_compare_pos_tc<=0)||(max_compare_time_tc<0)) 
        {
        max_compare_pos_tc=0; /* aix bsearch fix */
        }

return(time_information[max_compare_pos_tc]);
}


/*
 * decompress facility names and build up fac geometry
 */
static void build_facs(char *fname)
{
char *buf, *bufprev, *bufcurr;
int offs=facname_offset+8;
int i, j, clone;
char *pnt;
int total_mem = get_32(facname_offset+4);

buf=malloc_2(total_mem);
pnt=bufprev=buf;

for(i=0;i<numfacs;i++)
	{
        clone=get_16(offs);  offs+=2;
	bufcurr=pnt;
	for(j=0;j<clone;j++)
		{
		*(pnt++) = *(bufprev++);
		}
        while((*(pnt++)=get_byte(offs++)));
        mvlfacs[i].name=bufcurr;
	DEBUG(printf(LXTHDR"Encountered facility %d: '%s'\n", i, bufcurr));
	bufprev=bufcurr;
        }

if(!facgeometry_offset)
	{
	fprintf(stderr, "LXT '%s' is missing a facility geometry section, exiting.\n", fname);
	exit(255);
	}

offs=facgeometry_offset;
for(i=0;i<numfacs;i++)
	{
	mvlfacs[i].array_height=get_32(offs);
	mvlfacs[i].msb=get_32(offs+4);
	mvlfacs[i].lsb=get_32(offs+8);
	mvlfacs[i].flags=get_32(offs+12);
	mvlfacs[i].len=(mvlfacs[i].lsb>mvlfacs[i].msb)?(mvlfacs[i].lsb-mvlfacs[i].msb+1):(mvlfacs[i].msb-mvlfacs[i].lsb+1);
	offs+=0x10;
	}

}


/*
 * build up the lastchange entries so we can start to walk
 * through the aet..
 */
static void build_facs2(char *fname)
{
int i;
int offs;
int chg;
int maxchg=0, maxindx=0;
int last_position, last_time;

if(!time_table_offset)
	{
	fprintf(stderr, "LXT '%s' is missing a time table section, exiting.\n", fname);
	exit(255);
	}

offs = time_table_offset;

DEBUG(printf(LXTHDR"Time table position: %08x\n", time_table_offset + 12));

first_cycle=get_32(offs+4);
DEBUG(printf(LXTHDR"First cycle: %d\n", first_cycle));
last_cycle=get_32(offs+8);
DEBUG(printf(LXTHDR"Last cycle: %d\n", last_cycle));
total_cycles=get_32(offs+0);
DEBUG(printf(LXTHDR"Total cycles: %d\n", total_cycles));
DEBUG(printf(LXTHDR"Total cycles (actual): %d\n", last_cycle-first_cycle+1));

/* rebuild time table from its deltas... */

positional_information = (int *)malloc(total_cycles * sizeof(int));
last_position=0;
offs+=12;
for(i=0;i<total_cycles;i++)
	{
	last_position = positional_information[i] = get_32(offs) + last_position;
	offs+=4;
	}
time_information =       (int *)malloc(total_cycles * sizeof(int));
last_time=0;
for(i=0;i<total_cycles;i++)
	{
	last_time = time_information[i] = get_32(offs) + last_time;
	offs+=4;
	}

if(!sync_table_offset)
	{
	fprintf(stderr, "LXT '%s' is missing a sync table section, exiting.\n", fname);
	exit(255);
	}

offs = sync_table_offset;

for(i=0;i<numfacs;i++)
	{
	chg=get_32(offs);  offs+=4;
	if(chg>maxchg) {maxchg=chg; maxindx=i; }
	mvlfacs[i].lastchange=chg;
	DEBUG(printf(LXTHDR"Changes: %d '%s' %08x\n", i, mvlfacs[i].name, chg));
	}

DEBUG(printf(LXTHDR"Maxchange at: %08x for symbol '%s' of len %d\n", maxchg, mvlfacs[maxindx].name,  mvlfacs[maxindx].len));

maxchange=maxchg;
maxindex=maxindx;
}


/*
 * given a fac+offset, return the binary data for it
 */
static char *lt_buf=NULL;
static int lt_len = 0;

static char *parse_offset(struct fac *which, unsigned int offs)
{

int v, v2, j;
int k, l;
char *pnt;
char repeat;

l=which->len;
if(l>lt_len)
	{
	if(lt_buf) free_2(lt_buf);
	lt_buf=(char *)malloc_2(lt_len=l);
	}


pnt = lt_buf;
v=get_byte(offs);
v2=v&0x0f;

switch(v2)
	{
	case 0x00:	/* MVL2 */
			{
			unsigned int msk;
			int bitcnt=0;
			int ch;
			
			offs += ((v>>4)&3)+2;	/* skip value */
			for(j=0;;j++)
				{
				ch=get_byte(offs+j);
				msk=0x80;
				for(k=0;k<8;k++)
					{
					*(pnt++)= (ch&msk) ? '1' : '0';
					msk>>=1;
					bitcnt++;
					if(bitcnt==l) goto bail;
					}
				}
			}
			break;

	case 0x01:	/* MVL4 */
			{
			int bitcnt=0;
			int ch;
			int rsh;

			offs += ((v>>4)&3)+2;	/* skip value */
			for(j=0;;j++)
				{
				ch=get_byte(offs+j);
				rsh=6;
				for(k=0;k<4;k++)
					{
					*(pnt++)=convert_mvl((ch>>rsh)&0x3);
					rsh-=2;
					bitcnt++;
					if(bitcnt==l) goto bail;
					}
				}
			}
			break;

	case 0x02:	/* MVL9 */
			{
			int bitcnt=0;
			int ch;
			int rsh;

			offs += ((v>>4)&3)+2;	/* skip value */
			for(j=0;;j++)
				{
				ch=get_byte(offs+j);
				rsh=4;
				for(k=0;k<2;k++)
					{
					*(pnt++)=convert_mvl(ch>>rsh);
					rsh-=4;
					bitcnt++;
					if(bitcnt==l) goto bail;
					}
				}
			}
			break;

	case 0x03:	/* mvl repeat expansions */
	case 0x04:
	case 0x05:
	case 0x06:
	case 0x07:
	case 0x08:
	case 0x09:
	case 0x0a:
	case 0x0b:
			repeat = convert_mvl(v2-3);
			for(j=0;j<l;j++)
				{
				*(pnt++)=repeat;
				}
			*pnt=0;
			break;

	default:
			fprintf(stderr, LXTHDR"Unknown %02x at offset: %08x\n", v, offs);
			exit(255);
	}

bail:
return(lt_buf);
}


/*
 * mainline
 */
TimeType lxt_main(char *fname)
{
int i;
struct Node *n;
struct symbol *s, *prevsymroot=NULL, *prevsym=NULL;
unsigned int tagpnt;
int tag;

#ifndef _MSC_VER
int fd;

fd=open(fname, O_RDONLY);
if(fd<0)
        {
        fprintf(stderr, "Could not open '%s', exiting.\n", fname);
        exit(0);
        }

f_len=lseek(fd, 0, SEEK_END);
mm=mmap(NULL, f_len, PROT_READ, MAP_SHARED, fd, 0);
#else
HANDLE hIn, hInMap;

hIn = CreateFile(fname, GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
if(hIn == INVALID_HANDLE_VALUE)
	{
	fprintf(stderr, "Could not open '%s', exiting.\n", fname);
	exit(0);
	}
hInMap = CreateFileMapping(hIn, NULL, PAGE_READONLY, 0, 0, NULL);
mm = MapViewOfFile(hInMap, FILE_MAP_READ, 0, 0, 0);
f_len = GetFileSize(hIn, NULL);
#endif

if((i=get_16(0))!=0x0138)
	{
	fprintf(stderr, "Not an LXT format AET, exiting.\n");
	exit(255);
	}

if(version=get_16(2)>0x0001)
	{
	fprintf(stderr, "Version %d of LXT format AETs not supported, exiting.\n", version);
	exit(255);
	}	

if(get_byte(f_len-1)!=0xb4)
	{
	fprintf(stderr, "LXT '%s' is truncated, exiting.\n", fname);
	exit(255);
	}

DEBUG(printf(LXTHDR"Loading LXT '%s'...\n", fname));
DEBUG(printf(LXTHDR"Len: %d\n", (unsigned int)f_len));

tagpnt = f_len-2;
while((tag=get_byte(tagpnt))!=LT_SECTION_END)
	{
	int offset = get_32(tagpnt-4);
	tagpnt-=5;

	switch(tag)
		{
		case LT_SECTION_CHG:			change_field_offset=offset; DEBUG(printf(LXTHDR"LT_SECTION_CHG at %08x\n", offset)); break;
		case LT_SECTION_SYNC_TABLE:		sync_table_offset=offset; DEBUG(printf(LXTHDR"LT_SECTION_SYNC_TABLE at %08x\n", offset)); break;
		case LT_SECTION_FACNAME:		facname_offset=offset; DEBUG(printf(LXTHDR"LT_SECTION_FACNAME at %08x\n", offset)); break;
		case LT_SECTION_FACNAME_GEOMETRY:	facgeometry_offset=offset; DEBUG(printf(LXTHDR"LT_SECTION_FACNAME_GEOMETRY at %08x\n", offset)); break;
		case LT_SECTION_TIMESCALE:		timescale_offset=offset; DEBUG(printf(LXTHDR"LT_SECTION_TIMESCALE at %08x\n", offset)); break;
		case LT_SECTION_TIME_TABLE:		time_table_offset=offset; DEBUG(printf(LXTHDR"LT_SECTION_TIME_TABLE at %08x\n", offset)); break;
		case LT_SECTION_INITIAL_VALUE:		initial_value_offset=offset; DEBUG(printf(LXTHDR"LT_SECTION_INITIAL_VALUE at %08x\n", offset)); break;
		case LT_SECTION_DOUBLE_TEST:		double_test_offset=offset; DEBUG(printf(LXTHDR"LT_SECTION_DOUBLE_TEST at %08x\n", offset)); break;
		default: fprintf(stderr, "Skipping unknown section tag %02x.\n", tag); break;
		}	
	}

if(double_test_offset)
	{
	create_double_endian_mask(double_test_offset);
	}

if(timescale_offset)
	{
	signed char scale;

	scale=(signed char)get_byte(timescale_offset);
	switch(scale)
		{
		case 0:		time_dimension = 's'; break;
		case -3:	time_dimension = 'm'; break;
		case -6:	time_dimension = 'u'; break;
		case -12:	time_dimension = 'p'; break;
		case -15:	time_dimension = 'f'; break;
		case -9:
		default:	time_dimension = 'n'; break;
		}
	}
	else
	{
	time_dimension = 'n';
	}

if(!facname_offset)
	{
	fprintf(stderr, "LXT '%s' is missing a facility name section, exiting.\n", fname);
	exit(255);
	}

numfacs=get_32(facname_offset);
DEBUG(printf(LXTHDR"Number of facs: %d\n", numfacs));
mvlfacs=(struct fac *)calloc(numfacs,sizeof(struct fac));
build_facs(fname);
build_facs2(fname);

/* do your stuff here..all useful info has been initialized by now */

if(!hier_was_explicitly_set)    /* set default hierarchy split char */
        {
        hier_delimeter='.';
        }

for(i=0;i<numfacs;i++)
        {
	char buf[4096];
	char *str;	
	struct fac *f;

	if(mvlfacs[i].flags&LT_SYM_F_ALIAS)
		{
		int alias = mvlfacs[i].array_height;
		f=mvlfacs+alias;

		while(f->flags&LT_SYM_F_ALIAS)
			{
			f=mvlfacs+f->array_height;
			}
		}
		else
		{
		f=mvlfacs+i;
		}

	if((f->len>1)&& (!(f->flags&(LT_SYM_F_INTEGER|LT_SYM_F_DOUBLE|LT_SYM_F_STRING))) )
		{
		sprintf(buf, "%s[%d:%d]", mvlfacs[i].name,mvlfacs[i].msb, mvlfacs[i].lsb);
		str=malloc_2(strlen(buf)+1);
		if(!alt_hier_delimeter)
			{
			strcpy(str, buf);
			}
			else
			{
			strcpy_vcdalt(str, buf, alt_hier_delimeter);
			}
	        s=symadd(str,hash(str));
		prevsymroot = prevsym = NULL;
		}
	else if ( (f->len==1)&&(!(f->flags&(LT_SYM_F_INTEGER|LT_SYM_F_DOUBLE|LT_SYM_F_STRING)))&&
			((i!=numfacs-1)&&(!strcmp(mvlfacs[i].name, mvlfacs[i+1].name))) ||
			((i!=0)&&(!strcmp(mvlfacs[i].name, mvlfacs[i-1].name))) &&
			(mvlfacs[i].msb!=-1)&&(mvlfacs[i].lsb!=-1)
		)
		{
		sprintf(buf, "%s[%d]", mvlfacs[i].name,mvlfacs[i].msb);
		str=malloc_2(strlen(buf)+1);
		if(!alt_hier_delimeter)
			{
			strcpy(str, buf);
			}
			else
			{
			strcpy_vcdalt(str, buf, alt_hier_delimeter);
			}
	        s=symadd(str,hash(str));
		if((prevsym)&&(i>1)&&(!strcmp(mvlfacs[i].name, mvlfacs[i-1].name)))	/* allow chaining for search functions.. */
			{
			prevsym->vec_root = prevsymroot;
			prevsym->vec_chain = s;
			s->vec_root = prevsymroot;
			prevsym = s;
			}
			else
			{
			prevsymroot = prevsym = s;
			}
		}
		else
		{
		str=malloc_2(strlen(mvlfacs[i].name)+1);
		if(!alt_hier_delimeter)
			{
			strcpy(str, mvlfacs[i].name);
			}
			else
			{
			strcpy_vcdalt(str, mvlfacs[i].name, alt_hier_delimeter);
			}
	        s=symadd(str,hash(str));
		prevsymroot = prevsym = NULL;

		if(f->flags&LT_SYM_F_INTEGER)
			{
			mvlfacs[i].msb=31;
			mvlfacs[i].lsb=0;
			mvlfacs[i].len=32;
			}
		}
		
        if(!firstnode)
                {
                firstnode=curnode=s;   
                }
                else
                {
                curnode->nextinaet=s;
                curnode=s;   
                }

        n=(struct Node *)calloc_2(1,sizeof(struct Node));
        n->nname=s->name;
        n->mvlfac = mvlfacs+i;

	if((f->len>1)||(f->flags&&(LT_SYM_F_DOUBLE|LT_SYM_F_STRING)))
		{
		ExtNode *ext = (ExtNode *)calloc_2(1,sizeof(struct ExtNode));
		ext->msi = mvlfacs[i].msb;
		ext->lsi = mvlfacs[i].lsb;
		n->ext = ext;
		}
                 
        n->head.time=-1;        /* mark 1st node as negative time */
        n->head.v.val=1;
        s->n=n;
        }

facs=(struct symbol **)malloc_2(numfacs*sizeof(struct symbol *));
curnode=firstnode;
for(i=0;i<numfacs;i++)
	{
	char *subst, ch;
	int len;

	facs[i]=curnode;
        if((len=strlen(subst=curnode->name))>longestname) longestname=len;
	curnode=curnode->nextinaet;
	while((ch=(*subst)))
		{	
		if(ch==hier_delimeter) { *subst=0x01; }	/* forces sort at hier boundaries */
		subst++;
		}
	}
heapsort(facs,numfacs);
	
for(i=0;i<numfacs;i++)
	{
	char *subst, ch;

	subst=facs[i]->name;
	while((ch=(*subst)))
		{	
		if(ch==0x01) { *subst=hier_delimeter; }	/* restore back to normal */
		subst++;
		}

#ifdef DEBUG_FACILITIES
	printf("%-4d %s\n",i,facs[i]->name);
#endif
	}

facs_are_sorted=1;

init_tree();		
for(i=0;i<numfacs;i++)	
{
build_tree_from_name(facs[i]->name, i);
}
treeprune(treeroot);

#ifdef DEBUG_FACILITIES
treedebug(treeroot,"");
#endif

min_time = first_cycle; max_time=last_cycle;
printf("["TTFormat"] start time.\n["TTFormat"] end time.\n", min_time, max_time);
is_lxt = ~0;
return(max_time);
}


/*
 * this is the black magic that handles aliased signals...
 */
static void lxt_resolver(nptr np, nptr resolve)
{
np->ext = resolve->ext;
memcpy(&np->head, &resolve->head, sizeof(struct HistEnt));
np->curr = resolve->curr;
np->harray = resolve->harray;
np->numhist = resolve->numhist;
np->mvlfac=NULL;
}


/*
 * actually import an lxt trace but don't do it if
 * 1) it's already been imported
 * 2) an alias of this trace has been imported--instead
 *    copy over the relevant info and be done with it.
 */
void import_lxt_trace(nptr np)
{
unsigned int offs, offsdelta;
int v, w;
int ary_indx, ary_skip;
int tmval;
int prevtmval;
struct HistEnt *htemp;
struct HistEnt *histent_head, *histent_tail;
char *parsed;
int len, i, j;
struct fac *f;

if(!(f=np->mvlfac)) return;	/* already imported */

if(np->mvlfac->flags&LT_SYM_F_ALIAS)
	{
	int alias = np->mvlfac->array_height;
	f=mvlfacs+alias;

	if(f->resolve_lxt_alias_to)
		{
		if(!np->mvlfac->resolve_lxt_alias_to) np->mvlfac->resolve_lxt_alias_to = f->resolve_lxt_alias_to;
		}
		else
		{
		f->resolve_lxt_alias_to = np;
		}

	while(f->flags&LT_SYM_F_ALIAS)
		{
		f=mvlfacs+f->array_height;

		if(f->resolve_lxt_alias_to)
			{
			if(!np->mvlfac->resolve_lxt_alias_to) np->mvlfac->resolve_lxt_alias_to = f->resolve_lxt_alias_to;
			}
			else
			{
			f->resolve_lxt_alias_to = np; 
			}
		}
	}

/* f is the head minus any aliases, np->mvlfac is us... */
if(np->mvlfac->resolve_lxt_alias_to)	/* in case we're an alias head for later.. */
	{
	lxt_resolver(np, np->mvlfac->resolve_lxt_alias_to);
	return;
	}

np->mvlfac->resolve_lxt_alias_to = np;	/* in case we're an alias head for later.. */
offs=f->lastchange;

tmval=-1;
prevtmval = -1;
len = np->mvlfac->len;
         
histent_tail = htemp = histent_calloc();
if(len>1)
	{
	htemp->v.vector = (char *)malloc_2(len);
	for(i=0;i<len;i++) htemp->v.vector[i] = 2;
	}
	else
	{
	htemp->v.val = 2;		/* z */
	}
htemp->time = MAX_HISTENT_TIME;

histent_head = histent_calloc();
if(len>1)
	{
	histent_head->v.vector = (char *)malloc_2(len);
	for(i=0;i<len;i++) histent_head->v.vector[i] = 1;
	}
	else
	{
	histent_head->v.val = 1;	/* x */
	}
histent_head->time = MAX_HISTENT_TIME-1;
histent_head->next = htemp;	/* x */

np->numhist=2;

if(f->array_height < 1)	/* sorry, arrays not supported */
while(offs)
	{
	unsigned char val;

	if( (w=((v=get_byte(offs))&0xF)) >0xb)
		{
		unsigned int offsminus1, offsminus2;
		int delta, time_offsminus1;
		int skip;

		switch(v&0xF0)
			{
			case 0x00:
				skip = 2;
				offsdelta=get_byte(offs+1);
				break;
			case 0x10:
				skip = 3;
				offsdelta=get_16(offs+1);
				break;
			case 0x20:
				skip = 4;
				offsdelta=get_24(offs+1);
				break;
			case 0x30:
				skip = 5;
				offsdelta=get_32(offs+1);
				break;
			default:
				fprintf(stderr, "Unknown %02x at offset: %08x\n", v, offs);
				exit(0);
			}
		offsminus1 = offs-offsdelta-2;

		switch(get_byte(offsminus1)&0xF0)
			{
			case 0x00:
				offsdelta=get_byte(offsminus1+1);
				break;
			case 0x10:
				offsdelta=get_16(offsminus1+1);
				break;
			case 0x20:
				offsdelta=get_24(offsminus1+1);
				break;
			case 0x30:
				offsdelta=get_32(offsminus1+1);
				break;
			default:
				fprintf(stderr, "Unknown %02x at offset: %08x\n", get_byte(offsminus1), offsminus1);
				exit(0);
			}
		offsminus2 = offsminus1-offsdelta-2;

		delta = (time_offsminus1=bsearch_mvl_timechain(offsminus1)) - bsearch_mvl_timechain(offsminus2);

		if(!lxt_clock_compress_to_z)
			{
			int val = get_byte(offsminus1)&0xF;
			int reps;
			int rcnt;

			if((val<3)||(val>4))
				{
				fprintf(stderr, "Unexpected clk compress byte %02x at offset: %08x\n", get_byte(offsminus1), offsminus1);
				exit(0);
				}

			switch(w&3)
				{
				case 0:	reps = get_byte(offs+skip); break;
				case 1: reps = get_16(offs+skip); break;
				case 2: reps = get_24(offs+skip); break;
				case 3: reps = get_32(offs+skip); break;
				}

			reps++;
			val = (reps & 1) ^ (val==4);	/* because x3='0', x4='1' */
			val |= val<<1;			/* because 00='0', 11='1' */

			tmval = time_offsminus1 + (delta * reps);
			for(rcnt=0;rcnt<reps;rcnt++)
				{
				if(val!=histent_head->v.val)
					{
					htemp = histent_calloc();
					htemp->v.val = val;
					htemp->time = tmval;
					htemp->next = histent_head;
					histent_head = htemp;
					np->numhist++;
					}
					else
					{
					histent_head->time = tmval;
					}

				tmval-=delta;
				val=val^3;
				}
			}
			else
			{
			int val=2;

			if(val!=histent_head->v.val)
				{
				htemp = histent_calloc();
				htemp->v.val = val;
				htemp->time = time_offsminus1 + delta;
				htemp->next = histent_head;
				histent_head = htemp;
				np->numhist++;
				}
				else
				{
				histent_head->time = time_offsminus1 + delta;
				}
			tmval = time_offsminus1 + delta;
			}

		offs = offsminus1;	/* no need to recalc it again! */
		continue;
		}
	else
	if((tmval=bsearch_mvl_timechain(offs))!=prevtmval)	/* get rid of glitches (if even possible) */
		{
		DEBUG(printf(LXTHDR"offs: %08x is time %08x\n", offs, tmval));
		if(!(f->flags&(LT_SYM_F_DOUBLE|LT_SYM_F_STRING)))
			{
			parsed=parse_offset(f, offs);

			if(len==1)
				{
				switch(parsed[0])
					{
				        case '0':       val = 0; break;
				        case 'x':	val = 1; break;
				        case 'z':	val = 2; break;
				        case '1':       val = 3; break;
				        }
	
				if(val!=histent_head->v.val)
					{
					htemp = histent_calloc();
					htemp->v.val = val;
					htemp->time = tmval;
					htemp->next = histent_head;
					histent_head = htemp;
					np->numhist++;
					}
					else
					{
					histent_head->time = tmval;
					}
				}
				else
				{
				if(memcmp(parsed, histent_head->v.vector, len))
					{
					htemp = histent_calloc();
					htemp->v.vector = (char *)malloc_2(len);
					memcpy(htemp->v.vector, parsed, len);
					htemp->time = tmval;
					htemp->next = histent_head;
					histent_head = htemp;
					np->numhist++;
					}
					else
					{
					histent_head->time = tmval;
					}
				}
			}
		else
		if(f->flags&LT_SYM_F_DOUBLE)
			{
			int offs_dbl = offs + ((get_byte(offs)>>4)&3)+2;   /* skip value */

			htemp = histent_calloc();
			htemp->flags = HIST_REAL;

			if(double_is_native)
				{
				htemp->v.vector = ((unsigned char *)mm+offs_dbl);
				DEBUG(printf(LXTHDR"Added double '%.16g'\n", *((double *)(mm+offs_dbl))));
				}
				else
				{
				htemp->v.vector = swab_double_via_mask(offs_dbl);
				DEBUG(printf(LXTHDR"Added bytefixed double '%.16g'\n", *((double *)(htemp->v.vector))));
				}
			htemp->time = tmval;
			htemp->next = histent_head;
			histent_head = htemp;
			np->numhist++;
			}
			else
			{						/* defaults to if(f->flags&LT_SYM_F_STRING) */
			int offs_str = offs + ((get_byte(offs)>>4)&3)+2;   /* skip value */

			htemp = histent_calloc();
			htemp->flags = HIST_REAL|HIST_STRING;

			htemp->v.vector = ((unsigned char *)mm+offs_str);
			DEBUG(printf(LXTHDR"Added string '%s'\n", (unsigned char *)mm+offs_str));
			htemp->time = tmval;
			htemp->next = histent_head;
			histent_head = htemp;
			np->numhist++;
			}
		}

	prevtmval = tmval;
	
/*	v=get_byte(offs); */
	switch(v&0xF0)
		{
		case 0x00:
			offsdelta=get_byte(offs+1);
			break;
	
		case 0x10:
			offsdelta=get_16(offs+1);
			break;
	
		case 0x20:
			offsdelta=get_24(offs+1);
			break;

		case 0x30:
			offsdelta=get_32(offs+1);
			break;

		default:
			fprintf(stderr, "Unknown %02x at offset: %08x\n", v, offs);
			exit(0);
		}

	offs = offs-offsdelta-2;
	}

np->mvlfac = NULL;	/* it's imported and cached so we can forget it's an mvlfac now */

for(j=0;j>-2;j--)
	{
	if(tmval!=first_cycle)
		{
		char init;

		htemp = histent_calloc();

		if(!(f->flags&(LT_SYM_F_DOUBLE|LT_SYM_F_STRING)))
			{
			if(initial_value_offset)
				{
				switch(get_byte(initial_value_offset))
					{
					case 0:
					case 7:		init = 0; break;
		
					case 1:
					case 4:		init = 3; break;
		
					case 2:		init = 2; break;
		
					default:	init = 1; break;
					}
				}
				else
				{
				init = 1; /* x if unspecified */
				}
		
			if(len>1)
				{
				char *pnt = htemp->v.vector = (char *)malloc_2(len);	/* zeros */
				int i;
				for(i=0;i<len;i++)
					{
					*(pnt++) = init;
					}
				}
				else
				{
				htemp->v.val = init;
				}
			}
			else
			{
			htemp->flags = HIST_REAL;
			if(f->flags&LT_SYM_F_STRING) htemp->flags |= HIST_STRING;
			}
	
		htemp->time = first_cycle+j;
		htemp->next = histent_head;
		histent_head = htemp;
		np->numhist++;
		}

	tmval=first_cycle+1;
	}

if(!(f->flags&(LT_SYM_F_DOUBLE|LT_SYM_F_STRING)))
	{
	if(len>1)
		{
		np->head.v.vector = (char *)malloc_2(len);
		for(i=0;i<len;i++) np->head.v.vector[i] = 1;
		}
		else
		{
		np->head.v.val = 1;                     /* 'x' */
		}
	}
	else
	{
	np->head.flags = HIST_REAL;
	if(f->flags&LT_SYM_F_STRING) np->head.flags |= HIST_STRING;
	}
np->head.time  = -2;
np->head.next = histent_head;
np->curr = histent_tail;
np->numhist++;
}
