 /*  -*-objc-*-
 *  Finder.m: Implementation of the Finder Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "Finder.h"
#include "GWSplitView.h"
#include "Shelf/Shelf.h"
#include "Functions.h"
#include "GWorkspace.h"
#include "GNUstep.h"

@implementation ControlsView

- (void)drawRect:(NSRect)rect
{
  float w = [self frame].size.width;
  float h = [self frame].size.height;

	[super drawRect: rect];

	STROKE_LINE (darkGrayColor, 0, h -2, w, h -2);
	STROKE_LINE (whiteColor, 0, h -3, w, h -3);
}

@end

@implementation FileIconView

- (void)dealloc
{
  TEST_RELEASE (icon);
  RELEASE (highlightImage);
  RELEASE (nameField);  
  [super dealloc];
}

- (id)init
{
  self = [super init];
  if (self) {
    [self setFrame: NSMakeRect(0, 0, 64, 64)];
	  nameField = [[NSTextField alloc] initWithFrame: NSMakeRect(0, 0, 64, 12)];
	  [nameField setBackgroundColor: [NSColor windowBackgroundColor]];
		[nameField setFont: [NSFont systemFontOfSize: 10]];
	  [nameField setBezeled: NO];
	  [nameField setEditable: NO];
	  [nameField setSelectable: NO];
    [self addSubview: nameField];
    
    ASSIGN (highlightImage, [NSImage imageNamed: @"CellHighlight.tiff"]);
		        
    isactive = NO;
  }
  return self;  
}

- (void)activateForFileAtPath:(NSString *)fpath
{
  ASSIGN (icon, [[NSWorkspace sharedWorkspace] iconForFile: fpath]);
  [nameField setStringValue: cutFileLabelText([fpath lastPathComponent], nameField, 64)];
  isactive = YES;
  [self setNeedsDisplay: YES];
}

- (void)deactivate
{
  [nameField setStringValue: @" "];
  isactive = NO;
	[self setNeedsDisplay: YES];
}

- (void)drawRect:(NSRect)rect
{
	NSPoint p;

  if (isactive == NO) {
    return;
	}

  p = NSMakePoint(0, 12);
  [highlightImage compositeToPoint: p operation: NSCompositeSourceOver];

  p = NSMakePoint((60 - [icon size].width) / 2, (52 - [icon size].height) / 2 + 12);
	[icon compositeToPoint: p operation: NSCompositeSourceOver];    
}

@end

@implementation Finder

- (void)dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];
  if (task && [task isRunning]) {
    [task terminate];
	}
  TEST_RELEASE (task);
  TEST_RELEASE (pipe);
  TEST_RELEASE (currentSelection);
  TEST_RELEASE (fullFoundPaths);
  RELEASE (findButt);
  RELEASE (findField);
  RELEASE (findPopUp);
  RELEASE (fileIconView);  
  RELEASE (scrollView);
  RELEASE (foundMatrix);
  RELEASE (controlsView);
  RELEASE (shelf);
  RELEASE (split);
  RELEASE (fWin);
  [super dealloc];
}

- (id)init
{
	self = [super init];
  if (self) {
	  unsigned int style = NSTitledWindowMask | NSClosableWindowMask 
				              | NSMiniaturizableWindowMask | NSResizableWindowMask;
	  NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];	
	  NSDictionary *myPrefs;
    NSButtonCell* buttonCell;
		float shfwidth;   

	  fWin = [[NSPanel alloc] initWithContentRect: NSZeroRect
						  styleMask: style backing: NSBackingStoreBuffered defer: NO];
    if ([fWin setFrameUsingName: @"Finder"] == NO) {
      [fWin setFrame: NSMakeRect(100, 100, 413, 257) display: NO];
    }            
    [fWin setMinSize: NSMakeSize(413, 257)];
    [fWin setTitle: NSLocalizedString(@"Finder", @"")];
    [fWin setReleasedWhenClosed: NO]; 
    [fWin setDelegate: self];
		
		shfwidth = [[fWin contentView] frame].size.width;
		
    myPrefs = [defaults dictionaryForKey: @"finderprefs"];
    if (myPrefs != nil) {     
			NSArray *shelfDicts = [myPrefs objectForKey: @"shelfdicts"];  
			NSString *sheight = [myPrefs objectForKey: @"shelfheight"];

    	if(shelfDicts != nil) {
				shelf = [[FinderShelf alloc] initWithIconsDicts: shelfDicts rootPath: nil]; 
			} else {
				shelf = [[FinderShelf alloc] initWithIconsDicts: [NSArray array] rootPath: nil]; 
			}
		
    	if(sheight != nil) {
      	[shelf setFrame: NSMakeRect(0, 0, shfwidth, [sheight intValue])]; 
    	} else {
      	[shelf setFrame: NSMakeRect(0, 0, shfwidth, 75)];  
    	}
			
    } else {
			shelf = [[FinderShelf alloc] initWithIconsDicts: [NSArray array] rootPath: nil]; 
			[shelf setFrame: NSMakeRect(0, 0, shfwidth, 75)];  
    }

    [shelf setDelegate: self];
    [shelf setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];

    controlsView = [ControlsView new];
    [controlsView setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];    

	  findButt = [NSButton new];
	  [findButt setButtonType: NSMomentaryLight];
		[findButt setImage: [NSImage imageNamed: @"Magnify.tiff"]];
		[findButt setImagePosition: NSImageOnly];
		[findButt setTarget: self];
		[findButt setAction: @selector(startFind:)];
		[controlsView addSubview: findButt]; 

		findField = [NSTextField new];
		[controlsView addSubview: findField]; 

		findPopUp = [NSPopUpButton new];
    [findPopUp setPullsDown: NO];
		[findPopUp setTarget: self];
		[findPopUp setAction: @selector(popUpAction:)];
		[findPopUp addItemWithTitle: NSLocalizedString(@"Find items with names that match", @"")];
		[findPopUp addItemWithTitle: NSLocalizedString(@"Find items with contents that match", @"")];
		[controlsView addSubview: findPopUp];

    fileIconView = [FileIconView new];
    [controlsView addSubview: fileIconView]; 
		[fileIconView deactivate];
		
    scrollView = [NSScrollView new];
    [scrollView setHasHorizontalScroller: NO];
    [scrollView setHasVerticalScroller: YES]; 
    [scrollView setAutoresizingMask: NSViewHeightSizable];
    [controlsView addSubview: scrollView]; 

    buttonCell = AUTORELEASE([NSButtonCell new]);
    [buttonCell setButtonType: NSPushOnPushOffButton];
    [buttonCell setBordered: NO];
    [buttonCell setFont: [NSFont systemFontOfSize: 12]];
    [buttonCell setAlignment: NSLeftTextAlignment]; 

    foundMatrix = [[NSMatrix alloc] initWithFrame: NSZeroRect
			      	      mode: NSRadioModeMatrix prototype: buttonCell
		       							          numberOfRows: 0 numberOfColumns: 0];

    [foundMatrix setIntercellSpacing: NSZeroSize];
    [foundMatrix setTarget: self];		
    [foundMatrix setAction: @selector(choseFile:)];	
    [foundMatrix setDoubleAction: @selector(openFile:)];	
    [scrollView setDocumentView: foundMatrix];

    split = [[GWSplitView alloc] initWithFrame: [[fWin contentView] frame] viewer: self];
    [split setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];
    [split setDelegate: self];

		[split addSubview: shelf];
    [split addSubview: controlsView];
		[[fWin contentView] addSubview: split];

    currentSelection = nil;
    infoString = nil; 
    gw = [GWorkspace gworkspace];

  	[split resizeWithOldSuperviewSize: [split frame].size]; 
		[self adjustSubviews];			
  }
  
  return self;
}

- (void)activate
{
  [fWin orderFront: nil];
}

- (void)adjustSubviews
{
  NSRect r = [[fWin contentView] frame];
  float shelfheight;
  float w, h, d;
  
  h = [shelf frame].size.height;
  if (h < 35) {
    shelfheight = 2;
  } else if (h <= 75) {  
    shelfheight = 75;
  } else {
    shelfheight = 150;
  }
  
  [shelf setFrame: NSMakeRect(0, 0, r.size.width, shelfheight)];
  [shelf resizeWithOldSuperviewSize: [shelf frame].size]; 
  
  d = [split dividerThickness];
  w = r.size.width;
  h = r.size.height;    

  [controlsView setFrame: NSMakeRect(0, shelfheight + d, 
                                            w, h - shelfheight - d)];

  w = [controlsView frame].size.width;
  h = [controlsView frame].size.height;

  [findButt setFrame: NSMakeRect(11, h - 76, 64, 64)];
  
  [findField setFrame: NSMakeRect(85, h - 41, w - 188, 21)];
  
  [findPopUp setFrame: NSMakeRect(85, h - 76, 225, 20)];
  
  [fileIconView setFrame: NSMakeRect(w - 75, h - 76, 64, 64)];
    
  [scrollView  setFrame: NSMakeRect(11, 0, w - 22, h - 85)];
  
  [foundMatrix setCellSize: NSMakeSize(w - 45, 16)];
  [foundMatrix sizeToCells];
}

- (void)startFind:(id)sender
{
	NSString *toFind, *opt;
  NSMutableString *cline;
  NSArray *args;
	NSFileHandle *fileHandle;
	NSString *msg, *buttstr;
  int i;

	buttstr = NSLocalizedString(@"Continue", @"");
	
  if (currentSelection == nil) {
		msg = NSLocalizedString(@"No selection!", @"");
		NSRunAlertPanel(nil, msg, buttstr, nil, nil);
    return;
  }
  
  toFind = [findField stringValue];
  
  if ([toFind length] == 0) {
		msg = NSLocalizedString(@"No arguments!", @"");
		NSRunAlertPanel(nil, msg, buttstr, nil, nil);
    return;
  }
    
  [self emptyFoundMatrix];
  TEST_RELEASE (fullFoundPaths);
  fullFoundPaths = [[NSMutableArray alloc] initWithCapacity: 1];
  [findButt setEnabled: NO];
   
  cline = [NSMutableString stringWithCapacity: 1];
  [cline appendString: @"find "];

  for (i = 0; i < [currentSelection count]; i++) {
    [cline appendString: [currentSelection objectAtIndex: i]];
    [cline appendString: @" "];
  }

  opt = [findPopUp titleOfSelectedItem];
  
  if ([opt isEqualToString: NSLocalizedString(@"Find items with names that match", @"")] == YES) {
    [cline appendString: [NSString stringWithFormat: @"-name '*%@*' ", toFind]];
  } else {
    [cline appendString: @"-type 'f' "];
    [cline appendString: @"-print0 | xargs -r -0 grep -l "];
    [cline appendString: toFind];
  }

	ASSIGN (task, [NSTask new]); 
	AUTORELEASE (task);
  [task setLaunchPath: @"/bin/sh"];

  args = [NSArray arrayWithObjects: @"-c", cline, nil];
  [task setArguments: args];

  ASSIGN (pipe, [NSPipe pipe]);
  [task setStandardOutput: pipe];
    
  fileHandle = [pipe fileHandleForReading];
  [[NSNotificationCenter defaultCenter] addObserver: self
    								selector: @selector(dataFromTask:)
    										name: NSFileHandleReadToEndOfFileCompletionNotification
    									object: (id)fileHandle];

  [fileHandle readToEndOfFileInBackgroundAndNotify];    
  
  [[NSNotificationCenter defaultCenter] addObserver: self 
                   selector: @selector(endOfTask:) 
                       name: NSTaskDidTerminateNotification 
                     object: (id)task];
                     
  [task launch];            
}

- (void)dataFromTask:(NSNotification *)notification
{
  NSDictionary *userInfo;
  NSData *readData;
  NSString *buff;
  id cell;
  int i, count;
  
  userInfo = [notification userInfo];
  readData = [userInfo objectForKey: NSFileHandleNotificationDataItem];
  buff = [[NSString alloc] initWithData: readData encoding: NSNonLossyASCIIStringEncoding];
  ASSIGN (fullFoundPaths, [buff componentsSeparatedByString: @"\n"]);
  RELEASE (buff);
	  
	count = [fullFoundPaths count];
  if ([(NSString *)[fullFoundPaths objectAtIndex: count -1] length] == 0) {    
    count--;
  }
	
  for (i = 0; i < count; i++) {
    [foundMatrix addRow];      
    cell = [foundMatrix cellAtRow: [[foundMatrix cells] count] -1 column: 0];    
    [cell setStringValue: pathFittingInContainer(foundMatrix, [fullFoundPaths objectAtIndex: i], 20)];
  }
        
  [foundMatrix sizeToCells];
  [foundMatrix setNeedsDisplay: YES];      
}

- (void)endOfTask:(NSNotification *)notification
{
	if ([notification object] == task) {		
  	if([fullFoundPaths count] > 0) {
    	NSString *p = [fullFoundPaths objectAtIndex: 0];
    	if ([p length] > 0) {   
      	[fileIconView activateForFileAtPath: p];
    	} else {
      	[fileIconView deactivate];
    	}
  	} else {
    	[fileIconView deactivate];
  	}

  	[fileIconView setNeedsDisplay: YES];    
  	[findButt setEnabled: YES];

		[[NSNotificationCenter defaultCenter] removeObserver: self 
												name: NSTaskDidTerminateNotification object: task];
	}
}

- (void)popUpAction:(id)sender
{

}

- (void)choseFile:(id)sender
{
  int row, col;
  id cell;
  NSArray *paths;

  cell = [foundMatrix selectedCell]; 
  [foundMatrix getRow: &row column: &col ofCell: cell];   
  paths = [NSArray arrayWithObjects: [fullFoundPaths objectAtIndex: row], nil];
  [fileIconView activateForFileAtPath: [paths objectAtIndex: 0]];
  
  [gw setSelectedPaths: paths];   
}

- (void)openFile:(id)sender
{
  int row, col;
  id cell;
  NSArray *paths;
    
  cell = [foundMatrix selectedCell]; 
  [foundMatrix getRow: &row column: &col ofCell: cell]; 
  paths = [NSArray arrayWithObjects: [fullFoundPaths objectAtIndex: row], nil];

  [gw openSelectedPaths: paths newViewer: NO];   
}

- (void)emptyFoundMatrix
{
  NSArray *cellList = [foundMatrix cells];
  int count;
  
  if([cellList count] > 0) { 
    while (1) {
      count = [[foundMatrix cells] count];
      if (count == 0) {
        break;
      }
      [foundMatrix removeRow: count - 1];
    }
    [foundMatrix sizeToCells];
    [foundMatrix setNeedsDisplay: YES];
    [scrollView setNeedsDisplay: YES];
    
    [fileIconView deactivate];
  }
}

- (float)splitView:(NSSplitView *)sender
          constrainSplitPosition:(float)proposedPosition 
                                        ofSubviewAt:(int)offset
{
  if (proposedPosition < 35) {
    return 2;
  } else if (proposedPosition <= 75) {  
    return 75;
  } else {
    return 150;
  }
}

- (void)splitView:(NSSplitView *)sender 
                  resizeSubviewsWithOldSize:(NSSize)oldSize
{
	[self adjustSubviews];
}

- (void)splitViewDidResizeSubviews:(NSNotification *)aNotification
{
	[self adjustSubviews];
}

- (BOOL)windowShouldClose:(id)sender
{
  [self updateDefaults];
	return YES;
}

- (void)updateDefaults
{
	NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];		
  NSMutableDictionary *myPrefs;
  NSArray *shelfDicts;
  NSString *shelfHeight;

	if ([fWin isVisible]) {
  	[fWin saveFrameUsingName: @"Finder"];
	}
	
  myPrefs = [NSMutableDictionary dictionaryWithCapacity: 1];
  
  shelfDicts = [shelf iconsDicts];
  [myPrefs setObject: shelfDicts forKey: @"shelfdicts"];

  shelfHeight = [NSString stringWithFormat: @"%i", (int)[shelf frame].size.height];
  [myPrefs setObject: shelfHeight forKey: @"shelfheight"];

	[defaults setObject: myPrefs forKey: @"finderprefs"];
	[defaults synchronize];
}

- (NSWindow *)myWin
{
  return fWin;
}

- (void)updateInfoString
{
/*
  int count;

  path = [[[pathsMatrix selectedCell] paths] objectAtIndex: 0];
	attributes = [fm fileSystemAttributesAtPath: path];
	freeFs = [attributes objectForKey: NSFileSystemFreeSize];

  count = [fullFoundPaths count];
  ASSIGN (infoString, [NSString stringWithFormat: @"%i", count]);    

*/
}

- (NSString *)infoString
{
  return infoString;
}

- (BOOL)validateMenuItem:(id <NSMenuItem>)menuItem
{
	return YES;
}

//
// shelf delegate methods
//

- (void)shelf:(Shelf *)sender setCurrentSelection:(NSArray *)paths
{
  [self emptyFoundMatrix];
  ASSIGN (currentSelection, paths);
}

- (void)shelf:(Shelf *)sender openCurrentSelection:(NSArray *)paths newViewer:(BOOL)newv
{
//  [gw setSelectedPaths: paths];  

  // AGGIUNGERE UN SISTEMA PER AGGIORNARE IL/I VIEWER 
}

- (void)shelf:(Shelf *)sender keyDown:(NSEvent *)theEvent                                       
{

}

@end
