/*  -*-objc-*-
 *  GWBrowser.m: Implementation of the GWBrowser Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include <math.h> 

#include "GWBrowser.h"
#include "ColumnIcon.h"
#include "GWBrowserCell.h"
#include "BrowserViewer.h"
#include "Functions.h"
#include "GNUstep.h"

#define NSBR_COLUMN_SEP 6
#define NSBR_VOFFSET 2
#define BEZEL_BORDER_SIZE NSMakeSize(2, 2)

#define NSBR_COLUMN_IS_VISIBLE(i) \
(((i)>=firstVisibleColumn)&&((i)<=lastVisibleColumn))

#ifndef HAVE_RINTF
#define rintf rint
#endif
#ifndef HAVE_RINT
static double rint(double a)
{
  return (floor(a+0.5));
}
#endif

static float scrollerWidth;
static float iconsPathWidth;

extern NSString *NSPlainFileType;
extern NSString *NSDirectoryFileType;
extern NSString *NSApplicationFileType;
extern NSString *NSFilesystemFileType;
extern NSString *NSShellCommandFileType;

@implementation GWBrowserColumn

- (id)init
{
  [super init];
  isLoaded = NO;
	columnIcon = nil;
  return self;
}

- (void)dealloc
{
  TEST_RELEASE (columnScrollView);
  TEST_RELEASE (columnMatrix);
	TEST_RELEASE (columnIcon);
  [super dealloc];
}

- (void)setIsLoaded:(BOOL)flag
{
  isLoaded = flag;
}

- (BOOL)isLoaded
{
  return isLoaded;
}

- (void)setColumnScrollView:(NSScrollView *)aView
{
  ASSIGN (columnScrollView, aView);
}

- (NSScrollView *)columnScrollView
{
  return columnScrollView;
}

- (void)setColumnMatrix:(NSMatrix *)aMatrix
{
  ASSIGN (columnMatrix, aMatrix);
}

- (NSMatrix *)columnMatrix
{
  return columnMatrix;
}

- (void)setColumnIcon:(ColumnIcon *)aIcon
{
	if (columnIcon == nil) {
		ASSIGN (columnIcon, aIcon);
	}
}

- (ColumnIcon *)columnIcon
{
	return columnIcon;
}

- (void)setNumberOfRows:(int)num
{
  numberOfRows = num;
}

- (int)numberOfRows
{
  return numberOfRows;
}

@end


//
// Browser implementation
//

@implementation GWBrowser

//
// Init & C.
//

- (void)dealloc
{
  RELEASE (cellPrototype);
  RELEASE (pathSeparator);
  RELEASE (horizontalScroller);
  RELEASE (browserColumns);
	RELEASE (basePath);
  TEST_RELEASE(charBuffer);
  [super dealloc];
}

+ (void)initialize
{
  if (self == [GWBrowser class]) {
		scrollerWidth = [NSScroller scrollerWidth];
		iconsPathWidth = 96 - scrollerWidth;
	}
}

- (id)initWithFrame:(NSRect)rect basePath:(NSString *)bpath
{
  NSSize bs;

  self = [super initWithFrame: rect];
	ASSIGN (basePath, bpath);

  // Class setting
  cellPrototype = [[GWBrowserCell alloc] init];
  
  // Default values
  ASSIGN (pathSeparator, @"/");
  isLoaded = NO;
  browserDelegate = nil;
  doubleAction = NULL;  
  bs = BEZEL_BORDER_SIZE;
  minColumnWidth = scrollerWidth + (2 * bs.width);
  if (minColumnWidth < 100.0) {
    minColumnWidth = 100.0;
  }
	lastKeyPressed = 0.;
  charBuffer = nil;
  alphaNumericalLastColumn = -1;

  // Horizontal scroller
  scrollerRect.origin.x = bs.width;
  scrollerRect.origin.y = [self frame].size.height - iconsPathWidth - scrollerWidth;
  scrollerRect.size.width = [self frame].size.width - (2 * bs.width);
  scrollerRect.size.height = scrollerWidth;
  horizontalScroller = [[NSScroller alloc] initWithFrame: scrollerRect];
  [horizontalScroller setTarget: self];
  [horizontalScroller setAction: @selector(scrollViaScroller:)];
  [self addSubview: horizontalScroller];
  skipUpdateScroller = NO;

  // Columns
  browserColumns = [[NSMutableArray alloc] init];

  // Create a single column
  lastColumnLoaded = -1;
  firstVisibleColumn = 0;
  lastVisibleColumn = 0;
	currentshift = 0;
  [self createColumn];

  return self;
}

- (BOOL)setPath:(NSString *)path
{ 
  NSArray	*subStrings;
  NSString *aStr;
  unsigned numberOfSubStrings;
  unsigned i, j, column = 0;
  BOOL found = NO;		
	
  // Column Zero is always present. 
  [self loadColumnZero];

  // If that's all, return.
  if (path == nil || [path isEqualToString: basePath]) {
		[self setNeedsDisplay: YES];
		return YES;
	}
	
	if ([basePath isEqualToString: pathSeparator] == NO) { 
		NSRange range = [path rangeOfString: basePath]; 		 
																												 
		if (range.length == 0) {														 
			ASSIGN (basePath, pathSeparator); 								 
			[self loadColumnZero];		
			subStrings = [path componentsSeparatedByString: pathSeparator];
					
		} else {
			NSString *rpath = [path substringFromIndex: [basePath cStringLength]];
			subStrings = [rpath componentsSeparatedByString: pathSeparator];
		}
		
	} else {
  	subStrings = [path componentsSeparatedByString: pathSeparator];
	}
	
	numberOfSubStrings = [subStrings count];
	
  // Ignore a trailing void component. 
  if (numberOfSubStrings > 0
      			&& [[subStrings objectAtIndex: 0] isEqualToString: @""]) {
		numberOfSubStrings--;

		if (numberOfSubStrings) {
	  	NSRange theRange;
				
	  	theRange.location = 1;
	  	theRange.length = numberOfSubStrings;
	  	subStrings = [subStrings subarrayWithRange: theRange];
		}

		[self loadColumnZero];
	}

  column = lastColumnLoaded;
  if (column < 0) {
    column = 0;
	}
		
  // cycle thru str's array created from path
  for (i = 0; i < numberOfSubStrings; i++) {
		GWBrowserColumn	*bc = [browserColumns objectAtIndex: column + i];
		NSMatrix *matrix = [bc columnMatrix];
		NSArray *cells = [matrix cells];
		unsigned numOfRows = [cells count];
		NSBrowserCell	*selectedCell = nil;
      
		aStr = [subStrings objectAtIndex: i];

		if ([aStr isEqualToString: @""] == NO) {
	  	found = NO;

	  	for (j = 0; j < numOfRows; j++) {
	      NSString	*cellString;
	      
	      selectedCell = [cells objectAtIndex: j];
	      cellString = [selectedCell stringValue];
	      
	      if ([cellString isEqualToString: aStr]) {
		  		[matrix selectCellAtRow: j column: 0];
		  		found = YES;
		  		break;
				}
			}

	  	if (found == NO) {
				NSLog (@"GWBrowser: unable to find cell '%@' in column %d\n", aStr, column + i);
	    	break;
			}

			[self addColumn];
			[self updateIconOfColumn: i];
		}
	}
	
	[self updateIconOfColumn: [self lastColumn]];
	[self setLeafIconAtColumn: [self lastColumn]];
  [self setNeedsDisplay: YES];

  return found;
}

- (void)setPathSeparator:(NSString *)aString
{
  ASSIGN (pathSeparator, aString);
}

- (id)delegate
{
  return browserDelegate;
}

- (void)setDelegate:(id)anObject
{	
  browserDelegate = anObject;
}

- (void)setDoubleAction:(SEL)aSelector
{
  doubleAction = aSelector;
}

//
// Manipulating Columns 
//

- (void)addColumn
{
  int i;
			
  if (lastColumnLoaded + 1 >= [browserColumns count]) {
    i = [browserColumns indexOfObject: [self createColumn]];
	} else {
    i = lastColumnLoaded + 1;
	}
	
  if (i < 0) {
    i = 0;
	}
		
	[self addIconForColumn: i];
  [self performLoadOfColumn: i];
  [self setLastColumn: i];
  [self adjustMatrixOfColumn: i];

  isLoaded = YES;

  [self tile];

  if (i > 0 && i - 1 == lastVisibleColumn) {
    [self scrollColumnsRightBy: 1];
	}
}

- (GWBrowserColumn *)createColumn
{
  GWBrowserColumn *bc;
  NSScrollView *sc;
  NSRect rect = {{0, 0}, {100, 100}};
		
  bc = [[GWBrowserColumn alloc] init];

  // Create a scrollview
  sc = [[NSScrollView alloc] initWithFrame: rect];
  [sc setHasHorizontalScroller: NO];
  [sc setHasVerticalScroller: YES];
  [bc setColumnScrollView: sc];
  [self addSubview: sc];
  RELEASE(sc);
	
  [browserColumns addObject: bc];
  RELEASE(bc);
	
  return bc;
}

- (void)performLoadOfColumn:(int)column
{
  GWBrowserColumn *bc;
	NSScrollView *sc;
	NSMatrix *matrix;
	GWBrowserCell *cell;
  NSRect matrixRect = {{0, 0}, {100, 100}};
  NSSize matrixIntercellSpace = {0, 0};
	
  bc = [browserColumns objectAtIndex: column];

  if (!(sc = [bc columnScrollView])) {
    return;
	}
	
	// create a new col matrix
	matrix = [[NSMatrix alloc] initWithFrame: matrixRect
		   mode: NSListModeMatrix prototype: cellPrototype
		      												numberOfRows: 0 numberOfColumns: 0];
																		
	[matrix setIntercellSpacing: matrixIntercellSpace];
	[matrix setAutoscroll: YES];
	[matrix setTarget: self];
	[matrix setAction: @selector(doClick:)];
	[matrix setDoubleAction: @selector(doDoubleClick:)];

	// set new col matrix and release old
	[bc setColumnMatrix: matrix];
	[sc setDocumentView: matrix];

	// Tell the delegate to create the rows
	[browserDelegate gwbrowser: self
				 createRowsForColumn: column
			  					  inMatrix: matrix];

	cell = [matrix cellAtRow: 0  column: 0]; 
    
  if (cell != nil) {
		[matrix setCellSize: [cell cellSize]];
	}
	
  [sc setNeedsDisplay: YES];
  [bc setIsLoaded: YES];
	
	RELEASE (matrix);
}

- (ColumnIcon *)addIconForColumn:(int)column
{
	GWBrowserColumn *bc;
	ColumnIcon *icon = nil;

	if (!(bc = [browserColumns objectAtIndex: column])) {
		return nil;
	}

	icon = [[ColumnIcon alloc] init];		
	[icon setDelegate: self];
	[bc setColumnIcon: icon];
	[self addSubview: icon];
	[self addSubview: [icon label]]; 
	RELEASE (icon);

	return icon;
}

- (void)updateIconOfColumn:(int)column
{
	GWBrowserColumn *bc;
	ColumnIcon *icon;
		
	if ((column > ([browserColumns count] -1)) 
							|| (!(bc = [browserColumns objectAtIndex: column]))) {
		return;
	}
	
	icon = [bc columnIcon];

	if (icon != nil) {
  	if (column == 0) {
    	[icon setPaths: [NSArray arrayWithObject: basePath]];			
  	} else {  
			NSArray *selection = [self selectionInColumn: column - 1];
			if (selection != nil) {
    		[icon setPaths: selection];
  		}
		}
	} 
}

- (void)setLeafIconAtColumn:(int)column
{
	GWBrowserColumn *bc;
	NSMatrix *matrix;
	ColumnIcon *icn;
  int i;
         																									
  for (i = 0; i < column; i++) {
		if (!(bc = [browserColumns objectAtIndex: i])) {
			continue;
		}
		if (!(icn = [bc columnIcon])) {
			continue;
		}		
		if ([icn isDimmed] == NO) {
			[icn setBranch: YES];
		}
	}

	if ((bc = [browserColumns objectAtIndex: column])) {
		if ((icn = [bc columnIcon])) {
			[icn setBranch: NO];
			[icn select];
			
			if ((matrix = [self matrixInColumn: column])) {
				[matrix deselectAllCells];
			}
		}
	}
	
	column++;
	
  for (i = column; i < [browserColumns count]; i++) {
		if (!(bc = [browserColumns objectAtIndex: i])) {
			continue;
		}
		if (!(icn = [bc columnIcon])) {
			continue;
		}		
		[icn setDimmed: YES];
		[self setNeedsDisplayInRect: [[icn label] frame]];
	}
}

- (int)columnOfIcon:(ColumnIcon *)icon
{
	GWBrowserColumn *bc;
	ColumnIcon *icn = nil;
	int i;
		
  for (i = 0; i < [browserColumns count]; i++) {
		if (!(bc = [browserColumns objectAtIndex: i])) {
			continue;
		}
		if (!(icn = [bc columnIcon])) {
			continue;
		}
		
		if (icn == icon) {
			return i;
		}
	}
	
	return -1;
}

- (void)unloadFromColumn:(int)column
{
  GWBrowserColumn *bc; 
	NSScrollView *sc;
	ColumnIcon *icn;
  int i, count, num;
				
  // Unloads columns.
  count = [browserColumns count];
	num = [self numberOfVisibleColumns];
	
  for (i = column; i < count; ++i) {
		bc = [browserColumns objectAtIndex: i];
		sc = [bc columnScrollView];

		if ([bc isLoaded]) {
	  	if (sc) {
	      [sc setDocumentView: nil];
	      [sc setNeedsDisplay: YES];
	    }

			icn = [bc columnIcon];
			if (icn != nil) {
				[icn setPaths: nil];
				[self setNeedsDisplayInRect: [[icn label] frame]];
			}
			
	  	[bc setIsLoaded: NO];
		}
		
		if (i >= num) {
			if (icn != nil) {
				NSTextField *label = [icn label];
				[label setDelegate: nil];
				[label setEditable: NO];
  			[label removeFromSuperview];
				[icn removeFromSuperview];
			}
	  	[sc removeFromSuperview];
	  	[browserColumns removeObject: bc];
	  	count--;
	  	i--;
		}
  }
  
  if (column == 0) {
		isLoaded = NO;
	}
  
  // Scrolls if needed.
  if (column <= lastVisibleColumn) {
		[self scrollColumnsLeftBy: lastVisibleColumn - column + 1];
	}
	
  [self updateScroller];
}

- (int)columnOfMatrix:(NSMatrix *)matrix
{
  GWBrowserColumn *bc;
  int i, count;

  count = [browserColumns count];

  for (i = 0; i < count; ++i) {
		if (!(bc = [browserColumns objectAtIndex: i])) {
			continue;
		}
		if (matrix == [bc columnMatrix]) {
      return i;
		}		
  }

  return -1;
}

- (int)columnWithPath:(NSString *)path
{
  int i;
 
  for (i = 0; i <= [self lastColumn]; i++) {
    NSString *s = [self pathToColumn: i];
    if ([path isEqualToString: s]) {    
      return i;
    }
  }
  return -1;
}

- (int)lastValidColumn
{
	GWBrowserColumn *bc;
	ColumnIcon *icn;
	NSString *type;
	int i;
	
  for (i = 0; i < [browserColumns count]; i++) {
		if (!(bc = [browserColumns objectAtIndex: i])) {
			continue;
		}
		if (!(icn = [bc columnIcon])) {
			continue;
		}
		if (![icn isSelect]) {
			continue;
		}
		
		type = [icn type];
		
  	if ((type == NSDirectoryFileType) || (type == NSFilesystemFileType)) {					
			return i;		
  	}
		
  	if (type == NSApplicationFileType) {		
			if ([(BrowserViewer *)browserDelegate viewsApps] == YES) {
				return i;		
			}
			return i - 1;
  	}

		return i - 1;
	}
	
	return -1;
}

- (void)selectRow:(int)row inColumn:(int)column 
{
  NSMatrix *matrix;
  GWBrowserCell *cell;

  if (column < 0 || column > lastColumnLoaded) {
    return;
	}
	
  if (!(matrix = [self matrixInColumn: column])) {
    return;
	}
	
  if ((cell = [matrix cellAtRow: row column: 0])) {
		if (column < lastColumnLoaded) {
	  	[self setLastColumn: column];
		}

		[matrix deselectAllCells];
		[matrix selectCellAtRow: row column: 0];

		if (![cell isLeaf]) {
	  	[self addColumn];
		}
	}
}

- (GWBrowserCell *)loadedCellAtRow:(int)row column:(int)column
{
	GWBrowserColumn *bc;
  NSArray *columnCells;
  id matrix;
  int count = [browserColumns count];
  GWBrowserCell *cell;

  if (column >= count) {
		return nil;
	}

  bc = [browserColumns objectAtIndex: column];

  if (!(matrix = [bc columnMatrix])) {
		return nil;
	}

  if (!(columnCells = [matrix cells])) {
		return nil;
	}

  count = [columnCells count];

  if (row >= count) {
		return nil;
	}

  if (!(cell = [matrix cellAtRow: row column: 0])) {
		return nil;
	}

  if ([cell isLoaded]) {
		return cell;
	} else {
		[cell setLoaded: YES];
	}

	return cell;
}

- (ColumnIcon *)lastIcon
{
	GWBrowserColumn *bc;
	ColumnIcon *icn;
	int i;
	
  for (i = [browserColumns count] - 1; i >= 0; i--) {
		if (!(bc = [browserColumns objectAtIndex: i])) {
			continue;
		}
		if (!(icn = [bc columnIcon])) {
			continue;
		}
		
		if (![icn isSelect]) {
			continue;
		}
		
		if(([icn isDimmed] == NO) && ([icn isBranch] == NO)) {
			return icn;
		}
	}
	
	return nil;
}

- (NSPoint)lastIconPosition
{
	ColumnIcon *icon = [self lastIcon];
	
	if (icon != nil) {
		NSPoint p = [icon frame].origin;
		NSSize s = [icon iconShift];
		
		return NSMakePoint(p.x + s.width, p.y + s.height);
	}
	
	return NSMakePoint(0, 0);
}

- (void)emptyMatrix:(NSMatrix *)matrix inColumn:(int)column
{
  NSArray *cellList = [matrix cells];
  int count;
  			
  if([cellList count] > 0) { 
    while (1) {
      count = [[matrix cells] count];
      if (count == 0) {
        break;
      }
      [matrix removeRow: count - 1];
    }
  }
}

- (void)addCellsWithNames:(NSArray *)names 
                 inColumn:(int)column 
                   dimmed:(BOOL)dimm
{
  NSMatrix *matrix;
	GWBrowserCell *cell;
  NSString *colpath;
  NSArray *selection;
	NSMutableDictionary *sortDict;
	int stype;
  BOOL isdir;
  BOOL empty = NO;
  int i;
		
  matrix = [self matrixInColumn: column];
  selection = [matrix selectedCells];
  colpath = [self pathToColumn: column];
  
  for (i = 0; i < [names count]; i++) {
    NSString *name = [names objectAtIndex: i];
    NSString *cellpath = [colpath stringByAppendingPathComponent: name];
    
		[[NSFileManager defaultManager] fileExistsAtPath: cellpath isDirectory: &isdir];
    
		cell = [self cellWithTitle: name inColumn: column];    
    if (cell == nil) {
    
      if ([[matrix cells] count] == 0) {
        empty = YES;
      }
      
      [matrix addRow];
      cell = [matrix cellAtRow: [[matrix cells] count] -1 column: 0];
      [cell setStringValue: name];
			
      if (isdir == YES) {
		    [cell setLeaf: NO];
      } else {
		    [cell setLeaf: YES];
      }

      [cell setLoaded: YES];
      [cell setEnabled: !dimm];
			
    } else {
      [cell setEnabled: !dimm];
    }
  }
  
  [matrix sizeToCells];  
		
	stype = [browserDelegate getSortTypeForDirectoryAtPath: colpath];			
	sortDict = [NSMutableDictionary dictionaryWithCapacity: 1];
	[sortDict setObject: colpath forKey: @"path"];
	[sortDict setObject: [NSString stringWithFormat: @"%i", stype] forKey: @"type"];
  [matrix sortUsingFunction: (int (*)(id, id, void*))compCells context: (void *)sortDict];

	[self selectCells: selection inColumn: column sendAction: NO];
  
  if (empty == YES) {
    [self reloadColumn: column];
  }
  
  [matrix setNeedsDisplay: YES]; 
}

- (void)removeCellsWithNames:(NSArray *)names inColumn:(int)column
{
  NSMatrix *matrix;
	GWBrowserCell *cell;
	NSMutableArray *selection;
  int i;
	
  matrix = [self matrixInColumn: column]; 
  selection = [NSMutableArray arrayWithCapacity: 1];
  [selection addObjectsFromArray: [matrix selectedCells]];
 
  for (i = 0; i < [names count]; i++) {
    cell = [self cellWithTitle: [names objectAtIndex: i] inColumn: column];

    if (cell != nil) {    
			int row, col;
			
			if ([selection containsObject: cell]) {
				[selection removeObject: cell];
			}
      [matrix getRow: &row column: &col ofCell: cell];  
      [matrix removeRow: row];    			
    }
  }

  [matrix sizeToCells];
	
	if ([selection count] > 0) {
		[self selectCells: selection inColumn: column sendAction: NO];
	}
	
  [matrix setNeedsDisplay: YES];   
}

- (void)displayAllColumns
{	
  [self tile];
}

- (int)firstVisibleColumn
{
  return firstVisibleColumn;
}

- (BOOL)isLoaded
{
  return isLoaded;
}

- (int)lastColumn
{
  return lastColumnLoaded;
}

- (int)lastVisibleColumn
{
  return lastVisibleColumn;
}

- (void)loadColumnZero
{				
  [self setLastColumn: -1];
  [self addColumn];
  [self remapColumnSubviews: YES];
	[self updateIconOfColumn: 0];	
	[self setLeafIconAtColumn: 0];
}

- (int)numberOfVisibleColumns
{
  int num = lastVisibleColumn - firstVisibleColumn + 1;
  return (num > 0 ? num : 1);
}

- (void)reloadColumn:(int)column
{
  NSArray *selectedCells;
  NSMatrix *matrix;
  int i, count, max;
  int *selectedIndexes = NULL;		
		
  // Make sure the column even exists
  if (column > lastColumnLoaded) {
    return;
	}
	
  // Save the index of the previously selected cells
  matrix = [self matrixInColumn: column];
  selectedCells = [matrix selectedCells];
  count = [selectedCells count];

  if (count > 0) {
		selectedIndexes = NSZoneMalloc (NSDefaultMallocZone (), sizeof (int) * count);

		for (i = 0; i < count; i++) {
	  	NSCell *cell = [selectedCells objectAtIndex: i];
	  	int sRow, sColumn;
	  
	  	[matrix getRow: &sRow  column: &sColumn  ofCell: cell];
	  	selectedIndexes[i] = sRow;
		}
	}
  
  // Perform the data load
  [self performLoadOfColumn: column];
  [self adjustMatrixOfColumn: column];

  // Restore the selected cells
  if (count > 0) {
		matrix = [self matrixInColumn: column];
		max = [matrix numberOfRows];

		for (i = 0; i < count; i++) {
	  	// Abort when it stops making sense
	  	if (selectedIndexes[i] > max) {
	      break;
	    }
	  
	  	[matrix selectCellAtRow: selectedIndexes[i]  column: 0];
		}

		NSZoneFree (NSDefaultMallocZone (), selectedIndexes);
  }

  // set last column loaded
  [self setLastColumn: column];
}

- (int)selectedColumn
{
  int i;
  GWBrowserColumn *bc;
	NSMatrix *matrix;

  for (i = lastColumnLoaded; i >= 0; i--) {
		if (!(bc = [browserColumns objectAtIndex: i])) {
      continue;
		}
			
    if (![bc isLoaded] || !(matrix = [bc columnMatrix])) {
      continue;
		}
		
    if ([matrix selectedCell]) {
      return i;
		}
  }
  
  return -1;
}

- (void)setLastColumn:(int)column
{
  if (column < -1) {
    column = -1;
	}
	
  lastColumnLoaded = column;
  [self unloadFromColumn: column + 1];
}

- (NSMatrix *)matrixInColumn:(int)column
{
  GWBrowserColumn *bc;

  if (column == -1) {
		return nil;
	}

	bc = [browserColumns objectAtIndex: column];
  
  if (![bc isLoaded]) {
		return nil;
	}

  return [bc columnMatrix];
}

- (GWBrowserCell *)selectedCell
{
  NSMatrix *matrix;
  int i;

  if ((i = [self selectedColumn]) == -1) {
		return nil;
	}
  
  if (!(matrix = [self matrixInColumn: i])) {
  	return nil;
	}

  return [matrix selectedCell];
}

- (GWBrowserCell *)selectedCellInColumn: (int)column
{
  NSMatrix *matrix;

  if (!(matrix = [self matrixInColumn: column])) {
		return nil;
	}

  return [matrix selectedCell];
}

- (GWBrowserCell *)cellWithTitle:(NSString *)title inColumn:(int)column
{
  NSMatrix *matrix;
  NSArray *cells;
  id cell;
  int i;
  
  if (!(matrix = [self matrixInColumn: column])) {
		return nil;
	}
  cells = [matrix cells]; 

	for (i = 0; i < [cells count]; i++) {
    cell = [cells objectAtIndex: i];                  
		if ([[cell stringValue] isEqualToString: title]) {    
      return cell;
    }
  }
  
  return nil;
}

- (NSArray *)selectionInColumn:(int)column
{
  NSArray *selection = nil;
  NSString *columnPath = [self pathToColumn: column];
  NSMatrix *matrix;
  int i;

  matrix = [self matrixInColumn: column];

  if (matrix != nil) {
    NSMutableArray *a = [NSMutableArray array];
    NSString *cellpath;
    NSArray *selected = [matrix selectedCells];
    
		if (selected == nil) {
			return nil;
		}
		
    for (i = 0; i < [selected count]; i++) {    
      NSString *cellname = [[selected objectAtIndex: i] stringValue];  
      cellpath = [columnPath stringByAppendingPathComponent: cellname];    
      [a addObject: cellpath];
    }
    
    selection = [NSArray arrayWithArray: a];
  }

	if ([selection count] > 0) {
  	return selection;
	}
	
	return nil;
}

- (NSArray *)selectedCells
{
  NSMatrix *matrix;
  int i;

  // Nothing selected
  if ((i = [self selectedColumn]) == -1) {
		return nil;
	}
  
  if (!(matrix = [self matrixInColumn: i])) {
		return nil;
	}

  return [matrix selectedCells];
}

- (void)selectCellWithTitle:(NSString *)title 
                   inColumn:(int)column 
                 sendAction:(BOOL)act
{
  NSMatrix *matrix;
  GWBrowserCell *cell;
  int row, col;
	
  if (!(cell = [self cellWithTitle: title inColumn: column])) {
		return;
	}
		
	matrix = [self matrixInColumn: column];
  [matrix getRow: &row column: &col ofCell: cell];      
  [matrix deselectAllCells];
  [matrix selectCellAtRow: row column: 0];
  [matrix scrollCellToVisibleAtRow: row column: 0];
	
  if (act) {
    [matrix sendAction];
  }
}

- (void)selectCells:(NSArray *)cells 
           inColumn:(int)column 
      	 sendAction:(BOOL)act								 
{
  NSMatrix *matrix;
  int i, count, max;
  int *selectedIndexes = NULL;		
			
  matrix = [self matrixInColumn: column];
  count = [cells count];
	max = [matrix numberOfRows];
	selectedIndexes = NSZoneMalloc(NSDefaultMallocZone(), sizeof(int) * count);

	for (i = 0; i < count; i++) {
	  NSCell *cell;
	  int sRow, sColumn;
		
		cell = [cells objectAtIndex: i];
	  [matrix getRow: &sRow column: &sColumn ofCell: cell];
	  selectedIndexes[i] = sRow;
	}
  
	for (i = 0; i < count; i++) {
	  if (selectedIndexes[i] > max) {
	    break;
	  }
	  [matrix selectCellAtRow: selectedIndexes[i] column: 0];
	}

  if ((act) && ([cells count])) {
    [matrix sendAction];
  }

	NSZoneFree(NSDefaultMallocZone(), selectedIndexes);
}

- (void)setMultipleSelection:(NSArray *)titles 
                    inColumn:(int)column 
                  sendAction:(BOOL)act
{
	NSMatrix *matrix;
	NSMutableArray *selcells;
	int i;
		
  if (!(matrix = [self matrixInColumn: column])) {
		return;
	}
	
	selcells = [NSMutableArray arrayWithCapacity: 1];	

	for (i = 0; i < [titles count]; i++) {
		NSString *title = [titles objectAtIndex: i];
		GWBrowserCell *cell = [self cellWithTitle: title inColumn: column];
	
		if (cell != nil) {
  		[selcells addObject: cell];
		}
	}

	[matrix deselectAllCells];
	[self selectCells: selcells inColumn: column sendAction: act];							 
}

- (NSString *)pathToColumn:(int)column
{
	NSMutableString	*s = [basePath mutableCopy];
  unsigned i;
  NSString *string;
		
  if (column > lastColumnLoaded) {
		column = lastColumnLoaded + 1;
	}

  for (i = 0; i < column; ++i) {
		id c = [self selectedCellInColumn: i];

		if (c == nil) {
      return nil;
    }
		
		if ((i != 0) || ([basePath isEqualToString: pathSeparator] == NO)) {
	  	[s appendString: pathSeparator];
		} 
		
		string = [c stringValue];
      
		if (string == nil) {
	  	/* This should happen only when c == nil, in which case it
	      doesn't make sense to go with the path */
	  	break;
			
		} else {
	  	[s appendString: string];	  
		}
	}
	
  return AUTORELEASE (s);
}

- (NSRect)frameOfColumn:(int)column
{
  NSRect r = NSZeroRect;
  int n;

  // Number of columns over from the first
  n = column - firstVisibleColumn;

  // Calculate the frame
  r.size = columnSize;
  r.origin.x = n * columnSize.width;

  // Padding : columnSize.width is rounded in "tile" method
  if (column == lastVisibleColumn) {
		r.size.width = [self frame].size.width - r.origin.x;
	}

  if (r.size.width < 0) {
		r.size.width = 0;
	}
	
  if (r.size.height < 0) {
  	r.size.height = 0;
  }

  return r;
}

- (void)selectAllInLastColumn
{
  NSMatrix *matrix;
	
  if (!(matrix = [self matrixInColumn: [self lastValidColumn]])) {
		return;
	}
		
  [matrix selectAll: nil];
	[matrix sendAction];
}

//
// Arranging an Browser's Components 
//

- (void)setMaxVisibleColumns:(int)columnCount
{
  int i, delta;
	
  if ((columnCount < 1) || (maxVisibleColumns == columnCount)) {
    return;
	}
		
  // Scroll left as needed
  delta = columnCount - maxVisibleColumns;
	
  if ((delta > 0) && (lastVisibleColumn <= lastColumnLoaded)) {
		firstVisibleColumn = (firstVisibleColumn - delta > 0) ?
																	(firstVisibleColumn - delta) : 0;
	}
  
  // Adds columns as needed
  for (i = [browserColumns count]; i < columnCount; i++) {
		[self createColumn];
	}

  // Sets other variables
  lastVisibleColumn = firstVisibleColumn + columnCount - 1;
  maxVisibleColumns = columnCount;

  // Redisplay
  [self tile];
  [self updateScroller];
  [self remapColumnSubviews: YES];
  [self setNeedsDisplay: YES];
}

- (int)maxVisibleColumns
{
  return maxVisibleColumns;
}

- (void)setMinColumnWidth:(int)columnWidth
{
  float sw;
	
  sw = scrollerWidth;
  // Take the border into account

  // Column width cannot be less than scroller and border
  if (columnWidth < sw) {
    minColumnWidth = sw;
  } else {
    minColumnWidth = columnWidth;
	}
	
  [self tile];
}

- (int)minColumnWidth
{
  return minColumnWidth;
}

- (void)tile
{
  NSSize bs = BEZEL_BORDER_SIZE;
  GWBrowserColumn *bc;
	NSScrollView *sc;
	ColumnIcon *icn;
  int i, num;
		
  columnSize.height = [self frame].size.height;
  
  // Horizontal scroller
	scrollerRect.origin.x = bs.width;
	scrollerRect.origin.y = [self frame].size.height - (iconsPathWidth + scrollerWidth);
  scrollerRect.size.width = [self frame].size.width - (2 * bs.width);
	scrollerRect.size.height = scrollerWidth;

	columnSize.height -= iconsPathWidth + scrollerWidth + (3 * bs.height) + NSBR_VOFFSET;

  if (!NSEqualRects(scrollerRect, [horizontalScroller frame])) {
		[horizontalScroller setFrame: scrollerRect];
	}  

  // Columns
	num = maxVisibleColumns;
	
	columnSize.width = (int)([self frame].size.width / (float)num);    

  if (columnSize.height < 0) {
		columnSize.height = 0;
	}  

  for (i = firstVisibleColumn; i <= lastVisibleColumn; i++) {
		NSRect colRect, iconRect;
		
		if (!(bc = [browserColumns objectAtIndex: i])) {
	  	return;
		}
		
		if (!(sc = [bc columnScrollView])) {
	  	return;
		}
		
		colRect = [self frameOfColumn: i];
		[sc setFrame: colRect];
		
		icn = [bc columnIcon];
		
		if (icn != nil) {		
			iconRect = NSMakeRect(colRect.origin.x, 
											scrollerRect.origin.y + scrollerWidth + 19, 
																								colRect.size.width, 52);
			[icn setFrame: iconRect];
		}
		
		[self adjustMatrixOfColumn: i];
	}
}

- (void)remapColumnSubviews:(BOOL)fromFirst
{
  GWBrowserColumn *bc;
	NSScrollView *sc;
	ColumnIcon *icn;
  int i, count;
	id window;
  id firstResponder = nil;	
  BOOL setFirstResponder = NO;
																
  window = [self window];
  count = [browserColumns count];
  for (i = 0; i < count; i++) {
		bc = [browserColumns objectAtIndex: i];
		sc = [bc columnScrollView];
		icn = [bc columnIcon];
		
		if (!firstResponder && ((id)[bc columnMatrix] == (id)[window firstResponder])) {
			firstResponder = [bc columnMatrix];
		}
				
    if (sc) {		
			[sc removeFromSuperviewWithoutNeedingDisplay];
		}
		
    if (icn) {
			[icn removeFromSuperviewWithoutNeedingDisplay];
			[[icn label] removeFromSuperviewWithoutNeedingDisplay];
		}		
	}

  if (firstVisibleColumn > lastVisibleColumn) {
    return;
	}
	
  // Sets columns subviews order according to fromFirst (display order...).
  // All added subviews are automaticaly marked as needing display (-> NSView).
  if (fromFirst) {
		for (i = firstVisibleColumn; i <= lastVisibleColumn; i++) {
	  	bc = [browserColumns objectAtIndex: i];
	  	sc = [bc columnScrollView];
			icn = [bc columnIcon];
			
	  	[self addSubview: sc];
			
			if (icn) {
				[self addSubview: icn];
				[self addSubview: [icn label]]; 
				[self setNeedsDisplayInRect: [icn frame]];
				[self setNeedsDisplayInRect: [[icn label] frame]];
			}
			
	  	if ([bc columnMatrix] == firstResponder) {
	      [window makeFirstResponder: firstResponder];
	      setFirstResponder = YES;
	    }
		}

		if (firstResponder && setFirstResponder == NO) {
			[window makeFirstResponder:
		    		[[browserColumns objectAtIndex: firstVisibleColumn] columnMatrix]];
		}		 
				 
	} else {
		for (i = lastVisibleColumn; i >= firstVisibleColumn; i--) {
	  	bc = [browserColumns objectAtIndex: i];
	  	sc = [bc columnScrollView];
			icn = [bc columnIcon];
			
	  	[self addSubview: sc];
			
			if (icn) {
				[self addSubview: icn];
				[self addSubview: [icn label]]; 
				[self setNeedsDisplayInRect: [icn frame]];
				[self setNeedsDisplayInRect: [[icn label] frame]];
			}
			
	  	if ([bc columnMatrix] == firstResponder) {
	      [window makeFirstResponder: firstResponder];
	      setFirstResponder = YES;
	    }
		}

		if (firstResponder && setFirstResponder == NO) {
			[window makeFirstResponder:
		    		[[browserColumns objectAtIndex: lastVisibleColumn] columnMatrix]];
		}
  }
}

- (void)adjustMatrixOfColumn:(int)column
{
  GWBrowserColumn	*bc;
  NSScrollView *sc;
  NSMatrix *matrix;
  NSSize cs, ms;
	
  if (column >= (int)[browserColumns count]) {
		return;
	}

  if (!(bc = [browserColumns objectAtIndex: column])) {
		return;
	}
		
  sc = [bc columnScrollView];
  matrix = [bc columnMatrix];

  // Adjust matrix to fit in scrollview if column has been loaded
  if (sc && matrix && [bc isLoaded]) {
		cs = [sc contentSize];
		ms = [matrix cellSize];
		ms.width = cs.width;
		[matrix setCellSize: ms];
		[sc setDocumentView: matrix];
	}
}

- (void)setShift:(int)s
{
  int i, column = [self lastColumn];
    			
  for (i = 0; i <= s; i++) {  
    [self addColumn];
  }
	
	currentshift = s;  
  [self setLastColumn: column + s];
  [self scrollColumnsRightBy: s];
  [self tile];
  [self remapColumnSubviews: YES];
}

- (int)currentshift
{
  return currentshift;
}

- (void)scrollColumnsLeftBy:(int)shiftAmount
{	
  // Cannot shift past the zero column
  if ((firstVisibleColumn - shiftAmount) < 0) {
    shiftAmount = firstVisibleColumn;
	}
	
  // No amount to shift then nothing to do
  if (shiftAmount <= 0) {
    return;
	}
		
  // Shift
  firstVisibleColumn = firstVisibleColumn - shiftAmount;
  lastVisibleColumn = lastVisibleColumn - shiftAmount;

  // Update the scroller
  [self updateScroller];

  // Update the scrollviews
  [self tile];
  [self remapColumnSubviews: YES];
}

- (void)scrollColumnsRightBy:(int)shiftAmount
{	
  // Cannot shift past the last loaded column
  if ((shiftAmount + lastVisibleColumn) > lastColumnLoaded) {
    shiftAmount = lastColumnLoaded - lastVisibleColumn;
	}
	
  // No amount to shift then nothing to do
  if (shiftAmount <= 0) {
    return;
	}
		
  // Shift
  firstVisibleColumn = firstVisibleColumn + shiftAmount;
  lastVisibleColumn = lastVisibleColumn + shiftAmount;

  // Update the scroller
  [self updateScroller];

  // Update the scrollviews
  [self tile];
  [self remapColumnSubviews: NO];
}

- (void)scrollColumnToVisible:(int)column
{
  int i;
	
  // If its the last visible column then we are there already
  if (lastVisibleColumn == column) {
    return;
	}
	
  // If there are not enough columns to scroll with
  // then the column must be visible
  if (lastColumnLoaded + 1 <= maxVisibleColumns) {
    return;
	}
		
  i = lastVisibleColumn - column;
  if (i > 0) {
    [self scrollColumnsLeftBy: i];
  } else {
    [self scrollColumnsRightBy: (-i)];
	}
}

- (void)scrollViaScroller:(NSScroller *)sender
{
  NSScrollerPart hit;
	
  if ([sender class] != [NSScroller class]) {
    return;
  }
		
  hit = [sender hitPart];
  
  switch (hit) {
		// Scroll to the left
		case NSScrollerDecrementLine:
		case NSScrollerDecrementPage:
			[self scrollColumnsLeftBy: 1];
			
			if (currentshift > 0) {
				[self setLastColumn: [self lastColumn] - currentshift];
				[self setShift: currentshift - 1];
			}
			
			break;
      
		// Scroll to the right
		case NSScrollerIncrementLine:
		case NSScrollerIncrementPage:
			[self scrollColumnsRightBy: 1];
			break;
      
		// The knob or knob slot
		case NSScrollerKnob:
		case NSScrollerKnobSlot: 
			{
				int num = [self numberOfVisibleColumns];
	  		float f = [sender floatValue];
	  		float n = lastColumnLoaded + 1 - num;
				
	  		skipUpdateScroller = YES;
	  		[self scrollColumnToVisible: rintf(f * n) + num - 1];
	  		skipUpdateScroller = NO;
						
				[self setLastColumn: [self lastColumn] - currentshift];  
				currentshift = 0;
			}
      break;
      
		// NSScrollerNoPart ???
		default:
			break;
	}
}

- (void)updateScroller
{
  int num = [self numberOfVisibleColumns];
	
  // If there are not enough columns to scroll with
  // then the column must be visible
  if ((lastColumnLoaded == 0) || (lastColumnLoaded <= (num - 1))) {
		[horizontalScroller setEnabled: NO];
	
	} else {
		if (!skipUpdateScroller) {
			float prop = (float)num / (float)(lastColumnLoaded + 1);
			float i = lastColumnLoaded - num + 1;
			float f = 1 + ((lastVisibleColumn - lastColumnLoaded) / i);
			[horizontalScroller setFloatValue: f knobProportion: prop];
		}
	
		[horizontalScroller setEnabled: YES];
	}

  [horizontalScroller setNeedsDisplay: YES];
}


//
// Events & Actions
//

- (void)keyDown:(NSEvent *)theEvent
{
	NSString *characters;
	unichar character;
	NSMatrix *matrix;
	int column;
	
	if ((column = [self selectedColumn]) == -1) {
		return;
	}
	if (!(matrix = [self  matrixInColumn: column])) {
		return;
	}
	
	characters = [theEvent characters];
	character = 0;
		
  if ([characters length] > 0) {
		character = [characters characterAtIndex: 0];
	}
		
	switch (character) {
		case NSUpArrowFunctionKey:
		case NSDownArrowFunctionKey:
	  	return;
	
		case NSLeftArrowFunctionKey:
			{
				if ([theEvent modifierFlags] & NSControlKeyMask) {
	      	[super keyDown: theEvent];
	    	} else {
	    		[self moveLeft: self];
				}
			}
      return;
	
		case NSRightArrowFunctionKey:
			{
				if ([theEvent modifierFlags] & NSControlKeyMask) {
	      	[super keyDown: theEvent];
	    	} else {
	    		[self moveRight: self];
				}
			}
	  	return;

		case 13:
      [matrix sendDoubleAction];
      return;
	
		case NSTabCharacter:
	  	{
	    	if ([theEvent modifierFlags] & NSShiftKeyMask)
	      	[[self window] selectKeyViewPrecedingView: self];
	    	else
	      	[[self window] selectKeyViewFollowingView: self];
	  	}
			return;
	  	break;
	} 

  if ((character < 0xF700) && ([characters length] > 0)) {
		GWBrowserColumn *bc;
		NSMatrix *matrix;
		NSString *sv;
		int i, n, s;
		int selectedColumn;
		SEL lcarcSel = @selector(loadedCellAtRow:column:);
		IMP lcarc = [self methodForSelector: lcarcSel];
														
		selectedColumn = [self selectedColumn];
		if(selectedColumn != -1) {
	  	bc = [browserColumns objectAtIndex: selectedColumn];
	  	matrix = [bc columnMatrix];
	  	n = [matrix numberOfRows];
	  	s = [matrix selectedRow];
						
	  	if (charBuffer == nil) {
	      charBuffer = [characters substringToIndex: 1];
	      RETAIN(charBuffer);
	    } else {
	      if (([theEvent timestamp] - lastKeyPressed < 2000.0)
		  											&& (alphaNumericalLastColumn == selectedColumn)) {
		  		ASSIGN (charBuffer, ([charBuffer stringByAppendingString:
				    																[characters substringToIndex: 1]]));
				} else {
		  		ASSIGN (charBuffer, ([characters substringToIndex: 1]));
				}														
			}
			
			alphaNumericalLastColumn = selectedColumn;
			lastKeyPressed = [theEvent timestamp];
			
	  	sv = [((*lcarc)(self, lcarcSel, s, selectedColumn)) stringValue];

	  	if (([sv length] > 0) && ([sv hasPrefix: charBuffer])) {
	    	return;
	  	}
			
	  	for (i = s + 1; i < n; i++) {
	      sv = [((*lcarc)(self, lcarcSel, i, selectedColumn)) stringValue];
	      
				if (([sv length] > 0) && ([sv hasPrefix: charBuffer])) {
		  		[self selectRow: i inColumn: selectedColumn];	
		  		[matrix scrollCellToVisibleAtRow: i column: 0];
		  		[matrix performClick: self];
		  		return;
				}
	    }
	  	
			for (i = 0; i < s; i++) {
	      sv = [((*lcarc)(self, lcarcSel, i, selectedColumn)) stringValue];
	      
				if (([sv length] > 0) && ([sv hasPrefix: charBuffer])) {
		  		[self selectRow: i
					inColumn: selectedColumn];
		  		[matrix scrollCellToVisibleAtRow: i column: 0];
		  		[matrix performClick: self];
		  		return;
				}
			}
		}
		
		lastKeyPressed = 0.;			
	}
	
  [super keyDown: theEvent];
}

- (void)mouseDown:(NSEvent *)theEvent
{
	// Override NSControl handler (prevents highlighting).
}

- (void)doClick:(id)sender
{
  NSArray *selectedCells;
	NSString *columnPath;
	NSMutableArray *cellsPaths;
  int i, pos, column, selectedCellsCount;
  BOOL mustshift = ([self firstVisibleColumn] > 0) ? YES : NO;
				
  if ([sender class] != [NSMatrix class]) {
    return;
	}
		
  column = [self columnOfMatrix: sender];
  // If the matrix isn't ours then just return
  if (column == -1) {
    return;
	}
	
  selectedCells = [sender selectedCells];
  selectedCellsCount = [selectedCells count];
  if (selectedCellsCount == 0) {
    return;
	}

	columnPath = [self pathToColumn: column];
  cellsPaths = [NSMutableArray arrayWithCapacity: 1];
	  
  for (i = 0; i < [selectedCells count]; i++) {
    NSString *cellname = [[selectedCells objectAtIndex: i] stringValue];  
    NSString *cellpath = [columnPath stringByAppendingPathComponent: cellname];
    [cellsPaths addObject: cellpath];
  }

	[browserDelegate currentSelectedPaths: cellsPaths];		

  pos = column - [self firstVisibleColumn] + 1;  
  currentshift = 0;

	[self setLastColumn: column];
	[self addColumn];
	
	[self updateIconOfColumn: column + 1];
	[self setLeafIconAtColumn: column + 1];
	
	[sender scrollCellToVisibleAtRow: [sender selectedRow] column: column];
		
  if ((mustshift == YES) && (pos < ([self maxVisibleColumns] - 1))) {
		NSMatrix *matrix; 

		[self setShift: [self maxVisibleColumns] - pos - 1];

		matrix = [self matrixInColumn: [self selectedColumn]]; 
		if (matrix != nil) { 
			[[self window] makeFirstResponder: matrix]; 
		}
	}

  [self sendAction];
}

- (void)doDoubleClick:(id)sender
{
  NSArray *paths = [self selectionInColumn: [self selectedColumn]];  
	[browserDelegate openSelectedPaths: paths newViewer: NO];		
}

- (void)moveLeft:(id)sender
{
	NSMatrix *matrix;
	int selectedColumn;
				
	selectedColumn = [self selectedColumn];
	
	if (selectedColumn == 0) {
		return;
	}
	
	matrix = [self matrixInColumn: selectedColumn];

	[matrix deselectAllCells];

	if((selectedColumn + 1) <= [self lastColumn]) {
	  [self setLastColumn: selectedColumn];
		[self setLeafIconAtColumn: [self lastColumn]];
	}

	matrix = [self matrixInColumn: [self selectedColumn]];
	[[self window] makeFirstResponder: matrix];

	[super sendAction: action to: target];
}

- (void)moveRight:(id)sender
{
	NSMatrix *matrix;
	BOOL selectFirstRow = NO;
	int selectedColumn;
	
	selectedColumn = [self selectedColumn];
	
	if (selectedColumn == -1) {
	  matrix = [self matrixInColumn: 0];

	  if ([[matrix cells] count]) {
			[matrix selectCellAtRow: 0 column: 0];
			[[self window] makeFirstResponder: matrix];
			[self doClick: matrix];
			selectedColumn = 0;
		}
	} else {
	  matrix = [self matrixInColumn: selectedColumn];

	  if (![[matrix selectedCell] isLeaf] && [[matrix selectedCells] count] == 1) {
	    selectFirstRow = YES;
		}
	}

	if(selectFirstRow == YES) {
		int lastvalid = [self lastValidColumn];
		
		if (lastvalid == -1) {
			return;
		}
		
	  matrix = [self matrixInColumn: lastvalid];
	  if ([[matrix cells] count]) {
	    [matrix selectCellAtRow: 0 column: 0];
	    [[self window] makeFirstResponder: matrix];
	    [self doClick: matrix];
		}
	}

	[super sendAction: action to: target];
}

- (SEL)action
{
  return action;
}

- (SEL)doubleAction
{
  return doubleAction;
}

- (BOOL)sendAction
{
  return [self sendAction: [self action]  to: [self target]];
}

- (void)setAction:(SEL)s
{
  action = s;
}

- (void)setTarget:(id)atarget
{
  target = atarget;
}

- (id)target
{
  return target;
}

- (BOOL)acceptsFirstResponder
{
  return YES;
}

- (BOOL)becomeFirstResponder
{
  NSMatrix *matrix;
  int selectedColumn;

  selectedColumn = [self selectedColumn];
  if (selectedColumn == -1) {
    matrix = [self matrixInColumn: 0];
  } else {
    matrix = [self matrixInColumn: selectedColumn];
	}
	
  if (matrix) {
    [[self window] makeFirstResponder: matrix];
	}
	
  return YES;
}

- (void)resizeSubviewsWithOldSize:(NSSize)oldSize
{
  [self tile];
}

- (void)drawRect:(NSRect)rect
{	
	NSRect scrollerBorderRect = NSMakeRect(scrollerRect.origin.x, 
			scrollerRect.origin.y,
			scrollerRect.size.width, 
			scrollerRect.size.height + iconsPathWidth - 2);
	
	NSSize bs = BEZEL_BORDER_SIZE;
  NSRectClip(rect);
	
  [[[self window] backgroundColor] set];
  NSRectFill(rect);

  // Load the first column if not already done
  if (!isLoaded) {
		[self loadColumnZero];
	}

  // Draws scroller border
	scrollerBorderRect.origin.x = 0;
	scrollerBorderRect.origin.y = scrollerRect.origin.y - 2;
	scrollerBorderRect.size.width += 2 * bs.width;
	scrollerBorderRect.size.height += 2 * bs.height;

	if ((NSIntersectsRect (scrollerBorderRect, rect) == YES) && [self window]) {
		NSPoint p1 = NSMakePoint(scrollerBorderRect.origin.x, 
						scrollerBorderRect.origin.y + scrollerRect.size.height + 3);
		NSPoint p2 = NSMakePoint(scrollerBorderRect.origin.x + scrollerBorderRect.size.width - 2, 
																scrollerBorderRect.origin.y + scrollerRect.size.height + 3);
	
		NSDrawGrayBezel (scrollerBorderRect, rect);
		
		[[NSColor blackColor] set];
	
		[NSBezierPath strokeLineFromPoint: p1 toPoint: p2];
	}  
}

@end

//
// ColumnIcon Delegate Methods
//

@implementation GWBrowser (ColumnIconDelegateMethods)

- (void)icon:(ColumnIcon *)sender setFrameOfLabel:(NSTextField *)label
{
	float icnwidth, labwidth, labxpos, labypos;
  NSRect labelRect;
  
	icnwidth = [sender frame].size.width;
 	labwidth = [label frame].size.width;
	labypos = [sender frame].origin.y - 14;

	if(icnwidth > labwidth) {
		labxpos = [sender frame].origin.x + ((icnwidth - labwidth) / 2);
	} else {
		labxpos = [sender frame].origin.x - ((labwidth - icnwidth) / 2);
    
    // AGGIUNGERE TUTTA LA ROBBA !!!!!!!
  }
  
	labelRect = NSMakeRect(labxpos, labypos, labwidth, 14);
	[label setFrame: labelRect];
  [label setNeedsDisplay: YES];
}

- (void)unselectOtherIcons:(ColumnIcon *)selicon
{
	GWBrowserColumn *bc;
	ColumnIcon *icn;
	int i;
	
  for (i = 0; i < [browserColumns count]; i++) {
		if (!(bc = [browserColumns objectAtIndex: i])) {
			continue;
		}
		if (!(icn = [bc columnIcon])) {
			continue;
		}		
		if (([icn isSelect]) && (icn != selicon)) {
			[icn unselect];
		}
	}
}

- (void)clickOnIcon:(ColumnIcon *)clicked
{
	NSArray *paths;
	id window;
	int column;
		
	column = [self columnOfIcon: clicked];

	if (column == -1) {
		return;
	}
	
	paths = [clicked paths];  
  RETAIN (paths);
	
  if (column != 0) {
    [self selectCellWithTitle: [clicked name] 
										 inColumn: column - 1 
									 sendAction: YES];
  } else {
    [self setLastColumn: column];
  }
	
	[self setLeafIconAtColumn: column];

	[browserDelegate currentSelectedPaths: paths];		
	RELEASE (paths);
	
	window = [self window];
	[window makeFirstResponder: window];
}

- (void)doubleClickOnIcon:(ColumnIcon *)clicked newViewer:(BOOL)isnew
{
	[browserDelegate openSelectedPaths: [clicked paths] newViewer: isnew];
}

- (void)performFileOperationWithDictionary:(id)opdict
{
	[browserDelegate performFileOperationWithDictionary: opdict];
}

- (NSString *)getTrashPath
{
	return [browserDelegate getTrashPath];
}

@end
