/*  -*-objc-*-
 *  IconsPath.m: Implementation of the IconsPath Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "IconsPath.h"
#include "Functions.h"
#include "PathIcon.h"
#include "GNUstep.h"

@implementation IconsPath

- (void)dealloc
{
  RELEASE (root);
  TEST_RELEASE (currentPath);
  RELEASE (icons);
  [super dealloc];
}

- (id)initWithRootAtPath:(NSString *)rpath 
        		columnsWidth:(float)cwidth
						    delegate:(id)adelegate
{
  self = [super init];
  
  if (self) {
    ASSIGN (root, rpath);
    columnsWidth = cwidth;
		[self setDelegate: adelegate];
    [self setAutoresizingMask: (NSViewWidthSizable)];
    icons = [[NSMutableArray alloc] initWithCapacity: 1];
  }
   
  return self; 
}

- (void)setIconsForSelection:(NSArray *)selection
{
  NSString *fullPath;
  NSArray *components;
  NSMutableArray *subpaths;
  NSString *path;
  PathIcon *icon;
  int i, count;

  fullPath = [NSString stringWithString: [selection objectAtIndex: 0]]; 
  subpaths = [NSMutableArray arrayWithCapacity: 1];  
  path = [NSString string];     
  components = [fullPath pathComponents];  
  
  for (i = 0; i < [components count]; i++) {
    path = [path stringByAppendingPathComponent: [components objectAtIndex: i]];
    if (subPath(path, root) == NO) {
      [subpaths addObject: path];     
    }
  }

  count = [subpaths count];
  [self renewIcons: count];  
  
  for (i = 0; i < count; i++) {
    icon = [icons objectAtIndex: i];  
    [icon setBranch: YES];
    [icon setPaths: [NSArray arrayWithObject: [subpaths objectAtIndex: i]]];
  }
	
	if (count > 0) {
  	icon = [icons objectAtIndex: count - 1];
  	[icon setBranch: NO];
  	[icon setPaths: selection];
  	[icon select];
  }
	
  [self setIconsPositions];
}

- (void)setColumnWidth:(float)width
{
  columnsWidth = width;
  [self setIconsPositions];
}

- (void)renewIcons:(int)n
{
  while ([icons count] > n) {
    [self removeIcon: [self lastIcon]];
  }
  while ([icons count] < n) {  
    [self addIcon];
  } 
}

- (void)addIcon
{
  PathIcon *icon = [[PathIcon alloc] initWithDelegate: self];
  [self addSubview: icon];
  [self addSubview: [icon label]];  
  [icons addObject: icon];
  RELEASE (icon);
  [self setIconsPositions];  
}

- (void)removeIcon:(PathIcon *)icon
{
	NSTextField *label = [icon label];
	[label setDelegate: nil];
	[label setEditable: NO];
  [label removeFromSuperview];
  [icon removeFromSuperview];
  [icons removeObject: icon];
	[self setIconsPositions];  
}

- (void)removeIconAtIndex:(int)index
{
  [self removeIcon: [self iconAtIndex: index]];
}

- (void)setIconsPositions
{
  float posx = 0.0;
	int i;
  
	for (i = 0; i < [icons count]; i++) {
		PathIcon *icon = [icons objectAtIndex: i];
    [icon setFrame: NSMakeRect(posx, 18, columnsWidth, 52)];
    [icon setNeedsDisplay: YES];
    posx += columnsWidth;
  }
    
  if (posx != [self frame].size.width) {
    [self setFrame: NSMakeRect(0, 0, posx, 70)];
  }
  
	[self setNeedsDisplay: YES];
}

- (void)setLabelRectOfIcon:(PathIcon *)icon
{
	NSTextField *label;
	float labwidth, labxpos;
  NSRect labelRect;
  
	label = [icon label];
 	labwidth = [label frame].size.width;

	if(columnsWidth > labwidth) {
		labxpos = [icon frame].origin.x + ((columnsWidth - labwidth) / 2);
	} else {
		labxpos = [icon frame].origin.x - ((labwidth - columnsWidth) / 2);
    
    // AGGIUNGERE TUTTA LA ROBBA !!!!!!!
  }
  
	labelRect = NSMakeRect(labxpos, 4, labwidth, 14);
	[label setFrame: labelRect];
  [label setNeedsDisplay: YES];
}

- (void)resizeWithOldSuperviewSize:(NSSize)oldFrameSize
{
  [self setIconsPositions];
}

- (void)unselectOtherIcons:(PathIcon *)icon
{
  int i;
  
  for (i = 0; i < [icons count]; i++) {
    PathIcon *icn = [icons objectAtIndex: i];
    if (icn != icon) {    
      [icn unselect];
    }
  }
}

- (void)selectIconAtIndex:(int)index
{
  if (index < [icons count]) {
    [[icons objectAtIndex: index] select];
  }
}

- (NSArray *)icons
{
  return icons;
}

- (PathIcon *)iconAtIndex:(int)index
{
  if (index < [icons count]) {
    return [icons objectAtIndex: index];
  }
  return nil;
}

- (int)indexOfIcon:(PathIcon *)icon
{
  int i;
  
  for (i = 0; i < [icons count]; i++) {
    PathIcon *icn = [icons objectAtIndex: i];  
    if (icn == icon) {    
      return i;
    }
  }
  
  return i;
}

- (int)indexOfIconWithPath:(NSString *)path
{
  int i;
  
  for (i = 0; i < [icons count]; i++) {
    PathIcon *icon = [icons objectAtIndex: i];
    NSArray *ipaths = [icon paths];
    if (ipaths != nil) {   
      if ([ipaths containsObject: path] == YES) {
        return i;
      }
    }
  }
  
  return -1;
}

- (PathIcon *)lastIcon
{
  return [icons objectAtIndex: [icons count] - 1];
}

- (int)numberOfIcons
{
  return [icons count];
}

- (id)delegate
{
	return delegate;
}

- (void)setDelegate:(id)anObject
{
	delegate = anObject;
}

@end

//
// PathIcon Delegate Methods
//

@implementation IconsPath (PathIconDelegateMethods)

- (void)setLabelFrameOfIcon:(id)anicon
{
	[self setLabelRectOfIcon: anicon];
}

- (void)unselectIconsDifferentFrom:(id)anicon
{
	[self unselectOtherIcons: anicon];
}

- (void)clickedIcon:(id)anicon
{
	[delegate clickedIcon: anicon];
}

- (void)doubleClickedIcon:(id)anicon newViewer:(BOOL)isnew
{
	[delegate doubleClickedIcon: anicon newViewer: isnew];
}

- (void)performFileOperationWithDictionary:(id)opdict
{
	[delegate performFileOperationWithDictionary: opdict];
}

- (NSString *)getTrashPath
{
	return [delegate getTrashPath];
}

@end
