 /*  -*-objc-*-
 *  ViewersWindow.m: Implementation of the ViewersWindow Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "ViewersWindow.h"
#include "GWSplitView.h"
#include "Shelf/Shelf.h"
#include "IconViewsIcon.h"
#include "ViewersProtocol.h"
#include "History/History.h"
#include "GWorkspace.h"
#include "Functions.h"
#include "GNUstep.h"

extern NSString *GWViewersListDidChangeNotification;
extern NSString *GWBrowserColumnWidthChangedNotification;

@implementation ViewersWindow

- (void)dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];
  RELEASE (shelf);
  RELEASE (viewer);
  RELEASE (mainview);
  RELEASE (myPath);
	RELEASE (selectedPaths);
	RELEASE (viewers);
	RELEASE (viewersPaths);
  RELEASE (viewType);
	RELEASE (ViewerHistory);
  [super dealloc];
}

- (id)initForPath:(NSString *)path 
 withViewersPaths:(NSArray *)vpaths 
			canViewApps:(BOOL)canview
{
  unsigned int style;

  if ([path isEqualToString: @"/"] == YES) {
    style = NSTitledWindowMask | NSMiniaturizableWindowMask 
                                          | NSResizableWindowMask;
  } else {
    style = NSTitledWindowMask | NSClosableWindowMask 
				         | NSMiniaturizableWindowMask | NSResizableWindowMask;
  }

  self = [super initWithContentRect: NSZeroRect styleMask: style
                               backing: NSBackingStoreBuffered defer: NO];

  if (self) {
    NSUserDefaults *defaults;
    NSMutableDictionary *viewersPrefs, *myPrefs;
    NSArray *shelfDicts;
    id dictEntry;
		float shfwidth;   
		
		gw = [GWorkspace gworkspace];
  	fm = [NSFileManager defaultManager];
		
		[self setReleasedWhenClosed: NO];
    resizeIncrement = [gw browserColumnsWidth];
    [self setMinSize: NSMakeSize(resizeIncrement * 2, 250)];    
    [self setResizeIncrements: NSMakeSize(resizeIncrement, 1)];
		
    if ([path isEqualToString: @"/"] == YES) {
      [self setTitle: NSLocalizedString(@"File Viewer", @"")];
    } else {
			NSString *pth = [path stringByDeletingLastPathComponent];
			NSString *nm = [path lastPathComponent];
			NSString *title = [NSString stringWithFormat: @"%@ - %@", nm, pth];
      [self setTitle: title];   
    }

    if ([self setFrameUsingName: [NSString stringWithFormat: @"Viewer at %@", path]] == NO) {
      [self setFrame: NSMakeRect(200, 200, resizeIncrement * 3, 500) display: NO];
    }    
		
		ASSIGN (myPath, path);  
		selectedPaths = [[NSArray alloc] initWithObjects: path, nil];
  	viewsapps = canview;
		ViewerHistory = [[NSMutableArray alloc] init];
		[ViewerHistory addObject: path];
		currHistoryPos = 0;
			
    defaults = [NSUserDefaults standardUserDefaults];	
		
    dictEntry = [defaults dictionaryForKey: @"viewersprefs"];
    if (dictEntry) { 
 		  viewersPrefs = [dictEntry mutableCopy];
    } else {
      viewersPrefs = [[NSMutableDictionary alloc] initWithCapacity: 1];
    }

    dictEntry = [viewersPrefs objectForKey: myPath];
    if (dictEntry) { 
 		  myPrefs = [dictEntry mutableCopy];
    } else {
      myPrefs = [[NSMutableDictionary alloc] initWithCapacity: 1];
    }

		/* ************* */
		/*     shelf     */
		/* ************* */
		shfwidth = [[self contentView] frame].size.width;
		
    dictEntry = [myPrefs objectForKey: @"shelfdicts"];      
    if(dictEntry) {
      shelfDicts = [NSArray arrayWithArray: dictEntry];      
    } else {
			NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithCapacity: 1];
      NSArray *arr;

      if ([myPath isEqualToString: @"/"] == YES) {
        arr = [NSArray arrayWithObject: NSHomeDirectory()];
      } else {         
        arr = [NSArray arrayWithObject: myPath]; 
      }
			
			[dict setObject: arr forKey: @"paths"];
			[dict setObject: @"1" forKey: @"index"];
						       
      shelfDicts = [NSArray arrayWithObject: dict];         		
    }
    [myPrefs setObject: shelfDicts forKey: @"shelfdicts"];      

    shelf = [[Shelf alloc] initWithIconsDicts: shelfDicts rootPath: myPath];
    dictEntry = [myPrefs objectForKey: @"shelfheight"];
    if(dictEntry) {
      [shelf setFrame: NSMakeRect(0, 0, shfwidth, [dictEntry intValue])]; 
    } else {
      [shelf setFrame: NSMakeRect(0, 0, shfwidth, 75)];  
    }
		
    [shelf setDelegate: self];
    [shelf setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];

		/* ************* */
		/*    viewer     */
		/* ************* */
    dictEntry = [myPrefs objectForKey: @"viewtype"];
    if(dictEntry) {
			[self makeViewersWithBundlesPaths: vpaths type: dictEntry];
    } else {
			[self makeViewersWithBundlesPaths: vpaths type: nil];
    }
    [myPrefs setObject: viewType forKey: @"viewtype"];

		[viewer setPath: myPath delegate: self viewApps: viewsapps];

	  [viewersPrefs setObject: myPrefs forKey: myPath];
	  [defaults setObject: viewersPrefs forKey: @"viewersprefs"];
    [defaults synchronize];
    RELEASE (myPrefs);
    RELEASE (viewersPrefs);     

		/* ************* */
		/*   mainview    */
		/* ************* */
		if ([viewer usesShelf] == YES) {
			usingSplit = YES;
    	mainview = [[GWSplitView alloc] initWithFrame: [[self contentView] frame] viewer: self];
    	[mainview setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];
    	[mainview setDelegate: self];
			[mainview addSubview: shelf];
		
		} else {
			usingSplit = NO;
    	mainview = [[NSView alloc] initWithFrame: [[self contentView] frame]];
    	[mainview setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];
			[mainview setPostsFrameChangedNotifications: YES];
		}
		
    [mainview addSubview: viewer];
		[[self contentView] addSubview: mainview];

    [[NSNotificationCenter defaultCenter] addObserver: self 
                				  selector: @selector(viewersListDidChange:) 
                					    name: GWViewersListDidChangeNotification
                					  object: nil];

    [[NSNotificationCenter defaultCenter] addObserver: self 
                				  selector: @selector(columnsWidthChanged:) 
                					    name: GWBrowserColumnWidthChangedNotification
                					  object: nil];

    [[NSNotificationCenter defaultCenter] addObserver: self 
                				  selector: @selector(viewFrameDidChange:) 
                					    name: NSViewFrameDidChangeNotification
                					  object: nil];

  	[mainview resizeWithOldSuperviewSize: [mainview frame].size]; 
		
		[self adjustSubviews];	
  }
  
  return self;
}

- (void)makeViewersWithBundlesPaths:(NSArray *)bpaths type:(NSString *)vtype
{
	NSMenu *menu;
	BOOL found;
	int i;

	ASSIGN (viewersPaths, bpaths);

	TEST_RELEASE (viewers);
	viewers = [[NSMutableArray alloc] initWithCapacity: 1];

  for (i = 0; i < [bpaths count]; i++) {
		NSString *bpath = [bpaths objectAtIndex: i];	
		NSBundle *bundle = [NSBundle bundleWithPath: bpath]; 
		Class principalClass = [bundle principalClass];
		id<ViewersProtocol> vwr = [[principalClass alloc] init];
		[viewers addObject: vwr];
		RELEASE ((id)vwr);
  }

	menu = [[[NSApp mainMenu] itemWithTitle: NSLocalizedString(@"View", @"")] submenu];

	while (1) {
  	if ([menu numberOfItems] == 0) {
    	break;
  	}
  	[menu removeItemAtIndex: 0];
	}

	found = NO;
	
	for (i = 0; i < [viewers count]; i++) {
		id vwr = [viewers objectAtIndex: i];
		NSString *menuName = [vwr menuName];
		NSString *shortCut = [vwr shortCut];
		
		addItemToMenu(menu, menuName, @"", @"setViewerType:", shortCut);
		
		if (vtype && [vtype isEqualToString: menuName]) {
			ASSIGN (viewType, menuName);
			ASSIGN (viewer, (id<ViewersProtocol>)vwr);			
			found = YES;
		}
	}
	
	if (found == NO) {
		ASSIGN (viewer, (id<ViewersProtocol>)[viewers objectAtIndex: 0]);
		ASSIGN (viewType, [viewer menuName]);
	}
	
	[self setMiniwindowImage: [viewer miniicon]];
}

- (void)changeViewer:(NSString *)newViewType
{
  NSArray *selPaths = [[NSArray alloc] initWithArray: selectedPaths];
	NSRect r = [[self contentView] frame];
	
  [viewer removeFromSuperview];
	[self makeViewersWithBundlesPaths: viewersPaths type: newViewType];
	[viewer setPath: myPath delegate: self viewApps: viewsapps];
	
	if (([viewer usesShelf] == YES) && (usingSplit == NO)) {
		usingSplit = YES;
		[mainview removeFromSuperview];		
		RELEASE (mainview);
		mainview = [[GWSplitView alloc] initWithFrame: r viewer: self];
		[mainview setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];
		[mainview setDelegate: self];
		[mainview addSubview: shelf];
    [mainview addSubview: viewer];
		[[self contentView] addSubview: mainview];

	} else if (([viewer usesShelf] == NO) && (usingSplit == YES)) {	
		usingSplit = NO;
		[shelf removeFromSuperview];
		[mainview removeFromSuperview];
		RELEASE (mainview);
		mainview = [[NSView alloc] initWithFrame: r];
    [mainview setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];
		[mainview setPostsFrameChangedNotifications: YES];
    [mainview addSubview: viewer];
		[[self contentView] addSubview: mainview];
	
	} else {
  	[mainview addSubview: viewer];                
	}
	
	[self adjustSubviews];
	[viewer setCurrentSelection: selPaths];
	RELEASE (selPaths);
}

- (id)viewer
{
	return viewer;
}

- (void)adjustSubviews
{
  NSRect r = [mainview frame];
  float w = r.size.width;
	float h = r.size.height;   

	if (usingSplit == NO) {
  	[viewer setFrame: NSMakeRect(8, 0, w - 16, h)];
  	[viewer resizeWithOldSuperviewSize: [viewer frame].size]; 
	
	} else {
		float sh = [shelf frame].size.height;
		float d = [mainview dividerThickness];
		float shelfheight;
		
  	if (sh < 35) {
    	shelfheight = 2;
  	} else if (sh <= 75) {  
    	shelfheight = 75;
  	} else {
    	shelfheight = 150;
  	}

  	[shelf setFrame: NSMakeRect(0, 0, r.size.width, shelfheight)];
  	[shelf resizeWithOldSuperviewSize: [shelf frame].size]; 

  	[viewer setFrame: NSMakeRect(8, shelfheight + d, w - 16, h - shelfheight - d)];
  	[viewer resizeWithOldSuperviewSize: [viewer frame].size]; 
	}
}

- (void)viewFrameDidChange:(NSNotification *)notification
{
	if ((NSView *)[notification object] == mainview) {
  	[self adjustSubviews];
	}
}

- (void)setSelectedPaths:(NSArray *)paths
{
	ASSIGN (selectedPaths, paths);
}

- (NSString *)myPath
{
  return myPath;
}

- (NSPoint)locationOfIconForPath:(NSString *)apath
{
	return [viewer locationOfIconForPath: apath];
}

- (void)activate
{
  [self orderFront: nil];
}

- (void)viewersListDidChange:(NSNotification *)notification
{
	NSArray *vpaths = (NSArray *)[notification object];	
	
	if ([viewersPaths isEqual: vpaths] == NO) {
		ASSIGN (viewersPaths, vpaths);
		[self changeViewer: viewType];
	}
}

- (void)columnsWidthChanged:(NSNotification *)notification
{
  NSRect r = [self frame];
  float x = r.origin.x;
	float y = r.origin.y;
	float w = r.size.width;
	float h = r.size.height;
  int columnsWidth = [gw browserColumnsWidth];
  int columns = (int)(w / resizeIncrement); 

  resizeIncrement = columnsWidth;
     
  [viewer setAutoSynchronize: NO];
     
  [self setFrame: NSMakeRect(x, y, (columns * columnsWidth), h) display: YES];  
  [self setMinSize: NSMakeSize(resizeIncrement * 2, 250)];    
  [self setResizeIncrements: NSMakeSize(resizeIncrement, 1)];

  [viewer setResizeIncrement: columnsWidth];
  [mainview setNeedsDisplay: YES]; 
}

- (NSString *)viewType
{
  return viewType;
}

- (BOOL)viewsApps
{
  return viewsapps;
}

- (void)setViewerSelection:(NSArray *)selPaths
{
  [viewer setCurrentSelection: selPaths];
	
	if ([selPaths count] == 1) {
		[self addToHistory: [selPaths objectAtIndex: 0]];
	}
}

- (void)updateInfoString
{
  NSString *path;
  NSDictionary *attributes;
	NSNumber *freeFs;
  NSString *infoString;
  
	if (usingSplit == NO) {
		return;
	}
	
  path = [[viewer selectedPaths] objectAtIndex: 0];
	attributes = [fm fileSystemAttributesAtPath: path];
	freeFs = [attributes objectForKey: NSFileSystemFreeSize];
  
	if(freeFs == nil) {  
		infoString = [NSString stringWithString: @"unknown volume size"];    
	} else {
		infoString = [NSString stringWithFormat: @"%@ available on hard disk", 
                          fileSizeDescription([freeFs unsignedLongLongValue])];
	}
  
  [mainview updateInfo: infoString];
}

- (void)selectAll
{
  [viewer selectAll];
}

- (void)updateDefaults
{
	NSUserDefaults *defaults;
  NSMutableDictionary *viewersPrefs, *myPrefs;
  NSMutableArray *shelfDicts;
  NSString *shelfHeight;
  id dictEntry;
	
  [self saveFrameUsingName: [NSString stringWithFormat: @"Viewer at %@", myPath]];

  defaults = [NSUserDefaults standardUserDefaults];	
  viewersPrefs = [[defaults dictionaryForKey: @"viewersprefs"] mutableCopy];
	
  dictEntry = [viewersPrefs objectForKey: myPath];

  if (dictEntry) { 
 		myPrefs = [dictEntry mutableCopy];
  } else {
    myPrefs = [[NSMutableDictionary alloc] initWithCapacity: 1];
  }
  
  [myPrefs setObject: viewType forKey: @"viewtype"];

  shelfHeight = [NSString stringWithFormat: @"%i", (int)[shelf frame].size.height];
  [myPrefs setObject: shelfHeight forKey: @"shelfheight"];

  shelfDicts = [[shelf iconsDicts] mutableCopy];
  if ([shelfDicts count] == 0) {
		NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithCapacity: 1];
		NSArray *arr = [NSArray arrayWithObject: NSHomeDirectory()];
		
		[dict setObject: arr forKey: @"paths"];
		[dict setObject: @"1" forKey: @"index"];
		
    [shelfDicts addObject: dict];
  }      
  [myPrefs setObject: shelfDicts forKey: @"shelfdicts"];
  
	[viewersPrefs setObject: myPrefs forKey: myPath];
	[defaults setObject: viewersPrefs forKey: @"viewersprefs"];
	[defaults synchronize];
  
  RELEASE (viewersPrefs);
  RELEASE (myPrefs);
  RELEASE (shelfDicts);
}

- (void)keyDown:(NSEvent *)theEvent 
{
	NSString *characters;
	unichar character;

	characters = [theEvent characters];
	character = 0;
		
  if ([characters length] > 0) {
		character = [characters characterAtIndex: 0];
	}
		
	switch (character) {
    case NSLeftArrowFunctionKey:
			if ([theEvent modifierFlags] & NSControlKeyMask) {
				[self goBackwardInHistory: nil];
			}
      return;

    case NSRightArrowFunctionKey:			
			if ([theEvent modifierFlags] & NSControlKeyMask) {
	      [self goForwardInHistory: nil];
	    } 
			return;
	}
	
	[super keyDown: theEvent];
}

- (void)becomeMainWindow
{
	NSArray *selPaths = [viewer selectedPaths];
	ASSIGN (selectedPaths, selPaths);
  [gw setSelectedPaths: selPaths];
  [gw setCurrentViewer: self];  
  [self updateInfoString]; 
  [self makeFirstResponder: [viewer viewerView]];
	historyWin = [gw historyWindow];
	[historyWin setViewer: self];
	[self tuneHistory];
	[historyWin setHistoryPaths: ViewerHistory];
	[self setCurrentHistoryPosition: currHistoryPos];
}

- (BOOL)windowShouldClose:(id)sender
{
  if (self != [gw rootViewer]) {
    [[NSNotificationCenter defaultCenter] removeObserver: self];
    [viewer unsetWatchers]; 
		[historyWin setHistoryPaths: nil];
		[historyWin setViewer: nil];
  }
    
  [self updateDefaults];
  return YES;
}

- (void)close
{
  [super close];
  [gw viewerHasClosed: self]; 
}

//
// Menu operations
//
- (BOOL)validateMenuItem:(id <NSMenuItem>)menuItem
{	
	if ([[menuItem title] isEqualToString: NSLocalizedString(viewType, @"")]) {
		return NO;
	}
	
	if ([[menuItem title] isEqualToString: NSLocalizedString(@"Go forward", @"")]) {
		if ([ViewerHistory count] <= 1) {
			return NO;
		}
	  if (currHistoryPos < ([ViewerHistory count] - 1)) {
			return YES;
		} else {
			return NO;
		}
	}
	
	if ([[menuItem title] isEqualToString: NSLocalizedString(@"Go backward", @"")]) {
		if ([ViewerHistory count] <= 1) {
			return NO;
		}
	  if (currHistoryPos > 0) {
			return YES;
		}	else {
			return NO;
		}
	}
	
	return YES;
}

- (void)openSelection:(id)sender
{
	[gw openSelectedPaths: selectedPaths newViewer: NO];
}

- (void)openSelectionAsFolder:(id)sender
{
  [gw openSelectedPaths: selectedPaths newViewer: YES];
}

- (void)editFile:(id)sender
{
  int i;  
	BOOL isdir;

  for (i = 0; i < [selectedPaths count]; i++) {  
    NSString *path = [selectedPaths objectAtIndex: i];
    [fm fileExistsAtPath: path isDirectory: &isdir];    
    if (isdir == NO) {
      [gw openFile: path];
    }
  }
}

- (void)newFolder:(id)sender
{
  [gw newObject: YES];
}

- (void)newFile:(id)sender
{
  [gw newObject: NO];
}

- (void)duplicateFiles:(id)sender
{
	[gw duplicateFiles];
}

- (void)deleteFiles:(id)sender
{
	[gw deleteFiles];
}

- (void)setViewerType:(id)sender
{
	NSString *title = [sender title];
	int i;

	if ([NSLocalizedString(viewType, @"") isEqualToString: title]) {
    return;
  }

	for (i = 0; i < [viewers count]; i++) {
		id vwr = [viewers objectAtIndex: i];
		NSString *vwrName = [vwr menuName];
	
		if ([title isEqualToString: NSLocalizedString(vwrName, @"")]) {
			[self changeViewer: vwrName];
			break;
		}
	}
}

- (void)selectAllInViewer:(id)sender
{
	[viewer selectAll];
}

- (void)print:(id)sender
{
	[super print: sender];
}

//
// splitView delegate methods
//
- (float)splitView: (NSSplitView *)sender
          constrainSplitPosition: (float)proposedPosition 
                                        ofSubviewAt: (int)offset
{
  if (proposedPosition < 35) {
    return 2;
  } else if (proposedPosition <= 75) {  
    return 75;
  } else {
    return 150;
  }
}

- (void)splitView:(NSSplitView *)sender 
                  resizeSubviewsWithOldSize:(NSSize)oldSize
{
	[self adjustSubviews];
}

- (void)splitViewDidResizeSubviews:(NSNotification *)aNotification
{
	[self adjustSubviews];
}

@end

//
// history methods
//
@implementation ViewersWindow (historyMethods)

- (void)addToHistory:(NSString *)path
{
	NSString *defApp, *type;
	NSArray *tempHistory;

#define SETPOSITION(p) \
if ([ViewerHistory isEqual: tempHistory] == NO) { \
[self tuneHistory]; \
if (self == [historyWin viewer]) \
[historyWin setHistoryPaths: ViewerHistory]; \
} \
[self setCurrentHistoryPosition: p]; \
RELEASE (tempHistory); \
return
	
	currHistoryPos = (currHistoryPos < 0) ? 0 : currHistoryPos;
 
	[[NSWorkspace sharedWorkspace] getInfoForFile: path 
																		application: &defApp 
																					 type: &type]; 

	if ((type != NSDirectoryFileType) && (type != NSFilesystemFileType)) {
		return;
	}									

	tempHistory = [ViewerHistory copy];
	
	if (([ViewerHistory count] - 1) == currHistoryPos) {
		if ([[ViewerHistory lastObject] isEqual: path] == NO) {
			[ViewerHistory addObject: path];
		}
		SETPOSITION ([ViewerHistory count] - 1);
  }
	
	if ([ViewerHistory count] > (currHistoryPos + 1)) {
		if (([[ViewerHistory objectAtIndex: currHistoryPos + 1] isEqual: path] == NO)
				&& ([[ViewerHistory objectAtIndex: currHistoryPos] isEqual: path] == NO)) {
	
			currHistoryPos++;
			[ViewerHistory insertObject: path atIndex: currHistoryPos];
			
			while ((currHistoryPos + 1) < [ViewerHistory count]) {
				int last = [ViewerHistory count] - 1;
				[ViewerHistory removeObjectAtIndex: last];
			}
			SETPOSITION (currHistoryPos);
		}	
	}
	
	if ([[ViewerHistory lastObject] isEqual: path]) {
		SETPOSITION ([ViewerHistory count] - 1);
	}
	
	RELEASE (tempHistory);
}

- (void)tuneHistory
{
	int i, count = [ViewerHistory count];
	BOOL changed = NO;
	
#define CHECK_POSITION(n) \
if (currHistoryPos >= i) \
currHistoryPos -= n; \
currHistoryPos = (currHistoryPos < 0) ? 0 : currHistoryPos; \
currHistoryPos = (currHistoryPos >= count) ? (count - 1) : currHistoryPos	

	for (i = 0; i < count; i++) {
		NSString *hpath = [ViewerHistory objectAtIndex: i];
		BOOL isdir;
		
		if (([fm fileExistsAtPath: hpath isDirectory: &isdir] && isdir) == NO) {
			[ViewerHistory removeObjectAtIndex: i];
			changed = YES;
			CHECK_POSITION (1);		
			count--;
			i--;
		}
	}
			
	for (i = 0; i < count; i++) {
		NSString *hpath = [ViewerHistory objectAtIndex: i];

		if (i < ([ViewerHistory count] - 1)) {
			NSString *next = [ViewerHistory objectAtIndex: i + 1];
			
			if ([next isEqual: hpath]) {
				[ViewerHistory removeObjectAtIndex: i + 1];
				changed = YES;
				CHECK_POSITION (1);
				count--;
				i--;
			}
		}
	}
	
	if ([ViewerHistory count] > 4) {
		NSString *sa[2], *sb[2];
	
		count = [ViewerHistory count];
		
		for (i = 0; i < count; i++) {
			if (i < ([ViewerHistory count] - 3)) {
				sa[0] = [ViewerHistory objectAtIndex: i];
				sa[1] = [ViewerHistory objectAtIndex: i + 1];
				sb[0] = [ViewerHistory objectAtIndex: i + 2];
				sb[1] = [ViewerHistory objectAtIndex: i + 3];
		
				if ([sa[0] isEqual: sb[0]] && [sa[1] isEqual: sb[1]]) {
					[ViewerHistory removeObjectAtIndex: i + 3];
					[ViewerHistory removeObjectAtIndex: i + 2];
					changed = YES;
					CHECK_POSITION (2);
					count -= 2;
					i--;
				}
			}
		}
	}
	
	CHECK_POSITION (0);
		
	if (changed) {
		[historyWin setHistoryPaths: ViewerHistory position: currHistoryPos];
	}
}

- (void)setCurrentHistoryPosition:(int)newPosition 
{
	int count = [ViewerHistory count];
	currHistoryPos = (newPosition < 0) ? 0 : newPosition; 			
	currHistoryPos = (newPosition >= count) ? (count - 1) : newPosition;	
	[historyWin setHistoryPosition: currHistoryPos];
}

- (void)goToHistoryPosition:(int)position
{
	[self tuneHistory];
	if ((position >= 0) && (position < [ViewerHistory count])) {
		NSString *newpath = [ViewerHistory objectAtIndex: position];
		[self setCurrentHistoryPosition: position];
		[viewer setCurrentSelection: [NSArray arrayWithObject: newpath]];
	}
}

- (void)goBackwardInHistory:(id)sender
{
	[self tuneHistory];
  if (currHistoryPos > 0) {
    NSString *newpath = [ViewerHistory objectAtIndex: (currHistoryPos - 1)];
		[self setCurrentHistoryPosition: currHistoryPos - 1];
    [viewer setCurrentSelection: [NSArray arrayWithObject: newpath]];
  }
}

- (void)goForwardInHistory:(id)sender
{
	[self tuneHistory];
  if (currHistoryPos < ([ViewerHistory count] - 1)) {
		NSString *newpath = [ViewerHistory objectAtIndex: (currHistoryPos + 1)];
		[self setCurrentHistoryPosition: currHistoryPos + 1];					
    [viewer setCurrentSelection: [NSArray arrayWithObject: newpath]];  
  } 
}

@end

//
// shelf delegate methods
//

@implementation ViewersWindow (shelfDelegateMethods)

- (NSArray *)getSelectedPaths
{
	return selectedPaths;
}

- (void)shelf:(Shelf *)sender setCurrentSelection:(NSArray *)paths
{
	[self addPathToHistory: paths];
  [viewer setCurrentSelection: paths];
}

- (void)shelf:(Shelf *)sender openCurrentSelection:(NSArray *)paths newViewer:(BOOL)newv
{
  [viewer setSelectedPaths: paths];
  [gw openSelectedPaths: paths newViewer: newv];   
}

@end

//
// Viewers Delegate Methods
//

@implementation ViewersWindow (ViewerDelegateMethods)

- (void)setTheSelectedPaths:(id)paths
{
	ASSIGN (selectedPaths, paths);
	[gw setSelectedPaths: paths];
}

- (void)setTitleAndPath:(id)apath selectedPaths:(id)paths
{
	ASSIGN (myPath, apath);
	ASSIGN (selectedPaths, paths);
	[gw setSelectedPaths: paths];
	
  if ([myPath isEqualToString: @"/"] == YES) {
    [self setTitle: NSLocalizedString(@"File Viewer", @"")];
  } else {
		NSString *pth = [myPath stringByDeletingLastPathComponent];
		NSString *nm = [myPath lastPathComponent];
		NSString *title = [NSString stringWithFormat: @"%@ - %@", nm, pth];
    [self setTitle: title];   
  }
}

- (void)addWatcherForPath:(id)path
{
	[gw addWatcherForPath: path];
}

- (void)removeWatcherForPath:(id)path
{
	[gw removeWatcherForPath: path];
}

- (void)addPathToHistory:(NSArray *)paths
{
	if ((paths == nil) || ([paths count] == 0) || ([paths count] > 1)) {
    return; 
	}
	[self addToHistory: [paths objectAtIndex: 0]];
}

- (int)sortTypeForDirectoryAtPath:(id)path
{
	return [gw sortTypeForDirectoryAtPath: path];
}

- (void)openTheSelectedPaths:(id)paths newViewer:(BOOL)newv
{
	[gw openSelectedPaths: paths newViewer: newv];
}

- (void)newViewerAtPath:(id)path canViewApps:(BOOL)viewapps
{
	[gw newViewerAtPath: path canViewApps: viewapps];
}

- (BOOL)openFile:(id)fullPath
{
	return [gw openFile: fullPath];
}

- (void)performFileOperationWithDictionary:(id)opdict
{
	NSString *operation = [opdict objectForKey: @"operation"];
	NSString *source = [opdict objectForKey: @"source"];
	NSString *destination = [opdict objectForKey: @"destination"];
	NSArray *files = [opdict objectForKey: @"files"];
	int tag;
	
	[gw performFileOperation: operation source: source 
											destination: destination files: files tag: &tag];
}

- (NSImage *)smallIconForFile:(NSString*)aPath
{
	return [gw smallIconForFile: aPath];
}

- (NSImage *)smallIconForFiles:(NSArray*)pathArray
{
	return [gw smallIconForFiles: pathArray];
}

- (NSString *)getTrashPath
{
	return [gw trashPath];
}

- (void)updateTheInfoString
{
	[self updateInfoString];
}

- (int)getBrowserColumnsWidth
{
	return [gw browserColumnsWidth];
}

- (int)getIconViewCellsWidth
{
	return [gw iconViewCellsWidth];
}

- (int)getWindowFrameWidth
{
	return (int)[self frame].size.width;
}

- (int)getWindowFrameHeight
{
	return (int)[self frame].size.height;
}

@end
