/*C*

 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

#include "CtrlEvent.h"
#include "Events.h"
#include "MenuCtrl.h"
#include "../gl/Color.h"
#include "../gl/Console.h"
#include "../gl/Font.h"
#include "../gl/Mouse.h"
#include "../gl/VgaBlur.h"
#include "../gl/keycodes.h"
#include "../util/String.h"

//--------------------------------------------------------------------------------------------------------------------------------

class MenuItem : public Object
{
private:
 Events* events;
 Control* dest;

public:
 int id;
 String text;

 MenuItem(Events* _events, Control* _dest, int _id, const char* _text)
  { events = _events; dest = _dest; id = _id; text = _text; }
};

template Collection<MenuItem>;

//--------------------------------------------------------------------------------------------------------------------------------

MenuCtrl::MenuCtrl(Rect _rect, Events* _events, Control* _notify, Font* _font)
 : Control(_rect, _events)
{
 notify = _notify;
 font = _font;
 pSel = 0;
 pTop = 0;
 menuItems = new Collection<MenuItem>;
}

MenuCtrl::~MenuCtrl()
{
 delete menuItems;
}

//--------------------------------------------------------------------------------------------------------------------------------

Rect MenuCtrl::itemRect(int i)
{
 return Rect(rect().topLeft() + Point(0, i * font->h() - top()), Point(rect().w, font->h()));
}

bool MenuCtrl::bringIntoView()
{
 int y = font->h() * sel();

 // scroll up if we have to
 if(y < top())
  {
   top(y);
   return true;
  }

 // scroll down if we have to
 if(y + font->h() > top() + rect().h)
  {
   int newTop;
   if(font->h() > rect().h) newTop = y;
   else newTop = y + font->h() - rect().h;
   if(newTop != top())
    {
     top(newTop);
     return true;
    }
  }

 return false;
}

//--------------------------------------------------------------------------------------------------------------------------------

void MenuCtrl::evLBDown(Point p)
{
 int newSel = (p.y + top() - rect().top()) / font->h(); // confused?
 int oldSel = sel();
 if(newSel == oldSel || newSel >= menuItems->length()) return;

 sel(newSel);

 if(!rect().contains(itemRect(newSel)))
  {
   if(bringIntoView())
    events->redraw(rect());
  }
  /*
  {
   int newTop = font->h() * newSel;
   if(newTop + rect().h > menuItems->length() * font->h())
    newTop = menuItems->length() * font->h() - rect().h;
   if(newTop < 0) newTop = 0; // this should never happen...
   top(newTop);
   events->redraw(rect());
  }
  */
 else
  {
   events->redraw(itemRect(oldSel));
   events->redraw(itemRect(newSel));
  }
}

void MenuCtrl::evDbClick(Point p)
{
 int newSel = (p.y + top() - rect().top()) / font->h(); // confused?
 int oldSel = sel();
 if(newSel >= menuItems->length()) return;
 sel(newSel);

 if(newSel != oldSel)
  {
   events->redraw(itemRect(oldSel).clipTo(rect()));
   events->redraw(itemRect(newSel).clipTo(rect()));
  }

 events->add(new CommandEvent(notify, menuItems->at(sel())->id));
}

//--------------------------------------------------------------------------------------------------------------------------------

void MenuCtrl::evKeyDown(int key)
{
 if(IS_KEY_UP(key))
  {
   if(sel() > 0)
    {
     int oldSel = sel();
     int newSel = sel() - 1;
     sel(newSel);
     if(bringIntoView()) events->redraw(rect());
     else
      {
       events->redraw(itemRect(oldSel));
       events->redraw(itemRect(newSel));
      }
    }
  }
 if(IS_KEY_DOWN(key))
  {
   if(sel() < menuItems->length()-1)
    {
     int oldSel = sel();
     int newSel = sel() + 1;
     sel(newSel);
     if(bringIntoView()) events->redraw(rect());
     else
      {
       events->redraw(itemRect(oldSel));
       events->redraw(itemRect(newSel));
      }
    }
  }
}

void MenuCtrl::evKeyRep(int key)
{
 evKeyDown(key);
}

//--------------------------------------------------------------------------------------------------------------------------------

void MenuCtrl::draw(Rect redrawRect)
{
 font->hAlign = Font::HMiddle;
 font->vAlign = Font::VMiddle;
 // FIXME this should figure out which are on screen first
 for(int i=0; i<menuItems->length(); i++)
  {
   Rect miRect = itemRect(i);
   if(!miRect.meets(rect()) || !miRect.meets(redrawRect)) continue;
   MenuItem* mi = menuItems->at(i);
   if(sel() == i) screen->fillBox(miRect, Color(colorRed, 0x7F));
   font->writef(screen, miRect.middle(), colorWhite, "%s", (const char*) mi->text);
  }
}

//--------------------------------------------------------------------------------------------------------------------------------

void MenuCtrl::add(int id, const char* text)
{
 menuItems->add(new MenuItem(events, notify, id, text));
}

//--------------------------------------------------------------------------------------------------------------------------------

int MenuCtrl::itemId()
{
 return menuItems->at(sel())->id;
}

//--------------------------------------------------------------------------------------------------------------------------------
