/*C* -*-c++-*-
 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

#include "Color.h"
#include "Console.h"
#include "Font.h"
#include "VgaBlur.h"
#include "VgaContext.h"
#include "alphaTable.h"
#include "palette.h"
#include "../util/debug.h"
#include "../util/error.h"
#include <freetype.h>
#include <stdarg.h>
#include <string.h>

//--------------------------------------------------------------------------------------------------------------------------------

class Glyph : public Object
{
public:
 const unsigned char* data;
 Point z;
 int advance, bytewidth;
 Glyph();
 ~Glyph();
 void draw(class VgaContext* vc, Point p, Color c);
};

Glyph::Glyph()
{
 data = NULL;
 z = Point(0, 0);
 advance = bytewidth = 0;
}

Glyph::~Glyph()
{
 if(data) delete data;
}

void Glyph::draw(VgaContext* vc, Point p, Color c)
{
 Rect r = Rect(p, z);
 if(vc->clipping)
  {
   if(!r.meets(vc->clippingRect)) return;
   r.clipTo(vc->clippingRect);
  }

 switch(bpp)
  {

  case 8:
   for(int y = 0; y < r.h; y++)
    {
     const unsigned char* s = &data[(r.y - p.y + y) * bytewidth + (r.x - p.x)];
     unsigned char* d = &vc->fb()[r.y + y][r.x];
     for(int x = 0; x < r.w; x++)
      {
       if(*s++) *d = rgb2pal(c);
       d++;
      }
    }
   break;

  case 16:
   for(int y = 0; y < r.h; y++)
    {
     const unsigned char* s = &data[(r.y - p.y + y) * bytewidth + (r.x - p.x)];
     unsigned short* d = (unsigned short*) &vc->fb()[r.y + y][2 * r.x];
     for(int x = 0; x < r.w; x++)
      {
       unsigned char b = alphaTable[255 - *s][(*d & 0x001F) << 3] + alphaTable[*s][c.b];
       unsigned char g = alphaTable[255 - *s][(*d & 0x07E0) >> 3] + alphaTable[*s][c.g];
       unsigned char r = alphaTable[255 - *s][(*d & 0xF800) >> 8] + alphaTable[*s][c.r];
       *d++ = b >> 3 | (g & 0xFC) << 3 | (r & 0xF8) << 8;
       s++;
      }
    }
   break;

  case 24:
   for(int y = 0; y < r.h; y++)
    {
     const unsigned char* s = &data[(r.y - p.y + y) * bytewidth + (r.x - p.x)];
     unsigned char* d = &vc->fb()[r.y + y][3 * r.x];
     for(int x = 0; x < r.w; x++)
      {
       *d++ = alphaTable[255 - *s][*d] + alphaTable[*s][c.b];
       *d++ = alphaTable[255 - *s][*d] + alphaTable[*s][c.g];
       *d++ = alphaTable[255 - *s][*d] + alphaTable[*s][c.r];
       s++;
      }
    }
   break;

  case 32:
   for(int y = r.top(); y <= r.bottom(); y++)
    {
     const unsigned char* s = &data[(y - p.y) * bytewidth + (r.x - p.x)];
     unsigned char* d = &vc->fb()[y][4 * r.x];
     for(int x = 0; x < r.w; x++)
      {
       *d++ = alphaTable[255 - *s][*d] + alphaTable[*s][c.b];
       *d++ = alphaTable[255 - *s][*d] + alphaTable[*s][c.g];
       *d++ = alphaTable[255 - *s][*d] + alphaTable[*s][c.r];
       s++; d++;
      }
    }
   break;

  }
}

//--------------------------------------------------------------------------------------------------------------------------------

Font::Font()
{
 for(int i=0; i<96; i++) glyphs[i] = NULL;
 pH = 0;
 hAlign = HLeft;
 vAlign = VTop;
}

Font::~Font()
{
 for(int i=0; i<96; i++)
  if(glyphs[i]) delete glyphs[i];
}

//--------------------------------------------------------------------------------------------------------------------------------

bool Font::load(const char* filename, int _h)
{
 // FIXME the width of pixmaps should be individual
 // FIXME this is a bit of a mess now...

#define ttCheck(some_expr, if_error) \
if(TT_Error err = (some_expr)) \
{ setError("%s: \"%s\" failed with %ld", filename, #some_expr, err); if_error; }

 VPRINTF("Font::load(\"%s\", %d)\n", filename, _h);

 // variables to keep track of where we are so we can get out quickly in case of error

 bool openFace = false;

 // variables to keep TT data in

 TT_Face face;
 TT_Face_Properties properties;
 TT_CharMap ttCharMap;
 TT_Instance ttInstance;

 // open the face file

 if(TT_Error err = TT_Open_Face(ttEngine, filename, &face))
  switch(err)
   {
   case TT_Err_Could_Not_Open_File: setError("%s: couldn't open", filename); goto fail;
   default: setError("%s: TT_Open_Face failed (returned %ld)", filename, err); goto fail;
   }
 openFace = true;

 // load the properties structure

 ttCheck(TT_Get_Face_Properties(face, &properties), goto fail);

 // search for a unicode charmap and load one if found, otherwise fail

 for(int i=0; i<properties.num_CharMaps; i++)
  {
   unsigned short platform, encoding;
   TT_Get_CharMap_ID(face, i, &platform, &encoding);
   if((platform == 3 && encoding == 1) || (platform == 0 && encoding == 0))
    { TT_Get_CharMap(face, i, &ttCharMap); goto haveCharmap; }
  }
 setError("%s contains no unicode charmap", filename);
 goto fail;
haveCharmap:

 // create and initialise an instance on the face

 // FIXME the resolution could be adjusted with console size...?
 ttCheck(TT_New_Instance(face, &ttInstance), goto fail);
 ttCheck(TT_Set_Instance_CharSize(ttInstance, _h * 64), goto fail);

 {
  
  unsigned char palette[5] = {0, 63, 127, 191, 255};
  ttCheck(TT_Set_Raster_Gray_Palette(ttEngine, palette), goto fail);
  
  TT_Glyph ttGlyph;
  ttCheck(TT_New_Glyph(face, &ttGlyph), goto fail);

  // first pass to compute extremes of pixmaps size

  TT_F26Dot6 highest = 0, lowest = 0, rightmost = 0, leftmost = 0;
  for(int i=0; i<96; i++)
   {
    TT_Glyph_Metrics ttMetrics;
    ttCheck(TT_Load_Glyph(ttInstance, ttGlyph, TT_Char_Index(ttCharMap, i + 32), TTLOAD_DEFAULT), goto fail);
    ttCheck(TT_Get_Glyph_Metrics(ttGlyph, &ttMetrics), return false);
    
    if(ttMetrics.bbox.yMin < lowest) lowest = ttMetrics.bbox.yMin;
    if(ttMetrics.bbox.yMax > highest) highest = ttMetrics.bbox.yMax;
    if(ttMetrics.bbox.xMin < leftmost) leftmost = ttMetrics.bbox.xMin;
    if(ttMetrics.bbox.xMax > rightmost) rightmost = ttMetrics.bbox.xMax;
   }
  
  pH = (highest+63)/64 - (lowest-63)/64;
  
  // then go through and render each glyph

  {
   TT_F26Dot6 yOffset = -(lowest & -64);
   
   for(int i=0; i<96; i++)
    {
     if(glyphs[i]) delete glyphs[i];
     
     ttCheck(TT_Load_Glyph(ttInstance, ttGlyph, TT_Char_Index(ttCharMap, i + 32), TTLOAD_DEFAULT), goto fail);
     
     TT_Glyph_Metrics metrics;
     ttCheck(TT_Get_Glyph_Metrics(ttGlyph, &metrics), goto fail);
     TT_F26Dot6 xmin = metrics.bbox.xMin & -64;
     TT_F26Dot6 xmax = (metrics.bbox.xMax+63) & -64;

     TT_Raster_Map pixmap;
     pixmap.flow = TT_Flow_Down;
     pixmap.rows = pH;
     pixmap.width = (xmax - xmin) / 64 + 1;
     pixmap.cols = (pixmap.width + 3) & -4;
     pixmap.size = pixmap.rows * pixmap.cols;
     pixmap.bitmap = new unsigned char [pixmap.size];

     memset((unsigned char*)pixmap.bitmap, 0, pixmap.size);
     ttCheck(TT_Get_Glyph_Pixmap(ttGlyph, &pixmap, -xmin, yOffset), goto fail);
     
     glyphs[i] = new Glyph;
     glyphs[i]->data = (const unsigned char*) pixmap.bitmap;
     glyphs[i]->z = Point(pixmap.width, pixmap.rows);
     glyphs[i]->advance = (metrics.advance + 32) / 64;
     glyphs[i]->bytewidth = pixmap.cols;
    }
  }
 }

 // shut down in a lovely happy way

 TT_Close_Face(face);
 openFace = false;

 return true;

 // shut down and return an error

fail:
 if(openFace) TT_Close_Face(face);
 return false;
}

//--------------------------------------------------------------------------------------------------------------------------------

int Font::strw(const char* s)
{
 int ret = 0;
 for(; *s; s++)
  {
   if(*s < 32) continue; // skip any non-ASCII codes... NB: this catches > 127 since it is signed...
   Glyph *glyph = glyphs[*s - 32];
   ret += glyph->advance;
  }
 return ret;
}

Rect Font::writef(VgaContext* vc, Point p, Color c, const char* format, ...)
{
 // generate the completed string
 va_list ap; va_start(ap, format);
 char* str;
 vasprintf(&str, format, ap);
 va_end(ap);

 // do any fiddling for alignment
 switch(vAlign) { case VTop: break; case VMiddle: p.y -= h() / 2; break; case VBottom: p.y -= h(); break; }
 switch(hAlign) { case HLeft: break; case HMiddle: p.x -= strw(str) / 2; break; case HRight: p.x -= strw(str); break; }

 // output the glyphs
 Point p1 = p;
 for(const char* s = str; *s; s++)
  {
   if(*s < 32) continue; // skip any non-ASCII codes... NB: this catches > 127 since it is signed...
   Glyph *glyph = glyphs[*s - 32];
   glyph->draw(vc, p1, c);
   p1.x += glyph->advance;
  }

 p1.y += h() * 2;
 return Rect(p, p1 - p).clipTo(screenRect);
}

//--------------------------------------------------------------------------------------------------------------------------------
