/* directory.c
 * - Directory Server Functions
 *
 * Copyright (c) 1999 Jack Moffitt, Barath Raghavan, and Alexander Havng
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#ifdef _WIN32
#include <win32config.h>
#else
#include <config.h>
#endif
#endif

#include "definitions.h"
#include <stdio.h> 

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <ctype.h>
#include <time.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifndef _WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#endif

#include "avl.h"
#include "threads.h"
#include "icetypes.h"
#include "icecast.h"
#include "utility.h"
#include "ice_string.h"
#include "log.h"
#include "sock.h"
#include "avl_functions.h"
#include "logtime.h"
#include "timer.h"
#include "source.h"
#include "memory.h"
#include "http.h"

#include "directory.h"

extern server_info_t info;

/* Touch an x-audiocast style directory server. */
void directory_touch_xa(directory_server_t *ds)
{
	int sockfd, error, err;
	char s[65535], *temp;
	statistics_t stat;

	err = error = 0;
	temp = NULL;
	s[0] = '\0';

	if (!ds || !ds->host) {
		write_log(LOG_DEFAULT, "WARNING: Refuse to connect to NULL host");
		return;
	}

	xa_debug(1, "DEBUG: Touching xaudiocast directory server [%s:%d/%s]", ds->host, ds->port, ds->path);
	if ((sockfd = sock_connect_wto(ds->host, ds->port, 15)) != -1) {
		avl_traverser trav = {0};
		connection_t *sourcecon;
		int i = 1, c;
		char *url_p[7];

		xa_debug(2, "DEBUG: Connecting, sending info\n");

		for (c = 0; c < 7; c++) url_p[c] = NULL;

		get_running_stats_nl (&stat);

		/* Start query */
		sprintf(s, "GET /%s?", ds->path);

		/* Don't print id if we don't have it */
		if (ds->id != -1)
			catsprintf(s, "id=%d&", ds->id);

		/* Global server info */
		catsprintf(s, "force=%d&name=%s&type=%s&version=%s&port=%d&listeners=%ld&maxlisteners=%ld&maxstreamlisteners=%ld&alt=%ld&ast=%ld&maxsources=%ld&url=%s&email=%s&location=%s",
			    info.force_servername, url_encode(info.server_name, &url_p[0]), "icecast", url_encode(VERSION, &url_p[1]), info.port[0], info.num_clients, info.max_clients, 
			    info.max_clients_per_source, stat.client_connect_time / ((stat.client_connections > 0) ? stat.client_connections : 1), 
			    stat.source_connect_time / ((stat.source_connections > 0) ? stat.source_connections : 1), info.max_sources, info.server_url, 
			    info.rp_email ? url_encode(info.rp_email, &url_p[2]) : url_encode("n/a", &url_p[3]), info.location ? url_encode(info.location, &url_p[4]) : url_encode("n/a", &url_p[4]));

		/* Per stream */
		for (c = 0; c < 7; c++) {
			if (url_p[c]) nfree(url_p[c]);
			url_p[c] = NULL;
		}

		thread_mutex_lock(&info.source_mutex);
		while ((sourcecon = avl_traverse(info.sources, &trav))) {
			if (sourcecon->food.source->audiocast.public) {
				unsigned long int alt;
				if (sourcecon->food.source->stats.client_connections > 0)
					alt = (sourcecon->food.source->stats.client_connect_time / sourcecon->food.source->stats.client_connections) * 60;
				else
					alt = sourcecon->food.source->stats.client_connect_time * 60;
				
				catsprintf(s, "&source_mount_%d=%s&source_name_%d=%s&source_desc_%d=%s&source_genre_%d=%s&source_url_%d=%s&source_listeners_%d=%ld&source_alt_%d=%d&source_bitrate_%d=%d&stream_title_%d=%s&stream_url_%d=%s",
					    i, url_encode(sourcecon->food.source->audiocast.mount, &url_p[0]), 
					    i, url_encode(sourcecon->food.source->audiocast.name, &url_p[1]), 
					    i, url_encode(sourcecon->food.source->audiocast.description, &url_p[2]), 
					    i, url_encode(sourcecon->food.source->audiocast.genre, &url_p[3]), 
					    i, url_encode(sourcecon->food.source->audiocast.url, &url_p[4]), 
					    i, sourcecon->food.source->num_clients, 
					    i, alt, 
					    i, sourcecon->food.source->audiocast.bitrate, 
					    i, url_encode(sourcecon->food.source->info.streamtitle, &url_p[5]), 
					    i, url_encode(sourcecon->food.source->info.streamurl, &url_p[6]));
				i++;
			}
		}
		thread_mutex_unlock(&info.source_mutex);

		for (c = 0; c < 7; c++) {
			if (url_p[c]) nfree(url_p[c]);
			url_p[c] = NULL;
		}

		/* Finish query */
		catsprintf(s, " HTTP/1.0\nHost: %s\n\n", ds->host);

		/* send query */
		err = sock_write_bytes(sockfd, s, ice_strlen(s));
		if (err < 0) {
			if (ds->id == -1) /* First touch failed */
				ds->counter++;
			else 	/* Subsequent touch failed */
				ds->counter = 1;

			ds->id = -1; /* Next touch will be first */

			write_log(LOG_DEFAULT, "directory_touch_xa([%s:%d]) failed... could not send data to server... (retry in %d seconds)", (info.touch_freq * ds->counter * 60) - (get_time() - ds->touchtime));
		} else {

			/* Skip HTTP Header */
			do {
				err = sock_read_line(sockfd, s, BUFSIZE);
				xa_debug(2, "DEBUG: server-response: [%s]", s);
			} while (err == 1 && ice_strcmp(s, "") != 0);

			do {
				err = sock_read_line(sockfd, s, BUFSIZE);

				if (err == 1) {
					xa_debug(2, "DEBUG: server-response: [%s]", s);

					temp = strstr(s, "x-audiocast-yp-id: ");
					if (temp != NULL) 
						ds->id = atoi((char *)(temp + 19));
					temp = strstr(s, "x-audiocast-yp-error: ");
					if (temp != NULL) 
						error = atoi((char *)(temp + 22));
				}
			} while (err == 1 && ice_strcmp(s, "") != 0);

			if (error == 0) {
				write_log(LOG_DEFAULT, "directory_touch_xa([%s:%d]) completed...server id = %i", ds->host, ds->port, ds->id);
				ds->touches++;
			} else {
				if (ds->id == -1) /* First touch failed */
					ds->counter++;
				else 	/* Subsequent touch failed */
					ds->counter = 1;

				ds->id = -1; /* Next touch will be first */

				write_log(LOG_DEFAULT, "directory_touch_xa([%s:%d]) failed... directory server error #%i... (retry in %d seconds)", ds->host, ds->port, error, (info.touch_freq * ds->counter * 60) - (get_time() - ds->touchtime));

			}
		}
	} else {
		write_log(LOG_DEFAULT, "WARNING: Connect to %s failed.", ds->host);
	}
	
	sock_close(sockfd);
}

/* These functions are left to keep compability with old icy directory servers.
   They are stupid, but.. hey :) */
void directory_add(directory_server_t *ds, connection_t *con)
{
	int sockfd, error, err;
	char s[BUFSIZE];
	char  *temp;
	char *response;
	source_t *source;
	char *url_p[3];
	int c;
	
	for (c = 0; c < 3; c++) 
		url_p[c] = NULL;

	if (con) {
		source = con->food.source;
	} else {
		xa_debug(1, "DEBUG: No touch of %s:%d cause no encoders are connected", ds->host, ds->port);
		return;
	}

	error = err = 0;
	response = NULL;

	if (!ds || ds->host == NULL) {
		write_log(LOG_DEFAULT, "Refuse to connect to null hostname");
		return;
	}

	xa_debug(1, "DEBUG: Adding server to directory server [%s]", ds->host);

	
	xa_debug(2, "Sending GET /addsrv?v=1&br=%i&p=%i&m=%i&t=%s&g=%s&url=%s&irc=N/A&icq=N/A&aim=N/A HTTP/1.0", source->audiocast.bitrate, info.port[0], info.max_clients,
		 url_encode(source->audiocast.name, &url_p[0]), url_encode(source->audiocast.genre, &url_p[1]), url_encode(source->audiocast.url, &url_p[2]));

	for (c = 0; c < 3; c++) {
		if (url_p[c]) nfree(url_p[c]);
		url_p[c] = NULL;
	}

	if ((sockfd = sock_connect_wto (ds->host, ds->port, 15)) != -1) {
		sock_write_line (sockfd, "GET /addsrv?v=1&br=%i&p=%i&m=%i&t=%s&g=%s&url=%s&irc=N/A&icq=N/A&aim=N/A HTTP/1.0", source->audiocast.bitrate, info.port[0], info.max_clients,
				 url_encode(source->audiocast.name, &url_p[0]), url_encode(source->audiocast.genre, &url_p[1]), url_encode(source->audiocast.url, &url_p[2]));
		sock_write_line (sockfd, "Host: %s\r\nAccept: */*\r\n", ds->host);
	 
		for (c = 0; c < 3; c++) {
			if (url_p[c]) nfree(url_p[c]);
			url_p[c] = NULL;
		} 

		/* Skip HTTP Header */
		do {
			err = sock_read_line (sockfd, s, BUFSIZE);
			xa_debug (2, "DEBUG: server-response: [%s]", s);
		} while (err == 1 && ice_strcmp(s, "") != 0);

		do {
			err = sock_read_line (sockfd, s, BUFSIZE);
			xa_debug (2, "DEBUG: server-response: [%s]", s);
	
			temp = strstr(s, "icy-response:");
			if (temp != NULL) 
				response = (char *)(temp + 13);
			temp = strstr(s, "icy-id:");
			if (temp != NULL) 
				ds->id = atoi((char *)(temp + 7));
			temp = strstr(s, "icy-tchfrq:");
			if (temp != NULL) 
				info.touch_freq = atoi((char *)(temp + 11));
			temp = strstr(s, "icy-error:");
			if (temp != NULL) 
				error = atoi((char *)(temp + 10));
		} while (err == 1 && ice_strcmp(s, "") != 0);

		if (error == 0 && ds->id != -1) {
			write_log(LOG_DEFAULT, "directory_add([%s:%d]) completed...server id = %i", ds->host, ds->port, ds->id);
		} else {
			ds->id = -1;
			ds->counter++;
			write_log(LOG_DEFAULT, "directory_add([%s:%d]) failed... directory server error #%i... (retry in %d seconds)", ds->host, ds->port, error, (info.touch_freq * ds->counter * 60) - (get_time () - ds->touchtime));
		}
		
		sock_close(sockfd);
	} else {
		ds->id = -1;
		ds->counter++;
		write_log(LOG_DEFAULT, "directory_add([%s:%d]) failed, could not connect. (retry in %d seconds)", ds->host, ds->port, error, (info.touch_freq * ds->counter * 60) - (get_time () - ds->touchtime));
	}
}

void directory_touch(directory_server_t *ds, connection_t *con)
{
	int sockfd, error, err, avt = 0;
	char s[BUFSIZE], *temp;
	char *response, *title;
	statistics_t stat;

	error = err = 0;
	response = NULL;
	
	if (!ds || ds->host == NULL) {
		write_log(LOG_DEFAULT, "Refuse to touch a NULL host");
		return;
	}

	get_running_stats_nl(&stat);
	
	avt = stat.client_connect_time / ((stat.client_connections > 0) ? stat.client_connections : 1);
	
	xa_debug(2, "Sending: GET /cgi-bin/tchsrv?id=%i&p=%i&li=%i&alt=%d&ct=%s HTTP/1.0", ds->id, info.port[0], info.num_clients, avt, info.streamtitle);
	if ((sockfd = sock_connect_wto(ds->host, ds->port, 15)) != -1) {
		sock_write(sockfd, "GET /cgi-bin/tchsrv?id=%i&p=%i&li=%i&alt=%d&ct=%s HTTP/1.0\nHost: %s:%d\n\n", ds->id, info.port[0], info.num_clients, avt, url_encode(info.streamtitle,&title), ds->host, ds->port);
		nfree(title);

		/* Skip HTTP Header */
		do {
			err = sock_read_line(sockfd, s, BUFSIZE);
			xa_debug(2, "DEBUG: server-response: [%s]", s);
		} while (err == 1 && ice_strcmp(s, "") != 0);

		do {
			err = sock_read_line(sockfd, s, BUFSIZE);
			xa_debug(2, "DEBUG: server-response: [%s]", s);
			temp = strstr(s, "icy-response:");
			if (temp != NULL) 
				response = (char *)(temp + 13);
			temp = strstr(s, "icy-error:");
			if (temp != NULL) 
				error = atoi((char *)(temp + 10));
		} while (err == 1 && ice_strcmp(s, "") != 0);

		if (error == 0) {
			write_log(LOG_DEFAULT, "directory_touch([%s]) completed...", ds->host);
			ds->touches++;
		} else {
			if (ds->id == -1)
				ds->counter++;
			else
				ds->counter = 1;
			ds->id = -1;

			write_log(LOG_DEFAULT, "directory_touch([%s:%d]) failed... directory server error #%i... (retry in %d seconds)", ds->host, ds->port, error, (info.touch_freq * ds->counter * 60) - (get_time() - ds->touchtime));
		}
		
		sock_close(sockfd);
	} else {
		write_log(LOG_DEFAULT, "directory_touch([%s]) failed", ds->host);
	}
}

void directory_remove (directory_server_t *ds)
{
	int sockfd, error, err;
	char s[BUFSIZE], *temp;
	char *response;
	char buf[BUFSIZE];

	error = err = 0;
	response = NULL;

	if (!ds || ds->host == NULL) {
		write_log (LOG_DEFAULT, "Refuse to remove null host");
		return;
	}

	if ((sockfd = sock_connect_wto(ds->host, ds->port, 15)) != -1) {
		sock_write(sockfd, "GET /cgi-bin/remsrv?id=%i&p=%i HTTP/1.0\nHost: %s:%d\n\n", ds->id, info.port[0], ds->host, ds->port);

		/* Skip HTTP Header */
		do {
			err = sock_read_line(sockfd, buf, BUFSIZE);
			xa_debug(2, "DEBUG: server-response: [%s]", s);
		} while (err == 1 && ice_strcmp(s, "") != 0);

		do {
			err = sock_read_line(sockfd, buf, BUFSIZE);
			xa_debug(2, "DEBUG: server-response: [%s]", s);
			temp = strstr(s, "icy-response:");
			if (temp != NULL) 
				response = (char *)(temp + 13);
			temp = strstr(s, "icy-error:");
			if (temp != NULL) 
				error = atoi((char *)(temp + 10));
		} while (err == 1 && ice_strcmp(s, "") != 0);

		if (error == 0) {
			write_log(LOG_DEFAULT, "directory_remove([%s]) completed...", ds->host);
		} else {
			write_log(LOG_DEFAULT, "directory_remove([%s]) failed... directory server error #%i...", ds->host, error);
		}
		
		sock_close(sockfd);
	} else {
		write_log(LOG_DEFAULT, "directory_remove([%s]) failed", ds->host);
	}
}

void avl_touch_directory(void *data, void *param)
{
	directory_server_t *dir = (directory_server_t *)data;
	connection_t *defcon = get_default_mount ();

	if (dir->type == icy_e) {
		if (dir->id >= 0)
			directory_touch(dir, defcon);
		else
			directory_add(dir, defcon);
	} else {
		directory_touch_xa(dir);
	}
}
  
void update_stream_id(char *dshost, char *mountpoint, int id)
{
	avl_traverser trav = {0};
	connection_t *sourcecon;
 	relay_id_t *out, *rip = (relay_id_t *)nmalloc(sizeof(relay_id_t));

	rip->id = id;
	rip->host = nstrdup(dshost);

	thread_mutex_lock(&info.source_mutex);
  	while ((sourcecon = avl_traverse (info.sources, &trav))) {
		xa_debug(2, "DEBUG: update_stream_id () Looking for mountpoint [%s] on dshost: [%s]", mountpoint, dshost);
		if (ice_strcmp(sourcecon->food.source->audiocast.mount, mountpoint) == 0) {
			out = avl_replace(sourcecon->food.source->relay_tree, rip);
			xa_debug(2, "DEBUG: update_stream_id () Gave source %d new id for dshost: [%s] with %d", sourcecon->id, dshost, id);

			    if (out) {
				    nfree(out->host);
				    nfree(out);
			    }

			    thread_mutex_unlock(&info.source_mutex);
			    return;
		}
	}
  
	nfree(rip->host);
	nfree(rip);

	thread_mutex_unlock(&info.source_mutex);
	xa_debug(2, "DEBUG: update_stream_id (): Found no stream with mountpoint %s for dshost [%s]", mountpoint, dshost);
}

void free_directory_servers()
{
	directory_server_t *ds, *out;

	if (!info.d_servers) {
		write_log (LOG_DEFAULT, "WARNING: info.d_servers is NULL, weird!");
		return;
	}

	thread_mutex_lock(&info.directory_mutex);
	while ((ds = avl_get_any_node(info.d_servers))) {

		if (!ds) {
			write_log (LOG_DEFAULT, "WARNING: avl_get_any_node() returned NULL directory server");
			continue;
		}

		out = avl_delete(info.d_servers, ds);
		if (out) {
			nfree(out->path);
			nfree(out->host);
			nfree(out);
		}
	}
	thread_mutex_unlock(&info.directory_mutex);
}
