/* ****************************************************************************
 *
 * Copyright (c) Microsoft Corporation. 
 *
 * This source code is subject to terms and conditions of the Microsoft Public
 * License. A  copy of the license can be found in the License.html file at the
 * root of this distribution. If  you cannot locate the  Microsoft Public
 * License, please send an email to  dlr@microsoft.com. By using this source
 * code in any fashion, you are agreeing to be bound by the terms of the 
 * Microsoft Public License.
 *
 * You must not remove this notice, or any other, from this software.
 *
 * ***************************************************************************/

using System;
using System.Diagnostics;
using System.Threading;
using IronMath;
using IronPython.Runtime.Operations;
using IronPython.Runtime;
using IronPython.Runtime.Types;


namespace IronPython.Runtime.Operations {
    static partial class ByteOps {

        #region Generated ByteOps

        // *** BEGIN GENERATED CODE ***

        private static ReflectedType ByteType;
        public static DynamicType MakeDynamicType() {
            if (ByteType == null) {
                OpsReflectedType ort = new OpsReflectedType("Byte", typeof(Byte), typeof(ByteOps), null);
                if (Interlocked.CompareExchange<ReflectedType>(ref ByteType, ort, null) == null) {
                    return ort;
                }
            }
            return ByteType;
        }

        [PythonName("__new__")]
        public static object Make(DynamicType cls) {
            return Make(cls, default(Byte));
        }

        [PythonName("__new__")]
        public static object Make(DynamicType cls, object value) {
            if (cls != ByteType) {
                throw Ops.TypeError("Byte.__new__: first argument must be Byte type.");
            }
            IConvertible valueConvertible;
            if ((valueConvertible = value as IConvertible) != null) {
                switch (valueConvertible.GetTypeCode()) {
                    case TypeCode.Byte: return (Byte)(Byte)value;
                    case TypeCode.SByte: return (Byte)(SByte)value;
                    case TypeCode.Int16: return (Byte)(Int16)value;
                    case TypeCode.UInt16: return (Byte)(UInt16)value;
                    case TypeCode.Int32: return (Byte)(Int32)value;
                    case TypeCode.UInt32: return (Byte)(UInt32)value;
                    case TypeCode.Int64: return (Byte)(Int64)value;
                    case TypeCode.UInt64: return (Byte)(UInt64)value;
                    case TypeCode.Single: return (Byte)(Single)value;
                    case TypeCode.Double: return (Byte)(Double)value;
                }
            }
            if (value is String) {
                return Byte.Parse((String)value);
            } else if (value is BigInteger) {
                return (Byte)(BigInteger)value;
            } else if (value is ExtensibleInt) {
                return (Byte)((ExtensibleInt)value).value;
            } else if (value is ExtensibleLong) {
                return (Byte)((ExtensibleLong)value).Value;
            } else if (value is ExtensibleFloat) {
                return (Byte)((ExtensibleFloat)value).value;
            } else if (value is Enum) {
                return Converter.CastEnumToByte(value);
            }
            throw Ops.ValueError("invalid value for Byte.__new__");
        }

        [PythonName("__add__")]
        public static object Add(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__add__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Int16 result = (Int16)(((Int16)leftByte) + ((Int16)((Boolean)right ? (Byte)1 : (Byte)0)));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Byte: {
                            Int16 result = (Int16)(((Int16)leftByte) + ((Int16)(Byte)right));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.SByte: {
                            Int16 result = (Int16)(((Int16)leftByte) + ((Int16)(SByte)right));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Int16: {
                            Int32 result = (Int32)(((Int32)leftByte) + ((Int32)(Int16)right));
                            if (Int16.MinValue <= result && result <= Int16.MaxValue) {
                                return (Int16)result;
                            } else return result;
                        }
                    case TypeCode.UInt16: {
                            Int32 result = (Int32)(((Int32)leftByte) + ((Int32)(UInt16)right));
                            if (UInt16.MinValue <= result && result <= UInt16.MaxValue) {
                                return (UInt16)result;
                            } else return result;
                        }
                    case TypeCode.Int32: {
                            Int64 result = (Int64)(((Int64)leftByte) + ((Int64)(Int32)right));
                            if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                                return (Int32)result;
                            } else return result;
                        }
                    case TypeCode.UInt32: {
                            Int64 result = (Int64)(((Int64)leftByte) + ((Int64)(UInt32)right));
                            if (UInt32.MinValue <= result && result <= UInt32.MaxValue) {
                                return (UInt32)result;
                            } else return result;
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.Add(leftByte, (Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.AddImpl(leftByte, (UInt64)right);
                        }
                    case TypeCode.Single: {
                            return SingleOps.AddImpl((Single)leftByte, (Single)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.Add((Double)leftByte, (Double)(Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.Add(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.Add(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                Int64 result = (Int64)(((Int64)leftByte) + ((Int64)((ExtensibleInt)right).value));
                if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                    return (Int32)result;
                } else return result;
            } else if (right is ExtensibleLong) {
                return LongOps.Add(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.Add((Double)leftByte, (Double)((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.Add(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__div__")]
        public static object Divide(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__div__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            return ByteOps.DivideImpl((Byte)leftByte, (Byte)((Boolean)right ? (Byte)1 : (Byte)0));
                        }
                    case TypeCode.Byte: {
                            return ByteOps.DivideImpl((Byte)leftByte, (Byte)(Byte)right);
                        }
                    case TypeCode.SByte: {
                            return Int16Ops.DivideImpl((Int16)leftByte, (Int16)(SByte)right);
                        }
                    case TypeCode.Int16: {
                            return Int16Ops.DivideImpl((Int16)leftByte, (Int16)(Int16)right);
                        }
                    case TypeCode.UInt16: {
                            return UInt16Ops.DivideImpl((UInt16)leftByte, (UInt16)(UInt16)right);
                        }
                    case TypeCode.Int32: {
                            return IntOps.Divide((Int32)leftByte, (Int32)(Int32)right);
                        }
                    case TypeCode.UInt32: {
                            return UInt32Ops.DivideImpl((UInt32)leftByte, (UInt32)(UInt32)right);
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.Divide((Int64)leftByte, (Int64)(Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.DivideImpl((UInt64)leftByte, (UInt64)(UInt64)right);
                        }
                    case TypeCode.Single: {
                            return SingleOps.DivideImpl((Single)leftByte, (Single)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.Divide((Double)leftByte, (Double)(Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.Divide(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.Divide(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                return IntOps.Divide((Int32)leftByte, (Int32)((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.Divide(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.Divide((Double)leftByte, (Double)((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.Divide(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__floordiv__")]
        public static object FloorDivide(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__floordiv__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            return ByteOps.FloorDivideImpl((Byte)leftByte, (Byte)((Boolean)right ? (Byte)1 : (Byte)0));
                        }
                    case TypeCode.Byte: {
                            return ByteOps.FloorDivideImpl((Byte)leftByte, (Byte)(Byte)right);
                        }
                    case TypeCode.SByte: {
                            return Int16Ops.FloorDivideImpl((Int16)leftByte, (Int16)(SByte)right);
                        }
                    case TypeCode.Int16: {
                            return Int16Ops.FloorDivideImpl((Int16)leftByte, (Int16)(Int16)right);
                        }
                    case TypeCode.UInt16: {
                            return UInt16Ops.FloorDivideImpl((UInt16)leftByte, (UInt16)(UInt16)right);
                        }
                    case TypeCode.Int32: {
                            return IntOps.FloorDivide((Int32)leftByte, (Int32)(Int32)right);
                        }
                    case TypeCode.UInt32: {
                            return UInt32Ops.FloorDivideImpl((UInt32)leftByte, (UInt32)(UInt32)right);
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.FloorDivide((Int64)leftByte, (Int64)(Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.FloorDivideImpl((UInt64)leftByte, (UInt64)(UInt64)right);
                        }
                    case TypeCode.Single: {
                            return SingleOps.FloorDivideImpl((Single)leftByte, (Single)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.FloorDivide((Double)leftByte, (Double)(Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.FloorDivide(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.FloorDivide(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                return IntOps.FloorDivide((Int32)leftByte, (Int32)((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.FloorDivide(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.FloorDivide((Double)leftByte, (Double)((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.FloorDivide(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__mod__")]
        public static object Mod(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__mod__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            return ByteOps.ModImpl((Byte)leftByte, (Byte)((Boolean)right ? (Byte)1 : (Byte)0));
                        }
                    case TypeCode.Byte: {
                            return ByteOps.ModImpl((Byte)leftByte, (Byte)(Byte)right);
                        }
                    case TypeCode.SByte: {
                            return Int16Ops.ModImpl((Int16)leftByte, (Int16)(SByte)right);
                        }
                    case TypeCode.Int16: {
                            return Int16Ops.ModImpl((Int16)leftByte, (Int16)(Int16)right);
                        }
                    case TypeCode.UInt16: {
                            return UInt16Ops.ModImpl((UInt16)leftByte, (UInt16)(UInt16)right);
                        }
                    case TypeCode.Int32: {
                            return IntOps.Mod((Int32)leftByte, (Int32)(Int32)right);
                        }
                    case TypeCode.UInt32: {
                            return UInt32Ops.ModImpl((UInt32)leftByte, (UInt32)(UInt32)right);
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.Mod((Int64)leftByte, (Int64)(Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.ModImpl((UInt64)leftByte, (UInt64)(UInt64)right);
                        }
                    case TypeCode.Single: {
                            return SingleOps.ModImpl((Single)leftByte, (Single)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.Mod((Double)leftByte, (Double)(Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.Mod(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.Mod(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                return IntOps.Mod((Int32)leftByte, (Int32)((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.Mod(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.Mod((Double)leftByte, (Double)((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.Mod(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__mul__")]
        public static object Multiply(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__mul__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Byte result = (Byte)(((Byte)leftByte) * ((Byte)((Boolean)right ? (Byte)1 : (Byte)0)));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Byte: {
                            UInt16 result = (UInt16)(((UInt16)leftByte) * ((UInt16)(Byte)right));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.SByte: {
                            Int16 result = (Int16)(((Int16)leftByte) * ((Int16)(SByte)right));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Int16: {
                            Int32 result = (Int32)(((Int32)leftByte) * ((Int32)(Int16)right));
                            if (Int16.MinValue <= result && result <= Int16.MaxValue) {
                                return (Int16)result;
                            } else return result;
                        }
                    case TypeCode.UInt16: {
                            Int32 result = (Int32)(((Int32)leftByte) * ((Int32)(UInt16)right));
                            if (UInt16.MinValue <= result && result <= UInt16.MaxValue) {
                                return (UInt16)result;
                            } else return result;
                        }
                    case TypeCode.Int32: {
                            Int64 result = (Int64)(((Int64)leftByte) * ((Int64)(Int32)right));
                            if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                                return (Int32)result;
                            } else return result;
                        }
                    case TypeCode.UInt32: {
                            Int64 result = (Int64)(((Int64)leftByte) * ((Int64)(UInt32)right));
                            if (UInt32.MinValue <= result && result <= UInt32.MaxValue) {
                                return (UInt32)result;
                            } else return result;
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.Multiply(leftByte, (Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.MultiplyImpl(leftByte, (UInt64)right);
                        }
                    case TypeCode.Single: {
                            return FloatOps.Multiply((Double)leftByte, (Double)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.Multiply(leftByte, (Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.Multiply(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.Multiply(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                Int64 result = (Int64)(((Int64)leftByte) * ((Int64)((ExtensibleInt)right).value));
                if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                    return (Int32)result;
                } else return result;
            } else if (right is ExtensibleLong) {
                return LongOps.Multiply(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.Multiply(leftByte, ((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.Multiply(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__sub__")]
        public static object Subtract(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__sub__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Int16 result = (Int16)(((Int16)leftByte) - ((Int16)((Boolean)right ? (Byte)1 : (Byte)0)));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Byte: {
                            Int16 result = (Int16)(((Int16)leftByte) - ((Int16)(Byte)right));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.SByte: {
                            Int16 result = (Int16)(((Int16)leftByte) - ((Int16)(SByte)right));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Int16: {
                            Int32 result = (Int32)(((Int32)leftByte) - ((Int32)(Int16)right));
                            if (Int16.MinValue <= result && result <= Int16.MaxValue) {
                                return (Int16)result;
                            } else return result;
                        }
                    case TypeCode.UInt16: {
                            Int32 result = (Int32)(((Int32)leftByte) - ((Int32)(UInt16)right));
                            if (UInt16.MinValue <= result && result <= UInt16.MaxValue) {
                                return (UInt16)result;
                            } else return result;
                        }
                    case TypeCode.Int32: {
                            Int64 result = (Int64)(((Int64)leftByte) - ((Int64)(Int32)right));
                            if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                                return (Int32)result;
                            } else return result;
                        }
                    case TypeCode.UInt32: {
                            Int64 result = (Int64)(((Int64)leftByte) - ((Int64)(UInt32)right));
                            if (UInt32.MinValue <= result && result <= UInt32.MaxValue) {
                                return (UInt32)result;
                            } else return result;
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.Subtract(leftByte, (Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.SubtractImpl(leftByte, (UInt64)right);
                        }
                    case TypeCode.Single: {
                            return SingleOps.SubtractImpl((Single)leftByte, (Single)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.Subtract((Double)leftByte, (Double)(Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.Subtract(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.Subtract(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                Int64 result = (Int64)(((Int64)leftByte) - ((Int64)((ExtensibleInt)right).value));
                if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                    return (Int32)result;
                } else return result;
            } else if (right is ExtensibleLong) {
                return LongOps.Subtract(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.Subtract((Double)leftByte, (Double)((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.Subtract(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__radd__")]
        public static object ReverseAdd(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__radd__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Int16 result = (Int16)(((Int16)((Boolean)right ? (Byte)1 : (Byte)0)) + ((Int16)leftByte));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Byte: {
                            Int16 result = (Int16)(((Int16)(Byte)right) + ((Int16)leftByte));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.SByte: {
                            Int16 result = (Int16)(((Int16)(SByte)right) + ((Int16)leftByte));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Int16: {
                            Int32 result = (Int32)(((Int32)(Int16)right) + ((Int32)leftByte));
                            if (Int16.MinValue <= result && result <= Int16.MaxValue) {
                                return (Int16)result;
                            } else return result;
                        }
                    case TypeCode.UInt16: {
                            Int32 result = (Int32)(((Int32)(UInt16)right) + ((Int32)leftByte));
                            if (UInt16.MinValue <= result && result <= UInt16.MaxValue) {
                                return (UInt16)result;
                            } else return result;
                        }
                    case TypeCode.Int32: {
                            Int64 result = (Int64)(((Int64)(Int32)right) + ((Int64)leftByte));
                            if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                                return (Int32)result;
                            } else return result;
                        }
                    case TypeCode.UInt32: {
                            Int64 result = (Int64)(((Int64)(UInt32)right) + ((Int64)leftByte));
                            if (UInt32.MinValue <= result && result <= UInt32.MaxValue) {
                                return (UInt32)result;
                            } else return result;
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.ReverseAdd(leftByte, (Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.ReverseAddImpl(leftByte, (UInt64)right);
                        }
                    case TypeCode.Single: {
                            return SingleOps.ReverseAddImpl((Single)leftByte, (Single)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.ReverseAdd((Double)leftByte, (Double)(Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.ReverseAdd(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.ReverseAdd(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                Int64 result = (Int64)(((Int64)((ExtensibleInt)right).value) + ((Int64)leftByte));
                if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                    return (Int32)result;
                } else return result;
            } else if (right is ExtensibleLong) {
                return LongOps.ReverseAdd(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.ReverseAdd((Double)leftByte, (Double)((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.ReverseAdd(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rdiv__")]
        public static object ReverseDivide(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rdiv__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            return ByteOps.ReverseDivideImpl((Byte)leftByte, (Byte)((Boolean)right ? (Byte)1 : (Byte)0));
                        }
                    case TypeCode.Byte: {
                            return ByteOps.ReverseDivideImpl((Byte)leftByte, (Byte)(Byte)right);
                        }
                    case TypeCode.SByte: {
                            return Int16Ops.ReverseDivideImpl((Int16)leftByte, (Int16)(SByte)right);
                        }
                    case TypeCode.Int16: {
                            return Int16Ops.ReverseDivideImpl((Int16)leftByte, (Int16)(Int16)right);
                        }
                    case TypeCode.UInt16: {
                            return UInt16Ops.ReverseDivideImpl((UInt16)leftByte, (UInt16)(UInt16)right);
                        }
                    case TypeCode.Int32: {
                            return IntOps.ReverseDivide((Int32)leftByte, (Int32)(Int32)right);
                        }
                    case TypeCode.UInt32: {
                            return UInt32Ops.ReverseDivideImpl((UInt32)leftByte, (UInt32)(UInt32)right);
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.ReverseDivide((Int64)leftByte, (Int64)(Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.ReverseDivideImpl((UInt64)leftByte, (UInt64)(UInt64)right);
                        }
                    case TypeCode.Single: {
                            return SingleOps.ReverseDivideImpl((Single)leftByte, (Single)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.ReverseDivide((Double)leftByte, (Double)(Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.ReverseDivide(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.ReverseDivide(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                return IntOps.ReverseDivide((Int32)leftByte, (Int32)((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.ReverseDivide(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.ReverseDivide((Double)leftByte, (Double)((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.ReverseDivide(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rfloordiv__")]
        public static object ReverseFloorDivide(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rfloordiv__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            return ByteOps.ReverseFloorDivideImpl((Byte)leftByte, (Byte)((Boolean)right ? (Byte)1 : (Byte)0));
                        }
                    case TypeCode.Byte: {
                            return ByteOps.ReverseFloorDivideImpl((Byte)leftByte, (Byte)(Byte)right);
                        }
                    case TypeCode.SByte: {
                            return Int16Ops.ReverseFloorDivideImpl((Int16)leftByte, (Int16)(SByte)right);
                        }
                    case TypeCode.Int16: {
                            return Int16Ops.ReverseFloorDivideImpl((Int16)leftByte, (Int16)(Int16)right);
                        }
                    case TypeCode.UInt16: {
                            return UInt16Ops.ReverseFloorDivideImpl((UInt16)leftByte, (UInt16)(UInt16)right);
                        }
                    case TypeCode.Int32: {
                            return IntOps.ReverseFloorDivide((Int32)leftByte, (Int32)(Int32)right);
                        }
                    case TypeCode.UInt32: {
                            return UInt32Ops.ReverseFloorDivideImpl((UInt32)leftByte, (UInt32)(UInt32)right);
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.ReverseFloorDivide((Int64)leftByte, (Int64)(Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.ReverseFloorDivideImpl((UInt64)leftByte, (UInt64)(UInt64)right);
                        }
                    case TypeCode.Single: {
                            return SingleOps.ReverseFloorDivideImpl((Single)leftByte, (Single)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.ReverseFloorDivide((Double)leftByte, (Double)(Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.ReverseFloorDivide(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.ReverseFloorDivide(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                return IntOps.ReverseFloorDivide((Int32)leftByte, (Int32)((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.ReverseFloorDivide(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.ReverseFloorDivide((Double)leftByte, (Double)((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.ReverseFloorDivide(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rmod__")]
        public static object ReverseMod(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rmod__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            return ByteOps.ReverseModImpl((Byte)leftByte, (Byte)((Boolean)right ? (Byte)1 : (Byte)0));
                        }
                    case TypeCode.Byte: {
                            return ByteOps.ReverseModImpl((Byte)leftByte, (Byte)(Byte)right);
                        }
                    case TypeCode.SByte: {
                            return Int16Ops.ReverseModImpl((Int16)leftByte, (Int16)(SByte)right);
                        }
                    case TypeCode.Int16: {
                            return Int16Ops.ReverseModImpl((Int16)leftByte, (Int16)(Int16)right);
                        }
                    case TypeCode.UInt16: {
                            return UInt16Ops.ReverseModImpl((UInt16)leftByte, (UInt16)(UInt16)right);
                        }
                    case TypeCode.Int32: {
                            return IntOps.ReverseMod((Int32)leftByte, (Int32)(Int32)right);
                        }
                    case TypeCode.UInt32: {
                            return UInt32Ops.ReverseModImpl((UInt32)leftByte, (UInt32)(UInt32)right);
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.ReverseMod((Int64)leftByte, (Int64)(Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.ReverseModImpl((UInt64)leftByte, (UInt64)(UInt64)right);
                        }
                    case TypeCode.Single: {
                            return SingleOps.ReverseModImpl((Single)leftByte, (Single)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.ReverseMod((Double)leftByte, (Double)(Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.ReverseMod(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.ReverseMod(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                return IntOps.ReverseMod((Int32)leftByte, (Int32)((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.ReverseMod(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.ReverseMod((Double)leftByte, (Double)((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.ReverseMod(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rmul__")]
        public static object ReverseMultiply(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rmul__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Byte result = (Byte)(((Byte)((Boolean)right ? (Byte)1 : (Byte)0)) * ((Byte)leftByte));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Byte: {
                            UInt16 result = (UInt16)(((UInt16)(Byte)right) * ((UInt16)leftByte));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.SByte: {
                            Int16 result = (Int16)(((Int16)(SByte)right) * ((Int16)leftByte));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Int16: {
                            Int32 result = (Int32)(((Int32)(Int16)right) * ((Int32)leftByte));
                            if (Int16.MinValue <= result && result <= Int16.MaxValue) {
                                return (Int16)result;
                            } else return result;
                        }
                    case TypeCode.UInt16: {
                            Int32 result = (Int32)(((Int32)(UInt16)right) * ((Int32)leftByte));
                            if (UInt16.MinValue <= result && result <= UInt16.MaxValue) {
                                return (UInt16)result;
                            } else return result;
                        }
                    case TypeCode.Int32: {
                            Int64 result = (Int64)(((Int64)(Int32)right) * ((Int64)leftByte));
                            if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                                return (Int32)result;
                            } else return result;
                        }
                    case TypeCode.UInt32: {
                            Int64 result = (Int64)(((Int64)(UInt32)right) * ((Int64)leftByte));
                            if (UInt32.MinValue <= result && result <= UInt32.MaxValue) {
                                return (UInt32)result;
                            } else return result;
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.ReverseMultiply(leftByte, (Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.ReverseMultiplyImpl(leftByte, (UInt64)right);
                        }
                    case TypeCode.Single: {
                            return FloatOps.ReverseMultiply((Double)leftByte, (Double)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.ReverseMultiply(leftByte, (Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.ReverseMultiply(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.ReverseMultiply(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                Int64 result = (Int64)(((Int64)((ExtensibleInt)right).value) * ((Int64)leftByte));
                if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                    return (Int32)result;
                } else return result;
            } else if (right is ExtensibleLong) {
                return LongOps.ReverseMultiply(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.ReverseMultiply(leftByte, ((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.ReverseMultiply(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rsub__")]
        public static object ReverseSubtract(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rsub__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Int16 result = (Int16)(((Int16)((Boolean)right ? (Byte)1 : (Byte)0)) - ((Int16)leftByte));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Byte: {
                            Int16 result = (Int16)(((Int16)(Byte)right) - ((Int16)leftByte));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.SByte: {
                            Int16 result = (Int16)(((Int16)(SByte)right) - ((Int16)leftByte));
                            if (Byte.MinValue <= result && result <= Byte.MaxValue) {
                                return (Byte)result;
                            } else return result;
                        }
                    case TypeCode.Int16: {
                            Int32 result = (Int32)(((Int32)(Int16)right) - ((Int32)leftByte));
                            if (Int16.MinValue <= result && result <= Int16.MaxValue) {
                                return (Int16)result;
                            } else return result;
                        }
                    case TypeCode.UInt16: {
                            Int32 result = (Int32)(((Int32)(UInt16)right) - ((Int32)leftByte));
                            if (UInt16.MinValue <= result && result <= UInt16.MaxValue) {
                                return (UInt16)result;
                            } else return result;
                        }
                    case TypeCode.Int32: {
                            Int64 result = (Int64)(((Int64)(Int32)right) - ((Int64)leftByte));
                            if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                                return (Int32)result;
                            } else return result;
                        }
                    case TypeCode.UInt32: {
                            Int64 result = (Int64)(((Int64)(UInt32)right) - ((Int64)leftByte));
                            if (UInt32.MinValue <= result && result <= UInt32.MaxValue) {
                                return (UInt32)result;
                            } else return result;
                        }
                    case TypeCode.Int64: {
                            return Int64Ops.ReverseSubtract(leftByte, (Int64)right);
                        }
                    case TypeCode.UInt64: {
                            return UInt64Ops.ReverseSubtractImpl(leftByte, (UInt64)right);
                        }
                    case TypeCode.Single: {
                            return SingleOps.ReverseSubtractImpl((Single)leftByte, (Single)(Single)right);
                        }
                    case TypeCode.Double: {
                            return FloatOps.ReverseSubtract((Double)leftByte, (Double)(Double)right);
                        }
                }
            }
            if (right is BigInteger) {
                return LongOps.ReverseSubtract(leftByte, (BigInteger)right);
            } else if (right is Complex64) {
                return ComplexOps.ReverseSubtract(leftByte, (Complex64)right);
            } else if (right is ExtensibleInt) {
                Int64 result = (Int64)(((Int64)((ExtensibleInt)right).value) - ((Int64)leftByte));
                if (Int32.MinValue <= result && result <= Int32.MaxValue) {
                    return (Int32)result;
                } else return result;
            } else if (right is ExtensibleLong) {
                return LongOps.ReverseSubtract(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is ExtensibleFloat) {
                return FloatOps.ReverseSubtract((Double)leftByte, (Double)((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.ReverseSubtract(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__and__")]
        public static object BitwiseAnd(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__and__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Byte rightByte = (Byte)((Boolean)right ? (Byte)1 : (Byte)0);
                            return leftByte & rightByte;
                        }
                    case TypeCode.Byte: {
                            return leftByte & (Byte)right;
                        }
                    case TypeCode.SByte: {
                            Int16 leftInt16 = (Int16)leftByte;
                            Int16 rightInt16 = (Int16)(SByte)right;
                            return leftInt16 & rightInt16;
                        }
                    case TypeCode.Int16: {
                            Int16 leftInt16 = (Int16)leftByte;
                            return leftInt16 & (Int16)right;
                        }
                    case TypeCode.UInt16: {
                            UInt16 leftUInt16 = (UInt16)leftByte;
                            return leftUInt16 & (UInt16)right;
                        }
                    case TypeCode.Int32: {
                            Int32 leftInt32 = (Int32)leftByte;
                            return leftInt32 & (Int32)right;
                        }
                    case TypeCode.UInt32: {
                            UInt32 leftUInt32 = (UInt32)leftByte;
                            return leftUInt32 & (UInt32)right;
                        }
                    case TypeCode.Int64: {
                            Int64 leftInt64 = (Int64)leftByte;
                            return leftInt64 & (Int64)right;
                        }
                    case TypeCode.UInt64: {
                            UInt64 leftUInt64 = (UInt64)leftByte;
                            return leftUInt64 & (UInt64)right;
                        }
                }
            }
            if (right is BigInteger) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger & (BigInteger)right;
            } else if (right is ExtensibleInt) {
                Int32 leftInt32 = (Int32)leftByte;
                return leftInt32 & (Int32)((ExtensibleInt)right).value;
            } else if (right is ExtensibleLong) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger & (BigInteger)((ExtensibleLong)right).Value;
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rand__")]
        public static object ReverseBitwiseAnd(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rand__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Byte rightByte = (Byte)((Boolean)right ? (Byte)1 : (Byte)0);
                            return leftByte & rightByte;
                        }
                    case TypeCode.Byte: {
                            return leftByte & (Byte)right;
                        }
                    case TypeCode.SByte: {
                            Int16 leftInt16 = (Int16)leftByte;
                            Int16 rightInt16 = (Int16)(SByte)right;
                            return leftInt16 & rightInt16;
                        }
                    case TypeCode.Int16: {
                            Int16 leftInt16 = (Int16)leftByte;
                            return leftInt16 & (Int16)right;
                        }
                    case TypeCode.UInt16: {
                            UInt16 leftUInt16 = (UInt16)leftByte;
                            return leftUInt16 & (UInt16)right;
                        }
                    case TypeCode.Int32: {
                            Int32 leftInt32 = (Int32)leftByte;
                            return leftInt32 & (Int32)right;
                        }
                    case TypeCode.UInt32: {
                            UInt32 leftUInt32 = (UInt32)leftByte;
                            return leftUInt32 & (UInt32)right;
                        }
                    case TypeCode.Int64: {
                            Int64 leftInt64 = (Int64)leftByte;
                            return leftInt64 & (Int64)right;
                        }
                    case TypeCode.UInt64: {
                            UInt64 leftUInt64 = (UInt64)leftByte;
                            return leftUInt64 & (UInt64)right;
                        }
                }
            }
            if (right is BigInteger) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger & (BigInteger)right;
            } else if (right is ExtensibleInt) {
                Int32 leftInt32 = (Int32)leftByte;
                return leftInt32 & (Int32)((ExtensibleInt)right).value;
            } else if (right is ExtensibleLong) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger & (BigInteger)((ExtensibleLong)right).Value;
            }
            return Ops.NotImplemented;
        }
        [PythonName("__or__")]
        public static object BitwiseOr(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__or__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Byte rightByte = (Byte)((Boolean)right ? (Byte)1 : (Byte)0);
                            return leftByte | rightByte;
                        }
                    case TypeCode.Byte: {
                            return leftByte | (Byte)right;
                        }
                    case TypeCode.SByte: {
                            Int16 leftInt16 = (Int16)leftByte;
                            Int16 rightInt16 = (Int16)(SByte)right;
                            return leftInt16 | rightInt16;
                        }
                    case TypeCode.Int16: {
                            Int16 leftInt16 = (Int16)leftByte;
                            return leftInt16 | (Int16)right;
                        }
                    case TypeCode.UInt16: {
                            UInt16 leftUInt16 = (UInt16)leftByte;
                            return leftUInt16 | (UInt16)right;
                        }
                    case TypeCode.Int32: {
                            Int32 leftInt32 = (Int32)leftByte;
                            return leftInt32 | (Int32)right;
                        }
                    case TypeCode.UInt32: {
                            UInt32 leftUInt32 = (UInt32)leftByte;
                            return leftUInt32 | (UInt32)right;
                        }
                    case TypeCode.Int64: {
                            Int64 leftInt64 = (Int64)leftByte;
                            return leftInt64 | (Int64)right;
                        }
                    case TypeCode.UInt64: {
                            UInt64 leftUInt64 = (UInt64)leftByte;
                            return leftUInt64 | (UInt64)right;
                        }
                }
            }
            if (right is BigInteger) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger | (BigInteger)right;
            } else if (right is ExtensibleInt) {
                Int32 leftInt32 = (Int32)leftByte;
                return leftInt32 | (Int32)((ExtensibleInt)right).value;
            } else if (right is ExtensibleLong) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger | (BigInteger)((ExtensibleLong)right).Value;
            }
            return Ops.NotImplemented;
        }
        [PythonName("__ror__")]
        public static object ReverseBitwiseOr(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__ror__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Byte rightByte = (Byte)((Boolean)right ? (Byte)1 : (Byte)0);
                            return leftByte | rightByte;
                        }
                    case TypeCode.Byte: {
                            return leftByte | (Byte)right;
                        }
                    case TypeCode.SByte: {
                            Int16 leftInt16 = (Int16)leftByte;
                            Int16 rightInt16 = (Int16)(SByte)right;
                            return leftInt16 | rightInt16;
                        }
                    case TypeCode.Int16: {
                            Int16 leftInt16 = (Int16)leftByte;
                            return leftInt16 | (Int16)right;
                        }
                    case TypeCode.UInt16: {
                            UInt16 leftUInt16 = (UInt16)leftByte;
                            return leftUInt16 | (UInt16)right;
                        }
                    case TypeCode.Int32: {
                            Int32 leftInt32 = (Int32)leftByte;
                            return leftInt32 | (Int32)right;
                        }
                    case TypeCode.UInt32: {
                            UInt32 leftUInt32 = (UInt32)leftByte;
                            return leftUInt32 | (UInt32)right;
                        }
                    case TypeCode.Int64: {
                            Int64 leftInt64 = (Int64)leftByte;
                            return leftInt64 | (Int64)right;
                        }
                    case TypeCode.UInt64: {
                            UInt64 leftUInt64 = (UInt64)leftByte;
                            return leftUInt64 | (UInt64)right;
                        }
                }
            }
            if (right is BigInteger) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger | (BigInteger)right;
            } else if (right is ExtensibleInt) {
                Int32 leftInt32 = (Int32)leftByte;
                return leftInt32 | (Int32)((ExtensibleInt)right).value;
            } else if (right is ExtensibleLong) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger | (BigInteger)((ExtensibleLong)right).Value;
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rxor__")]
        public static object BitwiseXor(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rxor__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Byte rightByte = (Byte)((Boolean)right ? (Byte)1 : (Byte)0);
                            return leftByte ^ rightByte;
                        }
                    case TypeCode.Byte: {
                            return leftByte ^ (Byte)right;
                        }
                    case TypeCode.SByte: {
                            Int16 leftInt16 = (Int16)leftByte;
                            Int16 rightInt16 = (Int16)(SByte)right;
                            return leftInt16 ^ rightInt16;
                        }
                    case TypeCode.Int16: {
                            Int16 leftInt16 = (Int16)leftByte;
                            return leftInt16 ^ (Int16)right;
                        }
                    case TypeCode.UInt16: {
                            UInt16 leftUInt16 = (UInt16)leftByte;
                            return leftUInt16 ^ (UInt16)right;
                        }
                    case TypeCode.Int32: {
                            Int32 leftInt32 = (Int32)leftByte;
                            return leftInt32 ^ (Int32)right;
                        }
                    case TypeCode.UInt32: {
                            UInt32 leftUInt32 = (UInt32)leftByte;
                            return leftUInt32 ^ (UInt32)right;
                        }
                    case TypeCode.Int64: {
                            Int64 leftInt64 = (Int64)leftByte;
                            return leftInt64 ^ (Int64)right;
                        }
                    case TypeCode.UInt64: {
                            UInt64 leftUInt64 = (UInt64)leftByte;
                            return leftUInt64 ^ (UInt64)right;
                        }
                }
            }
            if (right is BigInteger) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger ^ (BigInteger)right;
            } else if (right is ExtensibleInt) {
                Int32 leftInt32 = (Int32)leftByte;
                return leftInt32 ^ (Int32)((ExtensibleInt)right).value;
            } else if (right is ExtensibleLong) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger ^ (BigInteger)((ExtensibleLong)right).Value;
            }
            return Ops.NotImplemented;
        }
        [PythonName("__xor__")]
        public static object ReverseBitwiseXor(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__xor__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean: {
                            Byte rightByte = (Byte)((Boolean)right ? (Byte)1 : (Byte)0);
                            return leftByte ^ rightByte;
                        }
                    case TypeCode.Byte: {
                            return leftByte ^ (Byte)right;
                        }
                    case TypeCode.SByte: {
                            Int16 leftInt16 = (Int16)leftByte;
                            Int16 rightInt16 = (Int16)(SByte)right;
                            return leftInt16 ^ rightInt16;
                        }
                    case TypeCode.Int16: {
                            Int16 leftInt16 = (Int16)leftByte;
                            return leftInt16 ^ (Int16)right;
                        }
                    case TypeCode.UInt16: {
                            UInt16 leftUInt16 = (UInt16)leftByte;
                            return leftUInt16 ^ (UInt16)right;
                        }
                    case TypeCode.Int32: {
                            Int32 leftInt32 = (Int32)leftByte;
                            return leftInt32 ^ (Int32)right;
                        }
                    case TypeCode.UInt32: {
                            UInt32 leftUInt32 = (UInt32)leftByte;
                            return leftUInt32 ^ (UInt32)right;
                        }
                    case TypeCode.Int64: {
                            Int64 leftInt64 = (Int64)leftByte;
                            return leftInt64 ^ (Int64)right;
                        }
                    case TypeCode.UInt64: {
                            UInt64 leftUInt64 = (UInt64)leftByte;
                            return leftUInt64 ^ (UInt64)right;
                        }
                }
            }
            if (right is BigInteger) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger ^ (BigInteger)right;
            } else if (right is ExtensibleInt) {
                Int32 leftInt32 = (Int32)leftByte;
                return leftInt32 ^ (Int32)((ExtensibleInt)right).value;
            } else if (right is ExtensibleLong) {
                BigInteger leftBigInteger = (BigInteger)leftByte;
                return leftBigInteger ^ (BigInteger)((ExtensibleLong)right).Value;
            }
            return Ops.NotImplemented;
        }
        [PythonName("__divmod__")]
        public static object DivMod(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__divmod__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean:
                        return ByteOps.DivModImpl(leftByte, ((Boolean)right ? (Byte)1 : (Byte)0));
                    case TypeCode.Byte:
                        return ByteOps.DivModImpl(leftByte, (Byte)right);
                    case TypeCode.SByte:
                        return Int16Ops.DivModImpl(leftByte, (SByte)right);
                    case TypeCode.Int16:
                        return Int16Ops.DivModImpl(leftByte, (Int16)right);
                    case TypeCode.UInt16:
                        return UInt16Ops.DivModImpl(leftByte, (UInt16)right);
                    case TypeCode.Int32:
                        return IntOps.DivMod(leftByte, (Int32)right);
                    case TypeCode.UInt32:
                        return UInt32Ops.DivModImpl(leftByte, (UInt32)right);
                    case TypeCode.Int64:
                        return Int64Ops.DivMod(leftByte, (Int64)right);
                    case TypeCode.UInt64:
                        return UInt64Ops.DivModImpl(leftByte, (UInt64)right);
                    case TypeCode.Single:
                        return SingleOps.DivModImpl(leftByte, (Single)right);
                    case TypeCode.Double:
                        return FloatOps.DivMod(leftByte, (Double)right);
                }
            }
            if (right is BigInteger) {
                return LongOps.DivMod(leftByte, (BigInteger)right);
            } else if (right is ExtensibleInt) {
                return IntOps.DivMod(leftByte, ((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.DivMod(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is Complex64) {
                return ComplexOps.DivMod(leftByte, (Complex64)right);
            } else if (right is ExtensibleFloat) {
                return FloatOps.DivMod(leftByte, ((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.DivMod(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rdivmod__")]
        public static object ReverseDivMod(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rdivmod__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean:
                        return ByteOps.ReverseDivModImpl(leftByte, ((Boolean)right ? (Byte)1 : (Byte)0));
                    case TypeCode.Byte:
                        return ByteOps.ReverseDivModImpl(leftByte, (Byte)right);
                    case TypeCode.SByte:
                        return Int16Ops.ReverseDivModImpl(leftByte, (SByte)right);
                    case TypeCode.Int16:
                        return Int16Ops.ReverseDivModImpl(leftByte, (Int16)right);
                    case TypeCode.UInt16:
                        return UInt16Ops.ReverseDivModImpl(leftByte, (UInt16)right);
                    case TypeCode.Int32:
                        return IntOps.ReverseDivMod(leftByte, (Int32)right);
                    case TypeCode.UInt32:
                        return UInt32Ops.ReverseDivModImpl(leftByte, (UInt32)right);
                    case TypeCode.Int64:
                        return Int64Ops.ReverseDivMod(leftByte, (Int64)right);
                    case TypeCode.UInt64:
                        return UInt64Ops.ReverseDivModImpl(leftByte, (UInt64)right);
                    case TypeCode.Single:
                        return SingleOps.ReverseDivModImpl(leftByte, (Single)right);
                    case TypeCode.Double:
                        return FloatOps.ReverseDivMod(leftByte, (Double)right);
                }
            }
            if (right is BigInteger) {
                return LongOps.ReverseDivMod(leftByte, (BigInteger)right);
            } else if (right is ExtensibleInt) {
                return IntOps.ReverseDivMod(leftByte, ((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.ReverseDivMod(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is Complex64) {
                return ComplexOps.ReverseDivMod(leftByte, (Complex64)right);
            } else if (right is ExtensibleFloat) {
                return FloatOps.ReverseDivMod(leftByte, ((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.ReverseDivMod(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__lshift__")]
        public static object LeftShift(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__lshift__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean:
                        return ByteOps.LeftShiftImpl(leftByte, ((Boolean)right ? (Byte)1 : (Byte)0));
                    case TypeCode.Byte:
                        return ByteOps.LeftShiftImpl(leftByte, (Byte)right);
                    case TypeCode.SByte:
                        return Int16Ops.LeftShiftImpl(leftByte, (SByte)right);
                    case TypeCode.Int16:
                        return Int16Ops.LeftShiftImpl(leftByte, (Int16)right);
                    case TypeCode.UInt16:
                        return UInt16Ops.LeftShiftImpl(leftByte, (UInt16)right);
                    case TypeCode.Int32:
                        return IntOps.LeftShift(leftByte, (Int32)right);
                    case TypeCode.UInt32:
                        return UInt32Ops.LeftShiftImpl(leftByte, (UInt32)right);
                    case TypeCode.Int64:
                        return Int64Ops.LeftShift(leftByte, (Int64)right);
                    case TypeCode.UInt64:
                        return UInt64Ops.LeftShiftImpl(leftByte, (UInt64)right);
                }
            }
            if (right is BigInteger) {
                return LongOps.LeftShift(leftByte, (BigInteger)right);
            } else if (right is ExtensibleInt) {
                return IntOps.LeftShift(leftByte, ((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.LeftShift(leftByte, ((ExtensibleLong)right).Value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rlshift__")]
        public static object ReverseLeftShift(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rlshift__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean:
                        return ByteOps.ReverseLeftShiftImpl(leftByte, ((Boolean)right ? (Byte)1 : (Byte)0));
                    case TypeCode.Byte:
                        return ByteOps.ReverseLeftShiftImpl(leftByte, (Byte)right);
                    case TypeCode.SByte:
                        return Int16Ops.ReverseLeftShiftImpl(leftByte, (SByte)right);
                    case TypeCode.Int16:
                        return Int16Ops.ReverseLeftShiftImpl(leftByte, (Int16)right);
                    case TypeCode.UInt16:
                        return UInt16Ops.ReverseLeftShiftImpl(leftByte, (UInt16)right);
                    case TypeCode.Int32:
                        return IntOps.ReverseLeftShift(leftByte, (Int32)right);
                    case TypeCode.UInt32:
                        return UInt32Ops.ReverseLeftShiftImpl(leftByte, (UInt32)right);
                    case TypeCode.Int64:
                        return Int64Ops.ReverseLeftShift(leftByte, (Int64)right);
                    case TypeCode.UInt64:
                        return UInt64Ops.ReverseLeftShiftImpl(leftByte, (UInt64)right);
                }
            }
            if (right is BigInteger) {
                return LongOps.ReverseLeftShift(leftByte, (BigInteger)right);
            } else if (right is ExtensibleInt) {
                return IntOps.ReverseLeftShift(leftByte, ((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.ReverseLeftShift(leftByte, ((ExtensibleLong)right).Value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__pow__")]
        public static object Power(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__pow__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean:
                        return ByteOps.PowerImpl(leftByte, ((Boolean)right ? (Byte)1 : (Byte)0));
                    case TypeCode.Byte:
                        return ByteOps.PowerImpl(leftByte, (Byte)right);
                    case TypeCode.SByte:
                        return Int16Ops.PowerImpl(leftByte, (SByte)right);
                    case TypeCode.Int16:
                        return Int16Ops.PowerImpl(leftByte, (Int16)right);
                    case TypeCode.UInt16:
                        return UInt16Ops.PowerImpl(leftByte, (UInt16)right);
                    case TypeCode.Int32:
                        return IntOps.Power(leftByte, (Int32)right);
                    case TypeCode.UInt32:
                        return UInt32Ops.PowerImpl(leftByte, (UInt32)right);
                    case TypeCode.Int64:
                        return Int64Ops.Power(leftByte, (Int64)right);
                    case TypeCode.UInt64:
                        return UInt64Ops.PowerImpl(leftByte, (UInt64)right);
                    case TypeCode.Single:
                        return SingleOps.PowerImpl(leftByte, (Single)right);
                    case TypeCode.Double:
                        return FloatOps.Power(leftByte, (Double)right);
                }
            }
            if (right is BigInteger) {
                return LongOps.Power(leftByte, (BigInteger)right);
            } else if (right is ExtensibleInt) {
                return IntOps.Power(leftByte, ((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.Power(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is Complex64) {
                return ComplexOps.Power(leftByte, (Complex64)right);
            } else if (right is ExtensibleFloat) {
                return FloatOps.Power(leftByte, ((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.Power(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rpow__")]
        public static object ReversePower(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rpow__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean:
                        return ByteOps.ReversePowerImpl(leftByte, ((Boolean)right ? (Byte)1 : (Byte)0));
                    case TypeCode.Byte:
                        return ByteOps.ReversePowerImpl(leftByte, (Byte)right);
                    case TypeCode.SByte:
                        return Int16Ops.ReversePowerImpl(leftByte, (SByte)right);
                    case TypeCode.Int16:
                        return Int16Ops.ReversePowerImpl(leftByte, (Int16)right);
                    case TypeCode.UInt16:
                        return UInt16Ops.ReversePowerImpl(leftByte, (UInt16)right);
                    case TypeCode.Int32:
                        return IntOps.ReversePower(leftByte, (Int32)right);
                    case TypeCode.UInt32:
                        return UInt32Ops.ReversePowerImpl(leftByte, (UInt32)right);
                    case TypeCode.Int64:
                        return Int64Ops.ReversePower(leftByte, (Int64)right);
                    case TypeCode.UInt64:
                        return UInt64Ops.ReversePowerImpl(leftByte, (UInt64)right);
                    case TypeCode.Single:
                        return SingleOps.ReversePowerImpl(leftByte, (Single)right);
                    case TypeCode.Double:
                        return FloatOps.ReversePower(leftByte, (Double)right);
                }
            }
            if (right is BigInteger) {
                return LongOps.ReversePower(leftByte, (BigInteger)right);
            } else if (right is ExtensibleInt) {
                return IntOps.ReversePower(leftByte, ((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.ReversePower(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is Complex64) {
                return ComplexOps.ReversePower(leftByte, (Complex64)right);
            } else if (right is ExtensibleFloat) {
                return FloatOps.ReversePower(leftByte, ((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return ComplexOps.ReversePower(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rshift__")]
        public static object RightShift(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rshift__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean:
                        return ByteOps.RightShiftImpl(leftByte, ((Boolean)right ? (Byte)1 : (Byte)0));
                    case TypeCode.Byte:
                        return ByteOps.RightShiftImpl(leftByte, (Byte)right);
                    case TypeCode.SByte:
                        return Int16Ops.RightShiftImpl(leftByte, (SByte)right);
                    case TypeCode.Int16:
                        return Int16Ops.RightShiftImpl(leftByte, (Int16)right);
                    case TypeCode.UInt16:
                        return UInt16Ops.RightShiftImpl(leftByte, (UInt16)right);
                    case TypeCode.Int32:
                        return IntOps.RightShift(leftByte, (Int32)right);
                    case TypeCode.UInt32:
                        return UInt32Ops.RightShiftImpl(leftByte, (UInt32)right);
                    case TypeCode.Int64:
                        return Int64Ops.RightShift(leftByte, (Int64)right);
                    case TypeCode.UInt64:
                        return UInt64Ops.RightShiftImpl(leftByte, (UInt64)right);
                }
            }
            if (right is BigInteger) {
                return LongOps.RightShift(leftByte, (BigInteger)right);
            } else if (right is ExtensibleInt) {
                return IntOps.RightShift(leftByte, ((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.RightShift(leftByte, ((ExtensibleLong)right).Value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rrshift__")]
        public static object ReverseRightShift(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rrshift__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean:
                        return ByteOps.ReverseRightShiftImpl(leftByte, ((Boolean)right ? (Byte)1 : (Byte)0));
                    case TypeCode.Byte:
                        return ByteOps.ReverseRightShiftImpl(leftByte, (Byte)right);
                    case TypeCode.SByte:
                        return Int16Ops.ReverseRightShiftImpl(leftByte, (SByte)right);
                    case TypeCode.Int16:
                        return Int16Ops.ReverseRightShiftImpl(leftByte, (Int16)right);
                    case TypeCode.UInt16:
                        return UInt16Ops.ReverseRightShiftImpl(leftByte, (UInt16)right);
                    case TypeCode.Int32:
                        return IntOps.ReverseRightShift(leftByte, (Int32)right);
                    case TypeCode.UInt32:
                        return UInt32Ops.ReverseRightShiftImpl(leftByte, (UInt32)right);
                    case TypeCode.Int64:
                        return Int64Ops.ReverseRightShift(leftByte, (Int64)right);
                    case TypeCode.UInt64:
                        return UInt64Ops.ReverseRightShiftImpl(leftByte, (UInt64)right);
                }
            }
            if (right is BigInteger) {
                return LongOps.ReverseRightShift(leftByte, (BigInteger)right);
            } else if (right is ExtensibleInt) {
                return IntOps.ReverseRightShift(leftByte, ((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return LongOps.ReverseRightShift(leftByte, ((ExtensibleLong)right).Value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__truediv__")]
        public static object TrueDivide(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__truediv__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean:
                        return FloatOps.TrueDivide(leftByte, ((Boolean)right ? (Byte)1 : (Byte)0));
                    case TypeCode.Byte:
                        return FloatOps.TrueDivide(leftByte, (Byte)right);
                    case TypeCode.SByte:
                        return FloatOps.TrueDivide(leftByte, (SByte)right);
                    case TypeCode.Int16:
                        return FloatOps.TrueDivide(leftByte, (Int16)right);
                    case TypeCode.UInt16:
                        return FloatOps.TrueDivide(leftByte, (UInt16)right);
                    case TypeCode.Int32:
                        return FloatOps.TrueDivide(leftByte, (Int32)right);
                    case TypeCode.UInt32:
                        return FloatOps.TrueDivide(leftByte, (UInt32)right);
                    case TypeCode.Int64:
                        return FloatOps.TrueDivide(leftByte, (Int64)right);
                    case TypeCode.UInt64:
                        return FloatOps.TrueDivide(leftByte, (UInt64)right);
                    case TypeCode.Single:
                        return FloatOps.TrueDivide(leftByte, (Single)right);
                    case TypeCode.Double:
                        return FloatOps.TrueDivide(leftByte, (Double)right);
                }
            }
            if (right is BigInteger) {
                return FloatOps.TrueDivide(leftByte, (BigInteger)right);
            } else if (right is ExtensibleInt) {
                return FloatOps.TrueDivide(leftByte, ((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return FloatOps.TrueDivide(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is Complex64) {
                return FloatOps.TrueDivide(leftByte, (Complex64)right);
            } else if (right is ExtensibleFloat) {
                return FloatOps.TrueDivide(leftByte, ((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return FloatOps.TrueDivide(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        [PythonName("__rtruediv__")]
        public static object ReverseTrueDivide(object left, object right) {
            if (!(left is Byte)) {
                throw Ops.TypeError("'__rtruediv__' requires Byte, but received {0}", Ops.GetDynamicType(left).__name__);
            }
            Byte leftByte = (Byte)left;
            IConvertible rightConvertible;
            if ((rightConvertible = right as IConvertible) != null) {
                switch (rightConvertible.GetTypeCode()) {
                    case TypeCode.Boolean:
                        return FloatOps.ReverseTrueDivide(leftByte, ((Boolean)right ? (Byte)1 : (Byte)0));
                    case TypeCode.Byte:
                        return FloatOps.ReverseTrueDivide(leftByte, (Byte)right);
                    case TypeCode.SByte:
                        return FloatOps.ReverseTrueDivide(leftByte, (SByte)right);
                    case TypeCode.Int16:
                        return FloatOps.ReverseTrueDivide(leftByte, (Int16)right);
                    case TypeCode.UInt16:
                        return FloatOps.ReverseTrueDivide(leftByte, (UInt16)right);
                    case TypeCode.Int32:
                        return FloatOps.ReverseTrueDivide(leftByte, (Int32)right);
                    case TypeCode.UInt32:
                        return FloatOps.ReverseTrueDivide(leftByte, (UInt32)right);
                    case TypeCode.Int64:
                        return FloatOps.ReverseTrueDivide(leftByte, (Int64)right);
                    case TypeCode.UInt64:
                        return FloatOps.ReverseTrueDivide(leftByte, (UInt64)right);
                    case TypeCode.Single:
                        return FloatOps.ReverseTrueDivide(leftByte, (Single)right);
                    case TypeCode.Double:
                        return FloatOps.ReverseTrueDivide(leftByte, (Double)right);
                }
            }
            if (right is BigInteger) {
                return FloatOps.ReverseTrueDivide(leftByte, (BigInteger)right);
            } else if (right is ExtensibleInt) {
                return FloatOps.ReverseTrueDivide(leftByte, ((ExtensibleInt)right).value);
            } else if (right is ExtensibleLong) {
                return FloatOps.ReverseTrueDivide(leftByte, ((ExtensibleLong)right).Value);
            } else if (right is Complex64) {
                return FloatOps.ReverseTrueDivide(leftByte, (Complex64)right);
            } else if (right is ExtensibleFloat) {
                return FloatOps.ReverseTrueDivide(leftByte, ((ExtensibleFloat)right).value);
            } else if (right is ExtensibleComplex) {
                return FloatOps.ReverseTrueDivide(leftByte, ((ExtensibleComplex)right).value);
            }
            return Ops.NotImplemented;
        }
        internal static object DivideImpl(Byte x, Byte y) {
            return (Byte)(x / y);
        }
        internal static object ModImpl(Byte x, Byte y) {
            return (Byte)(x % y);
        }


        internal static object DivModImpl(Byte x, Byte y) {
            object div = DivideImpl(x, y);
            if (div == Ops.NotImplemented) return div;
            object mod = ModImpl(x, y);
            if (mod == Ops.NotImplemented) return mod;
            return Tuple.MakeTuple(div, mod);
        }
        internal static object ReverseDivideImpl(Byte x, Byte y) {
            return DivideImpl(y, x);
        }
        internal static object ReverseModImpl(Byte x, Byte y) {
            return ModImpl(y, x);
        }
        internal static object ReverseDivModImpl(Byte x, Byte y) {
            return DivModImpl(y, x);
        }
        internal static object FloorDivideImpl(Byte x, Byte y) {
            return DivideImpl(x, y);
        }
        internal static object ReverseFloorDivideImpl(Byte x, Byte y) {
            return DivideImpl(y, x);
        }
        internal static object ReverseLeftShiftImpl(Byte x, Byte y) {
            return LeftShiftImpl(y, x);
        }
        internal static object ReversePowerImpl(Byte x, Byte y) {
            return PowerImpl(y, x);
        }
        internal static object ReverseRightShiftImpl(Byte x, Byte y) {
            return RightShiftImpl(y, x);
        }


        // *** END GENERATED CODE ***

        #endregion

    }
}
