#include <jmp-config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
 #include <unistd.h>
#endif

#include <gtk/gtk.h>

#include <hash.h>
#include <jmp.h>
#include <jmp-debug.h>
#include <jmpthread.h>
#include <comparators.h>
#include <heap_dump.h>

#ifdef CONFIG_UI_FILE
 #include <ui_file.h>
#endif

#include <ui_gtk.h>
#include <ui_gtk_prefs.h>
#include <ui_gtk_gtkutils.h>
#include <ui_gtk_filterconfig.h>
#include <ui_gtk_eventconfig.h>
#include <ui_gtk_threads_window.h>
#include <ui_gtk_heapgraph.h>
#include <ui_gtk_about_dialog.h>
#include <ui_gtk_monitor_dialog.h>
#include <ui_gtk_visible_row_config.h>
#include <ui_gtk_method_window.h>
#include <ui_gtk_class_window.h>

/** state tokens:
 * -2 embrionic (no UI windows)
 * -1 startup UI running (not yet operational)
 * 0  normal (UI operational)
 * 1  profile window done (UI operational)
 * 2  shutdown UI requested
 * 3  shutdown UI running
 * 4  shutdown UI completed
 */
static int        quit = -2;
static int        is_updating = 1;
static int        freeze_ui_disable = 0;
static GMutex *   mutex_quit = NULL;
static GCond *    cond_quit = NULL;
int exit_on_jvm_shutdown = 1;
static JNIEnv*    gtkthread_env_id = NULL;

/** Are we dumping on a timer basis? */
static guint  current_timer = 0; /* the timer identifier. */

static GtkWidget* statusbar;
static GtkTooltips* tooltips;
static GtkWidget* JMPmain;
static GtkItemFactory *menubar_item_factory;
static GtkWidget* buttonbar_freeze_ui;

#define ACTION_FREEZE_UI 1


#ifdef DLMDEBUG
#define DEBUG(str, args...) do { fprintf (stderr, "%s:%d:%s():" str, __FILE__, __LINE__, __FUNCTION__ , ## args); fflush(stderr); } while(0)
#endif

static int ui_state (int below, int set) {
    int rv = 0;
    g_mutex_lock (mutex_quit);
    if (quit <= below) {
        if (quit == below) {
            quit = set;
            g_cond_broadcast (cond_quit);
            rv = 1;
        }
    }
    g_mutex_unlock (mutex_quit);
    return rv;
}

static int checkAllowShutdownUI (void) {
    int rv = 0;
    g_mutex_lock (mutex_quit);
    if (quit >= 1)
        rv = 1;
    g_mutex_unlock (mutex_quit);
    return rv;
}

static int checkShutdownUI (void) {
    int rv = 0;
    g_mutex_lock (mutex_quit);
    if (quit >= 2)
        rv = 1;
    g_mutex_unlock (mutex_quit);
    return rv;
}

static int doProfileWindowDone (void) {
    int rv = 0;
    g_mutex_lock (mutex_quit);
    if (quit >= 0) {
        if (quit == 0) {
            quit = 1;
            g_cond_broadcast (cond_quit);
	    rv = 1;
	}
    }
    g_mutex_unlock (mutex_quit);
    return rv;
}

static int checkThenDoShutdownUI (void) {
    int rv = 0;
    g_mutex_lock (mutex_quit);
    if (quit >= 2) {
        if (quit == 2) {
            if (usingUI ()) {
                gtk_main_quit ();   /* time to shut down. */
            }
            quit = 3;
            g_cond_broadcast (cond_quit);
        }
        rv = 1;
    }
    g_mutex_unlock (mutex_quit);
    return rv;
}


/** Update the statistics. */
static void updateUI_internal (hashtab* classes, hashtab* methods) {
    if (tracing_objects ()) {
	update_class_tree (classes);
    }

    if (tracing_methods ()) {
	update_method_tree (methods);
    }
	
    update_threads_window (get_threads ());
}

void updateUI (hashtab* classes, hashtab* methods) {
    if (!is_updating)
	return;

    updateUI_internal (classes, methods);
}


/* This is a general purpose event system that allows the AGENT part
 *  pass events through to the UI and have the UI wake up right away
 *  and process them.
 * Perfect for shutdown notification, status bar and forcing a refresh.
 *
 */
static GSource *thread_events_source;
static GAsyncQueue *thread_events_queue;

#define UI_EVENT_JVM_SHUTDOWN		1
#define UI_EVENT_UI_SHUTDOWN		2
#define UI_EVENT_SET_STATUS		3
#define UI_EVENT_UPDATE_FULL_FORCE	4
#define UI_EVENT_UPDATE_FULL_TRY	5
#define UI_EVENT_PROFILE_WINDOW_BEGIN	6
#define UI_EVENT_PROFILE_WINDOW_END	7
#define UI_EVENT_PROFILE_WINDOW_CLOSE	8

struct ui_event {
    int type;
    unsigned int datalen;
    unsigned char data[];
};

static void
thread_events_send (struct ui_event *ui_event, const void *data, unsigned int len) {
    void *event_data;
    event_data = g_malloc (sizeof (struct ui_event) + len);
    memcpy (event_data, ui_event, sizeof (struct ui_event));
    if (data != NULL && len > 0)
	memcpy (event_data + sizeof (struct ui_event), data, len);
    g_async_queue_push (thread_events_queue, event_data);
    g_main_context_wakeup (NULL);
}

static void
thread_events_send_generic (int type, const char *data, unsigned int len) {
    char buf[sizeof (struct ui_event)];
    struct ui_event *ui_event;

    memset (buf, 0, sizeof (buf));
    ui_event = (struct ui_event *)buf;
    ui_event->type = type;
    ui_event->datalen = len;
    thread_events_send (ui_event, data, len);
}

static void
thread_events_send_string (int type, const char *data) {
    thread_events_send_generic (type, data, strlen (data) + 1);
}

static gboolean
thread_events_dispatch (GSource *source, GSourceFunc callback, gpointer user_data) {
    gpointer event_data = g_async_queue_pop (thread_events_queue);
    gboolean result = callback (event_data);
    g_free (event_data);
    return result;
}

static gboolean
thread_events_prepare (GSource *source, gint *timeout) {
    *timeout = -1;
    return (g_async_queue_length (thread_events_queue) > 0);
}

static gboolean
thread_events_check (GSource *source) {
    return (g_async_queue_length (thread_events_queue) > 0);
}

static GSourceFuncs thread_events_functions = {
    thread_events_prepare,
    thread_events_check,
    thread_events_dispatch,
    NULL,
};

static gboolean
thread_events_callback (gpointer event_data)
{
    struct ui_event *ui_event = (struct ui_event *)event_data;
    
    /*fprintf(stderr, "RUN event event_data=%p type=%d len=%d\n", 
	    event_data, ui_event->type, ui_event->datalen);*/
    trace5 ("EVENT %p type=%d datalen=%d data=%p\n", 
	    data, ui_event->type, ui_event->datalen, ui_event->data);
    switch (ui_event->type) {
    case UI_EVENT_JVM_SHUTDOWN:	  /* ask user */
	if (checkShutdownUI ())
	    break;
	set_status_internal (statusbar, _("JVM has shutdown"));
	/* TODO: present ask dialog for shutdown, auto-timer, preference options */
	break;

    case UI_EVENT_UI_SHUTDOWN:	  /* wakes us up */
	if (ui_state (0, 2) || ui_state (1, 2))
	    checkThenDoShutdownUI ();
	break;

    case UI_EVENT_SET_STATUS:
	if (checkShutdownUI ())
	    break;
	set_status_internal (statusbar, (char*)ui_event->data);
	break;

    case UI_EVENT_UPDATE_FULL_FORCE:
	if (checkShutdownUI ())
	    break;
	updateUI_internal (get_classes (), get_methods ());
	break;

    case UI_EVENT_UPDATE_FULL_TRY:
	if (checkShutdownUI ())
	    break;
	updateUI (get_classes (), get_methods ());
	break;

    case UI_EVENT_PROFILE_WINDOW_BEGIN:
	if (checkShutdownUI ())
	    break;
	if (is_updating)
	    set_status_internal (statusbar, _("Profile window started"));
	break;

    case UI_EVENT_PROFILE_WINDOW_END:
	if (checkShutdownUI ())
	    break;
	if (is_updating)
	    set_status_internal (statusbar, _("Profile window ended"));
	/* TODO: present ask dialog for shutdown, auto-timer, preference options */
	/* TODO: flush out a filedump if requested */
	updateUI (get_classes (), get_methods ());
	break;

    case UI_EVENT_PROFILE_WINDOW_CLOSE:
	/* TODO: flush out a filedump if requested */
	if (doProfileWindowDone ()) {
	    if (checkShutdownUI ())
		break;
	    /* TODO: usability improvement, if UI is in frozen state dont mess that up,
	     *  notify the user the session is finished when they unfreeze so they can
	     *  understand why things are not updating
	     */
	    if (is_updating)
		set_status_internal (statusbar, _("Profiling session finished"));

	    updateUI (get_classes (), get_methods ());

	    if (is_updating) {
		GtkWidget *w;
		freeze_ui_disable = 2;
		w = gtk_item_factory_get_widget_by_action (menubar_item_factory, 
							   ACTION_FREEZE_UI);
		if (w != NULL)
		    gtk_widget_set_sensitive (w, FALSE);
		if (buttonbar_freeze_ui != NULL)
		    gtk_widget_set_sensitive (buttonbar_freeze_ui, FALSE);
	    } else {
		freeze_ui_disable = 1;
	    }
	    
	    is_updating = 0;	/* closest thing to what I'm tying to achieve, but not perfect */
	}
	if (exit_on_jvm_shutdown) {
	    if (ui_state (1, 2))
		checkThenDoShutdownUI ();
	}
	break;
    }
    return TRUE;
}


void end_ui (void) {
    if (thread_events_queue) {
        /* GLib seems silly in its API here, the return type of unref should
         *  return boolean and track if the unref() caused a destroy().  There
         *  is no destroy() function and there is no way to get the current
         *  refcount.
         */
        g_async_queue_unref (thread_events_queue);
        thread_events_queue = NULL;
    }
    if (mutex_quit) {
        g_mutex_free (mutex_quit);
        mutex_quit = NULL;
    }
    if (cond_quit) {
        g_cond_free (cond_quit);
        cond_quit = NULL;
    }
}

/** Tell the ui system to shut itself down. */
static void quit_ui_internal (void) {
    GTimeVal waittime;

    if (ui_state (0, 2) || ui_state (1, 2))
        checkThenDoShutdownUI ();

    g_get_current_time (&waittime);
    g_time_val_add (&waittime, 5000000);
    g_mutex_lock (mutex_quit);
    for(;;) {
        if (quit >= 4)
            break;	/* UI has shutdown */
        if (g_cond_timed_wait (cond_quit, mutex_quit, &waittime) == FALSE) {
            fprintf (stderr, "ERROR: UI thread shutdown timeout after 5 seconds\n");
            break;
        }
    }
    g_mutex_unlock (mutex_quit);
}

int quit_ui (void) {
    if (exit_on_jvm_shutdown) {
        quit_ui_internal ();
        return 0;
    }
    return 1;
}


/** This is the timeout function called from gtk-main.
 *  Check if it is time to quit.
 */
gint run_updates (gpointer data) {
    if (checkThenDoShutdownUI ())
        return FALSE;

    if (usingUI ()) {
	gdk_threads_enter ();	
	updateUI (get_classes (), get_methods ());
	gdk_threads_leave ();
    }
    
#ifdef CONFIG_UI_FILE
    dumptimer_tick ();
#endif
    
    return TRUE;                /* keep timer */
}

void run_dump (void) {
    run_data_dump ();
}

void run_garbage_collector (void) {
    set_status (_("Requesting garbage collection ..."));
    run_GC ();
}

void run_heap_dumper (void) {
    set_status (_("Getting heap dump"));
    run_heap_dump ();
    set_status (_("Heap dumped"));
}

void run_monitor_dumper (void) {
    hashtab* monitors;
    set_status (_("Getting monitor dump"));
    monitors = run_monitor_dump ();
    show_monitor_dialog (monitors);
    set_status (_("Monitors dumped"));
    cleanup_monitor_information ();
}

void run_reset_counter (void) {
    set_status (_("Resetting counters..."));
    reset_counters ();
    set_status (_("Counters reseted"));
}

void run_restore_counter (void) {
    set_status (_("Resetoring counters..."));
    restore_counters ();
    set_status (_("Counters restored"));
}

static void freeze_label (GtkWidget *widget, int state) {
    GtkBin *bin = NULL;
    if (GTK_IS_MENU_ITEM(widget)) {
        GtkMenuItem *menuitem = GTK_MENU_ITEM(widget);
        GtkItem *item = GTK_ITEM(&menuitem->item);
        bin = GTK_BIN(&item->bin);
    } else if(GTK_IS_BUTTON(widget)) {
        GtkButton *button = GTK_BUTTON(widget);
        bin = GTK_BIN(button);
    }
    if(bin != NULL)
        gtk_label_parse_uline (GTK_LABEL(gtk_bin_get_child (bin)), state ? _("_Freeze UI") : _("Un_freeze UI"));
}

static void freeze_ui (gpointer callback_data, guint callback_action, GtkWidget *widget) {
    if (freeze_ui_disable != 0 && is_updating == 0) {
        if (freeze_ui_disable == 1) {
            freeze_label (widget, 1);	/* back to default state */
            freeze_label (buttonbar_freeze_ui, is_updating);
	    updateUI_internal (get_classes (), get_methods ());
	    freeze_ui_disable = 2;
        } else {
            gdk_beep ();
        }
    } else {
    is_updating = !is_updating;
    set_status (is_updating ? 
		_("UI updating continued") : 
		_("UI updating frozen"));
        freeze_label (widget, is_updating);
	freeze_label (buttonbar_freeze_ui, is_updating);
	updateUI (get_classes (), get_methods ());
    }
    if (freeze_ui_disable != 0) {
        /* Handle the case there we unfreeze */
	gtk_widget_set_sensitive (widget, FALSE);
	if (buttonbar_freeze_ui != NULL)
	    gtk_widget_set_sensitive (buttonbar_freeze_ui, FALSE);
    }
}

static void about_jmp (void) {
    GtkWidget *about = create_about_dialog ();	
    gtk_widget_show_all (about);
}

/** Add a button to the button box.
 * @param label the text on the button
 * @param tooltip the tool tip for the button giving a more detailed description
 *  of the actions of the button.
 * @param hbuttonbox the button box to add the button to.
 * @param handler the callback for the button.
 */
static GtkWidget* add_button (char* label, char* tooltip, 
			GtkWidget* hbuttonbox, GCallback* handler) {
    GtkWidget* button;
    guint button_key;
    button = gtk_button_new_with_label ("");
    button_key = 
	gtk_label_parse_uline (GTK_LABEL (GTK_BIN (button)->child), label);
    gtk_tooltips_set_tip (tooltips, button, tooltip, NULL);
    gtk_container_add (GTK_CONTAINER (hbuttonbox), button);
    g_signal_connect (G_OBJECT (button), "clicked",
		      G_CALLBACK (handler), button);
    return button;
}

static gboolean main_delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data) {
    if (checkAllowShutdownUI ()) {
        if (ui_state (0, 2) || ui_state (1, 2))
            checkThenDoShutdownUI ();
    } else {
        /* Let the user know we got the message, but you can't to that! */
        gdk_beep ();
    }
    return TRUE;
}

/** Build the main window.
 *  Names of widgets is not totally nice though...
 */
GtkWidget* create_JMP (void) {
    GtkWidget *JMP;
    GtkWidget *vbox;
    GtkWidget *menubar;
    GtkWidget *hbuttonbox;
    GtkWidget *heapgraph;
    GtkItemFactoryEntry menu_items[] = {
	{ _("/_File"),                  NULL, NULL,                  0, "<Branch>" },
	{ _("/File/_Dump"),             NULL, run_dump,              0, NULL },
	{ _("/File/_Reset counters"),   NULL, run_reset_counter,     0, NULL },
	{ _("/File/Re_store counters"), NULL, run_restore_counter,   0, NULL },
	{ _("/File/System GC"),         NULL, run_garbage_collector, 0, NULL },
	{ _("/File/_Heapdump"),         NULL, run_heap_dumper,       0, NULL },
	{ _("/File/_Monitors"),         NULL, run_monitor_dumper,    0, NULL },
	{ _("/File/_Freeze UI"),        NULL, freeze_ui,             ACTION_FREEZE_UI, NULL },
	{ _("/_Options"),               NULL, NULL,                  0, "<Branch>" },
	{ _("/Options/Filter"),         NULL, filter_edit_options,   0, NULL },
	{ _("/Options/Events"),         NULL, event_window,          0, NULL },
	{ _("/Options/Visible Rows"),   NULL, set_visible_rows,      0, NULL }, 
	{ _("/_View"),                  NULL, NULL,                  0, "<Branch>" },
	{ _("/View/Objects"),           NULL, toggle_class_window,   0, NULL },
	{ _("/View/Methods"),           NULL, toggle_method_window,  0, NULL },
	{ _("/View/Threads"),           NULL, toggle_threads_window, 0, NULL },
	{ _("/_Help"),                  NULL, NULL,                  0, "<LastBranch>" },
	{ _("/_Help/About"),            NULL, about_jmp,             0, NULL },
    };

    GtkItemFactory *item_factory;
    GtkAccelGroup *accel_group;
    gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);


    JMP = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_signal_connect (GTK_OBJECT (JMP), "delete-event",
			GTK_SIGNAL_FUNC (main_delete_event), NULL);

    gtk_window_set_title (GTK_WINDOW (JMP), _("Java Memory Profiler - Main"));
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (JMP), vbox);
    
    /* build menues. */
    accel_group = gtk_accel_group_new ();
    item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", 
					 accel_group);
    gtk_item_factory_create_items (item_factory, nmenu_items, menu_items, NULL);
    gtk_window_add_accel_group (GTK_WINDOW (JMP), accel_group);
    menubar = gtk_item_factory_get_widget (item_factory, "<main>");
    gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);
    
    /* build the button bar. */
    hbuttonbox = gtk_hbutton_box_new ();
    gtk_hbutton_box_set_spacing_default (0);
    gtk_box_pack_start (GTK_BOX (vbox), hbuttonbox, TRUE, TRUE, 0);

    add_button (_("_Dump"), _("Dump data to a text file"), 
		hbuttonbox, (GCallback*)run_dump);
    add_button (_("_Reset"), _("Reset counters to zero"), 
		hbuttonbox, (GCallback*)run_reset_counter);
    add_button (_("Re_store"), _("Restore counters to the standard (full) values"), 
		hbuttonbox, (GCallback*)run_restore_counter);
    add_button (_("System._GC"), _("Run the java virtual machines garbage collector"),
		hbuttonbox, (GCallback*)run_garbage_collector);
    add_button (_("_Heapdump"), _("Show the current heap"), 
		hbuttonbox, (GCallback*)run_heap_dumper);
    add_button (_("_Monitors"), _("Show the current monitors"), 
		hbuttonbox, (GCallback*)run_monitor_dumper);
    buttonbar_freeze_ui = add_button (_("_Freeze UI"),
		_("Stop updating values in the JMP windows"),
		hbuttonbox, (GCallback*)freeze_ui);
    add_button (_("Threads"), _("Show the current threads"),
		hbuttonbox, (GCallback*)show_refresh_threads_window);
 
    /* add a heap size graph */
    heapgraph = get_heap_graph ();
    gtk_box_pack_start (GTK_BOX (vbox), heapgraph, TRUE, TRUE, 0);   

    /* add status bar. */
    statusbar = gtk_statusbar_new ();
    gtk_box_pack_start (GTK_BOX (vbox), statusbar, FALSE, FALSE, 0);

    menubar_item_factory = item_factory;

    return JMP;
}


void set_ui_update_interval (int milliseconds) {
    if (current_timer)
        g_source_remove (current_timer);
    current_timer = g_timeout_add (milliseconds, run_updates, NULL);
}



void init_ui (void) {
    g_thread_init (NULL);
    mutex_quit = g_mutex_new ();
    cond_quit = g_cond_new ();
    quit = -2;

    thread_events_queue = g_async_queue_new ();

    if (usingUI ()) {
	int argc = 1;
	char** argv;
	argv = malloc (sizeof (*argv));
	argv[0] = "jmp";	

	gdk_threads_init ();
  	gtk_init (&argc, &argv);

	free (argv);
    }
}

void start_ui (void) {
    ui_state (-2, -1);
}

void stop_ui (void) {
    thread_events_send_generic (UI_EVENT_UI_SHUTDOWN, NULL, 0);
}

void notify_update_full_try_ui (void) {
    thread_events_send_generic (UI_EVENT_UPDATE_FULL_TRY, NULL, 0);
}

void notify_profile_window_begin_ui (void) {
    thread_events_send_generic (UI_EVENT_PROFILE_WINDOW_BEGIN, NULL, 0);
}

void notify_profile_window_end_ui (void) {
    thread_events_send_generic (UI_EVENT_PROFILE_WINDOW_END, NULL, 0);
}

void notify_profile_window_close_ui (void) {
    thread_events_send_generic (UI_EVENT_PROFILE_WINDOW_CLOSE, NULL, 0);
}

void notify_jvm_shutdown_ui (void) {
    thread_events_send_generic (UI_EVENT_JVM_SHUTDOWN, NULL, 0);
}

static int ui_gtk_setup (void) {
    tooltips = gtk_tooltips_new ();
    JMPmain = create_JMP ();
    ui_gtk_prefs_load_window (UI_GTK_PREFS_MAIN, 0, GTK_WINDOW (JMPmain));	/* inittsate=0 */

    if (tracing_objects ()) {
	setup_class_tracing ();
    }
    if (tracing_methods ()) {
	setup_method_tracing ();
    }
    return 0;
}

static void ui_gtk_cleanup (void) {
    quit_threads_window ();
    quit_class_window ();
    quit_method_window ();
    filter_destroy ();
    set_visible_rows_destroy ();
    event_window_destroy ();
    if (JMPmain != NULL) {
	ui_gtk_prefs_save_window (UI_GTK_PREFS_MAIN, (GtkWindow *)JMPmain);
        gtk_widget_destroy (JMPmain);
        JMPmain = NULL;
    }
}


/** The thread used to run gtk-main. 
 */
void gtkthread (void* data) {
    /* Hmm we are already a Java Thread so we dont need to attach it,
     *  we just want to get hold of our (JNIEnv *).  JNI says that reattaching
     *  an already attached thread is a noop, that make it sound safe to call
     *  multiple times and given that it returns a JNIEnv* that means we can get
     *  ours.  However WE SHOULD NOT CALL DetachCurrentThread() since the JavaVM
     *  owns the lifecycle of this thread.  If founds that when we call it we get
     *  malloc related errors anyway and it took me a while to deduce the situation
     *  I describe here.
     */
    gtkthread_env_id = JavaVM_AttachCurrentThread ();

    g_mutex_lock (mutex_quit);
    for(;;) {
        if (quit >= -1)
            break;	/* UI can startup */
        g_cond_timed_wait (cond_quit, mutex_quit, NULL);
    }
    g_mutex_unlock (mutex_quit);

    /* avoid gtk if we are not using any graphical ui */
    if (usingUI ()) {
        int timeout;

	gdk_threads_enter ();	

	thread_events_source = g_source_new (&thread_events_functions, sizeof (GSource));
	g_source_set_callback (thread_events_source,
			       thread_events_callback,
			       NULL, NULL);
        g_async_queue_ref (thread_events_queue);
	g_source_attach (thread_events_source, NULL);
	ui_gtk_setup ();

	timeout = 1000;		/* default */
	ui_gtk_prefs_int (UI_GTK_PREFS_INT_REFRESH, &timeout);
	set_ui_update_interval (timeout);

	set_status (_("Ready"));
	ui_state (-1, 0);
	updateUI_internal (get_classes (), get_methods ());	/* first update */

	gtk_main ();

	ui_gtk_cleanup ();
        g_source_destroy (thread_events_source);
        g_async_queue_unref (thread_events_queue);
	gdk_threads_leave ();
    } else {
        int loop = 1;

        g_mutex_lock (mutex_quit);
        while (loop) {
            GTimeVal waittime;

            g_mutex_unlock (mutex_quit);
	    run_updates (NULL);

            g_get_current_time (&waittime);
            g_time_val_add (&waittime, 1000000);	/* 1 sec */
            g_mutex_lock (mutex_quit);
            for (;;) {
                if (quit >= 1) {
                    loop = 0;
                    quit = 3;
                    g_cond_broadcast (cond_quit);
                    break;
                }
                if (g_cond_timed_wait (cond_quit, mutex_quit, &waittime) == FALSE)
                    break;
            }
	}
        g_mutex_unlock (mutex_quit);
    }

    ui_state (3, 4);

    if (gtkthread_env_id != NULL) {
        jvm_shutdown_thread_stop (gtkthread_env_id);
        gtkthread_env_id = NULL;
    }
}

/** Set the status text... maybe calling from any thread context */
void set_status (const char* text) {
    if (checkShutdownUI ())
        return;
    thread_events_send_string (UI_EVENT_SET_STATUS, text);
}


/** Check if we have any ui-handling to do. */
int events_pending (void) {
    return gtk_events_pending ();
}

/** Update the ui-toolkit (gtk_main_iteration ()) */
int ui_iteration (void) {
    if (checkShutdownUI ())
        return TRUE;
    return gtk_main_iteration ();
}

int ui_gtk_state () {
    return (quit < 0);
}

/** Show deadlock */
void show_deadlock (visited_threads* vt) {
   GtkWidget *dialog, *label;
   char buf[300];
   dialog = gtk_dialog_new_with_buttons (_("Deadlock detected"),
                                         GTK_WINDOW (JMPmain),
                                         GTK_DIALOG_DESTROY_WITH_PARENT,
                                         _("OK"),
                                         GTK_RESPONSE_NONE,
                                         NULL);

   label = gtk_label_new (_("Warning deadlock detected!\n"));  
   gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label);
   while (vt != NULL) {
       if (vt->next) 
	   snprintf (buf, 300, _("%s holding %s (%p), is trying to enter %s (%p)"), 
		     jmpthread_get_thread_name (vt->mi->owner), 
		     vt->mi->name, vt->mi->id, 
		     vt->next->mi->name, vt->next->mi->id);
       else {
	   snprintf (buf, 300, _("%s holding %s (%p)"), 
		     jmpthread_get_thread_name (vt->mi->owner), 
		     vt->mi->name, vt->mi->id); 
       }
       vt = vt->next;
       label = gtk_label_new (buf);  
       gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label);
   }    
   
   g_signal_connect_swapped (GTK_OBJECT (dialog), 
                             "response", 
                             G_CALLBACK (gtk_widget_destroy),
                             GTK_OBJECT (dialog));

   ui_gtk_prefs_load_window (UI_GTK_PREFS_DIALOG_DEADLOCK, ui_gtk_state(), GTK_WINDOW (dialog));
}

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
