// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the k3d::frames class for working with ranges of numbered files
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include "frames.h"
#include "result.h"
#include "string_modifiers.h"

#include <boost/filesystem/operations.hpp>

#include <cmath>
#include <iomanip>

namespace k3d
{

/////////////////////////////////////////////////////////////////////////////
// frames

frames::frames(const boost::filesystem::path& File)
{
	// Sanity checks ...
	assert_warning(!File.empty());

	std::string::size_type index;
	std::string buffer(File.native_file_string());

	// Get the post-digits string ...
	index = buffer.find_last_of("0123456789");
	if(std::string::npos != index)
		{
			m_post_digits = k3d::right(buffer, buffer.size() - (index+1));
			buffer = buffer.substr(0, index+1);
		}

	// Get the post-digit
	index = buffer.find_last_not_of("0123456789");
	if(std::string::npos != index)
		{
			m_frame_digits = buffer.size() - (index+1);
			m_pre_digits = buffer.substr(0, index+1);
		}
	else
		{
			m_frame_digits = buffer.size();
		}

	// Find the first available frame ...
	long maxframe = max_frame();
	for(long i = 0; i < maxframe; i++)
		{
			boost::filesystem::path file;
			frame(i, file);

			if(boost::filesystem::exists(file))
				{
					m_first_frame = i;
					break;
				}
		}

	// Find the last available frame ...
	for(long j = m_first_frame + 1; j < maxframe; j++)
		{
			boost::filesystem::path file;
			frame(j, file);

			if(!boost::filesystem::exists(file))
				{
					m_last_frame = j - 1;
					break;
				}
		}
}

frames::frames(const boost::filesystem::path& File, const long FirstFrame, const long LastFrame) :
	m_first_frame(FirstFrame),
	m_last_frame(LastFrame)
{
	// Sanity checks ...
	assert_warning(!File.empty());

	std::string::size_type index;
	std::string buffer(File.native_file_string());

	// Get the post-digits string ...
	index = buffer.find_last_of("0123456789");
	if(std::string::npos != index)
		{
			m_post_digits = k3d::right(buffer, buffer.size() - (index+1));
			buffer = buffer.substr(0, index+1);
		}

	// Get the digits ...
	index = buffer.find_last_not_of("0123456789");
	if(std::string::npos != index)
		{
			m_frame_digits = buffer.size() - (index+1);
			m_pre_digits = buffer.substr(0, index+1);
		}
	else
		{
			m_frame_digits = buffer.size();
		}
}

frames::frames(const std::string PreDigits, const std::string PostDigits, const long FirstFrame, const long LastFrame, const long FrameDigits) :
	m_pre_digits(PreDigits),
	m_post_digits(PostDigits),
	m_first_frame(FirstFrame),
	m_last_frame(LastFrame),
	m_frame_digits(FrameDigits)
{
	// Sanity checks ...
	assert_warning(FirstFrame <= LastFrame);
	assert_warning(FrameDigits);
}

bool frames::frame(const long FrameNumber, boost::filesystem::path& Result) const
{
	// Sanity checks ...
	return_val_if_fail(FrameNumber <= max_frame(), false);

	std::stringstream buffer;
	buffer << m_pre_digits << std::setfill('0') << std::setw(m_frame_digits) << k3d::to_string(FrameNumber) << m_post_digits;
	Result = boost::filesystem::path(buffer.str(), boost::filesystem::native);

	return true;
}

long frames::max_frame() const
{
	return long(pow(10, m_frame_digits) - 1);
}

const std::string& frames::pre_digits() const
{
	return m_pre_digits;
}

const std::string& frames::post_digits() const
{
	return m_post_digits;
}

const unsigned long frames::frame_digits() const
{
	return m_frame_digits;
}

long frames::first_frame() const
{
	return m_first_frame;
}

long frames::last_frame() const
{
	return m_last_frame;
}

} // namespace k3d


