#ifndef K3DUI_BITMAP_PREVIEW_H
#define K3DUI_BITMAP_PREVIEW_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares the bitmap_preview control, which provides a standard MVC UI for bitmap data that supports tutorials, undo/redo, etc.
		\author Tim Shead (tshead@k-3d.com)
*/

#include "k3dcontrol.h"

#include <k3dsdk/bitmap.h>
#include <k3dsdk/iproperty.h>
#include <k3dsdk/iwritable_property.h>

#include <sdpgtk/sdpgtkloopevents.h>
#include <sdpgtk/sdpgtkopengldrawingarea.h>

#include <boost/cstdint.hpp>

// Forward declarations
class sdpGtkIObjectContainer;
class sdpxml::Element;

namespace k3d
{

// Forward declarations
class istate_recorder;

namespace bitmap_preview
{

/////////////////////////////////////////////////////////////////////////////
// idata_proxy

/// Abstract interface for an object that proxies a data source for a bitmap_preview control (i.e. the "model" in model-view-controller)
class idata_proxy
{
public:
	virtual ~idata_proxy() {}

	/// Called to return the underlying data value
	virtual bitmap* value() = 0;
	/// Signal emitted if the underlying data changes
	typedef SigC::Signal0<void> changed_signal_t;
	/// Signal emitted if the underlying data changes
	virtual changed_signal_t& changed_signal() = 0;

protected:
	idata_proxy() {}
	idata_proxy(const idata_proxy& RHS) {}
	idata_proxy& operator=(const idata_proxy& RHS) { return *this; }
};

/////////////////////////////////////////////////////////////////////////////
// control

/// Provides a UI for previewing bitmap images (i.e. the view and the controller from model-view-controller)
class control :
	public k3dControl
{
	typedef k3dControl base;

public:
	control(iunknown* const CommandNodeParent, const std::string CommandNodeName);
	~control();

	/// Called by the framework when instantiating the button from a GTKML (XML) document
	bool Create(sdpGtkIObjectContainer* const ObjectContainer, sdpxml::Document& Document, sdpxml::Element& Element);

	/// Attaches the button to the data it fronts for
	bool attach(std::auto_ptr<idata_proxy> Data, istate_recorder* const StateRecorder, const std::string StateChangeName);

	const std::string CustomType() const;

private:
	/// Called whenever the the button state needs to be updated
	void update();

	/// Called to handle SDPGTK events
	void OnEvent(sdpGtkEvent* Event);
	/// Called when the GTK+ widgets are about to disappear
	void on_destroy();
	/// Called to redraw the RGB preview
	void on_redraw_rgb();
	/// Called to redraw the alpha preview
	void on_redraw_alpha();

	/// Storeas a reference to the underlying data object
	std::auto_ptr<idata_proxy> m_data;

	typedef basic_rgba<boost::uint8_t> preview_pixel_t;
	typedef basic_bitmap<preview_pixel_t> preview_bitmap_t;
	preview_bitmap_t m_preview_image;
	sdpGtkOpenGLDrawingArea m_rgb;
	sdpGtkOpenGLDrawingArea m_alpha;
};

/// Provides an implementation of bitmap_preview::idata_proxy that supports any data source that supports the value(), set_value(), and changed_signal() concepts
template<typename data_t>
class data_proxy :
	public idata_proxy
{
public:
	data_proxy(data_t& Data) :
		m_data(Data)
	{
	}

	bitmap* value()
	{
		return m_data.value();
	}

	changed_signal_t& changed_signal()
	{
		return m_data.changed_signal();
	}

private:
	data_proxy(const data_proxy& RHS);
	data_proxy& operator=(const data_proxy& RHS);
	~data_proxy() {}

	data_t& m_data;
};

/// Specialization of bitmap_preview::data_proxy for use with iproperty objects
template<>
class data_proxy<iproperty> :
	public idata_proxy
{
public:
	typedef iproperty data_t;
	
	data_proxy(data_t& Data) :
		m_readable_data(Data)
	{
	}

	bitmap* value()
	{
		return boost::any_cast<bitmap*>(m_readable_data.value());
	}

	changed_signal_t& changed_signal()
	{
		return m_readable_data.changed_signal();
	}

private:
	data_proxy(const data_proxy& RHS);
	data_proxy& operator=(const data_proxy& RHS);
	~data_proxy() {}

	data_t& m_readable_data;
};

/// Convenience factory function for creating bitmap_preview::idata_proxy objects
template<typename data_t>
std::auto_ptr<idata_proxy> proxy(data_t& Data)
{
	return std::auto_ptr<idata_proxy>(new data_proxy<data_t>(Data));
}

} // namespace bitmap_preview

} // namespace k3d

#endif // !K3DUI_BITMAP_PREVIEW_H


