#ifndef K3DUI_DYNAMIC_MENU_H
#define K3DUI_DYNAMIC_MENU_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares k3d::dynamic_menu, which encapsulates a dynamically-constructed menu
		\author Ed Millard (emillard@direcway.com)
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/signal_system.h>

#include <sdpgtk/sdpgtkmenu.h>

#include <boost/shared_ptr.hpp>

#include <string>

namespace k3d
{

// Forward declarations
class icommand_node;

namespace dynamic_menu
{

/////////////////////////////////////////////////////////////////////////////
// item

/// Default menu item that provides sorting and can activate a signal slot, for use with k3d::dynamic_menu::control
class item
{
public:
	typedef SigC::Slot0<void> slot_t;

	item();
	item(const std::string& Label, const slot_t& Slot = slot_t());
	item(icommand_node* const Parent, const std::string& Name, const std::string& Label, const slot_t& Slot = slot_t());
	item(const std::string& Label, sdpGtkMenu& SubMenu);

	operator GtkWidget*() const;

	std::string label;

	friend bool operator==(const item& lhs, const item& rhs)
	{
		return lhs.label == rhs.label;
	}

	friend bool operator<(const item& lhs, const item& rhs)
	{
		return lhs.label < rhs.label;
	}

private:
	class implementation;
	boost::shared_ptr<implementation> m_implementation;
};

/////////////////////////////////////////////////////////////////////////////
// control

/// STL container adapter that can be used to create dynamically-generated menus
template<typename container_t = std::vector<item> >
class control :
	public container_t,
	public sdpGtkMenu
{
public:
	/// Creates a menu object from scratch
	control()
	{
		sdpGtkMenu::Create();
	}

	/// Binds an existing menu
	control(const sdpGtkMenu& Menu) :
		sdpGtkMenu(Menu)
	{
	}

	/** \brief Builds the UI menu contents from the container contents
		\note You must call build() after making any changes to the container contents, and prior to calling popup()
	*/
	void build()
	{
		for(typename container_t::iterator i = container_t::begin(); i != container_t::end(); ++i)
			sdpGtkMenu::Append(*i);
	}

	/// Displays the menu as a popup
	void popup(const unsigned int Button = 0)
	{
		sdpGtkMenu::Popup(Button);
	}
};

} // namespace dynamic_menu

} // namespace k3d

#endif // !K3DUI_DYNAMIC_MENU_H


