// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the k3d::edit_control::control class, which provides a UI for string quantities
		\author Tim Shead (tshead@k-3d.com)
*/

#include "edit_control.h"
#include "gtkml.h"

#include <k3dsdk/application.h>
#include <k3dsdk/istate_recorder.h>
#include <k3dsdk/state_change_set.h>
#include <k3dsdk/ioptions.h>

#include <sdpgtk/sdpgtkevent.h>

namespace
{

const std::string control_value = "value";
const std::string control_value_focus_in = "value_focus_in";
const std::string control_value_focus_out = "value_focus_out";

} // namespace

namespace k3d
{

namespace edit_control
{

/////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::iunknown* CommandNodeParent, const std::string CommandNodeName) :
	base(CommandNodeParent, CommandNodeName)
{
	// Create and load our UI template ...
	std::istringstream uitemplate(
		"<gtkml>"
			"<entry name=\"value\">"
				"<event signal=\"destroy\" name=\"destroy\"/>"
				"<event signal=\"focus-in-event\" name=\"value_focus_in\" after=\"true\"/>"
				"<event signal=\"focus-out-event\" name=\"value_focus_out\"/>"
			"</entry>"
		"</gtkml>\n");

	return_if_fail(load_gtkml(uitemplate, "edit control builtin template", *this));

	RootWidget().Show();
}

control::~control()
{
	// No more events from this point forward ...
	DisconnectAllEvents();

	// Clean-up the GTK+ tree ...
	if(Root())
		RootWidget().Destroy();
}

const std::string control::CustomType() const
{
	return std::string("k3deditcontrol");
}

bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(Command == control_value)
		{
			return_val_if_fail(InteractiveSetText(control_value.c_str(), Arguments.c_str(), k3d::application().options().tutorial_speed(), true), false);
			on_value_focus_out();
			return true;
		}

	return base::execute_command(Command, Arguments);
}

bool control::Create(sdpGtkIObjectContainer* ObjectContainer, sdpxml::Document& Document, sdpxml::Element& Element)
{
	// Sanity checks ...
	assert_warning(ObjectContainer);
	assert_warning(Element.Name() == "k3deditcontrol");

	return true;
}

bool control::attach(std::auto_ptr<idata_proxy> Data, k3d::istate_recorder* const StateRecorder, const std::string StateChangeName)
{
	// Sanity checks ...
	return_val_if_fail(Data.get(), false);

	// Take ownership of the data source ...
	m_data = Data;

	// Complete our own initialization ...
	return_val_if_fail(base::Attach(StateRecorder, StateChangeName), false);

	// Update the display ...
	update();

	// We want to be notified if the data source changes ...
	m_data->changed_signal().connect(SigC::slot(*this, &control::update));

	return true;
}

void control::set_focus()
{
	Editable(control_value.c_str()).SelectRegion(0, -1);
	Editable(control_value.c_str()).GrabFocus();
}

void control::update()
{
	// Sanity checks ...
	return_if_fail(m_data.get());

	// Display the current value ...
	Editable(control_value.c_str()).SetText(m_data->value().c_str());
}

void control::OnEvent(sdpGtkEvent* Event)
{
	// Sanity checks ...
	assert_warning(Event);

	if(Event->Name() == control_value_focus_in)
		on_value_focus_in();
	else if(Event->Name() == control_value_focus_out)
		on_value_focus_out();
	else if(Event->Name() == "destroy")
		on_destroy();
	else
		base::OnEvent(Event);
}

void control::on_destroy()
{
	DisconnectAllEvents();
	Clear();
}

void control::on_value_focus_in()
{
	// Make sure we've got some storage, first!
	return_if_fail(m_data.get());

	// Make sure everything's selected ...
	Editable(control_value.c_str()).SelectRegion(0, -1);
}

void control::on_value_focus_out()
{
	// Make sure we've got some storage, first!
	return_if_fail(m_data.get());

	// Get rid of the selection ...
	Editable(control_value.c_str()).SelectRegion(0, 0);

	// Get the control text ...
	const std::string newvalue = Editable(control_value.c_str()).GetText();

	// If the value hasn't changed, we're done ...
	if(newvalue == m_data->value())
		return;

	// Record the command for posterity (tutorials) ...
	k3d::application().command_signal().emit(this, k3d::icommand_node::command_t::USER_INTERFACE, control_value, newvalue);

	// Turn this into an undo/redo -able event ...
	if(m_StateRecorder)
		m_StateRecorder->start_recording(k3d::create_state_change_set());

	// Update everything with the new value ...
	m_data->set_value(newvalue);

	// Turn this into an undo/redo -able event ...
	if(m_StateRecorder)
		m_StateRecorder->commit_change_set(m_StateRecorder->stop_recording(), m_StateChangeName + ' ' + '"' + newvalue + '"');
}

} // namespace edit_control

} // namespace k3d


