#ifndef K3DUI_FILTER_SELECTOR_H
#define K3DUI_FILTER_SELECTOR_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares the k3d::filter_selector class, which provides the user with a standard UI for selecting file import/export filters
		\author Tim Shead (tshead@k-3d.com)
*/

#include "filter_selector_private.h"

#include <k3dsdk/file_filter.h>
#include <boost/filesystem/path.hpp>

namespace k3d
{

/////////////////////////////////////////////////////////////////////////////
// filter_selector

/// Implements the application user interface
template<typename interface_t>
class filter_selector :
	private implementation_private::filter_selector
{
public:
	filter_selector(const std::string Title, const boost::filesystem::path& File) :
		implementation_private::filter_selector(Title, File)
	{
		// Our special "key" value for automatic filter selection ...
		static const std::string automatic("Automatic");
		m_choices[automatic] = std::string();

		// Get the list of available filters ...
		const k3d::iplugin_factory_collection::factories_t filters(k3d::plugins<interface_t>());

		// Create a mapping from human-readable descriptions to file filter names ...
		for(k3d::iplugin_factory_collection::factories_t::const_iterator filter = filters.begin(); filter != filters.end(); ++filter)
			m_choices[(*filter)->short_description()] = (*filter)->name();

		// Create a collection of human-readable choices for the user ...
		std::vector<std::string> combo_choices;
		combo_choices.push_back(automatic);
		for(k3d::iplugin_factory_collection::factories_t::const_iterator filter = filters.begin(); filter != filters.end(); ++filter)
			combo_choices.push_back((*filter)->short_description());

		Combo("filter").SetList(combo_choices);
		Combo("filter").SetEntryText(automatic.c_str());
	}

	/// Display the dialog, prompting the user for input; returns true iff the user OK'ed the dialog, otherwise false
	bool do_modal(GtkWindow* TransientOwner)
	{
		return implementation_private::filter_selector::do_modal(TransientOwner);
	}

	/// Returns the filter selected by the user, or an automatically-selected filter, or NULL
	interface_t* filter()
	{
		return choice().size() ? k3d::file_filter<interface_t>(choice()) : k3d::auto_file_filter<interface_t>(m_file);
	}
};

} // namespace k3d

#endif // !K3DUI_FILTER_SELECTOR_H


