// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/algebra.h>
#include <k3dsdk/computed_property.h>
#include <k3dsdk/itransform_sink.h>
#include <k3dsdk/itransform_source.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>

namespace libk3dcore
{

/////////////////////////////////////////////////////////////////////////////
// look_at_implementation

class look_at_implementation :
	public k3d::persistent<k3d::object>,
	public k3d::itransform_source,
	public k3d::itransform_sink
{
	typedef k3d::persistent<k3d::object>  base;

public:
	look_at_implementation(k3d::idocument& Document) :
		base(Document),
		m_input(k3d::init_name("input_matrix") + k3d::init_description("Input matrix [matrix4]") + k3d::init_value(k3d::identity3D()) + k3d::init_document(Document)),
		m_target(k3d::init_name("target_matrix") + k3d::init_description("Target matrix [matrix4]") + k3d::init_value(k3d::identity3D()) + k3d::init_document(Document)),
		m_output("output_matrix", "Output matrix [matrix4]", k3d::method_call(*this, &look_at_implementation::output_value))
	{
		register_property(m_input);
		register_property(m_target);
		register_property(m_output);
		
		m_input.changed_signal().connect(m_output.changed_signal().slot());
		m_target.changed_signal().connect(m_output.changed_signal().slot());
	}

	k3d::iproperty& transform_source_output()
	{
		return m_output;
	}
	
	k3d::iproperty& transform_sink_input()
	{
		return m_input;
	}

	k3d::matrix4 output_value()
	{
		const k3d::matrix4 input_matrix = m_input.property_value();
		const k3d::matrix4 target_matrix = m_target.property_value();
	
		const k3d::vector3 from = input_matrix * k3d::vector3(0, 0, 0);
		const k3d::vector3 to = target_matrix * k3d::vector3(0, 0, 0);
		const k3d::vector3 spherical = (to - from).Spherical(); 	 

		return input_matrix * rotation3D(k3d::quaternion(k3d::euler_angles(0, -spherical[2], spherical[1], k3d::euler_angles::ZXYstatic)));
	}
			
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<look_at_implementation>,
			k3d::interface_list<k3d::itransform_source,
			k3d::interface_list<k3d::itransform_sink > > > factory(
			k3d::uuid(0x4e2a30f5, 0x6d7d47ad, 0x943ccd36, 0x4b305b55),
			"LookAt",
			"Orients a transformation matrix to look at another",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

	k3d_data_property(k3d::matrix4, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_input;
	k3d_data_property(k3d::matrix4, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_target;
	k3d::computed_property<k3d::matrix4, k3d::method_call_t<look_at_implementation, k3d::matrix4> > m_output;
};

/////////////////////////////////////////////////////////////////////////////
// look_at_factory

k3d::iplugin_factory& look_at_factory()
{
	return look_at_implementation::get_factory();
}

} // namespace libk3dcore




