// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief An object that take two doubles as input and produce their remainder as output
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/object.h>
#include <k3dsdk/computed_property.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>

namespace libk3dcore
{

class scalar_modulo_implementation : 
	public k3d::persistent<k3d::object> 
{
	typedef k3d::persistent<k3d::object>  base;
public:
	scalar_modulo_implementation(k3d::idocument& Document) : 
		base(Document),
		m_input1(k3d::init_name("input1") + k3d::init_description("First input [double]") + k3d::init_value(0.0) + k3d::init_document(Document)),
		m_input2(k3d::init_name("input2") + k3d::init_description("Second input [double]") + k3d::init_value(1.0) + k3d::init_document(Document)),
		m_output("output", "Remainder of inputs [double]", k3d::method_call(*this, &scalar_modulo_implementation::get_value))
	{
		register_property(m_input1);
		register_property(m_input2);
		register_property(m_output);

		enable_serialization(k3d::persistence::proxy(m_input1));
		enable_serialization(k3d::persistence::proxy(m_input2));

		m_input1.changed_signal().connect(m_output.changed_signal().slot());
		m_input2.changed_signal().connect(m_output.changed_signal().slot());
	}
	
	double get_value()
	{
		const double input1 = m_input1.property_value();
		const double input2 = m_input2.property_value();

		return_val_if_fail(input2 != 0, input1);
		
		return std::fmod(input1, input2);
	}
	
	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<scalar_modulo_implementation> > factory(
			k3d::uuid(0x7c6cb94a, 0xe20d4fa2, 0xb56f0f2a, 0x50e822a0),
			"ScalarModulo", 
			"Modulo two scalar inputs and produce their remainder as output", 
			"Objects", 
			k3d::iplugin_factory::STABLE);
		return factory;
	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}
	
private:
	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_input1;
	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_input2;
	k3d::computed_property<double, k3d::method_call_t<scalar_modulo_implementation, double> > m_output;
	
};

k3d::iplugin_factory& scalar_modulo_factory()
{
	return scalar_modulo_implementation::get_factory();
}

} //namespace libk3dcore
