// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/axis.h>
#include <k3dsdk/bounding_box.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// linear_wave_points_implementation

class linear_wave_points_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >,
	public k3d::mouse_event_observer
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	linear_wave_points_implementation(k3d::idocument& Document) :
		base(Document),
		k3d::mouse_event_observer("LMB drag to taper points"),
		m_axis(k3d::init_name("axis") + k3d::init_description("Axis [enumeration]") + k3d::init_document(Document) + k3d::init_value(k3d::Y) + k3d::init_enumeration(k3d::axis_values())),
		m_along(k3d::init_name("along") + k3d::init_description("Along [enumeration]") + k3d::init_document(Document) + k3d::init_value(k3d::X) + k3d::init_enumeration(k3d::axis_values())),
		m_amplitude(k3d::init_name("amplitude") + k3d::init_description("Amplitude [number]") + k3d::init_document(Document) + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_wavelength(k3d::init_name("wavelength") + k3d::init_description("Wavelength [number]") + k3d::init_document(Document) + k3d::init_value(10.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_phase(k3d::init_name("phase") + k3d::init_description("Phase [number]") + k3d::init_document(Document) + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(k3d::radians(1.0)) + k3d::init_units(typeid(k3d::measurement::angle)))
	{
		enable_serialization(k3d::persistence::proxy(m_axis));
		enable_serialization(k3d::persistence::proxy(m_along));
		enable_serialization(k3d::persistence::proxy(m_amplitude));
		enable_serialization(k3d::persistence::proxy(m_wavelength));
		enable_serialization(k3d::persistence::proxy(m_phase));

		register_property(m_axis);
		register_property(m_along);
		register_property(m_amplitude);
		register_property(m_wavelength);
		register_property(m_phase);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &linear_wave_points_implementation::on_reset_geometry));

		m_axis.changed_signal().connect(SigC::slot(*this, &linear_wave_points_implementation::on_reshape_geometry));
		m_along.changed_signal().connect(SigC::slot(*this, &linear_wave_points_implementation::on_reshape_geometry));
		m_amplitude.changed_signal().connect(SigC::slot(*this, &linear_wave_points_implementation::on_reshape_geometry));
		m_wavelength.changed_signal().connect(SigC::slot(*this, &linear_wave_points_implementation::on_reshape_geometry));
		m_phase.changed_signal().connect(SigC::slot(*this, &linear_wave_points_implementation::on_reshape_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &linear_wave_points_implementation::on_create_geometry));
	}

	bool OnLButtonDrag(const k3d::imouse_event_observer::event_state& State, const k3d::vector2& Current, const k3d::vector2& Last, const k3d::vector2& Start, const drag_type_t DragType)
	{
		const double xfactor = (Current[0] - Last[0]);
		const double yfactor = (Current[1] - Last[1]);

		m_amplitude.set_value(m_amplitude.value() + (std::fabs(xfactor) >= std::fabs(yfactor) ? xfactor : yfactor));

		return true;
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		reshape_geometry(*input, *output);

		return output;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		assert(Source.points.size() == Target.points.size());

		const k3d::axis axis = m_axis.property_value();
		const k3d::axis along = m_along.property_value();
		const double amplitude = m_amplitude.property_value();
		const double wavelength = m_wavelength.property_value();
		const double phase = m_phase.property_value();

		const k3d::vector3 offset_filter(k3d::X == axis, k3d::Y == axis, k3d::Z == axis);

		// Filter-out infinite frequencies ...
		if(0 == wavelength)
			return;

		const bool ignore_selection = !k3d::contains_selection(Source);

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			{
				if(ignore_selection || Source.points[i]->selected)
					{
						k3d::vector3 position = Source.points[i]->position;

						const double wave_position = phase + (k3d::pi_times_2() * position[along] / wavelength);
						const double offset = amplitude * sin(wave_position);

						Target.points[i]->position = position + (offset * offset_filter);
					}
				else
					{
						Target.points[i]->position = Source.points[i]->position;
					}
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<linear_wave_points_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x8053da77, 0xef6743cd, 0x9d60f12a, 0x63d1fdc7),
				"LinearWavePoints",
				"Applies a wave function to points along an axis",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_enumeration_property(k3d::axis, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_axis;
	k3d_enumeration_property(k3d::axis, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_along;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_amplitude;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_wavelength;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_phase;
};

/////////////////////////////////////////////////////////////////////////////
// linear_wave_points_factory

k3d::iplugin_factory& linear_wave_points_factory()
{
	return linear_wave_points_implementation::get_factory();
}

} // namespace libk3ddeformation


