// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/utility.h>

#include <boost/multi_array.hpp>
#include <iomanip>
#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// cap_hole_implementation

class cap_hole_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	cap_hole_implementation(k3d::idocument& Document) :
		base(Document),
		m_cap_polyhedra(k3d::init_name("cap_polyhedra") + k3d::init_description("Cap holes (missing faces) in polyhedra [boolean]") + k3d::init_document(Document) + k3d::init_value(true)),
		m_cap_faces(k3d::init_name("cap_faces") + k3d::init_description("Cap holes in faces [boolean]") + k3d::init_document(Document) + k3d::init_value(true))
	{
		enable_serialization(k3d::persistence::proxy(m_cap_polyhedra));
		enable_serialization(k3d::persistence::proxy(m_cap_faces));

		register_property(m_cap_polyhedra);
		register_property(m_cap_faces);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &cap_hole_implementation::on_reset_geometry));
		m_cap_polyhedra.changed_signal().connect(SigC::slot(*this, &cap_hole_implementation::on_reset_geometry));
		m_cap_faces.changed_signal().connect(SigC::slot(*this, &cap_hole_implementation::on_reset_geometry));
		
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &cap_hole_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	/// Returns true iff the given object is selected
	struct is_selected
	{
		template<typename T>
		bool operator()(T* const Object)
		{
			return Object->selected;
		}
	};

	/// Returns the next exterior edge (if any) in *counter-clockwise* order
	static k3d::split_edge* next_edge(k3d::split_edge* const Edge)
	{
		return_val_if_fail(Edge, 0);
		return_val_if_fail(!Edge->companion, 0);
		
		k3d::split_edge* edge = Edge->face_clockwise;
		return_val_if_fail(edge != Edge, 0);
		
		while(edge)
			{
				if(!edge->companion)
					return edge;
					
				edge = edge->companion->face_clockwise;
			}
				
		return 0;
	}
	
	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		// For each polyhedron ...
		for(k3d::mesh::polyhedra_t::iterator polyhedron = output->polyhedra.begin(); polyhedron != output->polyhedra.end(); ++polyhedron)
			{
				const bool cap_polyhedra = m_cap_polyhedra.property_value();
				const bool cap_faces = m_cap_faces.property_value();
			
				// Get the set of all selected edges ...
				std::set<k3d::split_edge*> selected_edges;
				k3d::copy_if((*polyhedron)->edges.begin(), (*polyhedron)->edges.end(), std::inserter(selected_edges, selected_edges.end()), is_selected());

				if(selected_edges.empty())
					selected_edges.insert((*polyhedron)->edges.begin(), (*polyhedron)->edges.end());

				// If we aren't going to cap polyhedra, remove all face loops from the edge list ...
				if(!cap_polyhedra)
					{
						for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
							{
								for(k3d::split_edge* edge = (*face)->first_edge; edge; )
									{
										selected_edges.erase(edge);
										
										edge = edge->face_clockwise;
										if(edge == (*face)->first_edge)
											break;
									}
							}
					}
					
				// If we aren't going to cap faces, remove all face holes from the edge list ...
				if(!cap_faces)
					{
						for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
							{
								for(k3d::face::holes_t::const_iterator hole = (*face)->holes.begin(); hole != (*face)->holes.end(); ++hole)
									{
										for(k3d::split_edge* edge = *hole; edge; )
											{
												selected_edges.erase(edge);
										
												edge = edge->face_clockwise;
												if(edge == *hole)
													break;
											}
									}
							}
					}
												
				// For each edge ...
				for(std::set<k3d::split_edge*>::iterator e = selected_edges.begin(); e != selected_edges.end(); ++e)
					{
						// Skip interior edges ...
						if((*e)->companion)
							continue;

						// Get the entire loop ...
						k3d::polyhedron::edges_t old_edges;
						for(k3d::split_edge* edge = *e; edge; )
							{
								old_edges.push_back(edge);

								edge = next_edge(edge);
								
								if(edge == *e)
									break;								
							}

						// Create new edges ...
						k3d::polyhedron::edges_t new_edges;
						for(unsigned long i = 0; i != old_edges.size(); ++i)
							{
								k3d::split_edge* const old_edge = old_edges[(old_edges.size() - i) % old_edges.size()];
								k3d::split_edge* const new_edge = new k3d::split_edge(old_edge->face_clockwise->vertex);
								
								k3d::join_edges(*new_edge, *old_edge);
								new_edges.push_back(new_edge);
							}
						
						// Create a new face ...
						k3d::loop_edges(new_edges.begin(), new_edges.end());
						(*polyhedron)->edges.insert((*polyhedron)->edges.end(), new_edges.begin(), new_edges.end());
						(*polyhedron)->faces.push_back(new k3d::face(new_edges.front()));
					}
				
				assert_warning(is_valid(**polyhedron));
			}

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<cap_hole_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xd9900dbf, 0x3d854a36, 0xbd05a3eb, 0x1379a89a),
				"CapHole",
				"Caps (fills) holes in a polyhedron",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}
	
private:
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_cap_polyhedra;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_cap_faces;
};

/////////////////////////////////////////////////////////////////////////////
// cap_hole_factory

k3d::iplugin_factory& cap_hole_factory()
{
	return cap_hole_implementation::get_factory();
}

} // namespace libk3dmesh

