// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Defines a geometric collision-detection region within an ODE world
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Brett W. McCoy
*/

#include "icollidable.h"

#include <k3dsdk/classes.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/module.h>
#include <k3dsdk/property.h>
#include <sdpgl/sdpgl.h>

#include <boost/bind.hpp>

#include <vector>

namespace
{

/////////////////////////////////////////////////////////////////////////////
// geometry

class geometry
{
public:
	virtual ~geometry()
	{
	  if(m_geom) dGeomDestroy(m_geom);
	}

	dGeomID geom()
	{
		return m_geom;
	}

        geometry(const dGeomID Geom) :
		m_geom(Geom)
	{
	}

  void destroy()
  {
    if(m_geom) dGeomDestroy(m_geom);
    m_geom = 0;
  }

	virtual void update() = 0;

protected:

  dGeomID m_geom;

private:

	geometry(const geometry&) {}
	geometry& operator=(const geometry&) { return *this; }
};

typedef std::vector<geometry*> geometry_t;

/////////////////////////////////////////////////////////////////////////////
// sphere

class sphere :
	public geometry
{
public:
	sphere(k3d::iobject& Object) :
		geometry(dCreateSphere(0, 0)),
		m_object(Object)
	{
	}

	void update()
	{
		const boost::any radius = k3d::get_property_value(m_object, "radius");
		return_if_fail(radius.type() == typeid(double));

		dGeomSphereSetRadius(geom(), boost::any_cast<double>(radius));
	}

private:
	k3d::iobject& m_object;
};

/////////////////////////////////////////////////////////////////////////////
// create_geometry

class create_geometry
{
public:
	create_geometry(geometry_t& Geometry) :
		m_geometry(Geometry)
	{
	}

	void operator()(k3d::iobject& Object)
	{
		if(k3d::classes::Sphere() == Object.factory().class_id())
			m_geometry.push_back(new sphere(Object));
	}

private:
	geometry_t& m_geometry;
};

/////////////////////////////////////////////////////////////////////////////
// collidable_implementation

class collidable_implementation :
	public k3d::persistent<k3d::object> ,
	public libk3dode::icollidable
{
	typedef k3d::persistent<k3d::object>  base;

public:
	collidable_implementation(k3d::idocument& Document) :
		base(Document)
	{
		Document.hierarchy().changed_signal().connect(SigC::slot(*this, &collidable_implementation::on_hierarchy_changed));
	}

	~collidable_implementation()
	{
		destroy_geometry();
	}

	void create_geometry()
	{
		k3d::for_each_descendant(document().hierarchy(), this, ::create_geometry(m_geometry));
	}

	void update_geometry()
	{
		std::for_each(m_geometry.begin(), m_geometry.end(), boost::mem_fn(&geometry::update));
	}

	void destroy_geometry()
	{
		std::for_each(m_geometry.begin(), m_geometry.end(), k3d::delete_object());
		m_geometry.clear();
	}

	void on_hierarchy_changed()
	{
		destroy_geometry();
		create_geometry();
	}

	const geoms_t geoms()
	{
		update_geometry();

		geoms_t results;
		for(geometry_t::iterator geometry = m_geometry.begin(); geometry != m_geometry.end(); ++geometry)
			results.push_back((*geometry)->geom());

		return results;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<collidable_implementation> > factory(
			k3d::uuid(0xc2f25a51, 0xac124157, 0xb425bfbe, 0xe0bd5ab9),
			"ODECollidable",
			"Experimental ODE dynamics library plugin",
			"Objects",
			k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	geometry_t m_geometry;
};

} // namespace

namespace libk3dode
{

/////////////////////////////////////////////////////////////////////////////
// body_factory

k3d::iplugin_factory& collidable_factory()
{
	return ::collidable_implementation::get_factory();
}

} // namespace libk3dode


