// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the k3d::ri::render class, which provides a default implementation of k3d::ri::irender
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include "algebra.h"
#include "imaterial.h"
#include "irenderman_property.h"
#include "mesh.h"
#include "renderman.h"
#include "vectors.h"

#include <boost/array.hpp>

#include <iostream>
#include <numeric>
#include <set>
#include <stack>

namespace
{

long& indentation_storage(std::ios& Stream)
{
	static const int index = std::ios::xalloc();
	return Stream.iword(index);
}

std::ostream& reset_indentation(std::ostream& Stream)
{
	indentation_storage(Stream) = 0;
	return Stream;
}

std::ostream& push_indent(std::ostream& Stream)
{
	indentation_storage(Stream)++;
	return Stream;
}

std::ostream& pop_indent(std::ostream& Stream)
{
	long& indent = indentation_storage(Stream);
	indent -= (indent > 0);
	return Stream;
}

std::ostream& indentation(std::ostream& Stream)
{
	const long& indent = indentation_storage(Stream);
	for(long i = 0; i < indent; i++)
		Stream << "   ";

	return Stream;
}

long& inline_types_storage(std::ios& Stream)
{
	static const int index = std::ios_base::xalloc();
	return Stream.iword(index);
}

bool inline_types(std::ostream& Stream)
{
	return inline_types_storage(Stream) ? true : false;
}

bool set_inline_types(std::ostream& Stream, const bool Enabled)
{
	bool old_state = inline_types_storage(Stream) ? true : false;

	inline_types_storage(Stream) = Enabled;

	return old_state;
}

/// Formats a string with real-quotes for inclusion in a RIB file; designed to be used as an inline formatting object
class format_string
{
public:
	explicit format_string(const k3d::ri::string& Token) :
		token(Token)
	{
	}

	friend std::ostream& operator<<(std::ostream& Stream, const format_string& RHS)
	{
		Stream << "\"" << RHS.token << "\"";
		return Stream;
	}

private:
	const k3d::ri::string& token;
};

/// Formats a matrix for inclusion in a RIB file; designed to be used as an inline formatting object
class format_matrix
{
public:
	explicit format_matrix(const k3d::ri::matrix& M) :
		m(M)
	{
	}

	friend std::ostream& operator<<(std::ostream& Stream, const format_matrix& RHS)
	{
		Stream << "[";
		for(int i = 0; i != 4; ++i)
		{
			for(int j = 0; j != 4; ++j)
			{
				Stream << RHS.m[j][i] << " ";
			}
		}
		Stream << "]";
		
		return Stream;
	}

private:
	const k3d::ri::matrix& m;
};

/// Encapsulates a RenderMan RIB parameter type
struct predefined_type
{
	explicit predefined_type(const k3d::ri::parameter& Parameter) :
		storage_class(Parameter.storage_class),
		name(Parameter.name),
		type(Parameter.value.type()),
		array_dimension(Parameter.array_dimension)
	{
	}

	explicit predefined_type(const k3d::ri::storage_class_t StorageClass, const k3d::ri::string& Name, const std::type_info& Type, const k3d::ri::unsigned_integer ArrayDimension) :
		storage_class(StorageClass),
		name(Name),
		type(Type),
		array_dimension(ArrayDimension)
	{
	}

	friend bool operator<(const predefined_type& LHS, const predefined_type& RHS)
	{
		if(LHS.storage_class != RHS.storage_class)
			return LHS.storage_class < RHS.storage_class;

		if(LHS.array_dimension != RHS.array_dimension)
			return LHS.array_dimension < RHS.array_dimension;

		if(LHS.type != RHS.type)
			return LHS.type.before(RHS.type) ? true : false;

		return LHS.name < RHS.name;
	}

	k3d::ri::storage_class_t storage_class;
	const k3d::ri::string name;
	const std::type_info& type;
	const k3d::ri::unsigned_integer array_dimension;
};

/// A collection of RenderMan RIB parameter types
typedef std::set<predefined_type> predefined_types_t;

/// Returns the set of standard predefined RIB parameter types
predefined_types_t& predefined_types()
{
	static predefined_types_t types;
	if(types.empty())
	{
		types.insert(predefined_type(k3d::ri::VERTEX, "P", typeid(k3d::ri::point), 1));
		types.insert(predefined_type(k3d::ri::VERTEX, "P", typeid(k3d::ri::points), 1));
		types.insert(predefined_type(k3d::ri::VERTEX, "Pz", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::VERTEX, "Pz", typeid(k3d::ri::reals), 1));
		types.insert(predefined_type(k3d::ri::VERTEX, "Pw", typeid(k3d::ri::hpoint), 1));
		types.insert(predefined_type(k3d::ri::VERTEX, "Pw", typeid(k3d::ri::hpoints), 1));

/*
		types.insert(predefined_type(k3d::ri::VARYING, "N", typeid(k3d::ri::normal), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "N", typeid(k3d::ri::normals), 1));
*/

		types.insert(predefined_type(k3d::ri::VARYING, "Cs", typeid(k3d::ri::color), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "Cs", typeid(k3d::ri::colors), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "Os", typeid(k3d::ri::color), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "Os", typeid(k3d::ri::colors), 1));

		types.insert(predefined_type(k3d::ri::VARYING, "s", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "s", typeid(k3d::ri::reals), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "t", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "t", typeid(k3d::ri::reals), 1));

		types.insert(predefined_type(k3d::ri::VARYING, "blur", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "blur", typeid(k3d::ri::reals), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "sblur", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "sblur", typeid(k3d::ri::reals), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "tblur", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::VARYING, "tblur", typeid(k3d::ri::reals), 1));

		types.insert(predefined_type(k3d::ri::UNIFORM, "width", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::UNIFORM, "width", typeid(k3d::ri::reals), 1));
		types.insert(predefined_type(k3d::ri::UNIFORM, "swidth", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::UNIFORM, "swidth", typeid(k3d::ri::reals), 1));
		types.insert(predefined_type(k3d::ri::UNIFORM, "twidth", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::UNIFORM, "twidth", typeid(k3d::ri::reals), 1));

		types.insert(predefined_type(k3d::ri::UNIFORM, "filter", typeid(k3d::ri::string), 1));
		types.insert(predefined_type(k3d::ri::UNIFORM, "filter", typeid(k3d::ri::strings), 1));

		types.insert(predefined_type(k3d::ri::UNIFORM, "fill", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::UNIFORM, "fill", typeid(k3d::ri::reals), 1));

		types.insert(predefined_type(k3d::ri::UNIFORM, "fov", typeid(k3d::ri::real), 1));
		types.insert(predefined_type(k3d::ri::UNIFORM, "fov", typeid(k3d::ri::reals), 1));

		types.insert(predefined_type(k3d::ri::UNIFORM, "shader", typeid(k3d::ri::string), 1));
	}

	return types;
}

/// Formats a parameter name for inclusion in a RIB file
class format_parameter_name
{
public:
	explicit format_parameter_name(const k3d::ri::parameter& Parameter) :
		parameter(Parameter)
	{
	}

	friend std::ostream& operator<<(std::ostream& Stream, const format_parameter_name& RHS)
	{
		Stream << "\"";

		// Only generate type information if inlining is enabled ...
		if(inline_types(Stream))
		{
			// Only generate type information if this type isn't already predefined ...
			if(!predefined_types().count(predefined_type(RHS.parameter)))
			{
				Stream << RHS.parameter.storage_class << " ";

				const std::type_info& type = RHS.parameter.value.type();

				if(RHS.parameter.value.empty())
				{
					Stream << "unknown ";
					k3d::log() << error << "Cannot deduce type for empty parameter" << std::endl;
				}
				else if(typeid(k3d::ri::integer) == type || typeid(k3d::ri::integers) == type)
				{
					Stream << "integer";
				}
				else if(typeid(k3d::ri::real) == type || typeid(k3d::ri::reals) == type)
				{
					Stream << "float";
				}
				else if(typeid(k3d::ri::string) == type || typeid(k3d::ri::strings) == type)
				{
					Stream << "string";
				}
				else if(typeid(k3d::ri::point) == type || typeid(k3d::ri::points) == type)
				{
					Stream << "point";
				}
				else if(typeid(k3d::ri::vector) == type || typeid(k3d::ri::vectors) == type)
				{
					Stream << "vector";
				}
/*
				else if(typeid(k3d::ri::normal) == type || typeid(k3d::ri::normals) == type)
				{
					Stream << "normal";
				}
*/
				else if(typeid(k3d::ri::color) == type || typeid(k3d::ri::colors) == type)
				{
					Stream << "color";
				}
				else if(typeid(k3d::ri::hpoint) == type || typeid(k3d::ri::hpoints) == type)
				{
					Stream << "hpoint";
				}
				else if(typeid(k3d::ri::matrix) == type || typeid(k3d::ri::matrices) == type)
				{
					Stream << "matrix";
				}
				else
				{
					Stream << "unknown";
					k3d::log() << error << "Cannot deduce parameter type for [" << RHS.parameter.name << "]" << std::endl;
				}

				if(RHS.parameter.array_dimension > 1)
					Stream << "[" << RHS.parameter.array_dimension << "]";

				Stream << " ";
			}
		}

		Stream << RHS.parameter.name;

		Stream << "\"";

		return Stream;
	}

private:
	const k3d::ri::parameter& parameter;
};

/// Formats an array of values within square brackets for inclusion in a RIB file; designed to be used as an inline formatting object
template<typename iterator_t, typename value_t>
class format_array_t
{
public:
	format_array_t(const iterator_t Begin, const iterator_t End) :
		begin(Begin),
		end(End)
	{
	}

	friend std::ostream& operator << (std::ostream& Stream, const format_array_t& RHS)
	{
		Stream << "[ ";
		std::copy(RHS.begin, RHS.end, std::ostream_iterator<value_t>(Stream, " "));
		Stream << "]";

		return Stream;
	}

private:
	const iterator_t begin;
	const iterator_t end;
};

/// Partial specialization of format_array_t for use with string values
template<typename iterator_t>
class format_array_t<iterator_t, k3d::ri::string>
{
public:
	format_array_t(const iterator_t Begin, const iterator_t End) :
		begin(Begin),
		end(End)
	{
	}

	friend std::ostream& operator << (std::ostream& Stream, const format_array_t& RHS)
	{
		Stream << "[ ";
		for(iterator_t element = RHS.begin; element != RHS.end; ++element)
			Stream << format_string(*element) << " ";
		Stream << "]";

		return Stream;
	}

private:
	const iterator_t begin;
	const iterator_t end;
};

/// Convenience factory function for creating format_array_t objects
template<typename iterator_t>
format_array_t<iterator_t, typename std::iterator_traits<iterator_t>::value_type> format_array(const iterator_t Begin, const iterator_t End)
{
	return format_array_t<iterator_t, typename std::iterator_traits<iterator_t>::value_type>(Begin, End);
}

} // namespace

namespace k3d
{

namespace ri
{

std::ostream& operator<<(std::ostream& Stream, const parameter& RHS)
{
	Stream << format_parameter_name(RHS) << " ";

	if(RHS.value.empty())
	{
		Stream << "[ ]";
	}
	else if(typeid(integer) == RHS.value.type())
	{
		Stream << "[ " << boost::any_cast<integer>(RHS.value) << " ]";
	}
	else if(typeid(real) == RHS.value.type())
	{
		Stream << "[ " << boost::any_cast<real>(RHS.value) << " ]";
	}
	else if(typeid(string) == RHS.value.type())
	{
		Stream << "[ " << format_string(boost::any_cast<string>(RHS.value)) << " ]";
	}
	else if(typeid(point) == RHS.value.type())
	{
		Stream << "[ " << boost::any_cast<point>(RHS.value) << " ]";
	}
	else if(typeid(vector) == RHS.value.type())
	{
		Stream << "[ " << boost::any_cast<vector>(RHS.value) << " ]";
	}
/*
	else if(typeid(normal) == RHS.value.type())
	{
		Stream << "[ " << boost::any_cast<normal>(RHS.value) << " ]";
	}
*/
	else if(typeid(color) == RHS.value.type())
	{
		Stream << "[ " << boost::any_cast<color>(RHS.value) << " ]";
	}
	else if(typeid(hpoint) == RHS.value.type())
	{
		Stream << "[ " << boost::any_cast<hpoint>(RHS.value) << " ]";
	}
	else if(typeid(matrix) == RHS.value.type())
	{
		Stream << "[ " << boost::any_cast<matrix>(RHS.value) << " ]";
	}
	else if(typeid(integers) == RHS.value.type())
	{
		const integers* const p = boost::any_cast<integers>(&RHS.value);
		Stream << format_array(p->begin(), p->end());
	}
	else if(typeid(reals) == RHS.value.type())
	{
		const reals* const p = boost::any_cast<reals>(&RHS.value);
		Stream << format_array(p->begin(), p->end());
	}
	else if(typeid(strings) == RHS.value.type())
	{
		const strings* const p = boost::any_cast<strings>(&RHS.value);
		Stream << format_array(p->begin(), p->end());
	}
	else if(typeid(points) == RHS.value.type())
	{
		const points* const p = boost::any_cast<points>(&RHS.value);
		Stream << format_array(p->begin(), p->end());
	}
	else if(typeid(vectors) == RHS.value.type())
	{
		const vectors* const p = boost::any_cast<vectors>(&RHS.value);
		Stream << format_array(p->begin(), p->end());
	}
/*
	else if(typeid(normals) == RHS.value.type())
	{
		const normals* const p = boost::any_cast<normals>(&RHS.value);
		Stream << format_array(p->begin(), p->end());
	}
*/
	else if(typeid(colors) == RHS.value.type())
	{
		const colors* const p = boost::any_cast<colors>(&RHS.value);
		Stream << format_array(p->begin(), p->end());
	}
	else if(typeid(hpoints) == RHS.value.type())
	{
		const hpoints* const p = boost::any_cast<hpoints>(&RHS.value);
		Stream << format_array(p->begin(), p->end());
	}
	else if(typeid(matrices) == RHS.value.type())
	{
		const matrices* const p = boost::any_cast<matrices>(&RHS.value);
		Stream << format_array(p->begin(), p->end());
	}
	else
	{
		Stream << "[ ]";
		log() << error << k3d_file_reference << "Unknown type [" << RHS.value.type().name() << "] for [" << RHS.name << "] will be ignored" << std::endl;
	}

	return Stream;
}

std::ostream& operator<<(std::ostream& Stream, const parameter_list& RHS)
{
	std::copy(RHS.begin(), RHS.end(), std::ostream_iterator<parameter>(Stream, " "));
	return Stream;
}

/////////////////////////////////////////////////////////////////////////////
// motion_begin

void motion_begin(const render_state& State)
{
	if(motion_blur(State))
		State.engine.RiMotionBeginV(State.sample_times);
}

/////////////////////////////////////////////////////////////////////////////
// motion_end

void motion_end(const render_state& State)
{
	if(motion_blur(State))
		State.engine.RiMotionEnd();
}

/////////////////////////////////////////////////////////////////////////////
// motion_blur

bool motion_blur(const render_state& State)
{
	return State.sample_times.size() > 1;
}

/////////////////////////////////////////////////////////////////////////////
// first_sample

bool first_sample(const render_state& State)
{
	return 0 == State.sample_index;
}

/////////////////////////////////////////////////////////////////////////////
// last_sample

bool last_sample(const render_state& State)
{
	return State.sample_index == State.sample_times.size() - 1;
}

/////////////////////////////////////////////////////////////////////////////
// convert

const matrix convert(const k3d::matrix4& Matrix)
{
	return Matrix;
}

/////////////////////////////////////////////////////////////////////////////
// setup_material

void setup_material(iunknown* const Material, const render_state& State)
{
	k3d::ri::imaterial* ri_material = 0;

	if(k3d::imaterial* const material = dynamic_cast<k3d::imaterial*>(Material))
		ri_material = material->ri_material();

	if(ri_material)
	{
		ri_material->setup_renderman_material(State);
	}
	else
	{
		// We only generate RIB on the final sample ...
		if(!last_sample(State))
			return;

		State.engine.RiColor(k3d::color(1, 1, 1));
		State.engine.RiOpacity(k3d::color(1, 1, 1));
		State.engine.RiSurfaceV(path(), "null");
		State.engine.RiDisplacementV(path(), "null");
		State.engine.RiAtmosphereV(path(), "null");
		State.engine.RiInteriorV(path(), "null");
		State.engine.RiExteriorV(path(), "null");
	}
}

/////////////////////////////////////////////////////////////////////////////
// set_attributes

void set_attributes(iproperty_collection& Properties, irender_engine& Engine)
{
	const iproperty_collection::properties_t& properties = Properties.properties();

	typedef std::map<std::string, parameter_list> parameter_lists_t;
	parameter_lists_t parameter_lists;

	for(iproperty_collection::properties_t::const_iterator property = properties.begin(); property != properties.end(); ++property)
	{
		if(irenderman_property* const renderman_property = dynamic_cast<irenderman_property*>(*property))
		{
			if(renderman_property->property_parameter_list_type() != irenderman_property::ATTRIBUTE)
				continue;

			const std::string parameter_list_name = renderman_property->property_parameter_list_name();
			const std::string parameter_name = (*property)->property_name();

			parameter_lists[parameter_list_name].push_back(parameter(parameter_name, UNIFORM, (*property)->property_value()));
		}
	}

	for(parameter_lists_t::iterator parameter_list = parameter_lists.begin(); parameter_list != parameter_lists.end(); ++parameter_list)
		Engine.RiAttributeV(parameter_list->first, parameter_list->second);
}

/////////////////////////////////////////////////////////////////////////////
// set_options

/// Extracts RenderMan data from a property collection, generating corresponding calls to RiOption()
void set_options(iproperty_collection& Properties, irender_engine& Engine)
{
	const iproperty_collection::properties_t& properties = Properties.properties();

	typedef std::map<std::string, parameter_list> parameter_lists_t;
	parameter_lists_t parameter_lists;

	for(iproperty_collection::properties_t::const_iterator property = properties.begin(); property != properties.end(); ++property)
	{
		if(irenderman_property* const renderman_property = dynamic_cast<irenderman_property*>(*property))
		{
			if(renderman_property->property_parameter_list_type() != irenderman_property::OPTION)
				continue;

			const std::string parameter_list_name = renderman_property->property_parameter_list_name();
			const std::string parameter_name = (*property)->property_name();

			parameter_lists[parameter_list_name].push_back(parameter(parameter_name, UNIFORM, (*property)->property_value()));
		}
	}

	for(parameter_lists_t::iterator parameter_list = parameter_lists.begin(); parameter_list != parameter_lists.end(); ++parameter_list)
		Engine.RiAttributeV(parameter_list->first, parameter_list->second);
}

/////////////////////////////////////////////////////////////////////////////
// render_engine::implementation

class render_engine::implementation
{
public:
	implementation(std::ostream& Stream) :
		m_stream(Stream),
		m_light_handle(0),
		m_object_handle(0),
		m_frame_block(false),
		m_world_block(false),
		m_object_block(false),
		m_motion_block(false)
	{
	}

	/// Stores the RIB output stream
	std::ostream& m_stream;
	/// Stores the current light handle
	light_handle m_light_handle;
	/// Stores the current object handle
	object_handle m_object_handle;
	/// Set to true within a frame block
	bool m_frame_block;
	/// Set to true within a world block
	bool m_world_block;
	/// Set to true within an object block
	bool m_object_block;
	/// Set to true within a motion block
	bool m_motion_block;
	/// Stores the set of shaders in-use
	shaders_t m_shaders;
};

/////////////////////////////////////////////////////////////////////////////
// render_engine

render_engine::render_engine(std::ostream& Stream) :
	m_implementation(new implementation(Stream))
{
	// Enable inline type declarations by default ...
	::set_inline_types(m_implementation->m_stream, true);

	// Start out without any indentation ...
	::reset_indentation(m_implementation->m_stream);

	// Start writing the RIB file header ...
	RiStructure("RenderMan RIB-Structure 1.0");
	m_implementation->m_stream << "version 3.03" << "\n";
}

render_engine::~render_engine()
{
	delete m_implementation;
}

const render_engine::shaders_t render_engine::shaders()
{
	return m_implementation->m_shaders;
}

bool render_engine::set_inline_types(const bool Inline)
{
	return ::set_inline_types(m_implementation->m_stream, Inline);
}

void render_engine::use_shader(const path& Path)
{
	m_implementation->m_shaders.insert(Path);
}

void render_engine::RiDeclare(const string& Name, const string& Type)
{
	// Sanity checks ...
	return_if_fail(Name.size());
	return_if_fail(Type.size());

	m_implementation->m_stream << ::indentation << "Declare " << format_string(Name) << " " << format_string(Type) << "\n";
}

void render_engine::RiFrameBegin(const unsigned_integer FrameNumber)
{
	// Sanity checks ...
	if(m_implementation->m_frame_block)
	{
		log() << error << "Cannot nest calls to RiFrameBegin()" << std::endl;
		return;
	}

	m_implementation->m_frame_block = true;
	m_implementation->m_stream << ::indentation << ::indentation << "FrameBegin " << FrameNumber << "\n";
	::push_indent(m_implementation->m_stream);
}

void render_engine::RiFrameEnd()
{
	::pop_indent(m_implementation->m_stream);
	m_implementation->m_stream << ::indentation << "FrameEnd" << "\n";
	m_implementation->m_frame_block = false;
}

void render_engine::RiWorldBegin()
{
	// Sanity checks ...
	if(m_implementation->m_world_block)
	{
		log() << error << "Cannot nest calls to RiWorldBegin()" << std::endl;
		return;
	}

	m_implementation->m_world_block = true;
	m_implementation->m_stream << ::indentation << "WorldBegin" << "\n";
	::push_indent(m_implementation->m_stream);
}

void render_engine::RiWorldEnd()
{
	::pop_indent(m_implementation->m_stream);
	m_implementation->m_stream << ::indentation << "WorldEnd" << "\n";
	m_implementation->m_world_block = false;
}

void render_engine::RiFormat(const unsigned_integer XResolution, const unsigned_integer YResolution, const real AspectRatio)
{
	m_implementation->m_stream << ::indentation << "Format " << XResolution << " " << YResolution << " " << AspectRatio << "\n";
}

void render_engine::RiFrameAspectRatio(real AspectRatio)
{
	m_implementation->m_stream << ::indentation << "FrameAspectRatio " << AspectRatio << "\n";
}

void render_engine::RiScreenWindow(real Left, real Right, real Bottom, real Top)
{
	m_implementation->m_stream << ::indentation << "ScreenWindow " << Left << " " << Right << " " << Bottom << " " << Top << "\n";
}

void render_engine::RiCropWindow(real XMin, real XMax, real YMin, real YMax)
{
	m_implementation->m_stream << ::indentation << "CropWindow " << XMin << " " << XMax << " " << YMin << " " << YMax << "\n";
}

void render_engine::RiProjectionV(const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Projection " << format_string(Name) << " " << Parameters << "\n";
}

void render_engine::RiClipping(real NearPlane, real FarPlane)
{
	m_implementation->m_stream << ::indentation << "Clipping " << NearPlane << " " << FarPlane << "\n";
}

void render_engine::RiDepthOfField(real FStop, real FocalLength, real FocalDistance)
{
	m_implementation->m_stream << ::indentation << "DepthOfField " << FStop << " " << FocalLength << " " << FocalDistance << "\n";
}

void render_engine::RiShutter(real OpenTime, real CloseTime)
{
	m_implementation->m_stream << ::indentation << "Shutter " << OpenTime << " " << CloseTime << "\n";
}

void render_engine::RiPixelFilter(const string& FilterName, real XWidth, real YWidth)
{
	m_implementation->m_stream << ::indentation << "PixelFilter " << format_string(FilterName) << " " << XWidth << " " << YWidth << "\n";
}

void render_engine::RiPixelVariance(real Variation)
{
	m_implementation->m_stream << ::indentation << "PixelVariance " << Variation << "\n";
}
void render_engine::RiPixelSamples(real XSamples, real YSamples)
{
	m_implementation->m_stream << ::indentation << "PixelSamples " << XSamples << " " << YSamples << "\n";
}

void render_engine::RiExposure(real Gain, real Gamma)
{
	m_implementation->m_stream << ::indentation << "Exposure " << Gain << " " << Gamma << "\n";
}

void render_engine::RiImagerV(const path& Path, const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_shaders.insert(Path);
	m_implementation->m_stream << ::indentation << "Imager " << format_string(Name) << " " << Parameters << "\n";
}

void render_engine::RiQuantize(const string& Type, integer One, integer QMin, integer QMax, real Amplitude)
{
	m_implementation->m_stream << ::indentation << "Quantize " << format_string(Type) << " " << One << " " << QMin << " " << QMax << " " << Amplitude << "\n";
}

void render_engine::RiDisplayV(const string& Name, const string& Type, const string& Mode, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Display " << format_string(Name) << " " << format_string(Type) << " " << format_string(Mode) << " " << Parameters << "\n";
}

void render_engine::RiHiderV(const string& Type, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Hider " << format_string(Type) << " " << Parameters << "\n";
}

void render_engine::RiColorSamples(const unsigned_integer ParameterCount, const reals& nRGB, const reals& RGBn)
{
	return_if_fail(ParameterCount == nRGB.size());
	return_if_fail(ParameterCount == RGBn.size());

	m_implementation->m_stream << ::indentation << "ColorSamples " << format_array(nRGB.begin(), nRGB.end()) << " " << format_array(RGBn.begin(), RGBn.end()) << "\n";
}

void render_engine::RiRelativeDetail(real RelativeDetail)
{
	m_implementation->m_stream << ::indentation << "RelativeDetail " << RelativeDetail << "\n";
}

void render_engine::RiOptionV(const string& Name, const parameter_list& Parameters)
{
	const bool old_state = ::set_inline_types(m_implementation->m_stream, false);

	m_implementation->m_stream << ::indentation << "Option " << format_string(Name) << " " << Parameters << "\n";

	::set_inline_types(m_implementation->m_stream, old_state);
}

void render_engine::RiAttributeBegin()
{
	m_implementation->m_stream << ::indentation << "AttributeBegin" << "\n";
	::push_indent(m_implementation->m_stream);
}

void render_engine::RiAttributeEnd()
{
	::pop_indent(m_implementation->m_stream);
	m_implementation->m_stream << ::indentation << "AttributeEnd" << "\n";
}

void render_engine::RiColor(const color& Color)
{
	m_implementation->m_stream << ::indentation << "Color " << Color << "\n";
}

void render_engine::RiOpacity(const color& Opacity)
{
	m_implementation->m_stream << ::indentation << "Opacity " << Opacity << "\n";
}

void render_engine::RiTextureCoordinates(real S1, real T1, real S2, real T2, real S3, real T3, real S4, real T4)
{
	m_implementation->m_stream << ::indentation << "TextureCoordinates " << S1 << " " << T1 << " " << S2 << " " << T2 << " " << S3 << " " << T3 << " " << S4 << " " << T4 << "\n";
}

const light_handle render_engine::RiLightSourceV(const path& Path, const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_shaders.insert(Path);
	m_implementation->m_stream << ::indentation << "LightSource " << format_string(Name) << " " << ++m_implementation->m_light_handle << " " << Parameters << "\n";
	return m_implementation->m_light_handle;
}

const light_handle render_engine::RiAreaLightSourceV(const path& Path, const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_shaders.insert(Path);
	m_implementation->m_stream << ::indentation << "AreaLightSource " << format_string(Name) << " " << ++m_implementation->m_light_handle << " " << Parameters << "\n";
	return m_implementation->m_light_handle;
}

void render_engine::RiIlluminate(const light_handle LightHandle, bool OnOff)
{
	m_implementation->m_stream << ::indentation << "Illuminate " << LightHandle << " " << OnOff << "\n";
}

void render_engine::RiSurfaceV(const path& Path, const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_shaders.insert(Path);
	m_implementation->m_stream << ::indentation << "Surface " << format_string(Name) << " " << Parameters << "\n";
}

void render_engine::RiAtmosphereV(const path& Path, const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_shaders.insert(Path);
	m_implementation->m_stream << ::indentation << "Atmosphere " << format_string(Name) << " " << Parameters << "\n";
}

void render_engine::RiInteriorV(const path& Path, const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_shaders.insert(Path);
	m_implementation->m_stream << ::indentation << "Interior " << format_string(Name) << " " << Parameters << "\n";
}

void render_engine::RiExteriorV(const path& Path, const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_shaders.insert(Path);
	m_implementation->m_stream << ::indentation << "Exterior " << format_string(Name) << " " << Parameters << "\n";
}

void render_engine::RiShadingRate(real Size)
{
	m_implementation->m_stream << ::indentation << "ShadingRate " << Size << "\n";
}

void render_engine::RiShadingInterpolation(const string& Type)
{
	m_implementation->m_stream << ::indentation << "ShadingInterpolation " << format_string(Type) << "\n";
}

void render_engine::RiMatte(bool OnOff)
{
	m_implementation->m_stream << ::indentation << "Matte " << OnOff << "\n";
}

void render_engine::RiBound(const boost::array<real, 6>& Bound)
{
	m_implementation->m_stream << ::indentation << "Bound " << format_array(Bound.begin(), Bound.end()) << "\n";
}

void render_engine::RiDetail(const boost::array<real, 6>& Bound)
{
	m_implementation->m_stream << ::indentation << "Detail " << format_array(Bound.begin(), Bound.end()) << "\n";
}

void render_engine::RiDetailRange(const real MinVis, const real LowTran, const real UpTran, const real MaxVis)
{
	m_implementation->m_stream << ::indentation << "DetailRange " << MinVis << " " << LowTran << " " << UpTran << " " << MaxVis << "\n";
}

void render_engine::RiGeometricApproximation(const string& Type, real Value)
{
	m_implementation->m_stream << ::indentation << "GeometricApproximation " << format_string(Type) << " " << Value << "\n";
}

void render_engine::RiGeometricRepresentation(const string& Type)
{
	m_implementation->m_stream << ::indentation << "GeometricRepresentation " << format_string(Type) << "\n";
}

void render_engine::RiOrientation(const string& Orientation)
{
	m_implementation->m_stream << ::indentation << "Orientation " << format_string(Orientation) << "\n";
}

void render_engine::RiReverseOrientation()
{
	m_implementation->m_stream << ::indentation << "ReverseOrientation" << "\n";
}

void render_engine::RiSides(const unsigned_integer Sides)
{
	m_implementation->m_stream << ::indentation << "Sides " << Sides << "\n";
}

void render_engine::RiIdentity()
{
	m_implementation->m_stream << ::indentation << "Identity" << "\n";
}

void render_engine::RiTransform(const matrix& Transform)
{
	m_implementation->m_stream << ::indentation << "Transform " << format_matrix(Transform) << "\n";
}

void render_engine::RiConcatTransform(const matrix& Transform)
{
	m_implementation->m_stream << ::indentation << "ConcatTransform " << format_matrix(Transform) << "\n";
}

void render_engine::RiPerspective(real FieldOfView)
{
	m_implementation->m_stream << ::indentation << "Perspective " << FieldOfView << "\n";
}

void render_engine::RiTranslate(real DX, real DY, real DZ)
{
	m_implementation->m_stream << ::indentation << "Translate " << DX << " " << DY << " " << DZ << "\n";
}

void render_engine::RiRotate(real Angle, real DX, real DY, real DZ)
{
	m_implementation->m_stream << ::indentation << "Rotate " << Angle << " " << DX << " " << DY << " " << DZ << "\n";
}

void render_engine::RiScale(real DX, real DY, real DZ)
{
	m_implementation->m_stream << ::indentation << "Scale " << DX << " " << DY << " " << DZ << "\n";
}

void render_engine::RiSkew(real Angle, real DX1, real DY1, real DZ1, real DX2, real DY2, real DZ2)
{
	m_implementation->m_stream << ::indentation << "Skew " << Angle << " " << DX1 << " " << DY1 << " " << DZ1 << " " << DX2 << " " << DY2 << " " << DZ2 << "\n";
}

void render_engine::RiDeformationV(const path& Path, const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_shaders.insert(Path);
	m_implementation->m_stream << ::indentation << "Deformation " << format_string(Name) << " " << Parameters << "\n";
}

void render_engine::RiDisplacementV(const path& Path, const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_shaders.insert(Path);
	m_implementation->m_stream << ::indentation << "Displacement " << format_string(Name) << " " << Parameters << "\n";
}

void render_engine::RiCoordinateSystem(const string& Space)
{
	m_implementation->m_stream << ::indentation << "CoordinateSystem " << format_string(Space) << "\n";
}

void render_engine::RiCoordSysTransform(const string& Space)
{
	m_implementation->m_stream << ::indentation << "CoordSysTransform " << format_string(Space) << "\n";
}

void render_engine::RiTransformBegin()
{
	m_implementation->m_stream << ::indentation << "TransformBegin" << "\n";
	::push_indent(m_implementation->m_stream);
}

void render_engine::RiTransformEnd()
{
	::pop_indent(m_implementation->m_stream);
	m_implementation->m_stream << ::indentation << "TransformEnd" << "\n";
}

void render_engine::RiAttributeV(const string& Name, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Attribute " << format_string(Name) << " " << Parameters << "\n";
}

void render_engine::RiPointsV(const unsigned_integer VertexCount, const parameter_list& Parameters)
{
	// Sanity checks ...
	return_if_fail(VertexCount);

	m_implementation->m_stream << ::indentation << "Points " << Parameters << "\n";
}

void render_engine::RiPolygonV(const unsigned_integer VertexCount, const parameter_list& Parameters)
{
	// Sanity checks ...
	return_if_fail(VertexCount);

	m_implementation->m_stream << ::indentation << "Polygon " << Parameters << "\n";
}

void render_engine::RiGeneralPolygonV(const unsigned_integers& VertexCounts, const parameter_list& Parameters)
{
	// Do some simple sanity checks ...
	return_if_fail(VertexCounts.size());

	m_implementation->m_stream << ::indentation << "GeneralPolygon " << format_array(VertexCounts.begin(), VertexCounts.end()) << " " << Parameters << "\n";
}

void render_engine::RiPointsPolygonsV(const unsigned_integers& VertexCounts, const unsigned_integers& VertexIDs, const parameter_list& Parameters)
{
	// Sanity checks ...
	return_if_fail(VertexCounts.size());
	return_if_fail(VertexIDs.size() == std::accumulate(VertexCounts.begin(), VertexCounts.end(), 0UL));

	m_implementation->m_stream << ::indentation << "PointsPolygons " << format_array(VertexCounts.begin(), VertexCounts.end()) << " " << format_array(VertexIDs.begin(), VertexIDs.end()) << " " << Parameters << "\n";
}

void render_engine::RiPointsGeneralPolygonsV(const unsigned_integers& LoopCounts, const unsigned_integers& VertexCounts, const unsigned_integers& VertexIDs, const parameter_list& Parameters)
{
	// Sanity checks ...
	return_if_fail(LoopCounts.size());
	return_if_fail(VertexCounts.size() == std::accumulate(LoopCounts.begin(), LoopCounts.end(), 0UL));
	return_if_fail(VertexIDs.size() == std::accumulate(VertexCounts.begin(), VertexCounts.end(), 0UL));

	m_implementation->m_stream << ::indentation << "PointsGeneralPolygons " << format_array(LoopCounts.begin(), LoopCounts.end()) << " " << format_array(VertexCounts.begin(), VertexCounts.end()) << " " << format_array(VertexIDs.begin(), VertexIDs.end()) << " " << Parameters << "\n";
}

void render_engine::RiBasis(const matrix& UBasis, const unsigned_integer UStep, const matrix& VBasis, const unsigned_integer VStep)
{
	m_implementation->m_stream << ::indentation << "Basis " << format_matrix(UBasis) << " " << UStep << " " << format_matrix(VBasis) << " " << VStep << "\n";
}

void render_engine::RiBasis(const string& UBasis, const unsigned_integer UStep, const string& VBasis, const unsigned_integer VStep)
{
	m_implementation->m_stream << ::indentation << "Basis " << format_string(UBasis) << " " << UStep << " " << format_string(VBasis) << " " << VStep << "\n";
}

void render_engine::RiPatchV(const string& Type, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Patch " << format_string(Type) << " " << Parameters << "\n";
}

void render_engine::RiPatchMeshV(const string& Type, const unsigned_integer UCount, const string& UWrap, const unsigned_integer VCount, const string& VWrap, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "PatchMesh " << format_string(Type) << " " << UCount << " " << format_string(UWrap) << " " << VCount << " " << format_string(VWrap) << " " << Parameters << "\n";
}

void render_engine::RiNuPatchV(const unsigned_integer UCount, const unsigned_integer UOrder, const reals& UKnot, const real UMin, const real UMax, const unsigned_integer VCount, const unsigned_integer VOrder, const reals& VKnot, const real VMin, const real VMax, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "NuPatch " << UCount << " " << UOrder << " " << format_array(UKnot.begin(), UKnot.end()) << " " << UMin << " " << UMax << " " << VCount << " " << VOrder << " " << format_array(VKnot.begin(), VKnot.end()) << " " << VMin << " " << VMax << " " << Parameters << "\n";
}

void render_engine::RiTrimCurve(const unsigned_integer LoopCount, const unsigned_integers& CurveCounts, const unsigned_integers& Orders, const reals& Knots, const reals& Minimums, const reals& Maximums, const unsigned_integers& KnotCounts, const reals& U, const reals& V, const reals& W)
{
	m_implementation->m_stream << ::indentation << "TrimCurve " << LoopCount << " " << format_array(CurveCounts.begin(), CurveCounts.end()) << " " << format_array(Orders.begin(), Orders.end()) << " " << format_array(Knots.begin(), Knots.end()) << " " << format_array(Minimums.begin(), Minimums.end()) << " " << format_array(Maximums.begin(), Maximums.end()) << " " << format_array(KnotCounts.begin(), KnotCounts.end()) << " " << format_array(U.begin(), U.end()) << " " << format_array(V.begin(), V.end()) << " " << format_array(W.begin(), W.end()) << "\n";
}

void render_engine::RiSphereV(real Radius, real ZMin, real ZMax, real ThetaMax, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Sphere " << Radius << " " << ZMin << " " << ZMax << " " << ThetaMax << " " << Parameters << "\n";
}

void render_engine::RiConeV(real Height, real Radius, real ThetaMax, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Cone " << Height << " " << Radius << " " << ThetaMax << " " << Parameters << "\n";
}

void render_engine::RiCylinderV(real Radius, real ZMin, real ZMax, real ThetaMax, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Cylinder " << Radius << " " << ZMin << " " << ZMax << " " << ThetaMax << " " << Parameters << "\n";
}

void render_engine::RiHyperboloidV(const point& Point1, const point& Point2, real ThetaMax, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Hyperboloid " << Point1 << " " << Point2 << " " << ThetaMax << " " << Parameters << "\n";
}

void render_engine::RiParaboloidV(real RMax, real ZMin, real ZMax, real ThetaMax, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Paraboloid " << RMax << " " << ZMin << " " << ZMax << " " << ThetaMax << " " << Parameters << "\n";
}

void render_engine::RiDiskV(real Height, real Radius, real ThetaMax, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Disk " << Height << " " << Radius << " " << ThetaMax << " " << Parameters << "\n";
}

void render_engine::RiTorusV(real MajorRadius, real MinorRadius, real PhiMin, real PhiMax, real ThetaMax, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Torus " << MajorRadius << " " << MinorRadius << " " << PhiMin << " " << PhiMax << " " << ThetaMax << " " << Parameters << "\n";
}

void render_engine::RiCurvesV(const string& Type, const unsigned_integers& VertexCounts, const string& Wrap, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Curves " << format_string(Type) << " " << format_array(VertexCounts.begin(), VertexCounts.end()) << " " << format_string(Wrap) << " " << Parameters << "\n";
}

void render_engine::RiGeometryV(const string& Type, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Geometry " << format_string(Type) << " " << Parameters << "\n";
}

void render_engine::RiSolidBegin(const string& Type)
{
	m_implementation->m_stream << ::indentation << "SolidBegin " << format_string(Type) << "\n";
	::push_indent(m_implementation->m_stream);
}

void render_engine::RiSolidEnd()
{
	::pop_indent(m_implementation->m_stream);
	m_implementation->m_stream << ::indentation << "SolidEnd" << "\n";
}

const object_handle render_engine::RiObjectBegin()
{
	// Sanity checks ...
	if(m_implementation->m_object_block)
	{
		log() << error << "Cannot nest calls to RiObjectBegin()" << std::endl;
		return 0;
	}

	m_implementation->m_object_block = true;
	m_implementation->m_stream << ::indentation << "ObjectBegin " << ++m_implementation->m_object_handle << "\n";
	::push_indent(m_implementation->m_stream);
	return m_implementation->m_object_handle;
}

void render_engine::RiObjectEnd()
{
	::pop_indent(m_implementation->m_stream);
	m_implementation->m_stream << ::indentation << "ObjectEnd" << "\n";
	m_implementation->m_object_block = false;
}

void render_engine::RiObjectInstance(const object_handle Handle)
{
	m_implementation->m_stream << ::indentation << "ObjectInstance " << Handle << "\n";
}

void render_engine::RiMotionBeginV(const sample_times_t& Times)
{
	// Sanity checks ...
	if(m_implementation->m_motion_block)
	{
		log() << error << "Cannot nest calls to RiMotionBegin()" << std::endl;
		return;
	}

	m_implementation->m_motion_block = true;
	m_implementation->m_stream << ::indentation << "MotionBegin " << format_array(Times.begin(), Times.end()) << "\n";
	::push_indent(m_implementation->m_stream);
}

void render_engine::RiMotionEnd()
{
	::pop_indent(m_implementation->m_stream);
	m_implementation->m_stream << ::indentation << "MotionEnd" << "\n";
	m_implementation->m_motion_block = false;
}

void render_engine::RiErrorHandler(const string& Style)
{
	m_implementation->m_stream << ::indentation << "ErrorHandler " << format_string(Style) << "\n";
}

void render_engine::RiComment(const string& Comment)
{
	m_implementation->m_stream << ::indentation << "#" << Comment << "\n";
}

void render_engine::RiNewline()
{
	m_implementation->m_stream << "\n";
}

void render_engine::RiReadArchive(const path& Archive)
{
	m_implementation->m_stream << ::indentation << "ReadArchive " << format_string(Archive.native_file_string()) << "\n";
}

void render_engine::RiProcDelayedReadArchive(const path& Archive, const bound& Bound)
{
	m_implementation->m_stream << ::indentation << "Procedural " << format_string("DelayedReadArchive") << " [ " <<  format_string(Archive.native_file_string()) << " ] [ " << Bound.nx << " " << Bound.px << " " << Bound.ny << " " << Bound.py << " " << Bound.nz << " " << Bound.pz << " ]\n";
}

void render_engine::RiStructure(const string& Structure)
{
	m_implementation->m_stream << "##" << Structure << "\n";
}

void render_engine::RiSubdivisionMeshV(const string& Scheme, const unsigned_integers& VertexCounts, const unsigned_integers& VertexIDs, const strings& Tags, const unsigned_integers& ArgCounts, const integers& IntegerArgs, const reals& FloatArgs, const parameter_list& Parameters)
{
	// Sanity checks ...
	return_if_fail(VertexIDs.size() == std::accumulate(VertexCounts.begin(), VertexCounts.end(), 0UL));

	m_implementation->m_stream << ::indentation << "SubdivisionMesh " << format_string(Scheme) << " " << format_array(VertexCounts.begin(), VertexCounts.end()) << " " << format_array(VertexIDs.begin(), VertexIDs.end()) << " " << format_array(Tags.begin(), Tags.end()) << " " << format_array(ArgCounts.begin(), ArgCounts.end()) << " " << format_array(IntegerArgs.begin(), IntegerArgs.end()) << " " << format_array(FloatArgs.begin(), FloatArgs.end()) << " " << Parameters << "\n";
}

void render_engine::RiMakeCubeFaceEnvironmentV(const string& px, const string& nx, const string& py, const string& ny, const string& pz, const string& nz, const string& texturename, const real fov, const string& swrap, const string& twrap, const string& filterfunc, const real swidth, const real twidth, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "MakeCubeFaceEnvironment " << format_string(px) << " " << format_string(nx) << " " << format_string(py) << " " << format_string(ny) << " " << format_string(pz) << " " << format_string(nz) << " " << format_string(texturename) << " ";
	m_implementation->m_stream << fov << " " << format_string(swrap) << " " << format_string(twrap) << " " << format_string(filterfunc) << " " << swidth << " " << twidth << " " << Parameters;
}

void render_engine::RiMakeLatLongEnvironmentV(const string& picturename, const string& texturename, const string& filterfunc, const real swidth, const real twidth, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "MakeLatLongEnvironment " << format_string(picturename) << " " << format_string(texturename) << " " << format_string(filterfunc) << " " << swidth << " " << twidth << " " << Parameters;
}

void render_engine::RiMakeShadowV(const string& picturename, const string& texturename, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "MakeShadow " << format_string(picturename) << " " << format_string(texturename) << " " << Parameters;
}

void render_engine::RiMakeTextureV(const string& picturename, const string& texturename, const string& swrap, const string& twrap, const string& filterfunc, const real swidth, const real twidth, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "MakeTexture " << format_string(picturename) << " " << format_string(texturename) << " " << format_string(swrap) << " " << format_string(twrap) << " " << format_string(filterfunc) << " " << swidth << " " << twidth << " " << Parameters;
}

void render_engine::RiBlobbyV(const unsigned_integer NLeaf, const unsigned_integers& Codes, const reals& Floats, const strings& Strings, const parameter_list& Parameters)
{
	m_implementation->m_stream << ::indentation << "Blobby " << NLeaf << " " << format_array(Codes.begin(), Codes.end()) << " " << format_array(Floats.begin(), Floats.end()) << " " << format_array(Strings.begin(), Strings.end()) << " " << Parameters << "\n";
}

void render_engine::RiShaderLayerV(const std::string& type, const path& Path, const std::string& name, const std::string& layername, const parameter_list& Parameters)
{
	m_implementation->m_shaders.insert(Path);
	m_implementation->m_stream << ::indentation << "ShaderLayer " << format_string(type) << " " << format_string(name) << " " << format_string(layername) << " " << Parameters << "\n";
}

void render_engine::RiConnectShaderLayers(const std::string& type, const std::string& layer1, const std::string& variable1, const std::string& layer2, const std::string& variable2)
{
	m_implementation->m_stream << ::indentation << "ConnectShaderLayers " << format_string(type) << " " << format_string(layer1) << " " << format_string(variable1) << " " << format_string(layer2) << " " << format_string(variable2) << "\n";
}

} // namespace ri

} // namespace k3d

