// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/classes.h>
#include <k3dsdk/bounded.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/imesh_sink.h>
#include <k3dsdk/imesh_source.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/parentable.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/sgi_tesselator.h>
#include <k3dsdk/snappable.h>
#include <k3dsdk/snap_source.h>
#include <k3dsdk/snap_target.h>
#include <k3dsdk/transformable.h>

#include <subdivision_surface/k3d_sds_binding.h>
#include <subdivision_surface/sds_crease.h>
#include <surface_polygonizer/blobby_polygonizer.h>

#include "helpers.h"

#include <limits>
#include <map>
#include <stack>

/** \note Un-comment the following to test whether performance is application-bound or transformation-bound (assumes you've already verified that it isn't fill-bound, by resizing the viewport)
#define glVertex2d(x,y) glNormal3d(x,y,0)
#define glVertex2f(x,y) glNormal3f(x,y,0)
#define glVertex2i(x,y) glNormal3i(x,y,0)
#define glVertex2s(x,y) glNormal3s(x,y,0)
#define glVertex3d(x,y,z) glNormal3d(x,y,z)
#define glVertex3f(x,y,z) glNormal3f(x,y,z)
#define glVertex3i(x,y,z) glNormal3i(x,y,z)
#define glVertex3s(x,y,z) glNormal3s(x,y,z)
#define glVertex4d(x,y,z,w) glNormal3d(x,y,z)
#define glVertex4f(x,y,z,w) glNormal3f(x,y,z)
#define glVertex4i(x,y,z,w) glNormal3i(x,y,z)
#define glVertex4s(x,y,z,w) glNormal3s(x,y,z)
#define glVertex2dv(v) glNormal3d(v[0],v[1])
#define glVertex2fv(v) glNormal3f(v[0],v[1])
#define glVertex2iv(v) glNormal3i(v[0],v[1])
#define glVertex2sv(v) glNormal3s(v[0],v[1])
#define glVertex3dv(v) glNormal3dv(v)
#define glVertex3fv(v) glNormal3fv(v)
#define glVertex3iv(v) glNormal3iv(v)
#define glVertex3sv(v) glNormal3sv(v)
#define glVertex4dv(v) glNormal3dv(v)
#define glVertex4fv(v) glNormal3fv(v)
#define glVertex4iv(v) glNormal3iv(v)
#define glVertex4sv(v) glNormal3sv(v)
*/

namespace libk3dmesh
{

namespace detail
{

struct draw_all
{
	template<typename T>
	bool operator()(T*)
	{
		return true;
	}
};

struct draw_selected
{
	template<typename T>
	bool operator()(T* RHS)
	{
		return RHS && 0.0 != RHS->selection_weight;
	}
};

struct draw_unselected
{
	template<typename T>
	bool operator()(T* RHS)
	{
		return RHS && 0.0 == RHS->selection_weight;
	}
};

/** \todo Get rid of this */
template<typename base_t>
class mesh_modifier :
	public base_t,
	public k3d::imesh_sink,
	public k3d::imesh_source
{
public:
	mesh_modifier(k3d::idocument& Document) :
		base_t(Document),
		m_input_mesh(init_owner(*this) + init_name("input_mesh") + init_label(_("Input Mesh")) + init_description(_("Input mesh")) + init_value<k3d::mesh*>(0)),
		m_output_mesh(init_owner(*this) + init_name("output_mesh") + init_label(_("Output Mesh")) + init_description(_("Output mesh")) + init_slot(sigc::mem_fun(*this, &mesh_modifier<base_t>::create_mesh)))
	{
		m_input_mesh.changed_signal().connect(make_reset_mesh_slot());
	}

	k3d::iproperty& mesh_source_output()
	{
		return m_output_mesh;
	}

	k3d::iproperty& mesh_sink_input()
	{
		return m_input_mesh;
	}

	sigc::slot1<void, k3d::iunknown*> make_reset_mesh_slot()
	{
		return m_output_mesh.make_reset_slot();
	}

	sigc::slot1<void, k3d::iunknown*> make_update_mesh_slot()
	{
		return sigc::mem_fun(*this, &mesh_modifier<base_t>::update_mesh);
	}

protected:
	k3d_data(k3d::mesh*, immutable_name, change_signal, no_undo, local_storage, no_constraint, read_only_property, no_serialization) m_input_mesh;
	k3d_data(k3d::mesh*, immutable_name, change_signal, no_undo, demand_storage, no_constraint, read_only_property, no_serialization) m_output_mesh;

private:
	void create_mesh(k3d::mesh& Mesh)
	{
		if(const k3d::mesh* const input_mesh = m_input_mesh.value())
		{
			on_create_mesh(*input_mesh, Mesh);
			on_update_mesh(*input_mesh, Mesh);
		}
	}

	void update_mesh(k3d::iunknown* const Hint)
	{
		if(const k3d::mesh* const input_mesh = m_input_mesh.value())
		{
			if(k3d::mesh* const output_mesh = m_output_mesh.internal_value())
			{
				on_update_mesh(*input_mesh, *output_mesh);
				m_output_mesh.changed_signal().emit(Hint);
			}
		}
	}

	virtual void on_create_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh) = 0;
	virtual void on_update_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh) = 0;
};

/// Checks whether face forms a convex polygon
bool is_convex(k3d::face& Face)
{
	k3d::split_edge* edge = Face.first_edge;

	// Triangles are always convex
	if(edge->face_clockwise->face_clockwise->face_clockwise == edge)
		return true;

	k3d::vector3 first_normal;
	bool first = true;

	do
	{
		k3d::split_edge* next_edge = edge->face_clockwise;
		const k3d::vector3 e1 = k3d::to_vector(edge->vertex->position - next_edge->vertex->position);
		const k3d::vector3 e2 = k3d::to_vector(next_edge->face_clockwise->vertex->position - next_edge->vertex->position);
		const k3d::vector3 normal = e1 ^ e2;

		edge = next_edge;

		// Skip parallel edges
		if(normal.length2() < 1e-6)
			continue;

		if(first)
		{
			first_normal = normal;
			first = false;
		}
		else
		{
			if(first_normal * normal < 0)
				return false;
		}
	}
	while(edge != Face.first_edge);

	return true;
}

/// Functor object for applying a transformation to a collection of points
struct transform_points
{
	transform_points(const k3d::matrix4& Matrix) :
		matrix(Matrix)
	{
	}

	void operator()(k3d::point* const Point)
	{
		Point->position = matrix * Point->position;
	}

	const k3d::matrix4 matrix;
};

/////////////////////////////////////////////////////////////////////////////
// draw_blobby

/// Draws the components that make up a blobby (Visitor Design Pattern)
class draw_blobby :
	public k3d::blobby::visitor
{
public:
	void visit_constant(k3d::blobby::constant&)
	{
	}

	void visit_ellipsoid(k3d::blobby::ellipsoid& Ellipsoid)
	{
		glBegin(GL_POINTS);
			glVertex3dv(Ellipsoid.origin->position.n);
		glEnd();
	}

	void visit_segment(k3d::blobby::segment& Segment)
	{
		glLineWidth(Segment.radius);
		glBegin(GL_LINES);
			glVertex3dv(Segment.start->position.n);
			glVertex3dv(Segment.end->position.n);
		glEnd();
	}

	void visit_subtract(k3d::blobby::subtract& Subtract)
	{
		Subtract.subtrahend->accept(*this);
		Subtract.minuend->accept(*this);
	}

	void visit_divide(k3d::blobby::divide& Divide)
	{
		Divide.dividend->accept(*this);
		Divide.divisor->accept(*this);
	}

	void visit_add(k3d::blobby::add& Add)
	{
		Add.operands_accept(*this);
	}

	void visit_multiply(k3d::blobby::multiply& Multiply)
	{
		Multiply.operands_accept(*this);
	}

	void visit_min(k3d::blobby::min& Min)
	{
		Min.operands_accept(*this);
	}

	void visit_max(k3d::blobby::max& Max)
	{
		Max.operands_accept(*this);
	}
};

/////////////////////////////////////////////////////////////////////////////
// select_blobby

/// Draws the components that make up a blobby (Visitor Design Pattern)
class select_blobby :
	public k3d::blobby::visitor
{
public:
	void visit_constant(k3d::blobby::constant&)
	{
	}

	void visit_ellipsoid(k3d::blobby::ellipsoid& Ellipsoid)
	{
//		k3d::gl::push_name(&Ellipsoid);
		glBegin(GL_POINTS);
			glVertex3dv(Ellipsoid.origin->position.n);
		glEnd();
//		k3d::gl::pop_name();
	}

	void visit_segment(k3d::blobby::segment& Segment)
	{
//		k3d::gl::push_name(&Segment);
		glLineWidth(Segment.radius);
		glBegin(GL_LINES);
			glVertex3dv(Segment.start->position.n);
			glVertex3dv(Segment.end->position.n);
		glEnd();
//		k3d::gl::pop_name();
	}

	void visit_subtract(k3d::blobby::subtract& Subtract)
	{
		Subtract.subtrahend->accept(*this);
		Subtract.minuend->accept(*this);
	}

	void visit_divide(k3d::blobby::divide& Divide)
	{
		Divide.dividend->accept(*this);
		Divide.divisor->accept(*this);
	}

	void visit_add(k3d::blobby::add& Add)
	{
		Add.operands_accept(*this);
	}

	void visit_multiply(k3d::blobby::multiply& Multiply)
	{
		Multiply.operands_accept(*this);
	}

	void visit_min(k3d::blobby::min& Min)
	{
		Min.operands_accept(*this);
	}

	void visit_max(k3d::blobby::max& Max)
	{
		Max.operands_accept(*this);
	}
};

typedef std::vector<k3d::point3> vertices_t;
typedef std::vector<unsigned long> polygon_t;
typedef std::vector<polygon_t> polygons_t;

/////////////////////////////////////////////////////////////////////////////
// OpenGL tesselator callbacks

class glu_tesselator
{
public:
	glu_tesselator() :
		tesselator(sgiNewTess())
	{
		sgiTessCallback(tesselator, GLU_TESS_BEGIN_DATA, reinterpret_cast<_GLUfuncptr>(&raw_begin));
		sgiTessCallback(tesselator, GLU_TESS_VERTEX_DATA, reinterpret_cast<_GLUfuncptr>(&raw_vertex));
		sgiTessCallback(tesselator, GLU_TESS_COMBINE_DATA, reinterpret_cast<_GLUfuncptr>(&raw_combine));
		sgiTessCallback(tesselator, GLU_TESS_END_DATA, reinterpret_cast<_GLUfuncptr>(&raw_end));
		sgiTessCallback(tesselator, GLU_TESS_ERROR_DATA, reinterpret_cast<_GLUfuncptr>(&raw_error));
	}

	void tesselate(k3d::face& Face)
	{
		sgiTessBeginPolygon(tesselator, this);

		sgiTessBeginContour(tesselator);
		for(k3d::split_edge* edge = Face.first_edge; edge && edge->face_clockwise; edge = edge->face_clockwise)
		{
			sgiTessVertex(tesselator, edge->vertex->position.data(), edge->vertex);
			if(edge->face_clockwise == Face.first_edge)
				break;
		}
		sgiTessEndContour(tesselator);

		for(k3d::face::holes_t::const_iterator hole = Face.holes.begin(); hole != Face.holes.end(); ++hole)
		{
			sgiTessBeginContour(tesselator);
			for(k3d::split_edge* edge = *hole; edge && edge->face_clockwise; edge = edge->face_clockwise)
			{
				sgiTessVertex(tesselator, edge->vertex->position.data(), edge->vertex);
				if(edge->face_clockwise == *hole)
					break;
			}
			sgiTessEndContour(tesselator);
		}

		sgiTessEndPolygon(tesselator);
	}

	~glu_tesselator()
	{
		sgiDeleteTess(tesselator);

		std::for_each(new_points.begin(), new_points.end(), k3d::delete_object());
	}

private:
	void begin(GLenum Mode)
	{
		glBegin(Mode);
	}

	void vertex(void* VertexData)
	{
		k3d::point* point = reinterpret_cast<k3d::point*>(VertexData);
		k3d::gl::vertex3d(point->position);
	}

	void combine(GLdouble Coords[3], void* VertexData[4], GLfloat Weight[4], void** OutputData)
	{
		k3d::point* const new_point = new k3d::point(k3d::point3(Coords[0], Coords[1], Coords[2]));
		new_points.push_back(new_point);

		*OutputData = new_point;
	}

	void end()
	{
		glEnd();
	}

	void error(GLenum ErrorNumber)
	{
		const GLubyte* estring = gluErrorString(ErrorNumber);
		if(estring)
			k3d::log() << k3d::error << k3d_file_reference << ": Tessellation Error: " << estring << std::endl;
	}

	static glu_tesselator* instance(void* UserData)
	{
		return reinterpret_cast<glu_tesselator*>(UserData);
	}

	static void raw_begin(GLenum Mode, void* UserData)
	{
		instance(UserData)->begin(Mode);
	}

	static void raw_vertex(void* VertexData, void* UserData)
	{
		instance(UserData)->vertex(VertexData);
	}

	static void raw_combine(GLdouble Coords[3], void* VertexData[4], GLfloat Weight[4], void** OutputData, void* UserData)
	{
		instance(UserData)->combine(Coords, VertexData, Weight, OutputData);
	}

	static void raw_end(void* UserData)
	{
		instance(UserData)->end();
	}

	static void raw_error(GLenum ErrorNumber, void* UserData)
	{
		instance(UserData)->error(ErrorNumber);
	}

	SGItesselator* const tesselator;

	std::vector<k3d::point*> new_points;
};

} // namespace detail

namespace detail
{

class same_type
{
public:
	same_type(const std::type_info& Type) :
		m_type(Type)
	{
	}

	bool operator()(const boost::any& Value)
	{
		return Value.type() == m_type;
	}

private:
	const std::type_info& m_type;
};

typedef std::vector<boost::any> values_t;

typedef std::map<std::string, values_t> grouped_parameters_t;

template<typename data_t, typename container_t>
const container_t build_array(const values_t& Values)
{
	container_t result;

	for(values_t::const_iterator value = Values.begin(); value != Values.end(); ++value)
		result.push_back(boost::any_cast<data_t>(*value));

	return result;
}

k3d::ri::parameter_list build_parameters(const grouped_parameters_t& Parameters, const k3d::ri::storage_class_t StorageClass)
{
	k3d::ri::parameter_list results;

	// For each group of values ...
	for(grouped_parameters_t::const_iterator group = Parameters.begin(); group != Parameters.end(); ++group)
	{
		// Get some information about the group ...
		const std::string& name = group->first;
		const values_t& values = group->second;
		const std::type_info& type = values.front().type();

		// Check to see that all values have the same type; if not, skip the group ...
		if(values.size() != static_cast<size_t>(std::count_if(values.begin(), values.end(), same_type(type))))
		{
			k3d::log() << error << "Parameter [" << name << "] contains multiple types and will be ignored" << std::endl;
			continue;
		}

		// OK, everything looks good so let's turn the group into a set of parameters ...
		if(typeid(k3d::ri::integer) == type)
		{
			results.push_back(k3d::ri::parameter(name, StorageClass, build_array<k3d::ri::integer, k3d::ri::integers>(values)));
		}
		else if(typeid(k3d::ri::real) == type)
		{
			results.push_back(k3d::ri::parameter(name, StorageClass, build_array<k3d::ri::real, k3d::ri::reals>(values)));
		}
		else if(typeid(k3d::ri::string) == type)
		{
			results.push_back(k3d::ri::parameter(name, StorageClass, build_array<k3d::ri::string, k3d::ri::strings>(values)));
		}
		else if(typeid(k3d::ri::point) == type)
		{
			results.push_back(k3d::ri::parameter(name, StorageClass, build_array<k3d::ri::point, k3d::ri::points>(values)));
		}
		else if(typeid(k3d::ri::vector) == type)
		{
			results.push_back(k3d::ri::parameter(name, StorageClass, build_array<k3d::ri::vector, k3d::ri::vectors>(values)));
		}
		else if(typeid(k3d::ri::normal) == type)
		{
			results.push_back(k3d::ri::parameter(name, StorageClass, build_array<k3d::ri::normal, k3d::ri::normals>(values)));
		}
		else if(typeid(k3d::ri::color) == type)
		{
			results.push_back(k3d::ri::parameter(name, StorageClass, build_array<k3d::ri::color, k3d::ri::colors>(values)));
		}
		else if(typeid(k3d::ri::hpoint) == type)
		{
			results.push_back(k3d::ri::parameter(name, StorageClass, build_array<k3d::ri::hpoint, k3d::ri::hpoints>(values)));
		}
		else if(typeid(k3d::ri::matrix) == type)
		{
			results.push_back(k3d::ri::parameter(name, StorageClass, build_array<k3d::ri::matrix, k3d::ri::matrices>(values)));
		}
		else
		{
			k3d::log() << error << "Cannot deduce parameter type for [" << name << "]" << std::endl;
		}
	}

	return results;
}

void build_tags(const k3d::parameters_t::const_iterator Begin, const k3d::parameters_t::const_iterator& End, k3d::ri::strings& Tags, k3d::ri::unsigned_integers& TagCounts, k3d::ri::integers& TagIntegers, k3d::ri::reals& TagReals)
{

	for(k3d::parameters_t::const_iterator tag = Begin; tag != End; ++tag)
	{
		if(tag->first == "interpolateboundary" && (tag->second.type() == typeid(bool)))
		{
			if(boost::any_cast<bool>(tag->second))
			{
				Tags.push_back("interpolateboundary");
				TagCounts.push_back(0);
				TagCounts.push_back(0);
				continue;
			}
		}

		k3d::log() << warning << "Unknown or incorrectly typed tag [" << tag->first << "] will be ignored" << std::endl;
	}
}

void build_tags(const k3d::polyhedron::faces_t::const_iterator Begin, const k3d::polyhedron::faces_t::const_iterator& End, std::map<k3d::face*, k3d::ri::unsigned_integer>& FaceMap, k3d::ri::strings& Tags, k3d::ri::unsigned_integers& TagCounts, k3d::ri::integers& TagIntegers, k3d::ri::reals& TagReals, k3d::imaterial* const Material)
{
	for(k3d::polyhedron::faces_t::const_iterator face = Begin; face != End; ++face)
	{
		bool hole = false;
		for(k3d::parameters_t::const_iterator tag = (*face)->tags.begin(); tag != (*face)->tags.end(); ++tag)
		{
			if(tag->first == "hole" && (tag->second.type() == typeid(bool)))
			{
				hole = boost::any_cast<bool>(tag->second);
				continue;
			}

			k3d::log() << warning << "Unknown or incorrectly typed face tag [" << tag->first << "] will be ignored" << std::endl;
		}
		if((*face)->material != Material)
			hole = true;

		if(hole)
		{
			Tags.push_back("hole");
			TagCounts.push_back(1);
			TagCounts.push_back(0);
			TagIntegers.push_back(FaceMap[*face]);
		}
	}
}

typedef std::vector<k3d::split_edge*> edges_t;
void build_tags(const edges_t::const_iterator Begin, const edges_t::const_iterator& End, std::map<k3d::point*, k3d::ri::unsigned_integer>& PointMap, k3d::ri::strings& Tags, k3d::ri::unsigned_integers& TagCounts, k3d::ri::integers& TagIntegers, k3d::ri::reals& TagReals)
{
	// First, get the set of all "joined" edges (i.e. eliminate companions)
	typedef std::set<k3d::split_edge*> joined_edges_t;
	joined_edges_t joined_edges;
	for(edges_t::const_iterator e = Begin; e != End; ++e)
		joined_edges.insert(std::max((*e), (*e)->companion));
	joined_edges.erase(0);

	for(joined_edges_t::const_iterator e = joined_edges.begin(); e != joined_edges.end(); ++e)
	{
		k3d::split_edge& edge = **e;

		for(k3d::parameters_t::const_iterator tag = edge.tags.begin(); tag != edge.tags.end(); ++tag)
		{
			if(tag->first == "crease" && (tag->second.type() == typeid(k3d::ri::real)) && edge.vertex && edge.face_clockwise && edge.face_clockwise->vertex)
			{
				Tags.push_back("crease");
				TagCounts.push_back(2);
				TagCounts.push_back(1);
				TagIntegers.push_back(PointMap[edge.vertex]);
				TagIntegers.push_back(PointMap[edge.face_clockwise->vertex]);
				TagReals.push_back(boost::any_cast<k3d::ri::real>(tag->second));
				continue;
			}

			k3d::log() << warning << "Unknown or incorrectly typed edge tag [" << tag->first << "] will be ignored" << std::endl;
		}
	}
}

void build_tags(const std::vector<k3d::point*>::const_iterator Begin, const std::vector<k3d::point*>::const_iterator End, std::map<k3d::point*, k3d::ri::unsigned_integer>& PointMap, k3d::ri::strings& Tags, k3d::ri::unsigned_integers& TagCounts, k3d::ri::integers& TagIntegers, k3d::ri::reals& TagReals)
{
	for(std::vector<k3d::point*>::const_iterator point = Begin; point != End; ++point)
	{
		for(k3d::parameters_t::const_iterator tag = (*point)->tags.begin(); tag != (*point)->tags.end(); ++tag)
		{
			if(tag->first == "corner" && (tag->second.type() == typeid(k3d::ri::real)))
			{
				Tags.push_back("corner");
				TagCounts.push_back(1);
				TagCounts.push_back(1);
				TagIntegers.push_back(PointMap[*point]);
				TagReals.push_back(boost::any_cast<k3d::ri::real>(tag->second));
				continue;
			}

			k3d::log() << warning << "Unknown or incorrectly-typed point tag [" << tag->first << "] will be ignored" << std::endl;
		}
	}
}

k3d::ri::parameter_list build_parameters(const k3d::parameters_t::const_iterator& Begin, const k3d::parameters_t::const_iterator& End, const k3d::ri::storage_class_t StorageClass)
{
	grouped_parameters_t grouped_parameters;

	for(k3d::parameters_t::const_iterator parameter = Begin; parameter != End; ++parameter)
		grouped_parameters[parameter->first].push_back(parameter->second);

	return build_parameters(grouped_parameters, StorageClass);
}

k3d::ri::parameter_list build_parameters(const boost::array<k3d::parameters_t, 4>::const_iterator& Begin, const boost::array<k3d::parameters_t, 4>::const_iterator& End, const k3d::ri::storage_class_t StorageClass)
{
	grouped_parameters_t grouped_parameters;

	for(boost::array<k3d::parameters_t, 4>::const_iterator parameters = Begin; parameters != End; ++parameters)
	{
		for(k3d::parameters_t::const_iterator parameter = parameters->begin(); parameter != parameters->end(); ++parameter)
			grouped_parameters[parameter->first].push_back(parameter->second);
	}

	return build_parameters(grouped_parameters, StorageClass);
}

k3d::ri::parameter_list build_parameters(k3d::point** Begin, k3d::point** End, const k3d::ri::storage_class_t StorageClass)
{
	// Sanity check ...
	assert(k3d::ri::VERTEX == StorageClass);

	grouped_parameters_t grouped_parameters;

	for(k3d::point** point = Begin; point != End; ++point)
	{
		for(k3d::parameters_t::const_iterator parameter = (*point)->vertex_data.begin(); parameter != (*point)->vertex_data.end(); ++parameter)
			grouped_parameters[parameter->first].push_back(parameter->second);
	}

	return build_parameters(grouped_parameters, StorageClass);
}

k3d::ri::parameter_list build_parameters(const std::vector<k3d::point*>::const_iterator Begin, const std::vector<k3d::point*>::const_iterator End, const k3d::ri::storage_class_t StorageClass)
{
	// Sanity check ...
	assert(k3d::ri::VERTEX == StorageClass);

	grouped_parameters_t grouped_parameters;

	for(std::vector<k3d::point*>::const_iterator point = Begin; point != End; ++point)
	{
		for(k3d::parameters_t::const_iterator parameter = (*point)->vertex_data.begin(); parameter != (*point)->vertex_data.end(); ++parameter)
			grouped_parameters[parameter->first].push_back(parameter->second);
	}

	return build_parameters(grouped_parameters, StorageClass);
}

k3d::ri::parameter_list build_parameters(const std::vector<k3d::split_edge*>::const_iterator Begin, const std::vector<k3d::split_edge*>::const_iterator End, const k3d::ri::storage_class_t StorageClass)
{
	// Sanity check ...
	assert(k3d::ri::FACEVARYING == StorageClass);

	grouped_parameters_t grouped_parameters;

	for(std::vector<k3d::split_edge*>::const_iterator edge = Begin; edge != End; ++edge)
	{
		for(k3d::parameters_t::const_iterator parameter = (*edge)->facevarying_data.begin(); parameter != (*edge)->facevarying_data.end(); ++parameter)
			grouped_parameters[parameter->first].push_back(parameter->second);
	}

	return build_parameters(grouped_parameters, StorageClass);
}

k3d::ri::parameter_list build_parameters(const k3d::polyhedron::faces_t::const_iterator Begin, const k3d::polyhedron::faces_t::const_iterator End, const k3d::ri::storage_class_t StorageClass)
{
	// Sanity check ...
	assert(k3d::ri::UNIFORM == StorageClass);

	grouped_parameters_t grouped_parameters;

	for(k3d::polyhedron::faces_t::const_iterator face = Begin; face != End; ++face)
	{
		for(k3d::parameters_t::const_iterator parameter = (*face)->uniform_data.begin(); parameter != (*face)->uniform_data.end(); ++parameter)
			grouped_parameters[parameter->first].push_back(parameter->second);
	}

	return build_parameters(grouped_parameters, StorageClass);
}

k3d::ri::parameter_list build_parameters(const k3d::linear_curve_group::curves_t::const_iterator Begin, const k3d::linear_curve_group::curves_t::const_iterator End, const k3d::ri::storage_class_t StorageClass)
{
	grouped_parameters_t grouped_parameters;

	if(k3d::ri::UNIFORM == StorageClass)
	{
		for(k3d::linear_curve_group::curves_t::const_iterator curve = Begin; curve != End; ++curve)
		{
			for(k3d::parameters_t::const_iterator parameter = (*curve)->uniform_data.begin(); parameter != (*curve)->uniform_data.end(); ++parameter)
				grouped_parameters[parameter->first].push_back(parameter->second);
		}
	}
	else if(k3d::ri::VARYING == StorageClass)
	{
		for(k3d::linear_curve_group::curves_t::const_iterator curve = Begin; curve != End; ++curve)
		{
			for(k3d::linear_curve::varying_t::const_iterator varying_data = (*curve)->varying_data.begin(); varying_data != (*curve)->varying_data.end(); ++varying_data)
			{
				for(k3d::parameters_t::const_iterator parameter = varying_data->begin(); parameter != varying_data->end(); ++parameter)
					grouped_parameters[parameter->first].push_back(parameter->second);
			}
		}
	}

	return build_parameters(grouped_parameters, StorageClass);
}

k3d::ri::parameter_list build_parameters(const k3d::cubic_curve_group::curves_t::const_iterator Begin, const k3d::cubic_curve_group::curves_t::const_iterator End, const k3d::ri::storage_class_t StorageClass)
{
	grouped_parameters_t grouped_parameters;

	if(k3d::ri::UNIFORM == StorageClass)
	{
		for(k3d::cubic_curve_group::curves_t::const_iterator curve = Begin; curve != End; ++curve)
		{
			for(k3d::parameters_t::const_iterator parameter = (*curve)->uniform_data.begin(); parameter != (*curve)->uniform_data.end(); ++parameter)
				grouped_parameters[parameter->first].push_back(parameter->second);
		}
	}
	else if(k3d::ri::VARYING == StorageClass)
	{
		for(k3d::cubic_curve_group::curves_t::const_iterator curve = Begin; curve != End; ++curve)
		{
			for(k3d::cubic_curve::varying_t::const_iterator varying_data = (*curve)->varying_data.begin(); varying_data != (*curve)->varying_data.end(); ++varying_data)
			{
				for(k3d::parameters_t::const_iterator parameter = varying_data->begin(); parameter != varying_data->end(); ++parameter)
					grouped_parameters[parameter->first].push_back(parameter->second);
			}
		}
	}

	return build_parameters(grouped_parameters, StorageClass);
}

// RiBlobby

void push_matrix(const k3d::matrix4& Matrix, k3d::ri::reals& Floats)
{
	for(int i = 0; i != 4; ++i)
	{
		for(int j = 0; j != 4; ++j)
		{
			Floats.push_back(Matrix[i][j]);
		}
	}
}

void push_point3(const k3d::point3& Vector, k3d::ri::reals& Floats)
{
	Floats.push_back(Vector[0]);
	Floats.push_back(Vector[1]);
	Floats.push_back(Vector[2]);
}

/// Blobby virtual machine - builds the RiBlobby arrays
class blobby_vm :
	private k3d::blobby::visitor
{
public:
	blobby_vm(k3d::blobby& Blobby, k3d::ri::unsigned_integer& NLeaf, k3d::ri::unsigned_integers& Codes, k3d::ri::reals& Floats, k3d::ri::strings& Strings, grouped_parameters_t& Parameters) :
		nleaf(NLeaf),
		codes(Codes),
		floats(Floats),
		names(Strings),
		grouped_parameters(Parameters)
	{
		m_opcode_id = 0;
		Blobby.accept(*this);
	}

	virtual ~blobby_vm() {}

private:
	void visit_constant(k3d::blobby::constant& Constant)
	{
		codes.push_back(1000);
		codes.push_back(floats.size());
		floats.push_back(Constant.value);

		m_opcodes.push(m_opcode_id++);
		nleaf++;
	}

	void visit_ellipsoid(k3d::blobby::ellipsoid& Ellipsoid)
	{
		codes.push_back(1001);
		codes.push_back(floats.size());

		// Output matrix ...
		push_matrix(k3d::translation3D(Ellipsoid.origin->position) * Ellipsoid.transformation, floats);

		// Save parameters ...
		for(k3d::parameters_t::const_iterator parameter = Ellipsoid.vertex_data.begin(); parameter != Ellipsoid.vertex_data.end(); parameter++)
			grouped_parameters[parameter->first].push_back(parameter->second);

		m_opcodes.push(m_opcode_id++);
		nleaf++;
	}

	void visit_segment(k3d::blobby::segment& Segment)
	{
		codes.push_back(1002);
		codes.push_back(floats.size());

		// Output parameters ...
		push_point3(Segment.start->position, floats);
		push_point3(Segment.end->position, floats);
		floats.push_back(Segment.radius);
		push_matrix(Segment.transformation, floats);

		// Save extra parameters ...
		for(k3d::parameters_t::const_iterator parameter = Segment.vertex_data.begin(); parameter != Segment.vertex_data.end(); parameter++)
			grouped_parameters[parameter->first].push_back(parameter->second);

		m_opcodes.push(m_opcode_id++);
		nleaf++;
	}

	void visit_subtract(k3d::blobby::subtract& Subtract)
	{
		// Note - order matters, here !
		Subtract.subtrahend->accept(*this);
		Subtract.minuend->accept(*this);

		codes.push_back(4);
		k3d::ri::unsigned_integer opcode2 = m_opcodes.top();
		m_opcodes.pop();
		k3d::ri::unsigned_integer opcode1 = m_opcodes.top();
		m_opcodes.pop();

		codes.push_back(opcode1);
		codes.push_back(opcode2);

		m_opcodes.push(m_opcode_id++);
	}

	void visit_divide(k3d::blobby::divide& Divide)
	{
		// Note - order matters, here !
		Divide.dividend->accept(*this);
		Divide.divisor->accept(*this);

		codes.push_back(5);
		k3d::ri::unsigned_integer opcode2 = m_opcodes.top();
		m_opcodes.pop();
		k3d::ri::unsigned_integer opcode1 = m_opcodes.top();
		m_opcodes.pop();

		codes.push_back(opcode1);
		codes.push_back(opcode2);

		m_opcodes.push(m_opcode_id++);
	}

	void visit_add(k3d::blobby::add& Add)
	{
		Add.operands_accept(*this);

		codes.push_back(0);
		codes.push_back(Add.operands.size());

		for(unsigned long n = 0; n < Add.operands.size(); ++n)
		{
			codes.push_back(m_opcodes.top());
			m_opcodes.pop();
		}

		m_opcodes.push(m_opcode_id++);
	}

	void visit_multiply(k3d::blobby::multiply& Multiply)
	{
		Multiply.operands_accept(*this);

		codes.push_back(1);
		codes.push_back(Multiply.operands.size());

		for(unsigned long n = 0; n < Multiply.operands.size(); n++)
		{
			codes.push_back(m_opcodes.top());
			m_opcodes.pop();
		}

		m_opcodes.push(m_opcode_id++);
	}

	void visit_max(k3d::blobby::max& Max)
	{
		Max.operands_accept(*this);

		codes.push_back(2);
		codes.push_back(Max.operands.size());

		for(unsigned long n = 0; n < Max.operands.size(); n++)
		{
			codes.push_back(m_opcodes.top());
			m_opcodes.pop();
		}

		m_opcodes.push(m_opcode_id++);
	}

	void visit_min(k3d::blobby::min& Min)
	{
		Min.operands_accept(*this);

		codes.push_back(3);
		codes.push_back(Min.operands.size());

		for(unsigned long n = 0; n < Min.operands.size(); n++)
		{
			codes.push_back(m_opcodes.top());
			m_opcodes.pop();
		}

		m_opcodes.push(m_opcode_id++);
	}

	std::stack<unsigned long> m_opcodes;
	unsigned long m_opcode_id;

	k3d::ri::unsigned_integer& nleaf;
	k3d::ri::unsigned_integers& codes;
	k3d::ri::reals& floats;
	k3d::ri::strings& names;
	grouped_parameters_t& grouped_parameters;
};

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// mesh_instance

class mesh_instance :
	public k3d::snappable<k3d::bounded<k3d::gl::drawable<k3d::ri::renderable<k3d::mesh_selection_sink<detail::mesh_modifier<k3d::parentable<k3d::transformable<k3d::persistent<k3d::node> > > > > > > > >
{
	typedef k3d::snappable<k3d::bounded<k3d::gl::drawable<k3d::ri::renderable<k3d::mesh_selection_sink<detail::mesh_modifier<k3d::parentable<k3d::transformable<k3d::persistent<k3d::node> > > > > > > > > base;

public:
	mesh_instance(k3d::idocument& Document) :
		base(Document),
		m_display_lists(init_owner(*this) + init_name("display_lists") + init_label(_("Use OpenGL Display Lists")) + init_description(_("Use OpenGL display lists for performance (temporary)")) + init_value(true)),
		m_show_blobby_surface(init_owner(*this) + init_name("blobby_surface") + init_label(_("Show blobby surfaces")) + init_description(_("Show blobbies surfaces")) + init_value(true)),
		m_polyhedron_render_type(init_owner(*this) + init_name("polyhedron_render_type") + init_label(_("Polyhedron render type")) + init_description(_("Type is : default / polygons / Catmull-Clark")) + init_value(DEFAULT) + init_enumeration(polyhedron_render_values())),
		m_proxy_type(init_owner(*this) + init_name("proxy_type") + init_label(_("Preview Proxy")) + init_description(_("Preview mesh using a proxy for performance")) + init_value(PROXY_NONE) + init_enumeration(proxy_values())),
		m_preview_sds(init_owner(*this) + init_name("preview_sds") + init_label(_("Preview SDS")) + init_description(_("Show SDS Surfaces")) + init_value(true)),
		m_nurbs_sds(init_owner(*this) + init_name("nurbs_sds") + init_label(_("SDS NURBS preview")) + init_description(_("Experimental NURBS SDS Preview")) + init_value(false)),
		m_sds_crease(init_owner(*this) + init_name("sds_crease") + init_label(_("Show SDS creases")) + init_description(_("Show SDS creases")) + init_value(false)),
		m_sds_borders(init_owner(*this) + init_name("sds_borders") + init_label(_("Show SDS patch borders")) + init_description(_("Show SDS patch borders")) + init_value(true)),
		m_sds_level(init_owner(*this) + init_name("sds_level") + init_label(_("SDS Level")) + init_description(_("Subdivision Level")) + init_constraint(constraint::minimum(1)) + init_value(2) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_sds_render_level(init_owner(*this) + init_name("sds_render_level") + init_label(_("Render SDS Level")) + init_description(_("Subdivision Level for renderers not supporting SDS natively")) + init_constraint(constraint::minimum(1)) + init_value(3) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_color(init_owner(*this) + init_name("color") + init_label(_("Color")) + init_description(_("Instance color")) + init_value(k3d::color(0, 0, 0))),
		m_selected_color(init_owner(*this) + init_name("selected_color") + init_label(_("Selected color")) + init_description(_("Instance selected color")) + init_value(k3d::color(1, 0, 0))),
		m_draw_points(init_owner(*this) + init_name("draw_points") + init_label(_("Draw points")) + init_description(_("Draw mesh points")) + init_value(true)),
		m_draw_edges(init_owner(*this) + init_name("draw_edges") + init_label(_("Draw edges")) + init_description(_("Draw mesh edges")) + init_value(true)),
		m_draw_faces(init_owner(*this) + init_name("draw_faces") + init_label(_("Draw faces")) + init_description(_("Draw mesh faces")) + init_value(true)),
		m_draw_linear_curves(init_owner(*this) + init_name("draw_linear_curves") + init_label(_("Draw linear curves")) + init_description(_("Draw mesh linear curves")) + init_value(true)),
		m_draw_cubic_curves(init_owner(*this) + init_name("draw_cubic_curves") + init_label(_("Draw cubic curves")) + init_description(_("Draw mesh cubic curves")) + init_value(true)),
		m_draw_nucurves(init_owner(*this) + init_name("draw_nucurves") + init_label(_("Draw NURBS curves")) + init_description(_("Draw mesh NURBS curves")) + init_value(true)),
		m_draw_bilinear_patches(init_owner(*this) + init_name("draw_bilinear_patches") + init_label(_("Draw bilinear patches")) + init_description(_("Draw mesh points")) + init_value(true)),
		m_draw_bicubic_patches(init_owner(*this) + init_name("draw_bicubic_patches") + init_label(_("Draw bicubic patches")) + init_description(_("Draw mesh bicubic patches")) + init_value(true)),
		m_draw_nupatches(init_owner(*this) + init_name("draw_nupatches") + init_label(_("Draw NURBS patches")) + init_description(_("Draw mesh NURBS patches")) + init_value(true)),
		m_draw_blobbies(init_owner(*this) + init_name("draw_blobbies") + init_label(_("Draw blobbies")) + init_description(_("Draw mesh blobbies")) + init_value(true)),
		m_draw_two_sided(init_owner(*this) + init_name("draw_two_sided") + init_label(_("Draw two sided")) + init_description(_("Draw two sided faces")) + init_value(false)),
		m_transformed_output_mesh(init_owner(*this) + init_name("transformed_output_mesh") + init_label(_("Transformed Output Mesh")) + init_description(_("Transformed Output Mesh")) + init_slot(sigc::mem_fun(*this, &mesh_instance::on_create_transformed_mesh))),
		m_show_component_selection(init_owner(*this) + init_name("show_component_selection") + init_label(_("Show Component Selection")) + init_description(_("Show component selection")) + init_value(false))
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_mesh_selection.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_display_lists.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_polyhedron_render_type.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_proxy_type.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_preview_sds.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_nurbs_sds.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_sds_crease.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_sds_borders.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_sds_level.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_color.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_selected_color.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_show_blobby_surface.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_selection_weight.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));
		m_show_component_selection.changed_signal().connect(sigc::mem_fun(*this, &mesh_instance::on_reset_cache));

		m_draw_points.changed_signal().connect(make_async_redraw_slot());
		m_draw_edges.changed_signal().connect(make_async_redraw_slot());
		m_draw_faces.changed_signal().connect(make_async_redraw_slot());
		m_draw_linear_curves.changed_signal().connect(make_async_redraw_slot());
		m_draw_cubic_curves.changed_signal().connect(make_async_redraw_slot());
		m_draw_nucurves.changed_signal().connect(make_async_redraw_slot());
		m_draw_bilinear_patches.changed_signal().connect(make_async_redraw_slot());
		m_draw_bicubic_patches.changed_signal().connect(make_async_redraw_slot());
		m_draw_nupatches.changed_signal().connect(make_async_redraw_slot());
		m_draw_blobbies.changed_signal().connect(make_async_redraw_slot());
		m_draw_two_sided.changed_signal().connect(make_async_redraw_slot());

		m_mesh_selection.changed_signal().connect(make_reset_mesh_slot());

		m_input_mesh.changed_signal().connect(m_transformed_output_mesh.make_reset_slot());
		m_mesh_selection.changed_signal().connect(m_transformed_output_mesh.make_reset_slot());
		m_input_matrix.changed_signal().connect(m_transformed_output_mesh.make_reset_slot());

		m_input_matrix.changed_signal().connect(make_async_redraw_slot());

//		add_snap_source(new k3d::snap_source(_("Center"), sigc::mem_fun(*this, &mesh_instance::center_source_position), sigc::mem_fun(*this, &mesh_instance::center_source_orientation)));
	}

	~mesh_instance()
	{
	}

	const k3d::point3 center_source_position()
	{
		return k3d::point3();
	}

	bool center_source_orientation(k3d::vector3& Look, k3d::vector3& Up)
	{
		Look = k3d::vector3(0, 0, 1);
		Up = k3d::vector3(0, 1, 0);
		return true;
	}

	void on_create_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
		k3d::deep_copy(InputMesh, Mesh);

		const k3d::mesh_selection selection = m_mesh_selection.value();

		k3d::replace_selection(selection, Mesh);
	}

	void on_update_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
	}

	void on_create_transformed_mesh(k3d::mesh& Mesh)
	{
		if(k3d::mesh* const input_mesh = m_input_mesh.value())
		{
			k3d::deep_copy(*input_mesh, Mesh);

			const k3d::mesh_selection selection = m_mesh_selection.value();

			k3d::replace_selection(selection, Mesh);
			std::for_each(Mesh.points.begin(), Mesh.points.end(), detail::transform_points(matrix()));
		}
	}

	void on_reset_cache(k3d::iunknown*)
	{
		m_cache.reset();

		k3d::gl::redraw_all(document(), k3d::gl::irender_engine::ASYNCHRONOUS);
	}

	void on_create_cache()
	{
		if(m_cache.get())
			return;

		m_cache.reset(new cache(m_draw_two_sided.value()));

		const k3d::mesh* const mesh = m_output_mesh.value();
		return_if_fail(mesh);

		// Create blobby polygonized surfaces cache
		for(k3d::mesh::blobbies_t::const_iterator blobby = mesh->blobbies.begin(); blobby != mesh->blobbies.end(); ++blobby)
		{
			cache::blobby_cache_map_t::const_iterator cached_blobby = m_cache->blobby_cache_map.find(*blobby);
			if(cached_blobby == m_cache->blobby_cache_map.end())
			{
				// Cache polygonized surface for blobby
				detail::vertices_t blobby_vertices;
				detail::vertices_t blobby_normals;
				detail::polygons_t blobby_polygons;
				polygonize_blobby(*blobby, 0, blobby_vertices, blobby_normals, blobby_polygons);

				// Save surface
				m_cache->blobby_cache_map[*blobby] = m_cache->blobby_surfaces_vertices.size();
				m_cache->blobby_surfaces_vertices.push_back(blobby_vertices);
				m_cache->blobby_surfaces_normals.push_back(blobby_normals);
				m_cache->blobby_surfaces_polygons.push_back(blobby_polygons);
			}
		}

		// Create SDS cache ...
		if(m_preview_sds.value() && m_proxy_type.value() == PROXY_NONE)
		{
			const polyhedron_render_t render_type = m_polyhedron_render_type.value();

			std::map<k3d::point*, k3d::point*> point_map;
			point_map[0] = 0;
			if(mesh->polyhedra.size() > 0 && ((render_type == DEFAULT && (*mesh->polyhedra.begin())->type == k3d::polyhedron::CATMULL_CLARK_SUBDIVISION_MESH) || render_type == CATMULL_CLARK))
			{
				// set SDS rendering parameters:
				m_sds_cache.set_levels(m_sds_level.value());
				m_sds_cache.use_nurbs(m_nurbs_sds.value());
				m_sds_cache.draw_borders(m_sds_borders.value());

				if(m_sds_crease.value())
				{
					m_cache->m_creased_mesh = new k3d::mesh();
					k3d::deep_copy(*mesh, *m_cache->m_creased_mesh);
					for (k3d::mesh::polyhedra_t::const_iterator polyhedron = m_cache->m_creased_mesh->polyhedra.begin(); polyhedron != m_cache->m_creased_mesh->polyhedra.end(); ++polyhedron)
					{
						k3d::sds::crease(*m_cache->m_creased_mesh, **polyhedron);
					}

					m_sds_cache.set_input(m_cache->m_creased_mesh);
				}
				else
				{
					m_sds_cache.set_input(mesh);
				}
				m_sds_cache.update();
			}
		}
	}

	const k3d::bounding_box3 extents()
	{
		k3d::bounding_box3 results;

		const k3d::mesh* const mesh = m_output_mesh.value();
		return_val_if_fail(mesh, results);

		for(k3d::mesh::points_t::const_iterator point = mesh->points.begin(); point != mesh->points.end(); ++point)
		{
			results.px = std::max(results.px, (*point)->position[0]);
			results.py = std::max(results.py, (*point)->position[1]);
			results.pz = std::max(results.pz, (*point)->position[2]);
			results.nx = std::min(results.nx, (*point)->position[0]);
			results.ny = std::min(results.ny, (*point)->position[1]);
			results.nz = std::min(results.nz, (*point)->position[2]);
		}

		return results;
	}

	void on_gl_draw(const k3d::gl::render_state& State)
	{
		// No input, so we're done ...
		k3d::mesh* const mesh = m_output_mesh.value();
		return_if_fail(mesh);

		// Update the drawing cache as-needed ...
		on_create_cache();

		// Have a nurbs renderer cached if we need it ...
		nurbs_renderer_t nurbs = 0;

		if(m_display_lists.value())
			m_cache->activate_display_lists();
		else
			m_cache->deactivate_display_lists();

		const proxy_t proxy_type = m_proxy_type.value();
		if(PROXY_POINT == proxy_type)
		{
			k3d::gl::store_attributes attributes;

			glDisable(GL_LIGHTING);
			k3d::gl::color3d(get_selection_weight() ? m_selected_color.value() : m_color.value());
			glPointSize(5.0);

			glBegin(GL_POINTS);
				glVertex3d(0, 0, 0);
			glEnd();
		}
		else if(PROXY_BOUNDING_BOX == proxy_type)
		{
			k3d::gl::store_attributes attributes;

			glDisable(GL_LIGHTING);
			k3d::gl::color3d(get_selection_weight() ? m_selected_color.value() : m_color.value());

			const k3d::bounding_box3 bbox = k3d::bounds(*mesh);
			if(!bbox.empty())
				k3d::gl::draw_bounding_box(bbox);
		}
		else if(PROXY_NONE == proxy_type)
		{
			const k3d::color face_color = k3d::color(1, 1, 1);
			const k3d::color selected_face_color = m_selected_color.value();
			const k3d::color line_color = get_selection_weight() ? k3d::color(1, 1, 1) : m_color.value();
			const k3d::color selected_line_color = m_selected_color.value();
			const k3d::color point_color = get_selection_weight() ? k3d::color(1, 1, 1) : m_color.value();
			const bool show_selection = m_show_component_selection.value();
			const k3d::color selected_point_color = m_selected_color.value();

			if(State.draw_points && m_draw_points.value()
				&& m_cache->start_points_list())
			{
				if(show_selection)
				{
					draw_points(mesh->points.begin(), mesh->points.end(), selected_point_color, detail::draw_selected());
					draw_points(mesh->points.begin(), mesh->points.end(), point_color, detail::draw_unselected());

					draw_point_groups(mesh->point_groups.begin(), mesh->point_groups.end(), selected_point_color, detail::draw_selected());
					draw_point_groups(mesh->point_groups.begin(), mesh->point_groups.end(), point_color, detail::draw_unselected());
				}
				else
				{
					draw_points(mesh->points.begin(), mesh->points.end(), point_color, detail::draw_all());
					draw_point_groups(mesh->point_groups.begin(), mesh->point_groups.end(), point_color, detail::draw_all());
				}

				m_cache->end_list();
			}

			if(State.draw_linear_curves && m_draw_linear_curves.value()
				&& !mesh->linear_curve_groups.empty()
				&& m_cache->start_linear_curves_list())
			{
				if(show_selection)
				{
					draw_linear_curve_groups(mesh->linear_curve_groups.begin(), mesh->linear_curve_groups.end(), selected_line_color, detail::draw_selected());
					draw_linear_curve_groups(mesh->linear_curve_groups.begin(), mesh->linear_curve_groups.end(), line_color, detail::draw_unselected());
				}
				else
				{
					draw_linear_curve_groups(mesh->linear_curve_groups.begin(), mesh->linear_curve_groups.end(), line_color, detail::draw_all());
				}

				m_cache->end_list();
			}

			if(State.draw_cubic_curves && m_draw_cubic_curves.value()
				&& !mesh->cubic_curve_groups.empty()
				&& m_cache->start_cubic_curves_list())
			{
				if(show_selection)
				{
					draw_cubic_curve_groups(mesh->cubic_curve_groups.begin(), mesh->cubic_curve_groups.end(), selected_line_color, detail::draw_selected());
					draw_cubic_curve_groups(mesh->cubic_curve_groups.begin(), mesh->cubic_curve_groups.end(), line_color, detail::draw_unselected());
				}
				else
				{
					draw_cubic_curve_groups(mesh->cubic_curve_groups.begin(), mesh->cubic_curve_groups.end(), selected_line_color, detail::draw_all());
				}

				m_cache->end_list();
			}

			if(State.draw_nucurves && m_draw_nucurves.value()
				&& !mesh->nucurve_groups.empty()
				&& m_cache->start_nurbs_curves_list())
			{
				if(!nurbs)
					nurbs = nurbs_renderer(State);

				if(show_selection)
				{
					draw_nucurve_groups(nurbs, mesh->nucurve_groups.begin(), mesh->nucurve_groups.end(), selected_line_color, detail::draw_selected());
					draw_nucurve_groups(nurbs, mesh->nucurve_groups.begin(), mesh->nucurve_groups.end(), line_color, detail::draw_unselected());
				}
				else
				{
					draw_nucurve_groups(nurbs, mesh->nucurve_groups.begin(), mesh->nucurve_groups.end(), line_color, detail::draw_all());
				}

				m_cache->end_list();
			}

			if(State.draw_edges && m_draw_edges.value())
			{
				if(!mesh->polyhedra.empty()
					&& m_cache->start_edges_list())
				{
					if(show_selection)
					{
						draw_polyhedron_edges(mesh->polyhedra.begin(), mesh->polyhedra.end(), selected_line_color, detail::draw_selected());
						draw_polyhedron_edges(mesh->polyhedra.begin(), mesh->polyhedra.end(), line_color, detail::draw_unselected());
					}
					else
					{
						draw_polyhedron_edges(mesh->polyhedra.begin(), mesh->polyhedra.end(), line_color, detail::draw_all());
					}

					m_cache->end_list();
				}

				if(!mesh->bilinear_patches.empty()
					&& m_cache->start_bilinear_edges_list())
				{
					if(show_selection)
					{
						draw_bilinear_patch_edges(mesh->bilinear_patches.begin(), mesh->bilinear_patches.end(), selected_line_color, detail::draw_selected());
						draw_bilinear_patch_edges(mesh->bilinear_patches.begin(), mesh->bilinear_patches.end(), line_color, detail::draw_unselected());
					}
					else
					{
						draw_bilinear_patch_edges(mesh->bilinear_patches.begin(), mesh->bilinear_patches.end(), line_color, detail::draw_all());
					}

					m_cache->end_list();
				}

				if(!mesh->bicubic_patches.empty()
					&& m_cache->start_bicubic_edges_list())
				{
					if(show_selection)
					{
						draw_bicubic_patch_edges(mesh->bicubic_patches.begin(), mesh->bicubic_patches.end(), selected_line_color, detail::draw_selected());
						draw_bicubic_patch_edges(mesh->bicubic_patches.begin(), mesh->bicubic_patches.end(), line_color, detail::draw_unselected());
					}
					else
					{
						draw_bicubic_patch_edges(mesh->bicubic_patches.begin(), mesh->bicubic_patches.end(), line_color, detail::draw_all());
					}

					m_cache->end_list();
				}

				if(!mesh->nupatches.empty()
					&& m_cache->start_nurbs_edges_list())
				{
					if(!nurbs)
						nurbs = nurbs_renderer(State);

					if(show_selection)
					{
						draw_nupatch_edges(nurbs, mesh->nupatches.begin(), mesh->nupatches.end(), selected_line_color, detail::draw_selected());
						draw_nupatch_edges(nurbs, mesh->nupatches.begin(), mesh->nupatches.end(), line_color, detail::draw_unselected());
					}
					else
					{
						draw_nupatch_edges(nurbs, mesh->nupatches.begin(), mesh->nupatches.end(), line_color, detail::draw_all());
					}

					m_cache->end_list();
				}
			}

			const bool draw_two_sided = State.draw_two_sided && m_draw_two_sided.value();

			if(State.draw_faces && m_draw_faces.value()
				&& !mesh->polyhedra.empty()
				&& m_cache->start_faces_list(draw_two_sided))
			{
				const polyhedron_render_t render_type = m_polyhedron_render_type.value();
				const bool preview_sds = m_preview_sds.value();

				for(k3d::mesh::polyhedra_t::const_iterator polyhedron = mesh->polyhedra.begin(); polyhedron != mesh->polyhedra.end(); ++polyhedron)
				{
					switch(render_type)
					{
						case DEFAULT:
							if(!preview_sds || m_proxy_type.value() != PROXY_NONE)
								break;

							if(k3d::polyhedron::CATMULL_CLARK_SUBDIVISION_MESH == (*polyhedron)->type)
								continue;
							break;
						case POLYGONS:
							break;
						case CATMULL_CLARK:
							if(!preview_sds || m_proxy_type.value() != PROXY_NONE)
								break;
							continue;
					}

					if(show_selection)
					{
						draw_faces(**polyhedron, draw_two_sided, selected_face_color, detail::draw_selected());
						draw_faces(**polyhedron, draw_two_sided, face_color, detail::draw_unselected());
					}
					else
					{
						draw_faces(**polyhedron, draw_two_sided, face_color, detail::draw_all());
					}
				}

				m_cache->end_list();
			}

			if(State.draw_bilinear_patches && m_draw_bilinear_patches.value()
				&& !mesh->bilinear_patches.empty()
				&& m_cache->start_bilinear_patches_list(draw_two_sided))
			{
				if(show_selection)
				{
					draw_bilinear_patches(mesh->bilinear_patches.begin(), mesh->bilinear_patches.end(), draw_two_sided, selected_face_color, detail::draw_selected());
					draw_bilinear_patches(mesh->bilinear_patches.begin(), mesh->bilinear_patches.end(), draw_two_sided, face_color, detail::draw_unselected());
				}
				else
				{
					draw_bilinear_patches(mesh->bilinear_patches.begin(), mesh->bilinear_patches.end(), draw_two_sided, face_color, detail::draw_all());
				}

				m_cache->end_list();
			}

			if(State.draw_bicubic_patches && m_draw_bicubic_patches.value()
				&& !mesh->bicubic_patches.empty()
				&& m_cache->start_bicubic_patches_list(draw_two_sided))
			{
				if(show_selection)
				{
					draw_bicubic_patches(mesh->bicubic_patches.begin(), mesh->bicubic_patches.end(), draw_two_sided, selected_face_color, detail::draw_selected());
					draw_bicubic_patches(mesh->bicubic_patches.begin(), mesh->bicubic_patches.end(), draw_two_sided, face_color, detail::draw_unselected());
				}
				else
				{
					draw_bicubic_patches(mesh->bicubic_patches.begin(), mesh->bicubic_patches.end(), draw_two_sided, face_color, detail::draw_all());
				}

				m_cache->end_list();
			}

			if(State.draw_nupatches && m_draw_nupatches.value()
				&& !mesh->nupatches.empty()
				&& m_cache->start_nurbs_patches_list(draw_two_sided))
			{
				if(!nurbs)
					nurbs = nurbs_renderer(State);

				if(show_selection)
				{
					draw_nupatches(nurbs, mesh->nupatches.begin(), mesh->nupatches.end(), draw_two_sided, selected_face_color, detail::draw_selected());
					draw_nupatches(nurbs, mesh->nupatches.begin(), mesh->nupatches.end(), draw_two_sided, face_color, detail::draw_unselected());
				}
				else
				{
					draw_nupatches(nurbs, mesh->nupatches.begin(), mesh->nupatches.end(), draw_two_sided, face_color, detail::draw_all());
				}

				m_cache->end_list();
			}

			if(State.draw_blobbies && m_draw_blobbies.value()
				&& !mesh->blobbies.empty()
				&& m_cache->start_blobbies_list())
			{
				draw_blobbies(mesh->blobbies.begin(), mesh->blobbies.end(), true, selected_face_color);
				draw_blobbies(mesh->blobbies.begin(), mesh->blobbies.end(), false, face_color);

				m_cache->end_list();
			}

			// SDS cache, uses its own display lists
			if(m_preview_sds.value() && m_proxy_type.value() == PROXY_NONE && (mesh->polyhedra.size() > 0 && ((m_polyhedron_render_type.value() == DEFAULT && (*mesh->polyhedra.begin())->type == k3d::polyhedron::CATMULL_CLARK_SUBDIVISION_MESH) || m_polyhedron_render_type.value() == CATMULL_CLARK)))
			{
				m_sds_cache.set_render_state(State);
				m_sds_cache.output();
			}
		}
	}

	void on_gl_select(const k3d::gl::render_state& State, const k3d::gl::select_state& SelectState)
	{
		// Check instance selection against select_state flag
		if(SelectState.exclude_unselected_nodes && (!get_selection_weight()))
			return;

		// No input, so we're done ...
		k3d::mesh* const mesh = m_output_mesh.value();
		return_if_fail(mesh);

		// Update the drawing cache as-needed ...
		on_create_cache();

		k3d::gl::store_attributes attributes;

		// Have a nurbs renderer cached if we need it ...
		nurbs_renderer_t nurbs = 0;

		// don't forget to call the SDS display lists
		if(m_preview_sds.value() && m_proxy_type.value() == PROXY_NONE && (mesh->polyhedra.size() > 0 && ((m_polyhedron_render_type.value() == DEFAULT && (*mesh->polyhedra.begin())->type == k3d::polyhedron::CATMULL_CLARK_SUBDIVISION_MESH) || m_polyhedron_render_type.value() == CATMULL_CLARK)))
		{
			m_sds_cache.set_render_state(State);
			m_sds_cache.output();
		}

		// Use (optional) OpenGL display lists for performance ...
		/** \todo Restore display lists for selection */
		const bool display_lists = false;
//		const bool display_lists = m_display_lists.value();
		if(display_lists)
		{
			if(m_cache->selection_display_list)
			{
				glCallList(m_cache->selection_display_list);
				return;
			}

			m_cache->selection_display_list = glGenLists(1);
			glNewList(m_cache->selection_display_list, GL_COMPILE_AND_EXECUTE);
		}

		// At the top-level, provide selection of this object ...
		k3d::gl::push_selection_token(this);

		// Then, provide selection of the underlying mesh ...
		k3d::gl::push_selection_token(k3d::selection::MESH, 0);

		const proxy_t proxy_type = m_proxy_type.value();
		if(PROXY_POINT == proxy_type)
		{
			k3d::gl::store_attributes attributes;

			glPointSize(5.0);
			glBegin(GL_POINTS);
				glVertex3d(0, 0, 0);
			glEnd();
		}
		else if(PROXY_BOUNDING_BOX == proxy_type)
		{
			const k3d::bounding_box3 bbox = k3d::bounds(*mesh);
			if(!bbox.empty())
				k3d::gl::draw_bounding_box(bbox);
		}
		else if(PROXY_NONE == proxy_type)
		{
			if(SelectState.select_points && !mesh->points.empty())
				select_points(mesh->points.begin(), mesh->points.end());

			if(SelectState.select_point_groups && !mesh->point_groups.empty())
				select_point_groups(mesh->point_groups.begin(), mesh->point_groups.end());

			if(SelectState.select_edges && !mesh->polyhedra.empty())
				select_polyhedron_edges(mesh->polyhedra.begin(), mesh->polyhedra.end());

			if(SelectState.select_faces && !mesh->polyhedra.empty())
				select_polyhedron_faces(mesh->polyhedra.begin(), mesh->polyhedra.end());

			if(SelectState.select_linear_curves && !mesh->linear_curve_groups.empty())
				select_linear_curve_groups(mesh->linear_curve_groups.begin(), mesh->linear_curve_groups.end());

			if(SelectState.select_cubic_curves && !mesh->cubic_curve_groups.empty())
				select_cubic_curve_groups(mesh->cubic_curve_groups.begin(), mesh->cubic_curve_groups.end());

			if(SelectState.select_nucurves && !mesh->nucurve_groups.empty())
			{
				if(!nurbs)
					nurbs = nurbs_renderer(State);

				select_nucurves(nurbs, mesh->nucurve_groups.begin(), mesh->nucurve_groups.end());
			}

			if(SelectState.select_bilinear_patches && !mesh->bilinear_patches.empty())
				select_bilinear_patches(mesh->bilinear_patches.begin(), mesh->bilinear_patches.end());

			if(SelectState.select_bicubic_patches && !mesh->bicubic_patches.empty())
				select_bicubic_patches(mesh->bicubic_patches.begin(), mesh->bicubic_patches.end());

			if(SelectState.select_nupatches && !mesh->nupatches.empty())
			{
				if(!nurbs)
					nurbs = nurbs_renderer(State);

				select_nupatches(nurbs, mesh->nupatches.begin(), mesh->nupatches.end());
			}

			if(SelectState.select_blobbies && !mesh->blobbies.empty())
				select_blobbies(mesh->blobbies.begin(), mesh->blobbies.end());
		}

		k3d::gl::pop_selection_token(); // mesh
		k3d::gl::pop_selection_token(); // node

		if(display_lists)
			glEndList();
	}

	void on_renderman_render(const k3d::ri::render_state& State)
	{
		// No input, so we're done ...
		const k3d::mesh* const mesh = m_output_mesh.value();
		return_if_fail(mesh);

		const k3d::mesh& Mesh = *mesh;

		// For each point cloud in the mesh ...
		for(k3d::mesh::point_groups_t::const_iterator point_group = Mesh.point_groups.begin(); point_group != Mesh.point_groups.end(); ++point_group)
		{
			// Set the point cloud material ...
			k3d::ri::setup_material((*point_group)->material, State);

			// Setup point cloud parameters ...
			k3d::ri::parameter_list parameters;

			// Setup constant parameters ...
			parameters += detail::build_parameters((*point_group)->constant_data.begin(), (*point_group)->constant_data.end(), k3d::ri::CONSTANT);

			// Setup vertex parameters ...
			parameters += detail::build_parameters((*point_group)->points.begin(), (*point_group)->points.end(), k3d::ri::VERTEX);

			// Setup points ...
			k3d::ri::points points;
			for(k3d::point_group::points_t::const_iterator point = (*point_group)->points.begin(); point != (*point_group)->points.end(); ++point)
			{
				if(*point)
					points.push_back((*point)->position);
			}
			parameters.push_back(k3d::ri::parameter(k3d::ri::RI_P(), k3d::ri::VERTEX, points));

			State.engine.RiPointsV(points.size(), parameters);
		}

		const polyhedron_render_t render_type = m_polyhedron_render_type.value();

		// Render polyhedra as polygons ...
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
		{
			switch(render_type)
			{
				case DEFAULT:
					if((*polyhedron)->type != k3d::polyhedron::POLYGONS)
						continue;
					break;
				case POLYGONS:
					break;
				case CATMULL_CLARK:
					continue;
			}

			// Get the set of all materials used in this polyhedron ...
			typedef std::set<k3d::imaterial*> materials_t;
			materials_t materials;
			for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
				materials.insert((*face)->material);

			// For each material ...
			for(materials_t::iterator m= materials.begin(); m!= materials.end(); ++m)
			{
				k3d::imaterial* const material = *m;

				// Setup geometry ...
				k3d::ri::unsigned_integers loop_counts;
				k3d::ri::unsigned_integers vertex_counts;
				k3d::ri::unsigned_integers vertex_ids;
				k3d::ri::points ripoints;

				std::vector<k3d::point*> points;
				std::vector<k3d::split_edge*> edges;
				std::map<k3d::point*, k3d::ri::unsigned_integer> point_map;

				// For each polygon face ...
				for(k3d::polyhedron::faces_t::const_iterator f = (*polyhedron)->faces.begin(); f != (*polyhedron)->faces.end(); ++f)
				{
					k3d::face& face = **f;

					// Skip faces that have a different material ...
					if(face.material != material)
						continue;

					// List vertices for the face ...
					k3d::ri::unsigned_integer vertex_count = 0;
					for(k3d::split_edge* edge = face.first_edge; edge; edge = edge->face_clockwise)
					{
						++vertex_count;

						edges.push_back(edge);

						if(!point_map.count(edge->vertex))
						{
							point_map.insert(std::make_pair(edge->vertex, points.size()));
							points.push_back(edge->vertex);
							ripoints.push_back(edge->vertex->position);
						}

						vertex_ids.push_back(point_map[edge->vertex]);

						if(face.first_edge == edge->face_clockwise)
							break;
					}
					vertex_counts.push_back(vertex_count);

					// For each hole in the face ...
					for(k3d::face::holes_t::const_iterator hole = face.holes.begin(); hole != face.holes.end(); ++hole)
					{
						// List vertices for the hole ...
						k3d::ri::unsigned_integer vertex_count = 0;
						for(k3d::split_edge* edge = *hole; edge; edge = edge->face_clockwise)
						{
							++vertex_count;

							edges.push_back(edge);

							if(!point_map.count(edge->vertex))
							{
								point_map.insert(std::make_pair(edge->vertex, points.size()));
								points.push_back(edge->vertex);
								ripoints.push_back(edge->vertex->position);
							}

							vertex_ids.push_back(point_map[edge->vertex]);

							if(*hole == edge->face_clockwise)
								break;
						}
						vertex_counts.push_back(vertex_count);
					}

					// Total number of loops in the face (including holes) ...
					loop_counts.push_back(1 + face.holes.size());
				}

				if(loop_counts.size())
				{
					// Setup the polyhedron parameters ...
					k3d::ri::parameter_list parameters;

					// Setup constant data ...
					parameters += detail::build_parameters((*polyhedron)->constant_data.begin(), (*polyhedron)->constant_data.end(), k3d::ri::CONSTANT);

					// Setup uniform data ...
					parameters += detail::build_parameters((*polyhedron)->faces.begin(), (*polyhedron)->faces.end(), k3d::ri::UNIFORM);

					// Setup vertex data ...
					parameters += detail::build_parameters(points.begin(), points.end(), k3d::ri::VERTEX);

					// Setup points ...
					parameters.push_back(k3d::ri::parameter(k3d::ri::RI_P(), k3d::ri::VERTEX, ripoints));

					// Setup facevarying data ...
					parameters += detail::build_parameters(edges.begin(), edges.end(), k3d::ri::FACEVARYING);

					k3d::ri::setup_material(material, State);
					State.engine.RiPointsGeneralPolygonsV(loop_counts, vertex_counts, vertex_ids, parameters);
				}
			}
		}

		// Render polyhedra as subdivision surfaces ...
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
		{
			switch(render_type)
			{
				case DEFAULT:
					if((*polyhedron)->type != k3d::polyhedron::CATMULL_CLARK_SUBDIVISION_MESH)
						continue;
					break;
				case POLYGONS:
					continue;
				case CATMULL_CLARK:
					break;
			}

			// Get the set of all materials used in this polyhedron ...
			typedef std::set<k3d::imaterial*> materials_t;
			materials_t materials;
			for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
				materials.insert((*face)->material);

			for(materials_t::iterator m= materials.begin(); m!= materials.end(); ++m)
			{
				k3d::imaterial* const material = *m;

				k3d::ri::unsigned_integers vertex_counts;
				k3d::ri::unsigned_integers vertex_ids;
				k3d::ri::points ripoints;

				std::vector<k3d::point*> points;
				std::vector<k3d::split_edge*> edges;
				std::map<k3d::point*, k3d::ri::unsigned_integer> point_map;
				std::map<k3d::face*, k3d::ri::unsigned_integer> face_map;

				// For each polygon face ...
				for(k3d::polyhedron::faces_t::const_iterator f = (*polyhedron)->faces.begin(); f != (*polyhedron)->faces.end(); ++f)
				{
					k3d::face& face = **f;

					face_map.insert(std::make_pair(&face, face_map.size()));

					// List vertices for the face ...
					k3d::ri::unsigned_integer vertex_count = 0;
					for(k3d::split_edge* edge = face.first_edge; edge; edge = edge->face_clockwise)
					{
						++vertex_count;

						edges.push_back(edge);

						if(!point_map.count(edge->vertex))
						{
							point_map.insert(std::make_pair(edge->vertex, points.size()));
							points.push_back(edge->vertex);
							ripoints.push_back(edge->vertex->position);
						}

						vertex_ids.push_back(point_map[edge->vertex]);

						if(face.first_edge == edge->face_clockwise)
							break;
					}
					vertex_counts.push_back(vertex_count);
				}

				// Setup parameters ...
				k3d::ri::parameter_list parameters;

				// Setup constant data ...
				parameters += detail::build_parameters((*polyhedron)->constant_data.begin(), (*polyhedron)->constant_data.end(), k3d::ri::CONSTANT);

				// Setup uniform data ...
				parameters += detail::build_parameters((*polyhedron)->faces.begin(), (*polyhedron)->faces.end(), k3d::ri::UNIFORM);

				// Setup vertex data ...
				parameters += detail::build_parameters(points.begin(), points.end(), k3d::ri::VERTEX);

				// Setup points ...
				parameters.push_back(k3d::ri::parameter(k3d::ri::RI_P(), k3d::ri::VERTEX, ripoints));

				// Setup facevarying data ...
				parameters += detail::build_parameters(edges.begin(), edges.end(), k3d::ri::FACEVARYING);

				// Setup tags ...
				k3d::ri::strings tags;
				k3d::ri::unsigned_integers tag_counts;
				k3d::ri::integers tag_integers;
				k3d::ri::reals tag_reals;
				detail::build_tags((*polyhedron)->tags.begin(), (*polyhedron)->tags.end(), tags, tag_counts, tag_integers, tag_reals);
				detail::build_tags((*polyhedron)->faces.begin(), (*polyhedron)->faces.end(), face_map, tags, tag_counts, tag_integers, tag_reals, material);
				detail::build_tags(edges.begin(), edges.end(), point_map, tags, tag_counts, tag_integers, tag_reals);
				detail::build_tags(points.begin(), points.end(), point_map, tags, tag_counts, tag_integers, tag_reals);

				k3d::ri::setup_material(material, State);
				State.engine.RiSubdivisionMeshV("catmull-clark", vertex_counts, vertex_ids, tags, tag_counts, tag_integers, tag_reals, parameters);
			}
		}

		// For each linear curve group in the mesh ...
		for(k3d::mesh::linear_curve_groups_t::const_iterator group = Mesh.linear_curve_groups.begin(); group != Mesh.linear_curve_groups.end(); ++group)
		{
			// Set the group material ...
			k3d::ri::setup_material((*group)->material, State);

			// Keep track of curve control points ...
			std::vector<k3d::point*> points;
			k3d::ri::points ripoints;
			k3d::ri::unsigned_integers point_counts;

			// For each linear curve in the group ...
			for(k3d::linear_curve_group::curves_t::const_iterator curve = (*group)->curves.begin(); curve != (*group)->curves.end(); ++curve)
			{
				for(k3d::linear_curve::control_points_t::const_iterator control_point = (*curve)->control_points.begin(); control_point != (*curve)->control_points.end(); ++control_point)
				{
					points.push_back(*control_point);
					ripoints.push_back((*control_point)->position);
				}

				point_counts.push_back((*curve)->control_points.size());
			}

			// Setup group parameters ...
			k3d::ri::parameter_list parameters;

			// Setup constant parameters ...
			parameters += detail::build_parameters((*group)->constant_data.begin(), (*group)->constant_data.end(), k3d::ri::CONSTANT);

			// Setup uniform parameters ...
			parameters += detail::build_parameters((*group)->curves.begin(), (*group)->curves.end(), k3d::ri::UNIFORM);

			// Setup varying parameters ...
			parameters += detail::build_parameters((*group)->curves.begin(), (*group)->curves.end(), k3d::ri::VARYING);

			// Setup vertex parameters ...
			parameters += detail::build_parameters(points.begin(), points.end(), k3d::ri::VERTEX);

			// Setup points ...
			parameters.push_back(k3d::ri::parameter(k3d::ri::RI_P(), k3d::ri::VERTEX, ripoints));

			State.engine.RiCurvesV("linear", point_counts, (*group)->wrap ? "periodic" : "nonperiodic", parameters);
		}

		// For each cubic curve group in the mesh ...
		for(k3d::mesh::cubic_curve_groups_t::const_iterator group = Mesh.cubic_curve_groups.begin(); group != Mesh.cubic_curve_groups.end(); ++group)
		{
			// Set the group material ...
			k3d::ri::setup_material((*group)->material, State);

			// Keep track of curve control points ...
			std::vector<k3d::point*> points;
			k3d::ri::points ripoints;
			k3d::ri::unsigned_integers point_counts;

			// For each cubic curve in the group ...
			for(k3d::cubic_curve_group::curves_t::const_iterator curve = (*group)->curves.begin(); curve != (*group)->curves.end(); ++curve)
			{
				for(k3d::cubic_curve::control_points_t::const_iterator control_point = (*curve)->control_points.begin(); control_point != (*curve)->control_points.end(); ++control_point)
				{
					points.push_back(*control_point);
					ripoints.push_back((*control_point)->position);
				}

				point_counts.push_back((*curve)->control_points.size());
			}

			// Setup group parameters ...
			k3d::ri::parameter_list parameters;

			// Setup constant parameters ...
			parameters += detail::build_parameters((*group)->constant_data.begin(), (*group)->constant_data.end(), k3d::ri::CONSTANT);

			// Setup uniform parameters ...
			parameters += detail::build_parameters((*group)->curves.begin(), (*group)->curves.end(), k3d::ri::UNIFORM);

			// Setup varying parameters ...
			parameters += detail::build_parameters((*group)->curves.begin(), (*group)->curves.end(), k3d::ri::VARYING);

			// Setup vertex parameters ...
			parameters += detail::build_parameters(points.begin(), points.end(), k3d::ri::VERTEX);

			// Setup points ...
			parameters.push_back(k3d::ri::parameter(k3d::ri::RI_P(), k3d::ri::VERTEX, ripoints));

			// At the moment, we force all cubic curves to be Bezier ...
			State.engine.RiBasis("bezier", 3, "bezier", 3);
			State.engine.RiCurvesV("cubic", point_counts, (*group)->wrap ? "periodic" : "nonperiodic", parameters);
		}

		// For each bilinear patch in the mesh ...
		for(k3d::mesh::bilinear_patches_t::const_iterator patch = Mesh.bilinear_patches.begin(); patch != Mesh.bilinear_patches.end(); ++patch)
		{
			// Set the polyhedron material ...
			k3d::ri::setup_material((*patch)->material, State);

			// Setup patch parameters ...
			k3d::ri::parameter_list parameters;

			// Setup uniform parameters ...
			parameters += detail::build_parameters((*patch)->uniform_data.begin(), (*patch)->uniform_data.end(), k3d::ri::UNIFORM);

			// Setup varying parameters ...
			parameters += detail::build_parameters((*patch)->varying_data.begin(), (*patch)->varying_data.end(), k3d::ri::VARYING);

			// Setup vertex parameters ...
			parameters += detail::build_parameters((*patch)->control_points.begin(), (*patch)->control_points.end(), k3d::ri::VERTEX);

			// Setup points ...
			k3d::ri::points points;
			for(unsigned int i = 0; i != 4; ++i)
				points.push_back((*patch)->control_points[i]->position);
			parameters.push_back(k3d::ri::parameter(k3d::ri::RI_P(), k3d::ri::VERTEX, points));

			State.engine.RiPatchV("bilinear", parameters);
		}

		// For each bicubic patch in the mesh ...
		for(k3d::mesh::bicubic_patches_t::const_iterator patch = Mesh.bicubic_patches.begin(); patch != Mesh.bicubic_patches.end(); ++patch)
		{
			// Set the polyhedron material ...
			k3d::ri::setup_material((*patch)->material, State);

			// Setup patch parameters ...
			k3d::ri::parameter_list parameters;

			// Setup uniform parameters ...
			parameters += detail::build_parameters((*patch)->uniform_data.begin(), (*patch)->uniform_data.end(), k3d::ri::UNIFORM);

			// Setup varying parameters ...
			parameters += detail::build_parameters((*patch)->varying_data.begin(), (*patch)->varying_data.end(), k3d::ri::VARYING);

			// Setup vertex parameters ...
			parameters += detail::build_parameters((*patch)->control_points.begin(), (*patch)->control_points.end(), k3d::ri::VERTEX);

			// Setup points ...
			k3d::ri::points points;
			for(unsigned int i = 0; i != 16; ++i)
				points.push_back((*patch)->control_points[i]->position);
			parameters.push_back(k3d::ri::parameter(k3d::ri::RI_P(), k3d::ri::VERTEX, points));

			// At the moment, we force all bicubic patches to be Bezier ...
			State.engine.RiBasis("bezier", 3, "bezier", 3);
			State.engine.RiPatchV("bicubic", parameters);
		}

		// For each NURBS patch in the mesh ...
		for(k3d::mesh::nupatches_t::const_iterator nupatch = Mesh.nupatches.begin(); nupatch != Mesh.nupatches.end(); ++nupatch)
		{
			const k3d::nupatch& patch = **nupatch;

			// Set the patch material ...
			k3d::ri::setup_material(patch.material, State);

			// Setup patch parameters ...
			k3d::ri::parameter_list parameters;

			const k3d::ri::unsigned_integer u_count = patch.u_knots.size() - patch.u_order;
			const k3d::ri::unsigned_integer v_count = patch.v_knots.size() - patch.v_order;
			const k3d::ri::unsigned_integer u_order = patch.u_order;
			const k3d::ri::unsigned_integer v_order = patch.v_order;

			// Setup points ...
			k3d::ri::hpoints points;
			for(k3d::nupatch::control_points_t::const_iterator control_point = patch.control_points.begin(); control_point != patch.control_points.end(); ++control_point)
				points.push_back(k3d::point4(control_point->position->position * control_point->weight, control_point->weight));
			parameters.push_back(k3d::ri::parameter(k3d::ri::RI_PW(), k3d::ri::VERTEX, points));

			State.engine.RiNuPatchV(
				u_count,
				u_order,
				patch.u_knots,
				patch.u_knots[u_order-1],
				patch.u_knots[u_count],
				v_count,
				v_order,
				patch.v_knots,
				patch.v_knots[v_order-1],
				patch.v_knots[v_count],
				parameters
				);
		}

		// For each blobby in the mesh ...
		for(k3d::mesh::blobbies_t::const_iterator blobby = Mesh.blobbies.begin(); blobby != Mesh.blobbies.end(); ++blobby)
		{
			// Setup blobby parameters ...
			k3d::ri::unsigned_integer nleaf = 0;
			k3d::ri::unsigned_integers codes;
			k3d::ri::reals floats;
			k3d::ri::strings names;
			detail::grouped_parameters_t grouped_parameters;

			detail::blobby_vm(**blobby, nleaf, codes, floats, names, grouped_parameters);

			k3d::ri::parameter_list parameters;
			parameters += detail::build_parameters(grouped_parameters, k3d::ri::VERTEX);
			parameters += detail::build_parameters(grouped_parameters, k3d::ri::VARYING);

			State.engine.RiBlobbyV(nleaf, codes, floats, names, parameters);
		}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<mesh_instance>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink,
			k3d::interface_list<k3d::itransform_source,
			k3d::interface_list<k3d::itransform_sink > > > > > factory(
				k3d::classes::MeshInstance(),
				"MeshInstance",
				_("Renders an instance of a geometric mesh"),
				"Mesh",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	template<typename predicate_t>
	void draw_points(const k3d::mesh::points_t::const_iterator Begin, const k3d::mesh::points_t::const_iterator End, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		glColor3d(Color.red, Color.green, Color.blue);

		glBegin(GL_POINTS);
		for(k3d::mesh::points_t::const_iterator point = Begin; point != End; ++point)
		{
			if(!Predicate(*point))
				continue;

			glVertex3dv((*point)->position.n);
		}
		glEnd();
	}

	void select_points(const k3d::mesh::points_t::const_iterator Begin, const k3d::mesh::points_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);

		k3d::selection::id point_id = 0;
		for(k3d::mesh::points_t::const_iterator point = Begin; point != End; ++point, ++point_id)
		{
			k3d::gl::push_selection_token(k3d::selection::ABSOLUTE_POINT, point_id);

			glBegin(GL_POINTS);
			glVertex3dv((*point)->position.n);
			glEnd();

			k3d::gl::pop_selection_token();
		}
	}

	template<typename predicate_t>
	void draw_point_groups(const k3d::mesh::point_groups_t::const_iterator Begin, const k3d::mesh::point_groups_t::const_iterator End, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		glColor3d(Color.red, Color.green, Color.blue);

		for(k3d::mesh::point_groups_t::const_iterator group = Begin; group != End; ++group)
		{
			if(!Predicate(*group))
				continue;

			glBegin(GL_POINTS);
			for(k3d::point_group::points_t::const_iterator point = (*group)->points.begin(); point != (*group)->points.end(); ++point)
			{
				if(*point)
					glVertex3dv((*point)->position.n);
			}
			glEnd();
		}
	}

	void select_point_groups(const k3d::mesh::point_groups_t::const_iterator Begin, const k3d::mesh::point_groups_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);

		k3d::selection::id group_id = 0;
		for(k3d::mesh::point_groups_t::const_iterator group = Begin; group != End; ++group, ++group_id)
		{

			k3d::gl::push_selection_token(k3d::selection::POINT_GROUP, group_id);

			glBegin(GL_POINTS);
			k3d::selection::id point_id = 0;
			for(k3d::point_group::points_t::const_iterator point = (*group)->points.begin(); point != (*group)->points.end(); ++point, ++point_id)
			{
				if(*point)
				{
					k3d::gl::push_selection_token(k3d::selection::POINT, point_id);
					glVertex3dv((*point)->position.n);
					k3d::gl::pop_selection_token();
				}
			}
			glEnd();

			k3d::gl::pop_selection_token();
		}
	}

	template<typename predicate_t>
	void draw_linear_curve_groups(const k3d::mesh::linear_curve_groups_t::const_iterator Begin, const k3d::mesh::linear_curve_groups_t::const_iterator End, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		glColor3d(Color.red, Color.green, Color.blue);
		glLineWidth(1.0);

		for(k3d::mesh::linear_curve_groups_t::const_iterator group = Begin; group != End; ++group)
		{
			const GLenum mode = (*group)->wrap ? GL_LINE_LOOP : GL_LINE_STRIP;
			for(k3d::linear_curve_group::curves_t::const_iterator curve = (*group)->curves.begin(); curve != (*group)->curves.end(); ++curve)
			{
				if(!Predicate(*curve))
					continue;

				glBegin(mode);
				for(k3d::linear_curve::control_points_t::const_iterator control_point = (*curve)->control_points.begin(); control_point != (*curve)->control_points.end(); ++control_point)
				{
					return_if_fail(*control_point);
					glVertex3dv((*control_point)->position.n);
				}
				glEnd();
			}
		}
	}

	void select_linear_curve_groups(const k3d::mesh::linear_curve_groups_t::const_iterator Begin, const k3d::mesh::linear_curve_groups_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);

		k3d::selection::id group_id = 0;
		k3d::selection::id absolute_curve_id = 0;
		for(k3d::mesh::linear_curve_groups_t::const_iterator group = Begin; group != End; ++group, ++group_id)
		{
			k3d::gl::push_selection_token(k3d::selection::LINEAR_CURVE_GROUP, group_id);

			const GLenum mode = (*group)->wrap ? GL_LINE_LOOP : GL_LINE_STRIP;
			k3d::selection::id curve_id = 0;
			for(k3d::linear_curve_group::curves_t::const_iterator curve = (*group)->curves.begin(); curve != (*group)->curves.end(); ++curve, ++curve_id, ++absolute_curve_id)
			{
				k3d::gl::push_selection_token(k3d::selection::LINEAR_CURVE, curve_id);
				k3d::gl::push_selection_token(k3d::selection::ABSOLUTE_LINEAR_CURVE, absolute_curve_id);

				glBegin(mode);
				for(k3d::linear_curve::control_points_t::const_iterator control_point = (*curve)->control_points.begin(); control_point != (*curve)->control_points.end(); ++control_point)
				{
					return_if_fail(*control_point);
					glVertex3dv((*control_point)->position.n);
				}
				glEnd();

				k3d::gl::pop_selection_token();
				k3d::gl::pop_selection_token();
			}

			k3d::gl::pop_selection_token();
		}
	}

	template<typename predicate_t>
	void draw_cubic_curve_groups(const k3d::mesh::cubic_curve_groups_t::const_iterator Begin, const k3d::mesh::cubic_curve_groups_t::const_iterator End, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		glColor3d(Color.red, Color.green, Color.blue);
		glLineWidth(1.0);

		const unsigned int v_count = 8;
		const GLint v_order = 4;
		const GLint v_stride = 3;

		glEnable(GL_MAP1_VERTEX_3);
		glDisable(GL_AUTO_NORMAL);
		glMapGrid1d(v_count, 0.0, 1.0);

		GLdouble patch_points[4 * 3];
		for(k3d::mesh::cubic_curve_groups_t::const_iterator group = Begin; group != End; ++group)
		{
//				const GLenum mode = (*group)->wrap ? GL_LINE_LOOP : GL_LINE_STRIP;
			for(k3d::cubic_curve_group::curves_t::const_iterator curve = (*group)->curves.begin(); curve != (*group)->curves.end(); ++curve)
			{
				if(!Predicate(*curve))
					continue;

				GLdouble* pp = patch_points;
				for(k3d::cubic_curve::control_points_t::const_iterator control_point = (*curve)->control_points.begin(); control_point != (*curve)->control_points.end(); ++control_point)
				{
					return_if_fail(*control_point);
					const k3d::point3& v = (*control_point)->position;
					*pp++ = v[0];
					*pp++ = v[1];
					*pp++ = v[2];
				}
				glMap1d(GL_MAP1_VERTEX_3, 0, 1, v_stride, v_order, patch_points);
				glEvalMesh1(GL_LINE, 0, v_count);
			}
		}
	}

	void select_cubic_curve_groups(const k3d::mesh::cubic_curve_groups_t::const_iterator Begin, const k3d::mesh::cubic_curve_groups_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);

		const unsigned int v_count = 8;
		const GLint v_order = 4;
		const GLint v_stride = 3;

		glEnable(GL_MAP1_VERTEX_3);
		glDisable(GL_AUTO_NORMAL);
		glMapGrid1d(v_count, 0.0, 1.0);

		GLdouble patch_points[4 * 3];
		k3d::selection::id group_id = 0;
		k3d::selection::id absolute_curve_id = 0;
		for(k3d::mesh::cubic_curve_groups_t::const_iterator group = Begin; group != End; ++group, ++group_id)
		{
			k3d::gl::push_selection_token(k3d::selection::CUBIC_CURVE_GROUP, group_id);

			k3d::selection::id curve_id = 0;
			for(k3d::cubic_curve_group::curves_t::const_iterator curve = (*group)->curves.begin(); curve != (*group)->curves.end(); ++curve, ++curve_id, ++absolute_curve_id)
			{
				k3d::gl::push_selection_token(k3d::selection::CUBIC_CURVE, curve_id);
				k3d::gl::push_selection_token(k3d::selection::ABSOLUTE_CUBIC_CURVE, absolute_curve_id);

				GLdouble* pp = patch_points;
				for(k3d::cubic_curve::control_points_t::const_iterator control_point = (*curve)->control_points.begin(); control_point != (*curve)->control_points.end(); ++control_point)
				{
					return_if_fail(*control_point);
					const k3d::point3& v = (*control_point)->position;
					*pp++ = v[0];
					*pp++ = v[1];
					*pp++ = v[2];
				}
				glMap1d(GL_MAP1_VERTEX_3, 0, 1, v_stride, v_order, patch_points);
				glEvalMesh1(GL_LINE, 0, v_count);

				k3d::gl::pop_selection_token();
				k3d::gl::pop_selection_token();
			}

			k3d::gl::pop_selection_token();
		}
	}

	template<typename predicate_t>
	void draw_nucurve_groups(const nurbs_renderer_t Nurbs, const k3d::mesh::nucurve_groups_t::const_iterator Begin, const k3d::mesh::nucurve_groups_t::const_iterator End, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		glColor3d(Color.red, Color.green, Color.blue);

		for(k3d::mesh::nucurve_groups_t::const_iterator group = Begin; group != End; ++group)
		{
			for(k3d::nucurve_group::curves_t::const_iterator nucurve = (*group)->curves.begin(); nucurve != (*group)->curves.end(); ++nucurve)
			{
				k3d::nucurve& curve = **nucurve;
				if(!Predicate(*nucurve))
					continue;

				std::vector<GLfloat> gl_knot_vector(curve.knots.begin(), curve.knots.end());

				k3d::nucurve::control_points_t& control_points = curve.control_points;
				std::vector<GLfloat> gl_control_points;
				gl_control_points.reserve(4 * control_points.size());
				for(unsigned int i = 0; i != control_points.size(); ++i)
				{
					gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight * control_points[i].position->position[0]));
					gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight * control_points[i].position->position[1]));
					gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight * control_points[i].position->position[2]));
					gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight));
				}

				gluBeginCurve(Nurbs);
				gluNurbsCurve(Nurbs, gl_knot_vector.size(), &gl_knot_vector[0], 4, &gl_control_points[0], curve.order, GL_MAP1_VERTEX_4);
				gluEndCurve(Nurbs);
			}
		}
	}

	void select_nucurves(const nurbs_renderer_t Nurbs, const k3d::mesh::nucurve_groups_t::const_iterator Begin, const k3d::mesh::nucurve_groups_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;
		glDisable(GL_LIGHTING);

		k3d::selection::id group_id = 0;
		k3d::selection::id absolute_curve_id = 0;
		for(k3d::mesh::nucurve_groups_t::const_iterator group = Begin; group != End; ++group, ++group_id)
		{
			k3d::gl::push_selection_token(k3d::selection::NUCURVE_GROUP, group_id);

			k3d::selection::id curve_id = 0;
			for(k3d::nucurve_group::curves_t::const_iterator nucurve = (*group)->curves.begin(); nucurve != (*group)->curves.end(); ++nucurve, ++curve_id, ++absolute_curve_id)
			{
				k3d::nucurve& curve = **nucurve;
				k3d::nucurve::control_points_t& control_points = curve.control_points;

				std::vector<GLfloat> gl_knot_vector(curve.knots.begin(), curve.knots.end());

				std::vector<GLfloat> gl_control_points;
				gl_control_points.reserve(4 * control_points.size());
				for(unsigned int i = 0; i != control_points.size(); ++i)
				{
					gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight * control_points[i].position->position[0]));
					gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight * control_points[i].position->position[1]));
					gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight * control_points[i].position->position[2]));
					gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight));
				}

				k3d::gl::push_selection_token(k3d::selection::NUCURVE, curve_id);
				k3d::gl::push_selection_token(k3d::selection::ABSOLUTE_NUCURVE, absolute_curve_id);

				gluBeginCurve(Nurbs);
				gluNurbsCurve(Nurbs, gl_knot_vector.size(), &gl_knot_vector[0], 4, &gl_control_points[0], curve.order, GL_MAP1_VERTEX_4);
				gluEndCurve(Nurbs);

				k3d::gl::pop_selection_token();
				k3d::gl::pop_selection_token();
			}

			k3d::gl::pop_selection_token();
		}
	}

	template<typename predicate_t>
	void draw_polyhedron_edges(const k3d::mesh::polyhedra_t::const_iterator Begin, const k3d::mesh::polyhedra_t::const_iterator End, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;
		glDisable(GL_LIGHTING);

		glColor3d(Color.red, Color.green, Color.blue);

		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Begin; polyhedron != End; ++polyhedron)
		{
			// Draws all edges that are part of a specific face
			for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
			{
				for(k3d::split_edge* edge = (*face)->first_edge; edge; edge = edge->face_clockwise)
				{
					if(Predicate(edge) || Predicate(edge->companion) || Predicate(*face))
					{
						glBegin(GL_LINES);
							k3d::gl::vertex3d(edge->vertex->position);
							k3d::gl::vertex3d(edge->face_clockwise->vertex->position);
						glEnd();
					}

					if((*face)->first_edge == edge->face_clockwise)
						break;
				}

				// Draw hole edges
				for(k3d::face::holes_t::iterator hole = (*face)->holes.begin(); hole != (*face)->holes.end(); ++hole)
				{
					for(k3d::split_edge* edge = *hole; edge; edge = edge->face_clockwise)
					{
						if(Predicate(edge))
						{
							glBegin(GL_LINES);
								k3d::gl::vertex3d(edge->vertex->position);
								k3d::gl::vertex3d(edge->face_clockwise->vertex->position);
							glEnd();
						}

						if(edge->face_clockwise == (*hole))
							break;
					}
				}
			}
		}
	}

	void set_material_color(const k3d::color& color)
	{
		k3d::gl::material(GL_FRONT_AND_BACK, GL_DIFFUSE, color);
	}

	template<typename predicate_t>
	void draw_faces(const k3d::polyhedron& Polyhedron, const bool TwoSided, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;
		glEnable(GL_LIGHTING);
		glColor3d(0.8, 0.8, 1);

		glFrontFace(GL_CW);
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

		if(TwoSided)
			glDisable(GL_CULL_FACE);
		else
			glEnable(GL_CULL_FACE);

		glEnable(GL_POLYGON_OFFSET_FILL);
		glPolygonOffset(1.0, 1.0);

		k3d::imaterial* last_material = reinterpret_cast<k3d::imaterial*>(-1);
		for(k3d::polyhedron::faces_t::const_iterator face = Polyhedron.faces.begin(); face != Polyhedron.faces.end(); ++face)
		{
			if(!Predicate(*face))
				continue;

			if((*face)->material != last_material)
			{
				last_material = (*face)->material;
				k3d::gl::setup_material((*face)->material);
				set_material_color(Color);
			}

			// Check for point normals
			k3d::ri::normal face_normal = k3d::normal(**face);
			std::vector<k3d::point3> points;
			std::vector<k3d::ri::normal> normals;
			k3d::parameters_t::const_iterator N;
			unsigned long edge_number = 0;
			for(k3d::split_edge* edge = (*face)->first_edge; edge; edge = edge->face_clockwise)
			{
				points.push_back(edge->vertex->position);

				N = edge->facevarying_data.find("N");
				if(N != edge->facevarying_data.end())
				{
					normals.push_back(boost::any_cast<k3d::ri::normal>(N->second));
				}
				else
				{
					N = edge->vertex->vertex_data.find("N");
					if(N != edge->vertex->vertex_data.end())
					{
						normals.push_back(boost::any_cast<k3d::ri::normal>(N->second));
					}
					else
					{
						normals.push_back(face_normal);
					}
				}

				++edge_number;
				if(edge->face_clockwise == (*face)->first_edge)
					break;
			}

			// Draw polygon
			return_if_fail(edge_number >= 3);
			if(edge_number == 3 && !(*face)->holes.size())
			{
				// Trivial case
				glBegin(GL_TRIANGLES);
					glNormal3dv(normals[0].n);
					k3d::gl::vertex3d(points[0]);
					glNormal3dv(normals[1].n);
					k3d::gl::vertex3d(points[1]);
					glNormal3dv(normals[2].n);
					k3d::gl::vertex3d(points[2]);
				glEnd();
			}
			else if(edge_number == 4 && !(*face)->holes.size())
			{
				// Handle quads
				unsigned long n = 0;
				for(; n < 3; ++n)
				{
					k3d::vector3 vector1 = k3d::to_vector(points[(n + 3) % 4] - points[n]);
					k3d::vector3 vector2 = k3d::to_vector(points[(n + 1) % 4] - points[n]);
					k3d::vector3 normal = vector1 ^ vector2;
					if(normal.length2() > 1e-6 && (normal * k3d::to_vector(face_normal)) < 0)
						break;
				}

				glBegin(GL_TRIANGLES);
					glNormal3dv(normals[n].n);
					k3d::gl::vertex3d(points[n]);
					glNormal3dv(normals[(n + 1) % 4].n);
					k3d::gl::vertex3d(points[(n + 1) % 4]);
					glNormal3dv(normals[(n + 2) % 4].n);
					k3d::gl::vertex3d(points[(n + 2) % 4]);

					glNormal3dv(normals[n].n);
					k3d::gl::vertex3d(points[n]);
					glNormal3dv(normals[(n + 2) % 4].n);
					k3d::gl::vertex3d(points[(n + 2) % 4]);
					glNormal3dv(normals[(n + 3) % 4].n);
					k3d::gl::vertex3d(points[(n + 3) % 4]);
				glEnd();
			}
			else if(detail::is_convex(**face) && !(*face)->holes.size())
			{
				// Cut convex polygon into a triangle fan
				glBegin(GL_TRIANGLES);
				const unsigned long np = points.size();
				for(unsigned long n = 0; n < np - 2; ++n)
				{
					glNormal3dv(normals[0].n);
					k3d::gl::vertex3d(points[0]);
					glNormal3dv(normals[(n + 1) % np].n);
					k3d::gl::vertex3d(points[(n + 1) % np]);
					glNormal3dv(normals[(n + 2) % np].n);
					k3d::gl::vertex3d(points[(n + 2) % np]);
				}
				glEnd();
			}
			else
			{
				// General case
				glNormal3dv(face_normal.n);
				detail::glu_tesselator tesselator;
				tesselator.tesselate(**face);
			}
		}
	}

	void select_polyhedron_edges(const k3d::mesh::polyhedra_t::const_iterator Begin, const k3d::mesh::polyhedra_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;

		k3d::selection::id polyhedron_id = 0;
		k3d::selection::id absolute_edge_id = 0;
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Begin; polyhedron != End; ++polyhedron, ++polyhedron_id)
		{
			k3d::gl::push_selection_token(k3d::selection::POLYHEDRON, polyhedron_id);

			k3d::selection::id face_id = 0;
			for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face, ++face_id)
			{
				k3d::gl::push_selection_token(k3d::selection::FACE, face_id);

				k3d::selection::id edge_id = 0;
				for(k3d::split_edge* edge = (*face)->first_edge; edge; edge = edge->face_clockwise)
				{
					k3d::gl::push_selection_token(k3d::selection::SPLIT_EDGE, edge_id++);
					k3d::gl::push_selection_token(k3d::selection::ABSOLUTE_SPLIT_EDGE, absolute_edge_id++);

					glBegin(GL_LINES);
						k3d::gl::vertex3d(edge->vertex->position);
						k3d::gl::vertex3d(edge->face_clockwise->vertex->position);
					glEnd();

					k3d::gl::pop_selection_token();
					k3d::gl::pop_selection_token();

					if((*face)->first_edge == edge->face_clockwise)
						break;
				}

				k3d::selection::id hole_id = 0;
				for(k3d::face::holes_t::iterator hole = (*face)->holes.begin(); hole != (*face)->holes.end(); ++hole, ++hole_id)
				{
					k3d::gl::push_selection_token(k3d::selection::FACE_HOLE, hole_id);

					k3d::selection::id edge_id = 0;
					for(k3d::split_edge* edge = *hole; edge; edge = edge->face_clockwise)
					{
						k3d::gl::push_selection_token(k3d::selection::SPLIT_EDGE, edge_id++);
						k3d::gl::push_selection_token(k3d::selection::ABSOLUTE_SPLIT_EDGE, absolute_edge_id++);

						glBegin(GL_LINES);
							k3d::gl::vertex3d(edge->vertex->position);
							k3d::gl::vertex3d(edge->face_clockwise->vertex->position);
						glEnd();

						k3d::gl::pop_selection_token();
						k3d::gl::pop_selection_token();

						if(edge->face_clockwise == (*hole))
							break;
					}

					k3d::gl::pop_selection_token();
				}

				k3d::gl::pop_selection_token();
			}

			k3d::gl::pop_selection_token();
		}
	}

	void select_polyhedron_faces(const k3d::mesh::polyhedra_t::const_iterator Begin, const k3d::mesh::polyhedra_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;

		glFrontFace(GL_CW);
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		glDisable(GL_CULL_FACE);

		k3d::selection::id polyhedron_id = 0;
		k3d::selection::id absolute_face_id = 0;
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Begin; polyhedron != End; ++polyhedron, ++polyhedron_id)
		{
			k3d::gl::push_selection_token(k3d::selection::POLYHEDRON, polyhedron_id);

			k3d::selection::id face_id = 0;
			for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face, ++face_id, ++absolute_face_id)
			{
				k3d::gl::push_selection_token(k3d::selection::FACE, face_id);
				k3d::gl::push_selection_token(k3d::selection::ABSOLUTE_FACE, absolute_face_id);

				std::vector<k3d::point3> points;
				unsigned long edge_number = 0;
				for(k3d::split_edge* edge = (*face)->first_edge; edge; edge = edge->face_clockwise)
				{
					points.push_back(edge->vertex->position);

					++edge_number;
					if(edge->face_clockwise == (*face)->first_edge)
						break;
				}

				if(edge_number == 3 && !(*face)->holes.size())
				{
					// Trivial case
					glBegin(GL_TRIANGLES);
						k3d::gl::vertex3d(points[0]);
						k3d::gl::vertex3d(points[1]);
						k3d::gl::vertex3d(points[2]);
					glEnd();
				}
				else if(edge_number == 4 && !(*face)->holes.size())
				{
					// Handle quads
					k3d::normal3 face_normal = k3d::normal(**face);
					unsigned long n = 0;
					for(; n < 3; ++n)
					{
						k3d::vector3 vector1 = k3d::to_vector(points[(n + 3) % 4] - points[n]);
						k3d::vector3 point2 = k3d::to_vector(points[(n + 1) % 4] - points[n]);
						k3d::vector3 normal = vector1 ^ point2;
						if(normal.length2() > 1e-6 && (normal * face_normal) < 0)
							break;
					}

					glBegin(GL_TRIANGLES);
						k3d::gl::vertex3d(points[n]);
						k3d::gl::vertex3d(points[(n + 1) % 4]);
						k3d::gl::vertex3d(points[(n + 2) % 4]);

						k3d::gl::vertex3d(points[n]);
						k3d::gl::vertex3d(points[(n + 2) % 4]);
						k3d::gl::vertex3d(points[(n + 3) % 4]);
					glEnd();
				}
				else if(detail::is_convex(**face) && !(*face)->holes.size())
				{
					// Cut convex polygon into a triangle fan
					glBegin(GL_TRIANGLES);
					const unsigned long np = points.size();
					for(unsigned long n = 0; n < np - 2; ++n)
					{
						k3d::gl::vertex3d(points[0]);
						k3d::gl::vertex3d(points[(n + 1) % np]);
						k3d::gl::vertex3d(points[(n + 2) % np]);
					}
					glEnd();
				}
				else
				{
					// General case
					detail::glu_tesselator tesselator;
					tesselator.tesselate(**face);
				}

				k3d::gl::pop_selection_token();
				k3d::gl::pop_selection_token();
			}

/*
			if(DrawPoints)
			{
				for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
				{
					for(k3d::split_edge* edge = (*face)->first_edge; edge; edge = edge->face_clockwise)
					{
//						k3d::gl::push_name(edge->vertex);

						glBegin(GL_POINTS);
						glVertex3dv(edge->vertex->position.n);
						glEnd();

//						k3d::gl::pop_name();

						if((*face)->first_edge == edge->face_clockwise)
							break;
					}
				}
			}
*/

			k3d::gl::pop_selection_token();
		}
	}

	template<typename predicate_t>
	void draw_bilinear_patch_edges(const k3d::mesh::bilinear_patches_t::const_iterator Begin, const k3d::mesh::bilinear_patches_t::const_iterator End, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		glColor3d(Color.red, Color.green, Color.blue);

		const unsigned int u_count = 10;
		const unsigned int v_count = 10;
		const GLint u_order = 2;
		const GLint v_order = 2;
		const GLint u_stride = 3;
		const GLint v_stride = 2 * u_stride;

		glEnable(GL_MAP2_VERTEX_3);
		glDisable(GL_AUTO_NORMAL);
		glMapGrid2d(u_count, 0.0, 1.0, v_count, 0.0, 1.0);

		GLdouble patch_points[2 * 2 * 3];
		for(k3d::mesh::bilinear_patches_t::const_iterator patch = Begin; patch != End; ++patch)
		{
			if(!Predicate(*patch))
				continue;

			GLdouble* pp = patch_points;
			for(k3d::bilinear_patch::control_points_t::const_iterator control_point = (*patch)->control_points.begin(); control_point != (*patch)->control_points.end(); ++control_point)
			{
				return_if_fail(*control_point);

				const k3d::point3& v = (*control_point)->position;

				*pp++ = v[0];
				*pp++ = v[1];
				*pp++ = v[2];
			}

			glMap2d(GL_MAP2_VERTEX_3, 0, 1, u_stride, u_order, 0, 1, v_stride, v_order, &patch_points[0]);

			glEvalMesh2(GL_LINE, 0, 0, 0, v_count);
			glEvalMesh2(GL_LINE, u_count, u_count, 0, v_count);
			glEvalMesh2(GL_LINE, 0, u_count, 0, 0);
			glEvalMesh2(GL_LINE, 0, u_count, v_count, v_count);
		}
	}

	template<typename predicate_t>
	void draw_bilinear_patches(const k3d::mesh::bilinear_patches_t::const_iterator Begin, const k3d::mesh::bilinear_patches_t::const_iterator End, const bool TwoSided, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;
		glEnable(GL_LIGHTING);

		const unsigned int u_count = 10;
		const unsigned int v_count = 10;
		const GLint u_order = 2;
		const GLint v_order = 2;
		const GLint u_stride = 3;
		const GLint v_stride = 2 * u_stride;

		glFrontFace(GL_CCW);
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		if(TwoSided)
			glDisable(GL_CULL_FACE);
		else
			glEnable(GL_CULL_FACE);

		glEnable(GL_MAP2_VERTEX_3);
		glEnable(GL_AUTO_NORMAL);
		glMapGrid2d(u_count, 0.0, 1.0, v_count, 0.0, 1.0);

		GLdouble patch_points[2 * 2 * 3];
		for(k3d::mesh::bilinear_patches_t::const_iterator patch = Begin; patch != End; ++patch)
		{
			if(!Predicate(*patch))
				continue;

			k3d::gl::setup_material((*patch)->material);

			set_material_color(Color);

			GLdouble* pp = patch_points;
			for(k3d::bilinear_patch::control_points_t::const_iterator control_point = (*patch)->control_points.begin(); control_point != (*patch)->control_points.end(); ++control_point)
			{
				return_if_fail(*control_point);

				const k3d::point3& v = (*control_point)->position;

				*pp++ = v[0];
				*pp++ = v[1];
				*pp++ = v[2];
			}

			glMap2d(GL_MAP2_VERTEX_3, 0, 1, u_stride, u_order, 0, 1, v_stride, v_order, &patch_points[0]);
			glEvalMesh2(GL_FILL, 0, u_count, 0, v_count);
		}
	}

	void select_bilinear_patches(const k3d::mesh::bilinear_patches_t::const_iterator Begin, const k3d::mesh::bilinear_patches_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;
		glDisable(GL_LIGHTING);

		const unsigned int u_count = 10;
		const unsigned int v_count = 10;
		const GLint u_order = 2;
		const GLint v_order = 2;
		const GLint u_stride = 3;
		const GLint v_stride = 2 * u_stride;

		glFrontFace(GL_CCW);
		glDisable(GL_CULL_FACE);
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

		glEnable(GL_MAP2_VERTEX_3);
		glDisable(GL_AUTO_NORMAL);
		glMapGrid2d(u_count, 0.0, 1.0, v_count, 0.0, 1.0);

		GLdouble patch_points[2 * 2 * 3];
		k3d::selection::id patch_id = 0;
		for(k3d::mesh::bilinear_patches_t::const_iterator patch = Begin; patch != End; ++patch, ++patch_id)
		{
			k3d::gl::push_selection_token(k3d::selection::BILINEAR_PATCH, patch_id);

			GLdouble* pp = patch_points;
			for(k3d::bilinear_patch::control_points_t::const_iterator control_point = (*patch)->control_points.begin(); control_point != (*patch)->control_points.end(); ++control_point)
			{
				return_if_fail(*control_point);

				const k3d::point3& v = (*control_point)->position;

				*pp++ = v[0];
				*pp++ = v[1];
				*pp++ = v[2];
			}

			glMap2d(GL_MAP2_VERTEX_3, 0, 1, u_stride, u_order, 0, 1, v_stride, v_order, &patch_points[0]);
			glEvalMesh2(GL_FILL, 0, u_count, 0, v_count);

			k3d::gl::pop_selection_token();
		}
	}

	template<typename predicate_t>
	void draw_bicubic_patch_edges(const k3d::mesh::bicubic_patches_t::const_iterator Begin, const k3d::mesh::bicubic_patches_t::const_iterator End, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		glColor3d(Color.red, Color.green, Color.blue);

		const unsigned int u_count = 8;
		const unsigned int v_count = 8;
		const GLint u_order = 4;
		const GLint v_order = 4;
		const GLint u_stride = 3;
		const GLint v_stride = 4 * u_stride;

		glEnable(GL_MAP2_VERTEX_3);
		glDisable(GL_AUTO_NORMAL);
		glMapGrid2d(u_count, 0.0, 1.0, v_count, 0.0, 1.0);

		GLdouble patch_points[4 * 4 * 3];
		for(k3d::mesh::bicubic_patches_t::const_iterator patch = Begin; patch != End; ++patch)
		{
			if(!Predicate(*patch))
				continue;

			GLdouble* pp = patch_points;
			for(k3d::bicubic_patch::control_points_t::const_iterator control_point = (*patch)->control_points.begin(); control_point != (*patch)->control_points.end(); ++control_point)
			{
				return_if_fail(*control_point);

				const k3d::point3& v = (*control_point)->position;

				*pp++ = v[0];
				*pp++ = v[1];
				*pp++ = v[2];
			}

			glMap2d(GL_MAP2_VERTEX_3, 0, 1, u_stride, u_order, 0, 1, v_stride, v_order, patch_points);

			glEvalMesh2(GL_LINE, 0, 0, 0, v_count);
			glEvalMesh2(GL_LINE, u_count, u_count, 0, v_count);
			glEvalMesh2(GL_LINE, 0, u_count, 0, 0);
			glEvalMesh2(GL_LINE, 0, u_count, v_count, v_count);
		}
	}

	template<typename predicate_t>
	void draw_bicubic_patches(const k3d::mesh::bicubic_patches_t::const_iterator Begin, const k3d::mesh::bicubic_patches_t::const_iterator End, const bool TwoSided, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;
		glEnable(GL_LIGHTING);

		const unsigned int u_count = 5;
		const unsigned int v_count = 5;
		const GLint u_order = 4;
		const GLint v_order = 4;
		const GLint u_stride = 3;
		const GLint v_stride = 4 * u_stride;

		glFrontFace(GL_CCW);
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		if(TwoSided)
			glDisable(GL_CULL_FACE);
		else
			glEnable(GL_CULL_FACE);

		glEnable(GL_MAP2_VERTEX_3);
		glEnable(GL_AUTO_NORMAL);
		glMapGrid2d(u_count, 0.0, 1.0, v_count, 0.0, 1.0);

		GLdouble patch_points[4 * 4 * 3];
		for(k3d::mesh::bicubic_patches_t::const_iterator patch = Begin; patch != End; ++patch)
		{
			if(!Predicate(*patch))
				continue;

			k3d::gl::setup_material((*patch)->material);

			set_material_color(Color);

			GLdouble* pp = patch_points;
			for(k3d::bicubic_patch::control_points_t::const_iterator control_point = (*patch)->control_points.begin(); control_point != (*patch)->control_points.end(); ++control_point)
			{
				return_if_fail(*control_point);

				const k3d::point3& v = (*control_point)->position;

				*pp++ = v[0];
				*pp++ = v[1];
				*pp++ = v[2];
			}

			glMap2d(GL_MAP2_VERTEX_3, 0, 1, u_stride, u_order, 0, 1, v_stride, v_order, &patch_points[0]);
			glEvalMesh2(GL_FILL, 0, u_count, 0, v_count);
		}
	}

	void select_bicubic_patches(const k3d::mesh::bicubic_patches_t::const_iterator Begin, const k3d::mesh::bicubic_patches_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;
		glDisable(GL_LIGHTING);

		const unsigned int u_count = 8;
		const unsigned int v_count = 8;
		const GLint u_order = 4;
		const GLint v_order = 4;
		const GLint u_stride = 3;
		const GLint v_stride = 4 * u_stride;

		glFrontFace(GL_CCW);
		glDisable(GL_CULL_FACE);
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

		glEnable(GL_MAP2_VERTEX_3);
		glDisable(GL_AUTO_NORMAL);
		glMapGrid2d(u_count, 0.0, 1.0, v_count, 0.0, 1.0);

		GLdouble patch_points[4 * 4 * 3];
		k3d::selection::id patch_id = 0;
		for(k3d::mesh::bicubic_patches_t::const_iterator patch = Begin; patch != End; ++patch, ++patch_id)
		{
			k3d::gl::push_selection_token(k3d::selection::BICUBIC_PATCH, patch_id);

			GLdouble* pp = patch_points;
			for(k3d::bicubic_patch::control_points_t::const_iterator control_point = (*patch)->control_points.begin(); control_point != (*patch)->control_points.end(); ++control_point)
			{
				return_if_fail(*control_point);

				const k3d::point3& v = (*control_point)->position;

				*pp++ = v[0];
				*pp++ = v[1];
				*pp++ = v[2];
			}

			glMap2d(GL_MAP2_VERTEX_3, 0, 1, u_stride, u_order, 0, 1, v_stride, v_order, &patch_points[0]);
			glEvalMesh2(GL_FILL, 0, u_count, 0, v_count);

			k3d::gl::pop_selection_token();
		}
	}

	template<typename predicate_t>
	void draw_nupatch_edges(const nurbs_renderer_t Nurbs, const k3d::mesh::nupatches_t::const_iterator Begin, const k3d::mesh::nupatches_t::const_iterator End, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		glColor3d(Color.red, Color.green, Color.blue);

		gluNurbsProperty(Nurbs, GLU_DISPLAY_MODE, GLU_OUTLINE_PATCH);

		for(k3d::mesh::nupatches_t::const_iterator nupatch = Begin; nupatch != End; ++nupatch)
		{
			if(!Predicate(*nupatch))
				continue;

			render_nupatch(Nurbs, **nupatch);
		}
	}

	template<typename predicate_t>
	void draw_nupatches(const nurbs_renderer_t Nurbs, const k3d::mesh::nupatches_t::const_iterator Begin, const k3d::mesh::nupatches_t::const_iterator End, const bool TwoSided, const k3d::color& Color, predicate_t Predicate)
	{
		k3d::gl::store_attributes attributes;

		glEnable(GL_LIGHTING);
		glEnable(GL_AUTO_NORMAL);

		if(TwoSided)
			glDisable(GL_CULL_FACE);
		else
			glEnable(GL_CULL_FACE);

		glPolygonOffset(1.0, 1.0);
		glEnable(GL_POLYGON_OFFSET_FILL);

		gluNurbsProperty(Nurbs, GLU_DISPLAY_MODE, GLU_FILL);

		for(k3d::mesh::nupatches_t::const_iterator nupatch = Begin; nupatch != End; ++nupatch)
		{
			if(!Predicate(*nupatch))
				continue;

			k3d::gl::setup_material((*nupatch)->material);

			set_material_color(Color);
			render_nupatch(Nurbs, **nupatch);
		}

		glDisable(GL_POLYGON_OFFSET_FILL);
	}

	void select_nupatches(const nurbs_renderer_t Nurbs, const k3d::mesh::nupatches_t::const_iterator Begin, const k3d::mesh::nupatches_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		glDisable(GL_AUTO_NORMAL);
		glDisable(GL_CULL_FACE);

		gluNurbsProperty(Nurbs, GLU_DISPLAY_MODE, GLU_FILL);

		k3d::selection::id patch_id = 0;
		for(k3d::mesh::nupatches_t::const_iterator nupatch = Begin; nupatch != End; ++nupatch, ++patch_id)
		{
			k3d::gl::push_selection_token(k3d::selection::NUPATCH, patch_id);
			render_nupatch(Nurbs, **nupatch);
			k3d::gl::pop_selection_token();
		}
	}

	void render_nupatch(const nurbs_renderer_t Nurbs, const k3d::nupatch& Patch)
	{
		const unsigned int u_control_points_count = Patch.u_knots.size() - Patch.u_order;
		const unsigned int v_control_points_count = Patch.v_knots.size() - Patch.v_order;

		assert_warning(u_control_points_count * v_control_points_count == Patch.control_points.size());

		std::vector<GLfloat> gl_u_knot_vector(Patch.u_knots.begin(), Patch.u_knots.end());
		std::vector<GLfloat> gl_v_knot_vector(Patch.v_knots.begin(), Patch.v_knots.end());

		const GLint gl_u_stride = 4;
		const GLint gl_v_stride = gl_u_stride * u_control_points_count;

		const k3d::nupatch::control_points_t& control_points = Patch.control_points;
		std::vector<GLfloat> gl_control_points;
		gl_control_points.reserve(4 * control_points.size());
		for(unsigned int i = 0; i != control_points.size(); ++i)
		{
			gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight * control_points[i].position->position[0]));
			gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight * control_points[i].position->position[1]));
			gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight * control_points[i].position->position[2]));
			gl_control_points.push_back(static_cast<GLfloat>(control_points[i].weight));
		}

		gluBeginSurface(Nurbs);

		gluNurbsSurface(Nurbs,
			gl_u_knot_vector.size(),
			&gl_u_knot_vector[0],
			gl_v_knot_vector.size(),
			&gl_v_knot_vector[0],
			gl_u_stride,
			gl_v_stride,
			&gl_control_points[0],
			Patch.u_order,
			Patch.v_order,
			GL_MAP2_VERTEX_4);

		gluEndSurface(Nurbs);
	}

	void draw_blobbies(const k3d::mesh::blobbies_t::const_iterator Begin, const k3d::mesh::blobbies_t::const_iterator End, const bool SelectionState, const k3d::color& Color)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);
		glColor3d(Color.red, Color.green, Color.blue);

		detail::draw_blobby draw_blobby;
		for(k3d::mesh::blobbies_t::const_iterator blobby = Begin; blobby != End; ++blobby)
			(*blobby)->accept(draw_blobby);

		// Offset solid geometry, so it's cleanly visible ...
		glEnable(GL_POLYGON_OFFSET_FILL);
		glPolygonOffset(1.0, 1.0);

		// Solid drawing ...
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		glEnable(GL_LIGHTING);

		if(m_show_blobby_surface.value())
		{
			for(k3d::mesh::blobbies_t::const_iterator blobby = Begin; blobby != End; ++blobby)
				render_blobby_surface(*blobby);
		}
	}

	void select_blobbies(const k3d::mesh::blobbies_t::const_iterator Begin, const k3d::mesh::blobbies_t::const_iterator End)
	{
		k3d::gl::store_attributes attributes;

		glDisable(GL_LIGHTING);

		detail::select_blobby select_blobby;
		for(k3d::mesh::blobbies_t::const_iterator blobby = Begin; blobby != End; ++blobby)
			(*blobby)->accept(select_blobby);

		glFrontFace(GL_CW);
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		glDisable(GL_CULL_FACE);

		if(m_show_blobby_surface.value())
		{
			for(k3d::mesh::blobbies_t::const_iterator blobby = Begin; blobby != End; ++blobby)
			{
//				k3d::gl::push_name((*blobby)->root);
				render_blobby_surface(*blobby);
//				k3d::gl::pop_name();
			}
		}
	}

	void render_blobby_surface(k3d::blobby* Opcode)
	{
		return_if_fail(m_cache.get());

		// Get surface number
		cache::blobby_cache_map_t::const_iterator surface = m_cache->blobby_cache_map.find(Opcode);
		if(surface == m_cache->blobby_cache_map.end())
			return;

		unsigned long surface_number = surface->second;

		// Output cached surface
		const detail::vertices_t& blobby_vertices = m_cache->blobby_surfaces_vertices[surface_number];
		const detail::vertices_t& blobby_normals = m_cache->blobby_surfaces_normals[surface_number];
		const detail::polygons_t& blobby_polygons = m_cache->blobby_surfaces_polygons[surface_number];
		for(unsigned long p = 0; p < blobby_polygons.size(); ++p)
		{
			detail::polygon_t path = blobby_polygons[p];

			// Calculate normal
			unsigned long pointcount = path.size();
			if(pointcount < 3)
				continue;

			// Draw polygon
			glBegin(GL_POLYGON);
			for(unsigned long i = 0; i < path.size(); ++i)
			{
				// Invert normal (Blobby field decreases from in to out)
				k3d::point3 normal = -blobby_normals[path[i]];
				k3d::gl::normal3d(k3d::to_vector(normal));
				k3d::point3 vertex = blobby_vertices[path[i]];
				k3d::gl::vertex3d(vertex);
			}
			glEnd();
		}
	}

	/// Enumerates supported polyhedron styles
	typedef enum
	{
		/// Render input polyhedra as-is
		DEFAULT,
		/// Render input polyhedra as polygons, regardless of type
		POLYGONS,
		/// Render input polyhedra as subdivision surfaces, regardless of type
		CATMULL_CLARK,
	} polyhedron_render_t;

	friend std::ostream& operator << (std::ostream& Stream, const polyhedron_render_t& Value)
	{
		switch(Value)
		{
			case DEFAULT:
				Stream << "default";
				break;
			case POLYGONS:
				Stream << "polygons";
				break;
			case CATMULL_CLARK:
				Stream << "catmull-clark";
				break;
		}
		return Stream;
	}

	friend std::istream& operator >> (std::istream& Stream, polyhedron_render_t& Value)
	{
		std::string text;
		Stream >> text;

		if(text == "default")
			Value = DEFAULT;
		else if(text == "polygons")
			Value = POLYGONS;
		else if(text == "catmull-clark")
			Value = CATMULL_CLARK;
		else
			k3d::log() << error << k3d_file_reference << ": unknown enumeration [" << text << "]" << std::endl;

		return Stream;
	}

	static const k3d::ienumeration_property::enumeration_values_t& polyhedron_render_values()
	{
		static k3d::ienumeration_property::enumeration_values_t values;
		if(values.empty())
		{
			values.push_back(k3d::ienumeration_property::enumeration_value_t(_("Default"), "default", _("Render objects as-is")));
			values.push_back(k3d::ienumeration_property::enumeration_value_t(_("Polygons"), "polygons", _("Forces rendering as polygons")));
			values.push_back(k3d::ienumeration_property::enumeration_value_t(_("Catmull-Clark"), "catmull-clark", _("Forces rendering as subdivision surfaces")));
		}

		return values;
	}

	/// Enumerates proxy behavior
	typedef enum
	{
		/// Preview input geometry normally
		PROXY_NONE,
		/// Preview input geometry as a point
		PROXY_POINT,
		/// Preview input geometry as a bounding-box
		PROXY_BOUNDING_BOX,
	} proxy_t;

	friend std::ostream& operator << (std::ostream& Stream, const proxy_t& Value)
	{
		switch(Value)
		{
			case PROXY_NONE:
				Stream << "none";
				break;
			case PROXY_POINT:
				Stream << "point";
				break;
			case PROXY_BOUNDING_BOX:
				Stream << "bounding-box";
				break;
		}
		return Stream;
	}

	friend std::istream& operator >> (std::istream& Stream, proxy_t& Value)
	{
		std::string text;
		Stream >> text;

		if(text == "none")
			Value = PROXY_NONE;
		else if(text == "point")
			Value = PROXY_POINT;
		else if(text == "bounding-box")
			Value = PROXY_BOUNDING_BOX;
		else
			k3d::log() << error << k3d_file_reference << ": unknown enumeration [" << text << "]" << std::endl;

		return Stream;
	}

	static const k3d::ienumeration_property::enumeration_values_t& proxy_values()
	{
		static k3d::ienumeration_property::enumeration_values_t values;
		if(values.empty())
		{
			values.push_back(k3d::ienumeration_property::enumeration_value_t(_("None"), "none", _("Preview mesh normally")));
			values.push_back(k3d::ienumeration_property::enumeration_value_t(_("Point"), "point", _("Preview mesh using a single point")));
			values.push_back(k3d::ienumeration_property::enumeration_value_t(_("Bounding Box"), "bounding-box", _("Preview mesh using a bounding box")));
		}

		return values;
	}

	/// Caches data for doing fast drawing
	class cache
	{
	public:
		cache(const bool TwoSidedFaces) :
			m_two_sided_faces(TwoSidedFaces),
			m_creased_mesh(0),
			selection_display_list(0),
			m_display_lists(15, 0),
			m_building_list(false),
			m_activated(true)
		{
		}

		~cache()
		{
			delete m_creased_mesh;

			for(display_lists_t::iterator list = m_display_lists.begin(); list != m_display_lists.end(); ++list)
			{
				if(*list)
					glDeleteLists(*list, 1);
			}

			if(selection_display_list)
				glDeleteLists(selection_display_list, 1);
		}

		void activate_display_lists()
		{
			m_activated = true;
		}

		void deactivate_display_lists()
		{
			m_activated = false;
		}

		void end_list()
		{
			if(!m_activated)
				return;

			if(m_building_list)
				glEndList();

			m_building_list = false;
		}

		bool start_points_list() { return start_list(1); }
		bool start_linear_curves_list() { return start_list(2); }
		bool start_cubic_curves_list() { return start_list(3); }
		bool start_nurbs_curves_list() { return start_list(4); }
		bool start_edges_list() { return start_list(5); }
		bool start_bilinear_edges_list() { return start_list(6); }
		bool start_bicubic_edges_list() { return start_list(7); }
		bool start_nurbs_edges_list() { return start_list(8); }

		bool start_faces_list(const bool TwoSided)
		{
			if(TwoSided != m_two_sided_faces)
			{
				m_two_sided_faces = TwoSided;
				reset_list(9);
			}

			return start_list(9);
		}

		bool start_bilinear_patches_list(const bool TwoSided)
		{
			if(TwoSided != m_two_sided_faces)
			{
				m_two_sided_faces = TwoSided;
				reset_list(10);
			}

			return start_list(10);
		}

		bool start_bicubic_patches_list(const bool TwoSided)
		{
			if(TwoSided != m_two_sided_faces)
			{
				m_two_sided_faces = TwoSided;
				reset_list(11);
			}

			return start_list(11);
		}

		bool start_nurbs_patches_list(const bool TwoSided)
		{
			if(TwoSided != m_two_sided_faces)
			{
				m_two_sided_faces = TwoSided;
				reset_list(12);
			}

			return start_list(12);
		}

		bool start_blobbies_list() { return start_list(13); }

		/// Blobby polygonized surfaces caching variables
		typedef std::map<k3d::blobby*, unsigned long> blobby_cache_map_t;
		blobby_cache_map_t blobby_cache_map;
		typedef std::vector<detail::vertices_t> blobby_surfaces_vertices_t;
		blobby_surfaces_vertices_t blobby_surfaces_vertices;
		blobby_surfaces_vertices_t blobby_surfaces_normals;
		typedef std::vector<detail::polygons_t> blobby_surfaces_polygons_t;
		blobby_surfaces_polygons_t blobby_surfaces_polygons;

		/// Two sided faces
		bool m_two_sided_faces;

		/// SDS creased mesh cache
		k3d::mesh* m_creased_mesh;

		/// OpenGL display list used for performance during selection
		GLuint selection_display_list;

	private:
		/// OpenGL display lists used for performance during drawing
		typedef std::vector<GLuint> display_lists_t;
		display_lists_t m_display_lists;

		/// Starts an OpenGL list, returns true when the cache's empty
		bool start_list(const unsigned long N)
		{
			if(!m_activated)
				return true;

			return_val_if_fail(N < m_display_lists.size(), false);

			// Draw existing list
			if(m_display_lists[N])
			{
				glCallList(m_display_lists[N]);
				return false;
			}

			return_val_if_fail(!m_building_list, false);

			// Create a new list
			m_display_lists[N] = glGenLists(N);
			glNewList(m_display_lists[N], GL_COMPILE_AND_EXECUTE);

			// Mark the fact we're building a list
			m_building_list = true;

			return true;
		}

		void reset_list(const unsigned long N)
		{
			if(m_display_lists[N])
				glDeleteLists(N, 1);

			m_display_lists[N] = 0;
		}

		bool m_building_list;

		/// Available for testing display lists
		bool m_activated;
	};
	/// Stores cached data for doing fast drawing
	std::auto_ptr<cache> m_cache;

	/// SDS cache
	k3d::sds::k3d_opengl_sds_cache m_sds_cache;

	/// Use OpenGL display lists for performance
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_display_lists;
	/// Blobby preview type
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_show_blobby_surface;
	/// Controls how polyhedra will be rendered
	k3d_data(polyhedron_render_t, immutable_name, change_signal, with_undo, local_storage, no_constraint, enumeration_property, with_serialization) m_polyhedron_render_type;
	/// Controls how mesh data will be previewed
	k3d_data(proxy_t, immutable_name, change_signal, with_undo, local_storage, no_constraint, enumeration_property, with_serialization) m_proxy_type;
	/// Preview subdivision surfaces
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_preview_sds;
	/// Experimental NURBS SDS preview
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_nurbs_sds;
	/// Show SDS creases
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_sds_crease;
	/// Show SDS cage projection
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_sds_borders;
	/// SDS preview level
	k3d_data(int, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_sds_level;
	/// SDS render level
	k3d_data(int, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_sds_render_level;
	/// Stores the color used to draw unselected geometry
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_color;
	/// Stores the color used to draw selected geometry
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_selected_color;
	/// Store whether to draw components
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_points;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_edges;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_faces;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_linear_curves;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_cubic_curves;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_nucurves;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_bilinear_patches;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_bicubic_patches;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_nupatches;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_blobbies;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_draw_two_sided;
	k3d_data(k3d::mesh*, k3d::data::immutable_name, k3d::data::change_signal, k3d::data::no_undo, k3d::data::demand_storage, k3d::data::no_constraint, k3d::data::read_only_property, k3d::data::no_serialization) m_transformed_output_mesh;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_show_component_selection;
};

/////////////////////////////////////////////////////////////////////////////
// mesh_instance_factory

k3d::iplugin_factory& mesh_instance_factory()
{
	return mesh_instance::get_factory();
}

} // namespace libk3dmesh

