/***************************************************************************
                          commands.h  -  description
                             -------------------
    begin                : Don Dez 19 2002
    copyright            : (C) 2002 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef COMMANDS_H
#define COMMANDS_H

#include <kcommand.h>
#include <klocale.h>

#include "mycanvasitem.h"

struct barcodeData;
class DrawingRect;
class MyCanvasLine;
class MyCanvasRectangle;
class MyCanvasView;
class CanvasBarcode;
class PictureRectangle;
class QCanvasItem;
class QColor;
class QFont;
class QImage;
class QPen;
class QPixmap;
class QPoint;
class QString;
// Stuff for undo redo

class CommandUtils {
    public:
        CommandUtils() { }
        ~CommandUtils() { }

        void setCanvas( QCanvas* canvas ) {
            c = canvas;
        }
        
        bool canvasHasItem( QCanvasItem* item );

    protected:
        QCanvas* c;
};

/**
  * NewItemCommand is the base class for all classes
  * that create a new item in the label editor (e.g. a
  * barcode or a text element).
  * You have to implement void create() which creates
  * a QCanvasItem in item.
  * NewItemCommand takes care about possitioning and
  * undo/redo (because of KCommand).
  *
  * @author Dominik Seichter
  */
class NewItemCommand : public KCommand {
    public:
        NewItemCommand( MyCanvasView* view, const QString & name );
        ~NewItemCommand();

        void execute();
        void unexecute();
        QString name() const {
            return m_name;
        };

        /** returns a pointer to the QCanvasItem created by this class */
        QCanvasItem* createdItem() const { return item; }
        
    protected:
        /** This function has to be reimplemented in all subclasses
          * and has to create a QCanvasItem and store a pointer to it
          * in item. Otherwise KBarcode is going to crash.
          *
          * Example:
          *
          * <pre>
          * void create() {
          *     DrawingRect* r = new DrawingRect( 0 ); // Drawing rect is a subclass of QCanvasItem
          *     r->setCircle( m_circle );   // Do something with r
          *     item = r;   // save r into item, so that NewItemCommand knows about it
          * }
          * </pre>
          */
        virtual void create() = 0;
        
        MyCanvasView* cv;
        QCanvasItem* item;
        QPoint m_point;
        QString m_name;    
};

class ResizeCommand : public KCommand, CommandUtils {
    public:
        ResizeCommand( MyCanvasRectangle* it, bool shift = false ) {
            item = it;
            orect = rect = item->boundingRect();
            m_shift = shift;
            setCanvas( item->canvas() );
        }
        ~ResizeCommand() {}

        void setRect( int cx, int cy, int cw, int ch );
        
        void execute();
        void unexecute();
        QString name() const {
            return i18n("Resized Item");
        }

    protected:
        QRect orect;
        QRect rect;
        MyCanvasRectangle* item;
        bool m_shift;
};

class MoveCommand : public KCommand, CommandUtils {
    public:
        MoveCommand( int cx, int cy, MyCanvasRectangle* it ) {
            x = cx;
            y = cy;
            item = it;
            setCanvas( item->canvas() );
        }
        ~MoveCommand() {}

        void execute();
        void unexecute();
        QString name() const {
            return i18n("Moved Item");
        }

    protected:
        int x;
        int y;
        MyCanvasRectangle* item;
};

class LineMoveCommand : public KCommand, CommandUtils {
    public:
        LineMoveCommand( QPoint ca, QPoint cb, MyCanvasLine* l ) {
            line = l;
            a = ca;
            b = cb;
            setCanvas( line->canvas() );
        }

        ~LineMoveCommand() {}

        void execute();
        void unexecute();
        QString name() const {
            return i18n("Moved Line");
        }

    protected:
        QPoint a, b;
        // old coordinates, before first move
        QPoint a2, b2;
        MyCanvasLine* line;
};

class PictureCommand : public KCommand, CommandUtils {
    public:
        PictureCommand( double r, double ro, MyCanvasRectangle* it ) {
            rotate = r;
            orotate = ro;
            item = it;
            setCanvas( item->canvas() );
        }

        ~PictureCommand() {}

        void execute();
        void unexecute();
        QString name() const {
            return i18n("Changed Settings");
        }
    protected:
        double rotate, orotate;
        MyCanvasRectangle* item;
};

class TextChangeCommand : public KCommand, CommandUtils {
    public:
        TextChangeCommand( CanvasText* it, QString t ) {
            item = it;
            text = t;
            oldtext = item->text();
            setCanvas( item->canvas() );
        }
        ~TextChangeCommand() { }

        void execute();
        void unexecute();
        QString name() const {
            return i18n("Changed Text");
        }
    protected:
        QString oldtext, text;
        CanvasText* item;
};

class BarcodeCommand : public KCommand, CommandUtils {
    public:
        BarcodeCommand( CanvasBarcode* bcode, barcodeData* d ) {
            item = bcode;
            olddata = bcode->data();
            data = d;
            setCanvas( item->canvas() );
        }
        ~BarcodeCommand() {
            delete data;
        }

        void execute();
        void unexecute();
        QString name() const {
            return i18n("Changed Barcode");
        }
    protected:
        barcodeData olddata;
        barcodeData* data;
        CanvasBarcode* item;
};

class NewPictureCommand : public NewItemCommand {
    public:
        NewPictureCommand( QImage* img, MyCanvasView* v );
        ~NewPictureCommand() {
            delete image;
        }

    protected:
        void create();
        QImage* image;
};

class LabelEditor;
class NewTextCommand : public NewItemCommand {
    public:
        NewTextCommand( QString t, MyCanvasView* v, LabelEditor* editor );
        ~NewTextCommand() { }

    protected:
        void create();
        QString text;
        LabelEditor* m_editor;
};

class NewRectCommand : public NewItemCommand {
    public:
        NewRectCommand( MyCanvasView* v, bool circle = false );
        ~NewRectCommand() { }

    protected:
        void create();
        bool m_circle;
};

class NewLineCommand : public NewItemCommand {
    public:
        NewLineCommand( MyCanvasView* v );
        ~NewLineCommand() { }

    protected:
        void create();
};

class NewBarcodeCommand : public NewItemCommand {
    public:
        NewBarcodeCommand( MyCanvasView* v );
        ~NewBarcodeCommand() { }

    protected:
        void create();
};

class DeleteCommand : public KCommand, CommandUtils {
    public:
        DeleteCommand( QCanvasItem* it ) {
            item = it;
            c = item->canvas();
            setCanvas( c );            
        }
        ~DeleteCommand(){
            if( item->canvas() == 0 )
                delete item;
        }

        void execute();
        void unexecute();
        QString name() const {
            return i18n("Delete Item");
        }
    protected:
        QCanvasItem* item;
        QCanvas* c;
};

class RectCommand : public KCommand, CommandUtils {
    public:
        RectCommand( QColor c, QColor b, int bWidth, int pstyle, DrawingRect* r ) {
            fill = c;
            border = b;
            borderwidth = bWidth;
            penstyle = pstyle;
            rect = r;
            setCanvas( r->canvas() );            
        }
        ~RectCommand() { }

        void execute();
        void unexecute();
        QString name() const {
            return i18n("Modified Rectangle or Ellipse");
        }

    protected:
        QColor fill, fill2;
        QColor border, border2;
        int borderwidth, borderwidth2;
        int penstyle, penstyle2;
        DrawingRect* rect;
};

class LineCommand : public KCommand, CommandUtils {
    public:
        LineCommand( QPen p, MyCanvasLine* l ) {
            pen = p;
            line = l;
            setCanvas( line->canvas() );
        }
        ~LineCommand() { }

        void execute();
        void unexecute();
        QString name() const {
            return i18n("Modified Line");
        }

    protected:
        QPen pen, pen2;
        MyCanvasLine* line;
};

#endif
