/***************************************************************************
 *   Copyright (C) 2004-2009 by Thomas Fischer                             *
 *   fischer@unix-ag.uni-kl.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <qfile.h>
#include <qapplication.h>
#include <qspinbox.h>
#include <qregexp.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qlayout.h>

#include <klocale.h>
#include <klineedit.h>
#include <kpushbutton.h>
#include <kiconloader.h>
#include <kcombobox.h>
#include <kmessagebox.h>
#include <kio/netaccess.h>
#include <kurl.h>

#include <settings.h>
#include <fileimporterbibtex.h>
#include "webquerydblp.h"

namespace KBibTeX
{
    WebQueryDBLPWidget::WebQueryDBLPWidget( QWidget *parent, const char *name )
            : WebQueryWidget( parent, name )
    {
        init();

        Settings *settings = Settings::self();
        QString value = settings->getWebQueryDefault( "DBLP_query" );
        value = value == QString::null ? "" : value;
        lineEditQuery->setText( value );
        slotTextChanged( value, true );
        value = settings->getWebQueryDefault( "DBLP_keepSeparate" );
        value = value == QString::null || value.isEmpty() ? "0" : value;
        checkBoxKeepEntriesSeparate->setChecked( value == "1" );
    }

    void WebQueryDBLPWidget::init()
    {
        QVBoxLayout *vLayout = new QVBoxLayout( this, 0, KDialog::spacingHint() );

        QHBoxLayout *hLayout = new QHBoxLayout( );
        vLayout->addLayout( hLayout );

        KPushButton *clearSearchText = new KPushButton( this );
        clearSearchText->setIconSet( QIconSet( SmallIcon( "locationbar_erase" ) ) );
        hLayout->addWidget( clearSearchText );
        QLabel *label = new QLabel( i18n( "Search &term:" ), this );
        hLayout->addWidget( label );
        lineEditQuery = new KLineEdit( this );
        hLayout->addWidget( lineEditQuery );
        label->setBuddy( lineEditQuery );
        hLayout->addSpacing( KDialog::spacingHint() * 2 );
        connect( clearSearchText, SIGNAL( clicked() ), lineEditQuery, SLOT( clear() ) );
        connect( lineEditQuery, SIGNAL( textChanged( const QString& ) ), this, SLOT( slotTextChanged( const QString& ) ) );
        hLayout->setStretchFactor( lineEditQuery, 4 );
        KCompletion *completionQuery = lineEditQuery->completionObject();

        label = new QLabel( i18n( "&Number of results:" ), this );
        hLayout->addWidget( label );
        spinBoxMaxHits = new QSpinBox( 1, 250, 1, this );
        spinBoxMaxHits->setValue( 10 );
        hLayout->addWidget( spinBoxMaxHits );
        label->setBuddy( spinBoxMaxHits );

        hLayout = new QHBoxLayout( );
        vLayout->addLayout( hLayout );

        checkBoxKeepEntriesSeparate = new QCheckBox( i18n( "Do not merge corresponding entries" ), this );
        hLayout->addWidget( checkBoxKeepEntriesSeparate );

        vLayout->addStretch( 0 );

        connect( lineEditQuery, SIGNAL( returnPressed() ), this, SIGNAL( startSearch() ) );
        connect( lineEditQuery, SIGNAL( returnPressed( const QString& ) ), completionQuery, SLOT( addItem( const QString& ) ) );
    }

    WebQueryDBLP::WebQueryDBLP( QWidget* parent ): WebQuery( parent )
    {
        m_widget = new WebQueryDBLPWidget( parent );
    }

    WebQueryDBLP::~WebQueryDBLP()
    {
        delete m_widget;
    }

    QString WebQueryDBLP::title()
    {
        return i18n( "DBLP" );
    }

    QString WebQueryDBLP::disclaimer()
    {
        return i18n( "DBLP - Copyright" );
    }

    QString WebQueryDBLP::disclaimerURL()
    {
        return "http://www.informatik.uni-trier.de/~ley/db/copyright.html";
    }

    WebQueryWidget *WebQueryDBLP::widget()
    {
        return m_widget;
    }

    void WebQueryDBLP::query()
    {
        WebQuery::query();
        Settings *settings = Settings::self();
        settings->setWebQueryDefault( "DBLP_query", m_widget->lineEditQuery->text() );
        settings->setWebQueryDefault( "DBLP_keepSeparate", m_widget->checkBoxKeepEntriesSeparate->isChecked() ? "1" : "0" );

        int numberOfResults = m_widget->spinBoxMaxHits->value();
        setNumStages( numberOfResults + 1 );

        QString searchTerm = m_widget->lineEditQuery->text().stripWhiteSpace().replace( '$', "" );
        if ( searchTerm.isEmpty() )
        {
            setEndSearch( WebQuery::statusInvalidQuery );
            return;
        }

        KURL url = KURL( QString( "http://dblp.l3s.de/?newsearch=1&q=%1&search_opt=all&synt_query_exp=full" ).arg( searchTerm.replace( "%", "%25" ).replace( "+", "%2B" ).replace( " ", "%20" ).replace( "#", "%23" ).replace( "&", "%26" ).replace( "?", "%3F" ) ) );

        QString completeText = download( url );
        if ( completeText != QString::null && !m_aborted )
        {
            QRegExp findBibTeXurl( "<a href=\"(http://dblp.uni-trier.de/rec/bibtex/[^\"]+)\"" );
            findBibTeXurl.search( completeText );
            int pos = findBibTeXurl.pos( 1 );

            int maxCount = numberOfResults;
            while ( !m_aborted && pos > -1 && ( maxCount-- ) > 0 )
            {
                KURL keyUrl = KURL( findBibTeXurl.cap( 1 ) );
                BibTeX::File *tmpBibFile = downloadBibTeXFile( keyUrl );
                if ( tmpBibFile != NULL && !m_aborted )
                {
                    if ( tmpBibFile->count() == 2 )
                    {
                        /** if the data returned from DBLP contains only two entry, check if they are InCollection and Book. In this case, one is the conference article and the other is the conference proceedings. */
                        BibTeX::Entry *firstEntry = dynamic_cast<BibTeX::Entry*>( *tmpBibFile->begin() );
                        BibTeX::Entry *secondEntry = dynamic_cast<BibTeX::Entry*>( *( ++tmpBibFile->begin() ) );

                        if ( !m_widget->checkBoxKeepEntriesSeparate->isChecked() && firstEntry != NULL && secondEntry != NULL && ( firstEntry->entryType() == BibTeX::Entry::etInProceedings || firstEntry->entryType() == BibTeX::Entry::etInCollection ) && ( secondEntry->entryType() == BibTeX::Entry::etBook || secondEntry->entryType() == BibTeX::Entry::etProceedings ) )
                        {
                            /** We merge both conference article and proceedings, as otherwise 2*n entries will be shown in the search result and it is no longer clear, which conference and with article match. */
                            BibTeX::Entry *myEntry = tmpBibFile->completeReferencedFieldsConst( firstEntry );
                            myEntry->merge( secondEntry, BibTeX::Entry::msAddNew );
                            myEntry->deleteField( BibTeX::EntryField::ftCrossRef );
                            emit foundEntry( myEntry, false );
                        }
                        else
                        {
                            emit foundEntry( new BibTeX::Entry( firstEntry ), false );
                            emit foundEntry( new BibTeX::Entry( secondEntry ), m_widget->checkBoxKeepEntriesSeparate->isChecked() );
                        }
                    }
                    else
                        for ( BibTeX::File::ElementList::iterator it = tmpBibFile->begin(); it != tmpBibFile->end(); ++it )
                        {
                            BibTeX::Entry *entry = dynamic_cast<BibTeX::Entry*>( *it );
                            if ( entry != NULL )
                                emit foundEntry( new BibTeX::Entry( entry ), false );
                        }
                }

                if ( tmpBibFile != NULL )
                    delete tmpBibFile;

                findBibTeXurl.search( completeText, pos + 1 );
                pos = findBibTeXurl.pos( 1 );
            }

            setEndSearch( WebQuery::statusSuccess );
        }
        else if ( !m_aborted )
        {
            QString message = KIO::NetAccess::lastErrorString();
            if ( message.isEmpty() )
                message.prepend( '\n' );
            message.prepend( QString( i18n( "Querying database '%1' failed." ) ).arg( title() ) );
            KMessageBox::error( m_parent, message );
            setEndSearch( WebQuery::statusError );
        }
        else
            setEndSearch( WebQuery::statusAborted );
    }

}
