/***************************************************************************
                          user.h  -  description                              
                             -------------------                                         
    begin                : Thu Jan 6 2000                                           
    copyright            : (C) 2000 by Seth Hartbecke                         
    email                : gandalf@netins.net                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#ifndef USER_H
#define USER_H

#include <qwidget.h>
#include <qobject.h>
#include <qtabdialog.h>
#include <qlayout.h>
#include <qcheckbox.h>
#include <qlistview.h>
#include <qheader.h>
#include <qlist.h>
#include <qpushbutton.h>
#include <qgroupbox.h>
#include <qlineedit.h>
#include <qstrlist.h>
#include <qlabel.h>

#include <kmessagebox.h>

#include <xmlnode.h>
#include <infoquery.h>
#include <group.h>


class UserPresence;
class UserPgGenInfo;
class UserPgGroups;
class UserPgSubscription;
/**
 * This class does two things.  First it is the user record in the user info database, that is to say
 * this is the object you query when you want to know something about a user already on the roster (are
 * they currently online or offline?).  It is also the form for showing this information to the user.
 *
 * Having both of these things in one object is nice because I can simpally call show() to show the user
 * the info I'm looking at in the program.  It is bad in that it is poor programing to have the data and
 * the user presentation code in the same place.
 *
 * @short User database record and view
 * @author Seth Hartbecke (gandalf@netins.net)
*/

class User
	: public QTabDialog, public JabberID
{
   Q_OBJECT
public:
        /**
	 * Create a new user object with the specified JabberID.  User objects are usually created
	 * by JabberCore
	 */
	User(JabberID jid);
	~User();

	// You can use this to query the presence
	// info for a specific resource
	/**
	 * Returns the number of presences of this user the client knows about
	 */
	int numberResources();
	/**
	 * Returns the pointer to the nth presence record
	 */
	UserPresence *getPresence(int n);
	/**
	 * Returns a pointer to the first presence record where the strings match or null if it can't find one
	 */
	UserPresence *getPresence(QString presenceName);
	/**
	 * Returns a pointer to the highest presence, or null if there is no active presence for this user
	 */
	UserPresence *getHighestPresence();


	enum
	{
        	subscribedTo
		, subscribedFrom
		, subscribedBoth
		, subscribedNone
		, subscribedUnknown
	};
	int getSubscriptionType();

	/**
	 * Unsubscribe from the user's presence
	 */
	void subscribe();
	/**
	 * Subscribe to the user's presence
	 */
	void unsubscribe();
	/**
	 * Returns true if we have already called subscribe but not had any responce to the request yet
	 */
	bool watingSubscribeConfirm();

	// Check to see if they are asking to subscribe
	// or unsubscribe
	/**
	 * Returns true if the user is trying to subscribe to us.
	 */
	bool askingSubscribe();
	/**
	 * Allows the user to recieve our presence information.
	 */
	void allowSubscribe();
	/**
	 * The user is asking/notifiying us that they are nolonger interested in our presence.
	 */
	bool askingUnsubscribe();
	/**
	 * Allows the user not to recieve our presence anymore (after their request).  Is this required?
	 */
	void allowUnsubscribe();

	/**
	 * Gets the user's "real" name.  The real name is what shows in the roster and really is something
	 * that the user just invents
	 */
        QString getRealName();
	/**
	 * Sets the users "real" name.
	 */
        void setRealName(QString name);

	/**
	 * Add the user to the group specified, the group will be created if it does not exist
	 */
        void addGroup( QString group );
	/**
	 * Remove the user from the group specified, if this is the last user in that group then the
	 * group will auto delete itself
	 */
        void removeGroup( QString group );
        /**
	 * A list of groups we are in.  Please don't modify this list except via the addGroup() and removeGroup() methods.
	 *
	 * @see User#addGroup
	 * @see User#removeGroup
	 */
        QStrList getGroups();

	/**
	 * This pushes any group changes you may have made back to the server.  Utill this is called, the group changes
	 * are only known client side
	 */
	void commitGroups();

signals:
        /**
	 * Information known about the user has changed.  This is usually caused by a setRealName and the roster
	 * is usually the reciever.
	 *
	 * @see User#setRealName
	 */
        void informationChanged(User *u);

	/**
	 * If this objects groups change for _ANY_ reason
	 * (server sent group list or commitGroups was called)
         * this signal is emitted
	 */
	void groupsChanged(User *u);

	/**
	 * This user is requesting a change in subscription status
	 */
	void subscriptionChangeRequest(User *u);

	/**
	 * The subscription status to this user has changed
	 */
	void subscriptionChanged(User *u);

	/**
	 * The specified presence has changed
	 */
	void presenceChanged(User *u, UserPresence *p);

	/**
	 * This is called if you have been unsubscribed
	 */
	void subscriptionCanceled(User *u);

        void addedToGroup( User *, QString group );
        void removedFromGroup( User *, QString group );

	/**
	 * Called if this user object is being deleted
	 */
        void userDeleted( User * );

public slots:
	/**
	 * The <item ...></item> portion of the roster list is passed
         * here so that the object can update itself.
	 */
	void processItemTag(XMLNode *t);

	/**
	 * Processes <presence.../> tags are given to this slot (and passed on the correct user presence)
	 */
	void processPresence(XMLNode *t);

	/**
	 * Removes the user from the roster and the user database (totally gone, only to return
	 * via a user add)
	 */
        void removeUser();

private slots:
	void RosterUpdateResults(InfoQuery *request, InfoQuery *result);
	void memberResourceChanged(User *u, UserPresence *p);
        void memberResourceDeleted(User *u, UserPresence *p);

        void saveChanges();

private:
	// This is a list of pointers to qstrings that
	// are the names of the groups the user is in
	QStrList groups;

        QList<UserPresence> presenceList;
        QString _realName;
	int subscriptionType;
	bool tryingToSubscribe;
	bool isAskingSubscribe;
	bool isAskingUnsubscribe;

	/* GUI Stuff */
	UserPgGenInfo *pageGeneral;
	UserPgGroups *pageGroups;
	UserPgSubscription *pageSubscription;
};

/**
 * General info page of the user dialog
 *
 * This page shows the user's real name (allows them to change this) and givies a list of resources
 * the user object knows about
 */
class UserPgGenInfo
	: public QWidget
{
   Q_OBJECT
public:
	UserPgGenInfo(User *u);
	~UserPgGenInfo();

        QString getNewRealname() { return txtRealName->text(); };

public slots:
	void reloadInfo(User *u, UserPresence *p);
        void removedResource(User *u, UserPresence *p);

protected:
	void resizeEvent( QResizeEvent * );

private slots:
        void startChat( QListViewItem *item );
        void newRealname( User *u ) { txtRealName->setText( u->getRealName() ); };

private:

        User *user;

	QGridLayout *layout;
	int id;
	QLabel *lblJID;		 // Says "Jabber ID"
	QLabel *txtJID;		 // Displays the acutal jabber id
	QListView *resourceList; // List all of the know resources
        QLabel *lblRealName;
        QLineEdit *txtRealName;
};

/**
 * This is the groups page in the user info dialog.
 *
 * Gives a listing of all the avilable groups and allows them to choose which of those this user
 * is to be in, also allows them to create new groups.
 */
class UserPgGroups
	: public QWidget
{
   Q_OBJECT
public:
	UserPgGroups(User *u);
	~UserPgGroups();

public slots:
	void reloadGroups(User *u);

private slots:
	void addGroup();

        void newAvailGroup( Group * );
        void removeAvailGroup( Group * );
        void addToGroup( User *, QString );
        void removedFromGroup( User *, QString );

        void groupItemClicked( QListViewItem * );

private:
	QCheckListItem *getItem( QString groupName );

	User *user;

	QGridLayout *layout;
	QListView *groups;

	QGroupBox *addBox;
	QGridLayout *addLayout;
	QPushButton *btnAdd;
	QLineEdit *txtNewGroup;

};

/**
 * This page in the user info dialog shows the subscription status and allows the user to change it
 */
class UserPgSubscription
	: public QWidget
{
   Q_OBJECT
public:
	UserPgSubscription(User *u);
	~UserPgSubscription();

public slots:
	void subscriptionChanged(User *u);
	void subscriptionChangeRequest(User *u);
	void subscriptionCanceled(User *u);

private slots:
	void fromClicked();
	void toClicked();

private:
	void refreshCheckboxes();

	User *user;

	QGridLayout *layout;
	QCheckBox *subscribedTo;
	QCheckBox *subscribedFrom;
	QLabel *subscriptionStatus;

};

#endif
