/***************************************************************************
 *   Copyright (C) 2004 Nadeem Hasan <nhasan@kde.org>                      *
 *   Copyright (C) 2004 Stefan Kombrink <katakombi@web.de>                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "config.h"

#include <sys/types.h>

# include <sys/time.h>
# include <time.h>

#include <sys/shm.h>

#include <kdebug.h>
#include <kstaticdeleter.h>
#include <synparams.h>

#include "synconfig.h"
#include "touchpad.h"

TouchPad *TouchPad::m_self = 0;
static KStaticDeleter<TouchPad> staticTouchPadDeleter;

static int finger_low[] = { 53, 38, 25, 18, 11 };

TouchPad *TouchPad::self()
{
  if ( !m_self )
  {
    staticTouchPadDeleter.setObject( m_self, new TouchPad() );
  }

  return m_self;
}

TouchPad::TouchPad()
{
}

TouchPad::~TouchPad()
{
    if (  m_self == this )
        staticTouchPadDeleter.setObject( m_self, 0, false );
}

bool TouchPad::isValid()
{
    return Pad::hasDriver();
}

int TouchPad::isEnabled()
{
    return isValid()? ( !( Pad::getParam( TOUCHPADOFF ) == 0 ) ) : false;
}

void TouchPad::setEnabled( int enable )
{
    Pad::setParam( TOUCHPADOFF, (double) enable );
}

QRect TouchPad::edges()
{
    if ( !isValid() )
        return QRect();

    return QRect( QPoint( (int)Pad::getParam( TOPEDGE ), 
                          (int)Pad::getParam( LEFTEDGE) ),
                  QPoint( (int)Pad::getParam( BOTTOMEDGE ), 
                          (int)Pad::getParam( RIGHTEDGE ) ) );
}

void TouchPad::setEdges( const QRect &edge )
{
    if ( !isValid() )
      return;

    Pad::setParam( TOPEDGE, edge.top() );
    Pad::setParam( LEFTEDGE, edge.left() );
    Pad::setParam( BOTTOMEDGE, edge.bottom() );
    Pad::setParam( RIGHTEDGE, edge.right() );
}

bool TouchPad::isTappingEnabled()
{
    return isValid()? Pad::getParam( MAXTAPTIME ) > 0 : false;
}

int TouchPad::tapTime()
{
    return isValid()? (int)Pad::getParam( MAXTAPTIME ) : 0;
}

void TouchPad::setTapTime( int time )
{
    Pad::setParam( MAXTAPTIME, (double) time );
}

int TouchPad::sensitivity()
{
    if ( !isValid() )
        return 0;

    int i;

    for ( i = 0; i < 5; ++i )
    {
        if ( Pad::getParam( FINGERLOW ) >= finger_low[ i ] )
            return i;
    }

    return i - 1;
}

void TouchPad::setSensitivity( int i )
{
    if ( !isValid() )
        return;

    if ( i< 0 && i>4 )
        return;

    if ( !SynConfig::treatAsALPS() )
    {
        Pad::setParam( FINGERLOW, finger_low[ i ] );
        Pad::setParam( FINGERHIGH, finger_low[ i ] + 5 );
    }
    else
    {
        Pad::setParam( FINGERLOW, finger_low[ i ] - 11 );
        Pad::setParam( FINGERHIGH, finger_low[ i ] - 10 );
    }
}

bool TouchPad::isHorizontalScrollEnabled()
{
  return isValid()? Pad::getParam( HORIZSCROLLDELTA ) > 0 : false;
}

int TouchPad::horizontalScrollDelta()
{
  return isValid()? (int)Pad::getParam( HORIZSCROLLDELTA ) : 0;
}

void TouchPad::setHorizontalScrollDelta( int delta )
{
    Pad::setParam( HORIZSCROLLDELTA, delta );
}

bool TouchPad::isVerticalScrollEnabled()
{
  return isValid()? Pad::getParam( VERTSCROLLDELTA ) > 0 : false;
}

int TouchPad::verticalScrollDelta()
{
  return isValid()? (int)Pad::getParam( HORIZSCROLLDELTA ) : 0;
}

void TouchPad::setVerticalScrollDelta( int delta )
{
    Pad::setParam( VERTSCROLLDELTA, delta );
}

bool TouchPad::isEdgeMotionEnabled()
{
  if ( isValid() ) 
    return Pad::getParam( EDGEMOTIONUSEALWAYS );
  else 
    return false;
}

bool TouchPad::isCoastingEnabled()
{
  if ( isValid() )
    return ( Pad::getParam( COASTINGSPEED ) < 0.1 );
  else
    return false;
}

void TouchPad::setEdgeMotionEnabled( bool enable )
{
    Pad::setParam( EDGEMOTIONUSEALWAYS, enable? 1:0 );
}

void TouchPad::setCoastingEnabled( bool enable )
{
  if ( enable ) 
      Pad::setParam( COASTINGSPEED, 38 );
  else
      Pad::setParam( COASTINGSPEED, 0 ); // disable it completely
}

bool TouchPad::isCircularScrollEnabled()
{
  return isValid()? (bool)Pad::getParam( CIRCULARSCROLLING ) : false;
}

void TouchPad::setCircularScrollEnabled( bool enable )
{
    Pad::setParam( CIRCULARSCROLLING, enable? 1:0 );
}

int TouchPad::circularScrollDelta()
{
  return isValid()? ( int )( Pad::getParam( CIRCSCROLLDELTA ) * 1000 ) : 0;
}

void TouchPad::setCircularScrollDelta( int delta )
{
    Pad::setParam( CIRCSCROLLDELTA, delta * 0.001 );
}

ScrollTrigger TouchPad::circularScrollTrigger()
{
  return isValid()? ( ScrollTrigger )Pad::getParam( CIRCSCROLLTRIGGER ) : NoTrigger;
}

void TouchPad::setCircularScrollTrigger( ScrollTrigger t )
{
    Pad::setParam( CIRCSCROLLTRIGGER, (double)t );
}

Button TouchPad::buttonForTap( TapType tap )
{
    if ( !isValid() ) return NoButton;

    Button b;

    switch ( tap )
    {
        case RightTop:
            b = (Button) Pad::getParam( RTCORNERBUTTON );
            break;
        case RightBottom:
            b = (Button) Pad::getParam( RBCORNERBUTTON );
            break;
        case LeftTop:
            b = (Button) Pad::getParam( LTCORNERBUTTON );
            break;
        case LeftBottom:
            b = (Button) Pad::getParam( LBCORNERBUTTON );
            break;
        case OneFinger:
            b = (Button) Pad::getParam( TAPBUTTON1 );
            break;
        case TwoFingers:
            b = (Button) Pad::getParam( TAPBUTTON2 );
            break;
        case ThreeFingers:
            b = (Button) Pad::getParam( TAPBUTTON3 );
            break;
        default:
            // must not be the case!
            b = NoButton;
            break;
    }

    return b;
}

bool TouchPad::areFastTapsEnabled()
{
  return isValid()? (bool)Pad::getParam( FASTTAPS ) : false;
}

void TouchPad::setFastTaps( bool enable )
{
    Pad::setParam( FASTTAPS, enable? 1:0 );
}

void TouchPad::setButtonForTap( TapType tap, Button button )
{
    if ( !isValid() )
        return;

    switch ( tap )
    {
        case RightTop:
            Pad::setParam( RTCORNERBUTTON, button );
            break;
        case RightBottom:
            Pad::setParam( RBCORNERBUTTON, button );
            break;
        case LeftTop:
            Pad::setParam( LTCORNERBUTTON, button );
            break;
        case LeftBottom:
            Pad::setParam( LBCORNERBUTTON, button );
            break;
        case OneFinger:
            Pad::setParam( TAPBUTTON1, button );
            break;
        case TwoFingers:
            Pad::setParam( TAPBUTTON2, button );
            break;
        case ThreeFingers:
            Pad::setParam( TAPBUTTON3, button );
            break;
        default:
            // must not be the case!
            break;
    }

}

int TouchPad::absCoordX()
{
    return isValid()? (int)Pad::getParam( ABSCOORDX ) : 0;
}

int TouchPad::absCoordY()
{
    return isValid()? (int)Pad::getParam( ABSCOORDY ) : 0;
}

void TouchPad::applyConfig()
{
    setEnabled( SynConfig::enableTouchPad() );
    setSensitivity( SynConfig::sensitivity() );
    setTapTime( SynConfig::enableTapping()? SynConfig::tapTime() : 0 );
    setButtonForTap( OneFinger, ( Button )SynConfig::tapOneFinger() );
    setButtonForTap( TwoFingers, ( Button )SynConfig::tapTwoFingers() );
    setButtonForTap( ThreeFingers, ( Button )SynConfig::tapThreeFingers() );
    setButtonForTap( RightTop, ( Button )SynConfig::tapRightTop() );
    setButtonForTap( RightBottom, ( Button )SynConfig::tapRightBottom() );
    setButtonForTap( LeftTop, ( Button )SynConfig::tapLeftTop() );
    setButtonForTap( LeftBottom, ( Button )SynConfig::tapLeftBottom() );  
    setHorizontalScrollDelta( SynConfig::enableHorizontalScrolling()? SynConfig::horizontalScrollDelta() : 0 );
    setVerticalScrollDelta( SynConfig::enableVerticalScrolling()? SynConfig::verticalScrollDelta() : 0 );
    setEdgeMotionEnabled( SynConfig::enableEdgeMotion() );
    setCoastingEnabled( SynConfig::enableCoasting() );
    setCircularScrollEnabled( SynConfig::enableCircularScrolling() );
    setCircularScrollDelta( SynConfig::circularScrollDelta() );
    setCircularScrollTrigger( ( ScrollTrigger )SynConfig::circularScrollTrigger() );

    setEdges( SynConfig::edges() );
    setFastTaps( SynConfig::fastTaps() );
}

