//
//   File : kvi_process.cpp (/usr/build/NEW_kvirc/kvirc/kvilib/kvi_process.cpp)
//   Last major modification : Sun Jan 10 1999 17:20:34 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//
//  Simple versions of the KProcess and KProcessController classes
//  original code by (C) Christian Czezatke
//  e9025461@student.tuwien.ac.at
//  Really good work :)
//

//#define _KVI_DEBUG_CLASS_NAME_ "KviProcess"
//#define _KVI_DEBUG_CHECK_RANGE_
#ifndef _GNU_SOURCE
	// need this for SA_NOCLDSTOP...at least on my linux distro
	#define _GNU_SOURCE
#endif

#include "kvi_debug.h"

#include "kvi_process.h"
#include "kvi_malloc.h"
#include "kvi_locale.h"

#include <qwindowdefs.h> //for qAddPostRoutine
#include <qfileinfo.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

#ifndef SA_NOCLDSTOP
	// uh ?...
	#include <signal.h>
	#ifndef SA_NOCLDSTOP
		#define SA_NOCLDSTOP 1
	#endif
#endif

//
// KviProcessController class
// Handles signals from dying children.
//

KviProcessController * g_pProcessController = 0;

KviStr KviProcess::m_szShellName("");

void theSigCHLDHandler(int )
{
	// A signal from a child was received (SIGPIPE or SIGCHILD)
	int status;
	pid_t this_pid;
	int saved_errno = errno;
	// since waitpid and write change errno, we have to save it and restore it
	// (Richard Stevens, Advanced programming in the Unix Environment)
	this_pid = waitpid(-1, &status, WNOHANG);
	// J6t: g_pProcessController might be already destroyed
	if((-1 != this_pid) && (g_pProcessController != 0)) {
		::write(g_pProcessController->m_fd[1], &this_pid, sizeof(this_pid));
		::write(g_pProcessController->m_fd[1], &status, sizeof(status));
	}
	errno = saved_errno;
}

KviProcessController::KviProcessController()
{
	// We keep a list of processes currently alive.
	m_pProcessList = new QList<KviProcess>;
	m_pProcessList->setAutoDelete(false);      //newer delete the processes from here
 	// Need a pipe to communicate from the signal handler
	if(0>pipe(m_fd))printf(strerror(errno));
	if(fcntl(m_fd[0], F_SETFL, O_NONBLOCK)==-1)printf(strerror(errno));
	// And a notifier for that
	m_pNotifier = new QSocketNotifier(m_fd[0],QSocketNotifier::Read);
	m_pNotifier->setEnabled(true);
	QObject::connect(m_pNotifier, SIGNAL(activated(int)),this, SLOT(slotDoHousekeeping(int)));
	// Now we set the handler
	struct sigaction act;
	act.sa_handler=&theSigCHLDHandler;
	sigemptyset(&(act.sa_mask));
	sigaddset(&(act.sa_mask), SIGCHLD);
	act.sa_flags = SA_NOCLDSTOP;
	// CC: take care of SunOS which automatically restarts interrupted system
	// calls (and thus does not have SA_RESTART)
#ifdef SA_RESTART
	act.sa_flags |= SA_RESTART;
#endif
	sigaction( SIGCHLD, &act, 0L); 
	// kvi_thread takes care of the following SIGNAL
//	act.sa_handler=SIG_IGN;
//	sigemptyset(&(act.sa_mask));
//	sigaddset(&(act.sa_mask), SIGPIPE);
//	act.sa_flags = 0;
//	sigaction( SIGPIPE, &act, 0L);
	// ready
}



void KviProcessController::slotDoHousekeeping(int)
{
	__enter("Controller::doHouseKeeping");
	// A process has exited , and data was written to the pipe
	// in the handler...here we remove dead processes from the list
	KviProcess *proc;
	int bytes_read;
	pid_t pid;
	int status;
	bytes_read  = ::read(m_fd[0], &pid, sizeof(pid_t));
	bytes_read += ::read(m_fd[0], &status, sizeof(int));
	if (bytes_read != sizeof(int)+sizeof(pid_t)){
		if(bytes_read < 0){
			debug(__tr("Error: Could not read info from signal handler!"));
			debug(__tr("Error %d"),errno);
			if((errno == EINTR) || (errno == EINPROGRESS) || (errno == EAGAIN)){
				fprintf(stderr,__tr("Retrying"));
				slotDoHousekeeping(0);
				return;
			}
		} else {
			debug(__tr("Error: Could not read info from signal handler!"));
			debug(__tr("Readed %d bytes instead of %d+%d"),bytes_read,sizeof(int),sizeof(pid_t));
		}
	}
	proc = m_pProcessList->first();
	// Cleanup
	while (proc) {
		if (proc->pid() == pid) {
			// process has exited, so do emit the respective events
			proc->processHasExited(status);
			return; // only ONE process has exited
		}
		proc = m_pProcessList->next();
	}
	__leave("Controller::doHouseKeeping");
}

KviProcessController::~KviProcessController()
{
	struct sigaction act;

	// Turn off notification for processes that have exited
	act.sa_handler=SIG_IGN;
	sigemptyset(&(act.sa_mask));
	sigaddset(&(act.sa_mask), SIGCHLD);
	act.sa_flags = 0;
	sigaction( SIGCHLD, &act, 0L);

  	close(m_fd[0]);
	close(m_fd[1]);
	delete m_pProcessList;
	delete m_pNotifier;
}

void KviProcessController::addProcess(KviProcess *proc)
{
	__debug("Controller::addProcess");
	m_pProcessList->append(proc);
}

void KviProcessController::removeProcess(KviProcess *proc)
{
	__debug("Controller::removeProcess");
	m_pProcessList->removeRef(proc);
}


void kill_process_controller()
{
	// global handler to kill the g_pProcessController
	if(g_pProcessController){
		delete g_pProcessController;
		g_pProcessController = 0;
	}
}

KviProcess::KviProcess()
{
	__enter("KviProcess");
	if(!g_pProcessController){
		g_pProcessController = new KviProcessController();
		qAddPostRoutine(kill_process_controller);
	}

	m_pid = 0;
	m_bIsRunning = false;
	clearSockVariables();
	m_pStdoutNotifier = 0;
	m_pStderrNotifier = 0;
	g_pProcessController->addProcess(this);
	__leave("KviProcess");
}

KviProcess::~KviProcess()
{
	__enter("~KviProcess");
	// First kill all sockets
	killSockets();
	// If is running , and killOnClose is reequested , well , KILL it
	if(m_bKillOnClose && m_bIsRunning)sendSignal(SIGKILL);
	// And tell the process controller to forget about us.
	if(g_pProcessController)g_pProcessController->removeProcess(this);
	__leave("~KviProcess");
}

void KviProcess::clearSockVariables()
{
	m_stdinSock[0]=-1;
	m_stdinSock[1]=-1;
	m_stdoutSock[0]=-1;
	m_stdoutSock[1]=-1;
	m_stderrSock[0]=-1;
	m_stderrSock[1]=-1;
}

bool KviProcess::setupSockets()
{
	__enter("SetupSockets");
	int error = socketpair(AF_UNIX,SOCK_STREAM,0,m_stdinSock);
	if(error != 0)return false;
	error = socketpair(AF_UNIX,SOCK_STREAM,0,m_stderrSock);
	if(error != 0){
		close(m_stdinSock[0]);
		close(m_stdinSock[1]);
		clearSockVariables();
		return false;
	}
	error = socketpair(AF_UNIX,SOCK_STREAM,0,m_stdoutSock);
	if(error != 0){
		close(m_stdinSock[0]);
		close(m_stdinSock[1]);
		close(m_stdoutSock[0]);
		close(m_stdoutSock[1]);
		clearSockVariables();
		return false;
	}
	__leave("SetupSockets");
	return true;
}

void KviProcess::killSockets()
{
	__enter("killSockets");
	if(m_stdinSock[0]!=-1)close(m_stdinSock[0]);
	if(m_stdinSock[1]!=-1)close(m_stdinSock[1]);
	if(m_stdoutSock[0]!=-1)close(m_stdoutSock[0]);
	if(m_stdoutSock[1]!=-1)close(m_stdoutSock[1]);
	if(m_stderrSock[0]!=-1)close(m_stderrSock[0]);
	if(m_stderrSock[1]!=-1)close(m_stderrSock[1]);
	clearSockVariables();
	if(m_pStdoutNotifier != 0)delete m_pStdoutNotifier;
	if(m_pStderrNotifier != 0)delete m_pStderrNotifier;
	m_pStdoutNotifier = 0;
	m_pStderrNotifier = 0;
	__leave("killSockets");
}

bool KviProcess::run(const char * commandline,bool bCommunicate,bool bKillOnClose,bool bExecInSubshell)
{
	if(m_bIsRunning)return false;
	if((!commandline) || (!(*commandline)))return false;

	m_bKillOnClose = bKillOnClose;
	//Get the argument list
	QList<KviStr> argList;
	argList.setAutoDelete(true);

	KviStr arg;
	KviStr shellName;
	if(!findShell(shellName))shellName = "/bin/sh"; // just a default

	if(bExecInSubshell){
		argList.append(new KviStr(shellName.ptr()));
		argList.append(new KviStr("-c"));
		argList.append(new KviStr(commandline));
	} else {
		while(*commandline){
			commandline = kvi_extractToken(arg,commandline);
			if(arg.hasData())argList.append(new KviStr(arg));
		}
	}

	if(argList.isEmpty())return false;
	if(bCommunicate){
		if(!setupSockets())return false;
	}

	char ** pArgs=(char **)kvi_malloc((argList.count()+1) * sizeof(char*));
	uint cur=0;
	for(KviStr * pA=argList.first();pA && (cur<argList.count());pA=argList.next()){
		pArgs[cur]=pA->ptr();
		cur++;
	}
	pArgs[cur]=0; //Null terminator

	m_pid = fork();

	switch(m_pid){
		case 0:
			//////////////////////////////////////////
			// CHILD Process : Success
			//
			if(bCommunicate){
				if(!child_setupSockets()){
					killSockets();
					debug(__tr("Could not setup child communication"));
				}
			}//else setpgid(0,0); ?????
			execvp(pArgs[0],pArgs);
			debug(__tr("execvp failed for file %s : %s"),pArgs[0],strerror(errno));
//			debug("commandline was %s",commandline);
			exit(-1);
			//
			//
			//////////////////////////////////////////
		break;
		case -1: //Parent process : Failed
			killSockets();
			kvi_free(pArgs);
			return false;
		break;
		default: //Parent process : Success
			if(bCommunicate){
				if(!parent_setupSockets()){
					killSockets();
					debug(__tr("Could not setup parent communication"));
				}
			}
			m_bIsRunning = true;
			kvi_free(pArgs);
			return true;
		break;
	}
	//NEWER Here...
}

bool KviProcess::parent_setupSockets()
{
	// Kill the child socket side...
	close(m_stdinSock[PROC_SOCKET_CHILD]);
	close(m_stdoutSock[PROC_SOCKET_CHILD]);
	close(m_stderrSock[PROC_SOCKET_CHILD]);
	m_stdinSock[PROC_SOCKET_CHILD] =-1;
	m_stdoutSock[PROC_SOCKET_CHILD]=-1;
	m_stderrSock[PROC_SOCKET_CHILD]=-1;

	if(fcntl(m_stdinSock[PROC_SOCKET_PARENT],F_SETFL,O_NONBLOCK)==-1)return false;
	if(fcntl(m_stdoutSock[PROC_SOCKET_PARENT],F_SETFL,O_NONBLOCK)==-1)return false;
	if(fcntl(m_stderrSock[PROC_SOCKET_PARENT],F_SETFL,O_NONBLOCK)==-1)return false;

	m_pStdoutNotifier = new QSocketNotifier(m_stdoutSock[PROC_SOCKET_PARENT],QSocketNotifier::Read,this);
	QObject::connect(m_pStdoutNotifier,SIGNAL(activated(int)),this,SLOT(receivedStdout(int)));
	m_pStdoutNotifier->setEnabled(true);

	m_pStderrNotifier = new QSocketNotifier(m_stderrSock[PROC_SOCKET_PARENT],QSocketNotifier::Read,this);
	QObject::connect(m_pStderrNotifier,SIGNAL(activated(int)),this,SLOT(receivedStderr(int)));
	m_pStderrNotifier->setEnabled(true);

	return true;
}

bool KviProcess::child_setupSockets()
{
//	__enter("child_setupSockets");
	// Kill the parent socket side...
	close(m_stdinSock[PROC_SOCKET_PARENT]);
	close(m_stdoutSock[PROC_SOCKET_PARENT]);
	close(m_stderrSock[PROC_SOCKET_PARENT]);
	m_stdinSock[PROC_SOCKET_PARENT] =-1;
	m_stdoutSock[PROC_SOCKET_PARENT]=-1;
	m_stderrSock[PROC_SOCKET_PARENT]=-1;

	if(dup2(m_stdinSock[PROC_SOCKET_CHILD] ,STDIN_FILENO) ==-1)return false;
	if(dup2(m_stdoutSock[PROC_SOCKET_CHILD],STDOUT_FILENO)==-1)return false;
	if(dup2(m_stderrSock[PROC_SOCKET_CHILD],STDERR_FILENO)==-1)return false;

	// linger a while, to send all the output...
	struct linger l;
	l.l_onoff = 1;
	l.l_linger = 100; // linger 1 sec

	if(setsockopt(m_stdoutSock[PROC_SOCKET_CHILD],SOL_SOCKET,SO_LINGER,(char*)&l,sizeof(l))==-1)return false;
	if(setsockopt(m_stderrSock[PROC_SOCKET_CHILD],SOL_SOCKET,SO_LINGER,(char*)&l,sizeof(l))==-1)return false;
//	__leave("child_setupSockets");
	return true;
}

void KviProcess::receivedStdout(int fd)
{
	__enter("receivedStdout");
	char buffer[1024];
	int len;
	do {
//#warning "Check for read() errors here (EAGAIN ?)"
		len =read(fd,buffer,1024);
		if(len > 0)emit processStdout(this,buffer,len); //What means an EOF from a process ?
//		else m_pStdoutNotifier->setEnabled(false); //I'll follow the example of KProcess that disables the notifier
	} while(len == 1024);
}

void KviProcess::receivedStderr(int fd)
{
	__enter("receiveStderr");
	char buffer[1024];
	int len;
	do {
		len =read(fd,buffer,1024);
		if(len > 0)emit processStderr(this,buffer,len); //What means an EOF from a process ?
//		else m_pStderrNotifier->setEnabled(false); //I'll follow the example of KProcess that disables the notifier
	} while(len == 1024);
}

bool KviProcess::sendSignal(int sig)
{
	__enter("sendSignal");
	if(m_bIsRunning){
		if(::kill(m_pid,sig)==-1)return false;
	}
	__leave("sendSignal");
	return true;
}

void KviProcess::processHasExited(int status)
{
	__enter("processHasExited");
	// read the last data in the buffer (sometimes the SIGCHLD arrives before the in_socket notifier fires)
	receivedStderr(m_stderrSock[PROC_SOCKET_PARENT]);
	receivedStdout(m_stdoutSock[PROC_SOCKET_PARENT]);
	killSockets();
	m_bIsRunning= false;
	m_bKillOnClose = false;
	emit terminated(this,status);
	__leave("processHasExited");
}

bool KviProcess::findShell(KviStr &buffer)
{
	if(m_szShellName.hasData()){
		buffer = m_szShellName;
		return true;
	}

	KviStr tmp = getenv("SHELL");
	tmp.stripWhiteSpace();
	if(tmp.isEmpty()){
		debug(__tr("No SHELL enviroinement variable defined!"));
		return false;
	}
	QFileInfo f(_CHAR_2_QSTRING(tmp.ptr()));
	if(f.isExecutable()){
		m_szShellName = tmp;
		buffer = m_szShellName;
		return true;
	} else debug(__tr("SHELL is defined to %s but seems to be not executable"),tmp.ptr());
	return false;
}

bool KviProcess::writeData(const char *buffer,int len)
{
//	if(len == -1)len = strlen(buffer);
//	debug("Writing %s to process",buffer);
	return (write(m_stdinSock[PROC_SOCKET_PARENT],buffer,len) == len);
}



#include "m_kvi_process.moc"
