//
//   File : kvi_varcache.cpp (/usr/build/NEW_kvirc/kvirc/src/kvilib/kvi_varcache.cpp)
//   Last major modification : Mon Feb 15 1999 01:11:40 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"


#include "kvi_varcache.h"

KviVariableCache::KviVariableCache()
{
	m_pVarList = new QList<KviVariable>;
	m_pVarList->setAutoDelete(true);
	m_pDictList = new QList<KviDictionary>;
	m_pDictList->setAutoDelete(true);
}


KviVariableCache::~KviVariableCache()
{
	while(m_pDictList->first())deleteDict(m_pDictList->first());
	delete m_pDictList;
	delete m_pVarList;
}

KviVariable * KviVariableCache::set(const char *name,const char *value)
{
	__range_valid(name);
	__range_valid(value);
	KviVariable * v=getVariable(name);
	if(v){
		v->szValue = value;
		return v;
	}
	return insertNewVariable(name,value);
}

void KviVariableCache::concat(const char *name,const char *value)
{
	__range_valid(name);
	__range_valid(value);
	KviVariable * v=getVariable(name);
	if(v){
		v->szValue.append(value);
		return;
	}
	insertNewVariable(name,value);
}

void KviVariableCache::concatWithSeparator(const char *name,char sep,const char *value)
{
	__range_valid(name);
	__range_valid(value);
	KviVariable * v=getVariable(name);
	if(v){
		v->szValue.append(sep);
		v->szValue.append(value);
		return;
	}
	insertNewVariable(name,value);
}

KviVariable * KviVariableCache::insertNewVariable(const char *name,const char *value)
{
	KviVariable *v = new KviVariable;
	v->szName = name;
	v->szValue = value;
	// Insert in alphabetic order
	int idx = 0;
	for(KviVariable *b=m_pVarList->first();b;b=m_pVarList->next()){
		if(kvi_strcmpCI(b->szName.ptr(),name) < 0){
			m_pVarList->insert(idx,v);
			return v;
		}
		idx++;
	}
	m_pVarList->append(v);
	return v;
}

void KviVariableCache::unset(const char *name)
{
	__range_valid(name);
	KviVariable *v= getVariable(name);
	if(v)m_pVarList->removeRef(v);
}

KviVariable * KviVariableCache::getVariable(const char *name)
{
	__range_valid(name);
	int r;
	for(KviVariable * v=m_pVarList->first();v;v=m_pVarList->next()){
		r = kvi_strcmpCI(v->szName.ptr(),name);
		if(r == 0){
			return v; //equal
		}
		if(r < 0)return 0; //found a greater one
	}
	return 0;
}

const char *KviVariableCache::find(const char *name)
{
	__range_valid(name);
	KviVariable *v = getVariable(name);
	if(v)return v->szValue.ptr();
	return 0;
}


//================================================================================================
// Dictionaries

void KviVariableCache::deleteDict(KviDictionary * dict)
{
	__range_valid(dict->m_pVarCache);
	__range_valid(m_pDictList->find(dict) != -1);
	delete dict->m_pVarCache;
	m_pDictList->removeRef(dict);
}

KviDictionary * KviVariableCache::insertNewDict(const char *name)
{
	__range_invalid(getDict(name));
	KviDictionary * dict = new KviDictionary;
	dict->szName = name;
	dict->m_pVarCache = new KviVariableCache();
	// Insert in alphabetic order
	int idx = 0;
	for(KviDictionary *b=m_pDictList->first();b;b=m_pDictList->next()){
		if(kvi_strcmpCI(b->szName.ptr(),name) < 0){
			m_pDictList->insert(idx,dict);
			return dict;
		}
		idx++;
	}
	m_pDictList->append(dict);
	return dict;
}

KviDictionary * KviVariableCache::getDict(const char *name)
{
	__range_valid(name);
	int r;
	for(KviDictionary * v=m_pDictList->first();v;v=m_pDictList->next()){
		r = kvi_strcmpCI(v->szName.ptr(),name);
		if(r == 0){
			return v; //equal
		}
		if(r < 0)return 0; //found a greater one
	}
	return 0;	
}

KviVariable * KviVariableCache::getDictVariable(const char *dictname,const char *varname)
{
	__range_valid(dictname);
	__range_valid(varname);
	KviDictionary * dict = getDict(dictname);
	if(!dict)return 0;
	return dict->m_pVarCache->getVariable(varname);
}

const char * KviVariableCache::findDictVariable(const char *dictname,const char *varname)
{
	__range_valid(dictname);
	__range_valid(varname);
	KviVariable * v = getDictVariable(dictname,varname);
	if(v)return v->szValue.ptr();
	return 0;
}
/*
void KviVariableCache::setDictVariable(const char *dictname,const char *varname,const char *value)
{
	__range_valid(dictname);
	__range_valid(varname);
	__range_valid(value);
	KviVariable * v=getDictVariable(dictname,varname);
	if(v){
		v->szValue = value;
		return;
	}
	insertNewDictVariable(dictname,varname,value);
}
*/
KviVariable * KviVariableCache::setDictVariable(const char *dictname,const char *varname,const char *value)
{
	__range_valid(dictname);
	__range_valid(varname);
	__range_valid(value);
	KviDictionary * dict = getDict(dictname);
	if(!dict){
		dict = insertNewDict(dictname);
		return dict->m_pVarCache->insertNewVariable(varname,value);
	} else return dict->m_pVarCache->set(varname,value);
}

void KviVariableCache::unsetDictVariable(const char *dictname,const char *varname)
{
	__range_valid(dictname);
	__range_valid(varname);
	KviDictionary * dict = getDict(dictname);
	if(!dict)return;
	dict->m_pVarCache->unset(varname);
	if(dict->m_pVarCache->m_pVarList->count() == 0)deleteDict(dict);
}

void KviVariableCache::concatDictVariable(const char *dictname,const char *varname,const char *value)
{
	__range_valid(dictname);
	__range_valid(varname);
	__range_valid(value);
	KviDictionary *dict =  getDict(dictname);
	if(!dict)dict = insertNewDict(dictname);
	dict->m_pVarCache->concat(varname,value);
}

void KviVariableCache::concatWithSeparatorDictVariable(const char *dictname,const char *varname,char sep,const char *value)
{
	__range_valid(dictname);
	__range_valid(varname);
	__range_valid(value);
	KviDictionary *dict =  getDict(dictname);
	if(!dict)dict = insertNewDict(dictname);
	dict->m_pVarCache->concatWithSeparator(varname,sep,value);

}
