//
//   File : kvi_rawevent.cpp
//   Last major modification : Sat Feb 30 2000 17:16:14 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//



#include <qfile.h>
#include "kvi_defines.h"
#include "kvi_locale.h"
#include "kvi_rawevent.h"
#include "kvi_fileutils.h"

KviRawEventManager::KviRawEventManager()
{
	m_pLiteralEventsDict = 0;
	for(int i=0;i<1000;i++)m_numericEvents[i] = 0;
}

KviRawEventManager::~KviRawEventManager()
{
	cleanup();
}

bool KviRawEventManager::save(const char *filename)
{
	QFile f(filename);
	if(!f.open(IO_WriteOnly | IO_Truncate))return false;

	f.writeBlock(KVI_MAGIC_STRING,KVI_MAGIC_STRING_LENGTH);

	for(int i = 0;i< 1000;i++){
		if(m_numericEvents[i])
		{
			for(KviRawEvent * e = m_numericEvents[i]->first();e;e= m_numericEvents[i]->next())
			{
				f.writeBlock("[RAWEVENT ",10);
				KviStr tmp(KviStr::Format,"%d",i);
				f.writeBlock(tmp.ptr(),tmp.len());
				f.writeBlock(" ",1);
				f.writeBlock(e->szSource.ptr(),e->szSource.len());
				f.writeBlock(" bytes ",7);
				KviStr szNBytes(KviStr::Format,"%d",e->szBuffer.len());
				f.writeBlock(szNBytes.ptr(),szNBytes.len());
				f.writeBlock("]\n",2);
				f.writeBlock(e->szBuffer.ptr(),e->szBuffer.len());
				f.writeBlock("\n[ENDRAWEVENT]\n",15);
			}
		}
	}

	if(m_pLiteralEventsDict)
	{
		QAsciiDictIterator<KviRawEventList> it(*m_pLiteralEventsDict);
		QList<KviRawEvent> * l = 0;
		while((l = it.current()))
		{
			for(KviRawEvent * e = l->first();e;e= l->next())
			{
				f.writeBlock("[RAWEVENT ",10);
				KviStr tmp(it.currentKey());
				f.writeBlock(tmp.ptr(),tmp.len());
				f.writeBlock(" ",1);
				f.writeBlock(e->szSource.ptr(),e->szSource.len());
				f.writeBlock(" bytes ",7);
				KviStr szNBytes(KviStr::Format,"%d",e->szBuffer.len());
				f.writeBlock(szNBytes.ptr(),szNBytes.len());
				f.writeBlock("]\n",2);
				f.writeBlock(e->szBuffer.ptr(),e->szBuffer.len());
				f.writeBlock("\n[ENDRAWEVENT]\n",15);
			}			
			++it;
		}
	}

	f.close();
	return true;
}

void KviRawEventManager::cleanup()
{
	if(m_pLiteralEventsDict)delete m_pLiteralEventsDict;
	m_pLiteralEventsDict = 0;
	for(int i=0;i<1000;i++)
	{
		if(m_numericEvents[i])
		{
			delete m_numericEvents[i];
			m_numericEvents[i] = 0;
		}
	}
}

void KviRawEventManager::addHandler(const char * command,const char * source,const char * buffer)
{
	KviStr cmd(command);
	cmd.stripWhiteSpace();
	if(cmd.isEmpty())return;

	bool bOk = false;
	int num = cmd.toInt(&bOk);

	KviRawEvent * e = new KviRawEvent;
	e->szSource = source;
	e->szSource.stripWhiteSpace();
	if(e->szSource.isEmpty())e->szSource = "*";
	e->szBuffer = buffer;

	if(bOk)
	{
		if((num >= 0) && (num < 1000))
		{
			if(!m_numericEvents[num])
			{
				m_numericEvents[num] = new QList<KviRawEvent>;
				m_numericEvents[num]->setAutoDelete(true);
			}
			m_numericEvents[num]->append(e);
			return;
		}
	}

	if(!m_pLiteralEventsDict)
	{
		m_pLiteralEventsDict = new QAsciiDict<KviRawEventList>(17,false);
		m_pLiteralEventsDict->setAutoDelete(true);
	}
	QList<KviRawEvent> * l = m_pLiteralEventsDict->find(command);
	if(!l)
	{
		l = new QList<KviRawEvent>;
		l->setAutoDelete(true);
		m_pLiteralEventsDict->insert(command,l);
	}

	l->append(e);
}

bool KviRawEventManager::load(const char *filename,KviStr &error)
{
	cleanup();
	KviStr buffer;
	if(!kvi_loadFile(filename,buffer)){
		error = _i18n_("Unable to load file");
		return false;
	}
	if(!kvi_strEqualCSN(KVI_MAGIC_STRING,buffer.ptr(),KVI_MAGIC_STRING_LENGTH)){
		error = _i18n_("Invalid magic");
		return false; //not a kvirc file
	}
	buffer.cutLeft(KVI_MAGIC_STRING_LENGTH);
	buffer.stripLeftWhiteSpace(); //be flexible...allow some spaces...for manual editing by experts :)
	if(buffer.isEmpty())return true; //done

	while(buffer.hasData()){
		//[RAWEVENT command bytes <numofbytes>]\n
		//...
		//...
		//...
		//\n[ENDRAWEVENT]\n

		// Event label
		if(!kvi_strEqualCIN("[RAWEVENT ",buffer.ptr(),10)){
			error = _i18n_("Unrecognized [RAWEVENT] label.");
			return false;
		}

		buffer.cutLeft(10);
		buffer.stripLeftWhiteSpace(); //be flexible...allow some spaces...for manual editing by experts :)

		// Event name
		KviStr eventName;
		if(!buffer.getToken(eventName,' ')){
			error = _i18n_("Syntax error in [RAWEVENT] label");
			return false;
		} //there must be at least another token

		buffer.stripLeftWhiteSpace(); //be flexible...allow some spaces...for manual editing by experts :)

		KviStr eventSource;
		if(!buffer.getToken(eventSource,' '))
		{
			error = _i18n_("Missing event source in [RAWEVENT] label");
			return false;
		}

		buffer.stripLeftWhiteSpace(); //be flexible...allow some spaces...for manual editing by experts :)

		if(!kvi_strEqualCIN("bytes ",buffer.ptr(),6)){
			error = _i18n_("Syntax error in [RAWEVENT] label (event length)");
			return false;
		}
		buffer.cutLeft(6);

		buffer.stripLeftWhiteSpace(); //be flexible...allow some spaces...for manual editing by experts :)

		if(!isdigit(*(buffer.ptr()))){
			error = _i18n_("Syntax error in [RAWEVENT] label (digit expected)");
			return false;
		}

		KviStr numBytes;
		if(!buffer.getToken(numBytes,']')){
			error = _i18n_("Syntax error in [RAWEVENT] label (can not extract alias length)");
			return false;
		}

		bool bOk = false;
		int bytes = numBytes.toInt(&bOk);
		if(!bOk){
			error = _i18n_("Unrecogized token ");
			error +=numBytes;
			return false;
		} //syntax error again...someone messed with the file
		buffer.cutLeft(1); //cut the \n
		if(buffer.len() < bytes){
			error = _i18n_("Raw event buffer smaller than declared length (");
			error += numBytes;
			error +=")";
			return false;
		}

		KviStr eventBuffer(buffer.ptr(),bytes);

		addHandler(eventName.ptr(),eventSource.ptr(),eventBuffer.ptr());

		buffer.cutLeft(bytes);

		if(!kvi_strEqualCIN("\n[ENDRAWEVENT]",buffer.ptr(),14)){
			error = _i18n_("Unrecognized [ENDRAWEVENT] label");
			return false;
		}
		buffer.cutLeft(14);

		buffer.stripLeftWhiteSpace(); //be flexible...allow some spaces...for manual editing by experts :)
	}
	return true;
}



