//
//   File : kvi_winproperties.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_winproperties.cpp)
//   Last major modification : Wed Apr 28 1999 13:37:06 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_winproperties.h"
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "kvi_defines.h"
#include "kvi_options.h"


//============ KviWindowPropertiesList ============//

KviWindowPropertiesList::KviWindowPropertiesList()
{
	m_pPropertiesList = new QList<KviWindowPackedProperty>;
	m_pPropertiesList->setAutoDelete(true);
}
//============ ~KviWindowPropertiesList ============//

KviWindowPropertiesList::~KviWindowPropertiesList()
{
	delete m_pPropertiesList;
}

KviWindowPackedProperty * KviWindowPropertiesList::findProperty(const char *winName)
{
	for(KviWindowPackedProperty *p = m_pPropertiesList->first();p;p=m_pPropertiesList->next()){
		if(kvi_strEqualCI(winName,p->name.ptr()))return p;
	}
	return 0;
}

void KviWindowPropertiesList::setProperty(const char *winName,KviWindowProperty * prop)
{
	KviWindowPackedProperty * p = findProperty(winName);
	if(!g_pOptions->m_bWindowsRememberProprietes)return;
	if(!p){
		p = new KviWindowPackedProperty;
		p->name = winName;
		if(m_pPropertiesList->count() >= KVI_MAX_SAVED_WINDOW_PROPERTIES){
			m_pPropertiesList->removeFirst(); //Remove the oldest item...
		}
	} else {
		// Move it to the last place
		m_pPropertiesList->setAutoDelete(false);
		m_pPropertiesList->removeRef(p);
		m_pPropertiesList->setAutoDelete(true);
	}
	
	packProperty(prop,p);
	m_pPropertiesList->append(p); //The newest entry...
}


bool KviWindowPropertiesList::load(const char *filename,KviStr &error)
{
	KviStr buffer;
	if(!kvi_loadFile(filename,buffer)){
		error = __tr("Unable to load file");
		return false;
	}
	if(!kvi_strEqualCSN(KVI_MAGIC_STRING,buffer.ptr(),KVI_MAGIC_STRING_LENGTH)){
		error = __tr("Invalid magic");
		return false; //not a kvirc file
	}
	buffer.cutLeft(KVI_MAGIC_STRING_LENGTH);
	buffer.stripLeftWhiteSpace(); //be flexible...allow some spaces...for manual editing by experts :)
	if(buffer.isEmpty())return true; //done

	while(buffer.hasData()){
		KviStr name,prop;
		if(!(buffer.getLine(name) && buffer.getLine(prop))){
			error = __tr("Incomplete window property");
			return false;
		}
		KviWindowPackedProperty * p = new KviWindowPackedProperty;
		p->name = name;
		p->property = prop;
		if(m_pPropertiesList->count() >= KVI_MAX_SAVED_WINDOW_PROPERTIES){
			m_pPropertiesList->removeFirst(); //Remove the oldest item...
		}
		m_pPropertiesList->append(p);
		buffer.stripLeftWhiteSpace();
	}
	return true;
}

bool KviWindowPropertiesList::save(const char *filename)
{
	QFile f(filename);
	if(!f.open(IO_WriteOnly | IO_Truncate))return false;
	f.writeBlock(KVI_MAGIC_STRING,KVI_MAGIC_STRING_LENGTH);
	if(g_pOptions->m_bWindowsRememberProprietes){
		for(KviWindowPackedProperty *p=m_pPropertiesList->first();p;p=m_pPropertiesList->next()){
			f.writeBlock(p->name.ptr(),p->name.len());
			f.writeBlock("\n",1);
			f.writeBlock(p->property.ptr(),p->property.len());
			f.writeBlock("\n",1);
		}
	} //else forget everything...
	f.close();
	return true;
}

void KviWindowPropertiesList::packProperty(KviWindowProperty * prop,KviWindowPackedProperty * packed)
{
	__range_valid(prop);
	__range_valid(packed);
	packed->property.sprintf(
		"RECT(%d,%d,%d,%d);DOCK(%d,%d);SPL1(%d,%d);VIEW(%d,%d);SPL2(%d,%d,%d)",
		prop->rect.left(),prop->rect.top(),prop->rect.width(),prop->rect.height(),
		g_pOptions->m_bWindowsRememberDockState ? prop->isDocked : 1,
		g_pOptions->m_bWindowsRememberMaximizeState ? prop->isMaximized : 0,
		prop->splitWidth1,prop->splitWidth2,
		prop->timestamp,prop->imagesVisible,
		prop->topSplitWidth1,prop->topSplitWidth2,prop->topSplitWidth3);
//	debug("Packing property : %s",packed->property.ptr());
}

void KviWindowPropertiesList::unpackProperty(KviWindowProperty * prop,KviWindowPackedProperty * packed)
{
	__range_valid(prop);
	__range_valid(packed);
	int l,t; l = t = 0;
	int w,h; w = h = 200;
	prop->splitWidth1 = 50;
	prop->splitWidth2 = 50;
	prop->isDocked = true;
	prop->imagesVisible = true;
	prop->timestamp = true;
//#warning "sscanf IS BUGGY! (on many systems :)"
	// Problems probably here...sscanf fail to read some int values...
//	debug("Unpacking property : %s",packed->property.ptr());
	if(sscanf(packed->property.ptr(),
		"RECT(%d,%d,%d,%d);DOCK(%d,%d);SPL1(%d,%d);VIEW(%d,%d);SPL2(%d,%d,%d)",
		&l,&t,&w,&h,
		&(prop->isDocked),&(prop->isMaximized),
		&(prop->splitWidth1),&(prop->splitWidth2),
		&(prop->timestamp),&(prop->imagesVisible),
		&(prop->topSplitWidth1),&(prop->topSplitWidth2),&(prop->topSplitWidth3)) != 13)
			debug("Error while unpacking properties.");
//	debug("unpacked : RECT(%d,%d,%d,%d);DOCK(%d,%d);SPL1(%d,%d);VIEW(%d,%d);SPL2(%d,%d,%d)",
//		l,t,w,h,(prop->isDocked),(prop->isMaximized),
//		(prop->splitWidth1),(prop->splitWidth2),
//		(prop->timestamp),(prop->imagesVisible),
//		(prop->topSplitWidth1),(prop->topSplitWidth2),(prop->topSplitWidth3));
	prop->rect = QRect(l,t,w,h);
}
