/**vim: ts=4
  *
  *			--- libkvistat.h ---
  *
  * This is a plugin for KVirc 2.0 Phoenix. It allows you to see
  * statistics of your IRC sessions - total words said etc.
  * More features are still to come.
  *
  * (c) Krzysztof Godlewski <kristoff@poczta.wprost.pl>
  * 
  * This program is FREE software. You can redistribute it and/or
  * modify it under the terms of the GNU General Public License
  * as published by the Free Software Foundation; either version 2
  * of the License, or (at your opinion) any later version.
  *
  * This program is distributed in the HOPE that it will be USEFUL,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.

  * You should have received a copy of the GNU General Public License
  * along with this program. If not, write to the Free Software Foundation,
  * Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
  **/
 

#ifndef __LIBKVISTAT_H__
#define __LIBKVISTAT_H__

#define LIBKVISTAT_VERSION	"0.2.1"

#include <qlist.h>
#include <qtabdialog.h>
#include "kvirc_plugin.h"

class QTabDialog;
class QFile;
class QLabel;
class QString;
class QListView;
class QListViewItem;

class KviStatChan
{
public:
	KviStatChan( const char * _name, unsigned int _joins = 1,
		unsigned int  _words = 0, unsigned int _kicks = 0,
		unsigned int _bans = 0, unsigned int _topics = 0,
		unsigned int _actions = 0 )
		{ m_name =  _name; m_joins = _joins; m_words = _words;
		  m_kicks = _kicks; m_bans = _bans; m_topics = _topics;
		  m_actions = _actions; }
	~KviStatChan() {}
	
	void addJoins( int num = 1 )	{ m_joins += num; }
	void addWords( int num = 1 )	{ m_words += num; }
	void addKicks( int num = 1 )	{ m_kicks += num; }
	void addBans( int num = 1 )		{ m_bans += num; }
	void addTopics( int num = 1 )	{ m_topics += num; }
	void addActions( int num =1 )	{ m_actions += num; }
	
	const char * name()		const { return m_name.ptr(); }
	unsigned int joins()	const { return m_joins; }
	unsigned int words()	const { return m_words; }
	unsigned int bans()		const { return m_bans; }
	unsigned int kicks()	const { return m_kicks; }
	unsigned int topics()	const { return m_topics; }
	unsigned int actions()	const { return m_actions; }
	
private:
	KviStr       m_name;
	unsigned int m_joins;
	unsigned int m_words;
	unsigned int m_kicks;
	unsigned int m_bans;
	unsigned int m_topics;
	unsigned int m_actions;
};


// Main object : acts as a filter , data keeper and so on...

typedef struct _KviStatConfig
{
	// some options:
	bool          sysTrayOnStartup;

	// Type of stat showing on channel join (No/Full/Custom)
	int           joinStatType;
	// on channel join show:			
	bool          showJoins;
	bool          showWords;
	bool          showKicks;
	bool          showBans;
	bool          showTopics;
	// in systray widget show:
	bool          STWshowJoins;
	bool          STWshowWords;
	bool          STWshowKicks;
	bool          STWshowBans;
	bool          STWshowTopics;
	// systray widget scrolling options:
	int           ScrollingWhen;
	int           ScrollingDelay;
	int           ScrollingTextJump;
	bool          ScrollingDirLeft;

} KviStatConfig;

typedef struct _KviStatData
{
    unsigned int  queriesWords;
	unsigned int  queriesLetters;
	unsigned int  dccsWords;
	unsigned int  dccsLetters;
	unsigned int  chansWords;
	unsigned int  chansLetters;
	unsigned int  totalWords;
	unsigned int  totalLetters;
	unsigned int  kicks;
	unsigned int  bans;
	unsigned int  onIrc;
	unsigned int  joins;
	unsigned int  topics;
	unsigned int  sessionWords;
	unsigned int  wordsRecord;
	KviStr        startDate;

} KviStatData;


class KviStatController : public QObject
{
	Q_OBJECT
	
public:
	KviStatController();
	~KviStatController();
	
protected:
	QList<KviStatSysTrayWidget> * m_pSysTrayWidgetList;

	QList<KviStatChan>          * m_pStatChanList;
	KviStatData                   m_stats;
	KviStatConfig                 m_config;

	bool                          m_bShowConsoleStats;
	
private:
	void loadStats();
	void saveStats() const;
	void loadConfig();
	void saveConfig() const;

public:
	void registerStatTray(KviStatSysTrayWidget *ptr);
	void unregisterStatTray(KviStatSysTrayWidget *ptr);

	void setCurrentChan(KviStatChan * ptr);
	KviStatChan * const findStatChan(const char * channame) const;
	const KviStatChan * const currentChan() const { return
			m_pStatChanList->first(); }
			
    void addChan(KviStatChan *ptr)		{ m_pStatChanList->append(ptr); }
	void removeChan(KviStatChan *ptr)	{ m_pStatChanList->removeRef(ptr); }

	const KviStatData & statData() const { return m_stats; }
	QList<KviStatChan> * const chanList() const { return m_pStatChanList;}

	bool showConsoleStats() const		{ return m_bShowConsoleStats; }
	bool sysTrayOnStartup() const		{ return m_config.sysTrayOnStartup; }

	// add to number of total actions:	
    void addTotalKick()		{ ++m_stats.kicks; };
    void addTotalBan()		{ ++m_stats.bans; };
    void addOnIrc()			{ ++m_stats.onIrc; };
    void addTotalJoin()		{ ++m_stats.joins; };
    void addTotalTopic()	{ ++m_stats.topics; };

	// add letters to total stats:
    void addChansLetters( unsigned int num )   { m_stats.chansLetters += num; }
    void addQueriesLetters( unsigned int num ) { m_stats.queriesLetters += num;}
    void addDccsLetters( unsigned int num )    { m_stats.dccsLetters += num; }
    void addTotalLetters( unsigned int num )   { m_stats.totalLetters += num; }
    
	// add words to total stats:
    void addChansWords( unsigned int num )   { m_stats.chansWords += num; }
    void addQueriesWords( unsigned int num ) { m_stats.queriesWords += num; }
    void addDccsWords( unsigned int num )    { m_stats.dccsWords += num; }
    void addTotalWords( unsigned int num )   { m_stats.totalWords += num; }

	// return number of total words ...	
    unsigned int totalWords()		const	{ return m_stats.totalWords; }
    unsigned int chansWords()		const	{ return m_stats.chansWords; }
    unsigned int queriesWords()		const	{ return m_stats.queriesWords; }
    unsigned int dccsWords()		const	{ return m_stats.dccsWords; }
    
	// ... and letters :	
    unsigned int totalLetters()		const	{ return m_stats.totalLetters; }
    unsigned int chansLetters()		const	{ return m_stats.chansLetters; }
    unsigned int queriesLetters()	const	{ return m_stats.queriesLetters; }
    unsigned int dccsLetters()		const	{ return m_stats.dccsLetters; }

	// return some total stats:		
    unsigned int numOnIrc()		const	{ return m_stats.onIrc; }
    unsigned int numKicks()		const	{ return m_stats.kicks; }
    unsigned int numBans()		const	{ return m_stats.bans; }
    unsigned int numTopics()	const	{ return m_stats.topics; }
    unsigned int numJoins()		const	{ return m_stats.joins; }

	// return last reset date (stats start date)
    const char * startDate() const			{ return m_stats.startDate.ptr(); }
	// add session words
	void addSessionWords( int num )			{ m_stats.sessionWords += num; }
	// this session words
	unsigned int sessionWords() const		{ return m_stats.sessionWords; }
	// reset session words
	void resetSessionWords()				{ m_stats.sessionWords = 0; }
	// session record
	unsigned int wordsRecord() const		{ return m_stats.wordsRecord; }
	// set words record
	void setWordsRecord( unsigned int n ) 	{ m_stats.wordsRecord = n; }

    unsigned int joinStatType() const		{ return m_config.joinStatType; }
	// checks for plugin options	
    bool showJoins()	const	{ return m_config.showJoins; }
    bool showWords()	const	{ return m_config.showWords; }
    bool showKicks()	const	{ return m_config.showKicks; }
    bool showBans()		const	{ return m_config.showBans; }
    bool showTopics()	const	{ return m_config.showTopics; }

	bool doReset();
	
protected:
	virtual bool eventFilter( QObject *, QEvent * );

public slots:
	void slotKillOptionsWindow();
	void slotKillStatWindow();
	void slotApplyOptions();
	void setSysTrayWidgetOptions();
	void slotShowConfig();
	void slotShowStats();
	void slotReset();
	void slotSaveStats()			{ saveStats(); }
	void slotRemoveAllChans();
};
    
#endif	// ! __LIBKVISTAT_H__
