// =============================================================================
//
//      --- kvi_irc_view.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviIrcView"

#define _KVI_IRC_VIEW_CPP_

#include <qcursor.h>
#include <qdragobject.h>
#include <qfile.h>
#include <qpainter.h>
#include <qscrollbar.h>

#include "kvi_app.h"
#include "kvi_console.h"
#include "kvi_dcc_chat.h"
#include "kvi_debug.h"
#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_irc_view.h"
#include "kvi_locale.h"
#include "kvi_malloc.h"
#include "kvi_memmove.h"
#include "kvi_messagebox.h"
#include "kvi_mirccntrl.h"
#include "kvi_options.h"
#include "kvi_regusersdb.h"
#include "kvi_statusbar.h"
#include "kvi_userlistbox.h"
#include "kvi_userparser.h"

/**
 *
 * Damn complex class, but it works :)
 *
 * 1999-11-23:
 *      Just for fun, complicated the things a little bit more:
 *      Added precalculation of the text blocks and word wrapping
 *      and a fast scrolling mode (3 lines at once) for consecutive
 *      appendText() calls.
 *      Now the code becomes really not understandable... :)
 */

// Declared in kvi_app.cpp and managed by KviApp class
extern QPixmap              *g_pixViewOut[KVI_OUT_NUM_IMAGES];
extern QPixmap              *g_pIrcViewMemBuffer;
extern Qt::HANDLE            g_hIrcViewMemBuffer;
extern QPtrList<KviIrcView> *g_pIrcViewWidgetList;
extern KviEventManager      *g_pEventManager;

#define KVI_IRC_VIEW_BLOCK_SELECTION_TOTAL   0
#define KVI_IRC_VIEW_BLOCK_SELECTION_LEFT    1
#define KVI_IRC_VIEW_BLOCK_SELECTION_RIGHT   2
#define KVI_IRC_VIEW_BLOCK_SELECTION_CENTRAL 3

// TODO: PgUp and PgDn scrolls a fixed number of lines! Make it view height dependant

/**
 * ================ getColorBytes ===============
 * Scans the data_ptr for a mIrc color code XX, XX
 * and fills the color values in the two bytes
 */
const char *getColorBytes(const char *data_ptr, char *byte_1, char *byte_2)
{
	// First we can have a digit or a coma
	if( ((*data_ptr >= '0') && (*data_ptr <='9')) ) {
		// Something interesting
		(*byte_1) = (*data_ptr) - '0'; // Store the code
		data_ptr++;                    // And check the next
		if( ((*data_ptr >= '0') && (*data_ptr <= '9')) || (*data_ptr == ',') ) {
			// Yes, we can understand it
			if( *data_ptr == ',' ) {
				// A comma, need to check for background
				data_ptr++;
			} else {
				// A number.
				// TODO: maybe move the modulus operation to shifts?
				(*byte_1) = ((((*byte_1) * 10) + ((*data_ptr) - '0')) % 16);
				data_ptr++;
				if( *data_ptr == ',' ) {
					// A comma, need to check for background
					data_ptr++;
				} else {
					// Senseless, return
					(*byte_2) = KVI_NOCHANGE;
					return data_ptr;
				}
			}
		} else {
			// Senseless character control code OK, return
			(*byte_2) = KVI_NOCHANGE;
			return data_ptr;
		}
	} else {
		// Senseless character : only a CTRL+K code
		(*byte_1) = KVI_NOCHANGE;
		(*byte_2) = KVI_NOCHANGE;
		return data_ptr;
	}

	if( (*data_ptr >= '0') && (*data_ptr <= '9') ) {
		// Background, a color code
		(*byte_2) = (*data_ptr) - '0';
		data_ptr++;
		if( (*data_ptr >= '0') && (*data_ptr <= '9') ) {
			(*byte_2) = ((((*byte_2) * 10) + ((*data_ptr) - '0')) % 16);
			data_ptr++;
		}
		return data_ptr;
	} else {
		(*byte_2) = KVI_NOCHANGE;
		return data_ptr;
	}
}

/**
 * ============ KviIrcView ============
 */
KviIrcView::KviIrcView(QWidget *parent, KviFrame *pFrm, KviWindow *pWnd)
	: QWidget(parent, "irc_view")
{
	g_pIrcViewWidgetList->append(this); // Register
	setBackgroundMode(NoBackground);

	m_pFirstLine               = 0;
	m_pCurLine                 = 0;
	m_pLastLine                = 0;
	m_iNumLines                = 0;
	m_iMaxLines                = g_pOptions->m_iViewMaxBufferSize;
	m_bAcceptDrops             = false;
	m_pPrivateBackgroundPixmap = 0;
	m_pScrollBar               = new QScrollBar(0, 0, 1, 10, 0, QScrollBar::Vertical, this, "irc_view_scrollbar");
	m_iSelectionBottom         = 0;
	m_iSelectionTop            = 0;

	m_pScrollBar->setTracking(true);
	m_pScrollBar->hide();
	m_pScrollBar->setFocusProxy(this);
	m_bNeedScrollBar = false;
	connect(m_pScrollBar, SIGNAL(valueChanged(int)), this, SLOT(scrollBarPositionChanged(int)));

	m_iLastScrollBarValue = 0;
	m_bSelecting          = false;
	m_bDoubleClicked      = false;
	m_bShowImages         = g_pOptions->m_bShowImages;
	m_iSelectTimer        = 0;
	m_bTimestamp          = g_pOptions->m_bTimestamp;

	setFont(g_pOptions->m_fntView);
	// The following may be useful;
	// fontChange is not triggered when g_pOptions->m_fntView == font()
	recalcFontVariables(g_pOptions->m_fntView);

	m_bSkipScrollBarRepaint          = false;
	m_pLogFile                       = 0;
	m_pKviWindow                     = pWnd;
	m_pFrm                           = pFrm;
	m_iUnprocessedPaintEventRequests = 0;
	m_bPostedPaintEventPending       = false;
	// Links highlighting
	m_pLastLinkUnderMouse            = 0;
	m_iLastLinkRectTop               = -1;
	m_iLastLinkRectHeight            = -1;

	m_pMessagesStoppedWhileSelecting = new QPtrList<KviIrcViewTextLine>;
	m_pMessagesStoppedWhileSelecting->setAutoDelete(false);
	setMinimumWidth(KVI_IRC_VIEW_MINIMUM_WIDTH);

	if( m_pKviWindow->type() == KVI_WND_TYPE_QUERY )
		m_bBeeping = initBeeping(m_pKviWindow->caption());
	else
		m_bBeeping = false;

	setMouseTracking(true);
	resizeMemBuffer();
}

/**
 * ============ ~KviIrcView ============
 */
KviIrcView::~KviIrcView()
{
	g_pIrcViewWidgetList->removeRef(this); // Unregister

	if( m_iSelectTimer )
		killTimer(m_iSelectTimer);
	stopLogging();
	if( m_pPrivateBackgroundPixmap ) {
		delete m_pPrivateBackgroundPixmap;
		m_pPrivateBackgroundPixmap = 0;
	}
	emptyBuffer(false);
	KviIrcViewTextLine *l;
	while( (l = m_pMessagesStoppedWhileSelecting->first()) ) {
		m_pMessagesStoppedWhileSelecting->removeFirst();
		kvi_free(l->data_ptr);                              // Free string data
		if( l->attr_ptr->escape_cmd )
			kvi_free(l->attr_ptr->escape_cmd);
		kvi_free(l->attr_ptr);                              // Free attributes data
		if( l->num_text_blocks )
			kvi_free(l->text_blocks_ptr);
		delete l;
	}
	delete m_pMessagesStoppedWhileSelecting;
	m_pMessagesStoppedWhileSelecting = 0;
}

void KviIrcView::wheelEvent(QWheelEvent *e)
{
	if( m_bNeedScrollBar )
		g_pApp->sendEvent(m_pScrollBar, e);
}

void KviIrcView::enableDnD(bool bEnable)
{
	setAcceptDrops(bEnable);
	m_bAcceptDrops = bEnable;
}

void KviIrcView::dragEnterEvent(QDragEnterEvent *e)
{
	if( !m_bAcceptDrops ) return;

	e->accept(QUriDrag::canDecode(e));
	emit dndEntered();
}

void KviIrcView::dropEvent(QDropEvent *e)
{
	if( !m_bAcceptDrops ) return;

	QStringList list;
	if( QUriDrag::decodeLocalFiles(e, list) ) {
		if( !list.isEmpty() ) {
			QStringList::ConstIterator it = list.begin();
			for( ; it != list.end(); ++it ) {
				KviStr tmp = *it;
				emit fileDropped(tmp.ptr());
			}
		}
	}
}

/**
 * ============= BEEPING ===============
 */
void KviIrcView::toggleBeeping()
{
	if( isBeeping() ) {
		if( m_pKviWindow->type() == KVI_WND_TYPE_QUERY )
			stopBeeping(m_pKviWindow->caption());
		else if( m_pKviWindow->type() == KVI_WND_TYPE_CHAT )
			stopBeeping(((KviDccChat *) m_pKviWindow)->m_szRemoteNick.ptr());
	} else {
		if( m_pKviWindow->type() == KVI_WND_TYPE_QUERY )
			startBeeping(m_pKviWindow->caption());
		else if( m_pKviWindow->type() == KVI_WND_TYPE_CHAT )
			startBeeping(((KviDccChat *) m_pKviWindow)->m_szRemoteNick.ptr());
	}
}

void KviIrcView::stopBeeping(const char *user)
{
	KviRegisteredUser *tmpU;
	KviStr currentFlags;
	// Forms our hostname-and-username insensitive mask:
	KviStr tmpS(KviStr::Format, "%s!*@*", user); // We are only matching on the nick

	tmpU = g_pOptions->m_pRegUsersDb->findUserByMask(tmpS.ptr());
	if( tmpU ) {
		g_pOptions->m_pRegUsersDb->getFlags(tmpS.ptr(), currentFlags); // Find what other flags exist
		if( currentFlags.contains("a") )                     // The beep control code added this user.
			g_pOptions->m_pRegUsersDb->unregisterUser(tmpU); // For tidyness, delete them.
		else
			g_pOptions->m_pRegUsersDb->deleteFlags(tmpU, "b");
	} else if( !g_pOptions->m_bAddUsersForBeepControl ) {
		// If user cannot be found but the setting is false, it might be that
		// they were never added at user's behest. So it is not a problem.
		debug("Whoa... could not find the user that I put in there.\nEvil is afoot...\n");
	}

	m_bBeeping = false;
}

bool KviIrcView::isBeeping()
{
	return m_bBeeping;
}

void KviIrcView::startBeeping(const char *user)
{
	KviRegisteredUser *tmpU = new KviRegisteredUser();
	KviIrcUser tmpI;
	// Forms our hostname-and-username insensitive mask:
	KviStr tmpS(KviStr::Format, "%s!*@*", user); // We are only matching on the nick

	// If user is not in the list, they need to be.
	// That is just how I am doing it, okay? :)
	tmpU = g_pOptions->m_pRegUsersDb->findUserByMask(tmpS.ptr());
	if( !tmpU && g_pOptions->m_bAddUsersForBeepControl ) {
		// Need to fill out a KviRegisteredUser object.
		// For that we first need a KviIrcUser object
		tmpI.setNick(user);
		tmpI.setHost("");
		tmpI.setUsername("");

		tmpU = new KviRegisteredUser();
		tmpU->user = tmpI;
		tmpU->allFlags = KviStr("ab");
		tmpU->passwd   = KviStr("-");
		tmpU->comment  = KviStr("Auto-entry for beep control");
		tmpU->bNotify  = 0;
		tmpU->bIgnore  = 0;

		g_pOptions->m_pRegUsersDb->registerUser(tmpU, "ab");
	} else if( tmpU ) {
		g_pOptions->m_pRegUsersDb->addFlags(tmpU, "b");
	}
	m_bBeeping = true;
}

bool KviIrcView::initBeeping(const char *user)
{
	KviRegisteredUser *tmpU = 0;
	KviStr currentFlags;
	// Forms our hostname-and-username insensitive mask:
	KviStr tmpS(KviStr::Format, "%s!*@*", user); // We are only matching on the nick

	tmpU = g_pOptions->m_pRegUsersDb->findUserByMask(tmpS.ptr());
	if( tmpU ) {
		// Find what other flags exist
		g_pOptions->m_pRegUsersDb->getFlags(tmpS.ptr(), currentFlags);
		if( currentFlags.contains("b") )
			return true;
	}
	return false;
}

/**
 * ============= LOGGING ===============
 */
void KviIrcView::stopLogging()
{
	if( m_pLogFile ) {
		QCString tmp = QDateTime::currentDateTime().toString().local8Bit();
		KviStr szLogEnd(KviStr::Format, _i18n_("### Log session terminated at %s ###"), tmp.data());
		add2Log(szLogEnd.ptr());
		m_pLogFile->close();
		delete m_pLogFile;
		m_pLogFile = 0;
	}
}

bool KviIrcView::isLogging()
{
	return (m_pLogFile != 0);
}

void KviIrcView::getTextBuffer(KviStr &buffer)
{
	buffer = "";
	if( !m_pLastLine ) return;

	for( KviIrcViewTextLine *l = m_pFirstLine; l; l = l->next_line ) {
		buffer.append(l->data_ptr);
		buffer.append("\n");
	}
}

void KviIrcView::toggleLogging()
{
	if( isLogging() )
		stopLogging();
	else {
		KviStr tmp;
		m_pKviWindow->getDefaultLogName(tmp);
		startLogging(tmp.ptr());
	}
}

void KviIrcView::logToFile()
{
	// Yeah, this is powerful! :)
	QString cmd = "/DIALOG (savefile, Choose a log file name, $deflogfile($window), $window) "\
		"if((\"$dialogresult\" != \"\") && (\"$dialogmagic\" == \"$window\"))"\
		"log on $dialogresult";
	m_pFrm->m_pUserParser->parseUserCommand(cmd, m_pKviWindow);
}

void KviIrcView::saveBufferToFile()
{
	// Yeah, this is powerful! :)
	QString cmd = "/DIALOG (savefile, Choose a file name, $deflogfile($window).savebuf, $window) "\
		"if(\"$dialogresult\" != \"\")"\
		"window $dialogmagic savebuffer $dialogresult";
	m_pFrm->m_pUserParser->parseUserCommand(cmd, m_pKviWindow);
}

void KviIrcView::toggleTimestamp()
{
	setTimestamp(!timestamp());
}

void KviIrcView::toggleImages()
{
	setShowImages(!imagesVisible());
}

void KviIrcView::clearBuffer()
{
	emptyBuffer(true);
}

bool KviIrcView::saveBuffer(const char *filename)
{
	QFile f(filename);
	if( !f.open(IO_WriteOnly | IO_Truncate) )
		return false;
	KviStr tmp;
	getTextBuffer(tmp);
	f.writeBlock(tmp.ptr(), tmp.len());
	f.close();
	return true;
}

/**
 * ============== startLogging ===============
 */
bool KviIrcView::startLogging(const char *filename)
{
	stopLogging();

	m_pLogFile = new QFile(filename);
	if( m_pLogFile->exists() ) {
		if( !m_pLogFile->open(IO_Append | IO_WriteOnly) ) {
			delete m_pLogFile;
			m_pLogFile = 0;
			return false;
		}
	} else {
		if( !m_pLogFile->open(IO_WriteOnly) ) {
			delete m_pLogFile;
			m_pLogFile = 0;
			return false;
		}
	}
	QCString tmp = QDateTime::currentDateTime().toString().local8Bit();
	KviStr szLogStart(KviStr::Format, _i18n_("### Log session started at %s ###"), tmp.data());
	add2Log(szLogStart.ptr());
	return true;
}

/**
 * =============== add2Log ==============
 */
void KviIrcView::add2Log(const char *buffer, int buf_len)
{
	if( !m_pLogFile ) return;

	if( buf_len < 0 )
		buf_len = strlen(buffer);
	if( (m_pLogFile->writeBlock(buffer, buf_len) == -1) || (m_pLogFile->putch('\n') == -1) )
		debug("WARNING: cannot write to the log file.");
}

void KviIrcView::prevLine()    { m_pScrollBar->subtractLine(); }
void KviIrcView::nextLine()    { m_pScrollBar->addLine();      }
void KviIrcView::prevPage()    { m_pScrollBar->subtractPage(); }
void KviIrcView::nextPage()    { m_pScrollBar->addPage();      }

/**
 * ============ setPrivateBackgroundPixmap ===========
 */
void KviIrcView::setPrivateBackgroundPixmap(const QPixmap &pixmap, bool bRepaint)
{
	if( m_pPrivateBackgroundPixmap ) {
		delete m_pPrivateBackgroundPixmap;
		m_pPrivateBackgroundPixmap = 0;
	}
	if( !pixmap.isNull() )
		m_pPrivateBackgroundPixmap = new QPixmap(pixmap);
	if( bRepaint )
		paintEvent(0);
}

bool KviIrcView::hasPrivateBackgroundPixmap()
{
	return (m_pPrivateBackgroundPixmap != 0);
}

/**
 * ============= scrollBarPositionChanged =============
 */
void KviIrcView::scrollBarPositionChanged(int newValue)
{
	if( !m_pCurLine ) return;

	int diff = 0;
	if( newValue > m_iLastScrollBarValue ) {
		while( newValue > m_iLastScrollBarValue ) {
			if( m_pCurLine->next_line ) {
				m_pCurLine = m_pCurLine->next_line;
				diff++;
			}
			m_iLastScrollBarValue++;
		}
	} else {
		while( newValue < m_iLastScrollBarValue ) {
			if( m_pCurLine->prev_line )
				m_pCurLine = m_pCurLine->prev_line;
			m_iLastScrollBarValue--;
		}
	}
	__range_valid(newValue == m_iLastScrollBarValue);
	if( !m_bSkipScrollBarRepaint )
		paintEvent(0);
}

/**
 * ============= emptyBuffer ==============
 */
void KviIrcView::emptyBuffer(bool bRepaint)
{
	while( m_pLastLine )
		removeHeadLine();
	if( bRepaint )
		paintEvent(0);
}

/**
 * ============= setMaxBufferSize ===============
 */
void KviIrcView::setMaxBufferSize(int maxBufSize, bool bRepaint)
{
	if( maxBufSize < 32 )
		maxBufSize = 32;
	m_iMaxLines = maxBufSize;
	while( m_iNumLines > m_iMaxLines )
		removeHeadLine();
	m_pScrollBar->setRange(0, m_iNumLines);
	if( bRepaint )
		paintEvent(0);
}

/**
 * ================ setShowImages ===============
 */
void KviIrcView::setShowImages(bool bShow, bool bRepaint)
{
	if( m_bShowImages != bShow ) {
		m_bShowImages = bShow;
		if( bRepaint )
			paintEvent(0);
	}
}

/**
 * ================ setTimestamp ================
 */
void KviIrcView::setTimestamp(bool bTimestamp)
{
	m_bTimestamp = bTimestamp;
}

bool KviIrcView::timestamp()
{
	return m_bTimestamp;
}

bool KviIrcView::imagesVisible()
{
	return m_bShowImages;
}

/**
 * ============= appendLine =============
 */
bool KviIrcView::event(QEvent *e)
{
	if( e->type() == QEvent::User ) {
		__range_valid(m_bPostedPaintEventPending);
		if( m_iUnprocessedPaintEventRequests )
			paintEvent(0);
		// else: we just had a pointEvent that did the job
		m_bPostedPaintEventPending = false;
		return true;
	}
	return QWidget::event(e);
}

void KviIrcView::postUpdateEvent()
{
	if( !m_bPostedPaintEventPending ) {
		m_bPostedPaintEventPending = true;
		QEvent *e = new QEvent(QEvent::User);
		g_pApp->postEvent(this, e); // Queue a repaint
	}
	m_iUnprocessedPaintEventRequests++; // paintEvent() will set it to 0
	if( m_iUnprocessedPaintEventRequests == 3 ) {
		if( g_pOptions->m_pixViewBack->isNull() && (!m_pPrivateBackgroundPixmap) )
			fastScroll(3);
		else
			paintEvent(0);
	}
}

void KviIrcView::appendLine(KviIrcViewTextLine *ptr, bool bRepaint)
{
	if( isBeeping() && !m_pKviWindow->hasFocus() && g_pOptions->m_bEnableBeeping )
		KviApplication::beep();

	// This one appends a KviIrcViewTextLine to the buffer list
	if( m_bSelecting ) {
		// Do not move the view!
		// So we append the text line to a temp queue
		// and then we'll add it when the mouse button is released
		m_pMessagesStoppedWhileSelecting->append(ptr);
		return;
	}

	if( m_pLastLine ) {
		m_pLastLine->next_line = ptr;
		ptr->prev_line         = m_pLastLine;
		ptr->next_line         = 0;
		m_iNumLines++;
		if( m_iNumLines > m_iMaxLines ) {
			removeHeadLine();
			if( m_pCurLine == m_pLastLine ) {
				m_pCurLine = ptr;
				if( bRepaint )
					postUpdateEvent();
			} else {
				// The current line remains the same.
				// The scroll bar must move up one place to be in sync
				m_bSkipScrollBarRepaint = true;
				if( m_pScrollBar->value() > 0 ) {
					m_iLastScrollBarValue--;
					__range_valid(m_iLastScrollBarValue >= 0);
					m_pScrollBar->subtractLine();
				} // else: will stay in sync
				m_bSkipScrollBarRepaint = false;
			}
		} else {
			m_pScrollBar->setRange(0, m_iNumLines);
			if( m_pCurLine == m_pLastLine ) {
				m_bSkipScrollBarRepaint = true;
				m_pScrollBar->addLine();
				m_bSkipScrollBarRepaint = false;
				if( bRepaint )
					postUpdateEvent();
			}
		}
		m_pLastLine = ptr;
	} else {
		// First line
		m_pLastLine    = ptr;
		m_pFirstLine   = ptr;
		m_pCurLine     = ptr;
		ptr->prev_line = 0;
		ptr->next_line = 0;
		m_iNumLines    = 1;
		m_pScrollBar->setRange(0, 1);
		m_pScrollBar->addLine();
		if( bRepaint )
			paintEvent(0);
	}

	if( m_pLogFile ) {
		if( g_pOptions->m_bLogMsgType[ptr->msg_type] ) {
			add2Log(ptr->data_ptr, ptr->data_len - 1);
		}
	}
}

/**
 * ============== removeHeadLine ===============
 * Removes the first line of the text buffer.
 */
void KviIrcView::removeHeadLine(bool bRepaint)
{
	if( !m_pLastLine ) return;
	__range_valid(m_pFirstLine->prev_line == 0);
	__range_valid(m_pFirstLine->data_ptr);
	kvi_free(m_pFirstLine->data_ptr);                          // Free string data
	if( m_pFirstLine->attr_ptr->escape_cmd )
		kvi_free(m_pFirstLine->attr_ptr->escape_cmd);
	kvi_free(m_pFirstLine->attr_ptr);                          // Free attributes data
	if( m_pFirstLine->num_text_blocks )
		kvi_free(m_pFirstLine->text_blocks_ptr);
	if( m_pFirstLine->next_line ) {
		KviIrcViewTextLine *aux_ptr = m_pFirstLine->next_line; // Get the next line
		aux_ptr->prev_line = 0;                                // Becomes the first
		if( m_pFirstLine == m_pCurLine )
			m_pCurLine = aux_ptr;                              // Move the cur line if necessary
		delete m_pFirstLine;                                   // Delete the struct
		m_pFirstLine = aux_ptr;                                // Set the last...
		m_iNumLines--;                                         // ... and decrement the count
	} else { // Unique line
		delete m_pFirstLine;
		m_pFirstLine = 0;
		m_pCurLine   = 0;
		m_iNumLines  = 0;
		m_pLastLine  = 0;
	}
	if( bRepaint )
		paintEvent(0);
}

/**
 * ============= appendText ===============
 * Appends a text string to the buffer list; splits the lines
 */
void KviIrcView::appendText(int msg_type, const char *data_ptr, bool bRepaint)
{
	__range_valid(data_ptr);
	m_pLastLinkUnderMouse = 0;

	while( *data_ptr ) { // Have more data
		KviIrcViewTextLine *line_ptr = new KviIrcViewTextLine();
		line_ptr->msg_type        = msg_type;
		line_ptr->max_line_width  = -1;
		line_ptr->num_text_blocks = 0;
		data_ptr = getTextLine(msg_type, data_ptr, line_ptr);

		appendLine(line_ptr, bRepaint);
	}
}

void KviIrcView::getDoubleClickCommand(KviStr &buffer, const char *escape_cmd)
{
	while( *escape_cmd == ' ' )
		escape_cmd++;
	while( *escape_cmd == '[' ) {
		if( kvi_strEqualCIN(escape_cmd, "[dbl]", 5) ) {
			escape_cmd += 5;
			buffer = escape_cmd;
			int idx = buffer.findFirstIdx("[rbt]");
			if( idx != -1 )
				buffer.setLen(idx);
			return;
		}
		escape_cmd++;
		while( *escape_cmd && (*escape_cmd != '[') )
			escape_cmd++;
	}
	buffer = escape_cmd; // Whole command is a double click command
}

void KviIrcView::getRightClickCommand(KviStr &buffer, const char *escape_cmd)
{
	while( *escape_cmd == ' ' )
		escape_cmd++;
	while( *escape_cmd == '[' ) {
		if( kvi_strEqualCIN(escape_cmd, "[rbt]", 5) ) {
			escape_cmd += 5;
			buffer = escape_cmd;
			return;
		}
		escape_cmd++;
		while( *escape_cmd && (*escape_cmd != '[') )
			escape_cmd++;
	}
}

/**
 * ============= getTextLine ===============
 * Splits the text data in lines separated by '\n'
 */
const char *KviIrcView::getTextLine(int msg_type, const char *data_ptr, KviIrcViewTextLine *line_ptr)
{
	int buffer_pos_idx = 0;       // We are at the beginning in the buffer
	int curAttrIdx     = 0;
	int blockLen;
	register const char *p = data_ptr;

	// Alloc the first attribute
	line_ptr->attr_len = 1;
	line_ptr->attr_ptr = (KviIrcViewTextAttribute *) kvi_malloc(sizeof(KviIrcViewTextAttribute));
	// And fill it up
	line_ptr->attr_ptr[0].attribute  = KVI_TEXT_COLOR;
	line_ptr->attr_ptr[0].block_idx  = 0;
	line_ptr->attr_ptr[0].escape_cmd = 0;

	if( m_bTimestamp ) {
		line_ptr->data_ptr              = (char *) kvi_malloc(12);     // [hh:mm:ss]space\0
		line_ptr->data_len              = 12;
		line_ptr->attr_ptr[0].block_len = 11;
		buffer_pos_idx                  = 11;
		*(line_ptr->data_ptr)           = '[';
		kvi_fastmove((void *) (line_ptr->data_ptr + 1), (void *) QTime::currentTime().toString().latin1(), 8);
		*((line_ptr->data_ptr) + 9)     = ']';
		*((line_ptr->data_ptr) + 10)    = ' ';
	} else {
		line_ptr->data_ptr              = (char *) kvi_malloc(1);      // At least the terminator
		line_ptr->data_len              = 1;
		line_ptr->attr_ptr[0].block_len = 0;
	}

	line_ptr->attr_ptr[0].attr_back = g_pOptions->m_cViewOutTypeBack[msg_type];
	line_ptr->attr_ptr[0].attr_fore = g_pOptions->m_cViewOutTypeFore[msg_type];

	// Quick-and-dirty macros to make the code somewhat more simple:
#define APPEND_LAST_TEXT_BLOCK(__data_ptr, __data_len) \
	blockLen = __data_len; \
	line_ptr->data_len                       += blockLen; \
	line_ptr->attr_ptr[curAttrIdx].block_len += blockLen; \
	line_ptr->data_ptr = (char *) kvi_realloc((void *) line_ptr->data_ptr, line_ptr->data_len); \
	kvi_fastmove((void *) (line_ptr->data_ptr + buffer_pos_idx), (void *) __data_ptr, blockLen); \
	buffer_pos_idx += blockLen;

#define NEW_ATTRIBUTE_BLOCK(__attr_type) \
	line_ptr->attr_len++; \
	line_ptr->attr_ptr = (KviIrcViewTextAttribute *) kvi_realloc( \
		(void *) line_ptr->attr_ptr, line_ptr->attr_len * sizeof(KviIrcViewTextAttribute) \
	); \
	curAttrIdx++; \
	line_ptr->attr_ptr[curAttrIdx].attribute  = __attr_type; \
	line_ptr->attr_ptr[curAttrIdx].block_idx  = buffer_pos_idx; \
	line_ptr->attr_ptr[curAttrIdx].block_len  = 0; \
	line_ptr->attr_ptr[curAttrIdx].escape_cmd = 0;

	for( ;; ) {
		if( g_pOptions->m_bUrlHighlighting ) {
			while(
				(((unsigned char) *p) > 31) && (*p != '~') && (*p != 'h') &&
				(*p != 'f') && (*p != 'F') && (*p != 'w') && (*p != 'W')
			) {
				p++;
			}
		} else {
			while( (((unsigned char) *p) > 31) && (*p != '~') )
				p++;
		}
		switch( *p ) {
			case '\0':
				// Found the end of the string.
				APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr)
				// Terminate the string
				line_ptr->data_ptr[buffer_pos_idx] = '\0';
				return p;
				break;
			case '\t': {
				// Replaces tabs with spaces
				static QString tab;
				if( tab.isNull() )
					tab = tab.fill(' ', KVI_TAB_WIDTH);
				APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr);
				APPEND_LAST_TEXT_BLOCK(tab.ascii(), KVI_TAB_WIDTH);
				p++;
				data_ptr = p;
				break;
			}
			case '\r':
				// Command escape sequence
				// \r![dbl]double_click_command [rbt]right_click_command\rparameters string\r/!\r
				p++;
				if( *p == '!' ) {
					const char *next_cr = p;
					// Look up the next carriage return
					while( *next_cr && (*next_cr != '\r') )
						next_cr++;
					if( *next_cr ) {
						const char *term_cr = next_cr;
						term_cr++;
						while( *term_cr && (*term_cr != '\r') )
							term_cr++;
						if( *term_cr ) {
							// OK, the format is right:
							//  \r!    ... \r ...    \r
							//    ^p        ^next_cr  ^term_cr
							p--;
							//  \r!    ... \r ...    \r
							//   ^p         ^next_cr  ^term_cr
							APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr)

							// Create a new attribute block
							NEW_ATTRIBUTE_BLOCK(KVI_TEXT_ESCAPE)
							line_ptr->attr_ptr[curAttrIdx].escape_cmd = 0;
							p += 2; // Point after \r!
							blockLen = (next_cr - p);
							line_ptr->attr_ptr[curAttrIdx].escape_cmd = (char *) kvi_malloc((next_cr - p) + 1);
							kvi_fastmove((void *) (line_ptr->attr_ptr[curAttrIdx].escape_cmd), p, blockLen);
							line_ptr->attr_ptr[curAttrIdx].escape_cmd[blockLen] = '\0';
							line_ptr->attr_ptr[curAttrIdx].attr_fore            = KVI_NOCHANGE;
							++next_cr; // Point after the middle \r
							APPEND_LAST_TEXT_BLOCK(next_cr, term_cr - next_cr)

							// Create a new attribute block
							NEW_ATTRIBUTE_BLOCK(KVI_TEXT_UNESCAPE)
							p = ++term_cr;
							data_ptr = p;
						}
					}
					break;
				} else { // Lone '\r'
					p--;
				} // Fall-through to \n (replace lone '\r' with '\n')
			case '\n':
				// Found the end of a line
				APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr)
				// Terminate the string
				line_ptr->data_ptr[buffer_pos_idx] = '\0';
				// Move the current pointer to the next character... if it is '\0' we will simply stop
				p++;
				return p;
				break;
			case KVI_TEXT_BOLD:    // Fall-through
			case KVI_TEXT_RESET:   // Fall-through
			case KVI_TEXT_REVERSE: // Fall-through
			case KVI_TEXT_UNDERLINE:
				// Control code... need a new attribute struct
				// Found the end of a line
				APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr)
				// Create a new attribute block
				NEW_ATTRIBUTE_BLOCK(*p)
				// Move the current pointer to the next character... if it is '\0' we will simply stop
				data_ptr = ++p;
				break;
			case KVI_TEXT_COLOR:
				// Color control code... need a new attribute struct
				APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr)
				// Create a new attribute block
				NEW_ATTRIBUTE_BLOCK(*p)
				// Move the current pointer to the next character... if it is '\0' we will simply stop
				p++;
				p = getColorBytes(p,
					&(line_ptr->attr_ptr[curAttrIdx].attr_fore),
					&(line_ptr->attr_ptr[curAttrIdx].attr_back)
				);
				data_ptr = p;
				break;
			case '~':
				if( (g_pOptions->m_bUseKsircControlCodes) && (line_ptr->attr_ptr[curAttrIdx].attribute != *p) ) {
					// Control code... need a new attribute struct
					APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr)
					// Create a new attribute block
					NEW_ATTRIBUTE_BLOCK(*p)
					// Move the current pointer to the next character... if it is '\0' we will simply stop
					p++;
					switch( *p ) {
						case 'c':
							line_ptr->attr_ptr[curAttrIdx].attribute = KVI_TEXT_RESET;
							p++;
							break;
						case 'b':
							line_ptr->attr_ptr[curAttrIdx].attribute = KVI_TEXT_BOLD;
							p++;
							break;
						case 'i': // TODO: italics not yet supported
							p++;
							break;
						case 'r':
							line_ptr->attr_ptr[curAttrIdx].attribute = KVI_TEXT_REVERSE;
							p++;
							break;
						case 'u':
							line_ptr->attr_ptr[curAttrIdx].attribute = KVI_TEXT_UNDERLINE;
							p++;
							break;
						case '~': // Just a tilde
							break;
						default:
							line_ptr->attr_ptr[curAttrIdx].attribute = KVI_TEXT_COLOR;
							p = getColorBytes(p,
								&(line_ptr->attr_ptr[curAttrIdx].attr_fore),
								&(line_ptr->attr_ptr[curAttrIdx].attr_back)
							);
							break;
						}
					data_ptr = p;
				} else p++;
				break;
			case 'h': // Fall-through
			case 'f': // Fall-through
			case 'F':
				p++;
				if( (*p == 't') || (*p == 'T') ) {
					p--;
					int partLen = 0;
					if( kvi_strEqualCIN(p, "ftp.", 4) )          partLen = 4;
					else if( kvi_strEqualCSN(p, "ftp://", 6)   ) partLen = 6;
					else if( kvi_strEqualCSN(p, "http://", 7)  ) partLen = 7;
					else if( kvi_strEqualCSN(p, "https://", 8) ) partLen = 8;
					p += partLen;
					if( (partLen > 0) && (((unsigned char) *p) > 47) ) {
						p -= partLen;
						// URL highlighting block
						APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr)
						// Create a new attribute block
						NEW_ATTRIBUTE_BLOCK(KVI_TEXT_ESCAPE)
						line_ptr->attr_ptr[curAttrIdx].escape_cmd = (char *) kvi_malloc(11);
						kvi_fastmove((void *) (line_ptr->attr_ptr[curAttrIdx].escape_cmd), "openurl $1", 11);
						line_ptr->attr_ptr[curAttrIdx].attr_fore  = g_pOptions->m_cViewOutUrlFore;
						// Move the current pointer after the 'http:/' block
						// and run until the presumed end of the URL
						data_ptr = p;
						p += partLen;
						while( (((unsigned char) *p) > 32) &&
							(*p != '[') && (*p != ']') &&
							(*p != '(') && (*p != ')') &&
							(*p != '{') && (*p != '}')
						) {
							p++;
						}

						APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr)
						// Fire off event if necessary
						if( g_pEventManager->eventEnabled(KviEvent_OnUrl) ) {
							const char *tmp = data_ptr;
							KviStr tmpurl(tmp, blockLen);
							m_pFrm->m_pUserParser->callEvent(
								KviEvent_OnUrl, m_pKviWindow ? m_pKviWindow : m_pFrm->m_pConsole, tmpurl
							);
						}
						// Create a new attribute block
						NEW_ATTRIBUTE_BLOCK(KVI_TEXT_UNESCAPE)
						data_ptr = p;
					} else p -= partLen - 1;
				}
				break;
			case 'w': // Fall-through
			case 'W':
				p++;
				if( (*p == 'w') || (*p == 'W') ) {
					if( kvi_strEqualCIN(p, "ww.", 3) ) { // www.
						p += 3;
						if( ((unsigned char) *p) < 48 )
							p -= 3;
						else {
							p -= 4;
							// URL highlighting block
							APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr)
							// Create a new attribute block
							NEW_ATTRIBUTE_BLOCK(KVI_TEXT_ESCAPE)
							line_ptr->attr_ptr[curAttrIdx].escape_cmd = (char *) kvi_malloc(11);
							kvi_fastmove((void *) (line_ptr->attr_ptr[curAttrIdx].escape_cmd), "openurl $1", 11);
							line_ptr->attr_ptr[curAttrIdx].attr_fore  = g_pOptions->m_cViewOutUrlFore;
							// Move the current pointer after the 'http:/' block
							// And run until the presumed end of the URL
							data_ptr = p;
							p += 4;
							while( (((unsigned char) *p) > 32) &&
								(*p != '[') && (*p != ']') &&
						    	(*p != '(') && (*p != ')') &&
								(*p != '{') && (*p != '}')
							) {
								p++;
							}

							APPEND_LAST_TEXT_BLOCK(data_ptr, p - data_ptr)
							// Fire off event if necessary
							if( g_pEventManager->eventEnabled(KviEvent_OnUrl) ) {
								const char *tmp = data_ptr;
								KviStr tmpurl(tmp, blockLen);
								m_pFrm->m_pUserParser->callEvent(
									KviEvent_OnUrl, m_pKviWindow ? m_pKviWindow : m_pFrm->m_pConsole, tmpurl
								);
							}
							// Create a new attribute block
							NEW_ATTRIBUTE_BLOCK(KVI_TEXT_UNESCAPE)
							data_ptr = p;
						}
					}
				}
				break;
			default:
				p++;
				break;
		}
	}
}

/**
 * ================ drawBlockPart =================
 */
inline void KviIrcView::drawBlockPart(
	QPainter *pa, int sb_width, char curFore, char curBack, bool curBold, bool curUnderline,
	int &curLeftCoord, int curBottomCoord, const char *data, int len, int wdth)
{
	pa->setPen(*(g_pOptions->m_pMircColor[(unsigned char) curFore]));
	if( curBack != KVI_TRANSPARENT ) {
		int theWdth = wdth;
		if( theWdth <= 0 )
			theWdth = width() - (curLeftCoord + KVI_IRC_VIEW_HORIZONTAL_BORDER + sb_width);
		pa->fillRect(
			curLeftCoord, curBottomCoord - m_iFontLineSpacing + m_iFontDescent, theWdth,
			m_iFontLineSpacing, *(g_pOptions->m_pMircColor[(unsigned char) curBack])
		);
	}
	pa->drawText(curLeftCoord, curBottomCoord, data, len);
	if( curBold ) // Draw doubled font (simulate bold)
		pa->drawText(curLeftCoord + 1, curBottomCoord, data, len);
	if( curUnderline ) { // Draw a line under the text block
		int theWdth = wdth;
		// If width is negative, it is the last block of a line
		if( theWdth <= 0 )
			theWdth = width() - (curLeftCoord + KVI_IRC_VIEW_HORIZONTAL_BORDER + sb_width);
		pa->drawLine(curLeftCoord, curBottomCoord + 1, curLeftCoord + theWdth, curBottomCoord + 1);
	}
	curLeftCoord += wdth;
}

inline void KviIrcView::drawSelectedText(
	QPainter *pa, int sb_width, char curAttribute, char curFore, char curBack,
	int &curLeftCoord, int curBottomCoord, const char *text_ptr, int len, int wdth)
{
	pa->setPen(*(g_pOptions->m_pMircColor[(unsigned char) g_pOptions->m_cViewOutSeleFore]));
	int theWdth = wdth;
	if( theWdth <= 0 )
		theWdth = width() - (curLeftCoord + KVI_IRC_VIEW_HORIZONTAL_BORDER + sb_width);
	pa->fillRect(
		curLeftCoord, curBottomCoord - m_iFontLineSpacing + m_iFontDescent, theWdth, m_iFontLineSpacing,
		*(g_pOptions->m_pMircColor[(unsigned char) g_pOptions->m_cViewOutSeleBack])
	);
	pa->drawText(curLeftCoord, curBottomCoord, text_ptr, len);
	m_szLastSelectionLine.append(text_ptr, len);
	curLeftCoord += wdth;
}

void KviIrcView::fastScroll(int lines)
{
	m_iUnprocessedPaintEventRequests = 0;
	if( !isVisible() ) return;

	// Ok... the current line is the last one here
	// It is the only one that needs to be repainted
	int widgetWidth = width();
	if( m_bNeedScrollBar )
		widgetWidth -= m_pScrollBar->width();
	if( widgetWidth < 20 )
		return; // Cannot show stuff here
	int widgetHeight = height();
	int maxLineWidth = widgetWidth;
	int defLeftCoord = KVI_IRC_VIEW_HORIZONTAL_BORDER;
	if( m_bShowImages ) {
		maxLineWidth -= KVI_IRC_VIEW_PIXMAP_SEPARATOR_AND_DOUBLEBORDER_WIDTH;
		defLeftCoord += KVI_IRC_VIEW_PIXMAP_AND_SEPARATOR;
	}
	int heightToPaint = 1;
	KviIrcViewTextLine *l = m_pCurLine;
	while( lines > 0 ) {
		if( l ) {
			if( maxLineWidth != l->max_line_width )
				calculateLineWraps(l, maxLineWidth);
			heightToPaint += l->line_wraps * m_iFontLineSpacing;
			heightToPaint += m_iFontLineSpacing + m_iFontDescent;
			lines--;
			l = l->prev_line;
		} else lines = 0;
	}

	bitBlt(
		this, 1, 1, this, 1, heightToPaint, widgetWidth - 2, widgetHeight - (heightToPaint + KVI_IRC_VIEW_VERTICAL_BORDER)
	);

	QRect r(0, widgetHeight - (heightToPaint + KVI_IRC_VIEW_VERTICAL_BORDER),
		widgetWidth, heightToPaint + KVI_IRC_VIEW_VERTICAL_BORDER
	);

	QPaintEvent *e = new QPaintEvent(r);
	paintEvent(e);
	delete e;

	if( m_iLastLinkRectHeight > -1 ) {
		// Need to remove the last highlighted link
		m_iLastLinkRectTop -= heightToPaint;
		if( m_iLastLinkRectTop < 0 ) {
			m_iLastLinkRectHeight += m_iLastLinkRectTop;
			m_iLastLinkRectTop     = 0;
		}
	}
}

/**
 *
 * THIS FUNCTION IS A MONSTER
 *
 */
void KviIrcView::paintEvent(QPaintEvent *p)
{
	int scrollbarWidth = 0;
	if( m_bNeedScrollBar )
		scrollbarWidth = m_pScrollBar->width();
	int widgetWidth = width() - scrollbarWidth;
	if( !isVisible() || (widgetWidth < 20) ) {
		m_iUnprocessedPaintEventRequests = 0; // Assume a full repaint when this widget is shown...
		return; // Cannot show stuff here
	}
	int widgetHeight = height();

	static QRect r;
	if( p )
		r = p->rect();
	else {
		// A self triggered event
		m_iUnprocessedPaintEventRequests = 0; // Only full repaints reset
		r = rect();
	}

	int rectLeft   = r.x();
	int rectTop    = r.y();
	int rectHeight = r.height();
	int rectWidth  = r.width();
	int rectBottom = rectTop + rectHeight;
	if( rectWidth > widgetWidth )
		rectWidth = widgetWidth;

	QPainter pa(g_pIrcViewMemBuffer);

	if( m_pPrivateBackgroundPixmap ) {
		pa.drawTiledPixmap(rectLeft, rectTop, rectWidth, rectHeight, *m_pPrivateBackgroundPixmap, 0, 0);
	} else {
		if( g_pOptions->m_pixViewBack->isNull() ) {
			pa.fillRect(rectLeft, rectTop, rectWidth, rectHeight, g_pOptions->m_clrViewBack);
		} else {
			pa.drawTiledPixmap(rectLeft, rectTop, rectWidth, rectHeight, *(g_pOptions->m_pixViewBack));
		}
	}
	pa.setFont(font());

	// Have lines visible
	int curBottomCoord = widgetHeight - KVI_IRC_VIEW_VERTICAL_BORDER;
	int maxLineWidth   = widgetWidth;
	int defLeftCoord   = KVI_IRC_VIEW_HORIZONTAL_BORDER;
	int lineWrapsHeight;

	if( m_bShowImages ) {
		maxLineWidth -= KVI_IRC_VIEW_PIXMAP_SEPARATOR_AND_DOUBLEBORDER_WIDTH;
		defLeftCoord += KVI_IRC_VIEW_PIXMAP_AND_SEPARATOR;
	}

	KviIrcViewTextLine *pCurTextLine = m_pCurLine;

	if( m_bSelecting ) {
		if( m_bDoubleClicked ) {
			m_szLastDoubleClickedEscape     = "";
			m_szLastDoubleClickedEscapeCmd  = "";
		} else {
			m_szLastSelectionLine           = "";
			m_szLastStrippedSelectionLine   = "";
			m_szLastSelection               = "";
			m_szLastStrippedSelection       = "";
			m_iLastSelectionLineLen         = 0;
			m_iLastStrippedSelectionLineLen = 0;
		}
	}

	// Make sure that we have enough space to paint something...
	if( maxLineWidth < m_iMinimumPaintWidth )
		pCurTextLine = 0;
	bool notOnLastLine = false;
	if( pCurTextLine )
		notOnLastLine = (pCurTextLine->next_line);

	// And loop through lines until we do not run over the upper bound of the view
	while( (curBottomCoord >= KVI_IRC_VIEW_VERTICAL_BORDER) && pCurTextLine ) {
		// Paint pCurTextLine
		if( maxLineWidth != pCurTextLine->max_line_width ) {
			// Width of the widget or the font has been changed
			// from the last time that this line was painted
			calculateLineWraps(pCurTextLine, maxLineWidth);
		}

		// The evil multiplication.
		// On a 486 it can take up to 42 clock cycles
		lineWrapsHeight = (pCurTextLine->line_wraps) * m_iFontLineSpacing;
		curBottomCoord -= lineWrapsHeight;

		if( (curBottomCoord - m_iFontLineSpacing) > rectBottom ) {
			// Not in update rect... skip it
			curBottomCoord -= m_iFontLineSpacing + m_iFontDescent;
			pCurTextLine    = pCurTextLine->prev_line;
			continue;
		}

		if( m_bShowImages ) {
			// Paint the pixmap first
			// Calculate the position of the image
			int imageYPos = curBottomCoord - m_iRelativePixmapY;
			// Set the mask if needed
			pa.drawPixmap(
				KVI_IRC_VIEW_HORIZONTAL_BORDER,
				imageYPos,
				*(g_pixViewOut[pCurTextLine->msg_type])
			);
		}

		// Initialize for drawing this line of text
		char defaultBack      = pCurTextLine->text_blocks_ptr->attr_ptr->attr_back;
		char defaultFore      = pCurTextLine->text_blocks_ptr->attr_ptr->attr_fore;
		bool curBold          = false;
		bool curUnderline     = false;
		char foreBeforeEscape = KVI_BLACK;
		bool curLink          = false;
		char curFore          = defaultFore;
		char curBack          = defaultBack;
		int  curLeftCoord     = defLeftCoord;
		curBottomCoord       -= m_iFontDescent; // Raise the text...

		//
		// Single text line loop; paint all text blocks.
		// May correspond to more physical lines on the display if the text is wrapped
		//
		for( int i = 0; i < pCurTextLine->num_text_blocks; i++ ) {
			register KviIrcViewTextBlock *block = &(pCurTextLine->text_blocks_ptr[i]);
			// Play with the attributes
			if( block->attr_ptr ) {
				// Normal block
				switch( block->attr_ptr->attribute ) {
					case KVI_TEXT_COLOR:
						if( block->attr_ptr->attr_fore != KVI_NOCHANGE ) {
							curFore = block->attr_ptr->attr_fore;
							if( block->attr_ptr->attr_back != KVI_NOCHANGE )
								curBack = block->attr_ptr->attr_back;
						} else {
							// Only a CTRL+K; reset
							curFore = defaultFore;
							curBack = defaultBack;
						}
						break;
					case KVI_TEXT_ESCAPE:
						foreBeforeEscape = curFore;
						if( block->attr_ptr->attr_fore != KVI_NOCHANGE )
							curFore = block->attr_ptr->attr_fore;
						if( m_pLastLinkUnderMouse == block )
							curLink = true;
						break;
					case KVI_TEXT_UNESCAPE:
						curLink = false;
						curFore = foreBeforeEscape;
						break;
					case KVI_TEXT_BOLD:
						curBold = !curBold;
						break;
					case KVI_TEXT_UNDERLINE:
						curUnderline = !curUnderline;
						break;
					case KVI_TEXT_RESET:
						curBold      = false;
						curUnderline = false;
						curFore      = defaultFore;
						curBack      = defaultBack;
						break;
					case KVI_TEXT_REVERSE:
						if( curBack != KVI_TRANSPARENT ) {
							char aux = curFore;
							curFore  = curBack;
							curBack  = aux;
						} else {
							curBack = curFore;
							switch( curBack ) {
								case KVI_WHITE:       // Fall-through
								case KVI_ORANGE:      // Fall-through
								case KVI_YELLOW:      // Fall-through
								case KVI_LIGHTGREEN:  // Fall-through
								case KVI_BLUEMARINE:  // Fall-through
								case KVI_LIGHTBLUE:   // Fall-through
								case KVI_LIGHTVIOLET: // Fall-through
								case KVI_LIGHTGRAY:
									curFore = KVI_BLACK;
									break;
								default: // Transparent too
									curFore = KVI_LIGHTGREEN;
									break;
							}
						}
						break;
				}
			} else {
				// No attributes; it is a line wrap
				curLeftCoord = defLeftCoord;
				if( g_pOptions->m_bWrapMargin )
					curLeftCoord += m_iWrapMargin;
				curBottomCoord += m_iFontLineSpacing;
			}
			if( curLink )
				checkForDoubleClickedUrlOrEscape(pCurTextLine, curLeftCoord, curBottomCoord, i);
			if( m_bSelecting ) {
				// Check if the block or a part of it is selected
				if( checkSelectionBlock(pCurTextLine, curLeftCoord, curBottomCoord, i) ) {
					// Selected in some way
					char curAttribute = 0;
					char curAttrFore  = KVI_NOCHANGE;
					char curAttrBack  = KVI_NOCHANGE;
					if( (block->attr_ptr) && (i > 0) ) {
						curAttribute = block->attr_ptr->attribute;
						curAttrFore  = block->attr_ptr->attr_fore;
						curAttrBack  = block->attr_ptr->attr_back;
					}
					switch( m_TextBlockSelectionInfo.selection_type ) {
						case KVI_IRC_VIEW_BLOCK_SELECTION_TOTAL:
							drawSelectedText(
								&pa, scrollbarWidth, curAttribute, curAttrFore, curAttrBack,
								curLeftCoord, curBottomCoord, block->block_ptr, block->block_len, block->block_width
							);
							break;
						case KVI_IRC_VIEW_BLOCK_SELECTION_LEFT:
							drawSelectedText(
								&pa, scrollbarWidth, curAttribute, curAttrFore, curAttrBack,
								curLeftCoord, curBottomCoord, block->block_ptr,
								m_TextBlockSelectionInfo.part_1_length, m_TextBlockSelectionInfo.part_1_width
							);
							drawBlockPart(
								&pa, scrollbarWidth, curFore, curBack,
								curBold, curUnderline, curLeftCoord, curBottomCoord,
								block->block_ptr + m_TextBlockSelectionInfo.part_1_length,
								m_TextBlockSelectionInfo.part_2_length, m_TextBlockSelectionInfo.part_2_width
							);
							break;
						case KVI_IRC_VIEW_BLOCK_SELECTION_RIGHT:
							drawBlockPart(
								&pa, scrollbarWidth, curFore, curBack,
								curBold, curUnderline, curLeftCoord, curBottomCoord,
								block->block_ptr,
								m_TextBlockSelectionInfo.part_1_length, m_TextBlockSelectionInfo.part_1_width
							);
							drawSelectedText(
								&pa, scrollbarWidth, curAttribute, curAttrFore, curAttrBack,
								curLeftCoord, curBottomCoord,
								block->block_ptr + m_TextBlockSelectionInfo.part_1_length,
								m_TextBlockSelectionInfo.part_2_length, m_TextBlockSelectionInfo.part_2_width
							);
							break;
						case KVI_IRC_VIEW_BLOCK_SELECTION_CENTRAL:
							drawBlockPart(
								&pa, scrollbarWidth, curFore, curBack,
								curBold, curUnderline, curLeftCoord, curBottomCoord,
								block->block_ptr,
								m_TextBlockSelectionInfo.part_1_length, m_TextBlockSelectionInfo.part_1_width
							);
							drawSelectedText(
								&pa, scrollbarWidth, curAttribute, curAttrFore, curAttrBack,
								curLeftCoord, curBottomCoord,
								block->block_ptr + m_TextBlockSelectionInfo.part_1_length,
								m_TextBlockSelectionInfo.part_2_length, m_TextBlockSelectionInfo.part_2_width
							);
							drawBlockPart(
								&pa, scrollbarWidth, curFore, curBack,
								curBold, curUnderline, curLeftCoord, curBottomCoord,
								block->block_ptr + m_TextBlockSelectionInfo.part_1_length + m_TextBlockSelectionInfo.part_2_length,
								m_TextBlockSelectionInfo.part_3_length, m_TextBlockSelectionInfo.part_3_width
							);
							break;
					} // end of switch( m_TextBlockSelectionInfo.selection_type )
				} else {
					int wdth = block->block_width;
					// Last block before a word wrap, or a zero characters attribute block?
					if( wdth == 0 ) {
						// Last block before a word wrap, or a zero characters attribute block?
						if( (i < (pCurTextLine->num_text_blocks - 1)) ) {
							// There is another block; check if it is a wrap
							if( !(pCurTextLine->text_blocks_ptr[i + 1].attr_ptr) ) {
								wdth = widgetWidth - (curLeftCoord + KVI_IRC_VIEW_HORIZONTAL_BORDER);
							} // else simply a zero characters block
						} // else: simply a zero characters block
					}
					drawBlockPart(
						&pa, scrollbarWidth, curFore, curBack,
						curBold, curUnderline, curLeftCoord, curBottomCoord,
						block->block_ptr, block->block_len, wdth
					);
				}
			} else {
				// No selection... this is fast!
				int wdth = block->block_width;
				if( wdth <= 0 )
					wdth = widgetWidth - (curLeftCoord + KVI_IRC_VIEW_HORIZONTAL_BORDER);
				pa.setPen(*(g_pOptions->m_pMircColor[(unsigned char) curFore]));

				if( curBack != KVI_TRANSPARENT ) {
					pa.fillRect(
						curLeftCoord, curBottomCoord - m_iFontLineSpacing + m_iFontDescent, wdth,
						m_iFontLineSpacing, *(g_pOptions->m_pMircColor[(unsigned char) curBack])
					);
				}
				pa.drawText(curLeftCoord, curBottomCoord, block->block_ptr, block->block_len);

				if( curBold ) { // Draw doubled font (simulate bold)
					pa.drawText(curLeftCoord + 1, curBottomCoord, block->block_ptr, block->block_len);
				}
				if( curLink ) {
					pa.setPen(*(g_pOptions->m_pMircColor[(unsigned char) g_pOptions->m_cViewOutUrlFore]));
					pa.drawText(curLeftCoord - 1, curBottomCoord, block->block_ptr, block->block_len);
				}
				if( curUnderline ) { // Draw a line under the text block
					pa.drawLine(curLeftCoord, curBottomCoord + 1, curLeftCoord + wdth, curBottomCoord + 1);
				}
				curLeftCoord += block->block_width;
			}
		}

		if( m_bSelecting ) {
			if( m_szLastSelectionLine.hasData() ) {
				if( m_szLastSelection.hasData() )
					m_szLastSelection.prepend("\n", 1);
				m_szLastSelection.prepend(m_szLastSelectionLine);
				m_szLastSelectionLine.setLen(0);
			}
		}

		curBottomCoord -= lineWrapsHeight + m_iFontLineSpacing;
		pCurTextLine    = pCurTextLine->prev_line;
	}
	// Need to draw the sunken rect around the view now...
	pa.setPen(colorGroup().dark());
	pa.drawLine(0, 0, widgetWidth, 0);
	pa.drawLine(0, 0, 0, widgetHeight);
	pa.setPen(colorGroup().light());
	widgetWidth--;
	pa.drawLine(1, widgetHeight - 1, widgetWidth, widgetHeight - 1);
	pa.drawLine(widgetWidth, 1, widgetWidth, widgetHeight);
	pa.end();

	bool redraw = false;
	if( pCurTextLine || notOnLastLine )
		redraw = setNeedScrollBar(true);
	else
		redraw = setNeedScrollBar(false);

	if( redraw )
		update();
	else
		bitBlt(this, rectLeft, rectTop, g_pIrcViewMemBuffer, rectLeft, rectTop, rectWidth, rectHeight, Qt::CopyROP);
}

/**
 * ============ calculateLineWraps ==============
 */
void KviIrcView::calculateLineWraps(KviIrcViewTextLine *ptr, int maxWidth)
{
	if( ptr->num_text_blocks != 0 )
		kvi_free(ptr->text_blocks_ptr); // Free any previous wrap blocks
	ptr->text_blocks_ptr = (KviIrcViewTextBlock *) kvi_malloc(sizeof(KviIrcViewTextBlock)); // Alloc one block
	ptr->max_line_width  = maxWidth; // Calculus for this width
	ptr->num_text_blocks = 0;        // It will be ++
	ptr->line_wraps      = 0;        // No line wraps yet
	int curAttrBlock     = 0;        // Current attribute block
	int curLineWidth     = 0;

	// Init the first block
	ptr->text_blocks_ptr->block_ptr   = ptr->data_ptr;
	ptr->text_blocks_ptr->block_len   = 0;
	ptr->text_blocks_ptr->block_width = 0;
	ptr->text_blocks_ptr->attr_ptr    = &(ptr->attr_ptr[0]);

	int maxBlockLen = ptr->attr_ptr->block_len; // Equal to ptr->attr_ptr[0].block_len

	for( ;; ) {
		// Calculate the block_width
		register char *p = ptr->text_blocks_ptr[ptr->num_text_blocks].block_ptr;
		int curBlockLen   = 0;
		int curBlockWidth = 0;
		while( curBlockLen < maxBlockLen ) {
			curBlockWidth += m_iFontCharacterWidth[((unsigned char) *p)];
			curBlockLen++;
			p++;
		}
		// Check the length
		curLineWidth += curBlockWidth;
		if( curLineWidth < maxWidth ) {
			// OK, proceed to next block
			ptr->text_blocks_ptr[ptr->num_text_blocks].block_len   = curBlockLen;
			ptr->text_blocks_ptr[ptr->num_text_blocks].block_width = curBlockWidth;
			curAttrBlock++;
			ptr->num_text_blocks++;
			ptr->text_blocks_ptr = (KviIrcViewTextBlock *) kvi_realloc(
				ptr->text_blocks_ptr, (ptr->num_text_blocks + 1) * sizeof(KviIrcViewTextBlock)
			);
			// Process the next block of data in the next loop or return if have no more blocks
			if( curAttrBlock < ptr->attr_len ) {
				ptr->text_blocks_ptr[ptr->num_text_blocks].block_ptr   = &(ptr->data_ptr[ptr->attr_ptr[curAttrBlock].block_idx]);
				ptr->text_blocks_ptr[ptr->num_text_blocks].block_len   = 0;
				ptr->text_blocks_ptr[ptr->num_text_blocks].block_width = 0;
				ptr->text_blocks_ptr[ptr->num_text_blocks].attr_ptr    = &(ptr->attr_ptr[curAttrBlock]);
				maxBlockLen = ptr->text_blocks_ptr[ptr->num_text_blocks].attr_ptr->block_len;
			} else return;
		} else {
			// Need word wrap.
			// First go back to an admissible width
			while( (curLineWidth >= maxWidth) && curBlockLen ) {
				p--;
				curBlockLen--;
				curLineWidth -= m_iFontCharacterWidth[((unsigned char) *p)];
			}
			// Now look for a space
			while( (*p != ' ') && curBlockLen ) {
				p--;
				curBlockLen--;
				curLineWidth -= m_iFontCharacterWidth[((unsigned char) *p)];
			}
			// If we ran up to the beginning of the block.
			if( curBlockLen == 0 ) {
				// Do not like it. Forced wrap here.
				// Go ahead up to the biggest possible string
				do {
					curBlockLen++;
					p++;
					curLineWidth += m_iFontCharacterWidth[((unsigned char) *p)];
				}while( (curLineWidth < maxWidth) && (curBlockLen < maxBlockLen) );
				// Now overrun, go back one char
				p--;
				curBlockLen--;
				// OK... wrap
				ptr->text_blocks_ptr[ptr->num_text_blocks].block_len = curBlockLen;
				maxBlockLen -= curBlockLen;
				ptr->num_text_blocks++;
				ptr->text_blocks_ptr = (KviIrcViewTextBlock *) kvi_realloc(
					ptr->text_blocks_ptr, (ptr->num_text_blocks + 1) * sizeof(KviIrcViewTextBlock)
				);

				ptr->text_blocks_ptr[ptr->num_text_blocks].block_ptr   = p;
				ptr->text_blocks_ptr[ptr->num_text_blocks].block_len   = 0;
				ptr->text_blocks_ptr[ptr->num_text_blocks].block_width = 0;
				ptr->text_blocks_ptr[ptr->num_text_blocks].attr_ptr    = 0;
			} else {
				// Found a space...
				// Include it in the first block
				p++;
				curBlockLen++;
				// Block width is not important; we wrap significant block width
				ptr->text_blocks_ptr[ptr->num_text_blocks].block_len = curBlockLen;
				maxBlockLen -= curBlockLen;
				ptr->num_text_blocks++;
				ptr->text_blocks_ptr = (KviIrcViewTextBlock *) kvi_realloc(
					ptr->text_blocks_ptr, (ptr->num_text_blocks + 1) * sizeof(KviIrcViewTextBlock)
				);
				ptr->text_blocks_ptr[ptr->num_text_blocks].block_ptr   = p;
				ptr->text_blocks_ptr[ptr->num_text_blocks].block_len   = 0;
				ptr->text_blocks_ptr[ptr->num_text_blocks].block_width = 0;
				ptr->text_blocks_ptr[ptr->num_text_blocks].attr_ptr    = 0;
			}
			curLineWidth = 0;
			ptr->line_wraps++;
			if( ptr->line_wraps == 1 ) {
				if( g_pOptions->m_bWrapMargin )
					maxWidth -= m_iWrapMargin;
			}
		}
	}
	ptr->num_text_blocks++;
}

/**
 * ================= calculateSelectionBounds ==================
 */
void KviIrcView::calculateSelectionBounds()
{
	m_iSelectionTop    = ((m_iMouseButtonPressY < m_iMouseButtonCurrentY) ? m_iMouseButtonPressY : m_iMouseButtonCurrentY);
	m_iSelectionBottom = ((m_iMouseButtonPressY > m_iMouseButtonCurrentY) ? m_iMouseButtonPressY : m_iMouseButtonCurrentY);
	if( m_iMouseButtonPressY < m_iMouseButtonCurrentY ) {
		m_iSelectionBegin = m_iMouseButtonPressX;
		m_iSelectionEnd   = m_iMouseButtonCurrentX;
	} else {
		m_iSelectionBegin = m_iMouseButtonCurrentX;
		m_iSelectionEnd   = m_iMouseButtonPressX;
	}
	m_iSelectionLeft  = (m_iSelectionBegin < m_iSelectionEnd) ? m_iSelectionBegin : m_iSelectionEnd;
	m_iSelectionRight = (m_iSelectionBegin > m_iSelectionEnd) ? m_iSelectionBegin : m_iSelectionEnd;
}

/**
 * ============== checkForDoubleClickedUrlOrEscape =================
 */
void KviIrcView::checkForDoubleClickedUrlOrEscape(KviIrcViewTextLine *line, int left, int bottom, int bufIndex)
{
	__range_valid(bufIndex >= 0);
	int top   = bottom - m_iFontLineSpacing;
	int right = (line->text_blocks_ptr[bufIndex].block_width
		? left + line->text_blocks_ptr[bufIndex].block_width
		: width() - KVI_IRC_VIEW_SCROLLBAR_AND_HORIZONTAL_BORDER_WIDTH
	);
	if( (left < m_iMouseButtonPressX) && (right > m_iMouseButtonPressX ) &&
	    (bottom > m_iMouseButtonPressY) && (top < m_iMouseButtonPressY)
	) {
		// Yeah man! This block clicked... check if it is a part of a URL
		if( !(line->text_blocks_ptr[bufIndex].attr_ptr) ) {
			// Could be a word wrapped URL... go back until we find a block with attributes
			while( !(line->text_blocks_ptr[bufIndex].attr_ptr) && (bufIndex > 0) )
				bufIndex--;
		}
		if( bufIndex == 0 )
			return; // Cannot be... the first is always a color block
		if( line->text_blocks_ptr[bufIndex].attr_ptr ) {
			if( line->text_blocks_ptr[bufIndex].attr_ptr->attribute == KVI_TEXT_ESCAPE )
			{
				m_szLastDoubleClickedEscape = KviStr(
					line->text_blocks_ptr[bufIndex].block_ptr, line->text_blocks_ptr[bufIndex].block_len
				);
				m_szLastDoubleClickedEscapeCmd = line->text_blocks_ptr[bufIndex].attr_ptr->escape_cmd;
			} else
				return;
			// Continue while we do not find a non-wordwrap block
			for( ;; ) {
				bufIndex++;
				if( bufIndex == line->num_text_blocks )
					return;
				if( line->text_blocks_ptr[bufIndex].attr_ptr )
					return; // Finished; not a word wrap
				KviStr szBlockData(line->text_blocks_ptr[bufIndex].block_ptr, line->text_blocks_ptr[bufIndex].block_len);
				m_szLastDoubleClickedEscape.append(szBlockData.ptr());
			}
		}
	}
}

/**
 * =============== checkSelectionBlock ===============
 */
bool KviIrcView::checkSelectionBlock(KviIrcViewTextLine *line, int left, int bottom, int bufIndex)
{
	register char *p = line->text_blocks_ptr[bufIndex].block_ptr;
	int top   = bottom - m_iFontLineSpacing;
	int right = (line->text_blocks_ptr[bufIndex].block_width
		? left + line->text_blocks_ptr[bufIndex].block_width
		: width() - KVI_IRC_VIEW_SCROLLBAR_AND_HORIZONTAL_BORDER_WIDTH
	);
	if( bottom < m_iSelectionTop    ) return false; // The selection starts under this line
	if( top    > m_iSelectionBottom ) return false; // The selection ends over this line
	if( (top   >= m_iSelectionTop) && (bottom < m_iSelectionBottom) ) {
		// Whole line selected
		m_TextBlockSelectionInfo.selection_type = KVI_IRC_VIEW_BLOCK_SELECTION_TOTAL;
		return true;
	}
	if( (top < m_iSelectionTop) && (bottom >= m_iSelectionBottom) ) {
		// Selection begins and ends in this line
		if( right < m_iSelectionLeft  ) return false;
		if( left  > m_iSelectionRight ) return false;
		if( (right <= m_iSelectionRight) && (left > m_iSelectionLeft) ) {
			// Whole line selected
			m_TextBlockSelectionInfo.selection_type = KVI_IRC_VIEW_BLOCK_SELECTION_TOTAL;
			return true;
		}
		if( (right > m_iSelectionRight) && (left <= m_iSelectionLeft) ) {
			// Selection ends and begins in THIS BLOCK!
			m_TextBlockSelectionInfo.selection_type = KVI_IRC_VIEW_BLOCK_SELECTION_CENTRAL;
			m_TextBlockSelectionInfo.part_1_length  = 0;
			m_TextBlockSelectionInfo.part_1_width   = 0;
			while( (left <= m_iSelectionLeft) &&
			       (m_TextBlockSelectionInfo.part_1_length < line->text_blocks_ptr[bufIndex].block_len)
			) {
				left += m_iFontCharacterWidth[(unsigned char) *p];
				m_TextBlockSelectionInfo.part_1_width += m_iFontCharacterWidth[(unsigned char) *p];
				p++;
				m_TextBlockSelectionInfo.part_1_length++;
			}
			// Need to include the first character
			if( m_TextBlockSelectionInfo.part_1_length > 0 ) {
				m_TextBlockSelectionInfo.part_1_length--;
				p--;
				left -= m_iFontCharacterWidth[(unsigned char) *p];
				m_TextBlockSelectionInfo.part_1_width -= m_iFontCharacterWidth[(unsigned char) *p];
			}
			int maxLenNow   = line->text_blocks_ptr[bufIndex].block_len   - m_TextBlockSelectionInfo.part_1_length;
			int maxWidthNow = line->text_blocks_ptr[bufIndex].block_width - m_TextBlockSelectionInfo.part_1_width;
			m_TextBlockSelectionInfo.part_2_length = 0;
			m_TextBlockSelectionInfo.part_2_width  = 0;
			while( (left < m_iSelectionRight) && (m_TextBlockSelectionInfo.part_2_length < maxLenNow) ) {
				left += m_iFontCharacterWidth[(unsigned char) *p];
				m_TextBlockSelectionInfo.part_2_width += m_iFontCharacterWidth[(unsigned char) *p];
				p++;
				m_TextBlockSelectionInfo.part_2_length++;
			}
			m_TextBlockSelectionInfo.part_3_length = maxLenNow   - m_TextBlockSelectionInfo.part_2_length;
			m_TextBlockSelectionInfo.part_3_width  = maxWidthNow - m_TextBlockSelectionInfo.part_2_width;
			return true;
		}
		if( right > m_iSelectionRight ) {
			// Selection ends in THIS BLOCK!
			m_TextBlockSelectionInfo.selection_type = KVI_IRC_VIEW_BLOCK_SELECTION_LEFT;
			m_TextBlockSelectionInfo.part_1_length  = 0;
			m_TextBlockSelectionInfo.part_1_width   = 0;
			while( (left < m_iSelectionRight) &&
			       (m_TextBlockSelectionInfo.part_1_length < line->text_blocks_ptr[bufIndex].block_len)
			) {
				left += m_iFontCharacterWidth[(unsigned char) *p];
				m_TextBlockSelectionInfo.part_1_width += m_iFontCharacterWidth[(unsigned char) *p];
				p++;
				m_TextBlockSelectionInfo.part_1_length++;
			}
			m_TextBlockSelectionInfo.part_2_length = line->text_blocks_ptr[bufIndex].block_len   - m_TextBlockSelectionInfo.part_1_length;
			m_TextBlockSelectionInfo.part_2_width  = line->text_blocks_ptr[bufIndex].block_width - m_TextBlockSelectionInfo.part_1_width;
			return true;
		}
		// Selection begins in THIS BLOCK!
		m_TextBlockSelectionInfo.selection_type = KVI_IRC_VIEW_BLOCK_SELECTION_RIGHT;
		m_TextBlockSelectionInfo.part_1_length  = 0;
		m_TextBlockSelectionInfo.part_1_width   = 0;
		while( (left <= m_iSelectionLeft) &&
		       (m_TextBlockSelectionInfo.part_1_length < line->text_blocks_ptr[bufIndex].block_len)
		) {
			left += m_iFontCharacterWidth[(unsigned char) *p];
			m_TextBlockSelectionInfo.part_1_width += m_iFontCharacterWidth[(unsigned char) *p];
			p++;
			m_TextBlockSelectionInfo.part_1_length++;
		}
		// Need to include the first character
		if( m_TextBlockSelectionInfo.part_1_length > 0 ) {
			m_TextBlockSelectionInfo.part_1_length--;
			p--;
			left -= m_iFontCharacterWidth[(unsigned char) *p];
			m_TextBlockSelectionInfo.part_1_width -= m_iFontCharacterWidth[(unsigned char) *p];
		}
		m_TextBlockSelectionInfo.part_2_length = line->text_blocks_ptr[bufIndex].block_len   - m_TextBlockSelectionInfo.part_1_length;
		m_TextBlockSelectionInfo.part_2_width  = line->text_blocks_ptr[bufIndex].block_width - m_TextBlockSelectionInfo.part_1_width;
		return true;
	}

	if( top < m_iSelectionTop ) {
		// Selection starts in this line
		if( right < m_iSelectionBegin )
			return false;
		if( left > m_iSelectionBegin ) {
			// Whole block selected
			m_TextBlockSelectionInfo.selection_type = KVI_IRC_VIEW_BLOCK_SELECTION_TOTAL;
			return true;
		}
		// Selection begins in THIS BLOCK!
		m_TextBlockSelectionInfo.selection_type = KVI_IRC_VIEW_BLOCK_SELECTION_RIGHT;
		m_TextBlockSelectionInfo.part_1_length  = 0;
		m_TextBlockSelectionInfo.part_1_width   = 0;
		while( (left <= m_iSelectionBegin) &&
		       (m_TextBlockSelectionInfo.part_1_length < line->text_blocks_ptr[bufIndex].block_len)
		) {
			left += m_iFontCharacterWidth[(unsigned char) *p];
			m_TextBlockSelectionInfo.part_1_width += m_iFontCharacterWidth[(unsigned char) *p];
			p++;
			m_TextBlockSelectionInfo.part_1_length++;
		}
		// Need to include the first character
		if( m_TextBlockSelectionInfo.part_1_length > 0 ) {
			m_TextBlockSelectionInfo.part_1_length--;
			p--;
			left -= m_iFontCharacterWidth[(unsigned char) *p];
			m_TextBlockSelectionInfo.part_1_width -= m_iFontCharacterWidth[(unsigned char) *p];
		}
		m_TextBlockSelectionInfo.part_2_length = line->text_blocks_ptr[bufIndex].block_len   - m_TextBlockSelectionInfo.part_1_length;
		m_TextBlockSelectionInfo.part_2_width  = line->text_blocks_ptr[bufIndex].block_width - m_TextBlockSelectionInfo.part_1_width;
		return true;
	}
	// Selection ends in this line
	if( left  > m_iSelectionEnd )
		return false;
	if( right < m_iSelectionEnd ) {
		// Whole block selected
		m_TextBlockSelectionInfo.selection_type = KVI_IRC_VIEW_BLOCK_SELECTION_TOTAL;
		return true;
	}
	// Selection ends in THIS BLOCK!
	m_TextBlockSelectionInfo.selection_type = KVI_IRC_VIEW_BLOCK_SELECTION_LEFT;
	m_TextBlockSelectionInfo.part_1_length  = 0;
	m_TextBlockSelectionInfo.part_1_width   = 0;
	while( (left < m_iSelectionEnd) &&
	       (m_TextBlockSelectionInfo.part_1_length < line->text_blocks_ptr[bufIndex].block_len)
	) {
		left += m_iFontCharacterWidth[(unsigned char) *p];
		m_TextBlockSelectionInfo.part_1_width += m_iFontCharacterWidth[(unsigned char) *p];
		p++;
		m_TextBlockSelectionInfo.part_1_length++;
	}
	m_TextBlockSelectionInfo.part_2_length = line->text_blocks_ptr[bufIndex].block_len   - m_TextBlockSelectionInfo.part_1_length;
	m_TextBlockSelectionInfo.part_2_width  = line->text_blocks_ptr[bufIndex].block_width - m_TextBlockSelectionInfo.part_1_width;
	return true;
}

/**
 * ============ recalcFontVariables ==============
 */
void KviIrcView::recalcFontVariables(const QFont &fnt)
{
	QFontMetrics fm(fnt);
	m_iFontLineSpacing = fm.lineSpacing();
	if( m_iFontLineSpacing < 16 )
		m_iFontLineSpacing = 16;
	m_iFontDescent   = fm.descent();
	m_iFontLineWidth = fm.lineWidth();
	if( m_iFontLineWidth == 0 )
		m_iFontLineWidth = 1;
	m_iWrapMargin = fm.width("wwww");
	for( int i = 0; i < 256; i++ )
		m_iFontCharacterWidth[i] = fm.width((char)i);
	m_iMinimumPaintWidth = (fm.width('w') << 1) + m_iWrapMargin;
	m_iRelativePixmapY = (m_iFontLineSpacing + KVI_IRC_VIEW_PIXMAP_SIZE) >> 1;
}

/**
 * ================ resizeMemBuffer ===============
 */
void KviIrcView::resizeMemBuffer()
{
	// Check if we can make the mem buffer a bit smaller to save some memory
	int maxw = 16;
	int maxh = 16;
	int sbw  = 0;
	if( m_bNeedScrollBar )
		sbw = m_pScrollBar->width();
	for( KviIrcView *i = g_pIrcViewWidgetList->first(); i; i = g_pIrcViewWidgetList->next() ) {
		if( maxw < (i->width() - sbw) )
			maxw = i->width() - sbw;
		if( maxh < i->height() )
			maxh = i->height();
	}
	if( (maxw != g_pIrcViewMemBuffer->width()) || (maxh != g_pIrcViewMemBuffer->height()) ) {
		g_pIrcViewMemBuffer->resize(maxw, maxh);
		g_hIrcViewMemBuffer = g_pIrcViewMemBuffer->handle(); // Qt may change it while resizing
	}
	// OK. Mem buffer is big enough for all the IRC views alive
}

void KviIrcView::resizeEvent(QResizeEvent *)
{
	m_pScrollBar->setGeometry(width() - KVI_IRC_VIEW_SCROLLBAR_WIDTH, 0, KVI_IRC_VIEW_SCROLLBAR_WIDTH, height());
	resizeMemBuffer();
}

/**
 * ================ mousePressEvent ================
 */
void KviIrcView::mousePressEvent(QMouseEvent *e)
{
	m_iMouseButtonPressX   = e->pos().x();
	m_iMouseButtonPressY   = e->pos().y();
	m_iMouseButtonCurrentX = m_iMouseButtonPressX;
	m_iMouseButtonCurrentY = m_iMouseButtonPressY;
	if( e->button() & LeftButton ) {
		m_bSelecting = true;
		if( m_pKviWindow && (m_pKviWindow->type() == KVI_WND_TYPE_CHANNEL) ) {
			m_bDoubleClicked = true;
			paintEvent(0);
			m_bDoubleClicked = false;
			if( m_szLastDoubleClickedEscape.hasData() &&
			    m_pKviWindow->m_pListBox->findUser(m_szLastDoubleClickedEscape.ptr())
			) {
				m_pKviWindow->m_pListBox->deselectAll();
				m_pKviWindow->m_pListBox->select(m_szLastDoubleClickedEscape.ptr());
			}
		}
	} else if( e->button() & RightButton ) {
		m_bSelecting     = true;
		m_bDoubleClicked = true;
		paintEvent(0);
		m_bDoubleClicked = false;
		m_bSelecting     = false;
		KviStr tmp;
		getRightClickCommand(tmp, m_szLastDoubleClickedEscapeCmd.ptr());
		if( tmp.isEmpty() ) {
			// No link under mouse
			if( e->state() & ControlButton )
				m_pFrm->windowPopupRequested(this, m_pKviWindow);
			else
				emit contextPopupRequested(this);
		} else {
			if( m_pFrm && m_pKviWindow )
				m_pFrm->m_pUserParser->parseCommand(tmp.ptr(), m_pKviWindow, m_szLastDoubleClickedEscape.ptr());
		}
	} else if( e->button() & MidButton )
		m_pFrm->windowPopupRequested(this, m_pKviWindow);
}

/**
 * ================ mouseReleaseEvent ===============
 */
void KviIrcView::mouseReleaseEvent(QMouseEvent *e)
{
	if( m_iSelectTimer ) {
		killTimer(m_iSelectTimer);
		m_iSelectTimer = 0;
		if( e->state() & ShiftButton )
			m_pFrm->ircViewTextSelected(this, m_pKviWindow, m_szLastStrippedSelection.ptr());
		else
			m_pFrm->ircViewTextSelected(this, m_pKviWindow, m_szLastSelection.ptr());
	}
	if( m_bSelecting ) {
		m_bSelecting = false;
		// Insert the lines blocked while selecting
		KviIrcViewTextLine *l;
		while( (l = m_pMessagesStoppedWhileSelecting->first()) ) {
			m_pMessagesStoppedWhileSelecting->removeFirst();
			appendLine(l, false);
		}
		paintEvent(0);
	}
}

void KviIrcView::findNext(const char *text)
{
	KviIrcViewTextLine *l = m_pCurLine;
	int curScr = m_pScrollBar->value();
	if( l ) {
		curScr++;
		l = l->next_line;
		if( !l ) {
			l = m_pFirstLine;
			curScr = 1;
		}
		KviIrcViewTextLine *start = l;

		do {
			__range_valid(l->data_ptr);
			KviStr tmp = l->data_ptr;
			int idx = tmp.findFirstIdx(text, false);
			if( idx != -1 ) {
				m_pScrollBar->setValue(curScr); // Found
				return;
			}
			l = l->next_line;
			curScr++;
			if( !l ) {
				curScr = 1;
				l      = m_pFirstLine;
			}
		} while( l != start );
	}
	KviMessageBox::sorry(
		_CHAR_2_QSTRING(_i18n_("Text not found")),
		_CHAR_2_QSTRING(_i18n_("Find")),
		this
	);
}

void KviIrcView::findPrev(const char *text)
{
	KviIrcViewTextLine *l = m_pCurLine;
	int curScr = m_pScrollBar->value();
	if( l ) {
		curScr--;
		l = l->prev_line;
		if( !l ) {
			curScr = m_iNumLines;
			l      = m_pLastLine;
		}
		KviIrcViewTextLine *start = l;

		do {
			__range_valid(l->data_ptr);
			KviStr tmp = l->data_ptr;
			int idx = tmp.findFirstIdx(text, false);
			if( idx != -1 ) {
				m_pScrollBar->setValue(curScr); // Found
				return;
			}
			l = l->prev_line;
			curScr--;
			if( !l ) {
				curScr = m_iNumLines;
				l      = m_pLastLine;
			}
		} while( l != start );
	}
	KviMessageBox::sorry(
		_CHAR_2_QSTRING(_i18n_("Text not found")),
		_CHAR_2_QSTRING(_i18n_("Find")),
		this
	);
}

KviWindow *KviIrcView::parentKviWindow()
{
	return m_pKviWindow;
}

KviIrcViewTextBlock *KviIrcView::getLinkUnderMouse(int xPos, int yPos, int *rectTop, int *rectHeight)
{
	KviIrcViewTextLine *l = m_pCurLine;
	int iTop = height() - KVI_IRC_VIEW_VERTICAL_BORDER;

	while( iTop > yPos ) {
		if( l ) {
			iTop -= ((l->line_wraps + 1) * m_iFontLineSpacing) + m_iFontDescent;
			if( iTop <= yPos ) {
				// Got the right KviIrcViewTextLine
				int iLeft = KVI_IRC_VIEW_HORIZONTAL_BORDER;
				if( m_bShowImages )
					iLeft += KVI_IRC_VIEW_PIXMAP_AND_SEPARATOR;
				int firstRowTop = iTop;
				int i = 0;

				for( ;; ) {
					if( yPos <= iTop + m_iFontLineSpacing ) {
						// This row!!!
						if( iTop != firstRowTop ) {
							if( g_pOptions->m_bWrapMargin )
								iLeft += m_iWrapMargin;
						}
						if( xPos < iLeft )
							return 0;
						for( ;; ) {
							if( i >= l->num_text_blocks )
								return 0;
							if( l->text_blocks_ptr[i].block_width )
								iLeft += l->text_blocks_ptr[i].block_width;
							else {
								if( i < (l->num_text_blocks - 1) ) {
									// There is another block...
									// Check if it is a wrap...
									if( !(l->text_blocks_ptr[i + 1].attr_ptr) )
										iLeft = width();
									// else: simply a zero characters block
								}
							}
							if( i < 0 )
								return 0; // Word wrap block; no link here!
							if( xPos < iLeft ) {
								// Got it! Link?
								bool bHadWordWraps = false;
								while( !(l->text_blocks_ptr[i].attr_ptr) ) {
									// Word wrap?
									if( i >= 0 ) {
										i--;
										bHadWordWraps = true;
									} else return 0; // All word wraps?!!!
								}
								if( l->text_blocks_ptr[i].attr_ptr->attribute == KVI_TEXT_ESCAPE ) {
									*rectTop    = bHadWordWraps ? firstRowTop : iTop;
									*rectHeight = ((l->line_wraps + 1) * m_iFontLineSpacing) + m_iFontDescent;
									return &(l->text_blocks_ptr[i]);
								}
								return 0;
							}
							i++;
						}
					} else {
						// Run until a word wrap block
						i++; // At least one block!
						while( i < l->num_text_blocks ) {
							// Still OK to run right
							if( !(l->text_blocks_ptr[i].attr_ptr) )
								break;
							else
								i++;
						}
						if( i >= l->num_text_blocks )
							return 0;
						iTop += m_iFontLineSpacing;
					}
				}
			} else l = l->prev_line;
		} else return 0;
	}
	return 0;
}

void KviIrcView::mouseMoveEvent(QMouseEvent *e)
{
	if( m_bSelecting ) {
		if( m_iSelectTimer == 0 )
			m_iSelectTimer = startTimer(KVI_IRC_VIEW_SELECT_REPAINT_INTERVAL);
	} else {
		if( m_iSelectTimer ) {
			killTimer(m_iSelectTimer);
			m_iSelectTimer = 0;
		}
		int yPos = e->pos().y();
		int rectTop;
		int rectHeight;
		KviIrcViewTextBlock *newLinkUnderMouse = getLinkUnderMouse(e->pos().x(), yPos, &rectTop, &rectHeight);
		if( newLinkUnderMouse != m_pLastLinkUnderMouse ) {
			m_pLastLinkUnderMouse = newLinkUnderMouse;
			if( m_pLastLinkUnderMouse ) {
				KviStr dbl, rght;
				KviStr tmp(_i18n_("Link:"));
				getDoubleClickCommand(dbl, m_pLastLinkUnderMouse->attr_ptr->escape_cmd);
				if( dbl.hasData() )
					tmp.append(KviStr(KviStr::Format, _i18n_(" double-click: \"%s\""), dbl.ptr()));
				getRightClickCommand(rght, m_pLastLinkUnderMouse->attr_ptr->escape_cmd);
				if( rght.hasData() )
					tmp.append(KviStr(KviStr::Format, _i18n_(" right click: \"%s\""), rght.ptr()));
				m_pFrm->m_pStatusBar->tempText(tmp.ptr(), 5000);

				if( rectTop < 0 )
					rectTop = 0;
				if( (rectTop + rectHeight) > height() )
					rectHeight = height() - rectTop;

				if( m_iLastLinkRectHeight > -1 ) {
					// Prev link
					int top        = (rectTop < m_iLastLinkRectTop) ? rectTop : m_iLastLinkRectTop;
					int lastBottom = m_iLastLinkRectTop + m_iLastLinkRectHeight;
					int thisBottom = rectTop + rectHeight;
					QRect r(0, top, width(), ((lastBottom > thisBottom) ? lastBottom : thisBottom) - top);
					QPaintEvent *pev = new QPaintEvent(r);
					paintEvent(pev);
					delete pev;
				} else {
					// No prev link
					QRect r(0, rectTop, width(), rectHeight);
					QPaintEvent *pev = new QPaintEvent(r);
					paintEvent(pev);
					delete pev;
				}
				m_iLastLinkRectTop    = rectTop;
				m_iLastLinkRectHeight = rectHeight;
			} else {
				if( m_iLastLinkRectHeight > -1 ) {
					// There was a previous bottom rectangle
					QRect r(0, m_iLastLinkRectTop, width(), m_iLastLinkRectHeight);
					QPaintEvent *pev = new QPaintEvent(r);
					paintEvent(pev);
					delete pev;
					m_iLastLinkRectTop   = -1;
					m_iLastLinkRectHeight = -1;
				}
			}
		}
	}
}

/**
 * =============== timerEvent ===============
 */
void KviIrcView::timerEvent(QTimerEvent *e)
{
	if( e->timerId() == m_iSelectTimer ) {
		QPoint pnt = mapFromGlobal(QCursor::pos());
		m_iMouseButtonCurrentX = pnt.x();
		m_iMouseButtonCurrentY = pnt.y();
		calculateSelectionBounds();
		paintEvent(0);
	}
}

/**
 * ============= mouseDoubleClickEvent ===============
 */
void KviIrcView::mouseDoubleClickEvent(QMouseEvent *e)
{
	m_iMouseButtonPressX = e->pos().x();
	m_iMouseButtonPressY = e->pos().y();

	if( m_szLastDoubleClickedEscapeCmd.isEmpty() )
		return;
	KviStr tmp;
	getDoubleClickCommand(tmp, m_szLastDoubleClickedEscapeCmd.ptr());
	if( tmp.isEmpty() )
		return;
	if( m_pFrm && m_pKviWindow )
		m_pFrm->m_pUserParser->parseCommand(tmp.ptr(), m_pKviWindow, m_szLastDoubleClickedEscape.ptr());
}

/**
 *=============== fontChange ================
 */
void KviIrcView::fontChange(const QFont &oldFont)
{
	// Do not update
	recalcFontVariables(font());
	// Force recalculation of all the wraps in the next repaint
	KviIrcViewTextLine *l = m_pFirstLine;
	while( l ) {
		l->max_line_width = -1;
		l                 = l->next_line;
	}
}

bool KviIrcView::setNeedScrollBar(bool need)
{
	if( m_bNeedScrollBar == need )
		return false;
	
	m_bNeedScrollBar = need;
	if( m_bNeedScrollBar )
		m_pScrollBar->show();
	else
		m_pScrollBar->hide();
	resizeMemBuffer();
	return true;
}

#include "m_kvi_irc_view.moc"
