// =============================================================================
//
//      --- kvi_userparser.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviUserParser"

#include "kvi_alias.h"
#include "kvi_app.h"
#include "kvi_asyncdialog.h"
#include "kvi_asyncwhois.h"
#include "kvi_channel.h"
#include "kvi_command.h"
#include "kvi_console.h"
#include "kvi_dcc_chat.h"
#include "kvi_debug.h"
#include "kvi_defines.h"
#include "kvi_dns.h"
#include "kvi_error.h"
#include "kvi_event.h"
#include "kvi_exprtree.h"
#include "kvi_frame.h"
#include "kvi_input.h"
#include "kvi_irc_socket.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_script_object.h"
#include "kvi_script_objectcontroller.h"
#include "kvi_settings.h"
#include "kvi_slaveio.h"
#include "kvi_userparser.h"
#include "kvi_userpopupmenu.h"
#include "kvi_variablecache.h"
#include "kvi_window.h"
#ifdef COMPILE_PLUGIN_SUPPORT
	#include "kvi_plugin.h"
	extern KviPluginManager *g_pPluginManager;
#endif

// Created in KviApp constructor
extern KviAliasManager  *g_pAliasManager;
extern KviEventManager  *g_pEventManager;
extern KviVariableCache *g_pVarCache;

/**
 * === KviUserParser class ===================================================
 */
KviUserParser::KviUserParser(KviFrame *pFrm)
	: QObject()
{
	m_pFrm = pFrm;
	m_pAsyncDialogList        = new QPtrList<KviAsyncDialog>;
	m_pAsyncDialogList->setAutoDelete(false);
	m_pDnsList                = new QPtrList<KviDns>;
	m_pDnsList->setAutoDelete(true);
	m_pTimerList              = new QPtrList<KviWindowTimer>;
	m_pTimerList->setAutoDelete(true);
	m_pSlaveIOController      = new KviSlaveIO(pFrm, this);
	m_pAsyncWhoisController   = new KviAsyncWhois(pFrm, this);
	m_szLastTryResult         = "1";
	m_pScriptObjectController = new KviScriptObjectController(this);
}

KviUserParser::~KviUserParser()
{
	// The first to die is m_pScriptObjectController.
	// Removes all the objects -- that might call something in the OnDestroy events
	delete m_pScriptObjectController; m_pScriptObjectController = 0;
	m_pAsyncDialogList->setAutoDelete(true); // Remove all the remaining dialogs
	delete m_pAsyncDialogList;        m_pAsyncDialogList        = 0;
	delete m_pDnsList;                m_pDnsList                = 0 ;
	while( !m_pTimerList->isEmpty() ) {
		KviWindowTimer *timer = m_pTimerList->first();
		__range_valid(timer);
		killTimer(timer->id);
		m_pTimerList->removeFirst();
	}
	delete m_pTimerList;              m_pTimerList              = 0;
	delete m_pSlaveIOController;      m_pSlaveIOController      = 0;
	delete m_pAsyncWhoisController;   m_pAsyncWhoisController   = 0;
}

void KviUserParser::slaveAsyncProcessExited(
	KviStr &job, KviStr &commandline, KviStr &stdoutBuf, KviStr &stderrBuf,
	KviStr &magic, KviStr &window, unsigned int uPid, int iExitCode)
{
	m_szLastProcCommandline  = commandline;
	m_szLastProcMagic        = magic;
	m_szLastProcStdout       = stdoutBuf;
	m_szLastProcStderr       = stderrBuf;
	m_szLastProcPid.setNum(uPid);
	m_szLastProcExitCode.setNum(iExitCode);
	m_szLastProcWindow       = window;

	KviWindow *wnd = m_pFrm->findWindow(window.ptr());

	parseCommand(job.ptr(), wnd ? wnd : m_pConsole);

	m_szLastProcCommandline  = "";
	m_szLastProcMagic        = "";
	m_szLastProcStdout       = "";
	m_szLastProcStderr       = "";
	m_szLastProcPid          = "";
	m_szLastProcWindow       = "";
	m_szLastProcExitCode     = "";
}

void KviUserParser::asyncWhoisEnd(
	KviStr &asyncCmd, KviStr &magic, KviStr &window, KviStr &nick, KviStr &user, KviStr &host,
	KviStr &realName, KviStr &server, KviStr &idle, KviStr &status, bool bIrcOp, KviStr &chans, bool bSuccess)
{
	m_szLastAsyncWhoisNick     = nick;
	m_szLastAsyncWhoisUsername = user;
	m_szLastAsyncWhoisHost     = host;
	m_szLastAsyncWhoisRealName = realName;
	m_szLastAsyncWhoisServer   = server;
	m_szLastAsyncWhoisIdleTime = idle;
	m_szLastAsyncWhoisStatus   = status;
	m_szLastAsyncWhoisIrcOp    = (bIrcOp ? "1" : "0");
	m_szLastAsyncWhoisChannels = chans;
	m_szLastAsyncWhoisWindow   = window;
	m_szLastAsyncWhoisMagic    = magic;
	m_szLastAsyncWhoisSuccess  = (bSuccess ? "1" : "0");

	KviWindow *wnd = m_pFrm->findWindow(window.ptr());
	parseCommand(asyncCmd.ptr(), wnd ? wnd : m_pConsole);

	m_szLastAsyncWhoisNick     = "";
	m_szLastAsyncWhoisUsername = "";
	m_szLastAsyncWhoisHost     = "";
	m_szLastAsyncWhoisRealName = "";
	m_szLastAsyncWhoisServer   = "";
	m_szLastAsyncWhoisIdleTime = "";
	m_szLastAsyncWhoisStatus   = "";
	m_szLastAsyncWhoisIrcOp    = "";
	m_szLastAsyncWhoisChannels = "";
	m_szLastAsyncWhoisWindow   = "";
	m_szLastAsyncWhoisMagic    = "";
	m_szLastAsyncWhoisSuccess  = "";
}

int KviUserParser::completeCommand(KviStr &word, KviStr &match, KviStr &multiple)
{
	// Completes the command in "word".
	// Returns the number of matches.
	// 1 if there is an exact match (and it is in match)
	// 0 if there are no matches
	// n if there are n matches: multiple contains a list of the matches, match contains the common part
	KviCommandEntry *cmdTable;
	if( word.isEmpty() )
		return 0;

	char init = tolower(*(word.ptr()));
	if( init < 'o' )
		cmdTable = ((init < 'h') ? cmdTable_A_G : cmdTable_H_N);
	else
		cmdTable = ((init < 't') ? cmdTable_O_S : cmdTable_T_Z);

	int iMatches = 0;

	for( int i = 0; cmdTable[i].cmdName; i++ ) {
		if( kvi_strEqualNoLocaleCIN(word.ptr(), cmdTable[i].cmdName, word.len()) ) {
			if( match.hasData() ) {
				const char *c1 = match.ptr();
				const char *c2 = cmdTable[i].cmdName;
				while( *c1 && (tolower(*c1) == tolower(*c2)) ) {
					c1++;
					c2++;
				}
				match.cutRight(match.len() - (c1 - match.ptr()));
			} else match = cmdTable[i].cmdName;
			if( multiple.hasData() )
				multiple.append(", ");
			multiple.append(cmdTable[i].cmdName);
			iMatches++;
		}
	}

	if( iMatches == 1 )
		match.append(' ');
	return iMatches;
}

int KviUserParser::completeFunctionOrIdentifier(KviStr &word, KviStr &match, KviStr &multiple)
{
	// Completes the function or identifier in "word".
	// Returns the number of matches.
	// 1 if there is an exact match (and it is in match)
	// 0 if there are no matches
	// n is there are n matches: multiple contains a list of the matches, match contains the common part
	if( word.isEmpty() )
		return 0;
	int iMatches = 0;
	char init    = tolower(*(word.ptr()));

	KviFunctionEntry *fncTable;
	if( init < 'o' )
		fncTable = ((init < 'h') ? fncTable_A_G : fncTable_H_N);
	else
		fncTable = ((init < 't') ? fncTable_O_S : fncTable_T_Z);

	for( int i = 0; fncTable[i].fncName; i++ ) {
		if( kvi_strEqualNoLocaleCIN(word.ptr(), fncTable[i].fncName, word.len()) ) {
			if( match.hasData() ) {
				const char *c1 = match.ptr();
				const char *c2 = fncTable[i].fncName;
				while( *c1 && (tolower(*c1) == tolower(*c2)) ) {
					c1++;
					c2++;
				}
				match.cutRight(match.len() - (c1 - match.ptr()));
			} else match.sprintf("%s(", fncTable[i].fncName);
			if( multiple.hasData() )
				multiple.append(", ");
			multiple.append('$');
			multiple.append(fncTable[i].fncName);
			multiple.append("()");
			iMatches++;
		}
	}

	KviIdentifierEntry *idnTable;
	if( init < 'o' )
		idnTable = ((init < 'h') ? idnTable_A_G : idnTable_H_N);
	else
		idnTable = ((init < 't') ? idnTable_O_S : idnTable_T_Z);

	for( int i = 0; idnTable[i].idnName; i++ ) {
		if( kvi_strEqualNoLocaleCIN(word.ptr(), idnTable[i].idnName, word.len()) ) {
			if( match.hasData() ) {
				const char *c1 = match.ptr();
				const char *c2 = idnTable[i].idnName;
				while( *c1 && (tolower(*c1) == tolower(*c2)) ) {
					c1++;
					c2++;
				}
				match.cutRight(match.len() - (c1 - match.ptr()));
			} else match = idnTable[i].idnName;
			if( multiple.hasData() )
				multiple.append(", ");
			multiple.append('$');
			multiple.append(idnTable[i].idnName);
			iMatches++;
		}
	}
	if( iMatches == 1 ) {
		if( !match.lastCharIs('(') )
			match.append(' ');
	}
	return iMatches;
}

void KviUserParser::timerEvent(QTimerEvent *e)
{
	KviWindowTimer *t = findTimerById(e->timerId());
	__range_valid(t);
	if( m_pFrm->windowExists(t->window) ) {
		m_szLastTimerMagic = t->magic;
		m_szLastTimerName  = t->name;
		if( !parseCommand(t->job.ptr(), t->window) ) {
			// Need to remove the timer... there was an error
			t->singleShot = true;
		}
	} else t->singleShot = true; // Remove it... no window
	if( t->singleShot ) {
		killTimer(t->id);
		m_pTimerList->removeRef(t);
	}
}

void KviUserParser::unregisterTimersFor(KviWindow *wnd)
{
	QPtrList<KviWindowTimer> l;
	l.setAutoDelete(false);
	for( KviWindowTimer *t = m_pTimerList->first(); t; t = m_pTimerList->next() ) {
		if( t->window == wnd )
			l.append(t);
	}
	for(KviWindowTimer *tim = l.first(); tim; tim = l.next() ) {
		killTimer(tim->id);
		m_pTimerList->removeRef(tim);
	}
}

KviWindowTimer *KviUserParser::findTimerById(int id)
{
	for( KviWindowTimer *t = m_pTimerList->first(); t; t = m_pTimerList->next() ) {
		if( t->id == id )
			return t;
	}
	return 0;
}

KviWindowTimer *KviUserParser::findTimerByName(const char *name)
{
	for( KviWindowTimer *t = m_pTimerList->first(); t; t = m_pTimerList->next() ) {
		if( kvi_strEqualCI(name, t->name.ptr()) )
			return t;
	}
	return 0;
}

void KviUserParser::addTimer(KviWindowTimer *t)
{
	KviWindowTimer *old = findTimerByName(t->name.ptr());
	if( !m_pFrm->windowExists(t->window) ) {
		__range_valid(m_pFrm->windowExists(t->window));
		return; // Make sure that it exists
	}
	if( old ) {
		killTimer(old->id);
		m_pTimerList->removeRef(old);
	}
	m_pTimerList->append(t);
	t->id = startTimer(t->timeout);
}

bool KviUserParser::removeTimer(const char *name)
{
	KviWindowTimer *t = findTimerByName(name);
	if( !t ) return false;

	killTimer(t->id);
	m_pTimerList->removeRef(t);
	return true;
}

void KviUserParser::setup(KviIrcSocket *pSock, KviConsole *pConsole)
{
	m_pConsole = pConsole;
	m_pSocket  = pSock;
}

void KviUserParser::addAsyncDialog(KviAsyncDialog *dlg)
{
	m_pAsyncDialogList->append(dlg);
}

void KviUserParser::removeAsyncDialog(KviAsyncDialog *dlg)
{
	if( m_pAsyncDialogList->autoDelete() )
		return; // KviUserParser destructor deleted the dialog and removed it from the list
	m_pAsyncDialogList->removeRef(dlg);
}

/**
 * === parseUserCommand ======================================================
 *
 * This is called directly from the KviInput class
 *
 * This method  parses a command buffer that comes from the input line.
 * If the buffer contains an initial slash it is interpreted
 * as a buffer containing commands to be parsed,
 * otherwise it is simply sent to the appropriate target.
 */
void KviUserParser::parseUserCommand(QString &str, KviWindow *pWnd)
{
	KviStr pme   = m_pFrm->m_global.szCurrentNick;
	KviStr pmask = m_pFrm->m_global.szCurrentMaskFromServer;

	const char *ptr = str.ascii();
	while( *ptr && (((unsigned char) *ptr) < 33) )
		++ptr; // Skip initial whitespace
	if( !*ptr ) return; // Empty data buffer

	if( (*ptr == '/') || (*ptr == g_pOptions->m_cPersonalCommandPrefix) ) { // A command
		ptr++; // Skip the slash
		if( !*ptr )
			return; // Empty command
		parseCommand(ptr, pWnd);
	} else { // Non-command
		// Check for a backslash at the beginning
		// and remove it (escaped /)... is this the right way?
		if( *ptr == '\\' ) {
			ptr++;
			if( *ptr )
				str.remove((ptr - str.ascii()) - 1, 1);
		}
		switch( pWnd->type() ) {
			case KVI_WND_TYPE_CONSOLE:
				if( g_pEventManager->eventEnabled(KviEvent_OnConsoleInput) ) {
					KviStr eventparms(KviStr::Format, "%s", str.ascii());
					if( callEvent(KviEvent_OnConsoleInput, m_pConsole, eventparms) )
						return;
				}
				if( m_pSocket->sendData(str.ascii(), str.length()) )
					pWnd->output(KVI_OUT_RAW, _i18n_("Raw to server: %s"), str.ascii());
				else
					pWnd->outputNoFmt(KVI_OUT_ERROR, _i18n_("Cannot send data: not connected to server"));
				break;
			case KVI_WND_TYPE_CHANNEL:
				if( g_pEventManager->eventEnabled(KviEvent_OnChannelInput) ) {
					KviStr eventparms(KviStr::Format, "%s %s", pWnd->caption(), str.ascii());
					if( callEvent(KviEvent_OnChannelInput, pWnd, eventparms) )
						return;
				}
				if( ((KviChannel *) pWnd)->m_bOnChannel ) {
					if( m_pSocket->sendFmtData("PRIVMSG %s :%s", pWnd->caption(), str.ascii()) )
						m_pFrm->outputPrivmsg(pWnd, KVI_OUT_OWN, pme.ptr(), pmask.ptr(), str.ascii());
					else
						pWnd->outputNoFmt(KVI_OUT_ERROR, _i18n_("Cannot send data: not connected to server"));
				} else pWnd->outputNoFmt(KVI_OUT_ERROR, _i18n_("Cannot send data: not on channel"));
				break;
			case KVI_WND_TYPE_QUERY:
				if( g_pEventManager->eventEnabled(KviEvent_OnQueryInput) ) {
					KviStr eventparms(KviStr::Format, "%s %s", pWnd->caption(), str.ascii());
					if( callEvent(KviEvent_OnQueryInput, pWnd, eventparms) )
						return;
				}
				if( m_pSocket->sendFmtData("PRIVMSG %s :%s", pWnd->caption(), str.ascii()) ) {
					m_pFrm->outputPrivmsg(pWnd, KVI_OUT_OWN,
						m_pFrm->m_global.szCurrentNick.ptr(), m_pFrm->m_global.szCurrentMaskFromServer.ptr(), str.ascii()
					);
				} else pWnd->outputNoFmt(KVI_OUT_ERROR, _i18n_("Cannot send data: not connected to server"));
				break;
			case KVI_WND_TYPE_CHAT:
				if( g_pEventManager->eventEnabled(KviEvent_OnDCCInput) ) {
					KviStr eventparms(KviStr::Format, "%s %s", pWnd->caption(), str.ascii());
					if( callEvent(KviEvent_OnDCCInput, pWnd, eventparms) )
						return;
				}
				if( ((KviDccChat *) pWnd)->sendData(str.ascii()) ) {
					m_pFrm->outputPrivmsg(pWnd, KVI_OUT_OWN,
						((KviDccChat *) pWnd)->m_szLocalNick.ptr(), ((KviDccChat *) pWnd)->m_szLocalMask.ptr(), str.ascii()
					);
				} else pWnd->output(KVI_OUT_ERROR, _i18n_("Cannot send data: no connection"));
				break;
			case KVI_WND_TYPE_UWINDOW:
				if( g_pEventManager->eventEnabled(KviEvent_OnUserWindowInput) ) {
					KviStr eventparms(KviStr::Format, "%s %s", pWnd->caption(), str.ascii());
					if( callEvent(KviEvent_OnUserWindowInput, pWnd, eventparms) )
						return;
				}
				pWnd->output(KVI_OUT_NONE, str.ascii());
				break;
			default: // Just do nothing
				break;
		}
	}
}

bool KviUserParser::parseCommand(const char *ptr, KviWindow *pWnd, const char *params)
{
	KviCommand cmd(ptr, pWnd);
	if( params )
		cmd.setParams(KviStr("commandline"), params);
	if( !execCommandBuffer(&cmd) ) {
		// Stop the execution.
		// If there is no error then we simply halt;
		// If there is an error we spit it out.
		if( cmd.hasError() ) {
			printError(&cmd, ptr);
			return false;
		}
	}
	return true;
}

/**
 * === parseAsyncDialog callback =============================================
 *
 * This is called directly from the KviAsyncDialog class.
 * This method parses a command buffer that comes from the async dialog
 * Really similar to...(?)
 *
 */
void KviUserParser::parseAsyncDialogCallback(KviStr &str, KviWindow *pWnd)
{
	// First problem... the window pWnd may not exist
	if( !m_pFrm->windowExists(pWnd) )
		pWnd = m_pConsole; // Use the console instead
	char *ptr = str.ptr();
	while( *ptr && (*ptr < 33) )
		++ptr; // Skip initial whitespace...
	if( !*ptr )
		return; // Empty data buffer...
	KviCommand cmd(ptr, pWnd);
	if( !execCommandBuffer(&cmd) ) {
		if( cmd.hasError() )
			printError(&cmd, ptr);
	}
}

void KviUserParser::getErrorInfo(KviCommand *cmd, const char *bufferBegin, KviCommandErrorInfo *inf)
{
	inf->iLineNum = 1;
	const char *ptr      = bufferBegin;
	const char *lastLine = ptr;
	while( *ptr ) {
		if( ptr == cmd->m_ptr )
			break;
		if( *ptr == '\n' ) {
			++(inf->iLineNum);
			lastLine = ptr+1;
		}
		ptr++;
	}
	// The current line data
	KviStr tmp;
	tmp.setStr(lastLine, 35);
	tmp.getLine(inf->szLineStr);
	inf->szLineStr += "...";
	inf->iCharNum = cmd->m_ptr - lastLine;
}

void KviUserParser::printError(KviCommand *cmd, const char *bufferBegin)
{
	// The current line number
	KviCommandErrorInfo inf;
	getErrorInfo(cmd, bufferBegin, &inf);

	cmd->m_wnd->output(KVI_OUT_ERROR, "======================================================");
	cmd->m_wnd->output(KVI_OUT_ERROR, _i18n_("Error in command:   %s"), kvi_getErrorString(cmd->m_err));
	cmd->m_wnd->output(KVI_OUT_ERROR, "======================================================");
	cmd->m_wnd->output(KVI_OUT_ERROR, _i18n_("Parsing stopped at:"));
	cmd->m_wnd->output(KVI_OUT_ERROR, _i18n_("Line %d:            %s"), inf.iLineNum, inf.szLineStr.ptr());
	cmd->m_wnd->output(KVI_OUT_ERROR, _i18n_("Character:          %d"), inf.iCharNum);
	cmd->m_wnd->output(KVI_OUT_ERROR, _i18n_("Logical location:   %s"), cmd->m_szErrorLocation.ptr());
	if( cmd->m_szErrorToken.hasData() ) {
		int idx = cmd->m_szErrorToken.findFirstIdx('\n');
		KviStr tok((idx == -1) ? cmd->m_szErrorToken : cmd->m_szErrorToken.left(idx));
		if( tok.hasData() )
			cmd->m_wnd->output(KVI_OUT_ERROR, _i18n_("Error detail:       %s"), tok.ptr());
	}
}

/**
 * === THE REAL STUFF ========================================================
 *
 * This method parses a whole command buffer and executes it.
 * Returns false in case of execution stopped.
 * If cmd->err is non-zero in that moment it means that an
 * an error occurred, otherwise the execution was simply halted.
 *
 */
bool KviUserParser::execCommandBuffer(KviCommand *cmd)
{
	cmd->skipWhitespace();
	while( *(cmd->m_ptr) ) {
		if( *(cmd->m_ptr) == '{' ) {
			if( !execCommandBlock(cmd) )
				return false;
			// Here cmd->m_ptr points to the char immediately after the closing brace
		} else {
			if( !execSingleCommand(cmd) )
				return false;
			// Here cmd->m_ptr points to the char immediately after the separator
		}
		cmd->skipWhitespace();
	}
	return true;
}

bool KviUserParser::notConnectedToServer(KviCommand *cmd, const char *inCommand)
{
	if( g_pOptions->m_bPedanticParser )
		return cmd->setError(KVI_ERROR_NotConnectedToServer, inCommand);
	else
		return cmd->warning(_i18n_("%s: not connected to server: ignoring"), inCommand);
}

bool KviUserParser::recoverableError(KviCommand *cmd, int error, const char *cmdName, const char *additionalData)
{
	if( g_pOptions->m_bPedanticParser )
		return cmd->setError(error, cmdName, additionalData);
	else {
		if( additionalData && (*additionalData) )
			return cmd->warning(_i18n_("%s: %s (%s): ignoring"), cmdName, kvi_getErrorString(error), additionalData);
		else
			return cmd->warning(_i18n_("%s: %s: ignoring"), cmdName, kvi_getErrorString(error));
	}
}

/**
 *
 * This method parses a block of commands enclosed in braces.
 * Returns false in case of an error somewhere in the command
 * or if there are not enough matching braces.
 * Moves cmd->ptr up to the char after the matching closing brace.
 *
 */
bool KviUserParser::execCommandBlock(KviCommand *cmd)
{
	__range_valid(*(cmd->m_ptr) == '{');
	++(cmd->m_ptr);
	cmd->skipWhitespace();
	while( *(cmd->m_ptr) ) {
		switch( *(cmd->m_ptr) ) {
			case '}': ++(cmd->m_ptr);               return true;  break;
			case '{': if( !execCommandBlock(cmd) )  return false; break;
			default : if( !execSingleCommand(cmd) ) return false; break;
		}
		cmd->skipWhitespace();
	}
	// We should never reach this point
	cmd->setError(KVI_ERROR_MissingClosingBrace, _i18n_("COMMAND-BLOCK"));
	return false;
}

/**
 *
 * This method is called when cmd->m_ptr points to the beginning
 * of a command. (eg. echo -w $window some funny text;...).
 * If the first char is a KVI_GLOBAL_VAR_CHAR, we assume it to be
 * an assignment (or another unary operator).
 * Otherwise we look for the correct command procedure to call.
 *
 */
bool KviUserParser::execSingleCommand(KviCommand *cmd)
{
	cmd->clearBuffer();

	if( *(cmd->m_ptr) == ';' ) {
		++(cmd->m_ptr); // Empty command;
		return true;
	}

	if( (*(cmd->m_ptr) == KVI_GLOBAL_VAR_CHAR) || (*(cmd->m_ptr) == '$') )
		return parseCmd_LVALUE(cmd);

	if( *(cmd->m_ptr) == KVI_COMMENT_BEGIN_CHAR )
		return skipComment(cmd);

	const char *aux = cmd->m_ptr;
	while( *(cmd->m_ptr) && (isalnum(*(cmd->m_ptr)) || (*(cmd->m_ptr) == '_') || (*(cmd->m_ptr) == '.')) )
		++(cmd->m_ptr);
	KviStr tmp(aux, cmd->m_ptr - aux);

	// Alias?
	KviAlias *a = g_pAliasManager->findAlias(tmp.ptr());
	if( a )
		return parseAlias(cmd, a);

#ifdef COMPILE_PLUGIN_SUPPORT
	// Plugin command?
	KviPluginCommandHandler *h = g_pPluginManager->findCommandHandler(tmp.ptr());
	if( h )
		return parsePluginCommand(h, cmd);
#endif

	// Find the correct command table
	KviCommandEntry *cmdTable;
	char init = tolower(*(aux));

	if( init < 'o' )
		cmdTable = ((init < 'h') ? cmdTable_A_G : cmdTable_H_N);
	else
		cmdTable = ((init < 't') ? cmdTable_O_S : cmdTable_T_Z);

	for( int i = 0; cmdTable[i].cmdName; i++ ) {
		if( kvi_strEqualNoLocaleCIN(aux, cmdTable[i].cmdName, cmdTable[i].cmdLen) ) {
			// Check if the command is followed by a space or terminator,
			// or something that is not a letter or digit.
			if( (cmd->m_ptr - aux) == cmdTable[i].cmdLen ) {
				return (this->*(cmdTable[i].cmdProc))(cmd);
			}
		}
	}

	// Nope. Send command.
	if( g_pOptions->m_bSendUnknownCommandsToServer )
		return parseUnknownCommand(tmp, cmd);
	// Or return error?
	cmd->setError(KVI_ERROR_UnknownCommand, "COMMAND", cmd->m_ptr);
	return false;
}

bool KviUserParser::parseAlias(KviCommand *cmd, KviAlias *a)
{
	KviCommand command(a->szBuffer.ptr(), cmd->m_wnd);
	cmd->skipSpace();
	if( !command.setParamsFromCommand(a->szName, cmd, this) )
		return false;

	a->bLocked = true; // Protect against self modification
	if( !execCommandBuffer(&command) ) {
		a->bLocked = false; // Unlock
		if( command.hasError() ) {
			printError(&command, a->szBuffer.ptr());
			return false; // And halt
		}
	}
	cmd->setReturnValue(command.m_retBuffer.ptr());
	a->bLocked = false; // Unlock
	return true;
}

bool KviUserParser::parsePluginCommand(KviPluginCommandHandler *h, KviCommand *cmd)
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviPluginCommandStruct cmdstr;
	cmdstr.handle  = h->plugin_handle;
	cmdstr.app     = g_pApp;
	cmdstr.params  = new QPtrList<KviStr>;
	cmdstr.params->setAutoDelete(true);
	cmdstr.sock    = m_pSocket;
	cmdstr.console = m_pConsole;
	cmd->skipSpace();

	cmdstr.params->append(new KviStr(h->cmd_name.ptr()));

	while( (*(cmd->m_ptr) != ';') && (*(cmd->m_ptr) != '\n') && (*(cmd->m_ptr)) ) {
		if( !processCmdSingleToken(cmd) ) {
			while( cmdstr.params->first() )
				cmdstr.params->removeFirst();
			delete cmdstr.params;
			return false;
		}
		cmdstr.params->append(new KviStr(cmd->m_buffer));
		cmd->m_buffer = "";
		cmd->skipSpace();
	}
	if( (*(cmd->m_ptr)) )
		cmd->m_ptr++;

	cmdstr.frame  = m_pFrm;
	cmdstr.window = cmd->m_wnd;
	cmdstr.error  = KVI_ERROR_NoError;

	bool bResult = h->handler_routine(&cmdstr);
	delete cmdstr.params;

	if( !bResult ) // Error in plugin
		cmd->setError(cmdstr.error, h->cmd_name.ptr(), cmdstr.errorstr.ptr());

	return bResult;
#else // COMPILE_PLUGIN_SUPPORT
	return false;
#endif
}

bool KviUserParser::parsePluginFunction(KviPluginFunctionHandler *h, KviCommand *cmd, KviStr &buffer)
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviPluginCommandStruct cmdstr;
	cmdstr.handle  = h->plugin_handle;
	cmdstr.app     = g_pApp;
	cmdstr.params  = new QPtrList<KviStr>;
	cmdstr.params->setAutoDelete(true);
	cmdstr.sock    = m_pSocket;
	cmdstr.console = m_pConsole;
	cmd->skipSpace();

	cmdstr.params->append(new KviStr(h->fnc_name.ptr()));

	KviStr tmp;
	while( (*(cmd->m_ptr) != ')') && (*(cmd->m_ptr)) ) {
		if( !processFncSingleParam(cmd, tmp) ) {
			while( cmdstr.params->first() )
				cmdstr.params->removeFirst();
			delete cmdstr.params;
			return false;
		}
		cmdstr.params->append(new KviStr(tmp));
		tmp = "";
		cmd->skipSpace();
	}
	if( (*(cmd->m_ptr)) )
		cmd->m_ptr++;

	cmdstr.frame  = m_pFrm;
	cmdstr.window = cmd->m_wnd;
	cmdstr.error  = KVI_ERROR_NoError;

	bool bResult = h->handler_routine(&cmdstr, &tmp);
	buffer.append(tmp);

	delete cmdstr.params;

	if( !bResult ) // Error in plugin
		cmd->setError(cmdstr.error, h->fnc_name.ptr(), cmdstr.errorstr.ptr());

	return bResult;
#else // COMPILE_PLUGIN_SUPPORT
	return false;
#endif
}

bool KviUserParser::parsePluginIdentifier(KviPluginFunctionHandler *h, KviCommand *cmd, KviStr &buffer)
{
#ifdef COMPILE_PLUGIN_SUPPORT
	KviPluginCommandStruct cmdstr;
	cmdstr.handle  = h->plugin_handle;
	cmdstr.app     = g_pApp;
	cmdstr.params  = 0;
	cmdstr.sock    = m_pSocket;
	cmdstr.console = m_pConsole;
	cmd->skipSpace();
	cmdstr.frame   = m_pFrm;
	cmdstr.window  = cmd->m_wnd;
	cmdstr.error   = KVI_ERROR_NoError;

	KviStr tmp;
	bool bResult = h->handler_routine(&cmdstr, &tmp);
	buffer.append(tmp);

	if( !bResult ) // Error in plugin
		cmd->setError(cmdstr.error, h->fnc_name.ptr(), cmdstr.errorstr.ptr());

	return bResult;
#else // COMPILE_PLUGIN_SUPPORT
	return false;
#endif
}

/**
 * Returns TRUE if the execution was halted.
 * Otherwise FALSE.
 */
bool KviUserParser::callEvent(int index, KviWindow *pWnd, const KviStr &params)
{
#ifdef COMPILE_PLUGIN_SUPPORT
	__range_valid(g_pEventManager->m_bEventEnabled[index] || g_pEventManager->m_pEventHandlerList[index]);
#else
	__range_valid(g_pEventManager->m_bEventEnabled[index]);
#endif
	bool bHalt = false;

#ifdef COMPILE_PLUGIN_SUPPORT
	if( g_pEventManager->m_pEventHandlerList[index] ) {
		KviPluginCommandStruct cmd;
		cmd.app     = g_pApp;
		cmd.params  = new QPtrList<KviStr>;
		cmd.params->setAutoDelete(true);
		cmd.params->append(new KviStr(g_pEventManager->m_eventDescriptor[index].eventName));
		cmd.sock    = m_pSocket;
		cmd.console = m_pConsole;

		const char *aux = params.ptr();
		KviStr token;
		while( *aux ) {
			aux = kvi_extractToken(token, aux, ' ');
			token.stripSpace();
			if( token.hasData() )
				cmd.params->append(new KviStr(token));
		}
		cmd.window = pWnd;
		cmd.frame  = m_pFrm;
		cmd.error  = KVI_ERROR_NoError;

		for( KviPluginEventHandlerStruct *s = g_pEventManager->m_pEventHandlerList[index]->first()
		    ; s
		    ; s = g_pEventManager->m_pEventHandlerList[index]->next()
		) {
			cmd.handle = s->plugin_handle;
			bHalt = bHalt || (s->handler_routine)(&cmd);
		}
		delete cmd.params;
	}

	if( g_pEventManager->m_bEventEnabled[index] ) {
#endif
		KviCommand cmd(g_pEventManager->m_szEventBuffer[index].ptr(), pWnd);
		cmd.setParams(KviStr(g_pEventManager->m_eventDescriptor[index].eventName), params.ptr());
		if( !execCommandBuffer(&cmd) ) {
			if( cmd.hasError() ) {
				printError(&cmd, g_pEventManager->m_szEventBuffer[index].ptr());
				pWnd->output(KVI_OUT_ERROR,
					_i18n_("Caller event:       %s"), g_pEventManager->m_eventDescriptor[index].eventName
				);
				if( g_pOptions->m_bAutoDisableBrokenEvents ) {
					pWnd->output(KVI_OUT_ERROR,
						_i18n_("Event handler '%s' is broken: auto-disabling"),
						g_pEventManager->m_eventDescriptor[index].eventName
					);
					g_pEventManager->m_bEventEnabled[index] = false;
				}
				if( index != KviEvent_OnEventError ) {
					if( g_pEventManager->eventEnabled(KviEvent_OnEventError) ) {
						KviCommandErrorInfo inf;
						getErrorInfo(&cmd, g_pEventManager->m_szEventBuffer[index].ptr(), &inf);
						KviStr eventparms(KviStr::Format,
							"Event: %s Location: %s Line: %d Char: %d Error: %s Window: %s LastLine: %s",
							g_pEventManager->m_eventDescriptor[index].eventName,
							cmd.m_szErrorLocation.ptr(), inf.iLineNum, inf.iCharNum,
							kvi_getErrorString(cmd.m_err), pWnd->caption(), inf.szLineStr.ptr()
						);
						callEvent(KviEvent_OnEventError, m_pConsole, eventparms);
					}
				}
				return bHalt;
			} else return true; // Halted!
		}
#ifdef COMPILE_PLUGIN_SUPPORT
	}
#endif
	return bHalt;
}

bool KviUserParser::parseUnknownCommand(KviStr &command, KviCommand *cmd)
{
	if( !processCmdFinalPart(cmd) )
		return false;
	if( !m_pSocket->sendFmtData("%s :%s", command.ptr(), cmd->m_buffer.ptr()) )
		return notConnectedToServer(cmd, command.ptr());
	return true;
}

bool KviUserParser::callRawEvent(
	const KviStr &szCommand, const KviStr &szPrefix, const KviStr &szBuffer,
	const KviStr &szParams, const KviStr &szSourceMask, KviStr &retValue)
{
	KviCommand cmd(szBuffer.ptr(), m_pConsole);
	KviStr tmp(KviStr::Format, "%s %s", szPrefix.ptr(), szParams.ptr());
	cmd.setParams(szCommand, tmp.ptr());
	if( !execCommandBuffer(&cmd) ) {
		retValue = cmd.m_retBuffer;
		if( cmd.hasError() ) {
			printError(&cmd, szBuffer.ptr());
			m_pConsole->output(KVI_OUT_ERROR,
				_i18n_("Caller event:       raw %s (source mask: %s)"), szCommand.ptr(), szSourceMask.ptr()
			);
			if( g_pEventManager->eventEnabled(KviEvent_OnEventError) ) {
				KviCommandErrorInfo inf;
				getErrorInfo(&cmd, szBuffer.ptr(), &inf);
				KviStr eventparms(KviStr::Format,
					"Event: Raw %s Location: %s Line: %d Char: %d Error: %s SourceMask: %s LastLine: %s",
					szCommand.ptr(), cmd.m_szErrorLocation.ptr(), inf.iLineNum, inf.iCharNum,
					kvi_getErrorString(cmd.m_err), szSourceMask.ptr(), inf.szLineStr.ptr()
				);
				callEvent(KviEvent_OnEventError, m_pConsole, eventparms);
			}
			return false;
		} else return true; // Halted!
	} else retValue = cmd.m_retBuffer;
	return false;
}

bool KviUserParser::triggerObjectEvent(KviScriptObject *o, const char *eventName, const char *eventBuf, const KviStr &params)
{
	KviStr tmp = eventBuf;
	KviCommand cmd(tmp.ptr(), m_pConsole);
	cmd.setParams(KviStr(eventName), params.ptr());
	cmd.setThisId(o->id());
	if( !execCommandBuffer(&cmd) ) {
		if( cmd.hasError() ) {
			printError(&cmd, tmp.ptr());
			m_pConsole->output(KVI_OUT_ERROR, _i18n_("Caller event:       %s"), eventName);
			m_pConsole->output(KVI_OUT_ERROR, _i18n_("Caller object:      type(%s) name(%s) id(%s)"),
				o->getClass(), o->getName(), o->id()
			);
			if( g_pOptions->m_bAutoDisableBrokenEvents ) {
				m_pConsole->output(KVI_OUT_ERROR,
					_i18n_("Event handler '%s' for this object is broken: auto-removing"), eventName
				);
				o->removeEventHandler(eventName);
			}

			if( g_pEventManager->eventEnabled(KviEvent_OnEventError) ) {
				KviCommandErrorInfo inf;
				getErrorInfo(&cmd, tmp.ptr(), &inf);
				KviStr eventparms(KviStr::Format,
					"Event: Obj-%s Location: %s Line: %d Char: %d Error: %s CallerObjectClass: %s "\
					"CallerObjectName: %s LastLine: %s",
					eventName, cmd.m_szErrorLocation.ptr(), inf.iLineNum, inf.iCharNum,
					kvi_getErrorString(cmd.m_err), o->getClass(), o->getName(), inf.szLineStr.ptr()
				);
				callEvent(KviEvent_OnEventError, m_pConsole, eventparms);
			}
		} else return true; // Halted!
	}
	return false; // Do not stop
}

bool KviUserParser::callObjectFunction(
	KviScriptObject *o, const char *fncName, const char *fncBuf, QPtrList<KviStr> *params, KviStr &buffer)
{
	// If the params list is not null, it will be deleted!
	KviStr tmp = fncBuf;
	KviCommand cmd(tmp.ptr(), m_pConsole);

	cmd.setParams(KviStr(fncName), params);
	cmd.setThisId(o->id());

	bool bRet = execCommandBuffer(&cmd);
	buffer.append(cmd.m_retBuffer);
	if( !bRet ) {
		if( cmd.hasError() ) {
			printError(&cmd, tmp.ptr());
			m_pConsole->output(KVI_OUT_ERROR, _i18n_("Caller function:    %s"), fncName);
			m_pConsole->output(KVI_OUT_ERROR, _i18n_("Caller object:      type(%s) name(%s) id(%s)"),
				o->getClass(), o->getName(), o->id()
			);
			return false;
		}
	}
	return true; // Everything OK
}

bool KviUserParser::extractSwitches(KviCommand *cmd)
{
	cmd->clearSwitchList();
	cmd->skipSpace();
	while( *(cmd->m_ptr) == '-' ) {
		if( !extractSingleSwitch(cmd) )
			return false;
		cmd->skipSpace();
	}
	return true;
}

bool KviUserParser::extractSingleSwitch(KviCommand *cmd)
{
	__range_valid(*(cmd->m_ptr) == '-');
	++(cmd->m_ptr); // Skip the - char

	// Now a LETTER is required
	if( (!(*(cmd->m_ptr))) || (!isalpha(*(cmd->m_ptr))) ) {
		cmd->setError(KVI_ERROR_InvalidOption, _i18n_("SWITCH PARSING AFTER '-' DELIMITER"), cmd->m_ptr);
		return false;
	}

	KviStr *sw = new KviStr();
	sw->append(*(cmd->m_ptr));

	++(cmd->m_ptr); // Skip the switch letter
	cmd->skipSpace();

	// Now if we have a '=', a token follows the switch
	if( *(cmd->m_ptr) == '=' ) {
		// A token follows the switch
		++(cmd->m_ptr);
		cmd->skipSpace();
		sw->append('=');

		const char *aux_ptr = cmd->m_ptr;
		bool bInString = false;

		for( ;; ) {
			// Now skip all the uninteresting chars
			while( *aux_ptr && (*aux_ptr != '\\') && (*aux_ptr != '\n') &&
				(*aux_ptr != ' ') && (*aux_ptr != '\t') && (*aux_ptr != ';') &&
				(*aux_ptr != '$') && (*aux_ptr != '\"') && (*aux_ptr != KVI_GLOBAL_VAR_CHAR)
			) {
				aux_ptr++;
			}
			// Interesting char
			switch( *aux_ptr ) {
				case ' ' : // Fall-through
				case '\t': // Fall-through
				case ';' :
					if( !bInString ) { // End of the token; append the last block to the buffer and return
						sw->append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
						cmd->m_ptr = aux_ptr;
						cmd->m_switchList->append(sw);
						return true;
					} else while( (*aux_ptr == ' ') || (*aux_ptr == '\t') || (*aux_ptr == ';') ) {
						aux_ptr++; // In string; must skip it
					}
					break;
				case '\n': // Fall-through
				case '\0': // End of command buffer; append the last block to the buffer and return
					if( bInString ) {
						// Unescaped newline or end of data
						cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("SWITCH PARSING: PARAMETER EXTRACTION"));
						delete sw;
						return false;
					} // else: it is the end of the command
					sw->append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					cmd->m_ptr = aux_ptr;
					cmd->m_switchList->append(sw);
					return true;
					break;
				case '\"': // Beginning or end of a string
					sw->append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					bInString = !bInString;
					cmd->m_ptr = ++aux_ptr;
					break;
				case KVI_GLOBAL_VAR_CHAR: // Variable; append the last block to the buffer and process the var
					sw->append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					cmd->m_ptr = aux_ptr;
					if( !processVariable(cmd, *sw) ) {
						delete sw;
						return false;
					}
					aux_ptr = cmd->m_ptr;
					break;
				case '$': // System identifier; append the last block to the buffer and process the identifier
					sw->append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					cmd->m_ptr = aux_ptr;
					if( !processIdentifier(cmd, *sw) ) {
						delete sw;
						return false;
					}
					aux_ptr = cmd->m_ptr;
					break;
				case '\\': // Escape character; append the last block to the processed buffer
					sw->append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					cmd->m_ptr = ++aux_ptr;
					switch( *aux_ptr ) {
						case '\0':
							if( bInString ) {
								cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("SWITCH PARSING: PARAMETER EXTRACTION"));
								delete sw;
								return false;
							}
							cmd->m_switchList->append(sw);
							return true;
							break;
						case '\n':  // Escaped newline
							cmd->m_ptr = ++aux_ptr; // Skip it
							cmd->skipSpace();       // Skip leading spaces
							aux_ptr = cmd->m_ptr;   // Continue
							break;
						default: // Must be copied to the buffer without modification
							sw->append(*aux_ptr);
							cmd->m_ptr = ++aux_ptr;
						break;
					}
				break;
			}
		}
	}

	// It is not a '='
	cmd->m_switchList->append(sw);

	return true;
}

/**
 * This method must be called when cmd->m_ptr points to
 * a '%' or '$' char just after the '->' object scope operator.
 */
bool KviUserParser::processObjectScopeRValue(KviScriptObject *o, KviCommand *cmd, KviStr &buffer)
{
	KviScriptObject *aux = cmd->scopeObject(); // Save the current one
	cmd->setScopeObject(o);
	switch( *(cmd->m_ptr) ) {
		case '%': // Object variable
			if( !processVariable(cmd, buffer) )
				return false;
			break;
		case '$': // Object function or identifier call
			if( !processIdentifier(cmd, buffer) )
				return false;
			break;
		default: // Invalid!
			debug("Invalid char in object scope %s, check the code man!", cmd->m_ptr);
			return false;
			break;
	}
	cmd->setScopeObject(aux);
	return true;
}

/**
 *
 * This method parses the variable pointed to by ptr,
 * adds its value to cmd->m_buffer (or an empty string
 * if the var is not defined), then moves cmd->m_ptr
 * to the character immediately after the variable
 * and returns>
 *
 * <KVI_GLOBAL_VAR_CHAR>Variable        = global variable
 * <KVI_GLOBAL_VAR_CHAR>variable        = local variable
 * <KVI_GLOBAL_VAR_CHAR>Variable[index] = global dict
 * <KVI_GLOBAL_VAR_CHAR>variable[index] = local dict
 *
 */
bool KviUserParser::processVariable(KviCommand *cmd, KviStr &buffer)
{
	__range_valid(*cmd->m_ptr == KVI_GLOBAL_VAR_CHAR);
	const char *aux = ++(cmd->m_ptr);

	KviVariableCache *cache;
	if( cmd->scopeObject() ) {
		cache = cmd->scopeObject()->varCache();
	} else {
		if( islower(*aux) )
			cache = cmd->m_pLocalVarCache;
		else
			cache = g_pVarCache;
	}

	while( *aux && (isalnum(*aux) || (*aux == '.') || (*aux == '_')) )
		aux++;
	KviStr szVariable(cmd->m_ptr, aux - cmd->m_ptr);
	cmd->m_ptr = aux;

	const char *varVal = 0;
	if( szVariable.isEmpty() )
		buffer.append(KVI_GLOBAL_VAR_CHAR);
	else {
		if( *(cmd->m_ptr) == '[' ) {
			cmd->m_ptr++;
			KviStr szKey;
			// No scope object for variable keys
			KviScriptObject *scObj = cmd->scopeObject();
			cmd->setScopeObject(0);
			if( !processVariableKey(cmd, szKey) )
				return false;
			cmd->setScopeObject(scObj);
			// OK... got the key
			if( szKey.isEmpty() )
				return cmd->setError(KVI_ERROR_MissingDictionaryKey, _i18n_("VARIABLE EVALUATION"), szVariable.ptr());
			varVal = cache->findDictVariable(szVariable.ptr(), szKey.ptr());
		} else {
			varVal = cache->find(szVariable.ptr());
		}
		// Object scope?
		if( *(cmd->m_ptr) == '-' ) {
			if( *(cmd->m_ptr + 1) == '>' ) {
				cmd->m_ptr += 2;
				if( (*(cmd->m_ptr) == '%') || (*(cmd->m_ptr) == '$') ) {
					if( varVal ) {
						KviScriptObject *o = m_pScriptObjectController->findObjectById(varVal);
						if( o )
							return processObjectScopeRValue(o, cmd, buffer);
						else
							return cmd->setError(KVI_ERROR_ObjectNotFound, _i18n_("OBJECT OPERATOR -> (RVALUE)"), varVal);
					} else {
						return cmd->setError(KVI_ERROR_ObjectNotFound,
							_i18n_("OBJECT_OPERATOR -> (RVALUE)"),
							_i18n_("The variable contained no object id")
						);
					}
				} else {
					return cmd->setError(KVI_ERROR_VariableOrIdentifierExpected,
						_i18n_("OBJECT OPERATOR -> (RVALUE)"), cmd->m_ptr
					);
				}
			}
		}
		if( varVal )
			buffer.append(varVal);
	}
	return true;
}

/**
 * This method must be called when cmd->m_ptr points
 * somewhere in a variable parameters block.
 * The parsing is stopped by a '\n', ']' or '\0'.
 * On '\n' and '\0' an error is returned.
 * The parsing is NOT stopped if ']'
 * is in a string (delimitated by two '"' chars).
 * Skips the ending ']' character.
 * This function EATS leading and trailing spaces and tabs
 * and reduces the middle ones to one per group.
 * A key like this:
 *          string    "full  of  words"  and  spaces     .
 * Becomes like this:
 * string full  of  words and spaces.
 *
 */
bool KviUserParser::processVariableKey(KviCommand *cmd, KviStr &buffer)
{
	// Skip leading spaces and tabs (the escaped ones too)
	cmd->skipSpace();
	register const char *aux_ptr = cmd->m_ptr;

	bool inString = false;
	for( ;; ) {
		// First skip all the uninteresting chars
		while( *aux_ptr &&
			(*aux_ptr != '\n') && (*aux_ptr != ']') && (*aux_ptr != '$')  &&
			(*aux_ptr != ' ')  && (*aux_ptr != '"') && (*aux_ptr != '\\') &&
			(*aux_ptr != '\t') && (*aux_ptr != KVI_GLOBAL_VAR_CHAR)
		) {
			aux_ptr++;
		}
		// Interesting char
		switch( *aux_ptr ) {
			case '\0': // Fall-through
			case '\n': // End of the string ('\n' is not escaped); fail here
				if( inString )
					cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("DICTIONARY KEY"));
				else
					cmd->setError(KVI_ERROR_UnexpectedEndOfCommand, _i18n_("DICTIONARY KEY"));
				return false;
				break;
			case ' ':  // Fall-through
			case '\t': // Spaces; need to collapse it to one
				if( !inString ) {
					// Append the last part
					buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					// Skip the other spaces, including escaped ones
					cmd->m_ptr = aux_ptr;
					cmd->skipSpace();
					aux_ptr = cmd->m_ptr;
					if( (*aux_ptr != ']') )
						buffer.append(' ');
				} else while( (*aux_ptr == ' ') || (*aux_ptr == '\t') ) {
					aux_ptr++;
				}
				break;
			case '"': // Unescaped '"'; enter or exit the string -- skip the '"')
				buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = ++aux_ptr;
				inString   = !inString;
				break;
			case ']': // Exit if not in string
				if( !inString ) {
					buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					cmd->m_ptr = ++aux_ptr;
					return true;
				}
				++aux_ptr;
				break;
			case KVI_GLOBAL_VAR_CHAR: // Variable: append the last block to the buffer and process the var
				buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				if( !processVariable(cmd, buffer) )
					return false;
				aux_ptr = cmd->m_ptr;
				break;
			case '$': // System identifier: append the last block to the buffer and process the identifier
				buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				if( !processIdentifier(cmd, buffer) )
					return false;
				aux_ptr = cmd->m_ptr;
				break;
			case '\\': // Escape character: append the last block to the processed buffer
				buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = ++aux_ptr;
				switch( *aux_ptr ) {
					case '\0':
						if( inString )
							cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("DICTIONARY KEY"));
						else
							cmd->setError(KVI_ERROR_UnexpectedEndOfCommand, _i18n_("DICTIONARY KEY"));
						return false;
						break;
					case '\n':  // Escaped newline
						cmd->m_ptr = ++aux_ptr; // Skip it
						cmd->skipSpace();       // Skip leading spaces
						aux_ptr = cmd->m_ptr;   // Continue
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						cmd->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Never arrive here
	return true;
}

/**
 * This method expects cmd->m_ptr to point to a digit or '#'
 */
bool KviUserParser::processParameterIdentifier(KviCommand *cmd, KviStr &buffer)
{
	__range_valid(isdigit(*(cmd->m_ptr)) || ((*(cmd->m_ptr)) == '#'));
	if( *(cmd->m_ptr) == '#' ) {
		cmd->getParamCount(buffer);
		++(cmd->m_ptr);
		return true;
	}
	int param = (*(cmd->m_ptr) - '0');
	++(cmd->m_ptr);
	while( isdigit(*(cmd->m_ptr)) ) {
		param  = param * 10;
		param += (*(cmd->m_ptr) - '0');
		++(cmd->m_ptr);
	}
	if( *(cmd->m_ptr) == '-' ) {
		++(cmd->m_ptr);
		if( isdigit(*(cmd->m_ptr)) ) {
			int parEnd = (*(cmd->m_ptr) - '0');
			++(cmd->m_ptr);
			while( isdigit(*(cmd->m_ptr)) ) {
				parEnd  = parEnd * 10;
				parEnd += (*(cmd->m_ptr) - '0');
				++(cmd->m_ptr);
			}
			cmd->getParam(param, parEnd, buffer);
		} else {
			if( *(cmd->m_ptr) == '>' ) {
				// Oops! Was an object scope operator!
				--(cmd->m_ptr);
				cmd->getSingleParam(param, buffer);
			} else cmd->getParam(param, -1, buffer); // Up to the end
		}
	} else cmd->getSingleParam(param, buffer);
	return true;
}

/**
 * This method parses the identifier pointed by cmd->m_ptr,
 * adds its value to buffer (or stops if the identifier is not valid),
 * then moves cmd->m_ptr to the character immediately
 * after the variable and returns it.
 */
bool KviUserParser::processIdentifier(KviCommand *cmd, KviStr &buffer)
{
	__range_valid(*cmd->m_ptr == '$');
	const char *aux = ++(cmd->m_ptr); // Skip the $

	KviStr tmpBuffer;
	if( isdigit(*aux) || (*aux == '#') ) {
		// $0, $1, $2...
		// $0 name of the command
		// $# number of params
		if( !processParameterIdentifier(cmd, tmpBuffer) )
			return false;
	} else {
		if( *aux == '{' ) {
			// A command call
			cmd->m_ptr = aux;
			if( !skipCommand(cmd) )
				return false;
			KviStr tmp(aux, cmd->m_ptr);
			KviCommand _cmd(tmp.ptr(), cmd->m_wnd);
			if( !execCommandBuffer(&_cmd) ) {
				if( _cmd.hasError() ) {
					printError(&_cmd, tmp.ptr());
					return false; // We could do better?
				}
			}
			tmpBuffer.append(_cmd.m_retBuffer);
		} else {
			// Normal identifier
			if( cmd->scopeObject() ) {
				while(*aux && (isalnum(*aux) || (*aux == '.') || (*aux == '_') || (*aux == ':')) )
					aux++; // Run up to the end (first non-alphanumeric)
			} else {
				while( *aux && (isalnum(*aux) || (*aux == '.') || (*aux == '_')) )
					aux++;
			}
			KviStr ident(cmd->m_ptr, aux - cmd->m_ptr); // And get the identifier name
			cmd->m_ptr = aux;

			bool bDone = false;
			if( ident.isEmpty() ) {
				tmpBuffer.append('$');  // Single $... nope
			} else { // Have an identifier
				if( cmd->scopeObject() ) {
					QPtrList<KviStr> *pList = 0;
					if( *(cmd->m_ptr) == '(' ) {
						// Have parameters
						pList = new QPtrList<KviStr>;
						pList->setAutoDelete(true);
						++(cmd->m_ptr);
						// No scope object for function parameters
						KviScriptObject *scObj = cmd->scopeObject();
						cmd->setScopeObject(0);
						bool bRecognizedObjScopeOperator = cmd->recognizeObjectScopeOperator();
						cmd->setRecognizeObjectScopeOperator(true);
						KviStr tmp;
						do {
							tmp = "";
							if( !processFncSingleParam(cmd, tmp) ) {
								delete pList;
								return false;
							}
							if( tmp.hasData() )
								pList->append(new KviStr(tmp));
						} while( tmp.hasData() );
						if( !processFncFinalPart(cmd, tmp) ) {
							delete pList;
							return false;
						}
						cmd->setScopeObject(scObj);
						cmd->setRecognizeObjectScopeOperator(bRecognizedObjScopeOperator);
					}
					int retVal;
					int idx = ident.findFirstIdx(':');
					KviStr classOverride;
					if( idx != -1 ) {
						// Class override
						classOverride = ident.left(idx);
						ident.cutLeft(idx + 1);
						retVal = cmd->scopeObject()->callFunction(ident.ptr(), pList, tmpBuffer, classOverride.ptr());
					} else retVal = cmd->scopeObject()->callFunction(ident.ptr(), pList, tmpBuffer, 0);
					if( retVal != KVI_ERROR_Success ) {
						KviStr er;
						if( classOverride.hasData() )
							er.sprintf(_i18n_("function name: %s class override: %s"), ident.ptr(), classOverride.ptr());
						else
							er.sprintf(_i18n_("function name: %s"), ident.ptr());
						return cmd->setError(retVal, _i18n_("OBJECT FUNCTION CALL"), er.ptr());
					}
				} else {
					// Not a scope object
					if( *(cmd->m_ptr) == '(' ) {
						++(cmd->m_ptr);
						// The identifier is followed by parameters; it is a function
						ident.toUpper();

						KviFunctionEntry *fncTable;
						char init = *(ident.ptr());
						if( init < 'O' )
							fncTable = ((init < 'H') ? fncTable_A_G : fncTable_H_N);
						else
							fncTable = ((init < 'T') ? fncTable_O_S : fncTable_T_Z);

						// Evaluate it...
						for( int i = 0; fncTable[i].fncName; i++ ) {
							if( kvi_strEqualCS(ident.ptr(), fncTable[i].fncName) ) {
								if( !(this->*(fncTable[i].fncProc))(cmd, tmpBuffer) )
									return false;
								bDone = true;
								break;
							}
						}
#ifdef COMPILE_PLUGIN_SUPPORT
						if( !bDone ) {
							// Plugin function?
							KviPluginFunctionHandler *h = g_pPluginManager->findFunctionHandler(ident.ptr());
							if( h ) {
								if( !parsePluginFunction(h, cmd, tmpBuffer) )
									return false;
								bDone = true;
							}
						}
#endif
						// Cannot recover without parsing the function paramenters.
						if( !bDone )
							return cmd->setError(KVI_ERROR_UnknownFunction, _i18n_("FUNCTION"), ident.ptr());
					} else {
						ident.toUpper();
						// Evaluate it...
						KviIdentifierEntry *idnTable;
						char init = *(ident.ptr());

						if( init < 'O' )
							idnTable = ((init < 'H') ? idnTable_A_G : idnTable_H_N);
						else
							idnTable = ((init < 'T') ? idnTable_O_S : idnTable_T_Z);

						// Evaluate it...
						for( int i = 0; idnTable[i].idnName; i++ ) {
							if( kvi_strEqualCS(ident.ptr(), idnTable[i].idnName) ) {
								// Will set the cmd->err on errors!
								if( !(this->*(idnTable[i].idnProc))(cmd, tmpBuffer) )
									return false;
								bDone = true;
								break;
							}
						}
#ifdef COMPILE_PLUGIN_SUPPORT
						if( !bDone ) {
							// Plugin function?
							KviPluginFunctionHandler *h = g_pPluginManager->findFunctionHandler(ident.ptr());
							if( h ) {
								if( !parsePluginIdentifier(h, cmd, tmpBuffer) )
									return false;
								bDone = true;
							}
						}
#endif
						// Can recover from this
						if( !bDone ) {
							if( g_pOptions->m_bPedanticParser )
								return cmd->setError(KVI_ERROR_UnknownIdentifier, _i18n_("IDENTIFIER"), ident.ptr());
							else
								return cmd->warning(_i18n_("IDENTIFIER: unknown identifier (%s): ignoring"), ident.ptr());
						}
					}
				}
			}
		}
	}
	if( cmd->recognizeObjectScopeOperator() ) {
		// Object scope?
		if( *(cmd->m_ptr) == '-' ) {
			if( *(cmd->m_ptr + 1) == '>' ) {
				cmd->m_ptr += 2;
				if( (*(cmd->m_ptr) == '%') || (*(cmd->m_ptr) == '$') ) {
					if( tmpBuffer.hasData() ) {
						KviScriptObject *o = m_pScriptObjectController->findObjectById(tmpBuffer.ptr());
						if( o )
							return processObjectScopeRValue(o, cmd, buffer);
						else {
							return cmd->setError(KVI_ERROR_ObjectNotFound,
								_i18n_("OBJECT OPERATOR -> (RVALUE)"), tmpBuffer.ptr()
							);
						}
					} else {
						return cmd->setError(KVI_ERROR_ObjectNotFound,
							_i18n_("OBJECT_OPERATOR -> (RVALUE)"), _i18n_("The function/identifier call returned no object id")
						);
					}
				} else {
					return cmd->setError(KVI_ERROR_VariableOrIdentifierExpected,
						_i18n_("OBJECT OPERATOR -> (RVALUE)"), cmd->m_ptr
					);
				}
			}
		}
	}
	buffer.append(tmpBuffer);

	return true;
}

/**
 *
 * This method must be called when cmd->m_ptr points
 * somewhere in a command buffer.
 * It extracts the first token encountered and
 * preprocesses it (subst variables, etc...),
 * then it appends everything to the cmd->m_buffer.
 * The parsing is stopped by a '\n', ' ', ';' or '\0'.
 * Note that this function does nothing if cmd->m_ptr
 * points to the end of a command.
 *
 */
bool KviUserParser::processCmdSingleToken(KviCommand *cmd)
{
	// Skip leading spaces... even the escaped ones
	cmd->skipSpace();
	const char *aux_ptr = cmd->m_ptr;

	bool bInString = false;
	for( ;; ) {
		// Now skip all the uninteresting chars
		while( *aux_ptr && (*aux_ptr != '\\') && (*aux_ptr != '\n') &&
			(*aux_ptr != ' ') && (*aux_ptr != '\t') && (*aux_ptr != ';') &&
			(*aux_ptr != '$') && (*aux_ptr != '\"') && (*aux_ptr != KVI_GLOBAL_VAR_CHAR)
		) {
			aux_ptr++;
		}
		// Interesting char
		switch( *aux_ptr ) {
			case ' ' : // Fall-through
			case '\t': // Fall-through
			case ';' :
				if( !bInString ) { // End of the token; append the last block to the buffer and return
					cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					cmd->m_ptr = aux_ptr;
					return true;
				} else while( (*aux_ptr == ' ') || (*aux_ptr == '\t') || (*aux_ptr == ';') ) {
					aux_ptr++; // In string; must skip it
				}
				break;
			case '\n': // Fall-through
			case '\0': // End of command buffer; append the last block to the buffer and return
				if( bInString ) {
					// Unescaped newline or end of data
					cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("COMMAND-PARAMS"));
					return false;
				} // else: it is the end of the command
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				return true;
				break;
			case '\"': // Beginning or end of a string
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				bInString = !bInString;
				cmd->m_ptr = ++aux_ptr;
				break;
			case KVI_GLOBAL_VAR_CHAR: // Variable: append the last block to the buffer and process the var
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				if( !processVariable(cmd, cmd->m_buffer) )
					return false;
				aux_ptr = cmd->m_ptr;
				break;
			case '$': // System identifier: append the last block to the buffer and process the identifier
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				if( !processIdentifier(cmd, cmd->m_buffer) )
					return false;
				aux_ptr = cmd->m_ptr;
				break;
			case '\\': // Escape character: append the last block to the processed buffer
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = ++aux_ptr;
				switch( *aux_ptr ) {
					case '\0':
						if( bInString ) {
							cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("COMMAND-PARAMS"));
							return false;
						}
						return true;
						break;
					case '\n':  // Escaped newline
						cmd->m_ptr = ++aux_ptr; // Skip it
						cmd->skipSpace();       // Skip leading spaces
						aux_ptr = cmd->m_ptr;   // Continue
						break;
					default: // Must be copied to the buffer without modification
						cmd->m_buffer.append(*aux_ptr);
						cmd->m_ptr = ++aux_ptr;
						break;
				}
			break;
		}
	}
	// Never arrive here
	return true;
}

/**
 * Preprocesses the final part of a standard command.
 * It is stopped by a newline, a ';', or the end of the buffer.
 * This method extracts the command parameters, evaluates it,
 * and appends everything to cmd->buffer.
 */
bool KviUserParser::processCmdFinalPart(KviCommand *cmd)
{
	// Skip spaces, but keep the terminators!
	cmd->skipSpace();
	const char *aux_ptr = cmd->m_ptr;

	bool bInString = false;
	for( ;; ) {
		// Now skip all the uninteresting chars
		while( *aux_ptr && (*aux_ptr != '\\') && (*aux_ptr != '\n') && (*aux_ptr != '\t') &&
			(*aux_ptr != ';') && (*aux_ptr != '$') && (*aux_ptr != ' ') &&
			(*aux_ptr != KVI_GLOBAL_VAR_CHAR) && (*aux_ptr != '\"')
		) {
			++aux_ptr;
		}
		// Interesting char
		switch( *aux_ptr ) {
			case '\0': // Fall-through
			case '\n': // End of command buffer; append the last block to the buffer and return
				if( bInString ) {
					// Oops; end of buffer while looking for matching '"'
					cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("COMMAND-PARAMS"));
					return false;
				}
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				if( *aux_ptr == '\n' )
					aux_ptr++; // Point after the semicolon
				cmd->m_ptr = aux_ptr;
				return true;
				break;
			case '\"': // Beginning or end of a string
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				bInString = !bInString;
				cmd->m_ptr = ++aux_ptr; // Do not forget to point to next char
				break;
			case ' ' : // Fall-through
			case '\t': // Whitespace; need to collapse it to one
				if( !bInString ) {
					// Append the last part
					cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					// Skip the other spaces including escaped ones
					cmd->m_ptr = aux_ptr;
					cmd->skipSpace();
					aux_ptr = cmd->m_ptr;
					if( (*aux_ptr != '\0') && (*aux_ptr != '\n') && (*aux_ptr != ';') )
						cmd->m_buffer.append(' ');
				} else while( (*aux_ptr == ' ') || (*aux_ptr == '\t') ) {
					aux_ptr++;
				}
				break;
			case ';' : // End of command; append the last block to the buffer and return
				if( !bInString) {
					cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					cmd->m_ptr = ++aux_ptr; // Do not forget to point to next char
					return true;
				} else ++aux_ptr; // Just skip it
				break;
			case KVI_GLOBAL_VAR_CHAR: // Variable; append the last block to the buffer and process the var
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				if( !processVariable(cmd, cmd->m_buffer) )
					return false;
				aux_ptr = cmd->m_ptr;
				break;
			case '$': // System identifier; append the last block to the buffer and process the identifier
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				if( !processIdentifier(cmd, cmd->m_buffer) )
					return false;
				aux_ptr = cmd->m_ptr;
				break;
			case '\\': // Escape character; append the last block to the processed buffer
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = ++aux_ptr;
				switch( *aux_ptr ) {
					case '\0':
						if( bInString ) {
							cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("COMMAND-PARAMS"));
							return false;
						} else return true;
						break; // Escaped nothing
					case '\n': // Escaped newline
						cmd->m_ptr = ++aux_ptr; // Skip it
						cmd->skipSpace();       // Skip leading spaces
						aux_ptr = cmd->m_ptr;   // Continue
						break;
					default: // Must be copied to the buffer without modification
						cmd->m_buffer.append(*aux_ptr);
						cmd->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	return true;
}

/**
 * Preprocesses the final part of a standard command.
 * It is stopped by a newline, a ';', or the end of the buffer.
 * This method extracts the command parameters, does not evaluate it,
 * and appends everything to cmd->buffer.
 */
bool KviUserParser::processCmdFinalPartNoIdent(KviCommand *cmd)
{
	cmd->skipSpace();
	const char *aux_ptr = cmd->m_ptr;

	bool bInString = false;
	for( ;; ) {
		// Now skip all the uninteresting chars
		while( *aux_ptr && (*aux_ptr != '\\') && (*aux_ptr != '\n') && (*aux_ptr != '\t') &&
			(*aux_ptr != ';') && (*aux_ptr != ' ') && (*aux_ptr != '\"')
		) {
			++aux_ptr;
		}
		// Interesting char
		switch( *aux_ptr ) {
			case '\0': // Fall-through
			case '\n': // End of command buffer; append the last block to the buffer and return
				if( bInString ) {
					// Oops; end of buffer while looking for matching '"'
					cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("COMMAND-PARAMS"));
					return false;
				}
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				if( *aux_ptr == '\n' )
					aux_ptr++; // Point after the semicolon
				cmd->m_ptr = aux_ptr;
				return true;
				break;
			case '\"': // Beginning or end of a string
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				bInString = !bInString;
				cmd->m_ptr = ++aux_ptr; // Do not forget to point to next char
				break;
			case ' ' : // Fall-through
			case '\t': // Whitespace; need to collapse it to one.
				if( !bInString ) {
					// Append the last part
					cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					// Skip the other spaces including escaped ones
					cmd->m_ptr = aux_ptr;
					cmd->skipSpace();
					aux_ptr = cmd->m_ptr;
					if( (*aux_ptr != '\0') && (*aux_ptr != '\n') && (*aux_ptr != ';') )
						cmd->m_buffer.append(' ');
				} else while( (*aux_ptr == ' ') || (*aux_ptr == '\t') ) {
					aux_ptr++;
				}
				break;
			case ';' : // End of command; append the last block to the buffer and return
				if( !bInString ) {
					cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					cmd->m_ptr = ++aux_ptr; // Do not forget to point to next char
					return true;
				} else ++aux_ptr; // Just skip it
				break;
			case '\\': // Escape character; append the last block to the processed buffer
				cmd->m_buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = ++aux_ptr;
				switch( *aux_ptr ) {
					case '\0':
						if( bInString ) {
							cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("COMMAND-PARAMS"));
							return false;
						} else return true;
						break; // Escaped nothing...
					case '\n': // Escaped newline
						cmd->m_ptr = ++aux_ptr; // Skip it
						cmd->skipSpace();       // Skip leading spaces
						aux_ptr = cmd->m_ptr;   // Continue
						break;
					default: // Must be copied to the buffer without modification
						cmd->m_buffer.append(*aux_ptr);
						cmd->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	return true;
}

bool KviUserParser::processString(const char *ptr, KviWindow *pWnd, KviStr &buffer)
{
	bool retval = false;
	KviCommand *cmd = new KviCommand(ptr, pWnd);
	if( processCmdFinalPart(cmd) ) {
		buffer = cmd->m_buffer.ptr();
		retval = true;
	}
	delete cmd;
	return retval;
}

/**
 * ===========================================================================
 *
 * Process function parameters
 * The following three functions process parameters of functions:
 * blocks of data enclosed in () parentheses.
 * All expect cmd->m_ptr to point somewhere inside that block, so after the first
 * parenthesis and before (or directly to) the ending one.
 *
 */

/**
 * This method will extract the first parameter of the function
 * and if it is followed by ',' it will skip it.
 * It will not skip the ending ')' so the caller will be able
 * to know if we reached the end.
 * This is different in processFncFinalPart; that will skip the
 * ending ')' too.
 */
bool KviUserParser::processFncSingleParam(KviCommand *cmd, KviStr &buffer)
{
	if( !processFncSingleParamInternal(cmd, buffer) )
		return false;
	__range_valid((*(cmd->m_ptr) == ',') || (*(cmd->m_ptr) == ')'));
	if( *(cmd->m_ptr) == ',' )
		++(cmd->m_ptr);
	return true;
}

/**
 * This method must be called when cmd->m_ptr points
 * somewhere in a function parameter block.
 * It extracts the first parameter encountered and
 * preprocesses it (subst variables, etc...),
 * then it appends everything to the buffer.
 * The parsing is stopped by a '\n', ', ', ')' or '\0'.
 * On '\n' and '\0' an error is returned.
 * The parsing is NOT stopped if one of ',' or ')'
 * is in a string (delimitated by two '"' chars).
 * Note that this function does nothing if cmd->m_ptr
 * points to the end of the param buffer.
 * (e.g. cmd->m_ptr comes from previous calls to processFncSingleParam())
 * It will also skip matching parentheses:
 * so () is a valid parameter.
 * It will return false in case of an error;
 * If it returns true, cmd->m_ptr surely points to a ',' or a ')'
 * This function EATS leading and trailing spaces and tabs
 * and reduces the middle ones to one per group.
 * A param like this:
 *          string    "full  of  words"  and  spaces     ,
 * Becomes like this:
 * string full  of  words and spaces,
 */
bool KviUserParser::processFncSingleParamInternal(KviCommand *cmd, KviStr &buffer)
{
	// Skip leading spaces and tabs (the escaped ones too)
	cmd->skipSpace();
	register const char *aux_ptr = cmd->m_ptr;

	int parLevel  = 0;
	bool inString = false;
	for( ;; ) {
		// First skip all the uninteresting chars
		while( *aux_ptr &&
			(*aux_ptr != '\n') && (*aux_ptr != ')')  &&
			(*aux_ptr != ',')  && (*aux_ptr != '$')  &&
			(*aux_ptr != ' ')  && (*aux_ptr != '(')  &&
			(*aux_ptr != '"')  && (*aux_ptr != '\\') &&
			(*aux_ptr != '\t') && (*aux_ptr != KVI_GLOBAL_VAR_CHAR)
		) {
			aux_ptr++;
		}
		// Interesting char
		switch( *aux_ptr ) {
			case '\0': // Fall-through
			case '\n': // End of the string ('\n' is not escaped); fail here
				if( inString )
					cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("FUNCTION-PARAMS"));
				else
					cmd->setError(KVI_ERROR_UnexpectedEndOfCommand, _i18n_("FUNCTION-PARAMS"));
				return false;
				break;
			case ' ' : // Fall-through
			case '\t': // Whitespace; need to collapse it to one.
				if( !inString ) {
					// Append the last part
					buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
					// Skip the other spaces including escaped ones
					cmd->m_ptr = aux_ptr;
					cmd->skipSpace();
					aux_ptr = cmd->m_ptr;
					if( (*aux_ptr != ')') && (*aux_ptr != ',') )
						buffer.append(' ');
				} else while( (*aux_ptr == ' ') || (*aux_ptr == '\t') ) {
					aux_ptr++;
				}
				break;
			case '"': // Unescaped '"'; enter or exit the string (skip the '"')
				buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = ++aux_ptr;
				inString = !inString;
				break;
			case '(': // Keep track of matching parentheses (if not in string)
				++aux_ptr;
				if( !inString )
					++parLevel;
				break;
			case ')': // If not in string, exit if parenthesis level is 0, otherwise decrease it
				if( !inString ) {
					if( parLevel == 0 ) {
						buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
						cmd->m_ptr = aux_ptr;
						return true;
					} else --parLevel;
				}
				++aux_ptr;
				break;
			case ',' : // End of command buffer; append the last block to the buffer and return
				buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				if( !inString ) {
					cmd->m_ptr = aux_ptr;
					return true;
				}
				cmd->m_ptr = aux_ptr++; // In string; ',' must be included
				break;
			case KVI_GLOBAL_VAR_CHAR: // Variable; append the last block to the buffer and process the var
				buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				if( !processVariable(cmd, buffer) )
					return false;
				aux_ptr = cmd->m_ptr;
				break;
			case '$': // System identifier; append the last block to the buffer and process the identifier
				buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				if( !processIdentifier(cmd, buffer) )
					return false;
				aux_ptr = cmd->m_ptr;
				break;
			case '\\': // Escape character; append the last block to the processed buffer
				buffer.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = ++aux_ptr;
				switch( *aux_ptr ) {
					case '\0':
						if( inString )
							cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("FUNCTION-PARAMS"));
						else
							cmd->setError(KVI_ERROR_UnexpectedEndOfCommand, _i18n_("FUNCTION-PARAMS"));
						return false;
						break;
					case '\n': // Escaped newline
						cmd->m_ptr = ++aux_ptr; // Skip it
						cmd->skipSpace();       // Skip leading spaces
						aux_ptr = cmd->m_ptr;   // Continue
						break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						cmd->m_ptr = ++aux_ptr;
						break;
				}
			break;
		}
	}
	// Never arrive here
	return true;
}

/**
 * Preprocesses the final part of a function parameters block.
 * It is stopped by a newline, a ')', or the end of the buffer.
 * Extracts the command parameters, evaluates it,
 * and appends everything to the buffer.
 * If the return value is true, cmd->m_ptr points to the character
 * after the closing ')'.
 */
bool KviUserParser::processFncFinalPart(KviCommand *cmd, KviStr &buffer)
{
	for( ;; ) {
		if( !processFncSingleParamInternal(cmd, buffer) )
			return false;
		__range_valid((*(cmd->m_ptr) == ',') || (*(cmd->m_ptr) == ')'));
		if( *(cmd->m_ptr) == ',' ) {
			++(cmd->m_ptr);
			buffer.append(',');
		} else {
			++(cmd->m_ptr);
			return true;
		}
	}
	__range_invalid(true);
	return false; // Never here
}

/**
 * ===========================================================================
 *
 * Skip parts of command buffer.
 * Used in 'if', 'while' and 'do' parsing.
 *
 */
bool KviUserParser::skipCommand(KviCommand *cmd)
{
	if( *(cmd->m_ptr) == '{' ) {
		if( !skipCommandBlock(cmd) )
			return false;
		// Here cmd->m_ptr points to the char immediately after the closing brace
	} else {
		if( !skipSingleCommand(cmd) )
			return false;
		// Here cmd->m_ptr points to the char immediately after the separator...
	}
	return true;
}

bool KviUserParser::skipCommandBlock(KviCommand *cmd)
{
	__range_valid(*(cmd->m_ptr) == '{');
	++(cmd->m_ptr);
	cmd->skipWhitespace();
	while( *(cmd->m_ptr) ) {
		switch( *(cmd->m_ptr) ) {
			case '}': ++(cmd->m_ptr);               return true;  break;
			case '{': if( !skipCommandBlock(cmd) )  return false; break;
			default : if( !skipSingleCommand(cmd) ) return false; break;
		}
		cmd->skipWhitespace();
	}
	// We should never reach this point
	cmd->setError(KVI_ERROR_MissingClosingBrace, _i18n_("COMMAND-BLOCK"));
	return false;
}

bool KviUserParser::skipSingleCommand(KviCommand *cmd)
{
	while( *(cmd->m_ptr) && ((*(cmd->m_ptr) == ' ') || (*(cmd->m_ptr) == '\t')) )
		++(cmd->m_ptr);

	// Check for empty command
	if( *(cmd->m_ptr) == ';' ) {
		++(cmd->m_ptr);
		return true;
	}
	if( (*(cmd->m_ptr) == KVI_GLOBAL_VAR_CHAR) || (*(cmd->m_ptr) == '$') )
		return skipLValue(cmd);
	// A comment?
	if( *(cmd->m_ptr) == KVI_COMMENT_BEGIN_CHAR )
		return skipComment(cmd);
	// Extract the command name
	const char *aux = cmd->m_ptr;
	// Command names contain only ASCII chars, digits, _ and .
	while( *(cmd->m_ptr) && (isalnum(*(cmd->m_ptr)) || (*(cmd->m_ptr) ==  '_') || (*(cmd->m_ptr) == '.')) )
		++(cmd->m_ptr);
	KviStr tmp(aux, cmd->m_ptr);

	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "IF")           ) return skipIf(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "WHILE")        ) return skipWhile(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "DO")           ) return skipDo(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "ALIAS")        ) return skipAlias(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "SWITCH")       ) return skipSwitch(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "DIALOG")       ) return skipDialog(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "POPUP")        ) return skipPopup(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "FOREACH")      ) return skipForeach(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "TIMER")        ) return skipTimer(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "EXECV")        ) return skipExecv(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "AWHOIS")       ) return skipAwhois(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "UTOOLBAR")     ) return skipUtoolbar(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "TRY")          ) return skipTry(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "OBJ_SETEVENT") ) return skipObj_setEvent(cmd);
	if( kvi_strEqualNoLocaleCI(tmp.ptr(), "CLASS")        ) return skipClass(cmd);

	// Skip normal command
	if( !skipSwitches(cmd) )
		return false;

	for( ;; ) {
		// Now skip all the uninteresting chars
		while( *(cmd->m_ptr) &&
			(*(cmd->m_ptr) != '\\') && (*(cmd->m_ptr) != '\n') &&
			(*(cmd->m_ptr) != ';')  && (*(cmd->m_ptr) != '$')  &&
			(*(cmd->m_ptr) != KVI_GLOBAL_VAR_CHAR)
		) {
			++(cmd->m_ptr);
		}
		// Interesting char
		switch( *(cmd->m_ptr) ) {
			case '\0':
				return true;
				break;
			case '\n': // Fall-through
			case ';' : // End of command
				++(cmd->m_ptr);
				return true;
				break;
			case '$' :
				if( !skipIdentifier(cmd) )
					return false;
				break;
			case KVI_GLOBAL_VAR_CHAR :
				if( !skipVariable(cmd) )
					return false;
				break;
			case '\\': // Escape character; append the last block to the processed buffer.
				++(cmd->m_ptr);
				switch( *(cmd->m_ptr) ) {
					case '\0':
						return true;
						break; // Escaped nothing...
					default:
						++(cmd->m_ptr);
						break;
				}
			break;
		}
	}
	return true;
}

bool KviUserParser::skipExpressionBody(KviCommand *cmd)
{
	// Skips things like '( <expression like for calc> )'
	__range_valid(*(cmd->m_ptr) == '(');
	++(cmd->m_ptr);

	int parLevel  = 0;
	bool inString = false;
	for( ;; ) {
		// Now skip all the uninteresting chars
		while(*(cmd->m_ptr) &&
			(*(cmd->m_ptr) != '\\') && (*(cmd->m_ptr) != '\n') &&
			(*(cmd->m_ptr) != '(')  && (*(cmd->m_ptr) != ')')  &&
			(*(cmd->m_ptr) != '$')  && (*(cmd->m_ptr) != '"')  &&
			(*(cmd->m_ptr) != KVI_GLOBAL_VAR_CHAR)
		) {
			(cmd->m_ptr)++;
		}
		// Interesting char
		switch( *(cmd->m_ptr) ) {
			case '\0': // Fall-through
			case '\n': // Error
				cmd->setError(KVI_ERROR_UnexpectedEndOfCommand, _i18n_("EXPRESSION-BODY"));
				return false;
				break;
			case '"':
				++(cmd->m_ptr);
				inString = !inString;
				break;
			case '(':
				++(cmd->m_ptr);
				if( !inString )
					++parLevel;
				break;
			case ')' :
				if( !inString ) {
					if( parLevel == 0 ) {
						++(cmd->m_ptr);
						return true;
					} else {
						++(cmd->m_ptr);
						--parLevel;
					}
				} else ++(cmd->m_ptr);
				break;
			case KVI_GLOBAL_VAR_CHAR:
				if( !skipVariable(cmd) )
					return false;
				break;
			case '$':
				if( !skipIdentifier(cmd) )
					return false;
				break;
			case '\\': // Escape character
				++(cmd->m_ptr);
				switch( *(cmd->m_ptr) ) {
					case '\0':
						cmd->setError(KVI_ERROR_UnexpectedEndOfCommand, _i18n_("EXPRESSION-BODY"));
						return false;
					default:
						++(cmd->m_ptr);
						break;
				}
			break;
		}
	}
	// Never here
	return false;
}

bool KviUserParser::skipIdentifier(KviCommand *cmd)
{
	__range_valid(*cmd->m_ptr == '$');
	++(cmd->m_ptr); // Skip the $

	if( isdigit(*(cmd->m_ptr)) || (*(cmd->m_ptr) == '#') )
		cmd->m_ptr ++;
	else {
		if( *(cmd->m_ptr) == '{' ) {
			if( !skipCommand(cmd) )
				return false;
		} else {
			// Normal identifier
			while( *(cmd->m_ptr) && (isalnum(*(cmd->m_ptr)) || (*(cmd->m_ptr) == '.') ||
			      (*(cmd->m_ptr) == '_') || (*(cmd->m_ptr) == ':'))
			) {
				(cmd->m_ptr)++; // Run up to the end (first non alphanumeric)
			}
			if( *(cmd->m_ptr) == '(' ) {
				bool bRecognizedObjScopeOperator = cmd->recognizeObjectScopeOperator();
				cmd->setRecognizeObjectScopeOperator(true);
				if( !skipExpressionBody(cmd) )
					return false;
				cmd->setRecognizeObjectScopeOperator(bRecognizedObjScopeOperator);
			}
		}
	}

	if( cmd->recognizeObjectScopeOperator() ) {
		// Object scope?
		if( *(cmd->m_ptr) == '-' ) {
			if( *(cmd->m_ptr + 1) == '>' ) {
				cmd->m_ptr += 2;
				if( (*(cmd->m_ptr) == '%') || (*(cmd->m_ptr) == '$') )
					return skipObjectScopeRValue(cmd);
				else {
					return cmd->setError(KVI_ERROR_VariableOrIdentifierExpected,
						_i18n_("OBJECT OPERATOR -> (RVALUE)"), cmd->m_ptr
					);
				}
			}
		}
	}
	return true;
}

bool KviUserParser::skipObjectScopeRValue(KviCommand *cmd)
{
	switch( *(cmd->m_ptr) ) {
		case '%': // Object variable
			if( !skipVariable(cmd) )
				return false;
			break;
		case '$':
			// Object function or identifier call
			if( !skipIdentifier(cmd) )
				return false;
			break;
		default:
			// Invalid!
			return false;
		break;
	}
	return true;
}

bool KviUserParser::skipVariable(KviCommand *cmd)
{
	__range_valid(*cmd->m_ptr == KVI_GLOBAL_VAR_CHAR);
	++(cmd->m_ptr);

	while( *(cmd->m_ptr) && (isalnum(*(cmd->m_ptr)) || (*(cmd->m_ptr) == '.') || (*(cmd->m_ptr) == '_')) )
		(cmd->m_ptr)++;

	if( *(cmd->m_ptr) == '[' ) {
		(cmd->m_ptr)++;
		if( !skipVariableKey(cmd) )
			return false;
	}

	// Object scope?
	if( *(cmd->m_ptr) == '-' ) {
		if( *(cmd->m_ptr + 1) == '>' ) {
			cmd->m_ptr += 2;
			if( (*(cmd->m_ptr) == '%') || (*(cmd->m_ptr) == '$') )
				return skipObjectScopeRValue(cmd);
			else
				cmd->setError(KVI_ERROR_VariableOrIdentifierExpected, _i18n_("OBJECT OPERATOR -> (RVALUE)"), cmd->m_ptr);
		}
	}
	return true;
}

bool KviUserParser::skipVariableKey(KviCommand *cmd)
{
	cmd->skipSpace();
	bool inString = false;
	for( ;; ) {
		// First skip all the uninteresting chars
		while(*(cmd->m_ptr) &&
			(*(cmd->m_ptr) != '\n') && (*(cmd->m_ptr) != ']') &&
			(*(cmd->m_ptr) != '$')  && (*(cmd->m_ptr) != '"') &&
			(*(cmd->m_ptr) != '\\') && (*(cmd->m_ptr) != KVI_GLOBAL_VAR_CHAR)
		) {
			(cmd->m_ptr)++;
		}
		// Interesting char
		switch( *(cmd->m_ptr) ) {
			case '\0': // Fall-through
			case '\n': // End of the string ('\n' is not escaped); fail here
				if( inString )
					cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("DICTIONARY KEY"));
				else
					cmd->setError(KVI_ERROR_UnexpectedEndOfCommand, _i18n_("DICTIONARY KEY"));
				return false;
				break;
			case '"':
				cmd->m_ptr++;
				inString = !inString;
				break;
			case ']':
				cmd->m_ptr++;
				if( !inString )
					return true;
					break;
			case KVI_GLOBAL_VAR_CHAR:
				if( !skipVariable(cmd) )
					return false;
					break;
			case '$':
				if( !skipIdentifier(cmd) )
					return false;
					break;
			case '\\': // Escape character; append the last block to the processed buffer
				cmd->m_ptr++;
				switch( *(cmd->m_ptr) ) {
					case '\0':
						if( inString )
							cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("DICTIONARY KEY"));
						else
							cmd->setError(KVI_ERROR_UnexpectedEndOfCommand, _i18n_("DICTIONARY KEY"));
						return false;
						break;
					case '\n':  // Escaped newline
						cmd->m_ptr++;     // Skip it
						cmd->skipSpace(); // Skip leading spaces
						break;
					default:
						cmd->m_ptr++;
						break;
				}
			break;
		}
	}
	// Never arrive here
	return true;
}

bool KviUserParser::skipSwitches(KviCommand *cmd)
{
	cmd->skipSpace();
	while( *(cmd->m_ptr) == '-' ) {
		if( !skipSingleSwitch(cmd) )
			return false;
		cmd->skipSpace();
	}
	return true;
}

bool KviUserParser::skipSingleSwitch(KviCommand *cmd)
{
	__range_valid(*(cmd->m_ptr) == '-');
	++(cmd->m_ptr); // Skip the - char

	// Now a LETTER is required
	if( (!(*(cmd->m_ptr))) || (!isalpha(*(cmd->m_ptr))) ) {
		cmd->setError(KVI_ERROR_InvalidOption, _i18n_("SWITCH PARSING AFTER '-' DELIMITER"), cmd->m_ptr);
		return false;
	}
	++(cmd->m_ptr); // Skip the switch letter
	cmd->skipSpace();

	if( *(cmd->m_ptr) == '=' ) {
		// A token follows the switch
		++(cmd->m_ptr);
		cmd->skipSpace();
		const char *aux_ptr = cmd->m_ptr;
		bool bInString = false;
		for( ;; ) {
			// Now skip all the uninteresting chars
			while( *aux_ptr &&
				(*aux_ptr != '\\') && (*aux_ptr != '\n') &&
				(*aux_ptr != ' ')  && (*aux_ptr != '\t') &&
				(*aux_ptr != ';')  && (*aux_ptr != '$')  &&
				(*aux_ptr != '\"') && (*aux_ptr != KVI_GLOBAL_VAR_CHAR)
			) {
				aux_ptr++;
			}
			// Interesting char
			switch( *aux_ptr ) {
				case ' ' : // Fall-through
				case '\t': // Fall-through
				case ';' :
					if( !bInString ) { // End of the token; append the last block to the buffer and return
						cmd->m_ptr = aux_ptr;
						return true;
					} else while( (*aux_ptr == ' ') || (*aux_ptr == '\t') || (*aux_ptr == ';') ) {
						aux_ptr++; // In string; must skip it
					}
					break;
				case '\n': // Fall-through
				case '\0': // End of command buffer; append the last block to the buffer and return
					if( bInString ) {
						// Unescaped newline or end of data
						return cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("SWITCH PARSING: PARAMETER EXTRACTION"));
					} // else: it is the end of the command
					cmd->m_ptr = aux_ptr;
					return true;
					break;
				case '\"': // Beginning or end of a string
					bInString = !bInString;
					cmd->m_ptr = ++aux_ptr;
					break;
				case KVI_GLOBAL_VAR_CHAR: // Variable; append the last block to the buffer and process the var
					cmd->m_ptr = aux_ptr;
					if( !skipVariable(cmd) )
						return false;
					aux_ptr = cmd->m_ptr;
					break;
				case '$': // System identifier; append the last block to the buffer and process the identifier
					cmd->m_ptr = aux_ptr;
					if( !skipIdentifier(cmd) )
						return false;
					aux_ptr = cmd->m_ptr;
					break;
				case '\\': // Escape character; append the last block to the processed buffer
					cmd->m_ptr = ++aux_ptr;
					switch( *aux_ptr ) {
						case '\0':
							if( bInString ) {
								return cmd->setError(KVI_ERROR_UnexpectedEndInString,
									_i18n_("SWITCH PARSING: PARAMETER EXTRACTION")
								);
							}
							return true;
							break;
						case '\n': // Escaped newline
							cmd->m_ptr = ++aux_ptr; // Skip it
							cmd->skipSpace();       // Skip leading spaces
							aux_ptr = cmd->m_ptr;   // Continue
							break;
						default: // Must be copied to the buffer without modification
							cmd->m_ptr = ++aux_ptr;
							break;
					}
				break;
			}
		}
	}
	// Not '='
	return true;
}

bool KviUserParser::skipComment(KviCommand *cmd)
{
	__range_valid(*(cmd->m_ptr) == KVI_COMMENT_BEGIN_CHAR);

	while( *(cmd->m_ptr) && (*(cmd->m_ptr) != ';') && (*(cmd->m_ptr) != '\n') )
		++(cmd->m_ptr);
	if( *(cmd->m_ptr) )
		++(cmd->m_ptr); // Skip the terminator
	return true;
}

#include "m_kvi_userparser.moc"
