// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptTreeView"

#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_script_listview.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_treeview.h"
#include "kvi_script_treeview_item.h"

// TODO: Set column pixmap

/*
	@class: treeview
	@short:
		A tree view widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $addColumn(&lt;text&gt;)
		Adds a column with &lt;text&gt; to the treeview
		You need at least one column for this widget to work

		!fn: $removeColumn(&lt;index&gt;)
		Removes the column with zero-based index &lt;index&gt;

		!fn: $setColumnText(&lt;index&gt;, &lt;text&gt;)
		Sets a column's text to &lt;text&gt;.

		!fn: $setRootIsDecorated(&lt;boolean&gt;)
		Enables or disables the plus signs beside the items in this treeview.
		This is enabled by default.

		!fn: $setShowSortIndicator(&lt;boolean&gt;)
		Shows or hides the sort indicator

		!fn: $setAllColumnsShowFocus(&lt;boolean&gt;)
		Enables the focus visualisation for the whole items (instead of the first column only)

		!fn: $firstChild()
		Returns the id of the first [class]treeviewitem[/class] in this treeview.
		If there is no first child, this function returns an empty string.

	@events:
		!ev: OnSelectionChanged($1 = &lt;itemId&gt;)
		Triggered when the user selects one of the items of the treeview.<br>
		&lt;itemId&gt; contains the object id of the treeviewitem that has been selected or an empty string if the change
		caused to unset any current item.<br>
		The default event handler emits the [classsignal:treeview]selectionChanged[/classsignal] signal.<br>
		If you provide a new handler for this event, and still want the signal
		to be emitted, you must emit it by yourself:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:treeviewitem]selectionChanged[/classsignal], $1)
		</example>

		!ev: OnItemClicked($1 = &lt;itemId&gt;)
		Triggered when the user clicks on the treeview.<br>
		If the user clicks on an item, its id is passed as the first parameter.<br>
		The default event handler emits the [classsignal:treeview]itemClicked[/classsignal] signal.<br>
		If you provide a new handler for this event, and still want the signal
		to be emitted, you must emit it by yourself:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:treeviewitem]itemClicked[/classsignal], $1)
		</example>

		!ev: OnItemDoubleClicked($1 = &lt;itemId&gt;)
		Triggered when the user double-clicks on the treeview.<br>
		If the user double-clicks on an item, its id is passed as the first parameter.<br>
		The default event handler emits the [classsignal:treeview]itemDoubleClicked[/classsignal] signal.<br>
		If you provide a new handler for this event, and still want the signal
		to be emitted, you must emit it by yourself:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:treeviewitem]itemDoubleClicked[/classsignal], $1)
		</example>

	@signals:
		!sg: selectionChanged($1 = &lt;itemId&gt;)
		Emitted when the user selects one of the treeview items.<br>
		&lt;itemId&gt; contains the object id of the treeviewitem that has been selected or an empty string if the change
		caused the treeview to have no current item.<br>
		<b>This signal is emitted by the default [classevent:treeview]OnSelectionChanged[/classevent] event handler!<br>
		If you define a new event handler, and still want the signal to be emitted,
		you have to emit it by yourself!</b>

		!sg: itemClicked($1 = &lt;itemId&gt;)
		Emitted when the user clicks on one of the treeview items.<br>
		&lt;itemId&gt; contains the object id of the treeviewitem that has been clicked or an empty string if the user
		clicked on the empty part of the treeview.<br>
		<b>This signal is emitted by the default [classevent:treeview]OnItemClicked[/classevent] event handler!<br>
		If you define a new event handler, and still want the signal to be emitted,
		you have to emit it by yourself!</b>

		!sg: itemDoubleClicked($1 = &lt;itemId&gt;)
		Emitted when the user double-clicks on one of the treeview items.<br>
		&lt;itemId&gt; contains the object id of the treeviewitem that has been double-clicked or an empty string if the user
		clicked on the empty part of the treeview.<br>
		<b>This signal is emitted by the default [classevent:treeview]OnItemDoubleClicked[/classevent] event handler!<br>
		If you define a new event handler, and still want the signal to be emitted,
		you have to emit it by yourself!</b>

	@description:
		A tree view is a widget that is able to display items in a tree like manner.
		You will have to use also the [class]treeviewitem[/class] that reppresents
		a child item of this widget.<br>
		Basically you create a treeview widget and add a few child treeviewitem
		objects: the treeviewitem objects are arranged in a tree like manner:
		if you create them as children of the treeview, they will be
		appearing as "toplevel" in the treeview; if you create them as children
		of other treeviewitem objects, they will appear as children in the treeview too.
		To remove an item you have just to destroy it. All the items will be destroyed
		when this treeview object is destroyed.<br>
		The object class emits the OnSelectionChanged event (and signal in the default handler),
		when the user changes the currently selected item or unselects any item.<br>
		You can access the tree of items by using the function $firstChild() that returns the
		root item.

	@examples:
		<example>
			# Really simple one

			%tree = [fnc]$new[/fnc](treeview, [fnc]$root[/fnc], mytreeview)
			%tree->$addColumn("Friends")
			%tree->$addColumn("Age")
			%tree->$setRootIsDecorated(1)


			# Create the toplevel items

			%men = [fnc]$new[/fnc](treeviewitem, %tree, item, Men)
			%women = [fnc]$new[/fnc](treeviewitem, %tree, item, Women)

			# Create some child items
			%it = $new(treeviewitem, %men, it, Pippo, 25)
			%it = $new(treeviewitem, %men, it, Pluto, 11)
			%it = $new(treeviewitem, %men, it, Paperino, 45)
			%it = $new(treeviewitem, %women, it, Minnie, 12)
			%it = $new(treeviewitem, %women, it, Lucy, 999)
			# We want the men subtree open
			%men->$setOpen(1)

			[cmd]obj_setevent[/cmd](%tree, OnClose)
			{
				[cmd]destroy[/cmd] [fnc]$this[/fnc]
			}

			[cmd]obj_setevent[/cmd](%tree, OnSelectionChanged)
			{
				[cmd]if[/cmd]("$1" != "")[cmd]echo[/cmd] "Selection changed: item $1->$text(0)"
				else echo "No selection"
			}

			%tree->$show();

			# ...
			# ... navigate through the items
			# ...
			# this could be implemented gracefully with a recursive function...
			# (left as programming exercise to the scripter :)

			%it = %tree->$firstChild();
			while("%it" != "")
			{
				echo "Navigation: item: %it->$text(0)"
				%child = %it->$firstChild()
				if("%child" != "")%it = %child
				else %it = %it->$nextSibling()
			}

		</example>
	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * TREEVIEW class
 */
void KviScriptTreeView::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("addColumn",     (scriptObjectFunction) &KviScriptTreeView::builtinFunction_ADDCOLUMN);
	d->addBuiltinFunction("firstChild",    (scriptObjectFunction) &KviScriptTreeView::builtinFunction_FIRSTCHILD);
	d->addBuiltinFunction("removeColumn",  (scriptObjectFunction) &KviScriptTreeView::builtinFunction_REMOVECOLUMN);
	d->addBuiltinFunction("setColumnText", (scriptObjectFunction) &KviScriptTreeView::builtinFunction_SETCOLUMNTEXT);
	d->addBuiltinFunction("setRootIsDecorated",
		(scriptObjectFunction) &KviScriptTreeView::builtinFunction_SETROOTISDECORATED
	);
	d->addBuiltinFunction("setShowSortIndicator",
		(scriptObjectFunction) &KviScriptTreeView::builtinFunction_SETSHOWSORTINDICATOR
	);
	d->addBuiltinFunction("setAllColumnsShowFocus",
		(scriptObjectFunction) &KviScriptTreeView::builtinFunction_SETALLCOLUMNSSHOWFOCUS
	);

	KviScriptEventStruct *s = new KviScriptEventStruct();
	s->szName   = "OnSelectionChanged";
	s->szBuffer = "$this->$emit(selectionChanged, $1)";
	d->addDefaultEvent(s);
	s = new KviScriptEventStruct();
	s->szName   = "OnItemClicked";
	s->szBuffer = "$this->$emit(itemClicked, $1)";
	d->addDefaultEvent(s);
	s = new KviScriptEventStruct();
	s->szName   = "OnItemDoubleClicked";
	s->szBuffer = "$this->$emit(itemDoubleClicked, $1)";
	d->addDefaultEvent(s);
}

KviScriptTreeView::KviScriptTreeView(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptTreeView::~KviScriptTreeView()
{
	// Nothing here
}

bool KviScriptTreeView::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new KviListView(((KviScriptWidget *) parent())->m_pWidget, name());
		}
	}

	if( !m_pWidget )
		m_pWidget = new KviListView(0, name());
	((KviListView *) m_pWidget)->setRootIsDecorated(true);
	((KviListView *) m_pWidget)->setSelectionMode(QListView::Single);
	connect(m_pWidget, SIGNAL(selectionChanged(KviListViewItem *)), this, SLOT(selectionChanged(KviListViewItem *)));
	connect(m_pWidget, SIGNAL(clicked(KviListViewItem *)),          this, SLOT(itemClicked(KviListViewItem *)));
	connect(m_pWidget, SIGNAL(doubleClicked(KviListViewItem *)),    this, SLOT(itemDoubleClicked(KviListViewItem *)));
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	return true;
}

void KviScriptTreeView::selectionChanged(KviListViewItem *it)
{
	KviStr parms;
	if( it )
		parms.sprintf("%s", ((KviScriptListViewItem *) it)->m_pScriptObjectClass->id());
	triggerEvent("OnSelectionChanged", parms);
}

void KviScriptTreeView::itemClicked(KviListViewItem *it)
{
	KviStr parms;
	if( it )
		parms.sprintf("%s", ((KviScriptListViewItem *) it)->m_pScriptObjectClass->id());
	triggerEvent("OnItemClicked", parms);
}

void KviScriptTreeView::itemDoubleClicked(KviListViewItem *it)
{
	KviStr parms;
	if( it )
		parms.sprintf("%s", ((KviScriptListViewItem *) it)->m_pScriptObjectClass->id());
	triggerEvent("OnItemDoubleClicked", parms);
}

int KviScriptTreeView::builtinFunction_FIRSTCHILD(QPtrList<KviStr> *params, KviStr &buffer)
{
	KviScriptListViewItem *it = (KviScriptListViewItem *) ((KviListView *) m_pWidget)->firstChild();
	if( it )
		buffer.append(it->m_pScriptObjectClass->id());
	return KVI_ERROR_Success;
}

int KviScriptTreeView::builtinFunction_ADDCOLUMN(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			((KviListView *) m_pWidget)->addColumn(pS->ptr());
		else
			((KviListView *) m_pWidget)->addColumn(_i18n_("unnamed"));
	} else return KVI_ERROR_MissingParameter;
	return KVI_ERROR_Success;
}

int KviScriptTreeView::builtinFunction_REMOVECOLUMN(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int idx = pS->toInt(&bOk);
			if( bOk ) {
				((KviListView *) m_pWidget)->removeColumn(idx);
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTreeView::builtinFunction_SETCOLUMNTEXT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int idx = pS->toInt(&bOk);
			if( bOk ) {
				pS = params->next();
				((KviListView *) m_pWidget)->setColumnText(idx, pS ? pS->ptr() : "");
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTreeView::builtinFunction_SETROOTISDECORATED(QPtrList<KviStr> *params, KviStr &buffer)
{
	bool rd = true;
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			rd = !kvi_strEqualCI(pS->ptr(), "0");
	}
	((KviListView *) m_pWidget)->setRootIsDecorated(rd);
	return KVI_ERROR_Success;
}

int KviScriptTreeView::builtinFunction_SETALLCOLUMNSSHOWFOCUS(QPtrList<KviStr> *params, KviStr &)
{
	bool rd = true;
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			rd = !kvi_strEqualCI(pS->ptr(), "0");
	}
	((KviListView *) m_pWidget)->setAllColumnsShowFocus(rd);
	return KVI_ERROR_Success;
}

int KviScriptTreeView::builtinFunction_SETSHOWSORTINDICATOR(QPtrList<KviStr> *params, KviStr &buffer)
{
	bool rd = true;
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			rd = !kvi_strEqualCI(pS->ptr(), "0");
	}
	((KviListView *) m_pWidget)->setShowSortIndicator(rd);
	return KVI_ERROR_Success;
}

#include "m_kvi_script_treeview.moc"
