// =============================================================================
//
//      --- kvi_irc_proxy.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviIrcProxy"

#include "kvi_config.h"
#include "kvi_debug.h"
#include "kvi_irc_proxy.h"

KviIrcProxyManager::KviIrcProxyManager()
{
	m_pProxyList = new QPtrList<KviIrcProxy>;
	m_pProxyList->setAutoDelete(true);
	m_pCurrentProxy = 0;
}

KviIrcProxyManager::~KviIrcProxyManager()
{
	clear();
	if( m_pProxyList ) {
		delete m_pProxyList;
		m_pProxyList = 0;
	}
}

void KviIrcProxyManager::clear()
{
	while( !m_pProxyList->isEmpty() ) m_pProxyList->removeLast();
	m_pCurrentProxy = 0;
}

void KviIrcProxyManager::save(KviConfig *cfg)
{
	__range_valid(cfg);
	cfg->setGroup("IrcProxyManager");
	cfg->writeEntry("Hosts", m_pProxyList->count());
	for( uint i = 0; i < m_pProxyList->count(); i++ ) {
		KviIrcProxy *prx = m_pProxyList->at(i);
		__range_valid(prx);
		KviStr szNum(KviStr::Format,"Host_%d", i);
		KviIpAddresses::Iterator it = prx->addresses.begin();
		QString addr("");
		for( ; it != prx->addresses.end() ; ) {
			addr += (*it).toString();
			++it;
			if( it != prx->addresses.end() )
				addr += ',';
		}
		KviStr szPrx(KviStr::Format,"%s:%s:%s:%s:%s",
			prx->szHost.ptr(),
			addr.ascii(),
			prx->szPort.ptr(),
			prx->szUsername.ptr(),
			prx->szPassword.ptr()
		);
		cfg->writeEntry(szNum.ptr(), szPrx.ptr());
		if( prx == m_pCurrentProxy ) cfg->writeEntry("Current_Host", i);
	}
	cfg->sync();
}

void KviIrcProxyManager::load(KviConfig *cfg)
{
	clear();
	__range_valid(cfg);
	cfg->setGroup("IrcProxyManager");
	uint count = cfg->readUIntEntry("Hosts", 0);
	uint curr  = cfg->readUIntEntry("Current_Host", 0);
	m_pCurrentProxy = 0;
	KviIrcProxy *curPrx = 0;
	for( uint i = 0; i < count; i++ ) {
		KviIrcProxy *prx = new KviIrcProxy();
		__range_valid(prx);
		KviStr szNum(KviStr::Format,"Host_%d", i);
		KviStr szPrx = cfg->readEntry(szNum.ptr(), "proxy.localhost:127.0.0.1:1080::");

		m_pProxyList->append(prx);
		if( i == curr ) curPrx = prx;

		prx->szHost = "proxy.localhost";
		prx->szPort = "1080";
		prx->addresses.clear();

		const char *ptr = szPrx.ptr();
		if( (!*ptr) || (*ptr == ':') ) continue;
		ptr = kvi_extractUpTo(prx->szHost, ptr, ':');
		if(  *ptr ) ptr++;
		if( !*ptr ) continue;
		//
		// Create a list of IP addresses, separated by commas in the input string
		//
		QString addrs(ptr);
		addrs = addrs.mid(0, addrs.find(':'));
		ptr = (ptr + addrs.length());
		QStringList list = QStringList::split(",", addrs);
		QStringList::Iterator it = list.begin();
		for( ; it != list.end(); ++it ) {
			QString s = *it;
			QHostAddress addr;
			if( addr.setAddress(s) )
				prx->addresses.append(addr);
		}
		if(  *ptr ) ptr++;
		if( !*ptr ) continue;
		ptr = kvi_extractUpTo(prx->szPort, ptr, ':');
		if(  *ptr ) ptr++;
		if( !*ptr ) continue;
		ptr = kvi_extractUpTo(prx->szUsername, ptr, ':');
		if(  *ptr ) ptr++;
		if( !*ptr ) continue;
		kvi_extractUpTo(prx->szPassword, ptr, ':');
	}
	if( curPrx )
		m_pCurrentProxy = curPrx;
	else {
		if( m_pProxyList->isEmpty() )
			m_pCurrentProxy = 0;
		else
			m_pCurrentProxy = m_pProxyList->first();
	}
}

KviIrcProxy *KviIrcProxyManager::currentProxy()
{
	if( m_pCurrentProxy ) return m_pCurrentProxy;
	if( !m_pProxyList->isEmpty() ) {
		m_pCurrentProxy = m_pProxyList->first();
		return m_pCurrentProxy;
	}
	return 0;
}

KviIrcProxy *KviIrcProxyManager::getProxyByName(const char *szName)
{
	for( KviIrcProxy *p = m_pProxyList->first(); p; p = m_pProxyList->next() ) {
		if( kvi_strEqualCI(p->szHost.ptr(), szName) ) return p;
	}
	return 0;
}

bool KviIrcProxyManager::setCurrentProxy(KviIrcProxy *prx)
{
	__range_valid(m_pProxyList->findRef(prx) != -1);
	if( m_pProxyList->findRef(prx) != -1 ) {
		m_pCurrentProxy = prx;
		return true;
	}
	return false;
}

void KviIrcProxyManager::copyFrom(KviIrcProxyManager *m)
{
	clear();
	for( KviIrcProxy *p = m->m_pProxyList->first(); p; p = m->m_pProxyList->next() ) {
		KviIrcProxy *prx = new KviIrcProxy;
		prx->szHost      = p->szHost;
		prx->addresses   = p->addresses;
		prx->szUsername  = p->szUsername;
		prx->szPassword  = p->szPassword;
		prx->szPort      = p->szPort;
		m_pProxyList->append(prx);
		if( p == m->m_pCurrentProxy ) m_pCurrentProxy = prx;
	}
}

void KviIrcProxyManager::appendProxy(KviIrcProxy *ptr)
{
	__range_valid(ptr);
	m_pProxyList->append(ptr);
	m_pCurrentProxy = ptr;
}

bool KviIrcProxyManager::removeProxy(KviIrcProxy *ptr)
{
	__range_valid(ptr);
	bool bWasHere = m_pProxyList->removeRef(ptr);
	__range_valid(bWasHere);
	if( ptr == m_pCurrentProxy ) {
		if( m_pProxyList->isEmpty() )
			m_pCurrentProxy = 0;
		else
			m_pCurrentProxy = m_pProxyList->first();
	}
	return bWasHere;
}

void KviIrcProxyManager::updateProxyIp(const char *szName, KviIpAddresses list)
{
	KviIrcProxy *prx = getProxyByName(szName);
	if( !prx ) return;
	prx->addresses = list;
}

QPtrList<KviIrcProxy> *KviIrcProxyManager::proxyList()
{
	return m_pProxyList;
}
