//LabPlot : PlotQWT3D.cc

#include <iostream>
#include <qregexp.h>
#include <kmessagebox.h>
#include "PlotQWT3D.h"
#include "Label.h"
#ifdef HAVE_GL
#include "qwt3d_function.h"
#include "qwt3d_io.h"
#endif

PlotQWT3D::PlotQWT3D(Worksheet *p)
	: Plot(p)
#ifdef HAVE_GL
	, Qwt3D::SurfacePlot((QWidget *)p) 
#endif
{

	// order see qwt3d_types.h
	axis[0].setLabel(new Label(i18n("x-Axis")));
	axis[1].setLabel(new Label(i18n("y-Axis")));
	axis[2].setLabel(new Label(i18n("z-Axis")));
	axis[3].setLabel(new Label(i18n("x2-Axis")));
	axis[4].setLabel(new Label(i18n("x3-Axis")));
	axis[5].setLabel(new Label(i18n("x4-Axis")));
	axis[6].setLabel(new Label(i18n("y4-Axis")));
	axis[7].setLabel(new Label(i18n("y3-Axis")));
	axis[8].setLabel(new Label(i18n("y2-Axis")));
	axis[9].setLabel(new Label(i18n("z2-Axis")));
	axis[10].setLabel(new Label(i18n("z4-Axis")));
	axis[11].setLabel(new Label(i18n("z3-Axis")));

	setGraphBackground(Qt::black);
#ifdef HAVE_GL
	plotstyle = Qwt3D::FILLEDMESH;
	coordinatestyle = Qwt3D::BOX;
	floorstyle = Qwt3D::NOFLOOR;

	cv.clear();
	Qwt3D::RGBA rgb;

	for(int i=0;i<255;i++) {
		rgb.a = 1;
		rgb.r = i/255.0;
		rgb.g = 40.0/255.0;
		rgb.b = 1.0-i/255.0;
		cv.push_back(rgb);
	}

	// TODO : initial values
	setRotation(30,0,15);
	setShift(0.15,0,0);
	setZoom(0.9);
	
	for (int i=0;i<24;i++) {
		gridenabled[i] = FALSE;
	}
	
	Plot::legend.setPosition(.9,.05);
	aspect_ratio=TRUE;

	show();
#endif
}

QStringList PlotQWT3D::Info() {
	QStringList s;
	s<<"QWT";
	
	s<<QString::number(position.X())+QString(" , ")+QString::number(position.Y());
	s<<QString::number(Plot::size.X())+QString(" X ")+QString::number(Plot::size.Y());
	if (transparent)
		s<<QString("yes");
	else
		s<<QString("no");
	s<<bgcolor.name();
	s<<gbgcolor.name();
	
	return s;
}

void PlotQWT3D::Export(QString fn,QString format, int w, int h) {
	draw(0,w,h);
#ifdef HAVE_GL
	Qwt3D::Plot3D::save(fn,format);
#endif
}

void PlotQWT3D::draw(QPainter *p, int w, int h) {
#ifdef HAVE_GL
	kdDebug()<<"PlotQWT3D::draw()"<<endl;
	resize(w,h);
	
	kdDebug()<<"PlotStyle = "<<plotstyle<<endl;
	Qwt3D::Plot3D::setPlotStyle(plotstyle);
	// TODO : Enrichments : Qwt3D::Plot3D::setPlotStyle(Bar(0.007,0.5));
	 
	// TODO : set style of legend too
	kdDebug()<<"CoordinateStyle = "<<coordinatestyle<<endl;
	Qwt3D::Plot3D::setCoordinateStyle(coordinatestyle);
	kdDebug()<<"FloorStyle = "<<floorstyle<<endl;
	Qwt3D::SurfacePlot::setFloorStyle(floorstyle);

	QColor c = Background();
	setBackgroundColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));
	c = graphBackground();
	setMeshColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));
	

	// remove all html tags
	QString t = title->title();
	QRegExp rx("<.*>");
	rx.setMinimal(TRUE);
	t.replace(rx,"");
	t.replace(QRegExp(QString("\n")),"");	// replace newlines
	//kdDebug()<<"title = "<<t<<endl;	
	setTitle(t);
	setTitlePosition(1.0-title->Y(),title->X());		// TODO : Qwt3D::Anchor
	QFont f = title->font();
	setTitleFont(f.family(),f.pointSize(),f.weight(),f.italic());
	c = title->color();
	setTitleColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));  // TODO : a=?
	// TODO : boxed, rotation
	
	// set data color
	Qwt3D::StandardColor *col = new Qwt3D::StandardColor(this);
	col->setColorVector(cv);
	setDataColor(col);
	
	// legend
	// TODO : set isolines of legend
	showColorLegend(Plot::legend.enabled());
	Qwt3D::ColorLegend *l = Qwt3D::Plot3D::legend();
	// set legend position
	double lx = Plot::legend.X(), ly = 1.0-Plot::legend.Y();
//	kdDebug()<<"Legend X/Y "<<lx<<'/'<<ly<<endl; 
	l->setRelPosition(Qwt3D::Tuple(lx,ly),Qwt3D::Tuple(lx+.05,ly-0.3));
	
	// TODO : l->setTitleString(QString);
	// TODO : l->setTitleFont(s.a.);
	// TODO : border
	// TODO : Qwt3D::ColorLegend::LeftRight looks strange
//	l->setOrientation(Qwt3D::ColorLegend::BottomTop,Qwt3D::ColorLegend::Top);
	
	
	drawCurves(p,w,h);

	updateData();
	updateGL();
#endif
}

void PlotQWT3D::drawCurves(QPainter *p, int w, int h) {
#ifdef HAVE_GL
	for (unsigned int i=0; i < graphlist->getNumber() ; i++) {
		if(graphlist->getGraph(i)->isShown() == FALSE)
			continue;
		
		Style style;
		Symbol symbol;
		GRAPHType s = graphlist->getStruct(i);
		if(s == GRAPHM) {
			GraphM *g = graphlist->getGraphM(i);
			LRange r = g->Range(2);

			double *data = g->Data();
			int rows = g->NY(), cols = g->NX();
	
			double** sdata = new double* [cols] ;
	
			for ( int i = 0; i < cols; ++i)
				sdata[i] = new double [rows];
	
			// copy data
			for(int i=0;i<rows;i++) {
				for(int j=0;j<cols;j++) {
					sdata[j][i] = data[j+i*cols];
				}
			}
			// load data into surface plot
			loadFromData(sdata,cols,rows,0,cols,0,rows);
		
			// (x,y scale same as z range)
			// TODO : what about other graphs ?
			//double zscale=rows/(r.rMax()-r.rMin());
			
			if(aspect_ratio)
				setScale(1,cols/rows,rows/(r.rMax()-r.rMin()));
			else
				setScale(1,1,rows/(r.rMax()-r.rMin()));
		}
		else if(s == GRAPH3D) {
			Graph3D *g = graphlist->getGraph3D(i);
//			LRange r = g->Range(2);

			Point3D *data = g->Data();
			int rows = g->NY(), cols = g->NX();
			kdDebug()<<"rows/cols = "<<rows<<' '<<cols<<endl;
			
			Qwt3D::Triple **sdata = new Qwt3D::Triple* [cols];
			for ( int i = 0; i < cols; ++i)
				sdata[i] = new  Qwt3D::Triple[rows];
			
			// copy data
			for(int i=0;i<rows;i++) {
				for(int j=0;j<cols;j++) {
					sdata[j][i] = Qwt3D::Triple(data[j+i*cols].X(),data[j+i*cols].Y(),data[j+i*cols].Z());
				}
			}
			
			// load data into surface plot
			loadFromData(sdata,cols,rows);
			
			//TODO : convert for showing surface, etc. ??? 
			Qwt3D::Plot3D::setPlotStyle(Qwt3D::SPOINTS);
		}
		else
			continue;
		
		
		// use color of first axis
		QColor c = axis[0].TicsColor();
		coordinates()->setAxesColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));
		
		// TODO : grid lines
		int sum = Qwt3D::NOSIDEGRID;
		if(gridenabled[4] || gridenabled[20])
			sum |= Qwt3D::SLEFT;
		if(gridenabled[8] || gridenabled[16])
			sum |= Qwt3D::SRIGHT;
		if(gridenabled[6] || gridenabled[14])
			sum |= Qwt3D::CEIL;
		if(gridenabled[18] || gridenabled[2])
			sum |= Qwt3D::FLOOR;
		if(gridenabled[0] || gridenabled[10])
			sum |= Qwt3D::FRONT;
		if(gridenabled[12] || gridenabled[22])
			sum |= Qwt3D::BACK;
 		
		coordinates()->setGridLines(sum, sum, sum );
		
		// TODO : scale data too!
		// use scale from first x,y,z axis
		Qwt3D::SCALETYPE scale = Qwt3D::LINEARSCALE;
		switch(axis[0].Scale()) {
			case LINEAR : break;
			case LOG10 : scale = Qwt3D::LOG10SCALE; break;
			case LOG2 : case LN : case SQRT: break;
		}
		coordinates()->axes[Qwt3D::X1].setScale(scale);
		coordinates()->axes[Qwt3D::X2].setScale(scale);
		coordinates()->axes[Qwt3D::X3].setScale(scale);
		coordinates()->axes[Qwt3D::X4].setScale(scale);
		switch(axis[1].Scale()) {
			case LINEAR : scale = Qwt3D::LINEARSCALE; break;
			case LOG10 : scale = Qwt3D::LOG10SCALE; break;
			case LOG2 : case LN : case SQRT: break;
		}
		coordinates()->axes[Qwt3D::Y1].setScale(scale);
		coordinates()->axes[Qwt3D::Y2].setScale(scale);
		coordinates()->axes[Qwt3D::Y3].setScale(scale);
		coordinates()->axes[Qwt3D::Y4].setScale(scale);
		switch(axis[2].Scale()) {
			case LINEAR : scale = Qwt3D::LINEARSCALE; break;
			case LOG10 : scale = Qwt3D::LOG10SCALE; break;
			case LOG2 : case LN : case SQRT: break;
		}
		coordinates()->axes[Qwt3D::Z1].setScale(scale);
		coordinates()->axes[Qwt3D::Z2].setScale(scale);
		coordinates()->axes[Qwt3D::Z3].setScale(scale);
		coordinates()->axes[Qwt3D::Z4].setScale(scale);

		for(int i=0;i<12;i++) {
			Label *l = axis[i].label();
			QString t = l->title();
			QRegExp rx("<.*>");
			rx.setMinimal(TRUE);
			t.replace(rx,"");
			t.replace(QRegExp(QString("\n")),"");	// replace newlines

			// TODO : more axes settings
			// also : axes[Z1].setLabelString(QChar(0x38f);  // Omega - see http://www.unicode.org/charts/
			// dont use Qwt3D::Axis axes !
			
			coordinates()->axes[i].setNumbers(axis[i].enabled());

			coordinates()->axes[i].setLabelString(t);
			coordinates()->axes[i].setLabelFont(l->font());
			c = l->color();
			coordinates()->axes[i].setLabelColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));

			coordinates()->axes[i].setMajors(axis[i].MajorTics());
			coordinates()->axes[i].setMinors(axis[i].MinorTics());
			coordinates()->axes[i].setNumberFont(axis[i].TicsFont());
			c = axis[i].TicsLabelColor();
			coordinates()->axes[i].setNumberColor(Qwt3D::RGBA(c.red()/255.0,c.green()/255.0,c.blue()/255.0));
			
		}
	}
#endif
}
