/* $Id: drv_generic_graphic.c,v 1.34 2006/08/14 19:24:22 harbaum Exp $
 *
 * generic driver helper for graphic displays
 *
 * Copyright (C) 1999, 2000 Michael Reinelt <reinelt@eunet.at>
 * Copyright (C) 2004 The LCD4Linux Team <lcd4linux-devel@users.sourceforge.net>
 *
 * This file is part of LCD4Linux.
 *
 * LCD4Linux is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * LCD4Linux is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * $Log: drv_generic_graphic.c,v $
 * Revision 1.34  2006/08/14 19:24:22  harbaum
 * Umlaut support, added KVV HTTP-User-Agent
 *
 * Revision 1.33  2006/08/13 09:53:10  reinelt
 * dynamic properties added (used by 'style' of text widget)
 *
 * Revision 1.32  2006/08/13 06:46:51  reinelt
 * T6963 soft-timing & enhancements; indent
 *
 * Revision 1.31  2006/08/09 17:25:34  harbaum
 * Better bar color support and new bold font
 *
 * Revision 1.30  2006/08/08 20:16:29  harbaum
 * Added "extracolor" (used for e.g. bar border) and RGB support for LEDMATRIX
 *
 * Revision 1.29  2006/07/31 03:48:09  reinelt
 * preparations for scrolling
 *
 * Revision 1.28  2006/06/20 08:50:58  reinelt
 * widget_image linker error hopefully finally fixed
 *
 * Revision 1.27  2006/04/09 17:46:14  reinelt
 * vertical bar patch fro graphic displays by Ronald Hopfer
 *
 * Revision 1.26  2006/04/09 14:17:50  reinelt
 * autoconf/library fixes, image and graphic display inversion
 *
 * Revision 1.25  2006/02/27 07:53:52  reinelt
 * some more graphic issues fixed
 *
 * Revision 1.24  2006/02/27 06:15:55  reinelt
 * indent...
 *
 * Revision 1.23  2006/02/27 06:14:46  reinelt
 * graphic bug resulting in all black pixels solved
 *
 * Revision 1.22  2006/02/24 13:07:10  geronet
 * hollow bars for graphic lcd's
 *
 * Revision 1.21  2006/02/19 07:20:53  reinelt
 * image support nearly finished
 *
 * Revision 1.20  2006/02/08 04:55:05  reinelt
 * moved widget registration to drv_generic_graphic
 *
 * Revision 1.19  2006/02/07 05:36:13  reinelt
 * Layers added to Layout
 *
 * Revision 1.18  2006/01/30 05:47:38  reinelt
 * graphic subsystem changed to full-color RGBA
 *
 * Revision 1.17  2006/01/03 06:13:46  reinelt
 * GPIO's for MatrixOrbital
 *
 * Revision 1.16  2005/12/13 14:07:28  reinelt
 * LPH7508 driver finished
 *
 * Revision 1.15  2005/05/08 04:32:44  reinelt
 * CodingStyle added and applied
 *
 * Revision 1.14  2005/01/18 06:30:23  reinelt
 * added (C) to all copyright statements
 *
 * Revision 1.13  2005/01/09 10:53:24  reinelt
 * small type in plugin_uname fixed
 * new homepage lcd4linux.bulix.org
 *
 * Revision 1.12  2004/06/26 12:04:59  reinelt
 *
 * uh-oh... the last CVS log message messed up things a lot...
 *
 * Revision 1.11  2004/06/26 09:27:21  reinelt
 *
 * added '-W' to CFLAGS
 * changed all C++ comments to C ones
 * cleaned up a lot of signed/unsigned mistakes
 *
 * Revision 1.10  2004/06/20 10:09:55  reinelt
 *
 * 'const'ified the whole source
 *
 * Revision 1.9  2004/06/09 06:40:29  reinelt
 *
 * splash screen for T6963 driver
 *
 * Revision 1.8  2004/06/08 21:46:38  reinelt
 *
 * splash screen for X11 driver (and generic graphic driver)
 *
 * Revision 1.7  2004/06/01 06:45:30  reinelt
 *
 * some Fixme's processed
 * documented some code
 *
 * Revision 1.6  2004/03/03 03:47:04  reinelt
 * big patch from Martin Hejl:
 * - use qprintf() where appropriate
 * - save CPU cycles on gettimeofday()
 * - add quit() functions to free allocated memory
 * - fixed lots of memory leaks
 *
 * Revision 1.5  2004/02/29 14:30:59  reinelt
 * icon visibility fix for generic graphics from Xavier
 *
 * Revision 1.4  2004/02/24 05:55:04  reinelt
 *
 * X11 driver ported
 *
 * Revision 1.3  2004/02/22 17:35:41  reinelt
 * some fixes for generic graphic driver and T6963
 * removed ^M from plugin_imon (Nico, are you editing under Windows?)
 *
 * Revision 1.2  2004/02/18 06:39:20  reinelt
 * T6963 driver for graphic displays finished
 *
 * Revision 1.1  2004/02/15 21:43:43  reinelt
 * T6963 driver nearly finished
 * framework for graphic displays done
 * i2c_sensors patch from Xavier
 * some more old generation files removed
 *
 */

/* 
 *
 * exported functions:
 *
 * int drv_generic_graphic_init (char *section, char *driver);
 *   initializes the generic graphic driver
 *
 * int drv_generic_graphic_draw (WIDGET *W);
 *   renders Text widget into framebuffer
 *   calls drv_generic_graphic_real_blit()
 *
 * int drv_generic_graphic_icon_draw (WIDGET *W);
 *   renders Icon widget into framebuffer
 *   calls drv_generic_graphic_real_blit()
 *
 * int drv_generic_graphic_bar_draw (WIDGET *W);
 *   renders Bar widget into framebuffer
 *   calls drv_generic_graphic_real_blit()
 *
 * int drv_generic_graphic_quit (void);
 *   closes generic graphic driver
 *
 */


#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <termios.h>
#include <fcntl.h>

#include "debug.h"
#include "cfg.h"
#include "plugin.h"
#include "layout.h"
#include "widget.h"
#include "widget_text.h"
#include "widget_icon.h"
#include "widget_bar.h"
#include "widget_image.h"
#include "rgb.h"
#include "drv.h"
#include "drv_generic.h"
#include "drv_generic_graphic.h"
#include "property.h"
#include "font_6x8.h"
#include "font_6x8_bold.h"

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif

/* pixel colors */
RGBA FG_COL = { R: 0x00, G: 0x00, B: 0x00, A:0xff };
RGBA BG_COL = { R: 0xff, G: 0xff, B: 0xff, A:0xff };
RGBA BL_COL = { R: 0xff, G: 0xff, B: 0xff, A:0x00 };
RGBA NO_COL = { R: 0x00, G: 0x00, B: 0x00, A:0x00 };

static char *Section = NULL;
static char *Driver = NULL;

/* framebuffer */
static RGBA *drv_generic_graphic_FB[LAYERS] = { NULL, };

/* inverted colors */
static int INVERTED = 0;

/* must be implemented by the real driver */
void (*drv_generic_graphic_real_blit) () = NULL;


/****************************************/
/*** generic Framebuffer stuff        ***/
/****************************************/

static void drv_generic_graphic_resizeFB(int rows, int cols)
{
    RGBA *newFB;
    int i, l, row, col;

    /* Layout FB is large enough */
    if (rows <= LROWS && cols <= LCOLS)
	return;

    /* get maximum values */
    if (rows < LROWS)
	rows = LROWS;
    if (cols < LCOLS)
	cols = LCOLS;

    for (l = 0; l < LAYERS; l++) {

	/* allocate and initialize new Layout FB */
	newFB = malloc(cols * rows * sizeof(*newFB));
	for (i = 0; i < rows * cols; i++)
	    newFB[i] = (l == 0) ? BG_COL : NO_COL;

	/* transfer contents */
	if (drv_generic_graphic_FB[l] != NULL) {
	    for (row = 0; row < LROWS; row++) {
		for (col = 0; col < LCOLS; col++) {
		    newFB[row * cols + col] = drv_generic_graphic_FB[l][row * LCOLS + col];
		}
	    }
	    free(drv_generic_graphic_FB[l]);
	}
	drv_generic_graphic_FB[l] = newFB;
    }

    LCOLS = cols;
    LROWS = rows;

}

static void drv_generic_graphic_window(int pos, int size, int max, int *wpos, int *wsize)
{
    int p1 = pos;
    int p2 = pos + size;

    *wpos = 0;
    *wsize = 0;

    if (p1 > max || p2 < 0 || size < 1)
	return;

    if (p1 < 0)
	p1 = 0;

    if (p2 > max)
	p2 = max;

    *wpos = p1;
    *wsize = p2 - p1;
}

static void drv_generic_graphic_blit(const int row, const int col, const int height, const int width)
{
    if (drv_generic_graphic_real_blit) {
	int r, c, h, w;
	drv_generic_graphic_window(row, height, DROWS, &r, &h);
	drv_generic_graphic_window(col, width, DCOLS, &c, &w);
	if (h > 0 && w > 0) {
	    drv_generic_graphic_real_blit(r, c, h, w);
	}
    }
}

static RGBA drv_generic_graphic_blend(const int row, const int col)
{
    int l;
    RGBA p;
    RGBA ret;

    ret.R = BL_COL.R;
    ret.G = BL_COL.G;
    ret.B = BL_COL.B;
    ret.A = 0xff;
    for (l = LAYERS - 1; l >= 0; l--) {
	p = drv_generic_graphic_FB[l][row * LCOLS + col];
	if (p.A > 0) {
	    ret.R = (p.R * p.A + ret.R * (255 - p.A)) / 255;
	    ret.G = (p.G * p.A + ret.G * (255 - p.A)) / 255;
	    ret.B = (p.B * p.A + ret.B * (255 - p.A)) / 255;
	}
    }
    if (INVERTED) {
	ret.R = 255 - ret.R;
	ret.G = 255 - ret.G;
	ret.B = 255 - ret.B;
    }

    return ret;
}


/****************************************/
/*** generic text handling            ***/
/****************************************/

static void drv_generic_graphic_render(const int layer, const int row, const int col, const RGBA fg, const RGBA bg,
				       const char *style, const char *txt)
{
    int c, r, x, y, len;

    /* sanity checks */
    if (layer < 0 || layer >= LAYERS) {
	error("%s: layer %d out of bounds (0..%d)", Driver, layer, LAYERS - 1);
	return;
    }

    len = strlen(txt);

    /* maybe grow layout framebuffer */
    drv_generic_graphic_resizeFB(row + YRES, col + XRES * len);

    r = row;
    c = col;

    /* render text into layout FB */
    while (*txt != '\0') {
	unsigned char *chr;

	if (strcasestr(style, "bold") != NULL) {
	    chr = Font_6x8_bold[(int) *(unsigned char *) txt];
	} else {
	    chr = Font_6x8[(int) *(unsigned char *) txt];
	}

	for (y = 0; y < YRES; y++) {
	    int mask = 1 << XRES;
	    for (x = 0; x < XRES; x++) {
		mask >>= 1;
		if (chr[y] & mask)
		    drv_generic_graphic_FB[layer][(r + y) * LCOLS + c + x] = fg;
		else
		    drv_generic_graphic_FB[layer][(r + y) * LCOLS + c + x] = bg;
	    }
	}
	c += XRES;
	txt++;
    }

    /* flush area */
    drv_generic_graphic_blit(row, col, YRES, XRES * len);

}


/* say hello to the user */
int drv_generic_graphic_greet(const char *msg1, const char *msg2)
{
    char *line1[] = { "* LCD4Linux " VERSION " *",
	"LCD4Linux " VERSION,
	"* LCD4Linux *",
	"LCD4Linux",
	"L4Linux",
	NULL
    };

    char *line2[] = { "http://lcd4linux.bulix.org",
	"lcd4linux.bulix.org",
	NULL
    };

    int i;
    int flag = 0;

    unsigned int cols = DCOLS / XRES;
    unsigned int rows = DROWS / YRES;

    for (i = 0; line1[i]; i++) {
	if (strlen(line1[i]) <= cols) {
	    drv_generic_graphic_render(0, YRES * 0, XRES * ((cols - strlen(line1[i])) / 2), FG_COL, BG_COL, "norm",
				       line1[i]);
	    flag = 1;
	    break;
	}
    }

    if (rows >= 2) {
	for (i = 0; line2[i]; i++) {
	    if (strlen(line2[i]) <= cols) {
		drv_generic_graphic_render(0, YRES * 1, XRES * ((cols - strlen(line2[i])) / 2), FG_COL, BG_COL, "norm",
					   line2[i]);
		flag = 1;
		break;
	    }
	}
    }

    if (msg1 && rows >= 3) {
	unsigned int len = strlen(msg1);
	if (len <= cols) {
	    drv_generic_graphic_render(0, YRES * 2, XRES * ((cols - len) / 2), FG_COL, BG_COL, "norm", msg1);
	    flag = 1;
	}
    }

    if (msg2 && rows >= 4) {
	unsigned int len = strlen(msg2);
	if (len <= cols) {
	    drv_generic_graphic_render(0, YRES * 3, XRES * ((cols - len) / 2), FG_COL, BG_COL, "norm", msg2);
	    flag = 1;
	}
    }

    return flag;
}


int drv_generic_graphic_draw(WIDGET * W)
{
    WIDGET_TEXT *Text = W->data;
    RGBA fg, bg;

    fg = W->fg_valid ? W->fg_color : FG_COL;
    bg = W->bg_valid ? W->bg_color : BG_COL;

    drv_generic_graphic_render(W->layer, YRES * W->row, XRES * W->col, fg, bg, P2S(&Text->style), Text->buffer);

    return 0;
}


/****************************************/
/*** generic icon handling            ***/
/****************************************/

int drv_generic_graphic_icon_draw(WIDGET * W)
{
    WIDGET_ICON *Icon = W->data;
    RGBA fg, bg;
    unsigned char *bitmap = Icon->bitmap + YRES * Icon->curmap;
    int layer, row, col;
    int x, y;

    layer = W->layer;
    row = YRES * W->row;
    col = XRES * W->col;

    fg = W->fg_valid ? W->fg_color : FG_COL;
    bg = W->bg_valid ? W->bg_color : BG_COL;

    /* sanity check */
    if (layer < 0 || layer >= LAYERS) {
	error("%s: layer %d out of bounds (0..%d)", Driver, layer, LAYERS - 1);
	return -1;
    }

    /* maybe grow layout framebuffer */
    drv_generic_graphic_resizeFB(row + YRES, col + XRES);

    /* render icon */
    for (y = 0; y < YRES; y++) {
	int mask = 1 << XRES;
	for (x = 0; x < XRES; x++) {
	    int i = (row + y) * LCOLS + col + x;
	    mask >>= 1;
	    if (Icon->visible) {
		if (bitmap[y] & mask)
		    drv_generic_graphic_FB[layer][i] = fg;
		else
		    drv_generic_graphic_FB[layer][i] = bg;
	    } else {
		drv_generic_graphic_FB[layer][i] = BG_COL;
	    }
	}
    }

    /* flush area */
    drv_generic_graphic_blit(row, col, YRES, XRES);

    return 0;

}


/****************************************/
/*** generic bar handling             ***/
/****************************************/

int drv_generic_graphic_bar_draw(WIDGET * W)
{
    WIDGET_BAR *Bar = W->data;
    RGBA fg, bg, bar[2];
    int layer, row, col, len, res, rev, max, val1, val2;
    int x, y;
    DIRECTION dir;
    STYLE style;

    layer = W->layer;
    row = YRES * W->row;
    col = XRES * W->col;
    dir = Bar->direction;
    style = Bar->style;
    len = Bar->length;

    fg = W->fg_valid ? W->fg_color : FG_COL;
    bg = W->bg_valid ? W->bg_color : BG_COL;

    bar[0] = Bar->color_valid[0] ? Bar->color[0] : fg;
    bar[1] = Bar->color_valid[1] ? Bar->color[1] : fg;

    /* sanity check */
    if (layer < 0 || layer >= LAYERS) {
	error("%s: layer %d out of bounds (0..%d)", Driver, layer, LAYERS - 1);
	return -1;
    }

    /* maybe grow layout framebuffer */
    if (dir & (DIR_EAST | DIR_WEST)) {
	drv_generic_graphic_resizeFB(row + YRES, col + XRES * len);
    } else {
	drv_generic_graphic_resizeFB(row + YRES * len, col + XRES);
    }

    res = dir & (DIR_EAST | DIR_WEST) ? XRES : YRES;
    max = len * res;
    val1 = Bar->val1 * (double) (max);
    val2 = Bar->val2 * (double) (max);

    if (val1 < 1)
	val1 = 1;
    else if (val1 > max)
	val1 = max;

    if (val2 < 1)
	val2 = 1;
    else if (val2 > max)
	val2 = max;

    rev = 0;

    switch (dir) {
    case DIR_WEST:
	val1 = max - val1;
	val2 = max - val2;
	rev = 1;

    case DIR_EAST:
	for (y = 0; y < YRES; y++) {
	    int val = y < YRES / 2 ? val1 : val2;
	    RGBA bcol = y < YRES / 2 ? bar[0] : bar[1];

	    for (x = 0; x < max; x++) {
		if (x < val)
		    drv_generic_graphic_FB[layer][(row + y) * LCOLS + col + x] = rev ? bg : bcol;
		else
		    drv_generic_graphic_FB[layer][(row + y) * LCOLS + col + x] = rev ? bcol : bg;

		if (style) {
		    drv_generic_graphic_FB[layer][(row) * LCOLS + col + x] = fg;
		    drv_generic_graphic_FB[layer][(row + YRES - 1) * LCOLS + col + x] = fg;
		}
	    }
	    if (style) {
		drv_generic_graphic_FB[layer][(row + y) * LCOLS + col] = fg;
		drv_generic_graphic_FB[layer][(row + y) * LCOLS + col + max - 1] = fg;
	    }
	}
	break;

    case DIR_NORTH:
	val1 = max - val1;
	val2 = max - val2;
	rev = 1;

    case DIR_SOUTH:
	for (x = 0; x < XRES; x++) {
	    int val = x < XRES / 2 ? val1 : val2;
	    for (y = 0; y < max; y++) {
		if (y < val)
		    drv_generic_graphic_FB[layer][(row + y) * LCOLS + col + x] = rev ? bg : fg;
		else
		    drv_generic_graphic_FB[layer][(row + y) * LCOLS + col + x] = rev ? fg : bg;
	    }
	}
	break;
    }

    /* flush area */
    if (dir & (DIR_EAST | DIR_WEST)) {
	drv_generic_graphic_blit(row, col, YRES, XRES * len);
    } else {
	drv_generic_graphic_blit(row, col, YRES * len, XRES);
    }

    return 0;
}


/****************************************/
/*** generic image handling           ***/
/****************************************/

int drv_generic_graphic_image_draw(WIDGET * W)
{
    WIDGET_IMAGE *Image = W->data;
    int layer, row, col, width, height;
    int x, y;

    layer = W->layer;
    row = W->row;
    col = W->col;
    width = Image->width;
    height = Image->height;

    /* sanity check */
    if (layer < 0 || layer >= LAYERS) {
	error("%s: layer %d out of bounds (0..%d)", Driver, layer, LAYERS - 1);
	return -1;
    }

    /* if no size or no image at all, do nothing */
    if (width <= 0 || height <= 0 || Image->bitmap == NULL) {
	return 0;
    }

    /* maybe grow layout framebuffer */
    drv_generic_graphic_resizeFB(row + height, col + width);

    /* render image */
    for (y = 0; y < height; y++) {
	for (x = 0; x < width; x++) {
	    int i = (row + y) * LCOLS + col + x;
	    if (Image->visible) {
		drv_generic_graphic_FB[layer][i] = Image->bitmap[y * width + x];
	    } else {
		drv_generic_graphic_FB[layer][i] = BG_COL;
	    }
	}
    }

    /* flush area */
    drv_generic_graphic_blit(row, col, height, width);

    return 0;

}


/****************************************/
/*** generic init/quit                ***/
/****************************************/

int drv_generic_graphic_init(const char *section, const char *driver)
{
    int l;
    char *color;
    WIDGET_CLASS wc;

    Section = (char *) section;
    Driver = (char *) driver;

    /* init layout framebuffer */
    LROWS = 0;
    LCOLS = 0;

    for (l = 0; l < LAYERS; l++)
	drv_generic_graphic_FB[l] = NULL;

    drv_generic_graphic_resizeFB(DROWS, DCOLS);

    /* sanity check */
    for (l = 0; l < LAYERS; l++) {
	if (drv_generic_graphic_FB[l] == NULL) {
	    error("%s: framebuffer could not be allocated: malloc() failed", Driver);
	    return -1;
	}
    }

    /* init generic driver & register plugins */
    drv_generic_init();

    /* set default colors */
    color = cfg_get(Section, "foreground", "000000ff");
    if (color2RGBA(color, &FG_COL) < 0) {
	error("%s: ignoring illegal color '%s'", Driver, color);
    }
    if (color)
	free(color);

    color = cfg_get(Section, "background", "ffffff00");
    if (color2RGBA(color, &BG_COL) < 0) {
	error("%s: ignoring illegal color '%s'", Driver, color);
    }
    if (color)
	free(color);

    color = cfg_get(Section, "basecolor", "ffffff");
    if (color2RGBA(color, &BL_COL) < 0) {
	error("%s: ignoring illegal color '%s'", Driver, color);
    }
    if (color)
	free(color);

    /* inverted display? */
    cfg_number(section, "inverted", 0, 0, 1, &INVERTED);

    /* register text widget */
    wc = Widget_Text;
    wc.draw = drv_generic_graphic_draw;
    widget_register(&wc);

    /* register icon widget */
    wc = Widget_Icon;
    wc.draw = drv_generic_graphic_icon_draw;
    widget_register(&wc);

    /* register bar widget */
    wc = Widget_Bar;
    wc.draw = drv_generic_graphic_bar_draw;
    widget_register(&wc);

    /* register image widget */
#ifdef WITH_IMAGE
    wc = Widget_Image;
    wc.draw = drv_generic_graphic_image_draw;
    widget_register(&wc);
#endif

    /* clear framebuffer */
    drv_generic_graphic_clear();

    return 0;
}


int drv_generic_graphic_clear(void)
{
    int i, l;

    for (l = 0; l < LAYERS; l++)
	for (i = 0; i < LCOLS * LROWS; i++)
	    drv_generic_graphic_FB[l][i] = NO_COL;

    drv_generic_graphic_blit(0, 0, LROWS, LCOLS);

    return 0;
}


RGBA drv_generic_graphic_rgb(const int row, const int col)
{
    return drv_generic_graphic_blend(row, col);
}


unsigned char drv_generic_graphic_gray(const int row, const int col)
{
    RGBA p = drv_generic_graphic_blend(row, col);
    return (77 * p.R + 150 * p.G + 28 * p.B) / 255;
}


unsigned char drv_generic_graphic_black(const int row, const int col)
{
    return drv_generic_graphic_gray(row, col) < 127;
}


int drv_generic_graphic_quit(void)
{
    int l;

    for (l = 0; l < LAYERS; l++) {
	if (drv_generic_graphic_FB[l]) {
	    free(drv_generic_graphic_FB[l]);
	    drv_generic_graphic_FB[l] = NULL;
	}
    }
    widget_unregister();
    return (0);
}
