/***************************************************************************
                          unit.c  -  description
                             -------------------
    begin                : Fri Jan 19 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <SDL.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef WITH_SOUND
#include <SDL_mixer.h>
#include "audio.h"
#endif

#include "sdl.h"
#include "dynlist.h"
#include "config.h"
#include "tools.h"
#include "file.h"
#include "nation.h"
#include "unit.h"

extern Sdl sdl;
extern Config config;

/* this weather type must be set by set_player() in scenario.c */
int cur_weather = FAIR;

/* --- forwarded --- */
/* update life bar width */
void update_bar( Unit *unit );

/* open unit file and return file handle (adds the correct path)*/
FILE* open_unit_file( char *file_name )
{
    char full_path[256];

    /* don't forget to improve this later */
    sprintf(full_path, "%s/units/%s", SRC_DIR, file_name);
    printf( "parsing units/%s\n", file_name );

    return file_open( full_path );
}

/* prepares the dynamic list which will be used as unit library */
void init_unit_lib( Dyn_List *list )
{
    dl_init( list, AUTO_DELETE, delete_unit_lib_entry );
}

/* three white pixels measure the size of a single unit pic */
void get_unit_pic_size( SDL_Surface *unit_pics, int pic_id, int *width, int *height, int *offset, int *key )
{
    Uint32 mark;
    int y;
    int count = pic_id * 2; /* there are two pixels for one pic */

    /* nada white dot! take the first pixel found in the upper left corner as mark */
    mark = get_pixel( unit_pics, 0, 0 );

    /* compute offset */
    for ( y = 0; y < unit_pics->h; y++ )
        if ( get_pixel( unit_pics, 0, y ) == mark ) {
            if ( count == 0 ) break;
            count--;
        }
    *offset = y;

    /* compute height */
    y++;
    while ( y < unit_pics->h && get_pixel( unit_pics, 0, y ) != mark )
        y++;
   (*height) = y - (*offset) + 1;

    /* compute width */
    y = *offset;
    *width = 1;
    while ( get_pixel( unit_pics, (*width ), y ) != mark )
        (*width)++;
    (*width)++;

    /* pixel below left upper corner is set as color key */
    y = (*offset) + 1;
    *key = get_pixel( unit_pics, 0, y );
}

/* read single unit from file */
Unit_Lib_Entry* read_single_unit( FILE *file, SDL_Surface *unit_pics, Unit_Def *def, Nation **nations, int nation_count )
{
    Unit_Lib_Entry *entry = 0;
    char *cur_arg = 0;
    int i, j;
    int byte_size, y_offset;
    char *pix_buffer;
    int arg_count;
    char **args;
    int pic_id;
    int unit_width, unit_height, unit_offset;
    Uint32 color_key; /* unit transparent color key */

    /* create&clear struct */
    entry = malloc( sizeof( Unit_Lib_Entry ) );
    memset( entry, 0, sizeof( Unit_Lib_Entry ) );

    /* one more unit defined? */
    if ( !find_token( file, "entry_name", FROM_CURRENT_FILE_POS, NO_WARNING ) ) goto failure;

    /* read entry name */
    if ( ( cur_arg = get_arg( file, "entry_name", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->entry_name = strdup( cur_arg );
    FREE( cur_arg );

    /* read caption */
    if ( ( cur_arg = get_arg( file, "caption", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->cap = strdup( cur_arg );
    FREE( cur_arg );

    /* nation of origin */
    if ( ( cur_arg = get_arg( file, "nation", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->nation = -1;
    for ( i = 0; i < nation_count; i++ )
        if ( equal_str( cur_arg, nations[i]->entry_name ) )
            entry->nation = i;
    FREE( cur_arg );

    /* unit class */
    if ( ( cur_arg = get_arg( file, "unit_class", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->class = 0;
    for ( i = 0; i < def->class_count; i++ )
        if ( equal_str( cur_arg, def->classes[i].entry_name ) )
            entry->class = i;
    FREE( cur_arg );

    /* target type */
    if ( ( cur_arg = get_arg( file, "target_type", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->type = 0;
    for ( i = 0; i < def->type_count; i++ )
        if ( equal_str( cur_arg, def->type_names[i] ) )
            entry->type = i;
    FREE( cur_arg );
    /* check target type */
    if ( entry->type > TARGET_TYPE_LIMIT ) entry->type = 0;

    /* initiative */
    if ( ( cur_arg = get_arg( file, "initiative", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->ini = atoi( cur_arg );
    FREE( cur_arg );

    /* movement */
    if ( ( cur_arg = get_arg( file, "movement", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->mov = atoi( cur_arg );
    FREE( cur_arg );

    /* spotting */
    if ( ( cur_arg = get_arg( file, "spotting", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->spot = atoi( cur_arg );
    FREE( cur_arg );

    /* range */
    if ( ( cur_arg = get_arg( file, "range", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->range = atoi( cur_arg );
    FREE( cur_arg );

    /* number of attacks per turn */
    if ( ( cur_arg = get_arg( file, "attack_count", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->attack_count = atoi( cur_arg );
    FREE( cur_arg );

    /* attack */
    if ( ( args = get_arg_cluster( file, "attack", &arg_count, FROM_CURRENT_FILE_POS, WARNING ) ) == 0 ) goto failure;
    for ( i = 0; i < arg_count; i++ )
        if ( i < def->type_count )
            entry->attack[i] = atoi( args[i] );
    delete_arg_cluster( args, arg_count );

    /* and now let's point towards the unit definition struct, hehe */
    entry->unit_def = def;

    /* defence */
    if ( ( cur_arg = get_arg( file, "defence", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->def = atoi( cur_arg );
    FREE( cur_arg );

    /* ammo */
    if ( ( cur_arg = get_arg( file, "ammo", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->ammo = atoi( cur_arg );
    FREE( cur_arg );

    /* fuel */
    if ( ( cur_arg = get_arg( file, "fuel", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    entry->fuel = atoi( cur_arg );
    FREE( cur_arg );

    /* flags */
    if ( ( args = get_arg_cluster( file, "flags", &arg_count, FROM_CURRENT_FILE_POS, WARNING ) ) == 0 ) goto failure;
    entry->flags = 0;
    for ( i = 0; i < arg_count; i++ ) {

        cur_arg = args[i];

        if ( equal_str( "none", cur_arg ) )
            break;
        else
        if ( equal_str( "interceptor", cur_arg ) )
            entry->flags = entry->flags | INTERCEPTOR;
        else
        if ( equal_str( "air_defence", cur_arg ) )
            entry->flags = entry->flags | AIR_DEFENCE;
        else
        if ( equal_str( "swimming", cur_arg ) )
            entry->flags = entry->flags | SWIMMING;
        else
        if ( equal_str( "flying", cur_arg ) )
            entry->flags = entry->flags | FLYING;
        else
        if ( equal_str( "diving", cur_arg ) )
            entry->flags = entry->flags | DIVING;
        else
        if ( equal_str( "parachute", cur_arg ) )
            entry->flags = entry->flags | PARACHUTE;
        else
        if ( equal_str( "wheeled", cur_arg ) )
            entry->flags = entry->flags | WHEELED;
        else
        if ( equal_str( "tracked", cur_arg ) )
            entry->flags = entry->flags | TRACKED;
        else
        if ( equal_str( "transporter", cur_arg ) )
            entry->flags = entry->flags | TRANSPORTER;
        else
        if ( equal_str( "recon", cur_arg ) )
            entry->flags = entry->flags | RECON;
        else
        if ( equal_str( "artillery", cur_arg ) )
            entry->flags = entry->flags | ARTILLERY;
        else
        if ( equal_str( "bridge_eng", cur_arg ) )
            entry->flags = entry->flags | BRIDGE_ENG;
        else
            fprintf( stderr, "read_single_unit: unknown unit flag '%s'\n", cur_arg );

    }
    delete_arg_cluster( args, arg_count );

    /* picture type */
    if ( ( cur_arg = get_arg( file, "picture_type", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    if ( equal_str( "all_directions", cur_arg ) )
        entry->pic_type = ALL_DIRS;
    else
        entry->pic_type = SINGLE;
    FREE( cur_arg );

    /* picture */
    if ( ( cur_arg = get_arg( file, "picture_id", FROM_CURRENT_FILE_POS ) ) == 0 ) goto failure;
    pic_id = atoi( cur_arg );
    FREE( cur_arg );

    /* get unit picture info */
    get_unit_pic_size( unit_pics, pic_id, &unit_width, &unit_height, &unit_offset, &color_key );

    /* picture is copied from unit_pics first */
    /* if picture_type is not ALL_DIRS, picture is a single picture looking to the right;
        add a flipped picture looking to the left */
    if ( entry->pic_type == ALL_DIRS ) {

        entry->pic = create_surf( unit_width * 6, unit_height, SDL_SWSURFACE );
        entry->width = entry->pic->w;
        entry->height = entry->pic->h;
        FULL_DEST( entry->pic );
        SOURCE( unit_pics, 0, unit_offset );
        blit_surf();
        /* remove measure dots */
        set_pixel( entry->pic, 0, 0, color_key );
        set_pixel( entry->pic, 0, entry->height - 1, color_key );
        set_pixel( entry->pic, entry->width - 1, 0, color_key );
        /* set transparency */
        SDL_SetColorKey( entry->pic, SDL_SRCCOLORKEY, color_key );

    }
    else {

        /* set size */
        entry->width = unit_width;
        entry->height = unit_height;

        /* create pic and copy first pic */
        entry->pic = create_surf( entry->width * 2, entry->height, SDL_SWSURFACE );
        DEST( entry->pic, 0, 0, entry->width, entry->height );
        SOURCE( unit_pics, 0, unit_offset );
        blit_surf();

        /* remove measure dots */
        set_pixel( entry->pic, 0, 0, color_key );
        set_pixel( entry->pic, 0, entry->height - 1, color_key );
        set_pixel( entry->pic, entry->width - 1, 0, color_key );
        /* set transparency */
        SDL_SetColorKey( entry->pic, SDL_SRCCOLORKEY, color_key );

        /* get format info */
        byte_size = unit_pics->format->BytesPerPixel;
        y_offset = 0;
        pix_buffer = malloc( sizeof( char ) * byte_size );

        /* get second by flipping first one */
        lock_surf( unit_pics );
        lock_surf( entry->pic );
        for ( j = 0; j < entry->height; j++ ) {

            for ( i = 0; i < entry->width; i++ ) {

                memcpy( pix_buffer,
                        entry->pic->pixels +
                        y_offset +
                        ( entry->width - 1 - i ) * byte_size,
                        byte_size );
                memcpy( entry->pic->pixels +
                        y_offset +
                        entry->width * byte_size +
                        i * byte_size,
                        pix_buffer, byte_size );
            }
            y_offset += entry->pic->pitch;

        }
        unlock_surf( unit_pics );
        unlock_surf( entry->pic );

        /* free mem */
        FREE( pix_buffer );
    }

    /* entry->pic is completed, now create small_pic */
    entry->small_pic = create_surf( entry->pic->w / 2, entry->pic->h / 2, SDL_SWSURFACE );
    /* use color key of 'big' picture */
    SDL_SetColorKey( entry->small_pic, SDL_SRCCOLORKEY, get_pixel( entry->pic, 0, 0 ) );
    /* shrink */
    byte_size = entry->pic->format->BytesPerPixel;
    y_offset = 0;
    pix_buffer = malloc( sizeof( char ) * byte_size );
    for ( j = 0; j < entry->pic->h; j += 2 ) {

        for ( i = 0; i < entry->pic->w; i += 2 ) {

            memcpy( pix_buffer,
                    entry->pic->pixels +
                    y_offset +
                    i * byte_size,
                    byte_size );
            memcpy( entry->small_pic->pixels +
                    ( j / 2 ) * entry->small_pic->pitch +
                    ( i / 2 ) * byte_size,
                    pix_buffer, byte_size );

        }

        y_offset += entry->pic->pitch * 2;

    }
    FREE( pix_buffer );

    /* sounds */
    if ( ( args = get_arg_cluster( file, "sounds", &arg_count, FROM_CURRENT_FILE_POS, WARNING ) ) == 0 ) goto failure;
#ifdef WITH_SOUND
    /* moving sound */
    entry->own_sound = 0;
    if ( equal_str( args[0], "none" ) )
        entry->move_sound = 0;
    else
        if ( equal_str( args[0], "default" ) )
            entry->move_sound = def->classes[entry->class].move_sound;
        else
            if ( ( entry->move_sound = sound_chunk_load( args[0] ) ) != 0 )
                entry->own_sound = 1;
#endif
    delete_arg_cluster( args, arg_count );

    return entry;

failure:
    delete_unit_lib_entry( entry );
    return 0;
}

/* read all units from file to Unit_Lib_Entry structs and add them to the given dynamic list */
void read_units_from_file( char *file_name, Dyn_List *list, Unit_Def *def, Nation **nations, int nation_count )
{
    FILE *file = 0;
    Unit_Lib_Entry *entry;
    SDL_Surface *unit_pics; /* all unit graphics are saved in one surface in vertical order */
    char *cur_arg;
    char path[512];

    /* open file */
    file = open_unit_file( file_name );
    if ( file == 0 ) return;

    /* get unit pictures */
    if ( ( cur_arg = get_arg( file, "unit_pictures", RESET_FILE_POS ) ) == 0 ) {
        fclose( file );
        return;
    }
    sprintf( path, "units/%s", cur_arg );
    unit_pics = load_surf( path, SDL_SWSURFACE );
    FREE( cur_arg );

    while ( !feof( file ) ) {
        entry = read_single_unit( file, unit_pics, def, nations, nation_count );
        if ( entry != 0 )
            dl_add( list, entry );
        else
            break;
    }

    SDL_FreeSurface( unit_pics );
    fclose( file );
}

/* deletes a unit struct completely. compatible with dyn list callback because void* is given as
argument */
void delete_unit_lib_entry( void* poi )
{
    Unit_Lib_Entry *entry = (Unit_Lib_Entry*)poi;

    if ( entry == 0 ) return;

    /* delete within ...*/
    if ( entry->entry_name ) FREE( entry->entry_name );
    if ( entry->cap) FREE( entry->cap );
    if ( entry->pic ) SDL_FreeSurface( entry->pic );
    if ( entry->small_pic ) SDL_FreeSurface( entry->small_pic );
#ifdef WITH_SOUND
    if ( entry->own_sound ) {
        if ( entry->move_sound ) sound_chunk_free( entry->move_sound );
    }
#endif

    /* ... and the struct itself */
    FREE( entry ); entry = 0;

    return;
}

/* find unit lib entry by searching for entry_name */
Unit_Lib_Entry* find_unit_lib_entry( Dyn_List *list, char *entry_name )
{
    DL_Entry *entry;

    list->cur_entry = &list->head;

    while ( (entry = dl_next_entry( list ) ) )
        if ( equal_str( ((Unit_Lib_Entry*)entry->data)->entry_name, entry_name ) )
            return (Unit_Lib_Entry*)entry->data;

    return 0;
}

/* read unit definitions (target types/ unit_classes) */
Unit_Def* read_unit_def( char *file_name )
{
    FILE *file = 0;
    Unit_Def *unit_def;
    char **args;
    int arg_count;
    int i;
    int marked = 0;

    /* open file */
    if ( ( file = open_unit_file( file_name ) ) == 0 ) return 0;

    /* create scenario structure and memset to 0 */
    unit_def = malloc( sizeof( Unit_Def ) );
    memset( unit_def, 0, sizeof( Unit_Def ) );

    /* read target names */
    args = get_arg_cluster( file, "target_types", &arg_count, RESET_FILE_POS, WARNING );
    if ( args == 0 ) goto failure;
    unit_def->type_count = arg_count;
    /* check type number */
    if ( unit_def->type_count > TARGET_TYPE_LIMIT ) {
        fprintf( stderr, "read_unit_def: too many target types declared! limit is %i\n", TARGET_TYPE_LIMIT );
        unit_def->type_count = TARGET_TYPE_LIMIT;
    }
    /* go on */
    unit_def->type_names = calloc( unit_def->type_count, sizeof(char*) );
    unit_def->air_type = calloc( unit_def->type_count, sizeof( int ) );
    for ( i = 0; i < unit_def->type_count; i++ ) {
        /* if there is '*' in front it's an air attack value and */
        /* influenced by weather (if it's a ground unit) */
        if ( args[i][0] == '*' ) {
            marked = 1;
            unit_def->air_type[i] = 1;
        }
        unit_def->type_names[i] = strdup( args[i] + marked );
        marked = 0;
    }
    delete_arg_cluster( args, arg_count );

    /* count class names */
    unit_def->class_count = count_arg( file, "unit_class" );

    /* read class names */
    find_token( file, "unit_class", RESET_FILE_POS, WARNING );
    unit_def->classes = calloc( unit_def->class_count, sizeof( Unit_Class ) );
    for ( i = 0; i < unit_def->class_count; i++ ) {
        args = get_arg_cluster( file, "unit_class", &arg_count, FROM_CURRENT_FILE_POS, NO_WARNING );
        unit_def->classes[i].entry_name = strdup( args[0] );
        unit_def->classes[i].name = strdup( args[1] );
#ifdef WITH_SOUND
        /* load sounds */
        if ( equal_str( args[2], "none" ) )
            unit_def->classes[i].move_sound = 0;
        else
            unit_def->classes[i].move_sound = sound_chunk_load( args[2] );
#endif
        delete_arg_cluster( args, arg_count );
    }

    fclose( file );
    return unit_def;

failure:
    if ( file ) fclose( file );
    delete_unit_def( unit_def );
    return 0;
}

/* delete unit definitions (target types/ unit_classes) */
Unit_Def* delete_unit_def( Unit_Def *unit_def )
{
    int i;

    if ( unit_def == 0 ) return 0;

    if ( unit_def->type_names ) {
        for ( i = 0; i < unit_def->type_count; i++ )
            if ( unit_def->type_names[i] )
                FREE( unit_def->type_names[i] );
        FREE( unit_def->type_names );
    }

    if ( unit_def->air_type )
        FREE( unit_def->air_type );

    if ( unit_def->classes ) {
        for ( i = 0; i < unit_def->class_count; i++ ) {
            if ( unit_def->classes[i].name ) free( unit_def->classes[i].name );
            if ( unit_def->classes[i].entry_name ) free( unit_def->classes[i].entry_name );
#ifdef WITH_SOUND
            if ( unit_def->classes[i].move_sound ) sound_chunk_free( unit_def->classes[i].move_sound );
#endif
        }
        FREE( unit_def->classes );
    }

    FREE( unit_def ); unit_def = 0;

    return 0;
}

/* create and add unit */
Unit* create_unit( int delay, int player_id, Unit_Lib_Entry *unit_lib_entry, char *unit_name, int dir, int x, int y, int entr, int exp_level, Unit_Lib_Entry *tran )
{
    Unit *unit = 0;

    if ( unit_lib_entry == 0 ) return 0;

    unit = malloc( sizeof( Unit ) );
    memset( unit, 0, sizeof( Unit ) );

    memcpy( &unit->prop, unit_lib_entry, sizeof( Unit_Lib_Entry ) );
    unit->sel_prop = &unit->prop; /* no embark right now */

    /* compute life */
    update_bar( unit );

    unit->tran_prop_set = 0;
    unit->embark = NO_EMBARK;
    unit->ground_tran = 0;
    /* check if this unit can have a transporter */
    /* note: a unit can't be air/sea-embarked with a ground transporter */
    /* switch life_value if unit is embarked */
    if ( tran && CHECK_FLAGS( tran->flags, TRANSPORTER ) ) {

        if ( CHECK_FLAGS( tran->flags, SWIMMING ) &&
             !CHECK_FLAGS( unit_lib_entry->flags, FLYING | SWIMMING ) )
        {

            memcpy( &unit->tran_prop, tran, sizeof( Unit_Lib_Entry ) );
            unit->tran_prop_set = 1;
            unit->embark = SEA_EMBARK;
            unit->sel_prop = &unit->tran_prop;

        }
        else
            if ( CHECK_FLAGS( tran->flags, FLYING ) &&
                 !CHECK_FLAGS( unit_lib_entry->flags, FLYING | SWIMMING | WHEELED | TRACKED ) )
            {

                memcpy( &unit->tran_prop, tran, sizeof( Unit_Lib_Entry ) );
                unit->tran_prop_set = 1;
                unit->embark = AIR_EMBARK;
                unit->sel_prop = &unit->tran_prop;

            }
            else
                if ( !CHECK_FLAGS( unit_lib_entry->flags, WHEELED | TRACKED | SWIMMING | FLYING ) &&
                     !CHECK_FLAGS( tran->flags, FLYING | SWIMMING ) ) {

                    memcpy( &unit->tran_prop, tran, sizeof( Unit_Lib_Entry ) );
                    unit->tran_prop_set = 1;
                    unit->ground_tran = 1;

                }

    }

    unit->x = x;
    unit->y = y;
    unit->player_id = player_id;
    unit->delay = delay;
    unit->entr = entr;
    strcpy( unit->name, unit_name );
    unit->kill_next_turn = 0;
    unit->killed = 0;

    /* add experience */
    unit->exp = 0;
    unit->exp_level = 0;
    gain_exp( unit, exp_level * EXP_PER_LEVEL );

    /* looking direction */
    unit->dir = dir;
    unit->pic_offset = unit->sel_prop->width * dir;

    /* first time so clear everything */
    memset( &unit->cur_prop, 0, sizeof( Unit_Lib_Entry ) );

    /* unit starts with full ammo and fuel */
    give_fuel( unit, 100 );
    unit->old_fuel = unit->prop.fuel;
    give_ammo( unit, 100 );

    /* supply level is computed in scenario.c as map is needed therefore */
    unit->supply_level = 100;

    /* set current values */
    update_cur_unit_prop( unit );

    return unit;
}

/* delete unit */
void delete_unit( void *poi )
{
    Unit *unit = (Unit*)poi;

    if ( !unit ) return;

    free ( unit ); unit = 0;
}

/* adjust picture offset */
inline void adjust_unit_pic_offset( Unit *unit )
{
    unit->pic_offset = unit->sel_prop->width * unit->dir;
}

/* get direction in which unit looks when moving to passed position */
void check_unit_dir( Unit *unit, int x, int y )
{
    if ( unit->prop.pic_type == SINGLE ) {

        if ( x < unit->x )  {

            unit->dir = LEFT;
            unit->pic_offset = unit->sel_prop->width;

        }
        else
            if ( x > unit->x ) {

                unit->dir = RIGHT;
                unit->pic_offset = 0;

            }

    }
    else {

        /* not implemented yet */

    }
}

/* reset current properties */
void reset_cur_prop( Unit *unit )
{
    /* copy data */
    memcpy( &unit->cur_prop, unit->sel_prop, sizeof( Unit_Lib_Entry ) );
}

/* update current values by checking experience and entrenchement */
/* this does not include fuel or ammo ! */
void update_cur_unit_prop( Unit *unit )
{
    int i;
    int add;

    /* save these values */
    int fuel = unit->cur_prop.fuel;
    int ammo = unit->cur_prop.ammo;
    int mov = unit->cur_prop.mov;
    int attack_count = unit->cur_prop.attack_count;

    /* copy original properties */
    reset_cur_prop( unit );

    /* restore ammo,fuel ... */
    unit->cur_prop.fuel = fuel;
    unit->cur_prop.ammo = ammo;
    unit->cur_prop.mov = mov;
    unit->cur_prop.attack_count = attack_count;

    /* influence of current weather */
    if ( config.weather ) {
        if ( unit->sel_prop->flags & FLYING ) {
            /* cloudy weather cuts air attack in half */
            if ( cur_weather == CLOUDS )
                unit->cur_prop.attack[unit->sel_prop->type] /= 2;
            /* rain and snow doesn't allow any air attack */
            if ( cur_weather == SNOW || cur_weather == RAIN )
                if ( unit->sel_prop->flags & FLYING ) {
                    unit->cur_prop.attack_count = 0;
                    for ( i = 0; i < unit->sel_prop->unit_def->type_count; i++ )
                        unit->cur_prop.attack[i] = 0;
                }
        }
        else {
            if ( cur_weather == SNOW || cur_weather == RAIN ) {
                /* snow and rain cut air attack of ground unit in half */
                for ( i = 0; i < unit->sel_prop->unit_def->type_count; i++ )
                    if ( unit->sel_prop->unit_def->air_type[i] )
                        unit->cur_prop.attack[i] /= 2;
            }
        }
    }

    /* experience gives 20% better attack per level */
    /* entrenchment enhaces attack as well; about 5% per point */
    for ( i = 0; i < unit->sel_prop->unit_def->type_count; i++ )
        if ( unit->cur_prop.attack[i] != 0 ) {
            add = 0;
            if ( unit->exp_level > 0 )
                add += (int)( 0.1 * unit->exp_level * unit->cur_prop.attack[i] );
/*            add += (int)( 0.05 * unit->entr * unit->cur_prop.attack[i] ); */
            unit->cur_prop.attack[i] += add;
        }

    /* entrenchment enhances defence and rugged defence chance */
    if ( unit->sel_prop->type == INFANTRY )
        unit->cur_prop.def += (int)(0.1 * unit->entr * unit->cur_prop.def);
    else
        unit->cur_prop.def += (int)(0.05 * unit->entr * unit->cur_prop.def);
    /* rugged defence chance */
    unit->rugged_def_chance = unit->entr * 2 + unit->exp_level;
    /* some units does not have rugged defence */
    if ( unit->sel_prop->flags & TRANSPORTER || unit->sel_prop->flags & ARTILLERY )
        unit->rugged_def_chance = 0;
}

/* adds perc percentage fuel to current fuel and returns the value of fuel added */
int give_fuel( Unit *unit, int perc )
{
    int add = 0;
    float rel = (float)perc / 100;
    int fuel;

    if ( unit->embark == NO_EMBARK && unit->sel_prop->fuel == 0 && use_fuel( unit ) )
        fuel = unit->tran_prop.fuel;
    else
        fuel = unit->sel_prop->fuel;
    add = (int)( rel * fuel );
    unit->cur_prop.fuel += add;
    if ( unit->cur_prop.fuel > fuel )
        unit->cur_prop.fuel = fuel;

    /* backuped value not supported yet */

    return add;
}

/* adds perc percentage ammo to current ammo and returns the value of ammo added */
int give_ammo( Unit *unit, int perc )
{
    int add = 0;
    float rel = (float)perc / 100;

    /* transporters don't shoot, so directly use unit_prop */

    add = (int)( rel * unit->prop.ammo );
    unit->cur_prop.ammo += add;
    if ( unit->cur_prop.ammo > unit->prop.ammo )
        unit->cur_prop.ammo = unit->prop.ammo;

    /* backuped value not supported yet */

    return add;
}

/* return if unit uses fuel or is embarked and transporter uses fuel */
inline int use_fuel( Unit *unit )
{
    if ( unit->embark == NO_EMBARK ) {

        /* if ground transporter use fuel anyway */
        if ( unit->tran_prop_set && !(unit->tran_prop.flags & FLYING ) && !(unit->tran_prop.flags & SWIMMING ) )
            return 1;
        else
            if ( unit->prop.fuel )
                return 1;
            else
                return 0;

    }
    else {

        if ( unit->tran_prop.fuel )
            return 1;
        else
            return 0;

    }
}

/* mount unit on ground transporter */
void mount_unit( Unit *unit )
{
    if ( !unit->tran_prop_set || unit->embark != NO_EMBARK ) return;

    /* set prop pointer */
    unit->sel_prop = &unit->tran_prop;
    unit->embark = GROUND_EMBARK;
    /* adjust pic offset */
    adjust_unit_pic_offset( unit );

    /* no entrenchment when mounting */
    unit->entr = 0;

    /* change current props */
    update_cur_unit_prop( unit );
}

/* unmount unit from ground transporter */
void unmount_unit( Unit *unit )
{
    if ( unit->embark != GROUND_EMBARK ) return;

    /* set prop pointer */
    unit->sel_prop = &unit->prop;
    unit->embark = NO_EMBARK;
    /* adjust pic offset */
    adjust_unit_pic_offset( unit );

    /* no entrenchment when mounting */
    unit->entr = 0;

    /* change current props */
    update_cur_unit_prop( unit );
}

/* check if unit can supply */
int unit_can_supply( Unit *unit, int type )
{
    int ret = 0;
    int max_fuel = unit->sel_prop->fuel;

    if ( unit->supply_level == 0 ) return 0;
    if ( !unit->no_action_yet ) return 0;

    /* supply ammo? */
    if ( type == AMMO || type == ANYTHING )
        if ( unit->cur_prop.ammo < unit->prop.ammo )
            ret = 1;

    if ( type == AMMO )
        return ret;

    /* maybe ground units transporter */
    if ( unit->embark == NO_EMBARK && unit->sel_prop->fuel == 0 && use_fuel( unit ) )
        max_fuel = unit->tran_prop.fuel;

    if ( type == FUEL || type == ANYTHING )
        if ( unit->cur_prop.fuel < max_fuel )
            ret = 1;

    return ret;
}

/* supply unit */
int supply_unit( Unit *unit )
{
    int supplied = 0;

    if ( unit_can_supply( unit, AMMO ) ) {
        give_ammo( unit, unit->supply_level );
        supplied = 1;
    }
    if ( unit_can_supply( unit, FUEL ) ) {
        give_fuel( unit, unit->supply_level );
        supplied = 1;
    }

    if ( supplied ) {
        unit->no_action_yet = 0;
        unit->cur_prop.mov = 0;
        unit->cur_prop.attack_count = 0;
    }

    return supplied;
}

/* add to units experience and compute exp_level (0-5) -- return 1 if level has changed */
int gain_exp( Unit *unit, int exp )
{
    int old_level = unit->exp_level;

    unit->exp += exp;
    if ( unit->exp >= MAX_EXP ) unit->exp = MAX_EXP;

    unit->exp_level = unit->exp / EXP_PER_LEVEL;

    return ( old_level != unit->exp_level );
}

/* can unit attack other unit? check range, ammo, etc */
/* DOES not check for dimplomatic relations! */
/* must be done manually */
int can_attack_unit( Unit *att, Unit *def, int type )
{
    if ( att == def ) return 0;
    if ( att->player_id == def->player_id ) return 0;
    if ( att->cur_prop.ammo <= 0 ) return 0;
    if ( get_dist( att->x, att->y, def->x, def->y ) > att->cur_prop.range ) return 0;
    /* checks depending on type */
    if ( type == ATTACK ) {
        /* att is the agressor that starts the attack */
        if ( att->cur_prop.attack[def->sel_prop->type] <= 0 ) return 0;
        if ( att->cur_prop.attack_count <= 0 ) return 0;
    }
    else {
        /* att is the defender who just wants to shoot back */
        if ( att->cur_prop.attack[def->sel_prop->type] == 0 ) return 0;
        /* an artillery can't defend itself on long range */
        if ( att->sel_prop->flags & ARTILLERY )
            if ( get_dist( att->x, att->y, def->x, def->y ) > 1 )
                return 0;
        /* if you want to defend against artillery attack you must be beside the artillery */
        if ( def->sel_prop->flags & ARTILLERY )
            if ( get_dist( att->x, att->y, def->x, def->y ) > 1 )
                return 0;
        /* an airplan that's not above the attacker can't defend itself again ground units */
        if ( att->sel_prop->flags & FLYING && !(def->sel_prop->flags & FLYING ) )
            if ( att->x != def->x || att->y != def->y )
                return 0;
    }
    return 1;
}

/* update life bar width */
void update_bar( Unit *unit )
{
    /* bar width */
    unit->damage_bar_width = ( MAX_DAMAGE - unit->damage) * BAR_TILE_WIDTH;
    if ( unit->damage_bar_width == 0 && unit->damage < MAX_DAMAGE )
        unit->damage_bar_width = BAR_TILE_WIDTH;

    /* bar color is defined by vertical offset in map->life_icons */
    if ( unit->damage < 6 )
        unit->damage_bar_offset = 0;
    else
        if ( unit->damage < 7 )
            unit->damage_bar_offset = BAR_TILE_HEIGHT;
        else
            unit->damage_bar_offset = BAR_TILE_HEIGHT * 2;
}

/* either return a random value between 50 to 100% of passed value or a predictive 75% depending
on type is REAL_COMBAT or DAMAGE_PRED */
int get_hits( int v, int type )
{
    if ( type == REAL_COMBAT )
        return RANDOM( v / 2, v );
    else
        return (int)( 0.75 * v );
}

/* a unit can take up to 10 damage points before destruction; this funcition returns how much
damage the defender takes (caused by the attacker); this is either done as prediction
(type == DAMAGE_PRED) of for real (type == REAL_COMBAT) */
int get_damage( Unit *att_unit, Unit *def_unit, int type )
{
    int damage; /* damage points taken by defender */
    int perc; /* attack relative to defence in percent */
    int att, def; /* modified attack and defence values */
    int close_combat = 0;

    /* check if attacker can attack */
    if ( att_unit->cur_prop.attack[def_unit->sel_prop->type] == 0 ) return 0; /* def won't take damage */

    /* set close combat */
    if ( get_dist( att_unit->x, att_unit->y, def_unit->x, def_unit->y ) <= 1 )
        close_combat = 1;

    /* setup and modify attacker and defender value (note: experience and entrenchment already
    modified basic attack and defence in update_cur_unit_prop() ) */
    /* suppressed units don't contribute to attack/defence */
    att = abs ( att_unit->cur_prop.attack[def_unit->sel_prop->type] )
          * ( MAX_DAMAGE - att_unit->damage - att_unit->sup )
          / MAX_DAMAGE;
    def = def_unit->cur_prop.def
          * ( MAX_DAMAGE - def_unit->damage - def_unit->sup )
          / MAX_DAMAGE;

    /* get attack value relative to defend value */
    /* whatever happens: 4 is minimum defence */
    if ( def < 4 ) def = 4;
    if ( att < 4 ) att = 4;
    perc = 100 * att / def;

    /* if the defender got more initiative attack decreases up to 20% * difference in close combat */
    if ( close_combat )
        if ( def_unit->cur_prop.ini > att_unit->cur_prop.ini ) {
            if ( type == DAMAGE_PRED )
                perc -= ( def_unit->cur_prop.ini - att_unit->cur_prop.ini ) * 5;
            else
                perc -= RANDOM( 0, def_unit->cur_prop.ini - att_unit->cur_prop.ini ) * 10;
        }

    /* percentage may change a bit in  real combat */
    if ( type == REAL_COMBAT )
        perc = ( RANDOM( 0, 45 ) + 75 ) * perc / 100;

    /* if this value is <= 100% then there is a small chance (<=50%) to cause 1 damage */
    if ( perc <= 100 ) {
        if ( type == DAMAGE_PRED )
            return 0;
        else
            return RANDOM( 1, att + def ) <= att;
    }

    /* convert additional percentage to damage points */
    /* if we got here the attack value is higher than the defence; the damage points are computed
    as follows: first dp takes 20%, each dp more takes additional 10% */
    perc -= 100;
    damage = 0;
    while  ( perc > 0 ) {
        perc -= damage * 10 + 20;
        damage++;
    }

    /* if damage is too high, reduce */
    if ( def_unit->damage + damage > MAX_DAMAGE )
        damage = MAX_DAMAGE - def_unit->damage;

    return damage;
}

/* add damage to unit and set killed-flag if too much damage */
void take_damage( Unit *unit, int damage )
{
    unit->damage += damage;
    if ( unit->damage >= MAX_DAMAGE ) {
        unit->damage = MAX_DAMAGE;
        unit->killed = 1; /* goooood night, baby! */
    }
}

/* execute combat between two units both attack each other if possible, take damages and
gain experience; also checks for rugged defence;
returns real damage take by both units */
void combat( Unit *att, Unit *def, int type, int *att_damage, int *def_damage )
{
    int rugged_def_chance;
    int sup;

    *att_damage = *def_damage = 0;

    /* check for rugged defence; may only occur if both units are close to each other */
    if ( get_dist( att->x, att->y, def->x, def->y ) == 1 ) {
        rugged_def_chance = def->rugged_def_chance;
        if ( rugged_def_chance > 18 ) rugged_def_chance = 18;
        if ( def->entr > 3 )
            if ( RANDOM( 1, 20 ) <= rugged_def_chance )
                def->rugged_def = 1;
    }

    /* will be restored in engine::action END_BATTLE */
    if ( def->rugged_def ) {
        /* reduce attackers defence to a third and skip his attack */
        att->cur_prop.def /= 3;
        /* no initiative */
        att->cur_prop.ini = 0;
    }

    /* compute damage */
    if ( !def->rugged_def )
        *def_damage = get_damage( att, def, REAL_COMBAT );
    /* defender might not be able to shoot back */
    if ( can_attack_unit( def, att, DEFEND ) )
        *att_damage = get_damage( def, att, REAL_COMBAT );
    /* take damage */
    /* if its defensive fire cut the defender damage is given as suppression */
    take_damage( att, *att_damage);
    if ( type == DEF_FIRE ) {
        if ( *def_damage > 1 ) {
            sup = *def_damage / 2;
        }
        else
            sup = 1;
        *def_damage /= 2;
        take_damage( def, *def_damage );
        def->sup += sup;
    }
    else
        take_damage( def, *def_damage );

    /* costs an attack if not ambush */
    if ( type != AMBUSH )
        att->cur_prop.attack_count--;

    /* both must supply */
    if ( config.supply ) {
        att->cur_prop.ammo--;
        if ( can_attack_unit( def, att, DEFEND ) )
            def->cur_prop.ammo--;
    }

    /* gain experience */
    gain_exp( att, (*def_damage) + (*att_damage) );
    gain_exp( def, (*att_damage) + (*def_damage) );

    /* defender loses entrenchment */
    if ( def->entr > 0 ) def->entr--;

    /* update damage bars */
    update_bar( att );
    update_bar( def );
}

/*
====================================================================
Check if these two units are allowed to merge with each other.
====================================================================
*/
int units_may_merge( Unit *unit1, Unit *unit2 )
{
    /* units must not be sea/air embarked */
    if ( unit1->sel_prop == &unit1->tran_prop ) {
         if ( unit1->tran_prop.flags & FLYING ) return 0;
         if ( unit1->tran_prop.flags & SWIMMING ) return 0;
    }
    if ( unit2->sel_prop == &unit2->tran_prop ) {
         if ( unit2->tran_prop.flags & FLYING ) return 0;
         if ( unit2->tran_prop.flags & SWIMMING ) return 0;
    }
    /* same class? */
    if ( unit1->prop.class != unit2->prop.class ) return 0;
    /* same player? */
    if ( unit1->player_id != unit2->player_id ) return 0;
    /* first unit must not have moved so far */
    if ( !unit1->no_action_yet ) return 0;
    /* both must be tracked, wheeled or legged */
    if ( ( unit1->prop.flags & TRACKED ) != ( unit2->prop.flags & TRACKED ) ) return 0;
    if ( ( unit1->prop.flags & WHEELED ) != ( unit2->prop.flags & WHEELED ) ) return 0;
    /* the unit strength counted together must not exceed the limit of 10 */
    if ( MAX_DAMAGE - unit1->damage - unit2->damage > 0 ) return 0;
    /* not failed so far: allow merge */
    return 1;
}

/*
====================================================================
Melt these two units: unit1 is the new unit and unit2 should be
removed from map and memory after this function was called.
====================================================================
*/
void merge_units( Unit *unit1, Unit *unit2 )
{
    /* units relative weight */
    float weight1, weight2;
    int total;
    int i, neg;

    /* compute weight */
    weight1 = MAX_DAMAGE - unit1->damage;
    weight2 = MAX_DAMAGE - unit2->damage;
    total = weight1 + weight2;
    /* adjust so weight1 + weigth2 = 1 */
    weight1 /= total; weight2 /= total;

    /* no other actions allowed */
    unit1->no_action_yet = 0;
    unit1->cur_prop.mov = 0;
    unit1->cur_prop.attack_count = 0;
    /* repair damage */
    unit1->damage -= MAX_DAMAGE - unit2->damage;
    /* lose entrenchment */
    unit1->entr = (int)( weight1 * unit1->entr );
    /* update experience */
    unit1->exp = (int)( weight1 * unit1->exp + weight2 * unit2->exp );
    /* update unit1::prop */
    /* related initiative */
    unit1->prop.ini = (int)( weight1 * unit1->prop.ini + weight2 * unit2->prop.ini );
    /* minimum movement */
    if ( unit2->prop.mov < unit1->prop.mov )
        unit1->prop.mov = unit2->prop.mov;
    /* maximum spotting */
    if ( unit2->prop.spot > unit1->prop.spot )
        unit1->prop.spot = unit2->prop.spot;
    /* maximum range */
    if ( unit2->prop.range > unit1->prop.range )
        unit1->prop.range = unit2->prop.range;
    /* relative attack count */
    unit1->prop.attack_count = (int)( weight1 * unit1->prop.attack_count + weight2 * unit2->prop.attack_count );
    if ( unit1->prop.attack_count == 0 ) unit1->prop.attack_count = 1;
    /* relative attacks */
    /* if attack is negative simply use absolute value; only restore negative if both units are negative */
    for ( i = 0; i < unit1->prop.unit_def->type_count; i++ ) {
        neg = unit1->prop.attack[i] < 0 && unit2->prop.attack[i] < 0;
        unit1->prop.attack[i] = (int)( weight1 * abs( unit1->prop.attack[i] ) + weight2 * ( unit2->prop.attack[i] ) );
        if ( neg ) unit1->prop.attack[i] *= -1;
    }
    /* relative defence */
    unit1->prop.def = (int)( weight1 * unit1->prop.def + weight2 * unit2->prop.def );
    /* relative ammo */
    unit1->prop.ammo = (int)( weight1 * unit1->prop.ammo + weight2 * unit2->prop.ammo );
    unit1->cur_prop.ammo = (int)( weight1 * unit1->cur_prop.ammo + weight2 * unit2->cur_prop.ammo );
    /* relative fuel */
    unit1->prop.fuel = (int)( weight1 * unit1->prop.fuel + weight2 * unit2->prop.fuel );
    unit1->cur_prop.fuel = (int)( weight1 * unit1->cur_prop.fuel + weight2 * unit2->cur_prop.fuel );
    /* mergeed flags */
    unit1->prop.flags |= unit2->prop.flags;
    /* sounds, picture are kept */
    /* unit1::trans_prop isn't updated so far: */
    /* transporter of first unit is kept if any else second unit's transporter is used */
    if ( !unit1->tran_prop_set && unit2->tran_prop_set ) {
        memcpy( &unit1->tran_prop, &unit2->tran_prop, sizeof( Unit_Lib_Entry ) );
        unit1->tran_prop_set = 1;
        /* as this must be a ground transporter copy current fuel value */
        unit1->cur_prop.fuel = unit2->cur_prop.fuel;
    }
    /* update current properties */
    update_cur_unit_prop( unit1 );
    update_bar( unit1 );
}
