package tests::ReportParserLaTeXDocBookFormatterTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::ReportParser::LaTeXDocBookFormatter qw/ dbk2latex /;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_List {
    my $self = $_[0];

    my $latex = dbk2latex( <<EOF );
<itemizedlist>
  <title>This is an itemized list</title>
  <para>This list is rendered through the itemize list environment.</para>
  <listitem><para>First item.</para></listitem>
  <listitem><para>Second item.</para></listitem>
  <listitem><para>Third item.</para>
            <para>With two paragraphs.</para>
  </listitem>
  <listitem><para>Last item.</para></listitem>
</itemizedlist>
EOF

    $self->assert_str_equals( <<EOF, $latex );
\\large{\\textbf{This is an itemized list}}

\\nopagebreak[3]

This list is rendered through the itemize list environment.

\\nopagebreak[2]

\\begin{itemize}
\\item First item.

\\item Second item.

\\item Third item.

With two paragraphs.

\\item Last item.

\\end{itemize}

EOF
}

sub test_NamedItem {
    my $self = $_[0];

    my $latex = dbk2latex( <<EOF );
<variablelist>
  <varlistentry>
    <term>First term</term>
    <listitem><para>First definition.</para></listitem>
  </varlistentry>
  <varlistentry>
    <term>Second term</term>
    <listitem><para>Second definition.</para>
            <para>Using two paragraphs.</para>
    </listitem>
  </varlistentry>
</variablelist>
EOF

    $self->assert_str_equals( <<EOF, $latex );
\\begin{description}
\\item[First term] First definition.

\\item[Second term] Second definition.

Using two paragraphs.

\\end{description}

EOF
}

sub test_Para {
    my $self = $_[0];

    my $latex = dbk2latex( <<EOF );
<para>Paragraphs
   content\n is wrapped to 72 columns and ends with
 an extra newline.

The paragraph original white space is <emphasis>ignored</emphasis> and LaTex 
characters like '#' and '\\' are escaped.
</para>
EOF

    my $e_text = <<EOF;
Paragraphs content is wrapped to 72 columns and ends with an extra
newline. The paragraph original white space is \\emph{ignored} and LaTex
characters like '\\#' and '\$\\backslash\$' are escaped. 

EOF

    $self->assert_str_equals( $e_text, $latex );
}

sub test_Para_block_layout {
    my $self = $_[0];

    my $latex = dbk2latex( <<EOF );
<para>This is a paragraph which contain an admonition:

<important><para>This is an admonition.</para></important>

And this finishes the paragraph.</para>
EOF

    my $e_text = <<EOF;
This is a paragraph which contain an admonition: 

\\begin{quote}
\\large{\\textbf{Important:}}
This is an admonition.

\\end{quote}

\\noindent
 And this finishes the paragraph.

EOF
    $self->assert_str_equals( $e_text, $latex );
}

sub test_Admonition {
    my $self = $_[0];

    my $latex = dbk2latex( <<EOF );
<warning>
  <title>Perl can be hazardous to your mental health</title>
  <para>Too much perl hacking can cause headaches, depression and/or
     anxiety attacks.</para>
</warning>
EOF

    $self->assert_str_equals( <<EOF, $latex );
\\begin{quote}
\\large{\\textbf{Perl can be hazardous to your mental health}}
Too much perl hacking can cause headaches, depression and/or anxiety
attacks.

\\end{quote}

EOF
    $latex = dbk2latex( <<EOF );
<note>
  <para>Of course this was a joke.</para>
</note>
EOF

    $self->assert_str_equals( <<EOF, $latex );
\\begin{quote}
\\large{\\textbf{Note:}}
Of course this was a joke.

\\end{quote}

EOF
}

sub test_Href {
    my $self = $_[0];

    my $dbk_text = '<para>Go to <ulink url="http://www.logreport.org/">The LogReport Foundation</ulink> website.</para>';
    my $e_text = <<EOF;
Go to \\href{http://www.logreport.org/}{The LogReport Foundation}
website.

EOF
    $self->assert_str_equals( $e_text, dbk2latex( $dbk_text ) );
    return;
}

sub test_Email {
    my $self = $_[0];

    my $dbk_text = '<para>An <email>email@address.com</email>.</para>';
    my $e_text = <<EOF;
An \\href{mailto:email\@address.com}{email\@address.com}.

EOF
    $self->assert_str_equals( $e_text, dbk2latex( $dbk_text ) );
}

sub test_Superscript {
    my $self = $_[0];

    $self->assert_str_equals( "A \\raisebox{1ex}{superscript}.\n\n",
                              dbk2latex( '<para>A <superscript>superscript</superscript>.</para>') );
}

sub test_Subscript {
    my $self = $_[0];

    $self->assert_str_equals( "A \\raisebox{-1ex}{subscript}.\n\n",
                              dbk2latex( '<para>A <subscript>subscript</subscript>.</para>') );
}

sub test_TextTT {
    my $self = $_[0];

    $self->assert_str_equals( "Run the \\texttt{lire} command.\n\n",
                              dbk2latex( '<para>Run the <command>lire</command> command.</para>') );
}

sub test_Slanted {
    my $self = $_[0];

    $self->assert_str_equals( "A \\textsl{variable}.\n\n",
                              dbk2latex( '<para>A <varname>variable</varname>.</para>') );
}

1;
