/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file: alberta.h                                                          */
/*                                                                          */
/*                                                                          */
/* description: public header file of the ALBERTA package                   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*             Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.alberta-fem.de                                               */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2005)                          */
/*  (c) by D. Koester (2002-2005)                                           */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*  Global configuration header                                             */
/*--------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H        /* only defined during build of ALBERTA libs */
#include <config.h>
#endif

/*--------------------------------------------------------------------------*/
/*  Header-File for ALBERTA utilities                                       */
/*--------------------------------------------------------------------------*/
#ifndef _ALBERTA_H_
#define _ALBERTA_H_

#include "alberta_util.h"

#ifdef __cplusplus
extern "C" {
#endif

#define ALBERTA_VERSION  "ALBERTA: Version 2.0"

/*--------------------------------------------------------------------------*/
/*  Definition of the space dimension and of parameters depending on the    */
/*  space dimension:							    */
/*									    */
/*  DIM_OF_WORLD:   space dimension					    */
/*                                    					    */
/*  The ?D-suffix signals different simplex dimensions (formerly ==DIM).    */
/*--------------------------------------------------------------------------*/

#ifndef DIM_OF_WORLD
# error DIM_OF_WORLD UNDEFINED
#endif

#ifndef ALBERTA_DEBUG
#warning ALBERTA_DEBUG WAS NOT DEFINED! DEFAULTING TO 0.
#define ALBERTA_DEBUG 0
#endif

/* The master dimension limit */
#define DIM_MAX          3
  
/* Various constants for dimension dependent geometrical quantities */
#define N_VERTICES(DIM) ((DIM)+1)
#define N_EDGES(DIM)    ((DIM)*((DIM)+1)/2)
#define N_WALLS(DIM)    ((DIM)+1) /* number of codim 1 subsimplexes */
#define N_FACES(DIM)    (((DIM) == 3) * N_WALLS(DIM))
#define N_NEIGH(DIM)    (((DIM) != 0) * N_WALLS(DIM))
#define DIM_FAC(DIM)    ((DIM) < 2 ? 1 : (DIM) == 2 ? 2 : 6)

#define N_VERTICES_0D    N_VERTICES(0)
#define N_EDGES_0D       N_EDGES(0)
#define N_FACES_0D       N_FACES(0)
#define N_NEIGH_0D       N_NEIGH(0)
#define N_WALLS_0D       N_WALLS(0)

#define N_VERTICES_1D    N_VERTICES(1)
#define N_EDGES_1D       N_EDGES(1)
#define N_FACES_1D       N_FACES(1)
#define N_NEIGH_1D       N_NEIGH(1)
#define N_WALLS_1D       N_WALLS(1)

#define N_VERTICES_2D    N_VERTICES(2)
#define N_EDGES_2D       N_EDGES(2)
#define N_FACES_2D       N_FACES(2)
#define N_NEIGH_2D       N_NEIGH(2)
#define N_WALLS_2D       N_WALLS(2)

#define N_VERTICES_3D    N_VERTICES(3)
#define N_EDGES_3D       N_EDGES(3)
#define N_FACES_3D       N_FACES(3)
#define N_NEIGH_3D       N_NEIGH(3)
#define N_WALLS_3D       N_WALLS(3)

#define N_VERTICES_MAX   N_VERTICES(DIM_MAX)
#define N_EDGES_MAX      N_EDGES(DIM_MAX)
#define N_FACES_MAX      N_FACES(DIM_MAX)
#define N_NEIGH_MAX      N_NEIGH(DIM_MAX)
#define N_WALLS_MAX      N_WALLS(DIM_MAX)

/* N_LAMBDA is the maximal number of barycentric coordinates, formerly DIM+1 */
/* Barycentric coords are often initialized using constant values, therefore */
/* we set this to 4.							     */
#define N_LAMBDA 4
#define LAMBDA_MAX N_LAMBDA

typedef REAL            REAL_B[N_LAMBDA];
typedef REAL_B          REAL_BB[N_LAMBDA];
typedef REAL            REAL_D[DIM_OF_WORLD];
typedef REAL_D          REAL_DD[DIM_OF_WORLD];
typedef REAL_D          REAL_BD[N_LAMBDA];
typedef struct real_dds REAL_DDS;

struct real_dds {
  REAL_D row0;
  REAL   row1[DIM_OF_WORLD-1]; /* zero size array in case of DOW == 1 */
#if DIM_OF_WORLD > 1
  REAL   row2[DIM_OF_WORLD-2]; /* zero size array in case of DOW == 2 */
#else
  REAL   row2[0];
#endif
};
  
/*--------------------------------------------------------------------------*/
/*  some useful macros depending on DIM_OF_WORLD                            */
/*--------------------------------------------------------------------------*/

#include "alberta_inlines.h"

/*--------------------------------------------------------------------------*/
/*  access to element index via element or element_info structure           */
/*--------------------------------------------------------------------------*/

#if ALBERTA_DEBUG
#define INDEX(el)   ((el) ? (el)->index : -1)
#else
#define INDEX(el)   -1
#endif

/*--------------------------------------------------------------------------*/
/*  access to leaf data (only for leaf elements)                            */
/*--------------------------------------------------------------------------*/

#define IS_LEAF_EL(el) (!(el)->child[0])
#define LEAF_DATA(el)  ((void *)(el)->child[1])

/*--------------------------------------------------------------------------*/
/*  boundary types							    */
/*--------------------------------------------------------------------------*/

#define INTERIOR      0
#define DIRICHLET     1
#define NEUMANN       -1

#define IS_NEUMANN(bound) ((bound) <= NEUMANN)
#define IS_DIRICHLET(bound) ((bound) >= DIRICHLET)
#define IS_INTERIOR(bound) ((bound) == 0)

/*--------------------------------------------------------------------------*/
/*  node types (indices in n_dof[] vectors, e.g.)			    */
/*--------------------------------------------------------------------------*/

#define N_NODE_TYPES  (DIM_MAX+1)

#define VERTEX     0
#define CENTER     1
#define EDGE       2
#define FACE       3

/*--------------------------------------------------------------------------*/
/*  basic types of the grid						    */
/*--------------------------------------------------------------------------*/

typedef signed int              DOF;
typedef struct el               EL;
typedef struct macro_el         MACRO_EL;
typedef struct el_info          EL_INFO;
typedef struct rc_list_el       RC_LIST_EL;
typedef struct mesh             MESH;

typedef struct parametric       PARAMETRIC;
typedef struct traverse_stack   TRAVERSE_STACK;

typedef struct adapt_stat       ADAPT_STAT;
typedef struct adapt_instat     ADAPT_INSTAT;

#ifndef DOF_ADMIN_DEF
typedef struct dof_admin        DOF_ADMIN;
typedef struct dof_int_vec      DOF_INT_VEC;
typedef struct dof_dof_vec      DOF_DOF_VEC;
typedef struct dof_uchar_vec    DOF_UCHAR_VEC;
typedef struct dof_schar_vec    DOF_SCHAR_VEC;
typedef struct dof_real_vec     DOF_REAL_VEC;
typedef struct dof_real_d_vec   DOF_REAL_D_VEC;
typedef struct dof_ptr_vec      DOF_PTR_VEC;
typedef struct matrix_row       MATRIX_ROW;
typedef struct dof_matrix       DOF_MATRIX;
typedef struct dowb_matrix_row  DOWB_MATRIX_ROW;
typedef struct dof_dowb_matrix  DOF_DOWB_MATRIX;
#endif

typedef struct bas_fcts         BAS_FCTS;
typedef struct fe_space         FE_SPACE;

typedef struct quadrature       QUAD;
typedef struct quadrature       QUADRATURE;
typedef struct quad_fast        QUAD_FAST;

typedef struct macro_data       MACRO_DATA;
typedef struct node_projection  NODE_PROJECTION;

typedef struct list_node        LIST_NODE;
typedef struct traverse_hook    TRAVERSE_HOOK;
typedef void   (*TRAVERSE_HOOK_FCT)(const EL_INFO *el_info, void *hook_data);

/* Traverse-hook node. See also add_traverse_hook(),
 * remove_traverse_hook(). Traverse-Hooks are special hooks
 * dynamically added and removed from the mesh. The hooks are run
 * during mesh-traversal prior to calling the element function. They
 * are intended to hide certain details from the application program,
 * e.g. to modify basis functions on a per-element basis
 * (hp-methods?). In principle, the init_element() of the PARAMETRIC
 * structure could go into a traverse hook.
 */

struct list_node {
  LIST_NODE *next;
  LIST_NODE *prev;
};

#define LIST_ENTRY(node, type, nodename)				\
  ((type *)((ptrdiff_t)node - (ptrdiff_t)&((type *)0)->nodename))

struct traverse_hook
{
  TRAVERSE_HOOK_FCT function;
  void              *data;    /* Per hook data. */
  LIST_NODE         node; /* doubly-linked list for efficient add and remove */
};

/*--------------------------------------------------------------------------*/
/*  node projection descriptor:                                             */
/*  a function pointer which calculates the projected location of a new     */
/*  vertex resulting from refinement.                                       */
/*               curved boundary					    */
/*  type:        one of INTERIOR : DIRICHLET : NEUMANN			    */
/*--------------------------------------------------------------------------*/

struct node_projection
{
  void (*func)(REAL_D old_coord, const EL_INFO *eli, const REAL_B lambda);
};

/*--------------------------------------------------------------------------*/
/*  one single element (triangle) of the grid:			            */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  position of the nodes in 1d:                                            */
/*                                                                          */
/*  0 _____ 1   or  0 _____ 1                                               */
/*                      2                                                   */
/*                                                                          */
/*                          child[0]     child[1]                           */
/*  refinement:  0 _____ 1    0 ___ 1  0 ___ 1                              */
/*                               2        2                                 */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  position of the nodes in 2d                                             */
/*         2                 2                   2                 2        */
/*        /\       or       /\        or        /\       or       /\        */
/*       /  \             4/  \ 3              /  \             4/  \ 3     */
/*      /    \            /    \              /  3 \            /  6 \      */
/*    0/______\1        0/______\1          0/______\1        0/______\1    */
/*                           5                                     5        */
/*                                                                          */
/*  refinement:           2          child[0]  0    1   child[1]            */
/*                       /\                   /|    |\                      */
/*                     4/  \ 3  -->         5/ |4  3| \ 5                   */
/*                     /  6 \               /6 |    |6 \                    */
/*                   0/______\1           1/___|    |___\0                  */
/*                        5                  3  2  2  4                     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  3d refinement: vertex numbering after (Baensch +) Kossaczky             */
/*                                                                          */
/*  edges:                                                                  */
/*  E0:  between V0, V1                                                     */
/*  E1:  between V0, V2                                                     */
/*  E2:  between V0, V3                                                     */
/*  E3:  between V1, V2                                                     */
/*  E4:  between V1, V3                                                     */
/*  E5:  between V2, V3                                                     */
/*                                                                          */
/*  Always edge 0 (between vertices 0 and 1) is bisected.                   */
/*                                                                          */
/*                                        V1                                */
/*                                      -+                                  */
/*                                 ----- ||                                 */
/*                     E0    ------      | |                                */
/*                     ------           |   | E3                            */
/*               ------                 |    |                              */
/*         ------                      |      |                             */
/*     V0 +. . . . . . . . . . . . . . | . . . |                            */
/*         ---               (E1)     |         +  V2                       */
/*            ---                     |         /                           */
/*               ---                 |E4      /                             */
/*                  ---              |      /                               */
/*                 E2  ---          |     / E5                              */
/*                        ---       |   /                                   */
/*                           ---   |  /                                     */
/*                              ---|/                                       */
/*                                 +                                        */
/*                                 V3                                       */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*  child:        pointers to the two children of the element		    */
/*                if (child[0]==child[1]==nil) element is a leaf of the     */
/*                        tree                                              */
/*  dof:          vector of pointers to dof vectors :-)                     */
/*  new_coord:    in case of curved boundary, coords of ref.edge midpoint   */
/*  index:        global element index (only for test purposes)		    */
/*  mark:         element is a leaf:   					    */
/*                   mark == 0         do not refine/coarsen		    */
/*                   mark > 0          refine (mark times)                  */
/*                   mark < 0          may be coarsened (mark times)        */
/*--------------------------------------------------------------------------*/

struct el
{
  EL         *child[2];
  DOF        **dof;
  S_CHAR     mark;
  REAL       *new_coord;

#if ALBERTA_DEBUG
  int        index;
#endif
};

/*--------------------------------------------------------------------------*/
/*   child_vertex_3d[el_type][child][i] =                                   */
/*       parent's local vertex index of new vertex i                        */
/*       4 stands for the newly generated vertex                            */
/*--------------------------------------------------------------------------*/
/*   child_edge_3d[el_type][child][i] =                                     */
/*       parent's local edge index of new edge i                            */
/*       new edge 2 is half of old edge 0,                                  */
/*       new edges 4,5 are really new edges, and value is different:        */
/*         child_edge_3d[][][4,5] = index of same edge in other child	    */
/*--------------------------------------------------------------------------*/
/*  vertex_of_edge_?d[edge][i], i = 1,2 are the two vertices of edge        */
/*--------------------------------------------------------------------------*/
extern int vertex_of_edge_2d[3][2];              /* defined in traverse_r.c */

extern int child_vertex_3d[3][2][4];             /* defined in traverse_r.c */
extern int child_edge_3d[3][2][6];               /* defined in traverse_r.c */
extern int vertex_of_edge_3d[6][2];              /* defined in traverse_r.c */

/*--------------------------------------------------------------------------*/
/*  edge_of_dofs_3d[i][j]: gives the local index of edge with vertices i, j */
/*--------------------------------------------------------------------------*/
extern U_CHAR  edge_of_dofs_3d[4][4];               /* defined in refine.c  */

/*--------------------------------------------------------------------------*/
/* PARAMETRIC structure, entry in MESH structure                            */
/* description of parametric meshes and elements                            */
/*--------------------------------------------------------------------------*/

struct parametric 
{
  char *name;     /* textual description analogous to BAS_FCTS. */
  int (*init_element)(const EL_INFO *info, const PARAMETRIC *parametric);
  /* init_el = 0 : non-parametric element */
  /* init_el = 1 : parametric element     */

  void (*coord_to_world)(const EL_INFO *info, const QUAD *quad,
			 int n, const REAL lambda[][N_LAMBDA], REAL_D *world);
  int (*world_to_coord)(const EL_INFO *info,
			int n, const REAL_D world[], REAL lambda[][N_LAMBDA]);
  void (*det)(const EL_INFO *info, const QUAD *quad,
	      int n, const REAL lambda[][N_LAMBDA], REAL dets[]);
  void (*grd_lambda)(const EL_INFO *info,  const QUAD *quad, 
		     int n, const REAL lambda[][N_LAMBDA],
		     REAL_D grd_lam[][N_LAMBDA], REAL dets[]);

  int not_all;             /* true:  some elements may be non-parametric */
  int use_reference_mesh;  /* true: standard routines coord_to_world, etc. */
                           /* may be used to get data about the reference  */
                           /* triangulation. Set to "false" by default.    */

  /* private entries */
  void (*inherit_parametric)(MESH *slave);
  void *data;
};

/*--------------------------------------------------------------------------*/
/*  additional information to elements during hierarchy traversal	    */
/*--------------------------------------------------------------------------*/
/*  mesh:          pointer to the mesh structure                      	    */
/*  coord:         world coordinates of the nodes                     	    */
/*                 parametric case is handled via dof_vector          	    */
/*  vertex_bound:  boundary type of the vertices                 	    */
/*  edge_bound:    boundary type of the edges                    	    */
/*  face_bound:    boundary type of the faces                    	    */
/*  projection:    node projection function for the new vertex which would  */
/*                 result from a refinement of the current element.         */
/*  neigh:         pointer to the adjacent elements       		    */
/*                 nil-pointer for a part of the boundary		    */
/*  opp_coord:     world coordinates of opposite vertices 		    */
/*  opp_vertex:    local indices of opposite vertices       		    */
/*  orientation:   orientation of the tetrahedron (+1|-1) (only 3d)	    */
/*--------------------------------------------------------------------------*/

struct el_info
{
  MESH                *mesh;
  REAL_D               coord[N_VERTICES_MAX];
  const MACRO_EL      *macro_el;
  EL                  *el, *parent;
  FLAGS                fill_flag;

  S_CHAR               vertex_bound[N_VERTICES_MAX];
  S_CHAR               edge_bound[N_EDGES_MAX];
  S_CHAR               face_bound[N_FACES_MAX];

  NODE_PROJECTION     *active_projection;
  NODE_PROJECTION     *projections[N_NEIGH_MAX + 1];
  int                  level;

  EL                  *neigh[N_NEIGH_MAX];
  U_CHAR               opp_vertex[N_NEIGH_MAX];
  U_CHAR               el_type;
  REAL_D               opp_coord[N_NEIGH_MAX];

  S_CHAR               orientation;
};

/* A macro for accessing the boundary type of the i-th wall,
 * i.e. codim one sub-simplex.
 */
#define WALL_BOUND(eli, dim, i)			\
  ((dim) == 1					\
   ? (eli)->vertex_bound[i]			\
   : ((dim) == 2				\
      ? (eli)->edge_bound[i]			\
      : (eli)->face_bound[i]))

/*--------------------------------------------------------------------------*/
/* RC_LIST_EL structure to describe a refinement/coarsening patch.          */
/* el_info:        contains information about the patch element. This is not*/
/*                 a pointer since EL_INFO structures are often overwritten */
/*                 during mesh traversal.                                   */
/* no:             index of the patch element in the patch.                 */
/* flag:           for coarsening: true iff the coarsening edge of the      */
/*                 patch element is the coarsening edge of the patch.       */
/* neigh:          neighbours to the right/left in the orientation of the   */
/*                 edge, or nil pointer for a boundary face. (dim == 3 only)*/
/* opp_vertex:     the opposite vertex of neigh[0/1]. (dim == 3 only)       */
/*--------------------------------------------------------------------------*/

struct rc_list_el
{
  EL_INFO      el_info;
  int          no;
  int          flag;
  RC_LIST_EL   *neigh[2];
  int          opp_vertex[2];
};

/*--------------------------------------------------------------------------*/
/*  flags, which information should be present in the EL_INFO structure     */
/*--------------------------------------------------------------------------*/

#define FILL_NOTHING            0x00L
#define FILL_COORDS             0x01L
#define FILL_BOUND              0x02L
#define FILL_NEIGH              0x04L
#define FILL_OPP_COORDS         0x08L
#define FILL_ORIENTATION        0x10L
#define FILL_EL_TYPE            0x20L
#define FILL_PROJECTION         0x40L

#define FILL_ANY(mesh)							\
  (0x01L|0x02L|0x04L|0x08L|0x10L|0x20L|0x40L|(mesh)->active_hooks)

/* Hook-queue flags */
#define HOOK_QUEUE_OFFSET 24

enum hook_queue_enum {
  HOOK_QUEUE_0 = 0,
  BAS_FCTS_HOOK_QUEUE = HOOK_QUEUE_0,
  HOOK_QUEUE_1,
  HOOK_QUEUE_2,
  HOOK_QUEUE_3,
  HOOK_QUEUE_4,
  HOOK_QUEUE_5,
  HOOK_QUEUE_6,
  HOOK_QUEUE_7,
};
/* Note: definition of enum must come before typedef for C++ compilation. */
typedef enum hook_queue_enum    HOOK_QUEUE_ENUM;

#define RUN_HOOKS_0 (1UL << HOOK_QUEUE_0)
#define RUN_BAS_FCTS_HOOKS RUN_HOOKS_0
#define RUN_HOOKS_1 (1UL << HOOK_QUEUE_1)
#define RUN_HOOKS_2 (1UL << HOOK_QUEUE_2)
#define RUN_HOOKS_3 (1UL << HOOK_QUEUE_3)
#define RUN_HOOKS_4 (1UL << HOOK_QUEUE_4)
#define RUN_HOOKS_5 (1UL << HOOK_QUEUE_5)
#define RUN_HOOKS_6 (1UL << HOOK_QUEUE_6)
#define RUN_HOOKS_7 (1UL << HOOK_QUEUE_7)

/*--------------------------------------------------------------------------*/
/*  flags for mesh traversal                                                */
/*--------------------------------------------------------------------------*/

#define CALL_EVERY_EL_PREORDER  0x0100L
#define CALL_EVERY_EL_INORDER   0x0200L
#define CALL_EVERY_EL_POSTORDER 0x0400L
#define CALL_LEAF_EL            0x0800L
#define CALL_LEAF_EL_LEVEL      0x1000L
#define CALL_EL_LEVEL           0x2000L
#define CALL_MG_LEVEL           0x4000L    /* used in multigrid methods */

#define TEST_FLAG(flags, el_info) \
        TEST_EXIT(!((((el_info)->fill_flag)^(flags)) & (flags)),\
                 "flag %d not set\n", (flags))

#if ALBERTA_DEBUG==1
# define DEBUG_TEST_FLAG(flags, el_info) \
                        if((((el_info)->fill_flag)^(flags)) & (flags)) \
                          print_error_funcname(funcName, __FILE__, __LINE__),\
                          print_error_msg_exit("flag %d not set\n", (flags))
#else
# define DEBUG_TEST_FLAG(flags, el_info) do { funcName = funcName; } while (0)
#endif
/*--------------------------------------------------------------------------*/
/*  one single element of the macro triangulation:			    */
/*--------------------------------------------------------------------------*/
/*  el:            pointer to the element data of the macro element         */
/*  coord:         world coordinates of the nodes on the macro element	    */
/*  vertex_bound:  boundary type of the vertices                 	    */
/*  edge_bound:    boundary type of the edges                    	    */
/*  face_bound:    boundary type of the faces                    	    */
/*  projection:    possible node projection functions for all nodes [0]     */
/*                 or for specific edges or faces (dim > 1), which will     */
/*                 override entry [0].                                      */
/*  index:         unique global index of macro element                     */
/*  neigh:         pointer to the adjacent macro elements		    */
/*                 nil-pointer for a part of the boundary		    */
/*  opp_vertex:    local indices of opposite vertices       		    */
/*  el_type:       type of corresponding element.           		    */
/*  orientation:   orientation of corresponding element.    		    */
/*--------------------------------------------------------------------------*/

struct macro_el
{
  EL               *el;
  REAL             *coord[N_VERTICES_MAX];

  S_CHAR           vertex_bound[N_VERTICES_MAX];
  S_CHAR           edge_bound[N_EDGES_MAX];
  S_CHAR           face_bound[N_FACES_MAX];

  NODE_PROJECTION *projection[N_NEIGH_MAX + 1];

  int              index;

  MACRO_EL        *neigh[N_NEIGH_MAX];
  S_CHAR           opp_vertex[N_NEIGH_MAX];

  U_CHAR           el_type;
  S_CHAR           orientation;
};

/*--------------------------------------------------------------------------*/
/*---  index based storage of macro triangulations                       ---*/
/*--------------------------------------------------------------------------*/

struct macro_data
{
  int dim;                          /* dimension of the elements            */

  int n_total_vertices;
  int n_macro_elements;

  REAL_D *coords;                   /* Length will be n_total_vertices      */ 

  int *mel_vertices;                /* mel_vertices[i*n_vertices[dim]+j]:   */
                                /* global index of jth vertex of element i  */ 

  int *neigh;                      /* neigh[i*n_neigh[dim]+j]:              */
                            /* neighbour j of element i or -1 at boundaries */

  S_CHAR *boundary;                /* boundary[i*n_neigh[dim]+j]:           */
                      /* boundary type of jth vertex/edge/face of element i */ 
  /* WARNING: In 1D the local index 0 corresponds to vertex 1 & vice versa! */
  /* (Consistent with macro_data.neigh)                                     */

  U_CHAR *el_type;                  /* el_type[i]: type of element i        */
                                    /* only used in 3d!                     */
};

#ifndef DOF_ADMIN_DEF
#define DOF_ADMIN_DEF

/*--------------------------------------------------------------------------*/
/*  dof handling                 				            */
/*--------------------------------------------------------------------------*/
#define DOF_FREE_SIZE 32
#define DOF_UNIT_ALL_FREE 0xFFFFFFFF
typedef unsigned int DOF_FREE_UNIT;
extern  DOF_FREE_UNIT dof_free_bit[DOF_FREE_SIZE];        /* in dof_admin.c */

#define FOR_ALL_DOFS(admin, todo) \
  if ((admin)->hole_count == 0) { int dof; \
  for (dof = 0; dof < (admin)->used_count; dof++) { todo; } } \
      else { DOF_FREE_UNIT _dfu, *_dof_free = (admin)->dof_free; \
        int _i, _ibit, dof=0; \
        int _n= ((admin)->size_used + DOF_FREE_SIZE-1) / DOF_FREE_SIZE; \
        for (_i = 0; _i < _n; _i++) { \
	  if ((_dfu = _dof_free[_i])) { \
	    if (_dfu == DOF_UNIT_ALL_FREE) dof += DOF_FREE_SIZE; \
	    else for (_ibit = 0; _ibit < DOF_FREE_SIZE; _ibit++) { \
               if (!(_dfu & dof_free_bit[_ibit])) { todo; } dof++; } } \
          else { \
            for (_ibit = 0; _ibit < DOF_FREE_SIZE; _ibit++){ todo; dof++;}}}}

#define FOR_ALL_FREE_DOFS(admin, todo) \
      if ((admin)->hole_count == 0) { int dof; \
        for (dof = (admin)->used_count; \
	     dof < (admin)->size; dof++)   { todo; } } \
      else { DOF_FREE_UNIT _dfu, *_dof_free = (admin)->dof_free; \
        int _i, _ibit, dof=0; \
        int _n= ((admin)->size + DOF_FREE_SIZE-1) / DOF_FREE_SIZE; \
        for (_i = 0; _i < _n; _i++) { \
	  if ((_dfu = _dof_free[_i])) { \
	    if (_dfu == DOF_UNIT_ALL_FREE) \
              for (_ibit = 0 ; _ibit < DOF_FREE_SIZE; _ibit++)\
                { todo; dof++; } \
	    else for (_ibit = 0; _ibit < DOF_FREE_SIZE; _ibit++) { \
                   if (_dfu & dof_free_bit[_ibit]) { todo; } dof++; } } \
	  else dof += DOF_FREE_SIZE; }}

struct dof_admin
{
  MESH         *mesh;
  const char   *name;

  DOF_FREE_UNIT *dof_free;    /* flag bit vector                           */
  unsigned int  dof_free_size;/* flag bit vector size                      */
  unsigned int  first_hole;   /* index of first non-zero dof_free entry    */

  U_CHAR        preserve_coarse_dofs; /* preserve non-leaf DOFs or not     */

  DOF  size;                 /* allocated size of dof_list vector          */
  DOF  used_count;           /* number of used dof indices                 */
  DOF  hole_count;           /* number of FREED dof indices (NOT size-used)*/
  DOF  size_used;            /* > max. index of a used entry               */

  int  n_dof[N_NODE_TYPES];  /* dofs from THIS dof_admin                   */
  int  n0_dof[N_NODE_TYPES]; /* dofs from previous dof_admins              */

  DOF_INT_VEC     *dof_int_vec;           /* linked list of int vectors    */
  DOF_DOF_VEC     *dof_dof_vec;           /* linked list of dof vectors    */
  DOF_DOF_VEC     *int_dof_vec;           /* linked list of dof vectors    */
  DOF_UCHAR_VEC   *dof_uchar_vec;         /* linked list of u_char vectors */
  DOF_SCHAR_VEC   *dof_schar_vec;         /* linked list of s_char vectors */
  DOF_REAL_VEC    *dof_real_vec;          /* linked list of real vectors   */
  DOF_REAL_D_VEC  *dof_real_d_vec;        /* linked list of real_d vectors */
  DOF_PTR_VEC     *dof_ptr_vec;           /* linked list of void * vectors */
  DOF_MATRIX      *dof_matrix;            /* linked list of matrices       */
  DOF_DOWB_MATRIX *dof_dowb_matrix;       /* linked list of block matrices */

/*--------------------------------------------------------------------------*/
/*---  pointer for administration; don't touch!                          ---*/
/*--------------------------------------------------------------------------*/

  void            *mem_info;
};


/*--------------------------------------------------------------------------*/
/*  dof vector structures            					    */
/*--------------------------------------------------------------------------*/
/*  next:        pointer to next structure containing vector of same type   */
/*  fe_space:    pointer to fe_space  structure                             */
/*  refine_interpol: dof interpolation during refinement                    */
/*  coarse_restrict: restriction of linear functionals evaluated on a finer */
/*                   grid and stored in dof vector to the coarser grid      */
/*                   during coarsening                                      */
/*                or dof interpolation during coarsening                    */
/*  size:        allocated size of vector                         	    */
/*  vec[]:    vector entries (entry is used if dof index is used)	    */
/*--------------------------------------------------------------------------*/

struct dof_int_vec
{
  DOF_INT_VEC    *next;
  const FE_SPACE *fe_space;

  const char     *name;

  DOF            size;
  int            *vec;

  void  (*refine_interpol)(DOF_INT_VEC *, RC_LIST_EL *, int n);
  void  (*coarse_restrict)(DOF_INT_VEC *, RC_LIST_EL *, int n);

/*--------------------------------------------------------------------------*/
/*---  pointer for administration; don't touch!                          ---*/
/*--------------------------------------------------------------------------*/
  void            *mem_info;

};

struct dof_dof_vec
{
  DOF_DOF_VEC    *next;
  const FE_SPACE *fe_space;

  const char     *name;

  DOF            size;
  DOF            *vec;

  void  (*refine_interpol)(DOF_DOF_VEC *, RC_LIST_EL *, int n);
  void  (*coarse_restrict)(DOF_DOF_VEC *, RC_LIST_EL *, int n);

  void            *mem_info;
};

struct dof_uchar_vec
{
  DOF_UCHAR_VEC  *next;
  const FE_SPACE *fe_space;

  const char     *name;
  DOF            size;
  U_CHAR         *vec;

  void  (*refine_interpol)(DOF_UCHAR_VEC *, RC_LIST_EL *, int n);
  void  (*coarse_restrict)(DOF_UCHAR_VEC *, RC_LIST_EL *, int n);

  void            *mem_info;
};

struct dof_schar_vec
{
  DOF_SCHAR_VEC  *next;
  const FE_SPACE *fe_space;

  const char     *name;

  DOF            size;
  S_CHAR         *vec;

  void  (*refine_interpol)(DOF_SCHAR_VEC *, RC_LIST_EL *, int n);
  void  (*coarse_restrict)(DOF_SCHAR_VEC *, RC_LIST_EL *, int n);

  void            *mem_info;
};

struct dof_real_vec
{
  DOF_REAL_VEC   *next;
  const FE_SPACE *fe_space;

  const char     *name;
  DOF            size;
  REAL           *vec;

  void  (*refine_interpol)(DOF_REAL_VEC *, RC_LIST_EL *, int n);
  void  (*coarse_restrict)(DOF_REAL_VEC *, RC_LIST_EL *, int n);

  void            *mem_info;
};

struct dof_real_d_vec
{
  DOF_REAL_D_VEC *next;
  const FE_SPACE *fe_space;

  const char     *name;
  DOF            size;
  REAL_D         *vec;

  void  (*refine_interpol)(DOF_REAL_D_VEC *, RC_LIST_EL *, int n);
  void  (*coarse_restrict)(DOF_REAL_D_VEC *, RC_LIST_EL *, int n);

  void            *mem_info;
};

struct dof_ptr_vec
{
  DOF_PTR_VEC *next;
  const FE_SPACE *fe_space;

  const char     *name;
  DOF            size;
  void          **vec;

  void  (*refine_interpol)(DOF_PTR_VEC *, RC_LIST_EL *, int n);
  void  (*coarse_restrict)(DOF_PTR_VEC *, RC_LIST_EL *, int n);

  void            *mem_info;
};

/*--------------------------------------------------------------------------*/
/*  sparse matrix with one row for each dof,                                */
/*  entries are either REAL or REAL_DD				            */
/*--------------------------------------------------------------------------*/
/*  next:        pointer to next matrix (linked list in MESH)               */
/*  matrix_row[]: pointers to row structures (or nil if row index is unused)*/
/*  size:         currently allocated size of matrix_row[]		    */
/*--------------------------------------------------------------------------*/

struct dof_matrix
{
  DOF_MATRIX     *next;
  const FE_SPACE *row_fe_space;
  const FE_SPACE *col_fe_space;

  const char     *name;

  MATRIX_ROW     **matrix_row;                 /* lists of matrix entries   */
  DOF            size;                         /* size of vector matrix_row */

  void       (*refine_interpol)(DOF_MATRIX *, RC_LIST_EL *, int n);
  void       (*coarse_restrict)(DOF_MATRIX *, RC_LIST_EL *, int n);

  void            *mem_info;
};

/* "flag" values for "type" component */
#define DOWBM_FULL (1 << 0)
#define DOWBM_SYMM (1 << 1)
#define DOWBM_DIAG (1 << 2)

typedef enum dowbm_type {
  dowbm_full = DOWBM_FULL,
  dowbm_diag = DOWBM_DIAG,
  dowbm_symm = DOWBM_SYMM
} DOWBM_TYPE;

struct dof_dowb_matrix
{
  DOF_DOWB_MATRIX *next;

  const FE_SPACE  *row_fe_space;
  const FE_SPACE  *col_fe_space;

  const char      *name;

  DOWB_MATRIX_ROW **matrix_row;               /* lists of matrix entries   */
  DOF             size;                       /* size of vector matrix_row */

  DOWBM_TYPE      type;
  void       (*refine_interpol)(DOF_DOWB_MATRIX *, RC_LIST_EL *, int n);
  void       (*coarse_restrict)(DOF_DOWB_MATRIX *, RC_LIST_EL *, int n);

  void            *mem_info;
};

/*--------------------------------------------------------------------------*/
/*  row structure for sparse matrix, with either REAL or REAL_DD entries.   */
/*--------------------------------------------------------------------------*/
/*  next:        pointer to next structure containing entries of same row   */
/*  col[]:       column indices of entries (if >= 0; else unused)	    */
/*  entry[]:     matrix entries                        			    */
/*--------------------------------------------------------------------------*/

#define ROW_LENGTH 9

struct matrix_row
{
  MATRIX_ROW *next;
  DOF        col[ROW_LENGTH];    /* column indices */
  REAL       entry[ROW_LENGTH];  /* matrix entries */
};

/* full featured blocks */
struct dowb_matrix_row
{
  DOWB_MATRIX_ROW *next;
  DOF             col[ROW_LENGTH];    /* column indices */
  union {
    REAL_DD       full[0];   /* full */
    REAL_DDS      symm[0]; /* symmetric, only upper triangle is stored */
    REAL_D        diag[0]; /* diagonal */
  } entry;  /* matrix entries */
};

/* shared by DOF_MATRIX and DOF_DOWB_MATRIX */

#define ENTRY_USED(col)         ((col) >= 0)
#define ENTRY_NOT_USED(col)     ((col) < 0)
#define UNUSED_ENTRY    -1
#define NO_MORE_ENTRIES -2

#ifndef __CBLAS_H__
typedef enum { NoTranspose,
               Transpose,
               ConjugateTranspose } MatrixTranspose;
#endif

#endif  /* DOF_ADMIN_DEF */


/*--------------------------------------------------------------------------*/
/*  Here comes the MESH (giving access to the whole triangulation)	    */
/*--------------------------------------------------------------------------*/

struct mesh
{
  const char      *name;

  int             dim;

  int             n_vertices;
  int             n_elements;
  int             n_hier_elements;

  int             n_edges;                        /* Only used for dim > 1  */
  int             n_faces;                        /* Only used for dim == 3 */
  int             max_edge_neigh;                 /* Only used for dim == 3 */

  int             n_macro_el;
  MACRO_EL        *macro_els;

  REAL            diam[DIM_OF_WORLD];
  PARAMETRIC      *parametric;

  DOF_ADMIN       **dof_admin;
  int             n_dof_admin;
  
  int             n_dof_el;         /* sum of all dofs from all admins      */
  int             n_dof[N_NODE_TYPES]; /* sum of vertex/edge/... dofs from  */
                                      /* all admins */

  int             n_node_el;   /* number of used nodes on each element      */
  int             node[N_NODE_TYPES]; /* index of first vertex/edge/... node*/

  int             cookie;    /* changed on each refine/coarsen. Use this to */
     /* check consistency of meshes and DOF vectors when reading from files.*/

/*--------------------------------------------------------------------------*/
/*---  pointer for administration; don't touch!                          ---*/
/*--------------------------------------------------------------------------*/

  void            *mem_info;

  /* Linked list of hook functions to be run during mesh-traverse
   * before the per-element method is called.
   */
  LIST_NODE traverse_hooks[8];
  FLAGS active_hooks;
};

/*--------------------------------------------------------------------------*/
/*  stack data structure for non-recursive mesh traversal                   */
/*--------------------------------------------------------------------------*/

struct traverse_stack 
{
  MESH           *traverse_mesh;
  int             traverse_level;
  FLAGS           traverse_fill_flag;

  const MACRO_EL *traverse_mel;
  int             stack_size;
  int             stack_used;
  EL_INFO        *elinfo_stack;
  U_CHAR         *info_stack;
  
  const MACRO_EL *save_traverse_mel;
  EL_INFO        *save_elinfo_stack;
  U_CHAR         *save_info_stack;
  int             save_stack_used;

  int             el_count;

  int             marker;
  
  TRAVERSE_STACK *next;
};

/*--------------------------------------------------------------------------*/
/*---  data structure for basis function representation                  ---*/
/*--------------------------------------------------------------------------*/

typedef REAL        BAS_FCT(const REAL[N_LAMBDA]);
typedef const REAL  *GRD_BAS_FCT(const REAL[N_LAMBDA]);
typedef const REAL  (*D2_BAS_FCT(const REAL[N_LAMBDA]))[N_LAMBDA];

struct bas_fcts
{
  char         *name;       /*  textual description                         */
  int          dim;         /*  dimension of the corresponding mesh.        */
  int          n_bas_fcts;  /*  number of basisfunctions on one el          */
  int          degree;      /*  maximal degree of the basis functions       */
  const int    n_dof[N_NODE_TYPES];   /* dofs from these bas_fcts           */

  void         (*init_element)(const EL_INFO *, const FE_SPACE *, U_CHAR);
  BAS_FCT      **phi;
  GRD_BAS_FCT  **grd_phi;
  D2_BAS_FCT   **D2_phi;

  const DOF    *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const S_CHAR *(*get_bound)(const EL_INFO *, S_CHAR *);

/*************** entries must be set for interpolation   ********************/

  const REAL   *(*interpol)(const EL_INFO *el_info, int n, const int *indices, 
			    REAL (*f)(const REAL_D),
			    REAL (*f_loc)(const EL_INFO *el_info,
					  const REAL lambda[N_LAMBDA]),
			    REAL *coeff);

  const REAL_D *(*interpol_d)(const EL_INFO *el_info, int n,
			      const int *indices,
			      const REAL *(*f)(const REAL_D, REAL_D),
			      const REAL *(*f_loc)(const EL_INFO *el_info,
						   const REAL lambda[N_LAMBDA],
						   REAL_D val),
			      REAL_D *coeff);

/********************   optional entries  ***********************************/

  const int    *(*get_int_vec)(const EL *, const DOF_INT_VEC *, int *);
  const REAL   *(*get_real_vec)(const EL *, const DOF_REAL_VEC *, REAL *);
  const REAL_D *(*get_real_d_vec)(const EL *, const DOF_REAL_D_VEC *, REAL_D *);
  const U_CHAR *(*get_uchar_vec)(const EL *, const DOF_UCHAR_VEC *, U_CHAR *);
  const S_CHAR *(*get_schar_vec)(const EL *, const DOF_SCHAR_VEC *, S_CHAR *);

  void  (*real_refine_inter)(DOF_REAL_VEC *, RC_LIST_EL *, int);
  void  (*real_coarse_inter)(DOF_REAL_VEC *, RC_LIST_EL *, int);
  void  (*real_coarse_restr)(DOF_REAL_VEC *, RC_LIST_EL *, int);

  void  (*real_d_refine_inter)(DOF_REAL_D_VEC *, RC_LIST_EL *, int);
  void  (*real_d_coarse_inter)(DOF_REAL_D_VEC *, RC_LIST_EL *, int);
  void  (*real_d_coarse_restr)(DOF_REAL_D_VEC *, RC_LIST_EL *, int);

  void  *bas_fcts_data;
};

/* Barycentric coordinates of Lagrange nodes. */
#define LAGRANGE_NODES(bfcts) ((const REAL_B *)(bfcts)->bas_fcts_data)

/*--------------------------------------------------------------------------*/
/*---  FE spaces are a triple of DOFs and BAS_FCTs on a MESH             ---*/
/*--------------------------------------------------------------------------*/

struct fe_space
{
  const char       *name;
  const DOF_ADMIN  *admin;
  const BAS_FCTS   *bas_fcts;
  MESH             *mesh;
};

/*--------------------------------------------------------------------------*/
/*---  data structures for numerical integration                         ---*/
/*--------------------------------------------------------------------------*/

struct quadrature
{
  char          *name;
  int           degree;

  int           dim;
  int           n_points;
  const double  **lambda;
  const double  *w;
};

#define MAX_N_QUAD_POINTS  64

/*--------------------------------------------------------------------------*/
/*--- data structure with precomputed values of basis functions at       ---*/
/*--- quadrature nodes on the standard element                           ---*/
/*--------------------------------------------------------------------------*/

#define INIT_PHI       1
#define INIT_GRD_PHI   2
#define INIT_D2_PHI    4

struct quad_fast
{
  const QUAD        *quad;
  const BAS_FCTS    *bas_fcts;

  int               n_points;
  int               n_bas_fcts;
  const double      *w;

  U_CHAR            init_flag;

  REAL              **phi;
  REAL              (**grd_phi)[N_LAMBDA];
  REAL              (**D2_phi)[N_LAMBDA][N_LAMBDA];
};

/*--------------------------------------------------------------------------*/
/*---  data structure for adaptive methods                               ---*/
/*--------------------------------------------------------------------------*/

struct adapt_stat
{
  const char  *name;
  REAL        tolerance;
  REAL        p;                         /* power in estimator norm        */
  int         max_iteration;
  int         info;

  REAL   (*estimate)(MESH *mesh, ADAPT_STAT *adapt);
  REAL   (*get_el_est)(EL *el);          /* local error estimate           */
  REAL   (*get_el_estc)(EL *el);         /* local coarsening error estimate*/
  U_CHAR (*marking)(MESH *mesh, ADAPT_STAT *adapt);

  void   *est_info;                      /* estimator parameters           */
  REAL   err_sum, err_max;               /* sum and max of el_est          */

  void   (*build_before_refine)(MESH *mesh, U_CHAR flag);
  void   (*build_before_coarsen)(MESH *mesh, U_CHAR flag);
  void   (*build_after_coarsen)(MESH *mesh, U_CHAR flag);
  void   (*solve)(MESH *mesh);

  int    refine_bisections;
  int    coarsen_allowed;                /* 0 : 1                          */
  int    coarse_bisections;

  int    strategy;                       /* 1=GR, 2=MS, 3=ES, 4=GERS       */
  REAL   MS_gamma, MS_gamma_c;           /* maximum strategy               */
  REAL   ES_theta, ES_theta_c;           /* equidistribution strategy      */
  REAL   GERS_theta_star, GERS_nu, GERS_theta_c;  /* willy's strategy      */
};


struct adapt_instat
{
  const char  *name;

  ADAPT_STAT adapt_initial[1];
  ADAPT_STAT adapt_space[1];

  REAL   time;
  REAL   start_time, end_time;
  REAL   timestep;

  void   (*init_timestep)(MESH *mesh, ADAPT_INSTAT *adapt);
  void   (*set_time)(MESH *mesh, ADAPT_INSTAT *adapt);
  void   (*one_timestep)(MESH *mesh, ADAPT_INSTAT *adapt);
  REAL   (*get_time_est)(MESH *mesh, ADAPT_INSTAT *adapt);
  void   (*close_timestep)(MESH *mesh, ADAPT_INSTAT *adapt);

  int    strategy; 
  int    max_iteration;

  REAL   tolerance;
  REAL   rel_initial_error;
  REAL   rel_space_error;
  REAL   rel_time_error;
  REAL   time_theta_1;
  REAL   time_theta_2;
  REAL   time_delta_1;
  REAL   time_delta_2;
  int    info;
};

#define MESH_REFINED   1
#define MESH_COARSENED 2

#define H1_NORM  1
#define L2_NORM  2

/*--------------------------------------------------------------------------*/
/*---  data structures for matrix and vector update                      ---*/
/*--------------------------------------------------------------------------*/

typedef struct el_matrix_info  EL_MATRIX_INFO;
struct el_matrix_info
{
  int              n_row;
  const DOF_ADMIN  *row_admin;
  const DOF        *(*get_row_dof)(const EL *,const DOF_ADMIN *, DOF *);

  int              n_col;
  const DOF_ADMIN  *col_admin;
  const DOF        *(*get_col_dof)(const EL *,const DOF_ADMIN *, DOF *);

  const S_CHAR     *(*get_bound)(const EL_INFO *, S_CHAR *);

  REAL             factor;

  const REAL       **(*el_matrix_fct)(const EL_INFO *, void *);
  void             *fill_info;

  FLAGS            fill_flag;
};

typedef struct el_dowb_matrix_info  EL_DOWB_MATRIX_INFO;
struct el_dowb_matrix_info
{
  int              n_row;
  const DOF_ADMIN  *row_admin;
  const DOF        *(*get_row_dof)(const EL *,const DOF_ADMIN *, DOF *);

  int              n_col;
  const DOF_ADMIN  *col_admin;
  const DOF        *(*get_col_dof)(const EL *,const DOF_ADMIN *, DOF *);

  const S_CHAR     *(*get_bound)(const EL_INFO *, S_CHAR *);

  REAL             factor;

  const void       **(*el_matrix_fct)(const EL_INFO *, void *);
  void             *fill_info;

  FLAGS            fill_flag;
  DOWBM_TYPE       type; /* dowbm_full, _symm or _diag */
};

typedef struct el_vec_info  EL_VEC_INFO;
struct el_vec_info
{
  int              n_dof;
  const DOF_ADMIN  *admin;
  const DOF        *(*get_dof)(const EL *,const DOF_ADMIN *, DOF *);
  const S_CHAR     *(*get_bound)(const EL_INFO *, S_CHAR *);

  REAL             factor;

  const REAL       *(*el_vec_fct)(const EL_INFO *, void *);
  void             *fill_info;

  FLAGS            fill_flag;
};

typedef struct el_vec_d_info  EL_VEC_D_INFO;
struct el_vec_d_info
{
  int              n_dof;
  const DOF_ADMIN  *admin;
  const DOF        *(*get_dof)(const EL *,const DOF_ADMIN *, DOF *);
  const S_CHAR     *(*get_bound)(const EL_INFO *, S_CHAR *);

  REAL             factor;

  const REAL_D     *(*el_vec_fct)(const EL_INFO *, void *);
  void             *fill_info;

  FLAGS            fill_flag;
};

/*--------------------------------------------------------------------------*/
/*---  data structure about the differential operator for matrix assemblage */
/*--------------------------------------------------------------------------*/

typedef struct operator_info  OPERATOR_INFO;
struct operator_info
{
  const FE_SPACE *row_fe_space;
  const FE_SPACE *col_fe_space;

  const QUAD *quad[3];

  int       (*init_element)(const EL_INFO *, const QUAD *[3], void *);
  const REAL (*(*LALt)(const EL_INFO *, const QUAD *, int, void *))[N_LAMBDA];
  int        LALt_pw_const;
  int        LALt_symmetric;
  const REAL *(*Lb0)(const EL_INFO *, const QUAD *, int, void *);
  int        Lb0_pw_const;
  const REAL *(*Lb1)(const EL_INFO *, const QUAD *, int, void *);
  int        Lb1_pw_const;
  int        Lb0_Lb1_anti_symmetric;
  REAL       (*c)(const EL_INFO *, const QUAD *, int, void *);
  int        c_pw_const;

  int        use_get_bound;
  void       *user_data;
  FLAGS      fill_flag;
};

typedef struct dowb_operator_info DOWB_OPERATOR_INFO;
struct dowb_operator_info
{
  const FE_SPACE *row_fe_space;
  const FE_SPACE *col_fe_space;

  const QUAD *quad[3];

  int         (*init_element)(const EL_INFO *, const QUAD *[3], void *);
  union {
    const REAL_DD (*(*full)(const EL_INFO *,
			    const QUAD *, int, void *))[N_LAMBDA];
    const REAL_DDS (*(*symm)(const EL_INFO *,
			     const QUAD *, int, void *))[N_LAMBDA];
    const REAL_D (*(*diag)(const EL_INFO *,
			   const QUAD *, int, void *))[N_LAMBDA];
  } LALt;
  int           LALt_pw_const;
  int           LALt_symmetric;
  union {
    const REAL_DD  *(*full)(const EL_INFO *, const QUAD *, int, void *);
    const REAL_DDS *(*symm)(const EL_INFO *, const QUAD *, int, void *);
    const REAL_D   *(*diag)(const EL_INFO *, const QUAD *, int, void *);
  } Lb0;
  int           Lb0_pw_const;
  union {
    const REAL_DD  *(*full)(const EL_INFO *, const QUAD *, int, void *);
    const REAL_DDS *(*symm)(const EL_INFO *, const QUAD *, int, void *);
    const REAL_D   *(*diag)(const EL_INFO *, const QUAD *, int, void *);
  } Lb1;
  int           Lb1_pw_const;
  int           Lb0_Lb1_anti_symmetric;
  union {
    const REAL_D   *(*full)(const EL_INFO *, const QUAD *, int, void *);
    const REAL_DDS *(*symm)(const EL_INFO *, const QUAD *, int, void *);
    const REAL     *(*diag)(const EL_INFO *, const QUAD *, int, void *);
  } c;    
  int           c_pw_const;

  int        use_get_bound;
  void       *user_data;
  FLAGS      fill_flag;
  DOWBM_TYPE type; /* dowbm_full, _symm or _diag */
};

/*--------------------------------------------------------------------------*/
/*  calculate element stiffness matrices by preevaluated integrals over the */
/*  the reference element.                                                  */
/*--------------------------------------------------------------------------*/

typedef struct q11_psi_phi   Q11_PSI_PHI;
typedef struct q01_psi_phi   Q01_PSI_PHI;
typedef struct q10_psi_phi   Q10_PSI_PHI;
typedef struct q00_psi_phi   Q00_PSI_PHI;

struct q11_psi_phi
{
  const BAS_FCTS     *psi;
  const BAS_FCTS     *phi;
  const QUAD         *quad;

  const int   **n_entries;
  const REAL  ***values;
  const int   ***k;
  const int   ***l;
};

struct q01_psi_phi
{
  const BAS_FCTS     *psi;
  const BAS_FCTS     *phi;
  const QUAD         *quad;

  const int   **n_entries;
  const REAL  ***values;
  const int   ***l;
};

struct q10_psi_phi
{
  const BAS_FCTS     *psi;
  const BAS_FCTS     *phi;
  const QUAD         *quad;

  const int   **n_entries;
  const REAL  ***values;
  const int   ***k;
};

struct q00_psi_phi
{
  const BAS_FCTS     *psi;
  const BAS_FCTS     *phi;
  const QUAD         *quad;

  const REAL  **values;
};


/*--------------------------------------------------------------------------*/
/*   data type for preconditioning                                          */
/*--------------------------------------------------------------------------*/

typedef struct precon PRECON;
struct precon
{
  void    *precon_data;

  int     (*init_precon)(void *precon_data);
  void    (*precon)(void *precon_data, int n, REAL *vec);
  void    (*exit_precon)(void *precon_data);
};

const PRECON *get_diag_precon_s(const DOF_MATRIX *, const DOF_SCHAR_VEC *);
const PRECON *get_diag_precon_d(const DOF_MATRIX *, const DOF_SCHAR_VEC *);
const PRECON *get_diag_precon_dowb(const DOF_DOWB_MATRIX *,
				   const DOF_SCHAR_VEC *);

const PRECON *get_HB_precon_s(const FE_SPACE *, const DOF_SCHAR_VEC *, int, int);
const PRECON *get_HB_precon_d(const FE_SPACE *, const DOF_SCHAR_VEC *, int, int);

const PRECON *get_BPX_precon_s(const FE_SPACE *, const DOF_SCHAR_VEC *, int, int);
const PRECON *get_BPX_precon_d(const FE_SPACE *, const DOF_SCHAR_VEC *, int, int);

/*--------------------------------------------------------------------------*/
/*  solver identification for interface to OEM-lib                          */
/*--------------------------------------------------------------------------*/
typedef enum {NoSolver, BiCGStab, CG, TfQMR, GMRes, ODir, ORes} OEM_SOLVER;

/*--------------------------------------------------------------------------*/
/*--- abstract multigrid                                                 ---*/
/*--------------------------------------------------------------------------*/

typedef struct multi_grid_info MULTI_GRID_INFO;

struct multi_grid_info
{
  REAL             tolerance;                     /* tol. for resid         */
  REAL             exact_tolerance;               /* tol. for exact_solver  */

  int              cycle;                         /* 1=V-cycle, 2=W-cycle   */
  int              n_pre_smooth, n_in_smooth;     /* no of smoothing loops  */
  int              n_post_smooth;                 /* no of smoothing loops  */
  int              mg_levels;                     /* current no. of levels  */
  int              exact_level;                   /* level for exact_solver */
  int              max_iter;                      /* max. no of MG iter's   */
  int              info;

  int              (*init_multi_grid)(MULTI_GRID_INFO *mg_info);
  void             (*pre_smooth)(MULTI_GRID_INFO *mg_info, int level, int n);
  void             (*in_smooth)(MULTI_GRID_INFO *mg_info, int level, int n);
  void             (*post_smooth)(MULTI_GRID_INFO *mg_info, int level, int n);
  void             (*mg_restrict)(MULTI_GRID_INFO *mg_info, int level);
  void             (*mg_prolongate)(MULTI_GRID_INFO *mg_info, int level);
  void             (*exact_solver)(MULTI_GRID_INFO *mg_info, int level);
  REAL             (*mg_resid)(MULTI_GRID_INFO *mg_info, int level);
  void             (*exit_multi_grid)(MULTI_GRID_INFO *mg_info);

  void             *data;                         /* application dep. data */

};
int MG(MULTI_GRID_INFO *mg_info);

/*--------------------------------------------------------------------------*/
/*--- concrete multigrid                                                 ---*/
/*--------------------------------------------------------------------------*/

typedef struct mg_s_info MG_S_INFO;
struct mg_s_info
{
  MULTI_GRID_INFO  *mg_info;                      /* abstract MG info       */

  const FE_SPACE      *fe_space;
  DOF_MATRIX          *mat;
  const DOF_REAL_VEC  *f;
  DOF_REAL_VEC        *u;
  const DOF_SCHAR_VEC *bound;

  int              smoother,     exact_solver;
  REAL             smooth_omega, exact_omega;

  int              size;                          /* current size of vectors*/
  DOF_MATRIX       **matrix;                      /* one for each level     */
  REAL             **f_h;                         /* one for each level     */
  REAL             **u_h;                         /* one for each level     */
  REAL             **r_h;                         /* one for each level     */
  int              *dofs_per_level;               /* count dofs per level   */

  int              sort_size;                     /* size of sort vectors   */
  DOF              *sort_dof;                     /* dofs in order of levels*/
  DOF              *(dof_parent[2]);              /* (for linear elements)  */
  U_CHAR           *dof_level;
  S_CHAR           *sort_bound;                   /* sorted bound           */

  int              sort_invers_size;              /* size of inv. sort list */
  int              *sort_dof_invers;              /* inverse sort list      */
};
/*--------------------------------------------------------------------------*/
/*  sort_dof[ sorted dof ]          = unsorted dof                          */
/*  sort_dof_invers[ unsorted dof ] = sorted dof                            */
/*--------------------------------------------------------------------------*/

/* file MG_s1.c DOF_sort routines *******************************************/
void MG_s_setup_levels(MG_S_INFO *mg_s_info);
void MG_s_setup_mat_b(MG_S_INFO *mg_s_info,
		      DOF_MATRIX *mat, const DOF_SCHAR_VEC *bound);
void MG_s_dof_copy_to_sparse(MG_S_INFO *mg_s_info,
			     const DOF_REAL_VEC *x, REAL *y);
void MG_s_dof_copy_from_sparse(MG_S_INFO *mg_s_info, 
			       const REAL *x, DOF_REAL_VEC *y);
void MG_s_reset_mat(MG_S_INFO *mg_s_info);
void MG_s_sort_mat(MG_S_INFO *mg_s_info);
void MG_s_free_mem(MG_S_INFO *mg_s_info);

/* file MG_s2.c: DOF_sort independent routines ******************************/
void MG_s_restrict_mg_matrices(MG_S_INFO *mg_s_info);
void MG_s_restrict(MULTI_GRID_INFO *mg_info, int mg_level);
void MG_s_prolongate(MULTI_GRID_INFO *mg_info, int mg_level);
REAL MG_s_resid(MULTI_GRID_INFO *mg_info, int mg_level);
void MG_s_smoother(MULTI_GRID_INFO *mg_info, int mg_level, int n);
void MG_s_exact_solver(MULTI_GRID_INFO *mg_info, int mg_level);
void MG_s_gemv(MG_S_INFO *mg_s_info, int mg_level, MatrixTranspose transpose, 
	       REAL alpha, DOF_MATRIX *a, REAL *x, REAL beta, REAL *y);


/* file MG_s.c: *************************************************************/
int mg_s(DOF_MATRIX *matrix, DOF_REAL_VEC *u, const DOF_REAL_VEC *f,
	 const DOF_SCHAR_VEC *bound, 
	 REAL tol, int max_iter, int info, char *prefix);
MG_S_INFO *mg_s_init(DOF_MATRIX *matrix, const DOF_SCHAR_VEC *bound, 
		     int info, char *prefix);
int mg_s_solve(MG_S_INFO *mg_s_info,
	       DOF_REAL_VEC *u, const DOF_REAL_VEC *f, REAL tol, int max_iter);
void mg_s_exit(MG_S_INFO *mg_s_info);

/*--------------------------------------------------------------------------*/
/* Graphic output Definitions                                               */
/*--------------------------------------------------------------------------*/

typedef void * GRAPH_WINDOW;
typedef float  GRAPH_RGBCOLOR[3];

/** flags used by graph_mesh(): ****/
#define GRAPH_MESH_BOUNDARY        1
#define GRAPH_MESH_ELEMENT_MARK    2
#define GRAPH_MESH_VERTEX_DOF      4
#define GRAPH_MESH_ELEMENT_INDEX   8

/*--------------------------------------------------------------------------*/
/*  very usefull macro definitons                                           */
/*--------------------------------------------------------------------------*/

#define GET_MESH(dim,name,macro_data,init_node_proj) \
 check_and_get_mesh((dim),DIM_OF_WORLD,ALBERTA_DEBUG, \
                    ALBERTA_VERSION,(name),(macro_data), \
                    (init_node_proj))

#define GET_DOF_VEC(ptr, dof_vec)\
   TEST_EXIT((dof_vec)  && ((ptr) = (dof_vec)->vec),\
      "%s == nil\n", (dof_vec) ? NAME(dof_vec) : #dof_vec)


/*--------------------------------------------------------------------------*/
/*  defined in graphXO.c                                                    */
/*--------------------------------------------------------------------------*/
extern const GRAPH_RGBCOLOR rgb_black; 
extern const GRAPH_RGBCOLOR rgb_white;  
extern const GRAPH_RGBCOLOR rgb_red;    
extern const GRAPH_RGBCOLOR rgb_green;  
extern const GRAPH_RGBCOLOR rgb_blue;   
extern const GRAPH_RGBCOLOR rgb_yellow; 
extern const GRAPH_RGBCOLOR rgb_magenta;
extern const GRAPH_RGBCOLOR rgb_cyan;   
extern const GRAPH_RGBCOLOR rgb_grey50; 

extern const GRAPH_RGBCOLOR rgb_albert;
extern const GRAPH_RGBCOLOR rgb_alberta;


/*--------------------------------------------------------------------------*/
/*  used in wall_quad_fast.c                                                */
/*--------------------------------------------------------------------------*/

/* We use sort_face_indices() & friends and build the tables for every
 * relative permutation possible.
 */

typedef struct wall_quad_fast WALL_QUAD_FAST;

#define CARD_SN_MAX 6 /* this must be (N_LAMBDA-1)! */

struct wall_quad_fast
{
  const QUAD             *quad;
  
  int                    n_points;
  int                    n_bas_fcts;
  const REAL             *w;
  REAL_B                 *lambda[CARD_SN_MAX][N_FACES_MAX];
  U_CHAR                 init_flag;
  
  REAL                   **phi[CARD_SN_MAX][N_FACES_MAX];

  WALL_QUAD_FAST         *next;
  BAS_FCTS               *bas_fcts;

  REAL_B                 **grd_phi[CARD_SN_MAX][N_FACES_MAX];
  REAL_BB                **D2_phi[CARD_SN_MAX][N_FACES_MAX];
};

/*--------------------------------------------------------------------------*/
/*  functions supplied by ALBERTA                                           */
/*--------------------------------------------------------------------------*/

/***   file coarsen.c   *****************************************************/
extern U_CHAR coarsen(MESH *mesh);
extern U_CHAR global_coarsen(MESH *mesh, int no);
extern int  get_max_level(MESH *mesh);

/***   file dof_admin.c   ***************************************************/
/*extern void add_bas_fcts_to_admin(DOF_ADMIN *admin, const BAS_FCTS *bas_fcts);*/
extern const DOF_ADMIN *get_vertex_admin(MESH *mesh);

void add_element_matrix(DOF_MATRIX *, REAL, int, int, const DOF *, 
			const DOF *, const REAL **, const S_CHAR *);
void add_element_dowb_matrix(DOF_DOWB_MATRIX *, REAL, int, int, const DOF *, 
			     const DOF *, const void **, const S_CHAR *);
void add_element_vec(DOF_REAL_VEC *, REAL, int, const DOF *,
		     const REAL *, const S_CHAR *);
void add_element_d_vec(DOF_REAL_D_VEC *, REAL, int, const DOF *,
		       const REAL_D *, const S_CHAR *);

extern void dof_compress(MESH *mesh);
extern void clear_dof_matrix(DOF_MATRIX *matrix);
extern void clear_dof_dowb_matrix(DOF_DOWB_MATRIX *matrix);
extern void print_dof_matrix(const DOF_MATRIX *matrix);
extern void print_dof_dowb_matrix(const DOF_DOWB_MATRIX *matrix);
extern void print_dof_real_vec(const DOF_REAL_VEC *drv);
extern void print_dof_real_d_vec(const DOF_REAL_D_VEC *drdv);
extern void print_dof_ptr_vec(const DOF_PTR_VEC *dpv);
extern void print_dof_int_vec(const DOF_INT_VEC *div);
extern void print_dof_uchar_vec(const DOF_UCHAR_VEC *div);
extern void print_dof_schar_vec(const DOF_SCHAR_VEC *div);
/* BLAS 1 */
extern REAL dof_nrm2(const DOF_REAL_VEC *x);
extern REAL dof_asum(const DOF_REAL_VEC *x);
extern void dof_set(REAL alpha, DOF_REAL_VEC *x);
extern void dof_scal(REAL alpha, DOF_REAL_VEC *x);
extern REAL dof_dot(const DOF_REAL_VEC *x, const DOF_REAL_VEC *y);
extern void dof_copy(const DOF_REAL_VEC *x, DOF_REAL_VEC *y);
extern void dof_axpy(REAL alpha, const DOF_REAL_VEC *x, DOF_REAL_VEC *y);
/*  some non BLAS  */
extern void dof_xpay(REAL alpha, const DOF_REAL_VEC *x, DOF_REAL_VEC *y);
extern REAL dof_min(const DOF_REAL_VEC *x);
extern REAL dof_max(const DOF_REAL_VEC *x);
/* BLAS 2 */
extern void dof_gemv(MatrixTranspose transpose, REAL alpha,
		     const DOF_MATRIX *a, const DOF_REAL_VEC *x,
		     REAL beta, DOF_REAL_VEC *y);
extern void dof_mv(MatrixTranspose transpose, const DOF_MATRIX *a,
		   const DOF_REAL_VEC *x, DOF_REAL_VEC *y);

/* now the same for REAL_D */
extern void dof_axpy_d(REAL alpha, const DOF_REAL_D_VEC *x, DOF_REAL_D_VEC *y);
extern void dof_copy_d(const DOF_REAL_D_VEC *x, DOF_REAL_D_VEC *y);
extern REAL dof_dot_d(const DOF_REAL_D_VEC *x, const DOF_REAL_D_VEC *y);
extern REAL dof_nrm2_d(const DOF_REAL_D_VEC *x);
extern void dof_scal_d(REAL alpha, DOF_REAL_D_VEC *x);
extern void dof_set_d(REAL alpha, DOF_REAL_D_VEC *x);
extern void dof_xpay_d(REAL alpha, const DOF_REAL_D_VEC *x, DOF_REAL_D_VEC *y);
extern REAL dof_min_d(const DOF_REAL_D_VEC *x);
extern REAL dof_max_d(const DOF_REAL_D_VEC *x);
/*---    BLAS 2 for REAL_D                                               ---*/
extern void dof_mv_d(MatrixTranspose transpose, const DOF_MATRIX *a,
		     const DOF_REAL_D_VEC *x, DOF_REAL_D_VEC *y);
extern void dof_mv_dowb(MatrixTranspose transpose, const DOF_DOWB_MATRIX *a,
			const DOF_REAL_D_VEC *x, DOF_REAL_D_VEC *y);
extern void dof_gemv_d(MatrixTranspose transpose, REAL alpha,
		       const DOF_MATRIX *a, const DOF_REAL_D_VEC *x,
		       REAL beta, DOF_REAL_D_VEC *y);
extern void dof_gemv_dowb(MatrixTranspose transpose, REAL alpha,
			  const DOF_DOWB_MATRIX *a, const DOF_REAL_D_VEC *x,
			  REAL beta, DOF_REAL_D_VEC *y);

void update_matrix(DOF_MATRIX *dof_matrix, const EL_MATRIX_INFO *minfo);
void update_dowb_matrix(DOF_DOWB_MATRIX *dof_matrix,
			const EL_DOWB_MATRIX_INFO *minfo);
void update_real_vec(DOF_REAL_VEC *drv, const EL_VEC_INFO *vec_info);
void update_real_d_vec(DOF_REAL_D_VEC *drdv, const EL_VEC_D_INFO *vecd_info);

/***   file wall_quad_fast.c   *********************************************/

extern const WALL_QUAD_FAST *get_wall_quad_fast(const BAS_FCTS *, 
						const QUAD *, 
						U_CHAR init_flag);

/***   file macro.c   *******************************************************/
extern void macro_test(MACRO_DATA *data, const char *new_filename);

extern MACRO_DATA *read_macro(const char *name);
extern MACRO_DATA *read_macro_bin(const char *name);
extern MACRO_DATA *read_macro_xdr(const char *name);

extern int write_macro(MESH *mesh, const char *name);
extern int write_macro_bin(MESH *mesh, const char *name);
extern int write_macro_xdr(MESH *mesh, const char *name);

extern int write_macro_data(MACRO_DATA *data, const char *name);
extern int write_macro_data_bin(MACRO_DATA *data, const char *name);
extern int write_macro_data_xdr(MACRO_DATA *data, const char *name);

extern MACRO_DATA *alloc_macro_data(int dim, int nv, int ne, FLAGS);
extern void free_macro_data(MACRO_DATA *data);
extern void compute_neigh_fast(MACRO_DATA *data);
extern void dirichlet_boundary(MACRO_DATA *data);

extern MACRO_DATA *mesh2macro_data(MESH *mesh);
extern void macro_data2mesh(MESH *mesh, const MACRO_DATA *data,
			    NODE_PROJECTION *(*n_proj)(MESH *,MACRO_EL *,int));


/***   file memory.c   ******************************************************/
extern MESH *check_and_get_mesh(int dim, int dow, int neigh,
				const char *version, const char *name,
				const MACRO_DATA *macro_data,
				NODE_PROJECTION *(*init_node_proj)
				(MESH *, MACRO_EL *, int));
extern void add_traverse_hook(MESH *mesh,
			      TRAVERSE_HOOK *hook,
			      HOOK_QUEUE_ENUM queue);
extern void remove_traverse_hook(MESH *mesh,
				 TRAVERSE_HOOK *hook,
				 HOOK_QUEUE_ENUM queue);
extern void free_dof_admin(DOF_ADMIN *admin, MESH *mesh);
extern void free_int_dof_vec(DOF_DOF_VEC *vec);
extern void free_dof_int_vec(DOF_INT_VEC *vec);
extern void free_dof_dof_vec(DOF_DOF_VEC *vec);
extern void free_dof_matrix(DOF_MATRIX *mat);
extern void free_dof_dowb_matrix(DOF_DOWB_MATRIX *mat);
extern void free_dof_real_vec(DOF_REAL_VEC *vec);
extern void free_dof_real_d_vec(DOF_REAL_D_VEC *vec);
extern void free_dof_schar_vec(DOF_SCHAR_VEC *vec);
extern void free_dof_uchar_vec(DOF_UCHAR_VEC *vec);
extern void free_dof_ptr_vec(DOF_PTR_VEC *vec);
extern void free_fe_space(FE_SPACE *fe_space);
extern void free_real_d(MESH *mesh, REAL *ptr);
extern void free_matrix_row(const FE_SPACE *, MATRIX_ROW *);
extern void free_dowb_matrix_row(const FE_SPACE *, DOWBM_TYPE type, 
				 DOWB_MATRIX_ROW *);
extern void free_element(EL *el, MESH *mesh);
extern void free_rc_list(MESH *mesh, RC_LIST_EL *list); /* only for 3D */
extern void free_mesh(MESH *);
extern void free_dof(DOF *dof, MESH *mesh, int position,
		     const int is_coarse_dof);

extern DOF             *get_dof(MESH *mesh, int position);
extern const FE_SPACE  *get_fe_space(MESH *mesh, const char *name,
				     const int ndof[N_NODE_TYPES],
				     const BAS_FCTS *bas_fcts,
				     const U_CHAR preserve_coarse_dofs);
extern DOF_INT_VEC     *get_dof_int_vec(const char *name, const FE_SPACE *);
extern DOF_DOF_VEC     *get_int_dof_vec(const char *name, const FE_SPACE *);
extern DOF_DOF_VEC     *get_dof_dof_vec(const char *name, const FE_SPACE *);
extern DOF_MATRIX      *get_dof_matrix(const char *name, 
				       const FE_SPACE *row_fe_space,
				       const FE_SPACE *col_fe_space);
extern DOF_DOWB_MATRIX *get_dof_dowb_matrix(const char *name, 
					    const FE_SPACE *row_fe_space,
					    const FE_SPACE *col_fe_space,
					    DOWBM_TYPE type);
extern DOF_REAL_VEC    *get_dof_real_vec(const char *name, const FE_SPACE *);
extern DOF_REAL_D_VEC  *get_dof_real_d_vec(const char *name, const FE_SPACE *);
extern DOF_SCHAR_VEC   *get_dof_schar_vec(const char *name, const FE_SPACE *);
extern DOF_UCHAR_VEC   *get_dof_uchar_vec(const char *name, const FE_SPACE *);
extern DOF_PTR_VEC     *get_dof_ptr_vec(const char *name, const FE_SPACE *);
extern REAL            *get_real_d(MESH *mesh);
extern MATRIX_ROW      *get_matrix_row(const FE_SPACE *);
extern DOWB_MATRIX_ROW *get_dowb_matrix_row(const FE_SPACE *, DOWBM_TYPE type);
extern EL              *get_element(MESH *mesh);
extern RC_LIST_EL      *get_rc_list(MESH *mesh);           /* only for 3D */
extern size_t init_leaf_data(MESH *mesh, size_t size, 
		     void (*refine_leaf_data)(EL *parent, EL *child[2]),
		     void (*coarsen_leaf_data)(EL *parent, EL *child[2]));

/***   file submesh.c  ******************************************************/
extern MESH *get_submesh(MESH *master, const char *name,
			 int (*binding_method)(MESH *master, MACRO_EL *el,
					       int face, void *data),
			 void *data);

extern void unchain_submesh(MESH *slave);

extern MESH *read_submesh(MESH *master,
			  const char *slave_filename,
			  int (*binding_method)(MESH *master, MACRO_EL *el,
						int face, void *data),
			  NODE_PROJECTION *(*)(MESH *, MACRO_EL *, int),
			  void *data);

extern MESH *read_submesh_xdr(MESH *master,
			      const char *slave_filename,
			      int (*binding_method)(MESH *master, MACRO_EL *el,
						    int face, void *data),
			      NODE_PROJECTION *(*)(MESH *, MACRO_EL *, int),
			      void *data);

extern void trace_dof_real_vec(DOF_REAL_VEC *slave_vec,
			       DOF_REAL_VEC *master_vec);

extern void trace_dof_real_d_vec(DOF_REAL_D_VEC *slave_vec,
				 DOF_REAL_D_VEC *master_vec);

extern void get_slave_dof_mapping(const FE_SPACE *m_fe_space,
				  DOF_INT_VEC *s_map);

extern MESH *get_master(MESH *slave);
extern DOF_PTR_VEC *get_master_binding(MESH *slave);
extern DOF_PTR_VEC *get_slave_binding(MESH *slave);

/***   file level.c    ******************************************************/
extern REAL level_element_det_2d(const REAL_D coord[]);
extern void level_coord_to_world_2d(const REAL_D coord[],
				    const REAL_B lambda,
				    REAL_D world);
extern void level_coord_to_el_coord_2d(const REAL_B v_lambda[],
				       const REAL_B lambda,
				       REAL_B el_lambda);
extern REAL level_element_det_3d(const REAL_D coord[]);
extern void level_coord_to_world_3d(const REAL_D coord[],
				    const REAL_B lambda,
				    REAL_D world);
extern void level_coord_to_el_coord_3d(const REAL_B v_lambda[],
				       const REAL_B lambda,
				       REAL_B el_lambda);

extern int find_level(MESH *mesh, FLAGS fill_flag, const DOF_REAL_VEC *Level,
		      REAL value,
		      int (*init)(const EL_INFO *el_info,
				  REAL v[], 
				  int N, int face, const REAL_B lambda[]),
		      void (*cal)(const EL_INFO *el_info,
				  REAL v[],
				  int i, 
				  int face, const REAL_B lambda[], 
				  const REAL_D coord[]));
extern void set_element_mark(MESH *mesh, FLAGS fill_flag, S_CHAR mark);

/***   file numint.c   ******************************************************/
const QUAD *get_quadrature(int dim, int degree);
const QUAD *get_lumping_quadrature(int dim);
void print_quadrature(const QUAD *quad);
REAL  integrate_std_simp(const QUAD *quad, REAL (*f)(const REAL *));
const REAL   *f_at_qp(const QUAD*, REAL (*f)(const REAL [N_LAMBDA]), REAL*);
int   max_quad_points(void);
const REAL_D *f_d_at_qp(const QUAD *quad,
			const REAL *(*f)(const REAL[N_LAMBDA]),
			REAL_D *vec);
const REAL_D *grd_f_at_qp(const QUAD *,
			  const REAL *(*)(const REAL [N_LAMBDA]), 
			  REAL_D *);
const REAL_DD *grd_f_d_at_qp(const QUAD *, 
			     const REAL_D *(*)(const REAL [N_LAMBDA]),
			     REAL_DD *);
const QUAD_FAST *get_quad_fast(const BAS_FCTS *, const QUAD *, 
				      U_CHAR init_flag);

/***   file refine.c   ******************************************************/
extern U_CHAR refine(MESH *mesh);
extern U_CHAR global_refine(MESH *mesh, int mark);

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/***   file adapt.c   *******************************************************/
extern void adapt_method_stat(MESH *mesh, ADAPT_STAT *adapt);
extern void adapt_method_instat(MESH *mesh, ADAPT_INSTAT *adapt);
extern int marking(MESH *mesh, ADAPT_STAT *adapt);
extern ADAPT_INSTAT *get_adapt_instat(const int dim, const char *name,
				      const char *prefix,
				      int info, ADAPT_INSTAT *adapt_instat);
extern ADAPT_STAT *get_adapt_stat(const int dim, const char *name,
				  const char *prefix,
				  int info, ADAPT_STAT *adapt_stat);

/***   file assemble.c   ****************************************************/
const EL_MATRIX_INFO *fill_matrix_info(const OPERATOR_INFO *, 
				       EL_MATRIX_INFO *);
const Q00_PSI_PHI *get_q00_psi_phi(const BAS_FCTS *psi, const BAS_FCTS *phi,
				   const QUAD *quad);
const Q01_PSI_PHI *get_q01_psi_phi(const BAS_FCTS *psi, const BAS_FCTS *phi,
				   const QUAD *quad);
const Q10_PSI_PHI *get_q10_psi_phi(const BAS_FCTS *psi, const BAS_FCTS *phi,
				   const QUAD *quad);
const Q11_PSI_PHI *get_q11_psi_phi(const BAS_FCTS *psi, const BAS_FCTS *phi,
				   const QUAD *quad);

/***   file assemble_dowb.c  ************************************************/
const EL_DOWB_MATRIX_INFO *
fill_dowb_matrix_info(const DOWB_OPERATOR_INFO *operator_info, 
		      EL_DOWB_MATRIX_INFO *matrix_info);

/***   file bas_fct.c   *****************************************************/
extern const BAS_FCTS *get_bas_fcts(const char *name);
extern const BAS_FCTS *get_discontinuous_lagrange(int dim, int degree);
extern const BAS_FCTS *get_lagrange(int dim, int degree);
extern int new_bas_fcts(const BAS_FCTS * bas_fcts);

/***   file check.c   *******************************************************/
extern void check_mesh(MESH *mesh);

/***   file dof_admin.c   ***************************************************/
extern void free_dof_index(DOF_ADMIN *admin, int dof);
extern int  get_dof_index(DOF_ADMIN *admin);
extern void enlarge_dof_lists(DOF_ADMIN *admin, int minsize);
extern void update_dof_matrix(DOF_MATRIX *, REAL, int, const REAL **, 
			      const DOF *, const S_CHAR *);
extern void update_dof_dowb_matrix(DOF_DOWB_MATRIX *, REAL, int,
				   const void **, 
				   const DOF *, const S_CHAR *);
extern void update_dof_real_vec(DOF_REAL_VEC *, REAL, int, const REAL *,
				const DOF *, const S_CHAR *);
extern void update_dof_real_d_vec(DOF_REAL_D_VEC *, REAL, int, const REAL_D *,
				  const DOF *, const S_CHAR *);
extern void test_dof_matrix(DOF_MATRIX *matrix);
extern void test_dof_dowb_matrix(DOF_DOWB_MATRIX *matrix);

/***   file element.c   ***************************************************/
/* These routines are partially available as _?d-versions to avoid looking*/
/* up the dimension. This should be a small efficiency bonus.             */

extern int *sorted_wall_indices(const int dim, int face, int permno);
extern int wall_orientation(int dim, const EL *el, int face, int **vecp);
extern int *sort_wall_indices(const int dim,
				    const EL *el, int face, int *vec);

/* Dimension dependent routines, 0d, just dummies in most cases. */
extern int *sorted_wall_indices_0d(int face, int permno);
extern int wall_orientation_0d(const EL *el, int face, int **vec);
extern int *sort_wall_indices_0d(const EL *el, int face, int *vec);
extern int world_to_coord_0d(const EL_INFO *el_info, const REAL *,
			     REAL_B);
extern const REAL *coord_to_world_0d(const EL_INFO *, const REAL *, REAL_D);
extern REAL el_det_0d(const EL_INFO *);
extern REAL el_volume_0d(const EL_INFO *el_info);
extern REAL el_grd_lambda_0d(const EL_INFO *el_info,
			     REAL grd_lam[N_LAMBDA][DIM_OF_WORLD]);
extern REAL get_wall_normal_0d(const EL_INFO *, int , REAL *);

/* Dimension dependent routines, 1d */
extern int *sorted_wall_indices_1d(int face, int permno);
extern int wall_orientation_1d(const EL *el, int face, int **vec);
extern int *sort_wall_indices_1d(const EL *el, int face, int *vec);
extern int world_to_coord_1d(const EL_INFO *el_info, const REAL *,
			     REAL_B);
extern const REAL *coord_to_world_1d(const EL_INFO *, const REAL *, REAL_D);
extern REAL el_det_1d(const EL_INFO *);
extern REAL el_volume_1d(const EL_INFO *el_info);
extern REAL el_grd_lambda_1d(const EL_INFO *,
			     REAL grd_lam[N_LAMBDA][DIM_OF_WORLD]);
extern REAL get_wall_normal_1d(const EL_INFO *, int , REAL *);

#if DIM_OF_WORLD > 1
/* Dimension dependent routines, 2d */
extern int *sorted_wall_indices_2d(int face, int permno);
extern int wall_orientation_2d(const EL *el, int face, int **vec);
extern int *sort_wall_indices_2d(const EL *el, int face, int *vec);
extern int world_to_coord_2d(const EL_INFO *el_info, const REAL *,
			     REAL_B);
extern const REAL *coord_to_world_2d(const EL_INFO *, const REAL *, REAL_D);
extern REAL el_det_2d(const EL_INFO *);
extern REAL el_volume_2d(const EL_INFO *el_info);
extern REAL el_grd_lambda_2d(const EL_INFO *,
			     REAL grd_lam[N_LAMBDA][DIM_OF_WORLD]);
extern REAL get_wall_normal_2d(const EL_INFO *, int , REAL *);

#if DIM_OF_WORLD > 2
/* Dimension dependent routines, 3d */
extern int *sorted_wall_indices_3d(int face, int permno);
extern int wall_orientation_3d(const EL *el, int face, int **vec);
extern int *sort_wall_indices_3d(const EL *el, int face, int *vec);
extern int world_to_coord_3d(const EL_INFO *el_info, const REAL *,
			     REAL_B);
extern const REAL *coord_to_world_3d(const EL_INFO *, const REAL *, REAL_D);
extern REAL el_det_3d(const EL_INFO *);
extern REAL el_volume_3d(const EL_INFO *el_info);
extern REAL el_grd_lambda_3d(const EL_INFO *,
			     REAL grd_lam[N_LAMBDA][DIM_OF_WORLD]);
extern REAL get_wall_normal_3d(const EL_INFO *, int , REAL *);
#endif
#endif

/* Below we provide wrapper functions which distinguish the dimension
 * dependent routines by the co-dimension rather than by the dimension
 * of the underlying mesh. We start by defining a preprocessor macro
 * which spares us some typing and especially typos.
 *
 * In addition, we provide wrapper functions which decide by looking
 * at el_info->mesh->dim what to do.
 *
 */
#if DIM_OF_WORLD == 1

# define ALBERTA_CODIM_WRAPPER(ret, name, argtypes, argnames)	\
static inline ret name argtypes					\
{								\
  FUNCNAME(#name);						\
  								\
  switch (el_info->mesh->dim) {					\
  case 0: return name##_0d argnames;				\
  case 1: return name##_1d argnames;				\
  default:							\
    ERROR_EXIT("Illegal dim!\n");				\
    return (ret)0; /* just to make the compiler happy */	\
  }								\
}

# define ALBERTA_CODIM_ALIAS(ret, name, argtypes, argnames)		\
static inline ret name##_0cd argtypes { return name##_1d argnames; }	\
static inline ret name##_1cd argtypes { return name##_0d argnames; }

/* Variants which start at DOW == 2 and thus are empty here */
# define ALBERTA_CODIM_ALIAS_2(ret, name, argtypes, argnames)
# define ALBERTA_VOID_CODIM_ALIAS_2(name, argtypes, argnames)

#elif DIM_OF_WORLD == 2

# define ALBERTA_CODIM_WRAPPER(ret, name, argtypes, argnames)	\
static inline ret name argtypes					\
{								\
  FUNCNAME(#name);						\
								\
  switch (el_info->mesh->dim) {					\
  case 0: return name##_0d argnames;				\
  case 1: return name##_1d argnames;				\
  case 2: return name##_2d argnames;				\
  default:							\
    ERROR_EXIT("Illegal dim!\n");				\
    return (ret)0; /* just to make the compiler happy ... */	\
  }								\
}

# define ALBERTA_CODIM_ALIAS(ret, name, argtypes, argnames)		\
static inline ret name##_0cd argtypes { return name##_2d argnames; }	\
static inline ret name##_1cd argtypes { return name##_1d argnames; }	\
static inline ret name##_2cd argtypes { return name##_0d argnames; }

/* Variants which start at DOW == 2 */
# define ALBERTA_CODIM_ALIAS_2(ret, name, argtypes, argnames)		\
static inline ret name##_0cd argtypes { return name##_2d argnames; }
# define ALBERTA_VOID_CODIM_ALIAS_2(name, argtypes, argnames)		\
static inline void name##_0cd argtypes { name##_2d argnames; }

#elif DIM_OF_WORLD == 3

# define ALBERTA_CODIM_WRAPPER(ret, name, argtypes, argnames)	\
static inline ret name argtypes					\
{								\
  FUNCNAME(#name);						\
  								\
  switch (el_info->mesh->dim) {					\
  case 0: return name##_0d argnames;				\
  case 1: return name##_1d argnames;				\
  case 2: return name##_2d argnames;				\
  case 3: return name##_3d argnames;				\
  default:							\
    ERROR_EXIT("Illegal dim!\n");				\
    return (ret)0; /* just to make the compiler happy ... */	\
  }								\
}

# define ALBERTA_CODIM_ALIAS(ret, name, argtypes, argnames)		\
static inline ret name##_0cd argtypes { return name##_3d argnames; }	\
static inline ret name##_1cd argtypes { return name##_2d argnames; }	\
static inline ret name##_2cd argtypes { return name##_1d argnames; }	\
static inline ret name##_3cd argtypes { return name##_0d argnames; }

/* Variants which start at DOW == 2 */
# define ALBERTA_CODIM_ALIAS_2(ret, name, argtypes, argnames)		\
static inline ret name##_0cd argtypes { return name##_3d argnames; }	\
static inline ret name##_1cd argtypes { return name##_2d argnames; }
# define ALBERTA_VOID_CODIM_ALIAS_2(name, argtypes, argnames)	\
static inline void name##_0cd argtypes { name##_3d argnames; }	\
static inline void name##_1cd argtypes { name##_2d argnames; }

#else
# error Unsupported DIM_OF_WORLD
#endif

/* ..._Xcd() alias definitions */
ALBERTA_CODIM_ALIAS(int, world_to_coord,
		    (const EL_INFO *el_info,
		     const REAL *xy,
		     REAL_B lambda),
		    (el_info, xy, lambda))
ALBERTA_CODIM_ALIAS(const REAL *, coord_to_world,
		    (const EL_INFO *el_info, const REAL *l, REAL_D w),
		    (el_info, l, w))
ALBERTA_CODIM_ALIAS(REAL, el_volume, (const EL_INFO *el_info), (el_info))
ALBERTA_CODIM_ALIAS(REAL, el_det, (const EL_INFO *el_info), (el_info))
ALBERTA_CODIM_ALIAS(REAL, el_grd_lambda,
		    (const EL_INFO *el_info,
		     REAL grd_lam[N_LAMBDA][DIM_OF_WORLD]),
		    (el_info, grd_lam))
ALBERTA_CODIM_ALIAS(REAL, get_wall_normal,
		    (const EL_INFO *el_info, int i0, REAL *normal),
		    (el_info, i0, normal))
ALBERTA_CODIM_ALIAS(int *, sorted_wall_indices,
		    (int face, int permno),
		    (face, permno))
ALBERTA_CODIM_ALIAS(int, wall_orientation,
		    (const EL *el, int face, int **vecp),
		    (el, face, vecp))
ALBERTA_CODIM_ALIAS(int *, sort_wall_indices,
		    (const EL *el, int face, int *vec),
		    (el, face, vec))

/* Wrappers which look at el_info->mesh->dim */
ALBERTA_CODIM_WRAPPER(int, world_to_coord,
		      (const EL_INFO *el_info, const REAL *x, REAL_B lambda),
		      (el_info, x, lambda))
ALBERTA_CODIM_WRAPPER(const REAL *, coord_to_world,
		      (const EL_INFO *el_info, const REAL *lambda, REAL_D x),
		      (el_info, lambda, x))
ALBERTA_CODIM_WRAPPER(REAL, el_volume, (const EL_INFO *el_info), (el_info))
ALBERTA_CODIM_WRAPPER(REAL, el_det, (const EL_INFO *el_info), (el_info))
ALBERTA_CODIM_WRAPPER(REAL, el_grd_lambda,
		      (const EL_INFO *el_info,
		       REAL grd_lam[N_LAMBDA][DIM_OF_WORLD]),
		      (el_info, grd_lam))
ALBERTA_CODIM_WRAPPER(REAL, get_wall_normal,
		      (const EL_INFO *el_info, int i0, REAL *normal),
		      (el_info, i0, normal))

/* Some special wrapper functions, used for some stuff defined in
 * level.c 
 */
ALBERTA_CODIM_ALIAS_2(REAL, level_element_det, (const REAL_D coord[]), (coord))
ALBERTA_VOID_CODIM_ALIAS_2(level_coord_to_world,
			   (const REAL_D coord[],
			    const REAL_B lambda,
			    REAL_D world),
			   (coord, lambda, world))
ALBERTA_VOID_CODIM_ALIAS_2(level_coord_to_el_coord,
			   (const REAL_B v_lambda[],
			    const REAL_B lambda,
			    REAL_B el_lambda),
			   (v_lambda, lambda, el_lambda))

/***   file estimator.c   **************************************************/
#define INIT_UH 1
#define INIT_GRD_UH 2
REAL ellipt_est(const DOF_REAL_VEC *uh, ADAPT_STAT *adapt,
		REAL *(*rw_est)(EL *), REAL *(*rw_estc)(EL *),
                int degree, int norm, REAL C[3], const REAL_DD A,
                REAL (*f)(const EL_INFO *,const QUAD *,int,REAL,const REAL_D),
		FLAGS f_flag);
REAL heat_est(const DOF_REAL_VEC *uh, ADAPT_INSTAT *adapt,
	      REAL *(*rw_est)(EL *), REAL *(*rw_estc)(EL *),
	      int degree, REAL C[4], 
	      const DOF_REAL_VEC *uh_old, const REAL_DD A,
	      REAL (*f)(const EL_INFO *, const QUAD *, int iq, REAL t,
			REAL u, const REAL_D grd_u),
	      FLAGS f_flag);
/***   file estimator_dowb.c ************************************************/
REAL ellipt_est_d(const DOF_REAL_D_VEC *uh, ADAPT_STAT *adapt,
		  REAL *(*rw_est)(EL *), REAL *(*rw_estc)(EL *),
		  int degree, int norm, REAL C[3],
		  const REAL_DD A[DIM_OF_WORLD][DIM_OF_WORLD],
		  const REAL *(*f)(const EL_INFO *,
				   const QUAD *, int qp,
				   const REAL_D uh, const REAL_DD grd_uh,
                                   REAL_D res),
		  FLAGS f_flag);
REAL heat_est_d(const DOF_REAL_D_VEC *uh, ADAPT_INSTAT *adapt,
		REAL *(*rw_est)(EL *), REAL *(*rw_estc)(EL *),
		int degree, REAL C[4], 
		const DOF_REAL_D_VEC *uh_old,
		const REAL_DD A[DIM_OF_WORLD][DIM_OF_WORLD],
		const REAL *(*f)(const EL_INFO *, const QUAD *, int iq, REAL t,
				 const REAL_D u, const REAL_DD grd_u,
				 REAL_D res),
		FLAGS f_flag);
/***   file error.c   *******************************************************/
REAL max_err_at_qp(REAL (*)(const REAL_D), const DOF_REAL_VEC *, const QUAD *);
REAL max_err_d_at_qp(const REAL *(*)(const REAL_D, REAL_D), 
		     const DOF_REAL_D_VEC *, const QUAD *);
REAL max_err_at_vert(REAL (*)(const REAL_D), const DOF_REAL_VEC *);
REAL L2_err(REAL (*)(const REAL_D), const DOF_REAL_VEC *, const QUAD *, int, 
	     REAL *(*)(EL *), REAL *);
REAL H1_err(const REAL *(*)(const REAL_D, REAL_D), const DOF_REAL_VEC *, 
	     const QUAD *, int, REAL *(*)(EL *), REAL *);
REAL H1_err_d(const REAL_D *(*grd_u)(const REAL_D, REAL_DD),
	       const DOF_REAL_D_VEC *uh, const QUAD *quad, int rel_err,
	       REAL *(*rw_err_el)(EL *), REAL *max_h1_err2);
REAL L2_err_d(const REAL *(*)(const REAL_D, REAL_D), const DOF_REAL_D_VEC *,
	       const QUAD *, int, REAL *(*)(EL *), REAL *);

/***   file eval.c   ********************************************************/
REAL eval_uh(const REAL_B lambda, const REAL *uh_loc, const BAS_FCTS *b);
REAL eval_uh_fast(const REAL *uh_loc, const REAL *phi_val, int n_bfcts);
const REAL *eval_grd_uh(const REAL [N_LAMBDA], const REAL_D [N_LAMBDA],
			const REAL *, const BAS_FCTS *, REAL_D);
const REAL *eval_grd_uh_fast(const REAL_D [N_LAMBDA], const REAL *,
			     const REAL (*)[N_LAMBDA], int , REAL_D);
const REAL_D *eval_D2_uh(const REAL [N_LAMBDA], const REAL_D [N_LAMBDA],
			 const REAL *, const BAS_FCTS *, REAL_DD);
const REAL_D *eval_D2_uh_fast(const REAL_D [N_LAMBDA], const REAL *, 
			      const REAL (*)[N_LAMBDA][N_LAMBDA],
			      int, REAL_DD);

const REAL *eval_uh_d(const REAL [N_LAMBDA], const REAL_D *, const BAS_FCTS *,
		      REAL_D);
const REAL *eval_uh_d_fast(const REAL_D *, const REAL *, int, REAL_D);
const REAL_D *eval_grd_uh_d(const REAL [N_LAMBDA], const REAL_D [N_LAMBDA],
			     const REAL_D *, const BAS_FCTS *, REAL_DD);
const REAL_D *eval_grd_uh_d_fast(const REAL_D [N_LAMBDA], const REAL_D *, 
				 const REAL (*)[N_LAMBDA], int, REAL_DD);
const REAL_DD *eval_D2_uh_d(const REAL [N_LAMBDA], const REAL_D [N_LAMBDA],
			    const REAL_D *, const BAS_FCTS *, REAL_DD *);
const REAL_DD *eval_D2_uh_d_fast(const REAL_D [N_LAMBDA], const REAL_D *, 
				 const REAL (*)[N_LAMBDA][N_LAMBDA], int, 
				 REAL_DD *);

const REAL *uh_at_qp(const QUAD_FAST *, const REAL *uh_loc, REAL *vec);
const REAL_D *grd_uh_at_qp(const QUAD_FAST *,
			   const REAL_D grd_lambda[N_LAMBDA],
			   const REAL *uh_loc, REAL_D *vec);
const REAL_D *param_grd_uh_at_qp(const QUAD_FAST *fast,
				 REAL_D Lambda[][N_LAMBDA],
				 const REAL *uh_loc, REAL_D *vec);
const REAL_DD *D2_uh_at_qp(const QUAD_FAST *, const REAL_D [N_LAMBDA],
			   const REAL *, REAL_DD *);
const REAL_D *uh_d_at_qp(const QUAD_FAST *fast, const REAL_D *uh_loc,
			 REAL_D *vec);
const REAL_DD *grd_uh_d_at_qp(const QUAD_FAST *fast,
			      const REAL_D grd_lambda[N_LAMBDA], 
			      const REAL_D *uh_loc, REAL_DD *vec);
const REAL_DD *param_grd_uh_d_at_qp(const QUAD_FAST *fast,
				    REAL_D Lambda[][N_LAMBDA], 
				    const REAL_D *uh_loc, REAL_DD *vec);
const REAL_DD (*D2_uh_d_at_qp(const QUAD_FAST *, const REAL_D [N_LAMBDA],
			      const REAL_D *, 
			      REAL_DD (*)[DIM_OF_WORLD]))[DIM_OF_WORLD];
REAL eval_div_uh_d(const REAL lambda[N_LAMBDA],
		   const REAL_D grd_lambda[N_LAMBDA],
		   const REAL_D *uh_loc, const BAS_FCTS *b);
REAL eval_div_uh_d_fast(const REAL_D grd_lambda[N_LAMBDA],
			const REAL_D *uh_loc, 
			const REAL (*grd_phi)[N_LAMBDA], int n_bas_fcts);
const REAL *div_uh_d_at_qp(const QUAD_FAST *fast,
			   const REAL_D Lambda[N_LAMBDA], 
			   const REAL_D *uh_loc, REAL *vec);
const REAL *param_div_uh_d_at_qp(const QUAD_FAST *fast,
				 const REAL_D Lambda[][N_LAMBDA], 
				 const REAL_D *uh_loc, REAL *vec);
const REAL_DD *param_D2_uh_at_qp(const QUAD_FAST *fast,
				 const REAL_D Lambda[][N_LAMBDA],
				 const REAL *uh_loc, REAL_DD *vec);
const REAL_DD (*param_D2_uh_d_at_qp(const QUAD_FAST *fast,
				    const REAL_D grd_lam[][N_LAMBDA],
				    const REAL_D *uh_loc, 
			    REAL_DD (*vec)[DIM_OF_WORLD]))[DIM_OF_WORLD];
REAL H1_norm_uh(const QUAD *quad, const DOF_REAL_VEC *u_h);
REAL L2_norm_uh(const QUAD *quad, const DOF_REAL_VEC *u_h);
REAL H1_norm_uh_d(const QUAD *quad, const DOF_REAL_D_VEC *u_h);
REAL L2_norm_uh_d(const QUAD *quad, const DOF_REAL_D_VEC *u_h);
extern void interpol(REAL (*fct)(const REAL_D), DOF_REAL_VEC *);
extern void interpol_d(const REAL *(*)(const REAL_D, REAL_D), 
		       DOF_REAL_D_VEC *);

/***   file graphXO.c   ******************************************************/
GRAPH_WINDOW graph_open_window(const char *title, const char *geometry,
			       REAL *world, MESH *mesh);
void graph_close_window(GRAPH_WINDOW win);
void graph_clear_window(GRAPH_WINDOW win, const GRAPH_RGBCOLOR c);

void graph_mesh(GRAPH_WINDOW win, MESH *mesh, const GRAPH_RGBCOLOR c,
		FLAGS flag);
void graph_drv(GRAPH_WINDOW win, const DOF_REAL_VEC *uh,
	       REAL min, REAL max, int refine);
void graph_drv_d(GRAPH_WINDOW win, const DOF_REAL_D_VEC *uh,
		 REAL min, REAL max, int refine);
void graph_el_est(GRAPH_WINDOW win, MESH *mesh, REAL (*get_el_est)(EL *el),
		  REAL min, REAL max);

void graph_point(GRAPH_WINDOW, const REAL [2], const GRAPH_RGBCOLOR, float);
void graph_points(GRAPH_WINDOW win, int np, REAL (*p)[2], 
		  const GRAPH_RGBCOLOR c, float ps);
void graph_line(GRAPH_WINDOW, const REAL [2], const REAL [2], 
		const GRAPH_RGBCOLOR, float);

void graph_fvalues_2d(GRAPH_WINDOW win, MESH *mesh,
		      REAL(*fct)(const EL_INFO *el_info, const REAL *lambda),
		      FLAGS flags, REAL min, REAL max, int refine);
void graph_level_2d(GRAPH_WINDOW win, const DOF_REAL_VEC *v, REAL level,
		    const GRAPH_RGBCOLOR c, int refine);
void graph_levels_2d(GRAPH_WINDOW win, const DOF_REAL_VEC *v,
		     int n, REAL const *levels, const GRAPH_RGBCOLOR *color,
		     int refine);
void graph_level_d_2d(GRAPH_WINDOW, const DOF_REAL_D_VEC *,
		      REAL, const GRAPH_RGBCOLOR, int);
void graph_levels_d_2d(GRAPH_WINDOW, const DOF_REAL_D_VEC *,
		       int, const REAL *, const GRAPH_RGBCOLOR *, int);

/* multigrid level display routines:  */
void graph_mesh_mg_2d(GRAPH_WINDOW win, MESH *mesh, const GRAPH_RGBCOLOR c,
		      FLAGS flags, int mg_level);
void graph_values_mg_2d(GRAPH_WINDOW win, const DOF_REAL_VEC *v, 
			REAL min, REAL max, int refine,
			int mg_level, const FE_SPACE *fe_space,
			const int *sort_dof_invers);

/***   file l2scp.c   *******************************************************/
void dirichlet_bound(REAL (*)(const REAL_D), DOF_REAL_VEC *, DOF_REAL_VEC *,
		     DOF_SCHAR_VEC *);
void L2scp_fct_bas(REAL (*f)(const REAL_D), const QUAD *, DOF_REAL_VEC *fh);

void dirichlet_bound_d(const REAL *(*)(const REAL_D, REAL_D),
		       DOF_REAL_D_VEC *, DOF_REAL_D_VEC *, DOF_SCHAR_VEC *);
void L2scp_fct_bas_d(const REAL *(*f)(const REAL_D, REAL_D), const QUAD *,
		     DOF_REAL_D_VEC *fhd);

/*---  file oem_solve_s.c  -------------------------------------------------*/
int oem_solve_s(const DOF_MATRIX *A, const DOF_REAL_VEC *f, 
		DOF_REAL_VEC *u,  OEM_SOLVER, REAL, int, int, int, int);
int mat_vec_s(void *ud, int dim, const REAL *x, REAL *y);
void *init_mat_vec_s(MatrixTranspose,
		     const DOF_MATRIX *, const DOF_SCHAR_VEC *);
void exit_mat_vec_s(void *);

/*---  file oem_solve_d.c  -------------------------------------------------*/
int oem_solve_d(const DOF_MATRIX *A, const DOF_REAL_D_VEC *f, 
		DOF_REAL_D_VEC *u,  OEM_SOLVER, REAL, int, int, int, int);
int mat_vec_d(void *ud, int dim, const REAL *x, REAL *y);
void *init_mat_vec_d(MatrixTranspose, const DOF_MATRIX *,
		     const DOF_SCHAR_VEC *);
void exit_mat_vec_d(void *);

/*---  file oem_solve_dowb.c  ----------------------------------------------*/
int oem_solve_dowb(const DOF_DOWB_MATRIX *A, const DOF_REAL_D_VEC *f, 
		   DOF_REAL_D_VEC *u,  OEM_SOLVER, REAL, int, int, int, int);
int mat_vec_dowb(void *ud, int dim, const REAL *x, REAL *y);
void *init_mat_vec_dowb(MatrixTranspose, const DOF_DOWB_MATRIX *,
			const DOF_SCHAR_VEC *);
void exit_mat_vec_dowb(void *);

/*---  file parametric.c  --------------------------------------------------*/
void use_lagrange_parametric(MESH *mesh, int degree,
			     NODE_PROJECTION *n_proj, int strategy);
DOF_REAL_D_VEC *get_lagrange_coords(MESH *mesh);
DOF_UCHAR_VEC *get_lagrange_coord_flags(MESH *mesh);

/*--  file sor.c  ----------------------------------------------------------*/
int sor_dowb(DOF_DOWB_MATRIX *a, const DOF_REAL_D_VEC *f,
	     const DOF_SCHAR_VEC *b,
	     DOF_REAL_D_VEC *u, REAL omega, REAL tol, int max_iter, int info);
int sor_d(DOF_MATRIX *a, const DOF_REAL_D_VEC *f, const DOF_SCHAR_VEC *b,
	  DOF_REAL_D_VEC *u, REAL omega, REAL tol, int max_iter, int info);
int sor_s(DOF_MATRIX *a, const DOF_REAL_VEC *f, const DOF_SCHAR_VEC *b,
	  DOF_REAL_VEC *u, REAL omega, REAL tol, int max_iter, int info);

/***  file ssor.c  **********************************************************/
int ssor_dowb(DOF_DOWB_MATRIX *a,
	      const DOF_REAL_D_VEC *f, const DOF_SCHAR_VEC *b,
	      DOF_REAL_D_VEC *u, REAL omega, REAL tol, int max_iter, int info);
int ssor_d(DOF_MATRIX *a, const DOF_REAL_D_VEC *f, const DOF_SCHAR_VEC *b,
	   DOF_REAL_D_VEC *u, REAL omega, REAL tol, int max_iter, int info);
int ssor_s(DOF_MATRIX *a, const DOF_REAL_VEC *f, const DOF_SCHAR_VEC *b,
	   DOF_REAL_VEC *u, REAL omega, REAL tol, int max_iter, int info);

/***   file traverse_r.c  ***************************************************/
extern void mesh_traverse(MESH *mesh, int level, FLAGS fill_flag,
			  void (*el_fct)(const EL_INFO *, void *data), 
			  void *data);
extern void fill_macro_info(MESH *mesh, const MACRO_EL *mel, EL_INFO *elinfo);
extern void fill_elinfo(int ichild, const EL_INFO *parent_info, EL_INFO *elinfo);

/***   file traverse_nr.c ***************************************************/
extern TRAVERSE_STACK *get_traverse_stack(void);
extern void free_traverse_stack(TRAVERSE_STACK *stack);
extern const EL_INFO *traverse_first(TRAVERSE_STACK *stack,
				     MESH *mesh, int level, FLAGS fill_flag);
extern const EL_INFO *traverse_next(TRAVERSE_STACK *stack, const EL_INFO *);
extern const EL_INFO *traverse_neighbour(TRAVERSE_STACK *stack, const EL_INFO *,
					 int neighbour);
extern const EL_INFO *traverse_parent(const TRAVERSE_STACK *stack,
				      const EL_INFO *child);
extern const EL_INFO *subtree_traverse_first(TRAVERSE_STACK *stack,
					     const EL_INFO *local_root,
					     int level, FLAGS fill_flag);

#define TRAVERSE_START(mesh, stack, level, fill_flag, eli)	\
if (((eli) = traverse_first(stack, mesh, level, fill_flag))) do
#define TRAVERSE_STOP(stack, eli)		\
while (((eli) = traverse_next(stack, eli)))

/*---  file trav_xy.c  -----------------------------------------------------*/
extern int find_el_at_pt(MESH *mesh, const REAL_D xy,
			 EL_INFO **el_info_p, FLAGS flag, REAL bary[N_LAMBDA],
			 const MACRO_EL *start_mel,
			 const REAL_D xy0, REAL *sp);

/***   file read_mesh.c  *********************************************/
MESH *read_mesh(const char *fn, REAL *timeptr,
		NODE_PROJECTION *(*n_proj)(MESH *, MACRO_EL *, int));
DOF_REAL_VEC   *read_dof_real_vec(const char *, MESH *, FE_SPACE *);
DOF_REAL_D_VEC *read_dof_real_d_vec(const char *, MESH *, FE_SPACE *);
DOF_INT_VEC    *read_dof_int_vec(const char *, MESH *, FE_SPACE *);
DOF_SCHAR_VEC  *read_dof_schar_vec(const char *, MESH *, FE_SPACE *);
DOF_UCHAR_VEC  *read_dof_uchar_vec(const char *, MESH *, FE_SPACE *);

MESH *read_mesh_xdr(const char *, REAL *,
		    NODE_PROJECTION *(*)(MESH *, MACRO_EL *, int));
DOF_REAL_VEC   *read_dof_real_vec_xdr(const char *, MESH *, FE_SPACE *);
DOF_REAL_D_VEC *read_dof_real_d_vec_xdr(const char *, MESH *, FE_SPACE *);
DOF_INT_VEC    *read_dof_int_vec_xdr(const char *, MESH *, FE_SPACE *);
DOF_SCHAR_VEC  *read_dof_schar_vec_xdr(const char *, MESH *, FE_SPACE *);
DOF_UCHAR_VEC  *read_dof_uchar_vec_xdr(const char *, MESH *, FE_SPACE *);

/***   file write_mesh.c  ********************************************/
int write_mesh(MESH *, const char *, REAL);
int write_dof_real_vec(const DOF_REAL_VEC *, const char *);
int write_dof_real_d_vec(const DOF_REAL_D_VEC *, const char *);
int write_dof_int_vec(const DOF_INT_VEC *, const char *);
int write_dof_schar_vec(const DOF_SCHAR_VEC *, const char *);
int write_dof_uchar_vec(const DOF_UCHAR_VEC *, const char *);

int write_mesh_xdr(MESH *, const char *, REAL);
int write_dof_real_vec_xdr(const DOF_REAL_VEC *, const char *);
int write_dof_real_d_vec_xdr(const DOF_REAL_D_VEC *, const char *);
int write_dof_int_vec_xdr(const DOF_INT_VEC *, const char *);
int write_dof_schar_vec_xdr(const DOF_SCHAR_VEC *, const char *);
int write_dof_uchar_vec_xdr(const DOF_UCHAR_VEC *, const char *);

int write_dof_matrix_pbm(const DOF_MATRIX *matrix,
			 const char *filename);

/***   file write_mesh_gmv.c  ********************************************/

int write_mesh_gmv(MESH *mesh, const char *file_name, int write_ascii,
		   int use_refined_grid,
		   const int n_drv,
		   DOF_REAL_VEC **drv_ptr,
		   const int n_drdv,
		   DOF_REAL_D_VEC **drdv_ptr,
		   DOF_REAL_D_VEC *velocity, 
		   REAL sim_time);

int write_dof_vec_gmv(MESH *mesh,
		      const char *mesh_file, 
		      const char *file_name, int write_ascii,
		      int use_refined_grid,
		      const int n_drv,
		      DOF_REAL_VEC **drv_ptr,
		      const int n_drdv,
		      DOF_REAL_D_VEC **drdv_ptr,
		      DOF_REAL_D_VEC *velocity,
		      REAL sim_time);

/*--  file write_mesh_ps.c  ------------------------------------------------*/
void write_mesh_ps(MESH *mesh, const char *filename, const char *title,
                   const REAL x[2], const REAL y[2], int keepaspect, 
                   int draw_bound);

/*--------------------------------------------------------------------------*/
/*  interface for Lagrange elements for the gltools                         */
/*  file  gltools.c                                                         */
/*--------------------------------------------------------------------------*/
typedef void*    GLTOOLS_WINDOW;

GLTOOLS_WINDOW open_gltools_window(const char *, const char *, const REAL *,
				   MESH *, int);
void close_gltools_window(GLTOOLS_WINDOW);

extern int gltools_get_next_dialog(void);
extern void gltools_set_next_dialog(int dialog);
void gltools_est(GLTOOLS_WINDOW, MESH *, REAL (*)(EL *), REAL, REAL);
void gltools_disp_mesh(GLTOOLS_WINDOW, MESH *, int, const DOF_REAL_D_VEC *);
void gltools_mesh(GLTOOLS_WINDOW win, MESH *, int);
void gltools_disp_drv(GLTOOLS_WINDOW, const DOF_REAL_VEC *,
		      REAL, REAL, const DOF_REAL_D_VEC *);
void gltools_drv(GLTOOLS_WINDOW, const DOF_REAL_VEC *, REAL, REAL);
void gltools_disp_drv_d(GLTOOLS_WINDOW, const DOF_REAL_D_VEC *, 
			REAL, REAL, const DOF_REAL_D_VEC *);
void gltools_drv_d(GLTOOLS_WINDOW, const DOF_REAL_D_VEC *, REAL, REAL);
void gltools_disp_vec(GLTOOLS_WINDOW, const DOF_REAL_D_VEC *, 
		      REAL, REAL, const DOF_REAL_D_VEC *);
void gltools_vec(GLTOOLS_WINDOW, const DOF_REAL_D_VEC *, REAL, REAL);

/*--------------------------------------------------------------------------*/
/*  interface for Lagrange elements for the dxtools                         */
/*  file  dxtools.c                                                         */
/*--------------------------------------------------------------------------*/

typedef struct dxtools_window DXTOOLS_WINDOW;

extern DXTOOLS_WINDOW *open_dxtools_window(const char *title,
					   const char *geometry);

extern void close_dxtools_window(DXTOOLS_WINDOW *win);
extern void dxtools_mesh(DXTOOLS_WINDOW *win, MESH *mesh);
extern void dxtools_drv(DXTOOLS_WINDOW *win, const DOF_REAL_VEC *u);
extern void dxtools_drdv(DXTOOLS_WINDOW *win, const DOF_REAL_D_VEC *u);

#ifdef __cplusplus
}
#endif

#endif  /* !_ALBERTA_H_ */
