/*-------------------------------------------------------------------------------

Copyright (c) 2001 CerebraSoft

Module Name:

	bhBroadcast.cpp

Abstract:

	Application / NT service which broadcasts performance information
	to a mod backhand server.

License:

   All rights reserved.

   This product includes software developed at The Center for
   Networking and Distributed Systems at The Johns Hopkins University
   for use in the Backhand project (http://www.cnds.jhu.edu/backhand).
     Creator: Theo Schlossnagle 
     Guidance: Yair Amir

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer. 
 
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in
     the documentation and/or other materials provided with the
     distribution.
 
  3. All advertising materials mentioning features or use of this
     software must display the following acknowledgment:
     "This product includes software developed by Cerebrasoft
     (http://www.cerebrasoft.com).
        Creator: Rob Butler"

  4. The names "Backhand Broadcaster" and "NT Backhand Broadcaster" must
     not be used to endorse or promote products derived from this
     software without prior written permission. For written permission,
     please contact www.cerebrasoft.com.
 
  5. Products derived from this software may not be called "Backhand Broadcaster"
     or "NT Backhand Broadcaster" nor may "Backhand Broadcaster" or
     "NT Backhand Broadcaster" appear in their names without prior written
     permission. For written permission, please contact www.cerebrasoft.com.
 
  6. Redistributions of any form whatsoever must retain the following
     acknowledgment:
     "This product includes software developed by Cerebrasoft
     (http://www.cerebrasoft.com).
        Creator: Rob Butler"

  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
  ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
  OF THE POSSIBILITY OF SUCH DAMAGE.
---------------------------------------------------------------------------------*/

#include <windows.h>
#include <stdio.h>
#include "broadcast.h"
#include "eventLog.h"

/*-------------------------------------------------------------------------------
	Global Defines
---------------------------------------------------------------------------------*/

#define SERVICENAME	"NT Backhand Broadcaster"	// internal name of the service
#define DEPENDENCIES	""				// service dependencies - we have none
#define appName 	"NT Backhand Broadcaster"	// event log app name

/*-------------------------------------------------------------------------------
	Global Variables
---------------------------------------------------------------------------------*/

SERVICE_STATUS status;					// current status of the service
SERVICE_STATUS_HANDLE statusHandle;			// service status handle

/*-------------------------------------------------------------------------------
	Function Definitions
---------------------------------------------------------------------------------*/

VOID WINAPI service_main(DWORD, LPTSTR *);		// NT service main entry point
VOID WINAPI service_ctrl(DWORD);			// NT service control event dispatcher
BOOL ReportStatusToSCM(DWORD, DWORD, DWORD);		// NT service report status to SCM

VOID installService();					// installs app as NT service
VOID removeService();					// uninstalls app as NT service
VOID runApp();						// runs app as a console program
void showSvcErr(char *);				// displays error.  Used when (un)installing app.

/*-------------------------------------------------------------------------------

void main(int argc, char **argv)

	Program starting point.
---------------------------------------------------------------------------------*/

void main(int argc, char **argv){

    SERVICE_TABLE_ENTRY dispatchTable[] = {
        {SERVICENAME, (LPSERVICE_MAIN_FUNCTION) service_main},
        {NULL, NULL}
    };

    if ( argc > 1){
        if ( _stricmp( "-install", argv[1]) == 0 )
            installService();
        else if ( _stricmp( "-remove", argv[1]) == 0 )
            removeService();
        else if ( _stricmp( "-run", argv[1]) == 0 )
            runApp();
        else{
			printf("\n\tUnknown parameter.\n\nUSAGE:\n\n");
			printf("\t-install   to install the service\n");
			printf("\t-remove    to remove the service\n");
			printf("\t-run       to run as a console app\n");
        }
        exit(0);
    }else{	// try to start as service
		if (!StartServiceCtrlDispatcher(dispatchTable)){
			logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Unable to start NT Backhand Broadcaster service.");
			printf("\n\tUnable to start service.\n\nStart service in control panel or use commands below\n\nUSAGE:\n\n");
			printf("\t-install   to install the service\n");
			printf("\t-remove    to remove the service\n");
			printf("\t-run       to run as a console app\n");
		} 
	}
}

/*-------------------------------------------------------------------------------

void WINAPI service_main(DWORD dwArgc, LPTSTR *lpszArgv)

	perform initialization of the service

		dwArgc   - number of command line arguments
		lpszArgv - array of command line arguments
---------------------------------------------------------------------------------*/

void WINAPI service_main(DWORD dwArgc, LPTSTR *lpszArgv){

		// register our service control handler:
    statusHandle = RegisterServiceCtrlHandler(SERVICENAME, service_ctrl);

    if (!statusHandle)
		return;

		// SERVICE_STATUS members that don't change
    status.dwServiceType = SERVICE_WIN32_OWN_PROCESS;
    status.dwServiceSpecificExitCode = 0;

	    // report the status to the service control manager.
    ReportStatusToSCM(SERVICE_START_PENDING, NO_ERROR, 3000);

		// start broadcasting immediately, and report status
	if(startBroadcasting(0))
		ReportStatusToSCM(SERVICE_RUNNING, NO_ERROR, 3000);
	else
		exit(1);

    return;
}


/*-------------------------------------------------------------------------------

VOID WINAPI service_ctrl(DWORD ctrlCode)

	Called by SCM (NT Service Control Manager)
---------------------------------------------------------------------------------*/

VOID WINAPI service_ctrl(DWORD ctrlCode){

    switch(ctrlCode){

			// Stop the service.
        case SERVICE_CONTROL_STOP:
				// set the status
			ReportStatusToSCM(SERVICE_STOP_PENDING, NO_ERROR, 0);
				// stop broadcasting
			stopBroadcasting();
				// set the status
			ReportStatusToSCM(SERVICE_STOPPED, NO_ERROR, 0);
            break;

	        // Update the service status.	
        case SERVICE_CONTROL_INTERROGATE:
			ReportStatusToSCM(status.dwCurrentState, NO_ERROR, 0);
            break;

			// invalid control code, send status anyway.
        default:
			ReportStatusToSCM(status.dwCurrentState, NO_ERROR, 0);
            break;
    }
}

/*-------------------------------------------------------------------------------

BOOL ReportStatusToSCM(DWORD currentState, DWORD exitCode, DWORD waitHint)

	reports service status to SCM

		PARAMETERS:
			currentState - the state of the service
			exitCode - error code to report
			waitHint - worst case estimate to next checkpoint

		RETURN VALUE:
			TRUE  - success
			FALSE - failure
---------------------------------------------------------------------------------*/

BOOL ReportStatusToSCM(DWORD currentState, DWORD exitCode, DWORD waitHint){

    static DWORD checkPoint = 1;
    BOOL retVal = TRUE;

    if (currentState == SERVICE_START_PENDING)
		status.dwControlsAccepted = 0;
    else
        status.dwControlsAccepted = SERVICE_ACCEPT_STOP;

    status.dwCurrentState = currentState;
    status.dwWin32ExitCode = exitCode;
    status.dwWaitHint = waitHint;

    if (currentState == SERVICE_RUNNING || currentState == SERVICE_STOPPED )
        status.dwCheckPoint = 0;
    else
        status.dwCheckPoint = checkPoint++;

        // Report the status of the service to the service control manager.
    if (!(retVal = SetServiceStatus(statusHandle, &status)))
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "SetServiceStatus failed.");

    return retVal;
}

/*-------------------------------------------------------------------------------

void installService()

	Installs the service
---------------------------------------------------------------------------------*/

void installService(){

    SC_HANDLE   service;
    SC_HANDLE   SCManager;

    TCHAR path[1024];

    if (GetModuleFileName(NULL, path, 1024 ) == 0 ){
        showSvcErr("Unable to install service");
        return;
    }

    SCManager = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS );
    
    if (SCManager){
        service = CreateService(SCManager, SERVICENAME, SERVICENAME, SERVICE_ALL_ACCESS, 
						SERVICE_WIN32_OWN_PROCESS, SERVICE_AUTO_START, SERVICE_ERROR_NORMAL, path, NULL, 
						NULL, DEPENDENCIES, NULL, NULL); 

        if (service ){
            printf("%s installed.\n", SERVICENAME);
            CloseServiceHandle(service);
        }else{
            showSvcErr("CreateService failed");
        }

        CloseServiceHandle(SCManager);
    }else{
        showSvcErr("OpenSCManager failed");
	}
}

/*-------------------------------------------------------------------------------

void removeService()

	Stops and removes the service
---------------------------------------------------------------------------------*/

void removeService(){

    SC_HANDLE   service;
    SC_HANDLE   SCManager;

    SCManager = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS);

    if (SCManager){
        service = OpenService(SCManager, SERVICENAME, SERVICE_ALL_ACCESS);

        if (service){
				// try to stop the service
            if (ControlService(service, SERVICE_CONTROL_STOP, &status)){
                printf("Stopping %s.", SERVICENAME);
                Sleep( 1000 );

                while(QueryServiceStatus(service, &status)){
                    if (status.dwCurrentState == SERVICE_STOP_PENDING ){
                        printf(".");
                        Sleep(1000);
                    }else{
                        break;
					}
                }

                if (status.dwCurrentState == SERVICE_STOPPED)
                    printf("\n%s stopped.\n", SERVICENAME);
                else
                    printf("\n%s failed to stop.\n", SERVICENAME);
            }
				// now remove the service
            if(DeleteService(service))
                printf("%s removed.\n", SERVICENAME);
            else
                showSvcErr("DeleteService failed");

            CloseServiceHandle(service);
        }else{
            showSvcErr("OpenService failed");
		}

        CloseServiceHandle(SCManager);
    }
    else{
        showSvcErr("OpenSCManager failed");
	}
}

/*-------------------------------------------------------------------------------

void runApp()

	Run the program as a console app
---------------------------------------------------------------------------------*/

void runApp(){

		// start broadcasting immediately
	startBroadcasting(0);

	printf("\n\n\tBroadcasting mod_backhand performance data.");
	printf("\n\n\tHit enter (return) key to quit!\n");
	getchar();
	printf("\tQuitting....\n");

		// stop broadcasting
	stopBroadcasting();

	printf("\tBroadcasting Stopped\n");
}

/*-------------------------------------------------------------------------------

void showSvcErr(char *msg)

	Display error msg
---------------------------------------------------------------------------------*/

void showSvcErr(char *msg){

    DWORD retVal;
    LPTSTR tmp = NULL;

    retVal = FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_ARGUMENT_ARRAY,
							NULL, GetLastError(), LANG_NEUTRAL, (LPTSTR) &tmp, 0, NULL );

		// supplied buffer is not long enough
    if (!retVal)
        printf("Error occurred, and could not retrieve error message\n");
    else{
        tmp[strlen(tmp)-2] = '\0';  //remove cr and newline character
		printf("%s - %s  Code: 0x%x\n", msg, tmp, GetLastError());
		free(tmp);
    }
}