/* Copyright (c) 1992, 1999 John E. Davis.
 * This file is part of the S-Lang library.
 *
 * You may distribute under the terms of either the GNU General Public
 * License or the Perl Artistic License.
 */

#include <sys/types.h>
#include <stdio.h>

#include "types.h"

/*
 * tt_sprintf.c formats terminfo/termcap strings into something that can be
 * sent to the terminal.
 *
 * I stole this function from S-Lang and slightly modified it.  Thank you,
 * John Davis.
 *
 * Chris Bond <cbond@stormix.com>
 */

char *
tt_sprintf(buf, fmt, x, y)
	char *buf, *fmt;
	int x, y;
{
   char *fmt_max;
   register u_char *b, ch;
   int offset;
   int z, z1, parse_level;
   int zero_pad;
   int field_width;
   int variables [26];
   int stack [64];
   unsigned int stack_len;
   int parms [10];
#define STACK_POP (stack_len ? stack[--stack_len] : 0)

   if (fmt == NULL)
     {
	*buf = 0;
	return buf;
     }

   stack [0] = y;	       /* pushed for termcap */
   stack [1] = x;
   stack_len = 2;

   parms [1] = x;	       /* p1 */
   parms [2] = y;	       /* p2 */

   offset = 0;
   zero_pad = 0;
   field_width = 0;

   b = (unsigned char *) buf;
   fmt_max = fmt + strlen (fmt);

   while (fmt < fmt_max)
     {
	ch = *fmt++;

	if (ch != '%')
	  {
	     *b++ = ch;
	     continue;
	  }

	if (fmt == fmt_max) break;
	ch = *fmt++;

	switch (ch)
	  {
	   default:
	     *b++ = ch;
	     break;

	   case 'p':

	     if (fmt == fmt_max) break;
	     ch = *fmt++;
	     if ((ch >= '0') && (ch <= '9'))
	       stack [stack_len++] = parms [ch - '0'];
	     break;

	   case '\'':   /* 'x' */
	     if (fmt == fmt_max) break;
	     stack [stack_len++] = *fmt++;
	     if (fmt < fmt_max) fmt++;     /* skip ' */
	     break;

	   case '{':	       /* literal constant, e.g. {30} */
	     z = 0;
	     while ((fmt < fmt_max) && ((ch = *fmt) <= '9') && (ch >= '0'))
	       {
		  z = z * 10 + (ch - '0');
		  fmt++;
	       }
	     stack [stack_len++] = z;
	     if ((ch == '}') && (fmt < fmt_max)) fmt++;
	     break;

	   case '0':
	     if (fmt == fmt_max) break;
	     ch = *fmt;
	     if ((ch != '2') && (ch != '3'))
	       break;
	     zero_pad = 1;
	     fmt++;
	     /* drop */

	   case '2':
	   case '3':
	     if (fmt == fmt_max)
	     if (*fmt == 'x')
	       {
		  char x_fmt_buf [4];
		  char *x_fmt_buf_ptr;

		  x_fmt_buf_ptr = x_fmt_buf;
		  if (zero_pad) *x_fmt_buf_ptr++ = '0';
		  *x_fmt_buf_ptr++ = ch;
		  *x_fmt_buf_ptr++ = 'X';
		  *x_fmt_buf_ptr = 0;

		  z = STACK_POP;
		  z += offset;

		  sprintf ((char *)b, x_fmt_buf, z);
		  b += strlen ((char *)b);
		  zero_pad = 0;
		  break;
	       }

	     field_width = (ch - '0');
		  /* drop */

	   case 'd':
	     z = STACK_POP;
	     z += offset;
	     if (z >= 100)
	       {
		  *b++ = z / 100 + '0';
		  z = z % 100;
		  zero_pad = 1;
		  field_width = 2;
	       }
	     else if (zero_pad && (field_width == 3))
	       *b++ = '0';

	     if (z >= 10)
	       {
		  *b++ = z / 10 + '0';
		  z = z % 10;
	       }
	     else if (zero_pad && (field_width >= 2))
	       *b++ = '0';

	     *b++ = z + '0';
	     field_width = zero_pad = 0;
	     break;

	   case 'x':
	     z = STACK_POP;
	     z += offset;
	     sprintf ((char *) b, "%X", z);
	     b += strlen ((char *)b);
	     break;

	   case 'i':
	     offset = 1;
	     break;

	   case '+':
	     /* Handling this depends upon whether or not we are parsing
	      * terminfo.  Terminfo requires the stack so use it as an
	      * indicator.
	      */
	     if (stack_len > 2)
	       {
		  z = STACK_POP;
		  stack [stack_len - 1] += z;
	       }
	     else if (fmt < fmt_max)
	       {
		  ch = *fmt++;
		  if ((unsigned char) ch == 128) ch = 0;
		  ch = ch + (unsigned char) STACK_POP;
		  if (ch == '\n') ch++;
		  *b++ = ch;
	       }
	     break;

	     /* Binary operators */
	   case '-':
	   case '*':
	   case '/':
	   case 'm':
	   case '&':
	   case '|':
	   case '^':
	   case '=':
	   case '>':
	   case '<':
	   case 'A':
	   case 'O':
	     z1 = STACK_POP;
	     z = STACK_POP;
	     switch (ch)
	       {
		case '-': z = (z - z1); break;
		case '*': z = (z * z1); break;
		case '/': z = (z / z1); break;
		case 'm': z = (z % z1); break;
		case '&': z = (z & z1); break;
		case '|': z = (z | z1); break;
		case '^': z = (z ^ z1); break;
		case '=': z = (z == z1); break;
		case '>': z = (z > z1); break;
		case '<': z = (z < z1); break;
		case 'A': z = (z && z1); break;
		case 'O': z = (z || z1); break;
	       }
	     stack [stack_len++] = z;
	     break;

	     /* unary */
	   case '!':
	     z = STACK_POP;
	     stack [stack_len++] = !z;
	     break;

	   case '~':
	     z = STACK_POP;
	     stack [stack_len++] = ~z;
	     break;

	   case 'r':		       /* termcap -- swap parameters */
	     z = stack [0];
	     stack [0] = stack [1];
	     stack [1] = z;
	     break;

	   case '.':		       /* termcap */
	   case 'c':
	     ch = (unsigned char) STACK_POP;
	     if (ch == '\n') ch++;
	     *b++ = ch;
	     break;

	   case 'g':
	     if (fmt == fmt_max) break;
	     ch = *fmt++;
	     if ((ch >= 'a') && (ch <= 'z'))
	       stack [stack_len++] = variables [ch - 'a'];
	     break;

	   case 'P':
	     if (fmt == fmt_max) break;
	     ch = *fmt++;
	     if ((ch >= 'a') && (ch <= 'z'))
	       variables [ch - 'a'] = STACK_POP;
	     break;

	     /* If then else parsing.  Actually, this is rather easy.  The
	      * key is to notice that 'then' does all the work.  'if' simply
	      * there to indicate the start of a test and endif indicates
	      * the end of tests.  If 'else' is seen, then skip to
	      * endif.
	      */
	   case '?':		       /* if */
	   case ';':		       /* endif */
	     break;

	   case 't':		       /* then */
	     z = STACK_POP;
	     if (z != 0)
	       break;		       /* good.  Continue parsing. */

	     /* z == 0 and test has failed.  So, skip past this entire if
	      * expression to the matching else or matching endif.
	      */
	     /* drop */
	   case 'e':		       /* else */

	     parse_level = 0;
	     while (fmt < fmt_max)
	       {
		  unsigned char ch1;

		  ch1 = *fmt++;
		  if ((ch1 != '%') || (fmt == fmt_max))
		    continue;

		  ch1 = *fmt++;

		  if (ch1 == '?') parse_level++;   /* new if */
		  else if (ch1 == 'e')
		    {
		       if ((ch != 'e') && (parse_level == 0))
			 break;
		    }
		  else if (ch1 == ';')
		    {
		       if (parse_level == 0)
			 break;
		       parse_level--;
		    }
	       }
	     break;
	  }
     }
   *b = 0;
   return buf;
}
