/*
 libaudiomask - hybrid simultaneous audio masking threshold evaluation library
    Copyright (C) 2000-2010  Dr Matthew Raphael Flax

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef AUDIOMASKER_H_
#define AUDIOMASKER_H_

#include "AudioMask.H"
#include "depukfb.H"
#include <mffm/realFFT.H>
#include <stdlib.h>

#define DEFAULT_FBCOUNT 100
#define DEFAULT_SAMPLECOUNT 512
#define DEFAULT_SAMPLEFREQ 44100
//#define DEFAULT_LOWFERQ 25

// Recurse defines the number of times we recurse the output to input >=2
//#define RECURSE 7

/**
* Audio masking class - top level
* Class to transform input audio in the time domain to a simultaneous audio mask in the frequency domain.
*
* ========================= HOWTO ===============================
*
*    // First find the masking threshold
*
*    AudioMasker masker(sampleFreq, count); // Create the audio masker class using fs=sampleFreq and count filters
*
*    masker.excite(input, sampleCount); // find the mask for the array of input data which has sampleCount time samples.
*
*    // Now do something with the masking threshold ...
*
*    // The frequency domain mask is now located here
*
*    for (int j=0; j<count;j++)
*
*        masker.mask[j]; // This is the mask at each of the count frequencies of interest
*
*    // A more sophisticated example - find the threshold for each Fourier bin
*
*    double fact=(double)sampleFreq/((double)sampleCount-1.0); // convert from an index to the equivalent Fourier bin frequency
*
*    for (int j=0; j<halfSampleCount;j++){
*
*        cout<<"finding for freq "<<j*fact<<'\t'; // The frequency we are inspecting
*
*        double threshold=masker.findThreshold(j*fact); // The masking threshold
*
*        20*log10(threshold); // The threshold in decibels (dB)
*
*    }
*
*/
class AudioMasker : public AudioMask {
  double **output; //!< Filter bank output
  double **powOutput; //!< Filter bank output power
  double *input; //!<Filter bank input
  int sampleCount; //!<The sample count
  int bankCount; //!<The filter bank count

  realFFTData *fftData; //!< The FFT data
  realFFT *fft; //!< The FFT

  void FBDeMalloc(void);//!< Filter bank output matrix memory de-allocation

  void FBMalloc(void);  //!< Filter bank output matrix memory allocation

  void process(void); //!< Process the transformation
public:
  DepUKFB *pfb; //!< roex filters
  //  GTFB *gtfb; //!< gammatone filters

/**
* The Audio masking constructor
* @ sampFreq The sample frequency of the time domain data
* @ fBankCount The number of filter banks
*/
  AudioMasker(int sampFreq, int fBankCount);
  AudioMasker(void); //!< Audio masker constructor - allowing specification of fs and sub-band count later
  ~AudioMasker(void); //!< Audio masker deconstructor


/**
* Finds the excitation for input data
* @ Input Using short int input data
* @ sCount samples
*/
  void excite(short int *Input, int sCount);
/**
* Finds the excitation for input data
* @ Input Using double input data
* @ sCount samples
*/
  void excite(double *Input, int sCount);

  /**
  * Returns the simultaneous masking threshold at a particular frequency
  * @ freq The frequency of interest
  */
  double findThreshold(double freq);
};
#endif //AUDIOMASKER_H_
/** \example AudioMaskerExample.cc
 * This is an example of how to use the AudioMasker class
 * See the example file to work out how to use these Audio masking classes
 */

