/*
 * @(#)ReadUtil.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.IOException;
import java.io.Reader;

/**
 * Helps with standard read methods.  Package protected.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     December 15, 2002
 */
class ReadUtil
{
    /**
     * reads from the stream up to the given character, but not
     * including it.  If an end-of-stream is encountered before the
     * character, then an exception is thrown.
     */
    public static final String readTo( Reader in, char val )
            throws IOException
    {
        StringBuffer sb = new StringBuffer();
        int c = in.read();
        while (c != -1 && (char)c != val)
        {
            sb.append( (char)c );
            c = in.read();
        }
        if (c == -1)
        {
            throw new IOException( "Expected '"+val+
                "', but found end-of-stream." );
        }
        
        return sb.toString();
    }
    
    
    public static final String readCount( Reader in, int count )
            throws IOException
    {
        char c[] = new char[ count ];
        int read = in.read( c, 0, count );
        if (read < 0)
        {
            throw new IOException( "Expected to read "+(count)+
                " more characters, but encountered end-of-stream." );
        }
        int totRead = read;
        while (totRead < count)
        {
            read = in.read( c, totRead, (count - totRead) );
            if (read < 0)
            {
                throw new IOException( "Expected to read "+(count-totRead)+
                    " more characters, but encountered end-of-stream." );
            }
            totRead += read;
        }
        
        return new String( c );
    }

    
    
    public static final int toInt( String s )
            throws IOException
    {
        if (s == null || s.length() <= 0)
        {
            throw new IOException(
                "Invalid empty string; expected an integer." );
        }
        try
        {
            int i = Integer.parseInt( s );
            return i;
        }
        catch (NumberFormatException ex)
        {
            throw new IOException( "String '"+s+"' is not an integer." );
        }
    }

    
    
    
    public static final long toLong( String s )
            throws IOException
    {
        if (s == null || s.length() <= 0)
        {
            throw new IOException(
                "Invalid empty string; expected a long integer." );
        }
        try
        {
            long i = Long.parseLong( s );
            return i;
        }
        catch (NumberFormatException ex)
        {
            throw new IOException( "String '"+s+"' is not a long integer." );
        }
    }

}
