/*
 * @(#)DirectoryChannelLogReader.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.logger;

import java.io.File;
import java.io.IOException;

import net.sourceforge.groboutils.codecoverage.v2.IChannelLogReader;
import net.sourceforge.groboutils.codecoverage.v2.IClassChannelLogReader;

/**
 * Reads logs written by DirectoryChannelLogger.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     December 17, 2002
 */
public class DirectoryChannelLogReader implements IChannelLogReader
{
    private static final org.apache.log4j.Logger LOG =
        org.apache.log4j.Logger.getLogger( DirectoryChannelLogReader.class );
    private static final String EXTENTION =
            DirectoryChannelLogger.CLASS_LOG_EXTENTION;
    
    private File channelDir;
    
    
    public DirectoryChannelLogReader( File baseDir, short channel )
    {
        File newDir = new File( baseDir, Short.toString( channel ) );
        if (!newDir.exists() || !newDir.isDirectory())
        {
            throw new IllegalArgumentException( "File "+newDir+
                " is not a directory or doesn't exist." );
        }
        this.channelDir = newDir;
    }
    
    
    /**
     * Returns a channel reader for a specific class signature.
     * 
     * @return the reader for the given class signature.  If there are no
     *    logs for the given class, then the reader will not return anything.
     * @exception IOException thrown if there was an underlying problem reading
     *        from the log.
     */
    public IClassChannelLogReader getReaderForClassSignature( String signature )
            throws IOException
    {
        File logFile = new File( this.channelDir, signature+EXTENTION );
        if (!logFile.exists())
        {
            // this isn't a critical error: it means that nothing in this class
            // was ever executed.
            LOG.info( "Couldn't finding a log file for class "+signature+
                " in file "+logFile+"." );
            return new EmptyClassChannelLogReader();
        }
        
        LOG.debug( "Loading class log reader from file "+logFile+"." );
        return new DirectoryClassChannelLogReader( logFile, signature );
    }
}

